;;; -*- Syntax: Common-lisp; Package: USER -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: asyntax.lisp,v 1.1 92/04/16 09:30:23 clancy Exp $

;;;                        ****** ASYNTAX ******

; These functions define the syntax of predicates and rules in Algernon.

; Predicates:  A predicate can be either of form (not (slot frame value)) or (slot frame value).

; negated, frame, slot, facet, and value made into macros and moved to aglobals.

;(defun negated (pred)
;  (eql (car pred) 'not))

;(defun frame (pred)
;  (second (if (negated pred)
;              (second pred)
;              pred)))

;(defun slot (pred)
;  (first (if (negated pred)
;	     (second pred)
;             pred)))

;(defun facet (pred)
;  (if (negated pred)
;      @n-value
;      @value))

;(defun value (pred)
;  (third (if (negated pred)
;	     (second pred)
;             pred)))

(defun predicate (frame slot facet value)
  (if (eql facet @value)
      (list slot frame value)
      (list 'not (list slot frame value))))



; Negate: Returns the negation of p.
;   Complicated March 89 to handle representatives and skolem individuals correctly.
;   Complications taken out August 89 when reps changed to arbitrary objects
;   (and I don't know how to negate then at all yet ...).
;
(defun negate (p)
  (if (negated p)
      (list (slot p) (frame p) (value p))
      (list 'not (list (slot p) (frame p) (value p)))))


; Rules are classified by type and slot.  There are four types of rules:
;
; IF-ADDED rules have the form:
;     ((spouse _a _b) -> (spouse _b _a))
;
; while IF-NEEDED rules have form
;  ((has-disease _p flu)  <- (has-symptom _p fever) (has-symptom _p nausea))
;
; NONV-IF-ADDED and NONV-IF-NEEDED rules are similar but are associated with
; the non-value facet of slots.  For example:
;  ((not (flies ?x t)) <- (member ?x ?s) (name ?s (penguins)))
;
(defun RULE-TYPE (rule)
  (cond ((member '-> rule :test #'eq) (if (negated (car rule)) @n-if-added @if-added))
        ((member '<- rule :test #'eq) (if (negated (car rule)) @n-if-needed @if-needed))
	(t
	 (throw 'error (format nil "Illegal rule syntax --- No implication (-> or <-) in rule ~(~a~) ." rule)))))

;;; RULE-FACET: Computes the actual facet in which to store a
;;;   rule given its type and class.
;;;
;;; A rule type is one of @n-if-added, @if-added, @n-if-needed, or @if-needed.
;;; A rule class is one of 'frame or 'slot.
;;;
(defun rule-facet (type class)
  (if (eql class 'frame)
      type
      (cond ((eq type @n-if-added) @sn-if-added)
	    ((eq type @if-added)   @sif-added)
	    ((eq type @n-if-needed) @sn-if-needed)
	    ((eq type @if-needed) @sif-needed)
	    (T (error "Algernon Bug -- The type ~a was not a valid rule type." type)))))

; The slot for a rule is generally the slot of the predicate in the head
; of the rule.  The only exceptions to this are generic rules.
; Generic rules are rules which can apply to any
; slot.  They have form:
;  ((?p ?x ?v) <- (member ?p ?s) (representative ?s ?r) (?p ?r ?v)).
; The slot for a generic rule is @generic-rules.
;
; Modified 6/25/90 to take a result and apply the substitution in it before
; extracting slot.
;
(defun RULE-SLOT (rule &optional result)
  (let* ((rule-slot (slot (car rule))))
    (if result (setq rule-slot (substitute-bindings rule-slot (aresult-sub result))))
    (if (variable? rule-slot)
      @generic-rules
      rule-slot)))

(defun rule-frame (rule &optional result)
  (let* ((rule-frame (frame (car rule))))
    (if result (setq rule-frame (substitute-bindings rule-frame (aresult-sub result))))
    rule-frame))

(defun rule-value (rule &optional result)
  (let* ((rule-value (value (car rule))))
    (if result (setq rule-value (substitute-bindings rule-value (aresult-sub result))))
    rule-value))

; A rule has a list of antecedent clauses at the tail of the arrow, and a
; list of consequent clauses at the head of the arrow.

; First a utility function:
(defun copy-list-to (list end-flag)
  (if (eq (car list) end-flag)
    nil
    (cons (car list) (copy-list-to (cdr list) end-flag))))

(defun antecedent (rule)
  (let ((tail nil))
    (cond ((setq tail (member '<- rule :test #'eq))
	   (cdr tail))
	  ((setq tail (member '-> rule :test #'eq))
           (copy-list-to rule '->))
	  (t
	   (throw 'error
                  (format nil "Illegal rule syntax --- No implication (-> or <-) in rule ~(~a~) ."
                          rule))))))

(defun consequent (rule)
  (let ((tail nil))
    (cond ((setq tail (member '-> rule :test #'eq))
	   (cdr tail))
	  ((setq tail (member '<- rule :test #'eq))
           (copy-list-to rule '<-))
	  (t
	   (throw 'error
                  (format nil "Illegal rule syntax --- No implication (-> or <-) in rule ~(~a~) ."
                          rule))))))

