;;; -*- Syntax: Common-lisp; Package: USER -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: asymbolics.lisp,v 1.1 92/04/16 09:30:21 clancy Exp $

;;;                        ****** ASYMBOLICS ******

;;; Device dependent routines for symbolics.


(defun clear-window ()
  (send *standard-output* :clear-history))


;;; Dot tracing:

;;; Let Algernon know dot tracing is supported:
(push 'dot-trace *algy-features*)

(defparameter *dot-count* 0)
(defparameter *dot-limit* 100)
(defvar *dot-output* *standard-output*)

(defparameter *dot-trace-present* t)

(defun present-obj (thing string type D,#TD1PsT[Begin using 006 escapes](1 0 (NIL 0) (SAGE:SANS-SERIF-BODY SAGE::TYPEWRITER :NORMAL) "CPTFONT")&optional(2 0 (NIL 0) (NIL NIL NIL) "CPTFONT") 1(stream *standard-output*)2)
  (1dw:with-output-as-presentation
2      1(:single-box t :stream stream :type 2type 1:object2 thing)
    (princ string stream)))

(defun dot-present (symbol text)
  (when *dot-trace-present* 
    (present-obj text symbol '(satisfies algyp) *dot-output*)
    (dot-count)))

(defun dot-count ()
  (incf *dot-count*)
  (when (> *dot-count* *dot-limit*)
    (terpri *dot-output*)
    (princ "  ")
    (setq *dot-count* 2)))

(defun dot-terpri ()
  (terpri *dot-output*)
  (setq *dot-count* 0))

(defun reset-presenter ()
  (setq *dot-count* 0))

(defun algyp (x)
  (listp x))

(defun view-dot-trace ()
  (do ((input))
      ((eql input 'quit)
       (values))
    (setq input (accept '(or (satisfies algyp)
			     (member quit))
			:prompt "Select Algy object"
			:default 'Quit))
    (unless (eq input 'Quit)
      (terpri)(terpri)
      (format t "  ")
      (cond ((rule-pair-p input)
	     (pp-rule 2 input))
	    (t (pp-list 2 80 2 t input))))))

(defun rule-pair-p (x)
  (and (listp x)
       (listp (car x))
       (or  (member '-> (car  x))
	    (member '<- (car  x)))))


;;; Utilities for overviewing the knowledge-base.

;; First a window to do it in:
(defvar kb-win
	(tv:make-window 'dw:dynamic-window
			:edges '(100 100 500 500)
			:expose-p nil
			:blinker-p nil
			:default-character-style '(:fix :roman :normal)
			:save-bits t))

;; Here are the commands that can be sent to the window:
;;(send kb-win :kill)
;;(send kb-win :expose)
;;(send kb-win :clear-window)
;;(send kb-win :clear-history)

(defparameter *collected-relations* nil)

(defparameter *relations* nil)
(defparameter *roots* nil "Roots of current graphs.")
(defparameter *graph-stream* nil)
(defparameter *graph-depth* nil)
(defparameter *graph-border* nil)
(defparameter *reversed* nil)

(defun kb-overview (&key (relation 'subset) (root 'things) (stream kb-win) depth border (rtl t) (rl nil))
  (setq *relations* (if (consp relation) relation (list relation)))
  (setq *roots* (if (consp root) root (list root)))
  (setq *graph-stream* stream)
  (setq *graph-depth* depth)
  (setq *graph-border* border)
  (setq *reversed* rl)

  (collect-relations)
  (handle-nary-relations)
  (if *reversed*
      (setq *collected-relations*
	    (mapcar #'(lambda (rel) (list (second rel) (first rel)))
		    *collected-relations*)))
  (remove-loops *roots*)
  (if rtl (remove-trans-links))

  (send *graph-stream* :expose)
  (send *graph-stream* :clear-window)
  (terpri *graph-stream*) (terpri *graph-stream*)
  (format *graph-stream* "Showing relations: ")
  (dolist (relation *relations*) (format *graph-stream* " ~@(~s~)" relation))
  (format *graph-stream* ".~%")
  ;;(pp-labeled-list 0 "Showing relations:" *relations* nil)
  (if rtl
      (format *graph-stream* "(transitive links suppressed)~%~%")
      (format *graph-stream* ".~%~%"))

  (symbolics-graph *roots*))

(defun symbolics-graph (roots)
  (format-graph-from-root roots
			  #'(lambda (x stream)
			      (format stream "~@(~s~)" x))
			  #'children
			  :stream *graph-stream*
			  :orientation :horizontal
			  :direction (if *reversed* :before :after)
			  :root-is-sequence t
			  :cutoff-depth *graph-depth*
			  :border *graph-border*
			  :within-column-spacing 5
			  :column-spacing 25
			  :dont-draw-duplicates t))

(defun collect-relations ()
  (setq *collected-relations* nil)
  (flet ((my-push (item)
	   (push item *collected-relations*)))
    (dolist (relation *relations*)
      (with-no-output (dolist (f (all-frames))
			(a-query nil `((:retrieve (,relation (:quote ,f) ?x))
				       (:apply ,#'my-push ((,relation ,f ?x))))))))))

(defun children (f)
  (sort (mapcan #'(lambda (rel) (if (equal f (first rel)) (list (second rel))))
		*collected-relations*)
	#'alphalessp))

;; Quick and dirty function to remove most of the transitive links.
;; Should be replaced with a beter algorythm.
;;
(defun remove-trans-links ()
  (let ((hit-list nil))
    (dolist (rel1 *collected-relations*)
      (dolist (rel2 *collected-relations*)
	(if (eql (second rel1) (first rel2))
	    (push (list (first rel1) (second rel2))
		  hit-list))))
    (setq *collected-relations*
	  (set-difference *collected-relations* hit-list :test #'equal))))

(defun remove-loops (roots)
  (when (consp roots)
    (let* ((root (car roots))
	   (accessable-subtree (accessables (list root))))
      ;;(format t "~% root = ~a   subtree = ~a" root accessable-subtree)
      (when (find root accessable-subtree)	; When there is a loop.
	(setq *collected-relations*
	      (delete-if #'(lambda (rel)
			     (and (eql (cadr rel) root)
				  (find (car rel) accessable-subtree)))
			 *collected-relations*)))
      ;; Then search breadth first.
      (remove-loops (append (cdr roots) (children root))))))

(defun accessables (nodes)
  (let (new-nodes)
    (loop
      (setq new-nodes (mapcan #'children nodes))
      (if (subsetp new-nodes nodes) (return nodes))
      (setq nodes (union nodes new-nodes)))))

(defun handle-nary-relations ()
  (let ((counter 1))
    (setq *collected-relations*
          (mapcan #'(lambda (triple)
                      (let ((relation (first triple))
                            (arg1 (second triple))
                            (arg2 (third triple)))
                        (cond
                         ((consp arg2)
                          (setq relation (make-symbol (format nil "~a~a" relation (incf counter))))
                          (cons `(,arg1 ,relation)
                                (mapcar #'(lambda (x) `(,relation ,x))
                                        arg2)))
                         (t
                          (list (cdr triple))))))
                  *collected-relations*))
    t))


(defun graph-without-frame (frame)
  (setq *collected-relations*
        (remove-if #'(lambda (x) (find frame x)) *collected-relations*))
  (symbolics-graph *roots*))


;;; Graph-Path: Graphs the most recent access path (in a query or assertion).  One
;;; current limitations:
;;;
;;; 1. Special forms are ignored.
;;;

(defun graph-path (&key (stream kb-win) depth border)
  (setq *graph-stream* stream)
  (setq *graph-depth* depth)
  (setq *graph-border* border)

  (let ((algy-preds (preprocess *last-predicates*))
        (counter 0))
    (setq *roots* (list (frame (car algy-preds))))
    (setq *collected-relations* nil)
    
    (let* ((paths (mapcar #'(lambda (result)
                              (gp-substitute-bindings algy-preds (aresult-sub result)))
                          *last-results*))
           (big-path (list-union paths)))
      (dolist (pred big-path)
        ;; For the moment ignore special forms:
        (if (not (member (car pred) *special-forms*))
	    (let ((relation (intern (format nil "{~a~a}" (slot pred) (incf counter)))))
	      (push (list (frame pred) relation) *collected-relations*)
	      (let* ((pred-value (value pred))
		     (value-list (if (consp pred-value) pred-value (list pred-value))))
		(dolist (term value-list)
		  (push (list relation term) *collected-relations*)))))))

    (remove-loops *roots*)
    
    (send *graph-stream* :expose)
    (send *graph-stream* :clear-window)
    (symbolics-graph *roots*)))

(defun gp-substitute-bindings (exp alist)
  (cond ((consp exp)
	 (cons (gp-substitute-bindings (car exp) alist)
	       (gp-substitute-bindings (cdr exp) alist)))
	((variable? exp)
	 (let ((b (assoc (algy-variable-name exp) alist :key #'algy-variable-name)))
	   (if b
	       (cadr b)
	       exp)))
	(t exp)))

(defun list-union (list)
  (if (consp list)
    (union (car list) (list-union (cdr list)) :test #'equal)))

 
;; Then the code to collect the sets in the kb and show their members:


(defun show-sets ()
  (setq *relations* '(isa))
  (collect-relations)
  (let ((sets nil))
    (dolist (rel *collected-relations*)
      (if (not (end-search (string 'selfset) (string (second rel))))
	  (pushnew (second rel) sets)))
    ;; Explicitly take things out:
    (setq sets (delete 'things sets))
    (dolist (set (sort sets #'alphalessp))
      (pp-labeled-list 2
		       (format nil "~@(~a~):" set)
		       (sort (mapcan #'(lambda (rel)
					 (if (eql set (second rel))
					     (list (first rel))))
				     *collected-relations*)
			     #'alphalessp)
		       nil))))


(defun end-search (x y)
  (if (not (< (length y) (length x)))
      (string-equal x y :start2 (- (length y) (length x)))))
