;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: aprep.lisp,v 1.1 92/04/16 09:30:19 clancy Exp $

;;;                        ****** APREP ******

; Preprocessor for Algernon.
;
; Modified 10/11/90 so that :neq handled in alogic (for efficiency).
;
; Modified 4/20/91 so that variable replacement not done within :quote.
; It should now be the case that no preprocessing done within :quote.

; Modified 1/30/92 to treat quote exactly like :quote

(defparameter *var-num* 0 "Used to number new unique vars.")
(defparameter *pre-proc-forms* 
              '(:1-SLOTS
		:SLOTS
		:TAXONOMY
		:SLOT
		:ALL
		:EXISTS))
(defparameter *pre-proc-funcs*
              '(:quote 			; (:quote exp) = exp
		quote				; Used to say (name John (:quote (john))).
		:slot				; (:slot s) = the slot s.
                :lambda                         ; (:lambda ?x path).
		))

(defparameter *default-partition* 'main-partition)
(defparameter *orphan-warn* nil
  "When true warnings are printed out on variables used only once in rules.")
(defparameter *conseq-vars-warn* t
  "When true warnings are printed on variables used only in the consequents of rules.")
(defparameter *description-vars-warn* t
  "When true warnings are printed on variables appearing
   unbound in definite and indefinite descriptions.")

(defvar *retrieve* nil "When true all predicates are surounded by :retreive")

; RESET-PREPROC
; Resets the pre-processor.  This amounts to resetting the
; variable *var-num*.
(defun reset-preproc ()
  (setq *var-num* 0))


; PREPROCESS
;
; Takes a list of predicates and pre-processes it.
; Then does a small amount of error detection
; (could also do some optimization).
;
(defun preprocess (path &optional orphan-warn)
  (setq *orphan-warn* orphan-warn)
  (let ((new-path (prep-path path)))
    (setq *last-var-alist* (prep-vars new-path))
    new-path))


; PREP-PATH
; Takes a list of predicates and pre-processes it.
; Returns a list of predicates.
;
;   path = a path
;   arbs = assoc list of current arbitrary objects
;   prev-vars = a list of variables occuring in the current path before this rule
;               (used to detect orphan variables)
;
(defun prep-path (path &optional arbs prev-vars)
  (let ((new-path nil))
    (dolist (pred path)
      (setq new-path (nconc new-path (prep-pred pred arbs prev-vars)))
      (setq prev-vars (variables pred prev-vars)))
    new-path))


; PREP-PRED
; Takes a single predicate and pre-processes it.
; Returns a list of predicates.
;
;   p = a predicate
;   arbs = assoc list of current arbitrary objects
;   prev-vars = a list of variables occuring in the current path before this rule
;               (used to detect orphan variables)
;
(defun prep-pred (p arbs prev-vars)
  (cond ((not (listp p))
	 (list p))
	((member (car p) *special-forms* :test #'eq)
	 (prep-sp-form p arbs prev-vars))
	((member (car p) *pre-proc-forms* :test #'eq)
	 (prep-pp-form p arbs prev-vars))
	(t
	 (prep-normal-pred p arbs))))


; PREP-SP-FORMS
; Takes a single predicate which is a special form and
; pre-processes it.  Returns a list of Algernon predicates.
;
;   p = a special form
;   arbs = assoc list of current arbitrary objects
;
; There are four basic forms for special-forms:
;
;  1. Path From -- Those whose args form a path.
;  2. Term Form -- Those whose first arg is a term.
;  3. Pred Form -- Those whose only arg is a predicate.
;  4. Other     -- Those which are not affected by the pre-processor.
;
(defun prep-sp-form (p arbs prev-vars)
  (case (car p)
    ;;
    ;; Path form forms.
    ((:UNP :ANY)
     (list (cons (car p) (prep-path (cdr p) arbs prev-vars))))

    ((:THE :A :FORC)			; (second p) is a variable or list or variables.
     (description-var-check p prev-vars)
     (list (cons (car p) (cons (make-into-list (second p))

                               (prep-path (cddr p) arbs prev-vars)))))
    ;;
    ;; :ALL-PATHS fits in this class -- sort of.  Its not clear to me what I
    ;; should do with arbs and prev-vars here so I just pass them on unchanged ...
    ;; (perphaps vars in (second p) should be added to prev-vars in (third p) ?).
    (:ALL-PATHS
     (list (list (first p)
                 (prep-path (second p) arbs prev-vars)
                 (prep-path (third p) arbs prev-vars))))
    ;;
    ((:WO-CONTRA-POSITIVE
      :W-CONTRA-POSITIVE)
     (list (cons (car p) (prep-path (cdr p) arbs prev-vars))))
    ;;
    ;; Term form forms.
    ((:RULES
      :SRULES
      :DEL-RULE
      :DEL-SRULE)
     (prep-t-form (cons (first p) (cons (second p)
                                        (prep-rules (cddr p) arbs prev-vars)))
                  arbs))
    ;;
    ((:DEL-RULES
      :CLEAR-SLOT
      :SHOW)
     (prep-t-form p arbs))
    ;;
    ;; :bind-to-values has a strange syntax: (:bind-to-values ?x frame slot).
    ;; Would be nice if it were more like other forms ...
    (:BIND-TO-VALUES
     (list (cons :BIND-TO-VALUES (car (prep-t-form (cdr p) arbs)))))
    ;
    ; Pred form forms.
    ((:NO-COMPLETION
      :ASSUME
      :ASK
      :DELETE
      :IN-OWN-PARTITION)
     (prep-pred-form p arbs prev-vars))
    (:RETRIEVE
     (let ((*retrieve* t))
       (prep-pred-form p arbs prev-vars)))
    ;
    (t
     ;; The following forms do not change at all:
     ;; :CREATE
     ;; :DECL-SLOTS
     ;; :DEL-SRULES
     ;; :LISP
     ;; :TEST
     ;; :APPLY
     ;; :BIND
     ;; :NEQ
     ;; :BRANCH-ON-VALUES.
     (list (copy-tree p)))))

(defun make-into-list (l)
  (if (consp l) l (list l)))

; PREP-T-FORM
;   Pre-processes special forms which are in term form.
;   Expands the first argument and ignores the rest of the form.
;
(defun prep-t-form (p arbs)
  (let ((term (second p)))
    (let ((out-var (new-term term)))
      (append (expand term out-var arbs)
	      (list (cons (first p) (cons (strip-new-term-error out-var)
					  (copy-tree (cddr p)))))))))

; PREP-PRED-FORM
;   Pre-processes special forms which are in pred form.
;
(defun prep-pred-form (p arbs prev-vars)
  (let ((pred (prep-pred (second p) arbs prev-vars)))
    (append (butlast pred) (list (list (car p) (car (last pred)))))))

; PREP-RULES
; Takes a list of rules and pre-processes it.
; Hack -- A single rule is treated as a single list of predicates.
;         This works because (prep-pred '->) = '(->)
;
;   rlist = list of rules
;   arbs = assoc list of current arbitrary objects
;   prev-vars = a list of variables occuring in the current path before this rule
;               (used to detect orphan variables).
;
(defun prep-rules (rlist arbs prev-vars)
  (mapcar #'(lambda (plist)
	      (cond ((member (car plist) '(:eval :lisp))
		     (copy-tree plist))
		    (t
		     (if *orphan-warn*
			 (let ((orphans (find-orphans plist prev-vars)))
			   (if orphans
			       (algy-warning
				 (format nil
					 "(Preprocessor) The variable~P ~(~a~) ~
                                          appear~P only once in the rule:"
					 (length orphans) orphans (plural-invert (length orphans)))
				 plist))))
		     (if *conseq-vars-warn*
			 (let ((c-vars (nset-difference
					 (variables (consequent plist) nil)
					 (append (variables (antecedent plist) nil)
						 prev-vars
						 (described-vars (consequent plist))))))
			   (if c-vars
			       (algy-warning
				 (format nil
					 "(Preprocessor) The variable~P ~(~a~) ~
                                         appear~P only in the consequent of the rule:"
					 (length c-vars) c-vars (plural-invert (length c-vars)))
				 plist))))
		     (prep-path plist arbs prev-vars))))
	  rlist))

(defun plural-invert (x)
  (if (eql x 1)
      2
      1))

; PREP-PP-FORMS
; Takes a single predicate which is a pre-processor form and
; pre-processes it.  Returns a list of Algernon predicates.
;
;   p = a predicate
;   arbs = assoc list of current arbitrary objects
;
(defun prep-pp-form (p arbs prev-vars)
  (case (car p)
    ;
    (:1-SLOTS
     (let ((slots (cdr p))
           (partition (find-frame-with-name (list *default-partition*))))
       (append (list (cons ':decl-slots (mapcar #'(lambda (s) (list s 1)) slots)))
               (mapcar #'(lambda (s) (list 'slot-partition s partition)) slots))))
    ;
    (:SLOTS
     (let ((slots (cdr p))
           (partition (find-frame-with-name (list *default-partition*))))
       (append (list (cons ':decl-slots (mapcar #'(lambda (s) (list s nil)) slots)))
               (mapcar #'(lambda (s) (list 'slot-partition s partition)) slots))))
    ;
    (:TAXONOMY
     (prep-path (make-taxa (second p))))
    ;
    (:SLOT
     (make-slot (second p) (third p) (cdddr p) arbs prev-vars))
    ;
    ))


; PREP-NORMAL-PRED
; Pre-processes a 'normal' pred (i.e. not a special form or a pre-processor form).
; Returns a list of predicates.
;
;   p = a predicate
;   arbs = assoc list of current arbitrary objects
;
(defun prep-normal-pred (p arbs)
  (if (eql (car p) 'not)
      (let ((new-path (prep-normal-pred (cadr p) arbs)))
	(append (butlast new-path)
		(list (cons 'not (last new-path)))))
      (let ((app nil)                ; path to be appended to p
	    (out-var nil)
	    (pred (list (car p))))   ; new version of p (with functions replaced with new vars).
	(dolist (arg (cdr p))
	  (setq out-var (new-term arg))	; Replacement for arg or cons pair on error.
	  (setq app (append (expand arg out-var arbs) app))
	  (setq pred (append pred (list (strip-new-term-error out-var)))))
	(mapcar #'handle-n-place-preds
		(append app (list pred))))))

; NEW-TERM -- Generates replacement for term.  On error returns a cons pair
; according to error detected.
;
(defun new-term (term)
  (cond ((not (consp term))
	 (if (or (input-var? term) (null term))
	     term
	     (find-frame-with-name (list term))))
	((member (car term) *pre-proc-funcs* :test #'eq)
	 (case (car term)
	   (:slot (second term))	; Perhaps should check that term is a slot ?
	   ((:quote quote) term)	; Do nothing -- prep-vars will remove :quote.
           (:lambda (copy-tree term))))
	(t
	 (gen-new-input-var))))

; STRIP-NEW-TERM-ERROR -- A hack to remove the error tags from the
; result of a call to new-term.
;
(defun strip-new-term-error (x)
  (if (and (consp x) (eql (cdr x) '$no-name))
      (car x)
      x))

(defun strip-new-term-errors (list)
  (mapcar #'strip-new-term-error list))

; Find-Frame-With-Name -- Generates frame with name or
; returns the pair (<new-input-variable> $no-name).
;
; Hacked 1/30/90 to not affect numbers. 2/2/90 does not affect strings either.
;
(defun find-frame-with-name (name)
  (if (or (numberp (car name))
	  (stringp (car name))
	  (variable? (car name)))
      (car name)
      (let ((objects (filter-out-results (objects-from-name name))))
	(cond ((or (> (length objects) 1)
		   (< (length objects) 1))
	       (algy-warning (format nil "(Preprocessor) No frame known with name ~(~a~)." name))
	       (cons (gen-new-input-var) '$no-name))
	      (t
	       (car objects))))))

(defun filter-out-results (pairs)
  (mapcan #'(lambda (pair) (if (member nil (aresult-assump-ll (cdr pair)) :test #'eq)
                             (list (car pair))))
	  pairs))

; EXPAND -- Expands arb by calling NORM-EXPAND or ARB-EXPAND.
;
;  arb = a term of form (f1 (f2 ... (fn ?vn)))
;  out-var = a variable
;  arbs = assoc list of current arbitrary objects
;
;  Calls arb-expand iff ?vn in arbs.
;
(defun expand (arg out-var arbs)
  (let* ((v-pair (assoc (find-var arg) arbs)))
    (if v-pair
	(arb-expand arg (cadr v-pair) out-var)
	(norm-expand arg out-var))))


; NORM-EXPAND
; arg = (f1 (f2 ... (fn x)))
; (where x is a variable or preprocessor function)
; Returns a path of form:
;   ((fn x ?vn-1) (fn-1 ?vn-1 ?vn-2) ... (f1 ?v1 out-var))
;
; Two special cases:
;
;  1. If out-var is a cons pair this signals an "error" (detected by new-term).
;  Currently the only case of this is an unknown name which is signaled by the
;  pair (<new-variable> . $no-name).  In this case, norm-expand returns:
;  (name <new-variable> arg).
;
; 2. arg is nil or is just a special form.  In this case the output is just nil.
;
; Modified 9/27/90 to allow functions or arbitrary arity.
; Modified 5/16/91 to enclose returned preds in :retrieve when *retrieve* true.
;
; *** Hack -- This routine should not have to call handle-n-place-preds
; when it adds :retrieve ... ***
;
(defun norm-expand (arg out-var)
  (if *retrieve*
      (mapcar #'(lambda (pred) `(:retrieve ,(handle-n-place-preds pred)))
	      (norm-expand-internal arg out-var))
      (norm-expand-internal arg out-var)))

(defun norm-expand-internal (arg out-var)
  (cond
   ;; Special cases:
   ((and (consp out-var) (eql (cdr out-var) '$no-name))
    `((name ,(car out-var) (,arg))))
   ((or (not (consp arg))
        (member (car arg) *pre-proc-funcs* :test #'eq))
    nil)
   
   ;; Recurse:
   (t
    ;; First we generate the version of arg with new variables where needed: subst-arg:
    (let ((subst-arg (mapcar #'(lambda (x)
                                 (if (and (consp x)
                                          (not (member (car x) *pre-proc-funcs* :test #'eq)))
                                   (gen-new-input-var)
                                   (new-term x)))
                             (cdr arg))))
      ;; Then just return necessary expansions and new version of arg:
      (append (mapcan #'(lambda (x subst-x)
                          (norm-expand-internal x subst-x))
                      (cdr arg)
                      subst-arg)
              (list (append (list (car arg))
                            (strip-new-term-errors subst-arg)
                            (list out-var))))))))

; ARB-EXPAND --- OUTDATED
;
; arg = (f1 (f2 ... (fn ?obj0)))
; in-var =  <a variable name> Arb frame passed in which new arbs depend on.
; out-var = <a variable name> Obj frame which is bound by the created path.
;
; Returns a path of form:
;   ((type-res fn ?sn) (:CREATE ?arbn) (arb-member ?sn ?arbn) (depends-via ?arbn (fn in-var))
;    (type-res fn-1 ?sn-1) (:CREATE ?arbn-1) (arb-member ?sn-1 ?arbn-1) (depends-via ?arbn-1 (fn-1 ?arbn))
;    ...
;    (type-res f1 ?s1) (:CREATE out-var) (arb-member ?s1 out-var) (depends-via ?arb1 (f1 ?arb2))
;                                                                                        (obj-frame ?arb1 out-var))
;
(defun arb-expand (arg in-var out-var)
  (let ((arb-frame (gen-new-input-var)))
    (append (r-arb-expand arg in-var arb-frame)
	    (list `(obj-frame ,arb-frame ,out-var)))))

; R-ARB-EXPAND
;
; arg = (f1 (f2 ... (fn ?obj0)))
; in-var =  <a variable name> Arb frame passed in which new arbs depend on.
; out-var = <a variable name> Arb frame which is bound by the created path.
;
; Returns a path of form:
;   ((type-res fn ?sn) (:CREATE ?arbn) (arb-member ?sn ?arbn) (depends-via ?arbn (fn in-var))
;    (type-res fn-1 ?sn-1) (:CREATE ?arbn-1) (arb-member ?sn-1 ?arbn-1) (depends-via ?arbn-1 (fn-1 ?arbn))
;    ...
;    (type-res f1 ?s1) (:CREATE out-var) (arb-member ?s1 out-var) (depends-via out-var (f1 ?arb2)))
;
(defun r-arb-expand (arg in-var out-var)
  (cond ((not (consp arg))
	 nil)
	(t
	 (let ((new-set (gen-new-input-var))
	       (new-arb (if (listp (cadr arg)) (gen-new-input-var) in-var)))
	   (append (r-arb-expand (cadr arg) in-var new-arb)
		   `((type-res ,(car arg) ,new-set)
		     (:CREATE ,out-var) (arb-member ,new-set ,out-var)
		     (depends-via ,out-var (,(car arg) ,new-arb))))))))


; HANDLE-N-PLACE-PREDS
; Algernon internally represents n-place preds
; by:
;    (f t1 ... tn) = (f t1 (t2 ... tn))
;
; Unless n=2 in which case:
;
;    (f t1 t2) = (f t1 t2)
;
; This is wierd, but it is necessary because an n-place pred is
; representated in the K-base by putting an n-1 tuple in the
; appropriate frame slot.

(defun handle-n-place-preds (pred)
  (if (> (length pred) 3)
      (list (first pred) (second pred) (cddr pred))
      pred))

(defun find-var (arg)
  (if (and
	(consp arg)
	(consp (cdr arg)))
      (find-var (cadr arg))
      arg))
  

(defun gen-new-input-var ()
  (make-name '?$x (prin1-to-string (setq *var-num* (+ *var-num* 1)))))



; Code to detect unusual variable usage.

; Find-Orphans
;
; Returns a list of the variables occuring only once in path.
; prev-vars is a list of variables which have occured before this
; path and so can 'legally' occur only once.
;
(defun find-orphans (path prev-vars)
  (mapcan #'(lambda (pair) (if (and (= (cdr pair) 1) (not (find (car pair) prev-vars)))
			       (list (car pair))))
	  (variable-usage path nil)))

; Description-Var-Check
;
(defun description-var-check (p prev-vars)
  (if *description-vars-warn*
      (let ((unbound-vars (nset-difference (nset-difference (variables (third p) nil)
							    (variables (second p) nil))
					   prev-vars)))
	(if unbound-vars
	    (algy-warning
	      (format nil
		      "(Preprocessor) The variable~P ~(~a~) is unbound in the description:"
		      (length unbound-vars) unbound-vars)
	      (list p))))))

; Variable-Usage
;
; Returns an alist of variables in exp and the number of times
; they appear in exp.
;
(defun variable-usage (exp alist)
  (cond ((input-var? exp)
	 (let ((entry (assoc exp alist)))
	   (cond (entry
		  (incf (cdr entry)) alist)
		 (T (acons exp 1 alist)))))
	((consp exp)
	 (variable-usage (car exp) (variable-usage (cdr exp) alist)))
	(T alist)))

;;; Modified (hacked) 4/19/90  to not count variables inside a form (:lambda ...).
;;; 4/20/91 -- does not count inside :quote or quote either.
;;;
(defun variables (exp prev-vars)
  (cond ((input-var? exp)
         (pushnew exp prev-vars))
        ((consp exp)
         (cond ((member (car exp) '(:quote quote :lambda) :test #'eql)
                prev-vars)
               (t
                (setq prev-vars (variables (car exp) prev-vars))
                (variables (cdr exp) prev-vars))))
        (t prev-vars)))

;;; Described-Vars -- Given an exp which is a list of predicates, described-vars
;;; finds the variables appearing in forms (:the ?x ...) or (:forc ?x ...).
;;;
(defun described-vars (exp)
  (mapcan #'(lambda (pred)
	      (if (and (consp pred)
		       (member (car pred) '(:the :forc :a))
		       (input-var? (second pred)))
		  (list (second pred))))
	  exp))


; Code for taxonomy shorthand.

;;; Make-Taxa
;;;
;;; tree is of form:
;;;   (set (subset1 member1 member2 ...)
;;;        (subset2 member1 ...)
;;;        member1 ...)
;;;
;;; Returns path of form:
;;;
;;;   ((:the ?x1 (name ?x1 (:quote (subset1))))
;;;    (imp-superset ?x1 set)
;;;    (:the ?x2 (name ?x2 (:quote (member1))))
;;;    (member ?x1 ?x2)
;;;    ...
;;;   )
;;;
;;; Modified 2/13/90 to expand to imp-superset.
;;; Changed :forc to :the, enforcing uniqueness.  (2-6-92)

(defun make-taxa (tree)
  (mapcan #'(lambda (x)
	      (if (atom x)
		  (taxa-member x (car tree))
		  (taxa-superset x (car tree))))
	  (cdr tree)))

(defun taxa-member (new-member set)
  (let ((var (gen-new-input-var)))
    `((:the ,var (name ,var (:quote (,new-member))))
      (member ,set ,var))))

(defun taxa-superset (new-set superset)
  (let ((new-set-name (car new-set))
	(new-set-description (cdr new-set))
	(var (gen-new-input-var)))
    (append `((:the ,var (name ,var (:quote (,new-set-name))))
	      (imp-superset ,var ,superset))
	    (make-taxa (cons var new-set-description)))))


; Code for :slot shorthand.

(defun make-slot (name type-res description arbs prev-vars)
  (let ((partition (or (second (member :partition description :test #'eq))
                       *default-partition*))
        (num-res (second (member :cardinality description :test #'eq))))
    (list (cons
           :WO-CONTRA-POSITIVE
           (append `((:decl-slots (,name ,num-res)))
                   (prep-pred `(slot-partition (:slot ,name) ,partition) arbs prev-vars)
                   (if type-res (prep-pred `(type-slot (:slot ,name) ,@type-res) arbs prev-vars))
                   (prep-description name description))))))

(defun prep-description (name description)
  (if (null description)
      nil
      (append
	(case (car description)
	  (:cardinality nil)
	  (:partition nil)
	  (:backlink `((backlink ,name ,(second description))))
	  (:inverse `((inverse ,name ,(second description))))
	  (:comment `((comment ,name ,(second description))))	   
	  (t
	    (throw 'error (format nil "(Preprocessor) Illegal slot descriptor: ~(~a~)."
				  (list (first description) (second description))))))
	(prep-description name (cddr description)))))



; PREP-VARS takes a list l, and distructively replaces all variables
; (symbols beginning with ?) with structures. prep-vars is carefull to
; replace multiple occurances of variables with the same structure.
;
; Modified 4/19/90 to special case :lambda (sigh) and not replace vars in (:lambda ?x path).
;
; Modified 4/25/90 to take arg no-new-vars, and not replace new vars when it is true
; (this is currently done only within a :lambda).
;
; BUG: Variables should be scoped only with-in a path, but this is not done.
;
; Modified 3/6/91 to also prep variables in aresult-sub's.
;
; Modified 4/20/91 to not prep variables within :quote.
;
(defun prep-vars (l &optional alist no-new-vars in-lisp-context)
  (case (car l)
    (:lambda
	(setq no-new-vars t))
    ((:lisp :eval :test :funcall :bind :branch :branch-on-values :bind-to-value)
     (setq in-lisp-context t)))

  (cond ((consp (car l))
	 (if (and (not in-lisp-context)
		  (member (first (car l)) '(quote :quote) :test #'eql))
	     (rplaca l (second (car l)))   ; Just strip off :quote.
	     (setq alist (prep-vars (car l) alist no-new-vars in-lisp-context))))
	((input-var? (car l))
	 (let ((l-struct (cdr (assoc (car l) alist))))
	   (unless (or no-new-vars l-struct)
	     (setq l-struct (new-variable (car l)))
	     (setq alist (acons (car l) l-struct alist)))
	   (if l-struct (rplaca l l-struct))))
	((aresult-p (car l))
	 (setq alist (prep-vars (aresult-sub (car l)) alist no-new-vars in-lisp-context))))

  (cond ((consp (cdr l))
	 (if (and (not in-lisp-context)
		  (member (first (cdr l)) '(quote :quote) :test #'eql))
	     (rplacd l (second (cdr l)))   ; Just strip off :quote.
	     (setq alist (prep-vars (cdr l) alist no-new-vars in-lisp-context))))
	((input-var? (cdr l))
	 (let ((l-struct (cdr (assoc (cdr l) alist))))
	   (unless (or no-new-vars l-struct)
	     (setq l-struct (new-variable (cdr l)))
	     (setq alist (acons (cdr l) l-struct alist)))
	   (if l-struct (rplacd l l-struct))))
	((aresult-p (cdr l))
	 (setq alist (prep-vars (aresult-sub (cdr l)) alist no-new-vars in-lisp-context))))
	
  alist)

(defun input-var? (x)
  (and (symbolp x)
       (char= (schar (string x) 0) #\?)))