;;; -*- Mode:Common-Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: app.lisp,v 1.1 92/04/16 09:30:17 clancy Exp $

;;;                        ****** APP ******

; Algernon utilities to make readable output.

; Assumes facets can be compared using eq.

; Vars to define what sort of output to print:
(defparameter *show-sentence* t)
(defparameter *show-input* nil)
(defparameter *show-predicates* nil)
(defparameter *show-bindings* t)
(defparameter *show-insertions* nil)
(defparameter *show-other-kb-changes* t)
(defparameter *show-stats* t)
(defparameter *show-failures* t "Prints message when an operation fails")
(defparameter *cerror-on-failure* nil "Continuable error on failure.")

(defparameter *data-column* 23)
(defparameter *data-column2* (+ 7 *data-column*))

; Routines to set output mode.
(defun normal-output ()
  (setq *show-sentence* t)
  (setq *show-input* nil)
  (setq *show-predicates* nil)
  (setq *show-bindings* t)
  (setq *show-insertions* nil)
  (setq *show-other-kb-changes* t)
  (setq *show-stats* t)
  (setq *show-failures* t)
  (setq *cerror-on-failure* nil))

(defun minimal-output ()
  (setq *show-sentence* t)
  (setq *show-input* nil)
  (setq *show-predicates* nil)
  (setq *show-bindings* nil)
  (setq *show-insertions* nil)
  (setq *show-other-kb-changes* nil)
  (setq *show-stats* nil)
  (setq *show-failures* t)
  (setq *cerror-on-failure* nil))

(defun silent-output ()
  (setq *show-sentence* nil)
  (setq *show-input* nil)
  (setq *show-predicates* nil)
  (setq *show-bindings* nil)
  (setq *show-insertions* nil)
  (setq *show-other-kb-changes* nil)
  (setq *show-stats* nil)
  (setq *show-failures* nil)
  (setq *cerror-on-failure* nil))

(defun verbose-output ()
  (setq *show-sentence* t)
  (setq *show-input* t)
  (setq *show-predicates* t)
  (setq *show-bindings* t)
  (setq *show-insertions* t)
  (setq *show-other-kb-changes* t)
  (setq *show-stats* t)
  (setq *show-failures* t)
  (setq *cerror-on-failure* nil))

; And a macro to allow all output to be temporarily suppressed:
;
; with-no-output --- code appears in aglobals.


; Pretty Print the sentence input to Algernon.
;
(defun pp-sentence (op sentence)
  (if (and sentence *show-sentence*)
      (format t "~&~%~%~aING:  ~11T~a~%" op sentence)))

(defun pp-input (predicates &optional show-input)
  (if (or show-input *show-input*)
      (pp-labeled-list 0 (format nil "~&~% Input pred~P:" (length predicates))
		       predicates nil)))

; Pretty print the results of an operation.
;
(defun pp-output (&optional show-predicates show-bindings show-insertions show-other-kb-changes show-stats)
  (if (or *show-predicates* *show-bindings* *show-insertions* *show-other-kb-changes* *show-stats*
	  show-predicates show-bindings show-insertions show-other-kb-changes show-stats)
    (format t "~&~%"))
  (if (and (null *last-results*) *show-failures*)
    (format t "~% *~@(~a~) failed.*~%" *last-op*)
    (if (or show-bindings *show-bindings* show-predicates *show-predicates*)
      (if (or (not (eql (length *last-results*) 1))          ; If more than one result OR
              (aresult-sub (car *last-results*))             ; result has bindings OR
              (not (equal                                    ; result has assumptions.
                    (aresult-assump-ll (car *last-results*))
                    '(nil))))
        (pp-results *last-results* show-bindings show-predicates)
        (format t "~% ~@(~a~) succeeded.~%" *last-op*))))
  (if (and (or show-insertions *show-insertions*)
	   *last-inserted-values*)
      (pp-labeled-list 0 (format nil "Inserted value~P:" (length *last-inserted-values*))
		       *last-inserted-values* t))
  (if (and (or show-insertions *show-insertions*)
	   *last-inserted-assumptions*)
      (pp-labeled-list 0 (format nil "Inserted assump~P:" (length *last-inserted-assumptions*))
		       *last-inserted-assumptions* t))
  (if (and (or show-other-kb-changes *show-other-kb-changes*)
	   *last-deleted-values*)
      (pp-labeled-list 0 (format nil "Deleted value~P:" (length *last-deleted-values*))
		       *last-deleted-values* t))
  (if (and (or show-other-kb-changes *show-other-kb-changes*)
	   *last-creations*)
      (pp-labeled-list 0 (format nil "Created frame~P:" (length *last-creations*))
		       *last-creations* nil))
  (if (and (or show-other-kb-changes *show-other-kb-changes*)
	   *last-contradictions*)
      (pp-labeled-list 0 (format nil "Found contradiction~P:" (length *last-contradictions*))
		       *last-contradictions* t))
  (if (or *show-stats* show-stats)
      (prog ()
	(format t "~&~% Insertions: ~a ~25TRule applications: ~a ~50TIterations: ~a."
	          (+ (length *last-inserted-values*) (length *last-inserted-assumptions*))
		  *rule-count* *max-iterations*)
	(format t "~% Unifications: ~a ~25TMatches: ~a."
	          *unify-count* *match-count*)
	(format t "~% Frame insertions: ~a ~25TFrame accesses: ~a"
		  *frame-insertions* *frame-accesses*))))

; Pretty print the results of an operation.
;
(defun pp-results (results show-bindings show-predicates)
  (cond ((eql (length results) 1)
	 (format t "~% Result:")
	 (pp-single-result (car results) show-bindings show-predicates))
	(t
	 (let ((result-no 1))
	   (dolist (result results)
	     (format t "~% Result ~a:" result-no)
	     (pp-single-result result show-bindings show-predicates)
	     (setq result-no (+ 1 result-no)))))))

; Pretty print a result.
;
(defun pp-single-result (result show-bindings show-predicates)
  (if (or show-predicates *show-predicates*)
      (pp-labeled-list 2 (format nil "Predicate~P:" (length *last-predicates*))
		       (substitute-bindings *last-predicates* (aresult-sub result)) nil))
  (if (and (or show-bindings *show-bindings*) (aresult-sub result))
      (prog ()
	    (format t "~%   Binding~P: ~VT" (length (aresult-sub result)) *data-column*)
	    (pp-bindings (aresult-sub result))))
  (if (not (equal (aresult-assump-ll result) '(nil)))	; If there are any assumptions.
      (pp-labeled-list 2 (format nil "Assumption~P:" (length (aresult-assump-ll result)))
		       (aresult-assump-ll result) t)))

; Print out a list of variable bindings.
;
(defun pp-bindings (sub)
  (dolist (binding sub)
    (format t " ~(~a~) ~VT --- " (car binding) *data-column2*)
    (pp-list *data-column2* 80 *data-column2* nil (add-names (cdr binding)))
    (format t "~&~VT" *data-column*)))

; Pretty print a frame.
;
(defun PP-FRAME (object)
  (cond ((atom object)
	 (format t "~&~%~@(~a~): " object)
	 (if (slotp object) (format t "[slot]~%"))
	 (or (mapc #'(lambda (slot)
		       (pp-slot slot))
		   (cdr (get object 'frame)))
	     (do ((L (symbol-plist object) (cddr L)))
		 ((null L))
	       (format t "~%  ~@(~a~): ~14t~(~a~)" (car L) (cadr L)))))
	(t (format t "~%~@(~a~)" object)))
  (format t "~&~%")
  t)

(defun pp-slot (slot)
  (let* ((out-string (format nil "~%  ~(~a~):~14T" (car slot)))
	 (min-col (- (length out-string) 1)))
    (format t out-string)
    (let ((values (cdr (assoc @value (cdr slot) :test #'eq)))
	  (non-values (cdr (assoc @n-value (cdr slot) :test #'eq)))
	  (num (cdr (assoc @num-res (cdr slot) :test #'eq)))
	  (rules-list1 (mapcar #'(lambda (facet) (cdr (assoc facet (cdr slot) :test #'eq)))
			       (list @if-needed @if-added @n-if-needed @n-if-added)))
	  (rules-list2 (mapcar #'(lambda (facet) (cdr (assoc facet (cdr slot) :test #'eq)))
			       (list @sif-needed @sif-added @sn-if-needed @sn-if-added))))
      (if (consp values)
	  (pp-list min-col 80 min-col nil values))
      (when (consp non-values)
	(format t "~%     (non-value~P) " (length non-values))
	(pp-list min-col 80 min-col nil non-values))
      (if (consp num)
	  (format t "~%     (number restriction) ~a" num))
      (dolist (rules rules-list1)
	(dolist (rule-pair rules)
	  (format t "~&~VT" min-col)
	  (pp-rule min-col rule-pair)))
      (when (some #'consp rules-list2)
	(format t "~&       (Slot rules)")
	(dolist (rules rules-list2)
	  (dolist (rule-pair rules)
	    (format t "~&~VT" min-col)
	    (pp-rule min-col rule-pair)))))))

(defun add-names (frame-list)
  (if (consp frame-list)
      (let ((f (car frame-list)))
	(if (and (symbolp f) (framep f))
	    (append (list f (names f))
		    (add-names (cdr frame-list)))
	    (cons (add-names f) (add-names (cdr frame-list)))))
      frame-list))

; This could be improved to list all names separated by commas.
;
(defun names (f)
  (format nil "[~(~a~)]" (list-to-string (if (slotp 'name) (caar (fget f 'name @value))))))

(defun list-to-string (l)
  (if (consp l)
      (let ((rest-of-name (list-to-string (cdr l))))
	(if (not (null rest-of-name))
	    (format nil "~@(~a~) ~a" (car l) (list-to-string (cdr l)))
	    (format nil "~@(~a~)" (car l))))))

; PP-RULES prints out all rules for the slot 'slot' of frame 'frame'.
;
; Probably should apply substitutions in cdr of pairs returned by get-rules.
;
(defun PP-RULES (frame slot)
  (let* ((pred (list slot frame (new-variable '?x)))
	 (neg-pred (list 'not pred))
	 (if-needed (get-rules pred @if-needed))
	 (if-added (get-rules pred @if-added t))
	 (nonv-if-needed (get-rules neg-pred @n-if-needed))
	 (nonv-if-added (get-rules neg-pred @n-if-added t)))
    (if if-needed (pp-rule-list 'if-needed if-needed pred))
    (if if-added (pp-rule-list 'if-added if-added pred))
    (if nonv-if-needed (pp-rule-list 'nonv-if-needed nonv-if-needed neg-pred))
    (if nonv-if-added (pp-rule-list 'nonv-if-added nonv-if-added neg-pred))))

; Pretty print a list of rules.
;
(defun pp-rule-list (rule-type rule-list pred)
  (cond (rule-list
         (format t "~&~%  ~@(~a~) rules for ~(~a~):~%~%" rule-type pred)
         (dolist (pair rule-list)  ; Each 'pair' is a cons pair (rule . result).
           ;; Don't print rule if its assumptions are currently false:
           (unless (null (aresult-assump-ll (cdr pair)))
             (format t "~&~5T")
             (pp-rule 5 pair))))))

; Low level details for making a rule pretty.
;
(defun PP-RULE (min-col rule-pair)
  (let* ((rule (substitute-bindings (car rule-pair) (aresult-sub (cdr rule-pair))))
         (result (cdr rule-pair))
         (arrow (if (member '<- rule :test #'eq) '<- '->))
         (rhs (cdr (member arrow rule :test #'eq)))
         (lhs (ldiff rule rhs))   ; lhs includes arrow
         (col (pp-list min-col 80 min-col nil lhs)))
    (cond ((or (< col 40) (< (length rhs) 3))           ; heuristic for when to put rhs on same line.
           (format t " ")
           (setq col (+ 1 col))
           (pp-list col 80 col nil rhs))
          (t
           (format t "~%~VT" (+ min-col 2))
           (pp-list (+ min-col 2) 80 (+ min-col 2) nil rhs)))
    (pp-result min-col 80 min-col result t)))

; PP-UP
; Print results of partition update.
;
(defun pp-up (&optional show-insertions)
  (format t "~% Partition update complete.~%")
  (when (and (or *show-insertions* show-insertions)
	     *last-inserted-values*)
    (format t "~%")
    (pp-labeled-list 0 "Update inserted new values:" *last-inserted-values* nil)))

; PP-RESULT just consists of printing the assumptions.
;
(defun pp-result (min-col max-col cur-col result &optional new-line)
  (when (not (member nil (aresult-assump-ll result) :test #'eq))
    (when new-line
      (format t "~%~VT" min-col)
      (setq cur-col min-col))
    (format t "Assump-ll: ")
    (pp-list min-col max-col (+ cur-col 11) nil (aresult-assump-ll result))))


; Print list on separate-lines labeled with label at label-column.
;
(defun pp-labeled-list (label-column label list separate-lines)
  (let ((end-message-column (+ label-column (length label) 1)))
    (format t "~%~VT~@(~a~) " label-column label)
    (if (> end-message-column *data-column*)
	(format t "~%"))
    (format t "~VT" *data-column*))
  (cond (separate-lines
	 (dolist (element list)
	   (pp-list *data-column* 80 *data-column* t element)
	   (format t "~%~VT" *data-column*)))
	(t
	 (pp-list *data-column* 80 *data-column* nil list)
	 (format t "~%~VT" *data-column*))))


; Print long lists in readable format. 'parens' it true iff the
; list is to be enclosed in parens.
;
(defun pp-list (min-col max-col cur-col parens l)
  (let* ((out-string (format nil "~(~a~)" l))
	 (str-len (length out-string))
	 (current cur-col))
    (cond
      ((and parens (< (+ current str-len) max-col))   ; If it fits print it.
       (princ out-string)
       (+ current str-len))

      ((consp l)                                      ; Long lists.
       (cond ((and (> current (/ max-col 2))          ; New line if necessary and useful.
		   (not (eql current min-col)))
	      (format t "~%~VT" min-col)
	      (setq current min-col)))
       (cond (parens (format t "(") (setq current (+ 1 current))))
       (let ((out-list l)
	     (l-min-col current))
	 (cond ((and parens (not (consp (car out-list))))	
						; Special case if first element not a list.
						; Modified 11/20/89 to only special case if parens.
		(setq current (pp-list l-min-col max-col current t (car out-list)))
		(cond ((setq out-list (cdr out-list)) ; If there is more to go.
		       (format t " ")
		       (setq current (+ 1 current))
		       (setq l-min-col current)))))
         (loop
           (cond
            ((consp out-list)
             (setq current (pp-list l-min-col max-col current t (car out-list)))
             (cond ((cdr out-list)                      ; Space after every element but last.
                    (format t " ")
                    (setq current (+ 1 current))))
             (setq out-list (cdr out-list)))
            (out-list                                   ; One element remains but its not nil.
             (setq current (pp-list l-min-col max-col current t out-list))
             (return))
            (t
             (return)))))
       (cond (parens (format t ")") (+ 1 current))
	     (t current)))

      (t                                              ; Base case.
       (cond ((not (= current min-col))
	      (format t "~%~VT" min-col)
	      (setq current min-col)))
       (princ out-string)                          ; Print it.
       (+ current str-len)))))
 