;;; -*- Syntax: Common-lisp; Package: USER -*-

;;; Copyright (c) 1990 by James Crawford and Benjamin Kuipers.
;;;  $Id: amatch.lisp,v 1.1 92/04/16 09:30:14 clancy Exp $

;;;                        ****** AMATCH ******

; Recursive MATCH, based on the Abelson & Sussman matcher, with
; prolog-like variables matching a single subexpression.
; It returns an alist of variable bindings.
;  - explicit FAILED returned on failure; NIL means no bindings.
;
; dat must be variable free (otherwise call unify instead).
;
; Modified 3/4/90 to call standardize after match (like unify).

; Currently assumes variables can be compared using eq.

(defun MATCH (pat dat &optional alist dont-count)
  (if (not dont-count) (setq *match-count* (+ 1 *match-count*)))
  (if (eql alist 'failed)
    'failed
    (standardize (internal-match pat dat (copy-bindings alist)))))

(defun internal-match (pat dat alist)
  (loop
    (if (consp pat)
      (if (or (not (consp dat))
              (eql (setq alist (atom-match (pop pat) (pop dat) alist))
                   'failed))
        (return 'failed))
      (return (atom-match pat dat alist)))))

(defun atom-match (pat dat alist)
  (cond ((atom pat)
         (if (eql pat dat)
           alist
           (if (variable? pat)
             (check+update-binding pat dat alist)
             'failed)))
        ((atom dat) 'failed)
        (t
         (internal-match pat dat alist))))

; pred1 is less-general than (a special case of) pred2 if we can find a substitution
; theta such that theta(pred2) = pred1.
;
(defun less-general (pred1 pred2)
  (if (eql (match pred2 pred1) 'failed)
      nil
      t))

(defun more-general (pred1 pred2)
  (less-general pred2 pred1))

;;; variable? made macro and moved to aglobals.

;;; Can also add #+ti (eq (named-structure-p pat) 'algy-variable).
;;;(defun variable? (pat)
;;;  #-symbolics (algy-variable-p pat)
;;;  ;; Speed up the silly symbolics structure checker:
;;;  #+symbolics (and (arrayp pat) (eq (aref pat 0) 'algy-variable)))

(defun check+update-binding (pat dat alist)
  (if (eq pat dat)
      alist
      (let ((binding (assoc pat alist :test #'eq)))
	(cond ((null binding) (cons (list pat dat) alist))
	      (t (atom-match (cadr binding) dat alist))))))


; Unification allows patterns to be matched against each other.
; Modified 11/29/89 to return a "standard" unification --
; that is one which will work with only one pass by substitute-bindings.

(defun unify (pat exp &optional alist dont-count)
  (standardize (internal-unify pat exp (copy-bindings alist) dont-count)))

(defun internal-unify (pat exp &optional alist dont-count)
  (if (not dont-count) (setq *unify-count* (+ 1 *unify-count*)))
  (cond ((eql alist 'failed) 'failed)
        ((variable? pat)
         (unify-bindings pat exp alist))
        ((variable? exp)
         (unify-bindings exp pat alist))
        ((atom pat) (cond ((eql pat exp) alist)
                          (t 'failed)))
        ((atom exp) 'failed)
        (t (internal-unify (cdr pat)
                           (cdr exp)
                           (internal-unify (car pat) (car exp) alist t)
                           t))))

;;; Needs to be re-optimized and checked.
(defun standardize (alist)
  (if (eq alist 'failed)
      'failed
      (dolist (b alist alist)
	(when (has-variables (cadr b))
	  (setf (cadr b)
		(my-nsubstitute (cadr b) alist))))))

(defun my-nsubstitute (pat alist)
  (cond ((variable? pat)
         (nlookup pat alist))
        ((consp pat)
         (cons (my-nsubstitute (car pat) alist) (my-nsubstitute (cdr pat) alist)))
        (t
         pat)))

(defun nlookup (var alist)
  (let ((b (assoc var alist :test #'eq)))
    (if b
	(if (variable? (cadr b))
	    (setf (cadr b)
		  (nlookup (cadr b) alist))
	    (cadr b))
	var)))

; If there is a previous binding for this variable, we unify the
; new and old bindings against each other, possibly generating 
; further bindings.

(defun unify-bindings (var exp alist)
  (let ((binding (assoc var alist :test #'eq)))
    (cond ((equal var exp) alist)
	  ((null binding)
	   (cond ((freefor var exp alist)
		  (cons (list var exp) alist))))
	  ((internal-unify (cadr binding) exp alist t)))))

; Unification of a variable with another expression which includes that
; variable is arbitrarily difficult, so we exclude it.

(defun freefor (var exp alist)
  (cond ((variable? exp)
	 (cond ((eq var exp) nil)
	       (t (freefor var (cadr (assoc exp alist :test #'eq)) alist))))
	((atom exp))
	((and (freefor var (car exp) alist)
	      (freefor var (cdr exp) alist)))
	(t nil)))


; ALIST HANDLING FUNCTIONS
; Check recursively to determine whether an expression contains a variable at any level.

(defun has-variables (x)
  (cond ((not (consp x))
	 (variable? x))
        (t
	 (or (has-variables (car x))
	     (has-variables (cdr x))))))

;;; These two macros are specialized for substitute-bindings:
;;;
;;; Lookup exp in alist:
(defmacro sb-lookup (exp alist)
  `(let ((b (if (variable? ,exp)
		(assoc ,exp ,alist :test #'eq))))
    (if b
	(cadr b)
	,exp)))
;;;
;;; Substitute into expression which we expect to be an atom:
(defmacro atom-sb-lookup (exp alist)
  `(let ((mexp ,exp))
    (if (consp mexp)
	(substitute-bindings mexp ,alist)
	(sb-lookup mexp ,alist))))

;;; New 20% faster iterative version.
(defun substitute-bindings (exp alist)
  (cond ((null alist)
	 (copy-tree exp))
	((consp exp)
	 (do* ((input (cdr exp) (cdr input))
	       (result (list (atom-sb-lookup (car exp) alist)))
	       (last-cons result (cdr last-cons)))
	      ((not (consp input))
	       (unless (null input)
		 (rplacd last-cons
			 (sb-lookup input alist)))
	       result)
	   (rplacd last-cons (list (atom-sb-lookup (car input) alist)))))
	(t
	 (sb-lookup exp alist))))

#| OLD VERSION:
(defun substitute-bindings (exp alist)
  (cond ((consp exp)
	 (cons (substitute-bindings (car exp) alist)
	       (substitute-bindings (cdr exp) alist)))
	((variable? exp)
	 (let ((b (assoc exp alist :test #'eq)))
	   (if b
	       (cadr b)
	       exp)))
	(t exp)))
|#


; VERY OLD VERSION:
;(defun substitute-bindings (exp alist)
;  (cond ((null alist) exp)  ; a little efficiency hack ...
;	((variable? exp) (let ((val (cadr (assoc exp alist))))
;			   (cond ((null val) exp)
;				 (t (substitute-bindings val alist)))))
;	((listp exp) (mapcar #'(lambda (term)
;				 (substitute-bindings term alist))
;			     exp))
;	(t exp))
;  )

(defun copy-bindings (alist)
  (if (eql alist 'failed)
    'failed
    (mapcar #'(lambda (x) (if (variable? (cadr x))
                            (copy-list x)
                            x))
               alist)))