;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;Grapher.Lisp
;;;  $Id: agraph.lisp,v 1.1 92/04/16 09:30:03 clancy Exp $
;;
;;Copyright ) 1989, Apple Computer, Inc
;;
;;
;;  This file implements the base functionality for nodes and grapher-windows
;;  In order to use it, specific types of nodes must be defined.  The file
;;  list-nodes.lisp is an example.  Nodes should follow the node protocol by
;;  defining the the following functions:
;;    node-children --  returns a list of the node's children nodes
;;    node-parent   --  returns a list of the node's parent nodes
;;    node-draw     --  does the work of drawing a node.  usual-node-draw
;;                      should be called.
;;
;;  The redrawing could be sped up by caching the rectangles
;;  for all the nodes and lines in a quad-tree.  This would, however,
;;  consume a lot more space for a graph.
;;

;; Modified by James Crawford for use in Algernon.


;;(in-package :grapher :use '(:lisp :ccl))

(defparameter *grapher-font* '("geneva" 9))

(eval-when (eval load compile)
  (require 'records)
  (require 'traps)
  (require 'scrolling-windows "ccl;examples:scrolling-windows"))



;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; variables
;;

(defparameter *last-y* 0)
(defparameter *x-spacing* 20)
(defparameter *y-spacing* 10)

(defparameter *grapher-window-size* *fred-window-size*)

;;;;;;;;;;;;;;;
;;
;;  some utilities
;;

(defun point-max (a b)
  (make-point (max (point-h a) (point-h b))
              (max (point-v a) (point-v b))))

(defun point-min (a b)
  (make-point (min (point-h a) (point-h b))
              (min (point-v a) (point-v b))))

(defun halve-point (point)
  (make-point (truncate (point-h point) 2)
              (truncate (point-v point) 2)))


;;;;;;;;;;;;;;;;;;;
;;;
;;; node objects
;;;

(defobject *node*)

(defobfun (exist *node*) (init-list)
  (have 'node-position-iv #@(0 0))
  (have 'node-size-iv #@(150 20))    ;a default value
  (have 'node-center-iv nil)
  (usual-exist init-list))

(defobfun (node-position *node*) ()
  (objvar node-position-iv))

(defobfun (set-node-position *node*) (h &optional v)
  (setf (objvar node-position-iv) (make-point h v)
        (objvar node-center-iv) nil))

(defobfun (node-size *node*) ()
  (objvar node-size-iv))

(defobfun (set-node-size *node*) (h &optional v)
  (setf (objvar node-size-iv) (make-point h v)
        (objvar node-center-iv) nil))

(defobfun (node-center *node*) ()
  (or (objvar node-center-iv)
      (setf (objvar node-center-iv)
            (add-points (node-position)
                        (halve-point (node-size))))))

(defobfun (node-field-size *node*) (limit)
  (setq limit (point-max limit
                         (add-points (node-position)
                                     (node-size))))
  (dolist (child (node-children) limit)
    (setq limit (ask child (node-field-size limit)))))

(defobfun (node-click-event-handler *node*) (where)
  (declare (ignore where)))

(defun layout (root-node)
  (graph-init root-node)
  (ask root-node
    (set-node-position (make-point *x-spacing*
                                   (point-v (node-position)))))
  (setq *last-y* 0)
  (layout-y root-node)
  (leaf-funcall #'layout-x root-node))

(defun graph-init (node)
  "Zeros the coordinates of a node and all of its subnodes"
  (ask node
    (set-node-position #@(0 0))
    (setf (objvar node-center-iv) nil)
    (mapc #'graph-init (node-children))))

(defun layout-y (node)
  (ask node
    (when (zerop (point-v (node-position)))
      (let ((children (node-children)))
        (if (dolist (child children)
              (if (zerop (ask child (point-v (node-position))))
                (return t)))
          (progn
            (mapc #'layout-y children)
            (set-node-position
             (make-point (point-h (node-position))
                         (ceiling 
                          (reduce #'(lambda (a b) (+ a (ask b (point-v (node-position)))))
                                  children 
                                  :initial-value 0)
                          (length children)))))
          (set-node-position
           (make-point (point-h (node-position))
                       (setf *last-y* (+ *y-spacing* *last-y* (point-v (node-size)))))))))))

(defun layout-x (node &aux parents)
  (ask node
    (let* ((pos (node-position)))
      (when (and (zerop (point-h pos))
                 (setq parents (node-parents)))
        (dolist (parent parents)
          (layout-x parent))
        (set-node-position
         (make-point (+ *x-spacing*
                        (apply #'max (mapcar #'(lambda (node) 
                                                 (ask node
                                                   (point-h
                                                    (add-points (node-position)
                                                                (node-size)))))
                                             parents)))
                     (point-v pos)))))))

(defun leaf-funcall (fn node &aux (children (ask node (node-children))))
  "Calls fn on all the leaves of the graph starting at node"
  (if children
    (dolist (child children)
      (leaf-funcall fn child))
    (funcall fn node)))

(defobfun (node-draw-links *node*) (&aux (children (node-children)))
  (when children
    (let* ((center (node-center)))
      (dolist (child children)
        (ask child
          (let ((child-center (node-center)))
            (_MoveTo :long center)
            (_LineTo :long child-center)))))))

(defobfun (node-draw *node*) ()
  (let* ((children (node-children))
         (vis? (node-visible-p))
         (draw-links? (and (or vis? (node-on-right-p))
                           (some #'(lambda (kid)
                                     (ask kid (node-on-left-p)))
                                 children)))
         (do-kids? (or draw-links? (some #'(lambda (kid)
                                             (ask kid (node-on-right-p)))
                                         children))))
    (when draw-links?
      (node-draw-links))
    (when do-kids?
      (dolist (child children)
        (ask child (node-draw))))
    vis?))

(defobfun (node-on-right-p *node*) ()
  (< (point-h (node-center))
     (rref (ccl::%getport) :grafport.portrect.right)))
 
(defobfun (node-on-left-p *node*) ()
  (> (point-h (node-center)) (rref (ccl::%getport) :grafport.portrect.left)))

(defobfun (node-visible-p *node*) ()
  (let ((pos (node-position))
        (grafrect (rref (ccl::%getport) :grafport.portrect)))
    (rlet ((noderect :rect
                     :topleft pos
                     :bottomright (add-points pos (node-size))))
      (logbitp 8 (_SectRect :ptr grafrect :ptr noderect :ptr noderect :word)))))


(defun find-node-containing-point (node point &aux ret)
  (ask node
    (let* ((pos (node-position)))
      (rlet ((r :rect 
                :topleft pos
                :bottomright (add-points pos (node-size))))
        (if (logbitp 8 (_PtInRect :long point :ptr r :word))
          node
          (dolist (child (node-children))
            (if (setq ret (find-node-containing-point child point))
              (return ret))))))))


;;;;;;;;;;;;;;;;;;;;;;
;;
;; grapher window
;;

(defobject *grapher-window* ccl::*scrolling-window*)

(defobfun (exist *grapher-window*) (init-list)
  (let* ((rn (getf init-list :root-node))) 
    (unless rn (error "A root-node must be specified"))
    (have 'root-node rn)
    (layout rn)
    (let ((field-size (add-points (make-point *x-spacing* *y-spacing*)
                                  (ask rn (node-field-size 0)))))
      (without-interrupts
       (usual-exist (init-list-default 
                     init-list
                     :window-font *grapher-font*
                     :window-size (point-min field-size *grapher-window-size*)
                     :window-title "Untitled Grapher"
                     :window-type :document-with-zoom
                     :field-size field-size))))
    (set-window-font :patcopy)))

(defobfun (view-draw-contents *grapher-window*) ()
  (usual-view-draw-contents)
  (with-focused-view (objvar ccl::my-scroller)
    (ask (objvar root-node)
      (node-draw))))

(defobfun (window-click-event-handler *grapher-window*) (where)
  (let* ((other-where (convert-coordinates where
                                           (self)
                                           (objvar ccl::my-scroller)))
         (dialog-item (point-to-dialog-item where))
         (node (or dialog-item
                   (find-node-containing-point (objvar root-node)
                                           other-where))))
    (cond (dialog-item
           (usual-window-click-event-handler where))
          (node
           (ask node (node-click-event-handler other-where))))))



(provide :grapher)
