;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-
(in-package :user)

;;;  $Id: aglobals.lisp,v 1.1 92/04/16 09:30:02 clancy Exp $
;;; Copyright (c) 1990 by James Crawford.

;;;                        ****** AGLOBALS ******

;;; Global variables, constants, and structures for Algernon.

;; GLOBAL CONSTANTS
;;
;; The global constants to define the syntax of the Algernon knowledge-base.

;; Names for some special slots:
(defconstant @slot-props '$sp
	     "Appears in frames for slots and holds properties of slot.")
(defconstant @generic-rules '$gr
	     "Holds rules which apply to all slots in frame.")
(defconstant @dep-net '$dn
	     "Holds dependency net information (in frames for rules).")

;; And the list of slots reserved for system use:
;; (really should be:
;;       '(@slot-props @generic-rules @dep-net)
;; But constants within constants do not seem to work well in common-lisp ...

(defconstant @system-slots '($sp $gr $dn)
  "Slots required by system.")

;; Names for facets:

;;   Facets for holding values and assumptions:
(defconstant @value 'v "Values known to be true.")
(defconstant @n-value 'nv "Values known to be false.")

;;   Facets for holding rules:
;;     Modified 11/20/89 to separate rules for frames and rules for slots.
(defconstant @if-added 'ia "If-added (forward chaining) rules.")
(defconstant @if-needed 'in "If-needed (backward chaining) rules.")
(defconstant @n-if-added 'nia "If-added rules for non-values.")
(defconstant @n-if-needed 'nin "If-needed rules for non-values.")

(defconstant @sif-added 'sia "Slot if-added (forward chaining) rules.")
(defconstant @sif-needed 'sin "Slot if-needed (backward chaining) rules.")
(defconstant @sn-if-added 'snia "Slot if-added rules for non-values.")
(defconstant @sn-if-needed 'snin "Slot if-needed rules for non-values.")

;;    Facets for slot properties:
(defconstant @num-res 'nr "Maximum number of values slot can hold.")

;;   Facets for dependency net:
(defconstant @preds 'p 
	     "Predicates in this frame slot already queried.")
(defconstant @rules-dep 'rd
	     "Rules which depend on this frame slot.")
(defconstant @rule-text 'rt "Appears in @dn slot of rule frame and holds text or rule.")
(defconstant @frame-slot-dep 'fsd "Frame slots this rule depends on.")

(defconstant @debug t "Enables checking for Algernon errors.")
(defconstant @double-check-frames nil "When true arguments to aframes checked.")
;;


;; GLOBAL VARIABLES
;;
;; Version number
(defparameter *algy-version* 1.2)

;; Algernon features supported by a machine:
(defvar *algy-features* nil)

;;   Vars to keep track of the results of the last operation:
(defparameter *last-op* nil "Type of last operation (query or assert).")
(defparameter *last-predicates* nil "Predicates input in last operation.")
(defparameter *last-predicate* nil "Last predicate processed in last operation.")
(defparameter *last-results* nil "Bindings produced by last operation.")
(defparameter *last-inserted-values* nil "Values inserted in last operation.")
(defparameter *last-inserted-assumptions* nil "Assumptions made in last operation.")
(defparameter *last-deleted-values* nil "Values deleted in last operation.")
(defparameter *last-creations* nil "Frames created in last operation.")
(defparameter *last-contradictions* nil "Contradictions found in last operation.")
(defparameter *last-var-alist* nil "Bindings of input vars to Algernon vars in last op.")
(defparameter *unify-count* 0 "Number of unifications in last operation.")
(defparameter *match-count* 0 "Number of matches in last operation.")
(defparameter *rule-count* 0 "Number of rules applied in last operation.")
(defparameter *iteration-count* 0 "Number of iterations required for last primitive operation.")
(defparameter *max-iterations* 0 "Maximum number of iterations required for last operation.")
(defparameter *frame-insertions* 0 "Insertions into frames in last operation.")
(defparameter *frame-accesses* 0 "Accesses to frames in last operation.")
(defparameter *facets* (list @value @n-value
			     @if-added @if-needed @n-if-added @n-if-needed
			     @sif-added @sif-needed @sn-if-added @sn-if-needed
			     @num-res
			     @preds @rules-dep @rule-text @frame-slot-dep)
	      "Legal facets.")

;; Variable to recognize recursive calls to Algernon:
(defparameter *algy-recursive-callp* nil "True when Algernon called recursively")
;; Rule chaining:
(defparameter *search-strategy* 'depth-first)
;; A variable used in depth-first rule application:
(defparameter *rules-to-complete* nil "List of forward chaining rules which need to be completed.")
;; and one for breadth first:
(defparameter *top-level* t "True at top level of rule chaining.")
;; The next two switches are not checked in arules, but rather are checked
;; before calls are made into arules:
(defparameter *back-chain* t "Allows rule back-chaining -- set to nil by :retrieve.")
(defparameter *forward-chain* t "Allows rule forward-chaining -- almost always t.")

(defparameter *contra-positive* nil "Enables addition of contra-positives of rules.")

;; And switches for errors and warnings:
(defparameter *suppress-warnings* nil "Suppresses warnings")
(defparameter *algy-break-on-warning* nil "Call cerror on warnings")
(defparameter *algy-break-on-error* t "Call cerror on errors")

;; The function handle-special-forms in alogic.l defines the syntax of
;; special forms.  The allowed special forms:
(defparameter *special-forms*
	      '(:CREATE			; create new node of specified type
		:THE			; definite description
		:FORC			; find or create a frame to satisfy a path
		:A			; create new node, assert path
		:ANY			; return 0 or 1 binding from path
		:RULES                  ; add rules
		:SRULES                 ; add slot rules
		:DEL-RULE               ; delete a rule from a frame
		:DEL-SRULE              ; delete a slot rule from a slot
		:DEL-RULES              ; delete all rules from a frame-slot
		:DEL-SRULES             ; delete all slot rules from a slot
		:CLEAR-SLOT             ; remove all values from a frame-slot
		:DECL-SLOTS             ; declares slots
		:ASSUME    		; add a value to assumed-value facet
                :NEQ                    ; succeeds if its arguments are different frames
		:EVAL			; escape to lisp to eval predicate (returned value ignored)
		:LISP			;  (same)
		:TEST			; escape to lisp to eval predicate -- continue iff returns non nil
		:APPLY			; apply a lisp funciton
		:BIND                   ; bind algernon variable to result of lisp function
		:BRANCH			; bind algernon variable to each value in list
		:ASK			; query user
		:SHOW                   ; pretty-print frame
		:DELETE			; delete all values from a slot
		:RETRIEVE               ; retrieve known values but suppress rule firings
		:NO-COMPLETION          ; Suppresses rule completion for its argument
                :WO-CONTRA-POSITIVE     ; Suppresses addition of contra-positives of rules within its scope
                :W-CONTRA-POSITIVE       ; Enables addition of contra-positives of rules within its scope
                :BIND-TO-VALUES         ; bind variable to list of values in a frame-slot.
                :BRANCH-ON-VALUES       ; bind variable to each value in list returned by lisp function
		:UNP                    ; Unprovable
                :ALL-PATHS              ; succeeds if all paths for 1st arg extend to paths for 2nd.
                :IN-OWN-PARTITION       ; Queries or asserts predicate in its own partition.
		)
  "Special forms allowed in Algernon")

(defvar *kbs* nil
  "List of stored knowledge-bases")
(defvar *user-path* nil "The home directory of the current user.")
(defvar *user-kb-path* nil "The kb directory of the current user.")



;; STRUCTURES
(defstruct (algy-variable (:constructor new-variable (name))
			  (:print-function print-algy-variable))
  name)

(defun print-algy-variable (var str depth)
  (declare (ignore depth))
  (format str "~a" (algy-variable-name var)))

(defstruct (aresult)
  (sub 'nil)
  (assump-ll (list nil))
  alist)

(defun new-aresult (&optional alist)
  (make-aresult :alist alist))

;;; MACROS
;;;

(defparameter *no-depnet* nil "When true no dependency information is recorded.")

(defmacro with-no-depnet (&body exp)
  `(let ((*no-depnet* t))
     ,@exp))

(defmacro with-depnet (&body exp)
  `(let ((*no-depnet* nil))
     ,@exp))

(defmacro with-no-back-chaining (&body exp)
  `(let ((*back-chain* nil))
     ,@exp))

(defmacro with-no-forward-chaining (&body exp)
  `(let ((*forward-chain* nil))
     ,@exp))

(defmacro with-no-warnings (&body exp)
  `(let ((*suppress-warnings* t))
     ,@exp))

(defmacro with-no-output (&body exp)
  (let ((show-vars '(*show-sentence* *show-input* *show-predicates*
				     *show-bindings* *show-insertions*
				     *show-other-kb-changes* *show-stats*
				     *show-failures* *cerror-on-failure*)))
    `(let ,(mapcar #'(lambda (var) (list var nil))
		   show-vars)
       (declare (special ,@show-vars))
       (with-trace-off ,@exp))))

(defmacro with-no-output-unless-failure (&body exp)
  (let ((show-vars '(*show-sentence* *show-input* *show-predicates*
                     *show-bindings* *show-insertions*
                     *show-other-kb-changes* *show-stats*)))
    `(let ,(mapcar #'(lambda (var) (list var nil))
                   show-vars)
       (let ((*cerror-on-failure* t))
         (declare (special ,@show-vars))
         (with-trace-off ,@exp)))))

;;(with-no-output (foo) (var))

(defmacro with-trace-off (&body exp)
  (let ((trace-vars
	  '(*tracing* *dot-trace* *trace-names* *trace-frames*
		      *trace-frame-accesses* *trace-frame-creation* *trace-alogic*
		      *trace-alists* *trace-rules* *trace-iterations*
		      *trace-automatic-rule-addition* *trace-rule-completion*
		      *trace-values* *trace-new-values* *trace-assumptions*
		      *trace-contra* *trace-retrieval* *trace-depnet*
		      *trace-top-level-propagate* *trace-partitions*
		      *trace-partition-updates* *trace-prep*)))
    `(let ,(mapcar #'(lambda (var) (list var nil))
		   trace-vars)
       (declare (special ,@trace-vars))
       ,@exp)))

(defmacro silently (&body exp)
  `(with-no-output (with-no-warnings (with-trace-off ,@exp))))

(defmacro silently-unless-failure (&body exp)
  `(with-no-output-unless-failure (with-no-warnings (with-trace-off ,@exp))))

(defmacro quietly (&body exp)
  `(silently-unless-failure ,@exp))

(defmacro with-partitions (frame slot &body exp)
  `(or (and (eql ,frame *cur-frame*)
            (eql ,slot  *cur-slot*))
       (let ((*cur-partitions* (partitions ,frame ,slot))
             (*cur-frame* ,frame)
             (*cur-slot* ,slot))
         (update-partitions *cur-partitions*)
         (if (not *cur-partitions*)
           (algy-warning (format nil "No partitions for <~(~a~),~(~a~)>." ,frame ,slot)))
         ,@exp)))


(defmacro variable? (pat)
;;; Could add #+ti (eq (named-structure-p pat) 'algy-variable)
;;; but it does not seem to do any good.
  #-symbolics `(algy-variable-p ,pat)
  ;; Speed up the silly symbolics structure checker:
  #+symbolics `(let ((pat-var ,pat))
                 (and (arrayp pat-var) (eq (aref pat-var 0) 'algy-variable)))
  )

(defmacro framep (f)
  `(let ((f-var ,f))
     (and (symbolp f-var) (get f-var 'framep))))

(defmacro slotp (s)
  `(let ((s-var ,s))
     (and (symbolp s-var) (get s-var 'slotp))))

(defmacro facetp (f)
  `(let ((f-var ,f))
     (and (symbolp f-var) (get f-var 'facetp))))



(defmacro negated (pred)
  `(eq (car ,pred) 'not))

(defmacro frame (pred)
  `(let ((p ,pred))
     (second (if (negated p)
               (second p)
               p))))

(defmacro slot (pred)
  `(let ((p ,pred))
     (first (if (negated p)
              (second p)
              p))))

(defmacro facet (pred)
  `(if (negated ,pred)
     @n-value
     @value))

(defmacro value (pred)
  `(let ((p ,pred))
     (third (if (negated p)
              (second p)
              p))))

; Optimized version.  See atrace for original.
(defmacro trace-frame-access (frame slot facet value)
  `(cond ((not *trace-frame-accesses*)
          ,value)
         (t
          (let ((fr ,frame)
                (sl ,slot)
                (fa ,facet)
                (va ,value))
            (tab-over)
            (format t "Accessing value ~a from frame ~a slot ~a facet ~a."
                    va fr sl fa)
            va))))


; Optimized version.  See aframes for original.
(defmacro FGET (frame slot facet)
  `(let ((fr ,frame)
         (sl ,slot)
         (fa ,facet))
     (if @double-check-frames (check-frame fr sl fa nil))
     (incf *frame-accesses*)
     (trace-frame-access fr sl fa
                         (cdr (assoc fa
                                     (cdr (assoc sl
                                                 (cdr (get fr 'frame))
                                                 :test #'eq))
                                     :test #'eq)))))

(defmacro FPUT (frame slot facet value)
  `(let ((fr ,frame)
         (sl ,slot)
         (fa ,facet)
         (va ,value))
     (if @double-check-frames (check-frame fr sl fa va))
     (incf *frame-insertions*)
     (let ((value-list (follow-path (list sl fa)
                                    (fget-frame fr))))
       (trace-fput fr sl fa va)
       (cond ((member va (cdr value-list) :test #'equal)
              nil)
             (t
              (rplacd value-list (cons va (cdr value-list)))
              (if (eql fa @value) (incf *num-values*))
              t)))))

(defmacro listify (x)
  `(if (listp ,x)
    ,x
    (list ,x)))