;;; -*- Syntax: Common-lisp; Package: USER -*-

;;; Copyright (c) 1990 by James Crawford and Benjamin Kuipers.
;;;  $Id: aframes.lisp,v 1.1 92/04/16 09:30:01 clancy Exp $

;;;                        ****** AFRAMES ******

; Following Winston+Horn, chapter 22, we define frames to be generalized
; association lists stored on the property lists of (generated) atoms.
;   ATOM-27:
;      FRAME:  (ATOM-27 (slot1 (facet1 val1 val2 . . . valn)
;                              (facet2 val21 . . .  )
;                              . . .  )
;                       (slot2 (facet1 val31 . . .  )
;                              . . .  )
;                       . . .  ))
;

; We assume slots and facets are comparable using eq.

; Might be speed up by using eq hash tables for slots (on mac break even point
; is about 18, on a symbolics about 50, and on a TI assoc seems to be always faster
; up to at least 1000).

; All routines, except follow-path and find-unused-name, are inlined:
(proclaim '(inline FGET-FRAME))
(proclaim '(inline EXTEND))
(proclaim '(inline MAKE-NAME))
(proclaim '(inline FRAME-NAME))
(proclaim '(inline COUNT-VALUES))
(proclaim '(inline COUNT-FRAMES))
(proclaim '(inline FCLEAR-SLOT))
(proclaim '(inline FCLEAR-FACET))
(proclaim '(inline FREMOVE))
(proclaim '(inline CHECK-FRAME))
(proclaim '(inline ALL-FRAMES))
(proclaim '(inline RESET-FRAMES))
(proclaim '(inline MAKE-INTO-FACET))
(proclaim '(inline MAKE-INTO-SLOT))
(proclaim '(inline MAKE-NEW-FRAME))

; Parameters to keep track of how many frames there are, what frames there are
; and how many values there are in the frames:
(defvar *max-frame* 0)
(defvar *all-frames* nil)
(defvar *all-facets* nil)
(defvar *num-values* 0)

; First, routines to declare and recognize frames, slots and facets:

; MAKE-NEW-FRAME: Create a new frame.
;
(defun MAKE-NEW-FRAME (suggested-name)
  (let ((name (cond ((null suggested-name)
		     (frame-name (setq *max-frame* (+ 1 *max-frame*))))
		    (t
		     (find-unused-name suggested-name)))))
    (push name *all-frames*)
    (setf (get name 'frame) nil)             ; clear frame property.
    (setf (get name 'framep) t)              ; record that it is a frame.
    (push name *last-creations*)
    (trace-frame-creation name)))

; FIND-UNUSED-NAME
;
; Finds a variant of suggested-name which is not in use and is a symbol.
;
(defun find-unused-name (name &optional (root name) (counter 1))
  (if (framep name)
      (find-unused-name (intern (concatenate 'string
					     (string root)
					     (prin1-to-string counter)))
			root
			(+ counter 1))
      (if (symbolp name)
	  name
	  (frame-name (setq *max-frame* (+ 1 *max-frame*))))))

; FRAMEP -- made into macro and moved to aglobals.
;
;(defun FRAMEP (f)
;  (and (symbolp f) (get f 'framep)))

; make-into-slot
;
(defun make-into-slot (s)
  (if (framep s)
    (setf (get s 'slotp) t)
    (algy-error (format nil "Can't make non-frame ~(~a~) into a slot." s))))

; SLOTP -- made into macro and moved to aglobals.
;
;(defun slotp (s)
;  (and (symbolp s) (get s 'slotp)))

; make-into-facet
;
(defun make-into-facet (f)
  (cond ((symbolp f)
         (setf (get f 'facetp) t)
         (push f *all-facets*))
        (t
         (algy-error (format nil "Can't make non-symbol ~(~a~) into a facet." f)))))

; FACETP -- made into macro and moved to aglobals.
;
;(defun facetp (f)
;  (and (symbolp f) (get f 'facetp)))


; RESET-FRAMES: Clear all frames.
;
(defun RESET-FRAMES ()
  (dolist (frame *all-frames*)
    (remprop frame 'frame)
    (remprop frame 'framep)
    (remprop frame 'slotp))
  (dolist (facet *all-facets*)
    (remprop facet 'facetp))
  (setq *all-frames* nil)
  (setq *all-facets* nil)
  (setq *max-frame* 0)
  (setq *num-values* 0))

; All-Frames -- Returns a list of all frames in the knowledge-base.
;
(defun all-frames ()
  *all-frames*)


; Then routines to put things in frames and get them out.

; Changed to macros and moved to aglobals 10/14/90.

; Get all values in a specified frame.slot.facet:
;
;(defun FGET (frame slot facet)
;  (if @double-check-frames (check-frame frame slot facet nil))
;  (incf *frame-accesses*)
;  (trace-frame-access frame slot facet
;                      (cdr (assoc facet
;                                  (cdr (assoc slot
;                                              (cdr (get frame 'frame))
;                                              :test #'eq))
;                                  :test #'eq))))


; Put a value into frame.slot.facet if it's not already there.
;
; Returns true iff a new value was successfully inserted.
;
;(defun FPUT (frame slot facet value)
;  (if @double-check-frames (check-frame frame slot facet value))
;  (incf *frame-insertions*)
;  (let ((value-list (follow-path (list slot facet)
;                                 (fget-frame frame))))
;    (trace-fput frame slot facet value)
;    (cond ((member value (cdr value-list) :test #'equal)
;           nil)
;          (t
;           (rplacd value-list (cons value (cdr value-list)))
;           (if (eql facet @value) (incf *num-values*))
;           t))))

(defun check-frame (frame slot facet value)
  (if (not (framep frame))
      (algy-error (format nil "Non-existant frame ~a (slot = ~a, facet = ~a, value = ~a)"
			    frame slot facet value)))
  (if (not (slotp slot))
      (algy-error (format nil "Non-existant slot ~a (frame = ~a, facet = ~a, value = ~a)"
			    slot frame facet value)))
  (if (not (facetp facet))
      (error "Algernon Bug -- Illegal facet ~a (frame = ~a, slot = ~a, value = ~a)"
	     facet frame slot value)))

; Remove a value from the frame structure.
; (maybe these should not use follow-path (since it creates slots and facets) ??)

(defun FREMOVE (frame slot facet value &optional test)
  (cond ((not (and (consp frame) (eq 'or (car frame))))
	 (if @double-check-frames (check-frame frame slot facet value))
	 (trace-deletion frame slot facet value)
	 (let ((value-list (follow-path (list slot facet)
					(fget-frame frame)))
	       (delete-test (if (not test) #'(lambda (x) (equal x value)) test)))
	   (if (cdr value-list)
	       (setf (cdr value-list) (delete-if delete-test (cdr value-list))))
	   (if (null (cdr value-list)) (fclear-facet frame slot facet))
	   t))))

; Remove all values from a specified facet.
;
(defun FCLEAR-FACET (frame slot facet)
  (cond ((not (and (consp frame) (eq 'or (car frame))))
	 (if @double-check-frames (check-frame frame slot facet 'all-values))
	 (trace-deletion frame slot facet 'all-values)
	 (let ((slot-value (follow-path (list slot)
					(fget-frame frame))))
	   (cond ((cdr slot-value)
					; Here is some magic to remove an entire entry from an alist:
		  (setf (cdr slot-value) (delete-if #'(lambda (x) (eql facet (car x)))
						    (cdr slot-value)))))
	   (if (null (cdr slot-value)) (fclear-slot frame slot))
	   t))))

; Remove all values from a specified slot.
;
(defun FCLEAR-SLOT (frame slot)
  (cond ((not (and (consp frame) (eq 'or (car frame))))
	 (trace-deletion frame slot nil 'all-facets)
	 (let ((frame-value (fget-frame frame)))
	   (setf (cdr frame-value) (delete-if #'(lambda (x) (eql slot (car x)))
					      (cdr frame-value)))
	   (if (null (cdr frame-value)) (remprop frame 'frame))
	   t))))

(defun COUNT-FRAMES ()
  (length *all-frames*))

(defun COUNT-VALUES ()
  *num-values*)


; Utility routines.


;FRAME-NAME: Return the name of frame n.
;
(defun frame-name (n)
  (make-name 'frame (prin1-to-string n)))

; Make-Name: Make a string from var and suffix.
;
(defun make-name (var suffix)
  (intern (concatenate 'string (string var) suffix)))


; Do find-or-create retrieval in a frame.

(defun follow-path (path frame)
  (cond ((null path) frame)
	(t (follow-path (cdr path)
			(extend (car path) frame)))))

(defun extend (key frame)
  (let ((entry (assoc key (cdr frame) :test #'eq)))
    (cond ((null entry)				; empty frame = (NAME)
	   (setq entry (list key))
	   (rplacd (last frame) (list entry))))
    entry))

(defun fget-frame (frame)			; create property list entry if needed.
  (cond ((get frame 'frame))
	(t (setf (get frame 'frame) (list frame)))))
