;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-
(in-package :user)

;;;  $Id: aface.lisp,v 1.3 92/04/24 12:22:49 clancy Exp $
;;; Copyright (c) 1990, 1991 by James Crawford and Benjamin Kuipers

;;;                          ****** AFACE *******


; Initial copyright information:
(defparameter *copyright-notice* "
Algernon 1.2.2 of 6 February 1992.
Copyright (c) 1991, The University of Texas at Austin.

Algernon comes with ABSOLUTELY NO WARRANTY;  type `show-w' for details.
It is distributed for research purposes only;  type `show-c' for conditions.
Please complete the copyright agreement.  Complete instructions are provided
with both the show-w and show-c commands.")

(defparameter *show-copyright-notice* t)

(defparameter *UT-legal-notice* "
For authorization to use the Algernon software package please complete
the authorization request form.  This form can be found in the help
directory and is called algernon-authorization.ps.  A text version of
this form is also included in the directory.

Please complete this form and mail it to:

	University of Texas at Austin 
	Department of Computer Science
	Dr. Benjamin Kuipers
	Taylor Hall 2.124
	Austin, TX 78712-1188

Upon receipt of this form, we will send you a letter authorizing you
to use the Algernon software package.  You are authorized to install
and use the Algernon package in the interim according to the
conditions defined in the limited software license agreement below.


Copyright(c) 1991, The University of Texas at Austin (UTA).  All rights 
reserved.  By using this software, you, the LICENSEE, indicate you have
read, understood, and will comply with the following:

``Nonexclusive permission to use, copy and/or modify this software for
internal, noncommercial purposes is granted.  Any distribution, including
commercial sale, of this software, copies, associated documentation and/or
modifications is strictly prohibited without the prior written consent of
UTA.  Title to copyright to this software and associated documentation
shall at all times remain with UTA.  Appropriate copyright notice shall be 
placed on software copies, and a copy of this permission notice in
associated documentation.  No right is granted to use in advertising,
publicity or otherwise any trademark or the name of UTA.  Any software
and/or associated documentation identified as ``confidential'' will be
protected from unauthorized use/disclosure with the same degree of care
LICENSEE regularly employs to safeguard its own such information.  

**This software is provided ``as is'', and UTA MAKES NO REPRESENTATIONS OR
WARRANTIES, EXPRESS OR IMPLIED, INCLUDING THOSE OF MERCHANTABILITY OR
FITNESS FOR A PARTICULAR PURPOSE, OR THAT USE OF THE SOFTWARE,
MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL NOT INFRINGE ANY PATENTS,
COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.  UTA shall not be held liable for
any liability nor for any direct, indirect or consequential damages with 
respect to any claim by LICENSEE or any third party on account of or arising
from this Agreement.

***Submit software operation questions to: Prof. Benjamin Kuipers,
Computer Science Department, U.T. Austin, Austin, Texas 78712 USA.
Email:  kuipers@cs.utexas.edu.

Submit distribution and/or commercialization requests to:
Executive Vice President and Provost's Office, U.T. Austin, Main Building
201, Austin, Texas 78712, ATTN:  Technology Licensing Specialist.
")


; Interactive algernon interface.

; Local variables:
;
;   Vars for knowledge-base browser:
(defparameter *frame-stack* nil)
(defparameter *next-frames* nil)
(defparameter *prev-frames* nil)
;
;   Var to input commands:
(defparameter *command* nil)

; ALGY -- Command interpretor.
;
(defun ALGY ()
  (when *show-copyright-notice*
    (format t *copyright-notice*))
  (loop (format t "~&~%algy> ")
	;; peek-char does not work on mac (??) 
        ;;(peek-char)
	(setq *command* (read *query-io* nil nil))
	(if (eql *command* 'algy>)                         ; Strip off prompt (only needed on HP's).
          (setq *command* (read *query-io* nil nil)))
	(case *command*
	  
	  ; EXIT
	  ((exit e x quit)
	   (return t))
	  
	  ; QUERY
	  ((query q)
	   (acom-query (get-responce "Path to query:")))
	   
	  ; ASSERT
	  ((assert a)
	   (acom-assert (get-responce "Path to assert:")))

	  ; KB-DIRECTORY
	  ((kb-directory kbd)
           (acom-kb-directory))
	  ; KB-SNAPSHOT
	  ((kb-snapshot kbs)
	   (acom-kb-snapshot (get-responce "Name of new snapshot:")))
	  ; WRITE-SNAPSHOT
	  ((write-snapshot ws)
           (maybe-change-kb-dir)
	   (acom-write-snapshot (get-responce "Snapshot to write:")))
	  ; LIST-SNAPSHOTS
	  ((list-snapshots ls)
	   (acom-list-snapshots))
	  ; LOAD-KB
	  ((load-kb lkb)
           (maybe-change-kb-dir)
	   (acom-load-kb (get-responce "Knowledge-base to load:")))
	     
	  ; DUMP-KB
	  ((dump-kb dkb)
           (maybe-change-kb-dir)
	   (acom-dump-kb (get-responce "Name of new knowledge-base:")))
	  ; DELETE-SNAPSHOT
	  ((delete-snapshot ds)
	   (acom-delete-snapshot (get-responce "Name of snapshot to delete:")))		 
	   
	  ; RESET
	  (reset
	   (acom-reset))
	  ; DELETE-ALL-SNAPSHOTS
	  ((delete-all-snapshots das)
	   (acom-delete-all-snapshots))
	  
	  ; CLEAR WINDOW
	  ((clear-window cw)
	   (acom-clear-window))

	  ; UPDATE PARTITIONS
	  ((update-partitions up)
	   (acom-update-partitions (get-responce "Partition(s) to update:")))

	  ; HELP
	  ((help h ?)
	   (acom-help (get-responce "Help subtopic (type ? if not sure):")))

	  ; VERSION -- Temporarly disabled due to problem on
          ; Symbolics with long cond statements.
	  ; ((version v)
	  ;  (acom-version))
	  
	  ; PROFILE-KB
	  ((profile-kb pkb)
	   (acom-profile-kb))
	  ; KB-OVERVIEW
	  ((kb-overview kbo)
	   (acom-kb-overview (get-responce "Root(s):")
			     (get-responce "Slot(s):")
			     (yes-or-no-p "Reverse links")))

	  ; WHO
	  ((who w)
	   (acom-who (get-responce "Name:")))
	  ; VISIT-FRAME
	  ((visit-frame vf)
	   (acom-visit-frame (get-responce "Frame:")))
	  ; VISIT-SLOT
	  ((visit-slot vs)
	   (acom-visit-slot (get-responce "Slot:")))
	  ; VISIT-SLOT-NONV
	  ((visit-slot-nonv vsn)
	   (acom-visit-slot-nonv (get-responce "Slot:")))
	   
	  ; POP
	  ((pop p)
	   (acom-pop))
	  ; NEXT-FRAME
	  ((next-frame nf)
	   (acom-next-frame))
	  ; PREVIOUS-FRAME
	  ((previous-frame pf)
	   (acom-previous-frame))
	  ; CURRENT-FRAME
	  ((current-frame cf)
	   (acom-current-frame))
	  ; RULES
	  ((rules r)
	   (acom-rules (get-responce "Slot:")))

	  ; VIEW-DOT-TRACE
	  ((view-dot-trace vdt)
	   (acom-view-dot-trace))
	  ; DEFAULT-TRACE
	  ((default-trace dt)
	   (acom-default-trace))
	  ; TRACE-ALL
	  ((trace-all ta)
	   (acom-trace-all))
	  ; TRACE-OFF
	  ((trace-off to)
	   (acom-trace-off))
	  ; TRACE-LOGIC
	  ((trace-logic tl)
	   (acom-trace-logic))
	  ; TRACE-INTEREST
	  ((trace-interest ti)
	   (acom-trace-interest))

	  ; NORMAL-OUTPUT
	  ((normal-output no)
	   (acom-normal-output))
	  ; MINIMAL-OUTPUT
	  ((minimal-output mo)
	   (acom-minimal-output))
	  ; SILENT-OUTPUT
	  ((silent-output so)
	   (acom-silent-output))
	  ; VERBOSE-OUTPUT
	  ((verbose-output vo)
	   (acom-verbose-output))
	  ; LAST-OP
	  ((last-op lo)
	   (acom-last-op))

	  ; LISP
	  ((lisp l)
	   (acom-lisp (get-responce "Lisp form to evaluate:")))
	  ; LOAD
	  (load
	   (acom-load (get-responce "File to load:")))
	  ; COMPILE-LOAD
	  ((compile-load cl)
	   (acom-compile-load (get-responce "File to compile:")))
	  (t
	   (rest-of-algy-cond)))))

;;; On some machines the function Algy was too long, so we put part
;;; of the cond in this function ...
(defun rest-of-algy-cond ()
  (case *command*
    ; LOAD-EXAMPLE
    ((load-example le)
     (acom-load-example (get-responce "Example to load:")))
    ; RUN-EXAMPLE
    ((run-example re)
     (acom-run-example (get-responce "Example to run:")))
    ; SHOW-C and SHOW-W
    ((show-w show-c)
     (format t *UT-legal-notice*))
    (t
     (if (consp *command*)
	 (format t "~&  ~a~%" (eval *command*))
	 (format t "~&  Unknown command ~@(~a~).~%" *command*)))))

(defun acom-query (path)
  (a-query "User query." path))

(defun acom-assert (path)
  (a-assert "User assertion." path))

(defun acom-kb-directory ()
  (change-kb-dir))

(defun acom-kb-snapshot (name)
  (if (kb-snapshot name)
      (format t "~& Snapshot ~(~a~) taken." name)
      (error "Kb-snapshot failed ... (which should be impossible)")))

(defun acom-write-snapshot (name)
  (if (write-snapshot name)
      (format t "~& Knowledge-base ~(~a~) written." name)
      (algy-warning (format nil "Knowledge-base ~(~a~) not found." name))))

(defun acom-list-snapshots ()
  (format t "~& Current snapshots: ~a"
	  (mapcar #'(lambda (x) (kb-name x)) *kbs*)))

(defun acom-load-kb (name)
  (cond ((kb-exists? name)
	 (format t "~& Resetting Algernon.~%")
	 (reset-algy)
	 (if (load-kb name)
	     (format t "~& Knowledge-base ~(~a~) loaded." name)
	     (error "Snapshot exists but load failed ... (which should be impossible)")))
	(t
	 (algy-warning (format nil "Knowledge-base ~(~a~) could not be found." name)))))

(defun acom-dump-kb (name)
  (if (dump-kb name)
      (format t "~& Knowledge-base saved as ~(~a~)." name)
      (algy-warning (format nil "Knowledge-base ~(~a~) could not be found." name))))

(defun acom-delete-snapshot (name)
  (if (delete-snapshot name)
      (format t "~& Snapshop ~(~a~) deleted." name)
      (algy-warning (format nil "Snapshop ~(~a~) not found." name))))

(defun acom-reset ()
  (format t "~& Resetting Algernon.~%")
  (reset-algy)
  (load-common-sense-kb)
  (format t "~& Background knowledge-base loaded.~%"))

(defun acom-delete-all-snapshots ()
  (setq *kbs* nil)
  (format t "~& All snapshots deleted."))

(defun acom-clear-window ()
  (clear-window))

(defun acom-update-partitions (partitions)
  (init-trace)
  (reset-dnet)
  (reset-global-vars nil nil)
  (unwind-protect
       (update-partitions (if (consp partitions) partitions (list partitions)))
    (reset-dnet)
    (end-trace)
    (pp-up)))

(defun acom-help (input-name)
  (let* ((file-name (if (eql input-name '?) 'help input-name))
	 (file-path (format nil "~a~(~a~)~a" @algy-help-path file-name @algy-help-suffix)))
    (if (probe-file file-path)
	(with-open-file (help-file file-path)
	  (format t "~%")
	  (loop (let ((inline (read-line help-file nil 'eof)))
		  (if (eql inline 'eof)
		      (return)
		      (write-line inline)))))
	(format t "No help file for ~(~a~)." file-name))))

(defun acom-version ()
  (format t "~%Algernon version: ~a~%" *algy-version*))

(defun acom-profile-kb ()
  (profile-kb))

(defun acom-kb-overview (root slot rl)
  ;; Force root and slot to be lists:
  (if (not (consp root)) (setq root (list root)))
  (if (not (consp slot)) (setq slot (list slot)))
  
  ;; Check input and then do overview:
  (cond ((not (every #'(lambda (x) (slotp x)) slot))
	 (format t "~%~% ~@(~a~) is not a slot."
		 (find-if-not #'(lambda (x) (slotp x)) slot)))
	((not (every #'(lambda (x) (framep x)) root))
	 (format t "~%~% ~@(~a~) is not a frame."
		 (find-if-not #'(lambda (x) (framep x)) root)))
	(t
	 (kb-overview :relation slot :root root :rl rl))))

(defun acom-who  (name)
  (if (not (consp name))
      (setq name (list name)))
  (visit-frames (remove-assumps (objects-from-name name)))
  (show-current-frame))

(defun acom-visit-frame (frame)
  (cond ((framep frame)
	 (visit-frames (list frame))
	 (show-current-frame))
	(t
	 (format t "~% ~(~a~) is not a frame.~%" frame))))

(defun acom-visit-slot (slot)
  (if (null *frame-stack*)
      (format t "~% No current frame to visit a slot of.~%")
      (cond ((slotp slot)
	     (visit-frames (remove-assumps (fget (car *frame-stack*) slot @value)))
	     (show-current-frame))
	    (t
	     (format t "~% ~(~a~) is not a slot.~%" slot)))))

(defun acom-visit-slot-nonv (slot)
  (if (null *frame-stack*)
      (format t "~% No current frame to visit a slot of.~%")
      (cond ((slotp slot)
	     (visit-frames (remove-assumps (fget (car *frame-stack*) slot @n-value)))
	     (show-current-frame))
	    (t
	     (format t "~% ~(~a~) is not a slot.~%" slot)))))

(defun acom-pop ()
  (setq *frame-stack* (cdr *frame-stack*))
  (setq *next-frames* nil)
  (setq *prev-frames* nil)
  (show-current-frame))

(defun acom-next-frame ()
  (if (null *next-frames*)
      (format t "~&  No next frame.~%")
      (prog ()
	 (setq *prev-frames* (cons (car *frame-stack*) *prev-frames*))
	 (setq *frame-stack* (cons (car *next-frames*) (cdr *frame-stack*)))
	 (setq *next-frames* (cdr *next-frames*))
	 (show-current-frame))))

(defun acom-previous-frame ()
  (if (null *prev-frames*)
      (format t "~&  No previous frame.~%")
      (prog ()
	 (setq *next-frames* (cons (car *frame-stack*) *next-frames*))
	 (setq *frame-stack* (cons (car *prev-frames*) (cdr *frame-stack*)))
	 (setq *prev-frames* (cdr *prev-frames*))
	 (show-current-frame))))

(defun acom-current-frame ()
  (show-current-frame))

(defun acom-rules (slot)
  (if (null *frame-stack*)
      (format t "~% No current frame to show rules for.~%")
      (if (slotp slot)
	  (pp-rules (car *frame-stack*) slot)
	  (format t "~% ~(~a~) is not a slot.~%" slot))))

(defun acom-view-dot-trace ()
  (view-dot-trace))

(defun acom-default-trace ()
  (default-trace)
  (format t "~&  Default trace.~%"))

(defun acom-trace-all ()
  (trace-all)
  (format t "~&  Tracing.~%"))

(defun acom-trace-off ()
  (trace-off)
  (format t "~&  Tracing off.~%"))

(defun acom-trace-logic ()
  (trace-logic)
  (format t "~& Tracing logic.~%"))

(defun acom-trace-interest ()
  (trace-interest)
  (format t "~% Default trace.~%"))

(defun acom-normal-output ()
  (normal-output)
  (format t "Normal output."))

(defun acom-minimal-output ()
  (minimal-output)
  (format t "Minimal output."))

(defun acom-silent-output ()
  (silent-output)
  (format t "Silent output."))

(defun acom-verbose-output ()
  (verbose-output)
  (format t "Verbose output."))

(defun acom-last-op ()
  (pp-output t t t t))

(defun acom-lisp (form)
  (format t "~&  ~a~%" (eval form)))

(defun acom-load (name)
  (let ((success (algy-load-file name @algy-source-path)))
    (if success (format t "~&  Loaded file ~(~a~).~%" success))))

(defun acom-compile-load (name)
  (let ((success (algy-compile-load (format nil "~(~a~)" name))))
    (if success (format t "~& Compiled and loaded ~(~a~).~%" success))))

(defun acom-load-example (name)
  (let ((success (algy-load-file name @algy-example-path)))
    (if success (format t "~& Loaded file ~(~a~).~%" success))))

(defun acom-run-example (example)
  (let ((path (format nil "~a~(~a~)~(~a~)" @algy-help-path example @algy-help-suffix)))
    ;; Print out help file for example (if any).
    (if (probe-file path)
	(with-open-file (help-file path)
	  (format t "~%")
	  (loop (let ((inline (read-line help-file nil 'eof)))
		  (if (eql inline 'eof)
		      (return)
		      (write-line inline))))))
       (format t "~&~|&Example ~a --- Facts:~&~&" example)
       (eval (read-from-string (format nil "(facts-about-~(~a~))" example)))
       (format t "~&~|~&Example ~a --- Queries:~&~&" example)
       (eval (read-from-string (format nil "(queries-about-~(~a~))" example)))))


; Tell/Ask interface -- Intended for use when Algernon called
; from lisp.  Suppresses Algernon output and allows for collection
; of bindings for variables.
;
(defun tell (predicates &key comment)
  (quietly (a-assert comment predicates)))

; Ask should eventually be modified to take some account
; of the assumptions under which the results are valid ...
;
(defun ask (predicates &key comment collect execute retrieve)
  (when retrieve
    (setq predicates (mapcar #'(lambda (pred) `(:retrieve ,pred))
			     predicates)))
  (silently (a-query comment predicates))

  ;; Translate variables into internal Algernon variables:
  (setq collect (sublis *last-var-alist* collect))
  (setq execute (sublis *last-var-alist* execute))

  (when execute
    (dolist (result *last-results*)
	(let ((execute (substitute-bindings execute (aresult-sub result))))
	  (apply (car execute)
		 (cdr execute)))))
  (if collect
      (let (bindings)
	(dolist (result *last-results*)
	  (pushnew (substitute-bindings collect (aresult-sub result))
		   bindings
		   :test #'equal))
	bindings)
      ;; If nothing to collect then just return t or nil:
      (if *last-results* t nil)))

; The high level query and assert functions:
;
(defun a-query (sentence predicates)
  (let ((result (catch 'error (a-query-under-catch sentence predicates))))
    (cond ((eql result t)
	   result)
	  ((eql result nil)
	   (when (and *cerror-on-failure*
		      (not *algy-recursive-callp*))  ; Ignore failures of recursive calls.
	     (cerror "Ignore failed ~(~a~)."
		     "~@(~a~) failed on predicate: ~(~a~)."
		     *last-op* *last-predicate*))
	   nil)
	  (t
	   (format t "~&~%ALGERNON ERROR: ~@(~a~)~%" result)))))

(defun a-assert (sentence predicates)
  (let ((result (catch 'error (a-assert-under-catch sentence predicates))))
    (cond ((eql result t)
	   result)
	  ((eql result nil)
	   (when (and *cerror-on-failure*
		      (not *algy-recursive-callp*))  ; Ignore failures of recursive calls.
	     (cerror "Ignore failed ~(~a~)."
		     "~@(~a~) failed on predicate: ~(~a~)."
		     *last-op* *last-predicate*))
	   nil)
	  (t
	   (format t "~&~%ALGERNON ERROR: ~@(~a~)~%" result)))))

(defun reset-global-vars (op preds)
  (setq *last-op* op)
  (setq *last-predicates* preds)
  (setq *last-predicate* nil)
  (setq *last-results* nil)
  (setq *last-inserted-values* nil)
  (setq *last-inserted-assumptions* nil)
  (setq *last-deleted-values* nil)
  (setq *last-creations* nil)
  (setq *last-contradictions* nil)
  (setq *unify-count* 0)
  (setq *match-count* 0)
  (setq *rule-count* 0)
  (setq *iteration-count* 0)
  (setq *max-iterations* 0)
  (setq *frame-insertions* 0)
  (setq *frame-accesses* 0)
  (setq *rules-to-complete* nil)
  (setq *top-level* t)
  (setq *back-chain* t)
  (setq *forward-chain* t)
  (setq *no-depnet* nil)
  ;; Then some variables local to algy files which need to be reset: 
  (setq *cur-partitions* nil)
  (setq *cur-frame* nil)
  (setq *cur-slot* nil))

(defmacro with-new-global-vars (op preds &body exp)
  `(let ((*last-op* ,op)
	 (*last-predicates* ,preds)
	 (*last-predicate* nil)
	 (*last-results* nil)
	 (*last-inserted-values* nil)
	 (*last-inserted-assumptions* nil)
	 (*last-deleted-values* nil)
	 (*last-creations* nil)
	 (*last-contradictions* nil)
	 (*unify-count* 0)
	 (*match-count* 0)
	 (*rule-count* 0)
	 (*iteration-count* 0)
	 (*max-iterations* 0)
	 (*frame-insertions* 0)
	 (*frame-accesses* 0)
	 (*rules-to-complete* nil)
	 (*top-level* t)
	 (*back-chain* t)
	 (*forward-chain* t)
	 (*no-depnet* nil)
	 ;; Then some variables local to algy files which we need new coppies of when we recurse:
	 (assump-disjunct nil)
	 (*cur-partitions* nil)
	 (*cur-frame* nil)
	 (*cur-slot* nil)
	 (*as-list* nil)
	 (*qu-list* nil)
	 (*old-as-list* nil)
	 (*old-qu-list* nil))
     ,@exp))

(defun A-QUERY-UNDER-CATCH (sentence predicates)
  (cond ((and @debug (not (consp predicates)))
	 (algy-error
	   (format nil "Illegal query ~(~a~).  Can only query a list of predicates." predicates)))
	((and @debug (not (every #'consp predicates)))
	 (algy-error
	   (format nil "Query of predicate which is not a list: ~(~a~)."
		   (find-if-not #'consp predicates))))
	(t
	 (cond (*algy-recursive-callp*
		(with-new-global-vars
		  'Query predicates
		  (if (internal-query (trace-prep-path (preprocess predicates))
				      (list (new-aresult))
				      t)
		      t
		      nil)))
	       (t
		(pp-sentence 'Query sentence)
		(pp-input predicates)
		(init-trace)
		;; Make sure depnet is reset (should not be needed but just to be safe).
		(reset-dnet)
		(reset-global-vars 'Query predicates)
		(unwind-protect
		    (setq *last-results* 
			  (let ((*algy-recursive-callp* t))
			    (internal-query (trace-prep-path (preprocess predicates))
					    (list (new-aresult))
					    t)))
		  (reset-dnet)
		  (end-trace)
		  (pp-output))
		(if *last-results* t nil))))))

(defun A-ASSERT-UNDER-CATCH (sentence predicates)
  (cond ((and @debug (not (consp predicates)))
	 (algy-error
	   (format nil "Illegal assertion ~(~a~).  Can only assery a list of predicates." predicates)))
	((and @debug (not (every #'consp predicates)))
	 (algy-error
	   (format nil "Assertion of predicate which is not a list: ~(~a~)."
		   (find-if-not #'consp predicates))))
	(t
	 (cond (*algy-recursive-callp*
		(with-new-global-vars
		  'Assert predicates
		  (if (internal-assert (trace-prep-path (preprocess predicates t))
				       (list (new-aresult))
				       t)
		      t
		      nil)))
	       (t
		(pp-sentence 'Assert sentence)
		(pp-input predicates)
		(init-trace)
		;; Make sure depnet is reset (should not be needed but just to be safe).
		(reset-dnet)
		(reset-global-vars 'Assert predicates)
		(unwind-protect
		    (setq *last-results* 
			  (let ((*algy-recursive-callp* t))
			    (internal-assert (trace-prep-path (preprocess predicates t))
					    (list (new-aresult))
					    t)))
		  (reset-dnet)
		  (end-trace)
		  (pp-output))
		(if *last-results* t nil))))))

; RESET-ALGY
;
; Modified 12/6/89 to do no tracing durring the reset.
;
(defun RESET-ALGY ()
  ; First turn off all tracing:
  ; (untrace)
  (trace-off)

  ; Then reset other files:
  (reset-dnet)
  (reset-frames)
  (reset-names)
  (reset-rules)
  (reset-preproc)
  
  ; Reset algy variables:
  (setq *frame-stack* nil)
  (setq *next-frames* nil)
  (setq *prev-frames* nil)
  (setq *command* nil)
  
  ; Reset global vars:
  (reset-global-vars nil nil)

  ; Tell aframes about the facets we will need:
  (dolist (facet *facets*) (make-into-facet facet))
  
  ; And silently declare slots required by the system:
  (dolist (slot @system-slots) (decl-slot (list slot nil)))

  ; Set normal tracing and output:
  (default-trace)
  (normal-output)
  t)

; ALGY-ASK asks the user for a value, and returns a stream containing an extended result.
;
; Complicated extensively 1/18/90 for expert system class.
;
; (Note: bindings in alist of result are not applied to predicate (this should
; be done in calling function)).
;
(defun algy-ask (pred result)
  (cond
    ((variable? (slot pred))
     (algy-error
       (format nil "Illegal predicate (in :ask) ~(~a~) -- slot cannot be the variable (~(~a~))."
	       pred (slot pred))))
    ((variable? (frame pred))
     (algy-error
       (format nil "Illegal predicate (in :ask) ~(~a~) -- frame cannot be the variable (~(~a~))."
	       pred (frame pred))))

    ;; Asking about an uninstantiated pred.
    ((has-variables (value pred))
     (let ((current-values 
            (with-no-depnet (with-no-back-chaining (get-values pred)))))
       (cond
	 ;; Don't ask if there is an old value which (value pred) matches:
	 ((member (value pred) current-values :key #'car :test #'more-general)
	  (extend-with-values result (value pred) current-values))
	 (t
	  (let* ((var (get-variable (value pred)))
		 (type-res (car (remove-assumps 
                                 (with-no-depnet
                                   (with-no-back-chaining 
                                     (get-values `(type-slot ,(slot pred))))))))
		 (set (if type-res (find-corresponding-element var (cdr pred) type-res)))
		 (possible-values (if set
                                    (remove-assumps
                                     (with-no-depnet
                                       (with-no-back-chaining
                                         (get-values `(member ,set)))))))
		 (value nil))

	    (format t "~% Give me a value for ~(~a~) in ~(~a~)" var pred)
	    (if possible-values (format t " [possible values:") (format t ": "))

	    ;; Loop 'till get an acceptable value:
	    (loop
	      ;; Print out list of possible values for var (if known).
	      (when possible-values
		(mapc #'(lambda (x)
			  (format t " ~(~a~)" x))
		      possible-values)
		(format t "] "))

	      (setq value (read *query-io* nil nil))
	      (cond (possible-values
		     (if (find value possible-values)
			 (return)
			 (format t "~% [Enter a value in:")))
		    ((or (framep value)
			 (null set))		; set = nil => Even a non-frame can go in slot.
		     (return))
		    (t
		     (format t "~% ~(~a~) is not a known frame." value)
		     (cond ((y-or-n-p "Do you want to create it ?")
			    (make-new-frame value)
			    (return))
			   (t
			    (format t "~% Give me a value for ~(~a~) in ~(~a~)" var pred)
			    (if possible-values (format t " [possible values:") (format t ": ")))))))

	    (internal-assert (list pred)
			     (extend-with-values result var
						 (list (cons value
							     (new-aresult))))))))))

    ;; Instantiated.
    (t
     (let ((currently-known (with-no-depnet (with-no-back-chaining (known pred)))))
       (cond
	 (currently-known
	  (extend-result result (cdr currently-known)))
	 ((with-no-depnet (with-no-back-chaining (known (negate pred))))
	  nil)
	 ((yes-or-no-p " Is it true that ~(~a~)? " pred)
	  (internal-assert (list pred) (list result)))
	 (t					; User has responded "no".
	  (internal-assert (list (negate pred)) (list result))
	  nil))))))

;;; Find-Corresponding-Element -- Returns the element in l2 corresponding to x in l1.
;;;
;;; One complication: l1 is of form (e1 e2) or (e1 (e2 ... en)).  This is because
;;; of the strange way Algernon internally represents predicates of arity greater
;;; than 2.
;;;
(defun find-corresponding-element (x l1 l2)
  (find-corresponding-element-internal
    x
    (if (< (length l2) 3)
	l1
	(cons (car l1) (cadr l1)))
    l2))

(defun find-corresponding-element-internal (x l1 l2)
  (if (null l1)
      (error "~% Internal Algernon error: Illegal call to ~
             find-corresponding-element:  ~a does not occur in ~a." x l1))
  (if (eql x (car l1))
      (car l2)
      (find-corresponding-element-internal x (cdr l1) (cdr l2))))

(defun get-variable (x)
  (cond ((not (consp x))
	 (if (variable? x) x))
        (t
	 (or (get-variable (car x))
	     (get-variable (cdr x))))))

(defun get-responce (message)
  (if (not (listen)) (format t "~& ~@(~a~) " message))
  (read *query-io* nil nil))

(defun algy-warning (message &optional list)
  (unless *suppress-warnings*
    (format t "~&~% ")
    (if list
	(pp-labeled-list 2
			 (format nil "Algernon Warning --- ~@(~a~)" message)
			 list
			 nil)
	(format t "Algernon Warning --- ~@(~a~)" message))
    (if *algy-break-on-warning*
	(cerror "Continue, doing nothing about it"
		"Algy warning."))))

; Algy-Error: Needs to be called ...
;
(defun algy-error (string)
  (if *algy-break-on-error*
      (cerror "Throw to Algernon top level"
	      (format nil "Algy Error: ~a" string)))
  (throw 'error (format nil "Algy Error: ~a" string)))



; Auxillary routines for browser:
;
(defun show-current-frame ()
  (if (null *frame-stack*)
      (format t "~&  No current frame.~%")
      (pp-frame (car *frame-stack*))))

(defun visit-frames (frame-list)
  (if (null frame-list)
      (format t "~&  No such frames.~%")
      (prog ()
	(setq *frame-stack* (cons (car frame-list) *frame-stack*))
	(setq *prev-frames* nil)
	(setq *next-frames* (cdr frame-list)))))

(defun remove-assumps (pair-list)
  (mapcar #'car pair-list))

(defun profile-kb ()
  (format t "~% Frames: ~a.  Values: ~a.  Rules: ~a."
            (count-frames) (count-values) (count-rules)))


;;; Load-Common-Sense-Kb -- Load background knowledge-base if newer than stored
;;; knowledge-base.
;;;
;;; Expect to find source for kb in @kb-source.
;;; kb stored in @stored-kb.
;;;
(defun load-common-sense-kb ()
  (let* ((kbase-source (concatenate 'string @algy-kb-path @kb-name @lisp-file-extension))
	 (stored-kbase (concatenate 'string @algy-object-path @stored-kb-name @kb-file-extension))
	 (kbase-time (if (probe-file kbase-source)
			 (file-write-date kbase-source)
			 0))
	 (stored-kb-time (if (probe-file stored-kbase)
			     (file-write-date stored-kbase)
			     0)))
    (cond ((> kbase-time stored-kb-time)
	   (format t "~%  Updating stored knowledge-base.~%")
	   (algy-compile-load @kb-name @algy-kb-path) 
	   ;;(minimal-output) (common-sense-facts) (normal-output)
	   (with-no-warnings (with-no-output (common-sense-facts)))
	   (format t "~%  Saving knowledge-base as: ~a." stored-kbase)
	   (dump-kb @stored-kb-name @algy-object-path))
	  (t
	   (load-kb @stored-kb-name @algy-object-path)))))


; Routines to check/change the kb directory:

(defun maybe-change-kb-dir ()
  (if (and (equal *user-path* (namestring (user-homedir-pathname)))
           *user-kb-path*)
    (format t "~% Current knowledge-base directory: ~(~a~).~%" *user-kb-path*)
    (change-kb-dir)))

(defun change-kb-dir ()
  ;; If a new user has logged in then given them a new default:
  (if (not (eql *user-path* (namestring (user-homedir-pathname))))
    (setq *user-kb-path* (namestring (user-homedir-pathname))))
  (setq *user-path* (namestring (user-homedir-pathname)))
  (get-new-kb-dir))

(defun get-new-kb-dir ()
  (format t "~% Current knowledge-base directory: ~(~a~).~%" *user-kb-path*)
  (when (y-or-n-p " Do you want to change the knowledge-base directory? ")
    (format t "~% Your directory for storing knowledge-bases (end with ~a): "
            @path-terminator)
    (setq *user-kb-path* (read-line *query-io* nil nil))))
