;;; -*- Mode:Common-Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: acount.lisp,v 1.1 92/04/16 09:29:58 clancy Exp $

;;;                        ****** ACOUNT ******

;;; Acounting file to count rule applications, and queries.  Dedicated to Tracy.

;;; We assume frames and slots are comparable using eq.

(defparameter *accountants-only-slow-things-down* t)

(defparameter *total-rule-count* 0 "Total number of rule applications.")
(defparameter *rule-count-by-slot* nil "Alist of slots and the number of rules fired for each.")
(defparameter *rule-count-by-frame* nil "Alist of frames and the number of rules fired for each.")

(defparameter *total-query-count* 0 "Total number of queries.")
(defparameter *query-count-by-slot* nil "Alist of slots and the number of queries for each.")
(defparameter *query-count-by-frame* nil "Alist of frames and the number of queries for each.")

(defun enable-accountant ()
  (setq *accountants-only-slow-things-down* nil))

(defun disable-accountant ()
  (setq *accountants-only-slow-things-down* t))

(defun reset-accountant ()
  (format t "~%~%Final counts:")
  (show-counts)
  (setq *total-rule-count* 0)
  (setq *rule-count-by-slot* nil)
  (setq *rule-count-by-frame* nil)
  
  (setq *total-query-count* 0)
  (setq *query-count-by-slot* nil)
  (setq *query-count-by-frame* nil))


(defun rule-accountant (rule result)
  (unless *accountants-only-slow-things-down*
    (incf *total-rule-count*)
    (let ((slot (rule-slot rule result))
	  (frame (rule-frame rule result)))
      (setq *rule-count-by-slot* (alist-inc *rule-count-by-slot* slot))
      (setq *rule-count-by-frame* (alist-inc *rule-count-by-frame* frame)))))

;;; Bug - This routine really should take a result and apply it to pred.
;;;
;;; Modified 3/6/91 to count predicates surrounded by :no-completion.
;;;
(defun query-accountant (pred)
  (unless *accountants-only-slow-things-down*
    ;; Special case retrieve and no-completion
    ;; (prehaps should also deal with other special forms):
    (if (or (eq (car pred) :retrieve)
	    (eq (car pred) :no-completion))
	(query-accountant (second pred))
	(let ((slot (slot pred))
	      (frame (frame pred)))
	  (incf *total-query-count*)
	  (setq *query-count-by-slot* (alist-inc *query-count-by-slot* slot))
	  (setq *query-count-by-frame* (alist-inc *query-count-by-frame* frame))))))

(defun alist-inc (alist key)
  (let ((pair (assoc key alist :test #'eq)))
    (cond (pair
	   (incf (cdr pair))
	   alist)
	  (t
	   (acons key 1 alist)))))

(defun accountant ()
  (format t "~%Current counts:")
  (show-counts))

(defun show-counts (&optional slots frames)
  (format t "~%  Total rules fired = ~a" *total-rule-count*)

  (format t "~%  Top five slots (by number of rule applcations):")
  (setq *rule-count-by-slot* (sort *rule-count-by-slot* #'> :key #'cdr))
  (let ((counts *rule-count-by-slot*))
    (dotimes (x (min 5 (length counts)))
      (show-selected-count (car (first counts))
			   (cdr (first counts)))
      (pop counts)))
  
  (when slots
    (format t "%  Individual slots:")
    (dolist (slot slots)
      (show-selected-count slot (cdr (assoc slot *rule-count-by-slot* :test #'eq)))))
  
  (when frames
    (format t "%  Individual frames:")
    (dolist (frame frames)
      (show-selected-count frame (cdr (assoc frame *rule-count-by-frame* :test #'eq)))))
  
  (format t "~%  Total number of queries = ~a" *total-query-count*)

  (format t "~%  Top five slots (by number of queries):")
  (setq *query-count-by-slot* (sort *query-count-by-slot* #'> :key #'cdr))
  (let ((counts *query-count-by-slot*))
    (dotimes (x (min 5 (length counts)))
      (show-selected-count (car (first counts))
			   (cdr (first counts)))
      (pop counts))))

(defun show-selected-count (name value)
  (format t "~%     ~(~a~) --  ~a" name value))

	   
(defun show-all-counts (alist)
  (dolist (pair alist) (show-selected-count (car pair) (cdr pair))))