;;; -*- Mode:Common-Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: acontra.lisp,v 1.1 92/04/16 09:29:55 clancy Exp $

;;;                        ****** ACONTRA ******

; Recognizes and deals with contradictions in the knowledge-base.

;;; Modified 11/19/89 to add coreferent and cf-member links when number restrictions
;;; violated.
;;;
;;; Modofied 4/17/90 to allow user to provide functions to choose which assumptions to
;;; drop when resolving a contradiction.  The user should push onto *contradiction-resolvers*
;;; their functions.  If *contradiction-resolvers* includes foo (or, more accurately, if the user
;;; has executed "(push (symbol-function 'foo) *contradiction-resolvers*)") then foo should
;;; be declared with:
;;;
;;;       (defun foo (pred assump-ll not-assump-ll conjunct) ...)
;;;
;;; which means that pred is believed under the assump-ll, assump-ll and the negation
;;; of pred is believed under the assump-ll, not-assump-ll.  pred MAY be negated (i.e. of form
;;; (not (p t1 ... tn))).  conjunct is the conjunction of assump-ll and not-assump-ll.
;;; foo should return a nil or a list of assumptions to drop.  If it returns a list, then dropping the
;;; assumptions in the list MUST be sufficient to disprove EITHER assump-ll or
;;; neg-assump-ll (or, equivalently, to disprove each list in conjunct)
;;; (be careful as NO error checking is currently done on this).  If foo returns nil then
;;; the next function in *contradiction-resolvers*  is applied.  *contradiction-resolvers* 
;;; ends with a function which asks the user which assumption to drop.
;;;
;;; There are two problems with the current approach:
;;;
;;;   1. There is no way for one contradiction resolver to tell you how to choose between
;;;      the assumptions in one list in conjunct, and other resolvers to deal with the
;;;      others.  A resolver must resolve the entire contradiction or have no opinion.
;;;   2. If conjunct contains a list of length 1 then clearly that assumption must be
;;;      dropped, but each resolver must check this case ...

; Vars for this module:
;
(defvar assump-disjunct nil)   ; This does not seem to be used anywhere ?? (jc 1/8/91)
(defparameter *contradiction-resolvers* nil)

; CONSISTENT:  Check to see that pred is consistently believed, and
;   try to resolve any contradictions found.
;
;  * Error if contradiction without assumptions found.
;  * Contradiction with a single assumption -> assert negation of assumption.
;  * Contradiction with multiple assumptions: give up at random an assumption which
;    will remove only one side of contradiction (if possible).
;
; Returns assump-ll pred is believed under after all contradictions resolved.
;
(defun consistent (pred)
  (let* ((neg-pred (negate pred))
         (not-result (cdr (with-no-depnet (known neg-pred))))
         (result (cdr (with-no-depnet (with-no-back-chaining (known pred)))))
         (not-assump-ll (if not-result (aresult-assump-ll not-result)))
         (assump-ll (if result (aresult-assump-ll result))))
    
    (when assump-ll				; Only proceed if pred still believed.
      (if not-assump-ll
	  (let ((conjunct (conjunct-assump-ll assump-ll not-assump-ll)))
	    (trace-contradiction pred neg-pred assump-ll not-assump-ll)
	    (push (list pred neg-pred) *last-contradictions*)
	    
	    (if (and (member nil assump-ll :test #'eq) (member nil not-assump-ll :test #'eq))
		(algy-error (format nil "Contradiction ~a, ~a" pred (negate pred)))
		(let (assumptions-to-drop
		      (cr-functions *contradiction-resolvers*))
		  
		  ;; Look for a contradiction resolver which has an opinion:
		  (loop				; (format t "cr-functions = ~a~%" cr-functions)
		    (if assumptions-to-drop (return))
		    (if (not cr-functions)
			(algy-error "Internal Error -- No contradiction resolver succeeded."))
		    (setq assumptions-to-drop
			  (funcall (car cr-functions) pred assump-ll not-assump-ll conjunct))
		    (setq cr-functions (cdr cr-functions)))
		  
		  (dolist (assump assumptions-to-drop)
		    (drop-assumption assump)
		    ;; Assert negations of assump under assumptions appearing with it in conjunct:
		    (dolist (assump-list conjunct)
		      (when (member assump assump-list :test #'equal)
			(let ((result (new-aresult)))
			  (setf (aresult-assump-ll result)
				(list (remove assump assump-list :test #'equal)))
			  (assert-predicate (negate assump) result)))))))
	    
	    
	    ;; Find out how things turned out and return final assump-ll for pred:
	    (setq result (cdr (with-no-depnet (with-no-back-chaining (known pred)))))
	    (if result (aresult-assump-ll result)))
	  
	  ;; "Normal" case (no contradiction).
	  assump-ll))))

(defun drop-assumption (assump)
  (trace-assumption-drop assump)
  (delete-value assump))


;;; Contradic Resolvers.  The function appear in the file from the bottom of the list up
;;; (that is user-assump-drop is the last one tried ...).

;;; User-Assump-Drop -- Queries the user to find out which assumption to drop.
;;; A more elegant user interfact could easily be devised ...
;;;
(defun user-assump-drop (pred assump-ll not-assump-ll conjunct)
  (format t "~% Contradiction detected.")
  (format t "~% ~(~a~) is believed under assumptions: ~(~a~)" pred assump-ll)
  (format t "~% ~(~a~) is believed under assumptions: ~(~a~)" (negate pred) not-assump-ll)
  (format t "~% Specify by number which assumption to drop from each of the following lists:")
  (let (bad-assumps)
    (dolist (assumps conjunct)
      (if (eql (length assumps) 1)
        (push (car assumps) bad-assumps)
        (progn
          (format t "~%   ~(~a~) [1-~a] " assumps (length assumps))
          (push (nth (- (read) 1) assumps) bad-assumps))))
    (trace-contradiction-resolution 'user-assump-drop bad-assumps)
    bad-assumps))

(pushnew (symbol-function 'user-assump-drop) *contradiction-resolvers*)

;;; Dependent-Assumps -- Recognizes nested assumptions and returns
;;; the inside assumption.  Nested assumptions arise most often in proofs by
;;; contradiction:
;;;
;;;     assume(p)
;;;       assume(q)
;;;         assume(r)
;;;           -><-
;;;
;;; In such cases we want to always drop r.
;;; Such cases can be recognized because r appears in the knowledge-base taged
;;; with p and q, and q is tagged with p.
;;;
(defun dependent-assumps (pred assump-ll not-assump-ll conjunct)
  (declare (ignore pred) (ignore assump-ll) (ignore not-assump-ll))
  (let (bad-assumps)
    (dolist (assumps conjunct)
      ;; If the assumptions are nested then there is an ordering
      ;; of assumps such that a1 depends on (a2 ... an), a2 depends on (a3 ... an), ...
      ;; We detect this case by first sorting by dependency and then testing
      ;; that each assump depends on the next:
      (if (eql (length assumps) 1)
        (push (car assumps) bad-assumps)
        (let ((sorted-assumps (sort (copy-list assumps) #'assump-depends)))
          (if (every #'(lambda (assump)
                         (let ((later-assumps (cdr (member assump sorted-assumps :test #'equal))))
                           (or (null later-assumps)
                               (assump-depends assump (car later-assumps)))))
                     sorted-assumps)
            (push (car sorted-assumps) bad-assumps)
            (progn
              (setq bad-assumps nil)
              (return))))))
    (trace-contradiction-resolution 'dependent-assumps bad-assumps)
    bad-assumps))

(pushnew (symbol-function 'dependent-assumps) *contradiction-resolvers*)

;;; Assump-Depends -- Returns non-nil iff a1 depends on a2.
;;; As an efficiency hack goes directly to aframe level to check assumptions
;;; (this avoids the unnecessary check to see which assumptions are currently
;;; believed, which is done in aidg).
;;;
;;; Not currently cleaver enough to detect the case in which a1 depends on an
;;; assumption which depends on a2.
;;;
(defun assump-depends (a1 a2)
  (member a2
          (cadr (assoc (value a1) (fget (frame a1) (slot a1) (facet a1)) :test #'equal))
          :test #'equal))

