;;; -*- Syntax: Common-lisp; Package: USER -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: aassump.lisp,v 1.1 92/04/16 09:29:51 clancy Exp $

;;;                        ****** AASSUMP ******

;  Functions to manipulate assumptions.
;
; In the value facet of the algernon knowledge-base we keep an alist of values
; and lists of lists of assumptions making the value true.
;
; Example: Suppose V1 is always true, V2 is true only under
; the assumption A1, and V3 is true under the assumtion A2 or
; A3 and A4:
;
;         ((V1 ()) (V2 (A1)) (V3 (A2) (A3 A4))).
;
; This is understood as meaning:
;
;         V1
;         V2 <- A1
;         V3 <- A2 OR (A3 and A4)
;
; These routines manage such alists and lists of lists of assumptions.
; Some of these routines have distructive counter-parts ; if a routine
; is called x then its distructive counter-part is called nx.  The distructive
; versions will only distroy their Second argument.
;
; Several of these routines do things in fairly dumb ways (using the lisp functions union and
; subset).  If we ever deal with large numbers of assumptions this should be improved.

; DISJUNCT-ASSUMP-LL: Takes two assump-ll's and does a union with subsumptions checking.
;
(defun disjunct-assump-ll (assump-ll1 assump-ll2)
  (if (or (equal assump-ll1 '(nil))
          (equal assump-ll2 '(nil)))
    (list nil)
    (do ((input assump-ll1 (cdr input))
         (result assump-ll2 (disjunct-assumps (car input) result)))
        ((null input) result))))

; DISJUNCT-ASSUMPS: Add a list of assumptions to an assump-ll.
;
(defun disjunct-assumps (assumps assump-ll)
  (let ((temp-assump-ll (remove-if #'(lambda (x) (subsetp assumps x :test #'equal))
				   assump-ll)))
    (cond ((member assumps temp-assump-ll
		   :test #'(lambda (x y) (subsetp y x :test #'equal)))
	   temp-assump-ll)
          (t
	   (cons assumps temp-assump-ll)))))


; NDISJUNCT-ASSUMPS:
; Distructively do subsumption checking between a list of assumptions
; and a list of lists of assumptions.
;
; assumps is a list of assumptions and assumps-list
; is a list of lists of assumptions.
;
; Returns new list of lists of assumptions.
;
(defun ndisjunct-assumps (assumps assumps-list)
  (if assumps
    (let ((temp-assumps-list (delete-if #'(lambda (x) (subsetp assumps x :test #'equal))
                                        assumps-list)))
      (cond ((member assumps temp-assumps-list
                     :test #'(lambda (x y) (subsetp y x :test #'equal)))
             temp-assumps-list)
            (temp-assumps-list  ; if there is something left of the original assumptions
             (rplacd (last temp-assumps-list) (list assumps))
             temp-assumps-list)
            (t
             (list assumps))))
    (list nil)))



; CONJUNCT-ASSUMP-LL:
; Returns assump-ll which is the conjunction of assump-ll1 and assump-ll2.
;
; This routine assumes that disjuct-assump-ll does subsumtion checking
; amoung the elements of its first argument (which it currently does -- but
; this is not documented ...).
;
; This routine does NOT check for contradictions.  That is, it happily
; conjuncts ((a)) and (((not a))) to get ((a (not a))) ...
; The reason for this is that I do not think such cases can occur
; under the current implementation.
;
(defun conjunct-assump-ll (assump-ll1 assump-ll2)
  (cond ((equal assump-ll1 '(nil))
         (copy-assump-ll assump-ll2))
        ((equal assump-ll2 '(nil))
         (copy-assump-ll assump-ll1))
        (t
         (do ((input assump-ll1 (cdr input))
              (result nil (disjunct-assump-ll (conjunct-assumps-internal (car input) assump-ll2)
                                              result)))
             ((null input) result)))))

(defun copy-assump-ll (assump-ll)
  (mapcar #'(lambda (x) (copy-list x)) assump-ll))

; CONJUNCT-ASSUMPS
;
(defun conjunct-assumps (assumps assump-ll)
  (do ((input assump-ll (cdr input))
       (result nil (disjunct-assumps (union assumps (car input) :test #'equal)
				     result)))
    ((null input) result)))

; conjunct-assumps-internal: Faster version which does not do subsumption checking.
;
(defun conjunct-assumps-internal (assumps assump-ll)
  (do ((input assump-ll (cdr input))
       (result nil (cons (union assumps (car input) :test #'equal)
			 result)))
    ((null input) result)))

; ASSUMP-LL-IMPLIES
; Returns true if assump-ll1 -> assump-ll2.
; This is true if every list of assumptions in assump-111 is a superset of
; some set of assumptions in assump-112, or equivalently, for every list of
; assumptions in assump-ll1 there is a set of assumptions in assump-ll2
; which is a subset of it.
;
(defun assump-ll-implies (assump-ll1 assump-ll2)
  (or (null assump-ll1)
      (equal assump-ll2 '(nil))
      (and (member (car assump-ll1) assump-ll2
                   :test #'(lambda (x y) (subsetp y x :test #'equal)))
           (assump-ll-implies (cdr assump-ll1) assump-ll2))))
