;;;   -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; ROYAL FAMILY
;;;
;;; A slightly longer example on reasoning about genelogy.
;;;
;;; Mixed if-added and if-needed rules, and predicates at different
;;; levels of abstraction. 

(defun facts-about-families ()
  (a-assert "Family relations."
     '((:slot child (people people)
              :comment "(child p1 p2) = A child of p1 is p2.")
       (:slot son (people people)
              :comment "(son p1 p2) = A son of p1 is p2")
       (:slot daughter (people people)
              :comment "(daughter p1 p2) = A daughter of p1 is p2")

       (:slot parent (people people)
              :cardinality 2
              :inverse child
              :comment "(parent p1 p2) = A parent of p1 is p2")
       (:slot father (people people)
              :cardinality 1
              :comment "(father p1 p2) = The father of p1 is p2")
       (:slot mother (people people)
              :cardinality 1
              :comment "(mother p1 p2) = The mother of p1 is p2")

       (:slot sibling (people people)
              :inverse sibling
              :comment "(sibling p1 p2) = A sibling of p1 is p2.")
       (:slot brother (people people)
              :comment "(brother p1 p2) = A brother of p1 is p2.")
       (:slot sister (people people)
              :comment "(sister p1 p2) = A sister of p1 is p2.")

       (:slot grandchild (people people)
              :comment "(grandchild p1 p2) = A grandchild of p1 is p2.")
       (:slot grandson (people people)
              :comment "(grandson p1 p2) = A grandson of p1 is p2")
       (:slot grandaughter (people people)
              :comment "(granddaughter p1 p2) = A grandaughter of p1 is p2")

       (:slot grandparent (people people)
              :cardinality 4
              :inverse grandchild
              :comment "(grandparent p1 p2) = A grandparent of p1 is p2")
       (:slot grandfather (people people)
              :cardinality 2
              :comment "(grandfather p1 p2) = A grandfather of p1 is p2")
       (:slot grandmother (people people)
              :cardinality 2
              :comment "(grandmother p1 p2) = A grandmother of p1 is p2")

       (:slot uncle (people people)
              :comment "(uncle p1 p2) = An uncle of p1 is p2.")
       (:slot aunt (people people)
              :comment "(aunt p1 p2) = An aunt of p1 is p2.")
       ;;
       ;; Sex specific links are "shorthands" from simpler links:
       (:rules people
	((father ?x ?f) -> (parent ?x ?f) (gender ?f male))
	((father ?x ?f) <- (parent ?x ?f) (gender ?f male))
        ;;
	((mother ?x ?f) -> (parent ?x ?f) (gender ?f female))
	((mother ?x ?f) <- (parent ?x ?f) (gender ?f female))
        ;;
	((son ?x ?s) -> (child ?x ?s) (gender ?s male))
	((son ?x ?s) <- (child ?x ?s) (gender ?s male))
        ;;
	((daughter ?x ?d) -> (child ?x ?d) (gender ?d female))
	((daughter ?x ?d) <- (child ?x ?d) (gender ?d female))
        ;;
	((brother ?x ?b) -> (sibling ?x ?b) (gender ?b male))
	((brother ?x ?b) <- (sibling ?x ?b) (gender ?b male))
        ;;
	((sister ?x ?b) -> (sibling ?x ?b) (gender ?b female))
	((sister ?x ?b) <- (sibling ?x ?b) (gender ?b female))
        ;;
	((grandfather ?x ?gf) -> (grandparent ?x ?gf) (gender ?gf male))
	((grandfather ?x ?gf) <- (grandparent ?x ?gf) (gender ?gf male))
        ;;
	((grandmother ?x ?gf) -> (grandparent ?x ?gf) (gender ?gf female))
	((grandmother ?x ?gf) <- (grandparent ?x ?gf) (gender ?gf female))
        ;;
	((grandson ?x ?gs) -> (grandchild ?x ?gs) (gender ?gs male))
	((grandson ?x ?gs) <- (grandchild ?x ?gs) (gender ?gs male))
        ;;
	((grandaughter ?x ?gs) -> (grandchild ?x ?gs) (gender ?gs female))
	((grandaughter ?x ?gs) <- (grandchild ?x ?gs) (gender ?gs female))
        ;;
        ;; Aunt and Uncle are a bit different (there is no unisex term):
	((uncle ?x ?u) -> (gender ?u male))
        ((aunt ?x ?a) -> (gender ?a female))
        ;;
        ;; Then the rules to infer the more complex relations:
	((grandfather ?a ?c) <- (father (parent ?a) ?c))
	((grandmother ?a ?c) <- (mother (parent ?a) ?c))
	((grandson ?a ?c)    <- (son (child ?a) ?c))
	((grandaughter ?a ?c)  <- (daughter (child ?a) ?c))
        ;;
	((sibling ?x ?y)
	 <-
	 (child (parent ?x) ?y) (:NEQ ?x ?y) (:ASSUME (not (coreferent ?x ?y))))
        ;;
	((uncle ?x ?u) <- (brother (parent ?x) ?u))
	((uncle ?x ?u) <- (husband (aunt ?x) ?u))
	((aunt ?x ?a) <- (sister (parent ?x) ?a))
	((aunt ?x ?a) <- (wife (uncle ?x) ?a))
        ;;
	((husband ?w ?h)
	 <-
	 (gender ?w female) (spouse ?w ?h) (gender ?h male))
	((wife ?h ?w)
	 <-
	 (gender ?h male) (spouse ?h ?w) (gender ?w female))))))

;;; And the family tree of the British royal family.
;;;
(defun facts-about-royal-family ()
  (facts-about-families)
  (a-assert "Royal family."
	     '((:create ?ch Charles)
	       (:create ?di Diana)
	       (:create ?ha Harry)
	       (:create ?wi William)
	       (:create ?ph Philip Mountbatten)
	       (:create ?el Elizabeth II)
	       (:create ?an Andrew)
	       (:create ?sa Sarah)
	       ;;
	       (wife ?ch ?di)
	       (son ?ch ?ha)
	       (son ?ch ?wi)
	       (son ?di ?ha)
	       (son ?di ?wi)
	       (father ?ch ?ph)
	       (mother ?ch ?el)
	       (wife ?ph ?el)
	       (son ?ph ?an)
	       (son ?el ?an)
	       (wife ?an ?sa))))



;;; ROYAL FAMILY QUERIES
;;;
(defun queries-about-royal-family ()
  (a-query "Who are William's grandfathers and grandmothers?"
	   '((grandfather William ?gf)
	     (grandmother William ?gm)))
  ;;
  (a-query "Who are William's aunts and uncles?"
	   '((uncle William ?u)
	     (aunt William ?v)))
  ;;
  (a-query "Who are William's parents?"
	   '((parent William ?u)))
  ;;
  (a-query "Are all Charles' children male ?"
	   '((:ALL-PATHS ((child Charles ?b)) ((gender ?b male)))))
  ;;
  (a-query "Are all Charles' children female ?"
	   '((:ALL-PATHS ((child Charles ?b)) ((gender ?b female))))))