; -*- LISP -*-

;;;; Window support for ZGRAPH
;;  H. Kim, University of Illinois 

;; Make a window that simulates the normal Symbolics
;; window system.  

;;;; here there be FLAVORS

;; Sorry, that's just the easiest thing to use in Lucid right now.

(in-package 'user)

;; The vertical organization is always command menu on top, documentation
;; window on bottom, lisp listener above that, and the rest taken up by
;; the graph display port.

;; The height of the Zgraph description and display panes qare
;;  screen height - menu height - lisp height.
;; The vertical positions are computed based on these relative heights.
;; This allows users to easily customize zgraph.
;; (Kim: If the lucid window package allows it, it might be nice to
;;   offer the option of re-sizing while you are in zgraph. - KDF)

(defvar *zgraph-sidebar-size* 8)
(defvar *zgraph-screen-width* 1018)  ;; Determined by mouse hacking 
(defvar *zgraph-screen-height* 760)  ;; ditto

(defvar *zgraph-bitmap-width* 2000)  ;; Width of underlying bitmap
(defvar *zgraph-bitmap-height* 2000) ;; Height of underlying bitmap
(defvar *zgraph-scroll-bitmap-width* 1010)  ;; Width of underlying bitmap
(defvar *zgraph-scroll-bitmap-height* 2800) ;; Height of underlying bitmap

(defvar *zgraph-menu-height* (+ (font-height *default-font*) 10))
(defvar *zgraph-menu-width* *zgraph-screen-width*)    ;; All the way across
(defvar *zgraph-doc-height* (+ (font-height *default-font*) 2))
(defvar *zgraph-doc-width* *zgraph-screen-width*)     ;; Ditto
(defvar *zgraph-lisp-height* 120) ;; Remainder of screen 
(defvar *zgraph-lisp-width* *zgraph-screen-width*)
(defvar *zgraph-des-height* 600) ;; Large chunk of screen space 
(defvar *zgraph-des-width* *zgraph-screen-width*)
(defvar *zgraph-scroll-height* (- *zgraph-des-height* 50))
(defvar *zgraph-scroll-width* 400)

(defvar *zgraph-display-frame* nil)
(defvar *zgraph-display-pane* nil)
(defvar *dialog-box* nil)


(defflavor zgraph-display-frame
  ((zgraph-des-height *zgraph-des-height*)
   (zgraph-lisp-height *zgraph-lisp-height*))
  ()
  :gettable-instance-variables
  :settable-instance-variables) 

(defmacro default-mouse-enter-method ()
  `#'(lambda (viewport active-region mouse-event x y)
       (declare (ignore mouse-event x y))
       (bitblt-region (viewport-bitmap viewport) active-region
		      (viewport-bitmap viewport) active-region
		      boole-c1)
       (send *documentation-pane* :clear-window)
       (if (stringp documentation)
	   (send *documentation-pane* :draw-string documentation)
	   (send *documentation-pane* :draw-string
		 "Should be docs here!"))))

(defmacro default-mouse-exit-method ()
  `#'(lambda (viewport active-region mouse-event x y)
       (declare (ignore mouse-event x y))
       (bitblt-region (viewport-bitmap viewport) active-region
		      (viewport-bitmap viewport) active-region
		      boole-c1)
       (send *documentation-pane* :clear-window)))

(defmacro default-mouse-right-down-method ()
  `#'(lambda (viewport active-region mouse-event x y)
       (declare (ignore viewport active-region mouse-event x y))
       (send *zgraph-display-pane* value)))


(defflavor zgraph-window-flavor
  ((window nil)
   (currently-panning? NIL)
   (who-line-documentation-when-not-panning-desc
     "LHold: Fast Pan ,   R: Select Highlited Vertex or Edge,   L: Move Highlited Vertex")
   (who-line-documentation-when-not-panning
     "LHold: Pan by holding down Mouse-L while moving mouse around window,  L: Move Highlited Vertex")
   (who-line-documentation-when-panning
     "Rub mouse on window to pan.  Release mouse button to stop panning.")
   (window-edge-tolerance 10)
   (x-displacement 0)
   (y-displacement 0)
   (acceleration 2)
   (horizontal-pan-% .5)
   (vertical-pan-% .5))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defun zgraph-window-flavor-init (x)
  (send x :set-window ;; This builds the plotting window
	(make-window :x 0 :y *zgraph-menu-height*
		     :width *zgraph-bitmap-width*
		     :height *zgraph-bitmap-height*
		     :viewport-width (- *zgraph-screen-width*
					*zgraph-sidebar-size*)
		     ;; Height is what's left over from the rest.
		     :viewport-height (send *zgraph-display-frame* 
					    :zgraph-des-height)
		     :initial-font "MEDIUM-ROMAN"
        	     :parent (root-viewport) :scroll t :activate nil)))

(defmethod (zgraph-window-flavor :INSIDE-SIZE) ()
  (values (bitmap-width window) (bitmap-height window)))

(defmethod (zgraph-window-flavor :DRAW-CIRCLE) (x y radius)
  (draw-circle window (make-position x y) radius))

(defmethod (zgraph-window-flavor :DRAW-STRING) (text x-start y-start x-end y-end foo font op)
  (unless (stringp text) (setq text (format nil "~A not a string -- zgraph-window-flavor :DRAW-STRING"
					    text)))
  (stringblt window (make-position x-start (+ y-start (font-height font))) font text :operation op))

(defmethod (zgraph-window-flavor :DRAW-LINE) (x-from y-from x-to y-to)
  (draw-line window (make-position x-from y-from) (make-position x-to y-to)))

(defun display-pane-inside-size ()
  (values (- *zgraph-screen-width*
	     *zgraph-sidebar-size*)
	  (send *zgraph-display-frame* 
		:zgraph-des-height)))

(defun display-pane-inside-width () 
  (- *zgraph-screen-width* *zgraph-sidebar-size*)) ;; Was 1010

(defun display-pane-inside-height () 
  (send *zgraph-display-frame* 
	:zgraph-des-height))

(defmethod (zgraph-window-flavor :CLEAR-WINDOW) ()
  (clear-bitmap-active-regions window)
  (clear-bitmap window)
  (make-active-region
     (make-region :x 0 :y 0
		  :width *zgraph-bitmap-width*
		  :height *zgraph-bitmap-height*)
     :bitmap window
     :mouse-right-down
     #'(lambda (viewport active-region mouse-event x y)
         (declare (ignore viewport active-region mouse-event x y))
	 (when (send *zgraph-display-pane* :graphics-object-being-moved)
	   (send *zgraph-display-pane* :move-graphics-object)))
   
     :mouse-left-down
     #'(lambda (viewport active-region mouse-event x y)
	 (declare (ignore viewport active-region mouse-event x y))
	 (setq currently-panning? T)
	 (let ((the-string (send self :who-line-documentation-string)))
	   ;; Passing around a non-string is a disaster, so be paranoid.
	   (send *documentation-pane* :draw-string
		 (if (stringp the-string) the-string "No documentation here."))) 
	 (LET ((hit-window-edge? (send self :pan-around)))
	   (WHEN hit-window-edge?
	     (send self hit-window-edge?))))

     :mouse-left-up
     #'(lambda (viewport active-region mouse-event x y)
         (declare (ignore viewport active-region mouse-event x y))
	 (setq currently-panning? NIL)
	 (let ((the-string (send self :who-line-documentation-string)))
	   (send *documentation-pane* :draw-string
		 (if (stringp the-string) the-string
		   "No documentation here."))))

     :mouse-enter-region
     #'(lambda (viewport active-region mouse-event x y)
         (declare (ignore viewport active-region mouse-event x y))
	 (when (send *zgraph-display-pane* :graph)
	   (send *documentation-pane* :draw-string
		 (send self :who-line-documentation-string))))

     :mouse-exit-region
     #'(lambda (viewport active-region mouse-event x y)
         (declare (ignore viewport active-region mouse-event x y))
	 (send *documentation-pane* :clear-window))))

(defmethod (zgraph-window-flavor :left) ()
  (send self :pan (* horizontal-pan-% (display-pane-inside-width)) 0 "left..."))

(defmethod (zgraph-window-flavor :right) ()
  (send self :pan (- (* horizontal-pan-% (display-pane-inside-width))) 0 "right..."))

(defmethod (zgraph-window-flavor :up) ()
  (send self :pan 0 (* vertical-pan-% (display-pane-inside-height))  "up..."))

(defmethod (zgraph-window-flavor :down) ()
  (send self :pan 0 (- (* vertical-pan-% (display-pane-inside-height)))  "down..."))

(defmethod (zgraph-window-flavor :pan) (x y message)
  (WHEN (send *zgraph-display-pane* :graph)
    (let ((graph (send *zgraph-display-pane* :graph)))
      (WHEN (MEMQ :print-graph-computation-messages *graph-debug-actions*)
	(format t "~%Panning further ~a" message))
      (SEND graph :pan x y)
      (send *graph-output* :CLEAR-WINDOW)
      (SEND graph :draw)
      (WHEN (MEMQ :print-graph-computation-messages *graph-debug-actions*)
	(format t "done."))
      (terpri))))

(defmethod (zgraph-window-flavor :update-image) ()
  (setq (viewport-bitmap-x-offset window) x-displacement
	(viewport-bitmap-y-offset window) y-displacement))

(defmethod (zgraph-window-flavor :update-displacement) (x-delta y-delta &OPTIONAL absolute?)
  (setq x-delta (round x-delta)
	y-delta (round y-delta))
  (cond (absolute?
	  (setq  x-displacement x-delta
		 y-displacement y-delta))
	(T
	 (INCF x-displacement x-delta)
 	 (INCF y-displacement y-delta)))
  (let ((lose? NIL)
	(max-window-x (bitmap-width window))
	(max-window-y (bitmap-height window)))
    (WHEN (MINUSP x-displacement)
      (setq  x-displacement 0
	     lose? :LEFT))
    (WHEN (MINUSP y-displacement)
      (setq  y-displacement 0
	     lose? :UP))
    (WHEN (> x-displacement max-window-x)
      (setq  x-displacement max-window-x
	     lose? :RIGHT))
    (WHEN (> y-displacement max-window-y)
      (setq  y-displacement max-window-y
	     lose? :DOWN))
    lose?))

(defmethod (zgraph-window-flavor :pan-around) ()
  (LET ((previous-mouse-x *mouse-x*)
	(previous-mouse-y *mouse-y*)
	hitting-window-edge?)
    (setq currently-panning? T)
    (catch 'panning-past-window-edge
      (UNWIND-PROTECT
        (LOOP
	  (WHEN (AND hitting-window-edge? 
	             (EQ (LENGTH hitting-window-edge?) window-edge-tolerance))
	    (THROW 'panning-past-window-edge (CAR hitting-window-edge?)))
	  (COND
	    ;; If user releases the mouse button, exit loop.
	    ((ZEROP *mouse-buttons*)
	     (RETURN T))
	    (T
	     ;;Save these incase the mouse moves during computation of
	     ;;:UPDATE-DISPLACEMENT
	     (LET* ((x *mouse-x*)
		    (y *mouse-y*)
		    ;;Set SELF's displacement to the delta between the current mouse
		    ;;position and the position it was in when we were called,
		    ;; times the acceleration factor.
		    (result (SEND SELF :update-displacement
				  (* acceleration (- previous-mouse-x x))
				  (* acceleration (- previous-mouse-y y)))))
	       (WHEN result
		 (PUSH result hitting-window-edge?))
	       (SETQ previous-mouse-x x
		     previous-mouse-y y)
	       (SEND SELF :update-image)))))
	(SETQ currently-panning? NIL))
      ;;Return NIL, unless the THROW passes a value for the CATCH to return.
      NIL)))

(defmethod (zgraph-window-flavor :who-line-documentation-string) ()
  (IF currently-panning?
      who-line-documentation-when-panning
      (IF (send *zgraph-display-pane* :description-pane-active?)
          who-line-documentation-when-not-panning-desc
          who-line-documentation-when-not-panning)))


;; Zgraph menu pane flavor

(defflavor zgraph-menu-pane
  ((window nil))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defmethod (zgraph-menu-pane :CLEAR-WINDOW) ()
  (clear-bitmap-active-regions window)
  (clear-bitmap window))

(defun zgraph-menu-pane-init (x)
  (send x :set-window
	(make-window :x 0 :y 0 :width *zgraph-screen-width*
		     :height *zgraph-menu-height*
		     :initial-font "MEDIUM-ROMAN" 
		     :parent (root-viewport)  :activate nil)))

(defvar *command-menu*
  '(("Create Graph" :create-graph "Create and display a graph of some pre-specified type")
    ("Display Graph" :display-graph "Display a previlusly created graph")
    ("Status" :change-status "Alter the operation of the graph displayer.")
    ("Zoom" :zoom "Zoom in or out on the currently displayed graph.")
    ("Reset Scale" :reset-scale "Reset the scale to effectively undo all previous Zooms")
    ("Misc." :misc-cmd 
     "Menu of misc. commands. Applications can add to *miscellaneou-command-menu*")
    ("Exit" :exit "Exit Zgraph")))

(defun zgraph-menu-pane-show (command-menu)
  (let* ((margin 2)
	 (bitmap (send *menu-pane* :window))
	 (menu-width (truncate (/ *zgraph-screen-width* (length *command-menu*))))
	 (y-position (+ (font-height "MEDIUM-ROMAN") margin))
	 (choice-no 0))
    (dolist (command command-menu)
      (let* ((name (car command))
	     (value (cadr command))
	     (documentation (caddr command))
	     (x-position (+ (* choice-no menu-width)
			    (truncate (/ (- menu-width (string-width name *default-font*)) 2)))))
        (make-active-region
            (make-region :x x-position :y 3
			 :width (+ (string-width name *default-font*) (* margin 2)) :height 18)
            :bitmap bitmap

            :mouse-enter-region
            (default-mouse-enter-method)

            :mouse-exit-region
            (default-mouse-exit-method)

            :mouse-right-down
            (default-mouse-right-down-method))

            ;; Write the menu out to the menu-pane bitmap

        (stringblt bitmap
	           (make-position (+ margin x-position) y-position)
		   "MEDIUM-ROMAN" name))
      (incf choice-no))))


;; Zgraph status pane flavor

(defflavor zgraph-status-pane
  ((window nil))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defmethod (zgraph-status-pane :CLEAR-WINDOW) ()
  (clear-bitmap window))

(defun zgraph-status-pane-init (x)
  (send x :set-window
	(make-window :x 0 :y (+ *zgraph-des-height* ;; was 688
				*zgraph-menu-height*
				*zgraph-sidebar-size*)
		     :width *zgraph-screen-width* ;; was 1018
		     :height (send *zgraph-display-frame*
				   :zgraph-lisp-height)
		     :initial-font "MEDIUM-ROMAN"
		     :parent (root-viewport)  :activate nil)))



;; Zgraph description pane flavor

(defconstant *description-pane-width* 400)

(defflavor zgraph-description-pane
  ((window nil)
   (font "MEDIUM-ROMAN"))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defun zgraph-description-pane-init (x)
  (send x :set-window
	(make-window :x 0 :y *zgraph-menu-height* 
		     :width *zgraph-scroll-bitmap-width* 
		     :height *zgraph-scroll-bitmap-height* 
		     :viewport-width *zgraph-scroll-width* 
		     :viewport-height *zgraph-scroll-height* 
		     :initial-font (send x :font)
		     :parent (root-viewport) :scroll t :activate nil)))

(defmethod (zgraph-description-pane :change-font) (new-font &aux old-window)
  (unless (equal font new-font)
    (send self :set-font new-font)
    (setf old-window window)
    (zgraph-description-pane-init *description-pane*)
    (clear-bitmap-active-regions old-window)
    (delete-viewport old-window)
    (when (send *zgraph-display-pane* :description-pane-active?)
      (activate-viewport window)
      (expose-viewport window))))

(defmethod (zgraph-description-pane :CLEAR-WINDOW) ()
  (clear-bitmap window)
  (setf (stream-x-position window) 0)
  (setf (stream-y-position window) 0)
  (setf (viewport-bitmap-x-offset window) 0)
  (setf (viewport-bitmap-y-offset window) 0))

(defmethod (zgraph-description-pane :beginning) ()
  (cond
   ((> (stream-y-position window) (- *zgraph-scroll-bitmap-height* 100))
    (send self :CLEAR-WINDOW))
   (t
    (setf (viewport-bitmap-x-offset window) 0)
    (setf (viewport-bitmap-y-offset window) (stream-y-position window))))) 
  

(defmethod (zgraph-description-pane :append-item) (item)
  (when (< (stream-y-position window) (- *zgraph-scroll-bitmap-height* 20))
    (format window "~%~A" item)))
			    
;; Zgraph documentation pane flavor

(defflavor zgraph-documentation-pane
  ((window nil)
   (region nil))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defmethod (zgraph-documentation-pane :CLEAR-WINDOW) ()
  (clear-bitmap window))

(defun zgraph-documentation-pane-init (x)
  (send x :set-window
	(make-window :x 0 :y (+ *zgraph-menu-height* ;; Was 746 
				(send *zgraph-display-frame*
				      :zgraph-des-height) 
				(send *zgraph-display-frame*
				      :zgraph-lisp-height))
		     :width *zgraph-doc-width* :height *zgraph-doc-height*
		     :initial-font "MEDIUM-ROMAN"
		     :parent (root-viewport)  :activate nil))
  (send x :set-region
	(make-region :x 0 :y 0 :width *zgraph-doc-width* :height *zgraph-doc-height*)))

(defmethod (zgraph-documentation-pane :DRAW-STRING) (text)
  (send self :CLEAR-WINDOW)
  (cond ((stringp text)
	 (stringblt window (make-position 10 (- (font-height "MEDIUM-ROMAN") 2)) 
		    "MEDIUM-ROMAN"  text)
	 (bitblt-region window region window region boole-c1))
	(t (stringblt window (make-position 10 (- (font-height "MEDIUM-ROMAN") 2)) 
		      "MEDIUM-ROMAN"
		      (format nil "Zgraph doc text not string -- ~A" text)))))


;; Dialog window 

(defparameter *menu-inner-border-width* 2)
(defparameter *menu-outer-border-width* 1)

(defflavor dialog-box
 ((window nil)
   (input nil))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defmethod (dialog-box :build) (position width height prompt font)
  (let ((window (make-window :position position 
			     :width width :height height
			     :inner-border-width *menu-inner-border-width* 
			     :outer-border-width *menu-outer-border-width* 
			     :initial-font font
			     :parent (root-viewport))))
    (send self :set-window window)
    (expose-viewport window)
    (send self :set-input
	  (editor::dialogbox window prompt))
    (clear-bitmap-active-regions window)
    (delete-viewport window)))

(defun read-from-dialog-box (prompt font &optional number)
  (let* ((input-width 150)
	 (margin 3)
	 (menu-width (+ (string-width prompt font) input-width))
	 (menu-height (+ (font-height font) (* margin 2)))
	 (combined-border-width (+ *menu-inner-border-width*
				   *menu-outer-border-width*))
	 (min-mouse-x (+ combined-border-width 
			 (round menu-width 2)))
	 (max-mouse-x (- *zgraph-screen-width* combined-border-width 
			 (round menu-width 2) 1))
	 (min-mouse-y combined-border-width) 
	 (max-mouse-y (- *zgraph-screen-width* combined-border-width 
			 menu-height 1))
	 (desired-x (- *mouse-x* (round menu-width 2)))
	 (desired-y (- *mouse-y* (round menu-height 2)))
	 (actual-x (min (max min-mouse-x desired-x)
			max-mouse-x))
	 (actual-y (min (max min-mouse-y desired-y)
			max-mouse-y))
	 i-value)
    (unless *dialog-box* 
      (setf *dialog-box* (make-instance 'dialog-box)))
    (send *dialog-box* 
	  :build (make-position actual-x actual-y)
	         menu-width menu-height
	         prompt font)
    (setf i-value (send *dialog-box* :input))
    (when number
      (setf i-value (convert-string-to-pnumber i-value)))
    i-value))

(defun convert-string-to-pnumber (str)
  (let ((no 0))
    (dotimes (idx (length str) no)
      (setf no (* no 10))
      (case (schar str idx) 
	(#\1 (incf no 1))
	(#\2 (incf no 2))
	(#\3 (incf no 3))
	(#\4 (incf no 4))
	(#\5 (incf no 5))
	(#\6 (incf no 6))
	(#\7 (incf no 7))
	(#\8 (incf no 8))
	(#\9 (incf no 9))))))


;; 
(defvar *graph-status-menu*
  `(("Frame Configuration" :set-frame-configuration
			   "Configuration of window panes (Display pane only bu default)") 
    ("Graph Plotting Styles" :set-plotting-style
			     "Select the method to use in plotting graphs.")
    ("Display-pane : Lisp-Listener-pane" :change-graph-frame
     "Change the relative percentage of scrren space devoted to display versus lisp listener")
    ("Change Font" :change-font "Change the font in graph")
    ("Exit" :exit-graph-status-menu "Exit graph status menu")))

;;;; Status pane flavor

(defflavor zgraph-graph-status-pane
  ((window nil))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defmethod (zgraph-graph-status-pane :CLEAR-WINDOW) ()
  (clear-bitmap-active-regions window)
  (clear-bitmap window))

(defun zgraph-graph-status-pane-init (x)
  ;; 
  (let ((margin 2))
    (send x :set-window
	  (make-window :x (* 2 (round *zgraph-screen-width* 
				      (length *command-menu*)))
		       :y *zgraph-menu-height*
		       :width (+ (string-width "Display-pane : Lisp-Listener-pane" 
					       "MEDIUM-ROMAN") (* margin 2))
		       :height (* (+ (font-height "MEDIUM-ROMAN") (* margin 2))
				  (length *graph-status-menu*))
		       :initial-font "MEDIUM-ROMAN"
		       :parent (root-viewport)  :activate nil))))

(defun show-graph-status-menu ()
  (let* ((margin 2)
	 (window (send *graph-status-pane* :window))
	 (width (+ (string-width "Display-pane : Lisp-Listener-pane" 
				 "MEDIUM-ROMAN") (* margin 2)))
	 (height (+ (font-height "MEDIUM-ROMAN") (* margin 2)))
	 (choice-no 0))
    (dolist (command *graph-status-menu*)
      (let* ((name (car command))
	     (value (cadr command))
	     (documentation (caddr command))
	     (y-position (* choice-no height))) 
			   
        (make-active-region
            (make-region :x 0 :y y-position
			 :width width  :height height) 
            :bitmap window

            :mouse-enter-region
            (default-mouse-enter-method)

            :mouse-exit-region
            (default-mouse-exit-method)

            :mouse-right-down
            (default-mouse-right-down-method))

            ;; Write the menu out to the menu-pane bitmap

        (stringblt window
	           (make-position  margin
				   (+ y-position (- height (* margin 2))))
		   "MEDIUM-ROMAN" name))
      (incf choice-no))))

(defun activate-graph-status-menu ()
  (activate-viewport (send *graph-status-pane* :window))
  (expose-viewport (send *graph-status-pane* :window)))

(defun deactivate-graph-status-menu ()
  (deactivate-viewport (send *graph-status-pane* :window)))


;; Miscellaneous Command Menu Flavor

(defflavor zgraph-misc-cmd-menu
  ((window nil))
  ()
  :gettable-instance-variables
  :settable-instance-variables)

(defmethod (zgraph-misc-cmd-menu :clean-up) ()
  (clear-bitmap-active-regions window)
  (delete-viewport window))

(defmethod (zgraph-misc-cmd-menu :build) ()
  (let* ((margin 2)
	 (selection-height (+ (font-height "MEDIUM-ROMAN") (* margin 2)))
	 (inner-width (string-width "Miscellaneous Command Menu" "ITALIC"))
	 (menu-width 0)
	 (title-height (+ (font-height "ITALIC")  (* margin 2)))
	 (menu-height (+ title-height 1)))

    (dolist (choice *miscellaneous-command-menu*)
      (setf inner-width
	    (max inner-width (string-width (car choice) "MEDIUM-ROMAN")))
      (setf menu-width (+ inner-width (* margin 2)))
      (incf menu-height selection-height))

    ;; Create an inactive window. 
    (let* ((min-x (- *zgraph-screen-width* menu-width))
	   (window (make-window :x min-x :y 0
				:width menu-width :height menu-height
				:initial-font "MEDIUM-ROMAN"
				:activate nil))
	   (choice-no 0))

      (send self :set-window window)
      ;; First, the active region that covers the entire bitmap
      (make-active-region
       (make-region :x 0 :y 0 :extent (bitmap-extent window))
       :bitmap window
       :mouse-exit-region
       #'(lambda (viewport active-region mouse-event x y)
	   (declare (ignore viewport active-region mouse-event x y))
	   (deactivate-viewport window)))

      (stringblt window
		 (make-position 
		  (round (- menu-width (string-width "Miscellaneou Command Menu"
						     "ITALIC")) 2)
		  (- title-height margin))
		 "ITALIC" "Miscellaneou Command Menu")
      (draw-line window 
		 (make-position 0 (+ title-height 1))
		 (make-position menu-width
				(+ title-height 1)))

      (dolist (command *miscellaneous-command-menu*)
	(let* ((name (car command))
	       (flag (cadr command))
	       menu
	       prompt
	       form
	       arg
	       number
	       (documentation (cadr (member ':DOCUMENTATION command)))
	       (min-y (+ (* choice-no selection-height) (+ title-height 1)))
	       (x-position (round (- menu-width (string-width name *default-font*)) 2))
	       (y-position (- (+ min-y selection-height) (* margin 2))))

	  (cond ((equal flag ':CHOICES)
		 (setf menu (make-pop-up-menu (eval (caddr command)))))

		((equal flag ':READ)
		 (setf prompt (eval (caddr command)))
		 (if (stringp prompt)
		     (setf prompt (concatenate 'string prompt " => "))
		     (setf prompt " => "))
		 (setf form  (eval (cadddr command)))
		 (unless (functionp form) (setf form nil))
		 (setf number (member ':NUMBER command))))

	  (make-active-region
	   (make-region :x 0 :y min-y
			:width menu-width :height selection-height)
            :bitmap window

            :mouse-enter-region
            (default-mouse-enter-method)

            :mouse-exit-region
            (default-mouse-exit-method)

            :mouse-right-down
            #'(lambda (viewport active-region mouse-event x y)
                (declare (ignore viewport active-region mouse-event x y))
		(cond ((equal flag ':CHOICES)
		       (eval (pop-up-menu-choose menu)))
		      ((equal flag ':READ)
		       (setf arg (read-from-dialog-box prompt "MEDIUM-ROMAN" number))
		       (when arg (funcall form arg))))))

	  ;; Write the menu out to the menu-pane bitmap
	  
	  (stringblt window
		     (make-position x-position y-position)
		     "MEDIUM-ROMAN" name))

	(incf choice-no)))))

(defmethod (zgraph-misc-cmd-menu :activate) ()
  (activate-viewport window)
  (expose-viewport window))


;;;; Starting up the windows

(defvar *window-built* nil)

(defun init-lisp-listener ()
  ;; Build the status pane
  (setf *status-pane* (make-instance 'zgraph-status-pane))
  (zgraph-status-pane-init *status-pane*)

  ;; Attach editor to window
  (multiple-value-setq
    (*sys-lisp-buffer* *sys-editor-window* *sys-buffer-stream*)
    (editor::attach-editor-to-window (send *status-pane* :window)			  
				     :buffer-name "Lisp"
				     :modeline t)))

(defun activate-lisp-listener ()
  (activate-viewport (send *status-pane* :window))
  (expose-viewport (send *status-pane* :window))
  (setf *old-lisp-window* editor::*lisp-buffer-window*
	editor::*lisp-buffer-window* *sys-editor-window*)
  (editor::select-window editor::*lisp-buffer-window*))

(defun deactivate-lisp-listener ()
  (deactivate-viewport (send *status-pane* :window))
  (setf editor::*lisp-buffer-window* *old-lisp-window*)
  (editor::select-window editor::*lisp-buffer-window*))

(defun delete-lisp-listener ()
  (clear-bitmap-active-regions (send *status-pane* :window))
  (delete-viewport (send *status-pane* :window))
  (setf editor::*lisp-buffer-window* *old-lisp-window*)
  (editor::select-window editor::*lisp-buffer-window*))



(defun init-windows ()
  (setf *zgraph-display-frame* (make-instance 'zgraph-display-frame))
  (setf *zgraph-display-pane* (make-instance 'zgraph-display-pane))
  (setf *graph-output* (make-instance 'zgraph-window-flavor))
  (zgraph-window-flavor-init *graph-output*)
  ;; build the description pane
  (setf *description-pane* (make-instance 'zgraph-description-pane))
  (zgraph-description-pane-init *description-pane*)
  ;; Build the documentation pane
  (setf *documentation-pane* (make-instance 'zgraph-documentation-pane))
  (zgraph-documentation-pane-init *documentation-pane*)
  ;; Build the graph status menu pane
  (setf *graph-status-pane* (make-instance 'zgraph-graph-status-pane))
  (zgraph-graph-status-pane-init *graph-status-pane*)
  (show-graph-status-menu)
  (init-lisp-listener)
  ;; Build dialog window
  (send *zgraph-display-pane* :set-description-pane-active? nil) 
  (setf *window-built* t))
  
(defun activate-windows ()
  ;; Build the menu pane 
  (setf *menu-pane* (make-instance 'zgraph-menu-pane))
  (zgraph-menu-pane-init *menu-pane*)
  (zgraph-menu-pane-show *command-menu*)
  (activate-viewport (send *menu-pane* :window))
  (activate-viewport (send *graph-output* :window))
  (when (send *zgraph-display-pane* :description-pane-active?)
    (activate-viewport (send *description-pane* :window)))
  (activate-viewport (send *documentation-pane* :window))
  (expose-viewport (send *menu-pane* :window))
  (activate-lisp-listener)
  (expose-viewport (send *graph-output* :window))
  (when (send *zgraph-display-pane* :description-pane-active?)
    (expose-viewport (send *description-pane* :window)))
  (expose-viewport (send *documentation-pane* :window))
  (setf *graph-plotting-style* *default-graph-plotting-style*))


(defun deactivate-windows ()
  (clear-bitmap-active-regions (send *menu-pane* :window))
  (delete-viewport (send *menu-pane* :window))
  (deactivate-viewport (send *status-pane* :window))
  (when (send *zgraph-display-pane* :description-pane-active?)
    (deactivate-viewport (send *description-pane* :window)))
  (deactivate-viewport (send *graph-output* :window))
  (deactivate-viewport (send *documentation-pane* :window))
  (deactivate-lisp-listener))
 
(defun clean-up-windows ()
  (when *window-built*
    (send *zgraph-display-pane* :set-graph nil)
    (send *zgraph-display-pane* :set-graph-history nil)
    (send *graph-output* :CLEAR-WINDOW)))

