; -*- LISP -*-

;;;; Interface for RT Zgraph
;;   Translated from Hogge's code by H. Kim, University of Illinois

;;======================================================================================
;;
;;;;  The user interface (accessed via calling (z))
;;
;;======================================================================================
(in-package 'user)

(defvar *zgraph-display-pane* nil)
(defvar *miscellaneous-command-menu* nil)
(defvar *old-miscellaneous-command-menu* nil)
(defvar *misc-cmd-menu-pane* nil)

;; Zgraph's graph display pane.

(defflavor zgraph-display-pane
  ((graph nil)
   (graph-history nil)
   (graphics-object-being-moved nil)
   (configuration nil)
   (description-pane-active? nil))
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables

  ;Instance Variables:
  ;  GRAPH -- either NIL or a GRAPH struct which is
  ;  currently displayed on SELF.

  ;  GRAPH-HISTORY -- list of previously displayed graphs.
  ;  GRAPH is always the CAR of this list, unless GRAPH is NIL.
  ;  This is used to provide an option for viewing 
  ;  previously displayed graphs.

  ;  GRAPHICS-OBJECT-BEING-MOVED -- selected vertex to be moved.
  )


(defmethod (zgraph-display-pane :draw-graph) (new-graph &optional (update-display? t))
  (when update-display?
    (cond
     ((null new-graph)
      (send *graph-output* :CLEAR-WINDOW))
     ((not (typep new-graph 'graph))
      (FORMAT t "instance variable GRAPH must be a GRAPH flavor instance or NIL"))
     (t
      (send *graph-output* :CLEAR-WINDOW)
      (send *graph-output* :update-displacement 0 0 T)
      (send *graph-output* :update-image)
      (when (and description-pane-active?
		 (equal (send new-graph :pane) "display only"))
	(send new-graph :set-pane "desc pane")
	(send new-graph :set-x-displacement
	      (+ (send new-graph :x-displacement)
		 (round (/ *description-pane-width* (send new-graph :x-scale-factor))))))
      (send new-graph :draw))))
  (when (and new-graph (not (eq new-graph graph)))
    (setf graph-history (cons new-graph (remove new-graph graph-history))))
  (setf graph new-graph))

(defmethod (zgraph-display-pane :create-graph) ()
  (let* ((menu (make-pop-up-menu (mapcar #'(lambda (graph-type)
					   (cons (string (send graph-type :name)) graph-type))
					 *graph-types*)))
	 (graph-type (pop-up-menu-choose menu))
	 new-graph)
    (when graph-type
      (setq new-graph (funcall (send graph-type :instantiation-function) graph-type))
      ;;Traverse the graph given the root vertices
      (send new-graph :construct)
      ;;Plot the graph
      (send new-graph :plot-vertices)
      (debug-print t "~%Drawing graph on bit-array...")
      (send self :draw-graph new-graph)
      (debug-print t "done.~%> "))))

(defmethod (zgraph-display-pane :display-graph) ()
  (let* ((menu (make-pop-up-menu (mapcar #'(lambda (graph)
					   (cons (string (send graph :name)) graph))
					 graph-history)))
	 (selected-graph (pop-up-menu-choose menu)))
    (when selected-graph
      (case (pop-up-menu-choose
	     (make-pop-up-menu '(("Display" . 1) ("Delete" . 2))))
	(1
	 (debug-print t "~%Drawing graph on bit-array...")
	 (send self :draw-graph selected-graph)
	 (debug-print t " done~%> "))
	(2
	 (setf graph-history (remove selected-graph graph-history))
	 (when (eq selected-graph graph) 
	   (debug-print t "~%Current graph deleted.")
	   (debug-print t "~%Drawing most recent graph on bit-array...")
	   (send self :draw-graph (car graph-history))
	   (debug-print t " done~%> ")))))))


(defvar  *frame-configuration*
  '(("Display Pane Only" . :display-pane-only)
    ("With Description Pane" . :with-description-pane)))

;;; We should ask Lucid if it is better to cache the menu
;;; or build a new one each time.  Given how short the RT's
;;; are on page space, tossing it away seems the right choice. 

(defmethod  (zgraph-display-pane :set-frame-configuration) ()
  (let ((menu (make-pop-up-menu *frame-configuration*)))
    (setf configuration (pop-up-menu-choose menu))))

(defmethod (zgraph-display-pane :display-pane-only) ()
  (when (and graph description-pane-active?)
    (setf description-pane-active? nil)
    (deactivate-viewport (send *description-pane* :window))
    (unless (equal (send graph :pane) "display only")
      (send graph :set-x-displacement
	    (- (send graph :x-displacement) 
	       (round (/ *description-pane-width* (send graph :x-scale-factor)))))
      (send graph :set-pane "display only"))
    (send *graph-output* :CLEAR-WINDOW)
    (send graph :draw)
    (debug-print T "done~%> ")))


(defmethod (zgraph-display-pane :with-description-pane) ()
  (when (and graph (not description-pane-active?)) 
    (setf description-pane-active? t)
    (unless (equal (send graph :pane) "desc pane")
      (send graph :set-x-displacement
	    (+ (send graph :x-displacement)
	       (round (/ *description-pane-width* (send graph :x-scale-factor)))))
      (send graph :set-pane "desc pane"))
    (send *graph-output* :CLEAR-WINDOW)
    (send *description-pane* :CLEAR-WINDOW)
    (expose-viewport (send *description-pane* :window))
    (activate-viewport (send *description-pane* :window))
    (send graph :draw)
    (debug-print T "done~%> ")))


(defmethod (zgraph-display-pane :set-plotting-style) ()
  (let* ((menu (make-pop-up-menu *all-graph-plotting-styles*))
	 (plotting-style (pop-up-menu-choose menu)))
    (when plotting-style
      (setf *graph-plotting-style* plotting-style)
      (send graph :plot-vertices)
      (send graph :set-pane "display only")
      (send self :draw-graph graph)
      (debug-print T "done~%> "))))



(defmethod (zgraph-display-pane :exit-graph-status-menu) ()
  (deactivate-graph-status-menu)
  (when (and graph configuration) 
    (send self configuration)))

(defvar *frame-ratio-menu-item*
  '(("5:1" . 5) ("2:1" . 2) ("1:1" . 1) ("1:2" . 1/2) ("1:5" . 1/5)))

(defmethod (zgraph-display-pane :change-graph-frame) ())
#|
(defmethod (zgraph-display-pane :change-graph-frame) ()
  (let* ((menu (make-pop-up-menu *frame-ratio-menu-item*))
	 (ratio (pop-up-menu-choose menu))
	 (display&lisp-height (- *zgraph-screen-height*
				 *zgraph-menu-height*
				 *zgraph-doc-height*))
	 display-height)
    (when ratio
      (setf display-height
	    (round (* display&lisp-height ratio)
		   (+ ratio 1)))
      (send *zgraph-display-frame* 
	    :set-zgraph-des-height display-height)
      (send *zgraph-display-frame* 
	    :set-zgraph-lisp-height (- display&lisp-height display-height))
      (delete-lisp-listener[A)
      ;; Build the status pane
      (init-lisp-listener)
      (activate-lisp-listener))))
|#
(defvar *font-menu-item*
  '(("ROMAN" ."ROMAN") ("MEDIUM-ROMAN" ."MEDIUM-ROMAN") ("SMALL-ROMAN" ."SMALL-ROMAN")
    ("ITALIC" . "ITALIC")))
(defvar *font-change-item* 
  '(("Edge labels" . 1) ("Vertexs labels" . 2) ("Description pane" . 3)))

(defmethod (zgraph-display-pane :change-font) ()
  (let* ((menu (make-pop-up-menu *font-change-item*)) 
	 (x (pop-up-menu-choose menu))
	 font gtype)
    (when (and x
	       (setf font (pop-up-menu-choose 
			   (make-pop-up-menu *font-menu-item*)))) 
      (setf gtype (send graph :type))
      (case x (1 (send gtype :set-edge-label-font font)
		 (send self :draw-graph graph))
	      (2 (send gtype :set-vertex-label-font font)
		 (send self :draw-graph graph))
	      (3 (send *description-pane* :change-font font))))))

(defmethod (zgraph-display-pane :change-status) ()
  (setf configuration nil)
  (activate-graph-status-menu))
 

(defvar *zoom-menu-item*
  '(("1/4" . 1/4) ("1/3" . 1/3) ("1/2" . 1/2) ("2/3" . 2/3) ("3/4" . 3/4)
    ("1" . 1) ("4/3" . 4/3) ("3/2" . 3/2) ("2" . 2) ("3" . 3) ("4" . 4)))

(defmethod (zgraph-display-pane :zoom) ()
  (let* ((menu (make-pop-up-menu *zoom-menu-item*))
	 (scale (pop-up-menu-choose menu)))
    (when (and graph scale)
      (send graph :scale-relative scale scale)
      (send graph :set-x-displacement
	    (+ (round (/ (send graph :x-displacement) scale)) 30))
      (send graph :set-y-displacement
	    (round (/ (send graph :y-displacement) scale)))
      (when (< (send graph :y-displacement) 0)
	(send graph :set-y-displacement 100))
      (send self :draw-graph graph)
      (debug-print t "~% done~%> "))))


(defmethod (zgraph-display-pane :reset-scale) ()
  (when graph
    (send graph :reset-position-and-scale)
    (send graph :set-pane "display only")
    (send self :draw-graph graph)
    (debug-print T "done~%> ")))



(defmethod (zgraph-display-pane :misc-cmd) ()
  (unless (equal *miscellaneous-command-menu* *old-miscellaneous-command-menu*)
    (setf *old-miscellaneous-command-menu* *miscellaneous-command-menu*)
    (if *misc-cmd-menu-pane*
	(send *misc-cmd-menu-pane* :clean-up)
	(setf *misc-cmd-menu-pane* (make-instance 'zgraph-misc-cmd-menu)))
    (send *misc-cmd-menu-pane* :build))
  (when *misc-cmd-menu-pane*
    (send *misc-cmd-menu-pane* :activate)))


(defmethod (zgraph-display-pane :exit) ()
  (deactivate-windows)
  (setf *default-font* *old-font*)
  (format t "~%> "))

(defmethod (zgraph-display-pane :move-graphics-object) (&optional selected-vertex)
  ;; Handles MOUSE-L clicks near the vertex. This is used to dynamically
  ;; move vertices.
  (when graph
    (cond
     (graphics-object-being-moved
      (setf (vertex-location graphics-object-being-moved)
	    (cons (send graph :inverse-scale-x *mouse-x*)
		  (send graph :inverse-scale-y *mouse-y*))) 
      (send graph :vertex-has-moved graphics-object-being-moved)
      (setf graphics-object-being-moved nil)
      (format t "~%Redrawing graph.")
      (send self :draw-graph graph)
      (format t "~%> "))
     (t
      (when (typep selected-vertex 'vertex)
	(setf graphics-object-being-moved selected-vertex)
	(format t "~%You have selected vertex ~s to be moved. Now move the mouse to the desired location 
(pan to it if need be) and click MOUSE-R again to complete the move."		
		(vertex-data selected-vertex)))))))

(defun z ()
  (setf *old-font* *default-font*)
  (setf *default-font* "MEDIUM-ROMAN")
  (unless *window-built*
    (init-windows))
  (activate-windows))
