;; -*- LISP -*-

;;;; Definition of graph types
;;   RT version of ZGRAPH
;;   Translated from Hogge's version by H. Kim, University of Illinois

(in-package 'user)

(defflavor graph-type
  ((name (INTERN (FORMAT NIL "GRAPH-TYPE-~a" (LENGTH *graph-types*))
		 *package*))
   (traversal-function NIL)
   (default-root-finding-form NIL)
   (instantiation-function 'default-instantiation-function)
   (traverse-recursively? T)
   (vertex-label-font nil)
   (vertex-print-string-function 'default-vertex-print-string-function)
   (edge-label-font nil)
   (edge-print-string-function 'default-edge-print-string-function)
   (vertex/edge-description-function 'default-vertex/edge-description-function) 
   (vertex/edge-selection-handler 'default-vertex/edge-selection-handler) 
   (default-name-count 0))
  ()
  :SETTABLE-INSTANCE-VARIABLES
  :GETTABLE-INSTANCE-VARIABLES
  :INITABLE-INSTANCE-VARIABLES)

(defmethod (graph-type :after :init) (ignore)
  (push self *graph-types*))

(defmethod (graph-type :genname) ()
  (format nil "~a #~s" name (incf default-name-count)))

(defun default-vertex-print-string-function (vertex-struct)
  (values
   (format nil "~s" (vertex-data vertex-struct))
   "SMALL-ROMAN"))

(defun default-edge-print-string-function (edge-struct)
  (let ((edge-data (edge-data edge-struct)))
    (values
     (if (cdr edge-data)
	 (with-output-to-string (stream)
	   (if (stringp (car edge-data))
	       (format stream (car edge-data))
	       (format stream "~S" (car edge-data)))
	   (dolist (label (cdr edge-data))
	     (if (stringp label)
		 (format stream " & ~A" label)
		 (when label
                   (format stream " & ~S" label)))))
         (if (stringp (car edge-data))
 	     (car edge-data)
       	     (when (car edge-data)
   	       (format nil "~S" (car edge-data)))))
     "SMALL-ROMAN")))

(defun default-vertex/edge-description-function (vertex-or-edge)
  (display-on-scroll-window
   (with-output-to-string (stream)
			  (let ((*standard-output* stream))
			    (describe (if (typep vertex-or-edge 'vertex)
					  (vertex-data vertex-or-edge)
					  (edge-data vertex-or-edge)))))
   *description-pane*))

;; This next function should be moved into a generic scroll window
;; definition.  Flavorized, with methods for changing sizes and such.

(defun display-on-scroll-window (description scroller)
  (let ((description-lines ;; Decompose it into lines
         (with-input-from-string (stream description)
           (do ((line (read-line stream nil)(read-line stream nil))
		(lines nil))
	       ((null line) (nreverse lines))
	     (push line lines)))))
    ;; Now display it
    (send scroller :beginning)
    (send scroller :append-item "----------------------------------")
    (dolist (line description-lines)
      (send scroller :append-item line))))

(defmethod (graph-type :handle-selection-of-object) (vertex-or-edge-struct)
  (funcall vertex/edge-selection-handler vertex-or-edge-struct))

(defun default-vertex/edge-selection-handler (vertex-or-edge-struct)
  (let ((graph-type (send (send *zgraph-display-pane* :graph) :type)))
    (cond
     ((typep vertex-or-edge-struct 'vertex)
      (funcall (send graph-type :vertex/edge-description-function) vertex-or-edge-struct))
     ((typep vertex-or-edge-struct 'edge)
      (funcall (send graph-type :vertex/edge-description-function) vertex-or-edge-struct)))))

(defun default-instantiation-function (graph-type)
  (let (*name*
	(*form* (send graph-type :default-root-finding-form)))
    (declare (special *name* *form*))
    (send *documentation-pane* :CLEAR-WINDOW)
    (send *documentation-pane* :draw-string 
	  "Identifier for this graph instance.")
    (setf *name* (read-from-dialog-box "name: " "ROMAN")) 
    (if (equal *name* "")
	(setf *name* nil)
	(setf *name* (make-symbol *name*)))
     (setf *form* (send graph-type :default-root-finding-form))
     (let ((root-vertices (eval *form*)))
       (if root-vertices 
	   (debug-print t "~%Root vertices are: ~s" root-vertices)
	   (debug-print t "~%Warning, the default root finding form, ~s, evaluated to NIL."
		       *form*))
      (make-instance 'graph
		     :name *name* 
		     :type graph-type
		     :root-vertices root-vertices)))) 

(compile-flavor-methods graph-type)
