;;; -*- Mode: LISP -*-

(in-package 'user)

;; Examples using RT Zgraph
;; Translated from Hogge's code by H. Kim, University of Illinois

;;This file illustrates how to use Zgraph to display data objects and the relationships
;;among them.  Before studying it, it would be helpful to go back to Zgraph,
;;execute the Create Graph command, and select SAMPLE-GRAPH-TYPE from the menu of graph 
;;types.
;;
;;The data objects which SAMPLE-GRAPH-TYPE is set up to display are simply the
;;following symbols:
;;
(DEFVAR v1)
(DEFVAR v2)
(DEFVAR v3)
(DEFVAR v4)
(DEFVAR v5)
(DEFVAR v6)

;;We define two simple relationships among these data objects which we want to display
;;using Zgraph:
;;1.  A symbol's value is a list of the other symbols.
;;2.  A symbol's FOO property is either NIL or one of the other symbols.
;;
;;The following forms set up these relationships by setting each symbol to either NIL or to
;;a list of other symbols and by adding a FOO property to V1.
;;
(SETQ v1 '(v2 v3 v6)
      v2 '(v1 v3)
      v3 '(v1 v2 v4)
      v4 '(v5 v6)
      v5 NIL
      v6 NIL)
(SETF (GET 'v1 'foo) 'v6)

;;
;;This gives us a very simple data structure to display.  If you've clicked on Create Graph
;;and created a graph of type SAMPLE-GRAPH-TYPE, you will have seen the above data structure
;;displayed as a graph with V1, V2, V3, V4, V5, and V6 as the vertices, several directed
;;edges labelled "VALUE", and one edge labelled "FOO PROPERTY" (leading from V1 to V6).
;;
;;In order to define a graph type (such as SAMPLE-GRAPH-TYPE), you must provide the Define
;;Graph Type command with the name of a graph traversal function which you define elsewhere
;;(in a Zmacs buffer, or in a loaded LISP file).  The traversal function should accept as
;;argument *any* data object (vertex) in your graph and return an alist of conses of the
;;following form:
;;
;;   ({vertex to which the argument vertex is connected} . {edge label})
;;
;;In otherwords, the function takes any vertex and returns an alist describing edges leading
;;from the vertex.  The traversal function will recursively be applied to the CARs of this
;;alist in a depth-first manner.  The CDRs may be any LISP object.  If they are strings, they
;;are printed without quote marks.  Otherwise they are printed using the ~S FORMAT directive.
;;
;;The following defines the traversal function for SAMPLE-GRAPH-TYPE:
;;
(DEFUN sample-graph-traversal-function (vertex)
  "Traversal function for the graph type SAME-GRAPH-TYPE.
This function is a simple example of a graph travesal function."
  (LET ((value-vertices 
        (do ((connected-vertex (car (SYMBOL-VALUE vertex)) (car cv))
	     (cv (cdr (SYMBOL-VALUE vertex)) (cdr cv))
	     (result nil))
	    ((null connected-vertex) (nreverse result)) 
	  (push (CONS connected-vertex "VALUE") result))))
;        (LOOP FOR connected-vertex IN (SYMBOL-VALUE vertex)
;			      COLLECT (CONS connected-vertex "VALUE"))))

    (WHEN (GET vertex 'foo)
      (PUSH (CONS (GET vertex 'foo) "FOO PROPERTY")
	    value-vertices))
    value-vertices))


;;Try evaluating the following calls to the traversal function:
(sample-graph-traversal-function 'v1)
(sample-graph-traversal-function 'v2)
(sample-graph-traversal-function 'v3)
(sample-graph-traversal-function 'v4)
(sample-graph-traversal-function 'v5)  ;;returns NIL--no outgoing edges from V5.
(sample-graph-traversal-function 'v6)  ;;returns NIL--no outgoing edges from V6.


;;The Define Graph Type command also accepts a default root finding form which
;;is evaluated to provide a list of vertices you want the graph traversal function
;;to be applied to.  SAMPLE-GRAPH-TYPE's default root finding form is '(v1).
;;V1 is related to V2, V3, and V6 through its value cell.  (It is also related to V6 through
;;its FOO property, and Zgraph can handle such multiple relations.)  Therefore, 
;;SAMPLE-GRAPH-TRAVERSAL-FUNCTION finds vertices V2, V3, and V6 when applied to V1.  Recursive
;;applications finds V4 since (MEMQ 'v4 v3) and V5 since (MEMQ 'v5 v4).  All this illustrates
;;that the application of SAMPLE-GRAPH-TRAVERSAL-FUNCTION in a depth-first manner means that
;;you may only have to supply a few vertices as the root-finding form.  Your traversal
;;function will find the rest of them. 

;;If you are still unsure about how to define graph types, try playing with 
;;SAMPLE-GRAPH-TRAVERSAL-FUNCTION to add more edge relationships and types of vertices.  For
;;instance, extend it to handle some sort of STRUCT as well as the six symbols.

;;The following form defines the SAMPLE-GRAPH-TYPE.  Instead of defining graph types through
;;the Zgraph user interface, you can put calls such as the following in your LISPM-INIT file.
;;See documentation and code for DEFAULT-VERTEX-PRINT-STRING-FUNCTION, 
;;DEFAULT-EDGE-PRINT-STRING-FUNCTION, DEFAULT-VERTEX/EDGE-SELECTION-HANDLER, and
;;DEFAULT-INSTANTIATION-FUNCTION for information on how they should behave.  (The print-string
;;functions return strings to print, while the selection handler controls what to do when the
;;user clicks on a displayed vertex or edge.)
;;

(MAKE-INSTANCE 'graph-type
	       :name 'sample-graph-type 
	       :traversal-function 'sample-graph-traversal-function
	       :default-root-finding-form ''(v1)
	       :instantiation-function 'default-instantiation-function
	       :traverse-recursively? T
	       :vertex-print-string-function 'default-vertex-print-string-function
	       :edge-print-string-function 'default-edge-print-string-function
	       :vertex/edge-description-function 'default-vertex/edge-description-function
	       :vertex/edge-selection-handler 'default-vertex/edge-selection-handler)

;;If you evaluate this global variable, you'll get a list which holds the GRAPH-TYPE instance
;;created above.  This variable provides a handle for the Zgraph user interface into all
;;defined graph types.

 *graph-types*


;;This section sets up a graph type for generating random graphs.

(DEFVAR *generated-graph* NIL)

(DEFUN gengraph (num-vertices num-edges)
  ;;Clear out previous
  (DOLIST (symbol *generated-graph*)
    (SET symbol NIL))
  ;;Make sure we have enough vertices.
  (DOTIMES (i (- num-vertices (LENGTH *generated-graph*)))
    (PUSH (GENSYM) *generated-graph*)
    (SET (CAR *generated-graph*) NIL))
  ;;Throw out excess vertices (wasteful)
  (SETQ *generated-graph* (NTHCDR (- (LENGTH *generated-graph*) num-vertices)
				  *generated-graph*))
  (DOTIMES (i num-edges)
    (PUSH (NTH (RANDOM num-vertices) *generated-graph*)
	  (SYMBOL-VALUE (NTH (RANDOM num-vertices) *generated-graph*)))))


(DEFUN get-random-graph ()
  (LET ((v 50)
	(e 50))
    (DECLARE (SPECIAL v e))
    (gengraph v e)
    *generated-graph*))

#|
(DEFUN get-random-graph ()
  (LET ((v 100)
	(e 100))
    (DECLARE (SPECIAL v e))
    (TV:CHOOSE-VARIABLE-VALUES
      `((v "Number of Vertices" :NUMBER)
	(e  "Number of Edges" :NUMBER))
      :LABEL "Enter Specifications for Randomly Generated Graph")
    (gengraph v e)
    *generated-graph*))
|#
(DEFUN random-graph-traversal-function (symbol)
  (do ((v (SYMBOL-VALUE symbol) (cdr v))
       (result nil))
      ((null v) (nreverse result))
    (push (CONS (car v) "Value") result))
  
;  (LOOP FOR v IN (SYMBOL-VALUE symbol)
;	COLLECT (CONS v "Value"))
  )

(MAKE-INSTANCE 'graph-type
	       :name 'random-graph-type
	       :traversal-function 'random-graph-traversal-function
	       :default-root-finding-form '(get-random-graph)
               :traverse-recursively? NIL)



;;This section sets up more vertices and edges to provide several additional demos
;;(besides the one you get when first creating a graph of type SAMPLE-GRAPH-TYPE).  If you
;;would like to see these demos, execute the STATUS command and select from one of the
;;four forms in the "Roots for the SAMPLE-GRAPH-TYPE" field.  Then create a graph of type
;;SAMPLE-GRAPH-TYPE.

(DEFVAR *sample-root-list*
	'('(v1) '(v12) '(v20) '(v41 v48)) "Used as quick way to demo different graphs.")


(PROCLAIM '(SPECIAL v7 v8 v9 v10 v11 v12 v13 v14 v15
		    v16 v17 v18 v19 v20 v21 v22 v23 v24 v25 v26 v27 v28 v29 v30 v31
		    v32 v33 v34 v35 v36 v37 v38 v39 v41 v42 v43 v44 v45 v46 v47 v52
		    v53 v55 v56 v54 v58 v59 v60 v61 v62 v63 v48 v49 v50 v51))
;;Second demo
(LET ((vertices '(v11 v12 v14 v15 v17 v18)))
  (DOLIST (v vertices)
    (SET v (REMOVE v vertices)))
  (PUSH 'v13 v11)
  (PUSH 'v13 v14)
  (SETQ v11 (APPEND v11 '(v16)))
  (SETQ v14 (APPEND v14 '(v16)))
  (SETQ v13 NIL)
  (SETQ v16 NIL))
;;Third demo (the monster)
(LET ((vertices '(v20 v21 v22 v23 v24 v25 v26 v27 v28 v29 v30 v31 v32 v33 v34 v35 v36 v37 v38
		      v39)))
  (DOLIST (v vertices)
    (SET v (REMOVE v vertices))))
;;Fourth demo
(SETQ v41 '(v42 v43 v44)
      v42 '(v41 v43)
      v43 '(v41 v42)
      
      v44 '(v45 v46 v47)
      v45 '(v46)
      v46 '(v52)
      v47 '(v45)
      v52 '(v53 v54 v55)
      v53 NIL
      v55 '(v56)
      v56 NIL
      
      v54 '(v59 v58)
      v58 '(v61 v62)
      v59 '(v58 v60 v62)
      v60 '(v61 v62 v63)
      v61 NIL
      v62 NIL
      v63 NIL
      
      v48 '(v49 v50 v51)
      v49 NIL
      v50 '(v51)
      v51 NIL)

