;;; -*- Mode: LISP; Syntax: Common-lisp; Base: 10; Fonts: CPTFONT,CPTFONTB; Package: ZG -*-

1;;Copyright (c) 1986 by John C. Hogge, The University of Illinois.
;;
;;File INTERFACE.LISP of system Zgraph.

;;
;;This file sets up the Zgraph user interface.  The display pane of this interface is set up
;;for standalone use in an application--to see how to do this, refer to the documentation 
;;string on flavor ZG:GRAPH-DISPLAY-PANE.
;;

;;==========================================================================================
@;;
1;;;; The user interface (accessed via SYSTEM G).
@;;
1;;==========================================================================================

0(DEFVAR *default-graph-display-pane-keystroke-commands*
#+Symbolics
	'((1089. :change-status) 1;;#\s-a
0	  (1114. :zoom) 1;;#\s-z
0	  (1094. :keyboard-zoom) 1;;#\s-f
0	  (1106. :reset-scale) 1;;#\s-r
0	  (1092. :display-graph) 1;;#\s-d
0	  (1098. :display-self-loops) 1;;#\s-j
0	  )
#+Explorer
	'((#\s-a :change-status)
	  (#\s-z :zoom) 
	  (#\s-f :keyboard-zoom) 
	  (#\s-r :reset-scale) 
	  (#\s-d :display-graph) 
	  (#\s-j :display-self-loops) 
	  )
  "1Keystroke commands interpreted (by default) by flavor GRAPH-DISPLAY-PANE.
2CDRs are messages to send to the graph display pane to execute keystroke command.0")

(DEFVAR *zgraph-frame-keystroke-commands*
#+Symbolics
	`((1095. (define-graph-type)) 1;;#\s-g
0	  (1093. (edit-graph-type)) 1;;#\s-e
0	  (1091. (SEND *gd* :create-graph)) 1;;#\s-c
0	  (1107. (save-graph-types)) 1;;#\s-s
0	  (1100. (load-graph-types)) 1;;#\s-l

0	  (134. (SEND *gd* :help)) 1;;#\HELP
0	  1;;Puts user into Zmacs with buffer containing sample graph, traversal function, and
0	  1;;default root finding forms.
0	  (646. (ED 'v1)) 1;;#\META-HELP (doesn't appear in command menu)
0	  1;;Puts user into Zmacs with buffer containing a sample of how to integrate Zgraph
0	  1;;into application code.
0	  (1158. (ED 'functional-interface-example)) 1;;#\SUPER-HELP, ditto
0	  . ,*default-graph-display-pane-keystroke-commands*)
#+Explorer
	`((#\s-g (define-graph-type))
	  (#\s-e (edit-graph-type))
	  (#\s-c (SEND *gd* :create-graph))
	  (#\s-s (save-graph-types))
	  (#\s-l (load-graph-types))

	  (#\HELP (SEND *gd* :help))
	  1;;Puts user into Zmacs with buffer containing sample graph, traversal function, and
0	  1;;default root finding forms.
0	  (#\META-HELP (ED 'v1)) 1;; (doesn't appear in command menu)
0	  1;;Puts user into Zmacs with buffer containing a sample of how to integrate Zgraph
0	  1;;into application code.
0	  (#\SUPER-HELP (ED 'functional-interface-example)) 1;; ditto
0	  . ,*default-graph-display-pane-keystroke-commands*)
  "1Keystroke commands interpreted by0 1the Zgraph user interface.  These include three of the
standard Graph-Display-Pane commands, our own version of the Status command (which
contains more options than Graph-Display-Pane's Status command, command local to Zgraph.0")


(DEFVAR *miscellaneous-command-menu*
	`(("Display Self Loops" :EVAL (SEND *display-io* :display-self-loops)
	   :documentation
	   ,(si:string "{SUPER-j}  Display any self loops of vertices in the current graph."))
	  ("Clear Description Scroll Window" :EVAL
	   (WHEN *description-output*
	     (IF (NOT (SEND *description-output* :EXPOSED-P))
		 (FORMAT T 2"~%Description Pane not exposed."0)
		 (SEND *description-output* :SET-ITEMS NIL)))
	   :documentation
	   ,(si:string "2Clear the scroll window when in WITH-DESCRIPTION-PANE configuration 1(see Status command)0"))
	  ("Save Description Scroll Window" :EVAL 
	   (WHEN *description-output*
	     (IF (NOT (SEND *description-output* :EXPOSED-P))
		 (FORMAT T 2"~%Description Pane not exposed."0)
		 (SEND *gd* :save-scroll-window)))
	   :documentation
	   ,(si:string "2Save the text displayed in the scroll window into a file.0"))
	  )
  2"Menu of misc. commands.  Applications should feel free to push onto this.
Used as first argument to TV:MENU-CHOOSE, so your own items should be of type :EVAL or
:FUNCALL."0)

(DEFVAR *default-graph-display-pane-menu-commands*
	`(("Display Graph" :VALUE :display-graph
	   :documentation
	   ,(si:string "{SUPER-d}  Display a previously created graph"))
	  ("Status" :VALUE :change-status
	   :documentation
	   ,(si:string "{SUPER-a}  Alter the operation of the graph displayer."))
	  ("Menu Zoom" :VALUE :zoom
	   :documentation
	   ,(si:string "{SUPER-z}  Zoom in or out on the currently displayed graph."))
	  ("Kbd2.0 Zoom" :VALUE :keyboard-zoom
	   :documentation
	   ,(si:string "{SUPER-f}  Zoom in or out by providing a flonum scale."))
	  ("Reset Scale" :VALUE :reset-scale
	   :documentation
	   ,(si:string "{SUPER-r}  Reset the scale to effectively undo all previous Zooms"))
	  ("Misc2.0" :VALUE :miscellaneous-command-menu
	   :documentation
	   ,(si:string "2Submenu of misc. commands.  Applications can add to this via 1ZG:*MISCELLANEOUS-COMMAND-MENU*0")))
 "1Menu item list used in Graph-Display-Pane's MOUSE-RIGHT pop-up menu command.
2Item types are all :VALUE and pass back a message to send to the graph display pane if a
symbol,0 2otherwise they are forms to eval.0")



(DEFVAR *zgraph-frame-menu-commands*
	`(("Help" :VALUE (SEND *gd* :help)
	   :documentation ,(si:string "{HELP}  Prints a short help message."))
	  ("Graph Types" :VALUE
	   (TV:MENU-CHOOSE '(("Define Graph Type" :FUNCALL define-graph-type
			      :documentation
			2      0,(si:string "{SUPER-g}  Define a new type of graph to display"))
			     ("Edit Graph Type" :FUNCALL edit-graph-type
			      :documentation
			2      0,(si:string "{SUPER-e}  Edit the attributes of a previously defined graph type."))
			     ("Save Graph Types" :FUNCALL save-graph-types
			      :documentation
			      ,(si:string "{SUPER-s}  Save currently defined graph types into a file"))
			     ("Load Graph Types" :FUNCALL load-graph-types
			      :documentation
			      ,(si:string "{SUPER-l}  Load previously saved graph types from a2 0file")))
			   (si:string 2"Graph Type Commands"0))
	   :documentation
	   ,(si:string "2Brings up a submenu of commands for operating on graph types.0"))
			   
	  ("Create Graph" :VALUE (SEND *gd* :create-graph)
	   :documentation
	   ,(si:string "{SUPER-c}  Create and display a graph of some pre-specified type"))
	  . ,*default-graph-display-pane-menu-commands*)
  "1Menu item list used in the Zgraph user interface.  These include three of the
standard Graph-Display-Pane commands, our own version of the Status command (which
contains more options than Graph-Display-Pane's Status command, command local to Zgraph.
2Item types are all :VALUE and pass back a message to send to the graph display pane if a
symbol,0 2otherwise they are forms to eval.0")


1;;
;; ZG:GRAPH-DISPLAY-PANE, Zgraph's graph display pane.
;; Also usable as a standalone graph displayer. 
;;

0(DEFFLAVOR graph-display-pane ((graph NIL)
			       (graph-history NIL)
			       (keystroke-commands
				 *default-graph-display-pane-keystroke-commands*)
			       (menu-commands
				 *default-graph-display-pane-menu-commands*)
			       (horizontal-pan-% .5)
			       (vertical-pan-% .5)
			       (graphics-object-being-moved NIL))
	   (TV:DONT-SELECT-WITH-MOUSE-MIXIN
	    mouse-sensitive-graphics-mixin      1;for mouse sensitive vertices and edges
0	    meta-window)			1;for windowing into a large output bitarray
0  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables
  (:INIT-KEYWORDS :real-window-flavor)
  (:REQUIRED-INIT-KEYWORDS :extended-width :extended-height)
  (:DEFAULT-INIT-PLIST
   1;;NIL label gives us precious more screen space.
0   :LABEL NIL
   1;;Who line message to display when not panning about (when MOUSE-L isn't held down).
0   1;;The difference between this and the default (for META-WINDOW) is the mention of
0   1;;MOUSE-M and MOUSE-R operations.
0   :WHO-LINE-DOCUMENTATION-WHEN-NOT-PANNING
   (SI:STRING (FORMAT NIL
    "LHold: Fast Pan,  M: Select Highlited Vertex or Edge,  CTRL-M: Move Highlited2 0Vertex, R: Command Menu"))) 
  (:DOCUMENTATION :COMBINATION "
1ZG:GRAPH-DISPLAY-PANE is set up for use both as a pane in the Zgraph user interface and as a
standalone window an application can use.  GRAPH-DISPLAY-PANE is initialized with a default
set of keystroke and menu commands (stored in instance variables KEYSTROKE-COMMANDS and 
MENU-COMMANDS) which are appropriate for standalone use.  An application can modify this set
as needed.  The Zgraph user interface (ZG:GRAPH-DISPLAY-FRAME) does so and is a good model 
for using GRAPH-DISPLAY-PANE.

Instance Variables:
  GRAPH0 1--0 1either NIL or a GRAPH struct which is currently displayed on SELF.

  GRAPH-HISTORY -- list of previously displayed graphs.  GRAPH is always the CAR of this list,
unless GRAPH is NIL.  This is used to provide an option for viewing previously displayed 
graphs.

  KEYSTROKE-COMMANDS0 1--0 1alist of (KEY . COMMAND) where COMMAND, if a symbol, is sent as a
message to SELF.  If COMMAND is instead a list, it is evaluated.  The latter case is provided
so that an application can add its own commands to those normally executed by the graph
display pane.  The former case is the default mode of command execution, since the default
value of KEYSTROKE-COMMANDS is ZG:*DEFAULT-GRAPH-DISPLAY-PANE-KEYSTROKE-COMMANDS*.

  MENU-COMMANDS0 1--0 1menu item list whose entries are of the form (string :VALUE COMMAND...)
where COMMAND is just as described under KEYSTROKE-COMMANDS.  This menu item list is used only
for handling MOUSE-R on SELF, which allows the user to execute a command from a pop-up menu.
MENU-COMMANDS is especially nice to modify so that an application can stuff some of its own
commands into the pop-up menu.  To do this, it would add to MENU-COMMANDS entries of the form
   (string :VALUE (form-to-eval)...)
where (FORM-TO-EVAL), when evalled, executes some application command.  Note that you CAN
use items of type other than :VALUE, such as :CHOOSE or :EVAL.  However, the execution of
these menu blips better not return any symbols, since they will then be sent as messages to
the graph display pane.
  The default value for MENU-COMMANDS is ZG:*DEFAULT-GRAPH-DISPLAY-PANE-MENU-COMMANDS*.


Steps for using the GRAPH-DISPLAY-PANE in an application:

1.  Initialize instance variables KEYSTROKE-COMMANDS and MENU-COMMANDS to whatever commands
    you want the pane to interpret0,1 or accept the defaults.

2.  In your application command loop, any blip or keystroke which isn't yours you should send
    to the graph diplay pane via the :HANDLE-INPUT method.  :HANDLE-INPUT takes the input
    as argument and processes it as follows:
    1. If input is a menu blip, it is executed to return a message to send to SELF or a list
       to EVAL (as described above).
    2. If input is a mouse blip, left = fast pan the graph, middle = select vertex or edge,
       right = command menu.  The menu command calls TV:MENU-CHOOSE with instance variable
       MENU-COMMANDS as value.
    3. If input is a keystroke, it is looked up in the KEYSTROKE-COMMANDS instance var.
       If found, the CADR is EVALed if its a list or sent to SELF otherwise.

0    1Otherwise, the pane doesn't recognize the input.  What it does depends on the value of 
0    1the optional argument to :HANDLE-INPUT.  If it is NIL, we beep on bad mouse buttons and 
0    1keystrokes and err on everthing else.  (This is handled by method 
0    1:HANDLE-UNRECOGNIZED-INPUT.)  If it is not NIL, we return it as a value and let the caller
0    1handle bad input.  For instance, in a command loop the caller might pass us a single 
0    1keystroke to be processed.  If we don't recognize it and it has no control bits, the 
0    1caller0 1might want to :UNTYI it and read an expression from the user.  As an example, this
0   1 is done in0 1method (ZG:GRAPH-DISPLAY-FRAME :COMMAND).

3.  Include ZG:GRAPH-DISPLAY-FRAME as a pane in your constraint frame.  Alternatively, it
    can be used as a pop-up window via TV:WINDOW-CALL.

4.  Set up a graph for display as follows:
    1.  Instantiate an instance of GRAPH-TYPE appropriate for displaying your data structure.
    2.  Then instantiate an instance of GRAPH to build a graphic model of your data structure
        which Zgraph can read.  A required init keyword of GRAPH is :TYPE, which should be
        an instance of GRAPH-TYPE.
    3.  Send the graph instance a :CONSTRUCT message, followed by a :PLOT-VERTICES message.
    4.  Send the graph display window a :SET-GRAPH message with your graph instance as 
        argument
    All this is somewhat complex.  Function ZG:FUNCTIONAL-INTERFACE-EXAMPLE provides an
    example of all this.  Do meta-point on it to see the code.0"))



(DEFMETHOD (graph-display-pane :set-graph) (new-graph &OPTIONAL (update-display? T))
  "1Sets instance variable GRAPH to NEW-GRAPH and updates the graph history.
If the optional argument is not non-NIL (the default), do the following:
If0 1NEW-GRAPH is NIL, clear the REAL-WINDOW and SELF.  Otherwise, we
reset our displacement, draw the graph onto REAL-WINDOW, and display a portion of
REAL-WINDOW on SELF.0"
1  ;;Do this stuf before setting GRAPH, since the value of GRAPH directs the mouse sensitivity
0  1;;and related who-line help code. If we didn't do this, it would be possible to display a
0  1;;graph other than GRAPH, while having mouse-sensitivity activated for GRAPH.  (Invisible 
0  1;;vertices and edges would be mouse-sensitive).
0  (WHEN update-display?
    (COND
      ((NULL new-graph)
       (SEND real-window :CLEAR-WINDOW)
       (SEND SELF :update-image))
      ((NOT (TYPEP new-graph 'graph))
       (FERROR "Instance variable ZG:GRAPH must be a ZG:GRAPH flavor instance or NIL."))
      (T
       1;;Reset the displacement incase we've panned a previously displayed graph
0       (SEND SELF :update-displacement 0 0 T)
       1;;Output graph onto REAL-WINDOW.
0       (SEND new-graph :draw real-window)
       1;;Copy bit array from real window to SELF.
0       (SEND SELF :update-image))))
  (WHEN (AND new-graph (NEQ new-graph graph))
    (SETQ graph-history (CONS new-graph (REMOVE new-graph graph-history))))
  (SETQ graph new-graph))


(DEFMACRO minimizing ((variable evaluator &OPTIONAL (max-value MOST-POSITIVE-FIXNUM)) &BODY body)
  2"Not quite general enough for external use, but what the hey.  Here's an example call:
1  (minimizing (x (+ (CAR x) (CADR x)))
    (DOLIST (thing '((2 5) (3 4) (3 9)))
      (try-value-of thing)))2"
0  (LET ((internal-function-name (GENSYM)))
    `(LET ((minimum-value ,max-value)
	   thing-with-minimum-value
	   value)
       (LABELS ((,internal-function-name (,variable)
		 (SETQ value ,evaluator)
		 (WHEN (< value minimum-value)
		   (SETQ minimum-value value
			 thing-with-minimum-value ,variable))))
	 ,@(SUBST internal-function-name 'try-value-of body))
       (VALUES thing-with-minimum-value minimum-value))))


(DEFUN sqr (x) (* x x))

(DEFMETHOD (graph-display-pane :item-at?) (x y)
  "1Required by flavor ZG:MOUSE-SENSITIVE-GRAPHICS-MIXIN.
Returns NIL or the mouse-sensitive item at coordinates X,Y.  In our case, the item is
a VERTEX or EDGE struct.0"
  (WHEN graph
      1;;The calls to :INVERSE-SCALE-X and :INVERSE-SCALE-Y go from screen
0      1;;coordinates to the graph's world coordinates.
0      1;;The additions take into account the fast-panning which displaces the graph.
0      (LET* ((max-delta (- (SEND graph :inverse-scale-x 10)
			   (SEND graph :inverse-scale-x 0)))
	     (x (SEND graph :inverse-scale-x (+ x x-displacement)))
	     (y (SEND graph :inverse-scale-y (+ y y-displacement)))
	1     ;;Get location groupings (clumps) of vertices and edges.
0	     (clumps (SEND graph :location-clumps)))
	1;;Find the closest vertex or edge to X,Y and its squared distance to it
0	(MULTIPLE-VALUE-BIND (closest-thing distance)
	    (minimizing (thing (LET ((loc (IF (TYPEP thing 'vertex)
					      (vertex-location thing)
					      (edge-misc thing))))
				 (+ (sqr (- x (CAR loc))) (sqr (- y (CDR loc))))))
	      (LABELS ((try-clump (c)
			 (WHEN c
			   (DOLIST (vertex (THIRD c))
			     (try-value-of vertex))
			   (DOLIST (edge (FOURTH c))
			     (try-value-of edge)))))
		1;;Look for closest vertex/edge not only in the clump engulfing X,Y but also in its
0		1;;neighboring clumps (if they exist).  Neglecting the neighbors would cause weird
0		1;;behavior in the mouse sensitivity when the mouse is near the border of a clump.
0		(try-clump (get-clump x y clumps))
		(try-clump (get-clump (- x max-delta) y clumps))	;left neighbor
		(try-clump (get-clump (+ x max-delta) y clumps))	;right neighbor
		(try-clump (get-clump x (- y max-delta) clumps))	;down neighbor
		(try-clump (get-clump x (+ y max-delta) clumps))))	;up neighbor
	  1;;Eliminate if not within MAX-DELTA (otherwise the mouse will always detect something)0.
	1  ;;Square distance because mins. are squared.
0	  (WHEN (<= distance (sqr max-delta))
	    closest-thing)))))


(DEFMETHOD (graph-display-pane :box-item-at) (item IGNORE IGNORE)
  "1Required by flavor ZG:MOUSE-SENSITIVE-GRAPHICS-MIXIN.
Highlites ITEM, which is a VERTEX or EDGE struct which the mouse has moved near.
Since we use XOR, ZG:MOUSE-SENSITIVE-GRAPHICS-MIXIN can also use this method to
erase the highliting.0"
  (COND
    ((TYPEP item 'vertex)
     (LET ((loc (vertex-location item)))
       (SEND SELF :DRAW-CIRCLE
	     (- (SEND graph :scale-x (CAR loc)) x-displacement)
	     (- (SEND graph :scale-y (CDR loc)) y-displacement)
	     10 TV:ALU-XOR)))
    ((TYPEP item 'edge)
     (LET* ((loc (edge-misc item))
	    (x (- (SEND graph :scale-x (CAR loc)) x-displacement))
	    (y (- (SEND graph :scale-y (CDR loc)) y-displacement))
	    (slope (edge-slope item)))
       1;;First call blanks out the pointer already drawn as part of the edge.
0       1;;Second call then writes a larg version of the pointer, as mouse-sensitive
0       1;;indicator.  Using XOR on both drawns means the original pointer will be restored.
0       (draw-pointer SELF x y slope 8. TV:ALU-XOR)
       (draw-pointer SELF x y slope 20. TV:ALU-XOR))
     (LET* ((head-loc (vertex-location (edge-vertex-1 item)))
	    (tail-loc (vertex-location (edge-vertex-2 item)))
	    (head-x (- (SEND graph :scale-x (CAR head-loc)) x-displacement))
	    (head-y (- (SEND graph :scale-y (CDR head-loc)) y-displacement))
	    (tail-x (- (SEND graph :scale-x (CAR tail-loc)) x-displacement))
	    (tail-y (- (SEND graph :scale-y (CDR tail-loc)) y-displacement)))
       1;;First blank out the edge's line, then redraw it as a dashed line.  If
0       1;;we don't do the bank-out, the results looks trashy since the jaggies on a solid
0       1;;line are different from those on a dashed line.  Note however: this looks great 
0       1;;only if the entire edge lies within the display pane.  If any of the edge is
0       1;;outside the display pane, this usually looks sloppy since these lines are being
0       1;;clipped to the display pane, while the original line was clipped to the real
0       1;;window (ie. the endpoints of the lines don't coincide, necessarily).  Is there a
0       1;;good solution??  Doing the XOR onto the real window and then updating takes too
0       1;;long--that's the ultimately solution.  A speed hack for this would be to update
0       1;;(bitblt) only the extents of the line.
0       (SEND SELF :DRAW-LINE head-x head-y tail-x tail-y TV:ALU-XOR)
       (SEND SELF :DRAW-DASHED-LINE head-x head-y tail-x tail-y TV:ALU-XOR)))))

(DEFMETHOD (graph-display-pane :item-who-line-documentation) (item)
  "1This method is used in conjunction with flavor ZG:MOUSE-SENSITIVE-GRAPHICS-MIXIN0.
1Returns who-line doc to display when the mouse is over a vertex or edge.0"
  (COND
    ((TYPEP item 'vertex)
     (si:string
       (FORMAT NIL "LHold: Fast pan,  M: Menu of operations on vertex ~s,  R: Command Menu"
	       1;;In the normal case, (GET-REAL-VERTEX ITEM) just returns ITEM.  But if
0	       1;;we are displaying a metagraph (during recursive display of a vertex or edge),
0	       1;;GET-REAL-VERTEX returns what we want.
0	       (vertex-data (get-real-vertex item)))))
    ((TYPEP item 'edge)
       (si:string
	 (FORMAT NIL
	     "LHold: Fast pan, M: Menu of operations on edge from ~s to ~s,  R: Command Menu"
	1     ;;Comment above applies.
0	     (vertex-data (get-real-vertex (edge-vertex-1 item)))
	     (vertex-data (get-real-vertex (edge-vertex-2 item))))))))


(DEFMETHOD (graph-display-pane :handle-input) (input &OPTIONAL do-nothing-on-bad-input)
  "1Tries to interpret INPUT as a command (relevant to the currently stored GRAPH).
INPUT is typically read by a call to :ANY-TYI in an application command loop and
is found not to be among the application commands.  It can be a menu blip, mouse blip,
or keystroke and is handled as follows:
  1. If INPUT is a menu blip, it is executed to return a message handlable by SELF.
     The message is sent to SELF with no arguments.
  2. If INPUT is a mouse blip, left = fast pan the graph, middle = select vertex or edge,
     right = command menu.
  3. If INPUT is a keystroke, it is looked up in the keystroke-commands instance var.
     If found, the cooresponding message is sent to SELF with no arguments.

Otherwise, we don't recognize the input.  What we do depends on the value of the optional
argument.  If it is NIL, we beep on bad mouse buttons and keystrokes and err on everthing 
else.  (This is handled by method :HANDLE-UNRECOGNIZED-INPUT.)  If it is not NIL, we return
it as a value and let the caller handle bad input.  For instance, in a command loop the
caller might pass us a single keystroke to be processed.  If we don't recognize it and it has
no control bits, the caller might want to :UNTYI it and read an expression from the user.0"
  (IF (LISTP input)
      (CASE (CAR input)
	1;;Execute a menu command.
0	(:MENU
	 (LET ((command (SEND (FOURTH input) :EXECUTE (SECOND input))))
	   (IF command
	       (SEND SELF :execute-command command)
	       (OR do-nothing-on-bad-input
		   (SEND SELF :handle-unrecognized-input input)))))
	(:MOUSE-BUTTON
	 (COND
	   ((= (CADR input) #\MOUSE-L)
	    (LET ((hit-window-edge? (SEND SELF :pan-around)))
	      1;;The meta-window is set up to return a flag when the user
0	      1;;hits up against the window edge.
0	      1;;during :PAN-AROUND.  If he does this, we do a real pan.
0	      1;;SELF handles four messages,
0	      1;;one for each keyword returned by :PAN-AROUND--:LEFT, :RIGHT,
0	      1;;:UP, and :DOWN.
0	      (WHEN hit-window-edge?
		(SEND SELF hit-window-edge?))))
	   ((= (CADR input) #\MOUSE-M)
	    (SEND SELF :select-graphics-object input))
	   ((= (CADR input) #\MOUSE-R)
	    (SEND SELF :pop-up-menu-command))
	   ((= (CADR input) #+Symbolics #\CONTROL-MOUSE-M #+Explorer #\MOUSE-M-2)
	    (SEND SELF :move-graphics-object input))
	   1;;Bad mouse button
0	   (T (OR do-nothing-on-bad-input
		  (SEND SELF :handle-unrecognized-input input)))))
	(T (OR do-nothing-on-bad-input
	       (SEND SELF :handle-unrecognized-input input))))
      1;;Else key input.  If key is a keystroke command, send ourself the command
0      1;;message.  Otherwise put the input back into the buffer and read in a Lisp
0      1;;expression to evaluate.
0      (LET ((command (ASSOC input keystroke-commands
			    #+Explorer :TEST #+Explorer #'=)))
	(IF command
	    (SEND SELF :execute-command (CADR command))
	    (OR do-nothing-on-bad-input
		(SEND SELF :handle-unrecognized-input input))))))

(DEFMETHOD (graph-display-pane :left) ()
  "1Does a real pan left.0"
  (SEND SELF :pan (* horizontal-pan-% (SEND SELF :INSIDE-WIDTH)) 0 "left..."))

(DEFMETHOD (graph-display-pane :right) ()
  "1Does a real pan right.0"
  (SEND SELF :pan (- (* horizontal-pan-% (SEND SELF :INSIDE-WIDTH))) 0 "right..."))

(DEFMETHOD (graph-display-pane :up) ()
  "1Does a real pan up.0"
  (SEND SELF :pan 0 (* vertical-pan-% (SEND SELF :INSIDE-HEIGHT)) "up..."))

(DEFMETHOD (graph-display-pane :down) ()
  "1Does a real pan down.0"
  (SEND SELF :pan 0 (- (* vertical-pan-% (SEND SELF :INSIDE-HEIGHT))) "down..."))

(DEFMETHOD (graph-display-pane :pan) (x y message)
  "1Pans the graph by x,y relative.0"
  (WHEN graph
    (WHEN (MEMQ :print-graph-computation-messages  *graph-debug-actions*)
      (SEND *STANDARD-OUTPUT* :CLEAR-WINDOW)
      (FORMAT T "Panning further ~a" message))
    (SEND graph :pan x y)
    (SEND graph :draw)
    (SEND SELF :CLEAR-WINDOW)
    (SEND SELF :update-image)
    (WHEN (MEMQ :print-graph-computation-messages  *graph-debug-actions*)
      (FORMAT T "done."))
    (TERPRI)))


(DEFMETHOD (graph-display-pane :execute-command) (command)
  "1Attempts to handle COMMAND, obtained either from the CDR of an element in
KEYSTROKE-COMMANDS or the value of the execution of a menu blip from the MENU-COMMANDS. 
If COMMAND is a list, it is a form to eval (and was provided by the application using
GRAPH-DISPLAY-PANE).  Otherwise it is assumed to be a message to send to SELF (a local
command method).0"
  (IF (LISTP command)
      (EVAL command)
      (SEND SELF command)))

(DEFMETHOD (graph-display-pane :handle-unrecognized-input) (input)
  "1Handles unrecognized keystrokes and mouse buttons by beeping--errs on other input.0"
  (IF (LISTP input)
      (IF (EQ (CAR input) :MOUSE-BUTTON)
	  (BEEP)
	  (FERROR "Unable to handle weird blip ~s" input))
      (BEEP)))


(DEFMETHOD (graph-display-pane :select-graphics-object) (mouse-blip)
  "1Handles MOUSE-M clicks.0  1MOUSE-M is used for selecting vertices and edges in the graph.0"
  (IF (OR (NULL graph) (NOT (EQ SELF (THIRD mouse-blip))))
      (BEEP)
      (LET ((selected-object (SIXTH mouse-blip)))
	(WHEN selected-object
	  (SEND (SEND graph :type) :handle-selection-of-object selected-object SELF)))))

(DEFMETHOD (graph-display-pane :move-graphics-object) (mouse-blip)
  2"Handles MOUSE-CONTROL-M clicks.  This is used to dynamically move vertices and edges.
User might want to do this to organze the output for easier viewing or for pretty screen
dumps."
0  (IF (OR (NULL graph) (NOT (EQ SELF (THIRD mouse-blip))))
      (BEEP)
      1;;If the user previously selected an object to be moved, MOUSE-BLIP indicates
0      1;;where to put it.  Otherwise, store the selected object and tell the user
0      1;;how to put it somewhere.
0      (COND
	(graphics-object-being-moved
	 (SETF (vertex-location graphics-object-being-moved)
	       (CONS (SEND graph :inverse-scale-x (+ (FOURTH mouse-blip) x-displacement))
		     (SEND graph :inverse-scale-y (+ (FIFTH mouse-blip) y-displacement))))
	 (SEND graph :vertex-has-moved graphics-object-being-moved)
	 (SETQ graphics-object-being-moved NIL)
	 (FORMAT T 2"~%Redrawing graph."0)
	 (SEND graph :draw)
	 (SEND SELF :CLEAR-WINDOW)
	 (SEND SELF :update-image))
	(T
	 (LET ((selected-object (SIXTH mouse-blip)))
	   (COND
	     ((TYPEP selected-object 'vertex)
	      (SETQ graphics-object-being-moved selected-object)
	      (FORMAT T 2"~%You have selected vertex ~s to be moved.
Now move the mouse to the desired location (pan to it if need be) and click
MOUSE-CONTROL-M again to complete the move."0 (vertex-data selected-object)))
	     (T (BEEP))))))))


(DEFMETHOD (graph-display-pane :pop-up-menu-command) ()
  (LET ((command (TV:MENU-CHOOSE menu-commands)))
    (WHEN command
      (SEND SELF :execute-command command))))

1;;
;; Commands
;;


0(DEFVAR *zoom-menu-items* 
	(LOOP FOR factor IN '(1/4 1/3 1/2 2/3 3/4 1 4/3 3/2 2 3 4) COLLECT
	      `(,(FORMAT NIL "        ~s        " factor) :VALUE ,factor
		:DOCUMENTATION ,(SI:STRING (FORMAT NIL "Zoom ~:[In~;Out~] by ~s"
						   (< factor 1)
						   (IF (< factor 1)
						       (/ 1 factor)
						       factor))))))
(DEFMETHOD (graph-display-pane :zoom) ()
  (IF graph
      (LET ((scale (TV:MENU-CHOOSE *zoom-menu-items* (SI:STRING "Zoom factor (multiplier)"))))
	(WHEN scale
	  (SEND graph :scale-relative scale scale)
	  (SEND graph :draw)
	  (SEND SELF :CLEAR-WINDOW)
	  (SEND SELF :update-image)))
      (BEEP)))

(DEFMETHOD (graph-display-pane :keyboard-zoom) ()
  (IF graph
      (LET ((scale (prompt-and-read ':number "Scale (numbers over 1.0 zoom In): ")))
	(WHEN scale
	  (SEND graph :scale-relative scale scale)
	  (SEND graph :draw)
	  (SEND SELF :CLEAR-WINDOW)
	  (SEND SELF :update-image)))
      (BEEP)))

(DEFMETHOD (graph-display-pane :reset-scale) ()
  (COND
    (graph
     (SEND graph :reset-position-and-scale)
     (SEND SELF :update-displacement 0 0 T)
     (SEND graph :DRAW)
     (SEND SELF :CLEAR-WINDOW)
     (SEND SELF :update-image))
    (T
     (BEEP))))


#+Symbolics
(DEFMETHOD (graph-display-pane :display-graph) ()
  (WHEN graph
    1;;Before displaying another graph, make whatever fast panning has occurred (since the
0    1;;most recent real pan) a real pan.  This way, when the user reselects
0    1;;GRAPH, it will be positioned the way he last saw it.
0    (SEND graph :pan (- x-displacement) (- y-displacement)))
  (MULTIPLE-VALUE-BIND (selected-graph button)
      (TV:MENU-CHOOSE (LOOP FOR g IN graph-history
			    COLLECT `(,(SEND g :name)
				      1;;Returns GRAPH as first value, and the 1st or
0				      1;;second cons (depending on mouse button) as 2nd value
0				      :BUTTONS ((1 . ,g) (2 . ,g))
				      :DOCUMENTATION ,(SI:STRING "L: Display,  M: Delete"))))
    (WHEN selected-graph
      (CASE (CAR button)
	(1
	  (debug-print T "~%Drawing graph on hidden bit-array...")
	  (SEND SELF :set-graph selected-graph)
	  (debug-print T "done."))
	(2
	  (SETQ graph-history (REMOVE selected-graph graph-history))
	  (WHEN (EQ selected-graph graph)
	    (debug-print T "~%Current graph deleted.")
	    (debug-print T "~%Drawing most recent graph on hidden bit-array...")
	    (SEND SELF :set-graph (CAR graph-history))
	    (debug-print T "done.")))))))
#+Explorer
(DEFMETHOD (graph-display-pane :display-graph) ()
  (WHEN graph
    1;;Before displaying another graph, make whatever fast panning has occurred (since the
0    1;;most recent real pan) a real pan.  This way, when the user reselects
0    1;;GRAPH, it will be positioned the way he last saw it.
0    (SEND graph :pan (- x-displacement) (- y-displacement)))
  (LET ((selection (TV:MENU-CHOOSE
		     (LOOP FOR g IN graph-history
			   COLLECT `(,(SEND g :name)
				     1;;Returns GRAPH and indicator of which mouse button was pressed.
0				     :BUTTONS ((NIL :VALUE (1 . ,g)) (NIL :VALUE (2 . ,g)))
				     :DOCUMENTATION ,(SI:STRING "L: Display,  M: Delete"))))))
    (WHEN selection
      (CASE (CAR selection)
	(1
	  (debug-print T "~%Drawing graph on hidden bit-array...")
	  (SEND SELF :set-graph (CDR selection))
	  (debug-print T "done."))
	(2
	  (SETQ graph-history (REMOVE (CDR selection) graph-history))
	  (WHEN (EQ (CDR selection) graph)
	    (debug-print T "~%Current graph deleted.")
	    (debug-print T "~%Drawing most recent graph on hidden bit-array...")
	    (SEND SELF :set-graph (CAR graph-history))
	    (debug-print T "done.")))))))

(DEFMETHOD (graph-display-pane :miscellaneous-command-menu) ()
  (TV:MENU-CHOOSE *miscellaneous-command-menu* (SI:STRING 2"Miscellaneous Commands"0)))

(DEFMETHOD (graph-display-pane :change-status) (&OPTIONAL additional-cvv-items)
  "1This command is set up so that an application can call it, adding its own 
choose-variable-values items to be edited by the user.0"
  (LET ((acc acceleration)
	(width (SEND real-window :inside-width))
	(height (SEND real-window :inside-height))
	(horiz horizontal-pan-%)
	(vert vertical-pan-%))
    (DECLARE (SPECIAL acc width height horiz vert))
    (TV:CHOOSE-VARIABLE-VALUES
      `((acc ,(si:string "Panning Acceleration")
	 :DOCUMENTATION ,(si:string
         "A number which determines the ratio of pixels panned to pixels the mouse is moved.")
	 :NUMBER)
	(horiz ,(si:string "Percentage of screen to pan horizontally")
	 :DOCUMENTATION ,(si:string
         "Used when the fast mouse panning gets to an edge of the displayed graph")
	 :NUMBER)
	(vert ,(si:string "Percentage of screen to pan vertically")
	 :DOCUMENTATION ,(si:string
	 "Used when the fast mouse panning gets to an edge of the displayed graph")
	 :NUMBER)
	(*dashed-line-margin* ,(si:string "Dashed Line Margin")
	 :DOCUMENTATION ,(si:string
         "# pixels between dashed line and actual boundary of visible portion of graph.  NIL disables the drawing of dashed line."))
	(width ,(si:string "Output plane width")
	 :DOCUMENTATION ,(si:string "Width in pixels of the area we window into.")
	 :NUMBER)
	(height ,(si:string "Output plane height")
	 :DOCUMENTATION ,(si:string "Height in pixels of the area we window into.")
	 :NUMBER)
	(*smallest-edge-length-to-bother-labelling*
	  ,(si:string "Length of shortest edge to bother labelling")
	 :DOCUMENTATION ,(si:string "Edges shorter than this value won't be labelled.")
	 :NUMBER)
	(*percentage-radius-for-leaves-in-circular-arrangement*
	  ,(si:string "Percentage (by radius) of circle for graph leaves")
	 :DOCUMENTATION ,(si:string
	 "Percentage of the circle to allocate to edges leading to/from leaves of the graph.")
	 :NUMBER)
	. ,additional-cvv-items)
      :LABEL
      (SI:STRING "Move mouse over data fields.  Field descriptions will appear in who-line."))
    (SEND SELF :set-acceleration acc)
    (SEND SELF :set-horizontal-pan-% horiz)
    (SEND SELF :set-vertical-pan-% vert)
    (UNLESS (AND (EQ width (SEND real-window :inside-width))
		 (EQ height (SEND real-window :inside-height)))
      (SEND real-window :set-inside-size width height)
      1;;Rescale and redraw currently displayed graph so user sees more of it (assuming he
      ;;grew the window)
0      (WHEN graph
	(SEND graph :scale-for-initial-viewing real-window SELF)
	(SEND SELF :set-graph graph)))))

(DEFMETHOD (graph-display-pane :display-self-loops) ()
  1;;This is a quick kludge.  Self loops could be displayed graphically.
0  (COND
    (graph
     (SEND SELF :CLEAR-WINDOW)
     (FORMAT SELF "Vertices connected to themselves:")
     (IF (SEND graph :GET :self-loops)
	 (DOLIST (self-loop (SEND graph :GET :self-loops))
	   (FORMAT SELF "~%  ~s via edges: ~{~a ~}"
		   (vertex-data (edge-vertex-1 self-loop)) (edge-data self-loop)))
	 (FORMAT SELF "~%-none-"))
     (FORMAT SELF "~2%Press any key to continue")
     (SEND SELF :TYI)
     1;;Redisplay graph
0     (SEND SELF :update-image))
    (T (BEEP))))


1;;
;;;; Metagraphs -- Code for recursively displaying subgraphs of a GRAPH flavor instance.
;;
;; MetaGraphs are used to display graphs (or previously created metagraphs), with infinite
;; levels of recursion supported.  The command loop is also called recursively.
;;
0(DEFVAR *graph-type-for-displaying-subsets-of-graphs* NIL)
(DEFVAR *vertices-to-display* NIL)
(DEFVAR *vertices-to-highlight* NIL)
(DEFVAR *graph-stack* NIL
 "1Last element is always the user's GRAPH instance.  Anything else is a metagraph.0")


(DEFUN get-metagraph-type ()
  "1Returns the metagraph type, creating it first if it hasn't been created.0"
  (UNLESS *graph-type-for-displaying-subsets-of-graphs*
    (SETQ *graph-type-for-displaying-subsets-of-graphs*
	  (MAKE-INSTANCE 'graph-type
			 :name 'meta-graph-type
			 :traversal-function 'metagraph-traversal-function
			 :vertex-print-string-function 'metagraph-vertex-print-string-function
			 :edge-print-string-function 'metagraph-edge-print-string-function
			 :vertex/edge-selection-handler 'metagraph-vertex/edge-selection-handler
			 :vertex/edge-description-function 'metagraph-vertex/edge-description-function))
    1;;Remove METAGRAPH type from list, since user can't make use of it in Zgraph interface.
0    (POP *graph-types*))
  *graph-type-for-displaying-subsets-of-graphs*)


(DEFUN metagraph-traversal-function (vertex-struct)
  1;;Here the argument is a VERTEX-STRUCT--the data item for meta-graph-type.
0  1;;Unless it is in the global list, we completely ignore it.
0  (WHEN (MEMQ vertex-struct *vertices-to-display*)
    (LOOP FOR edge IN (vertex-edges vertex-struct)
	  FOR neighbor = (edge-vertex-2 edge)
	  1;;Blow off the edge if it isn't between two vertices of the display subset.
0	  WHEN (MEMQ neighbor *vertices-to-display*) 
	  COLLECT (CONS neighbor edge))))

(DEFUN get-real-vertex (vertex-struct)
  "1If VERTEX-STRUCT is a vertex of a metagraph, the real vertex struct which it represents
is returned.  If not, VERTEX-STRUCT *is* a real vertex struct and is returned.  The
former case involves recurisively looking through VERTEX data fields until a real
vertex is found (one whose data field doesn't hold a vertex.0"
  (DO ((data (vertex-data vertex-struct) (vertex-data vertex-struct)))
      ((NOT (TYPEP data 'vertex))
       vertex-struct)
    (SETQ vertex-struct data)))

(DEFUN metagraph-vertex-print-string-function (vertex-struct)
  1;;Have the original graph provide a print string for the vertex stored in VERTEX-STRUCT's
0  1;;data field.  (We might be nested hopelessly in recursion, with metagraphs displaying 
0  1;;metagraphs...)
0  (LET ((real-vertex (get-real-vertex vertex-struct)))
    (MULTIPLE-VALUE-BIND (string font)
	(FUNCALL (SEND (SEND (CAR (LAST *graph-stack*)) :type)
		       :vertex-print-string-function)
		 real-vertex)
      1;;Kludgy but simple way to highlight the root vertices of the subset. (In the case
0      1;;of the Display Vertex option, its just the vertex on which the user clicked
0      1;;MOUSE-M.)
0      (VALUES
	(IF (MEMQ real-vertex *vertices-to-highlight*)
	    (FORMAT NIL "--> ~a <--" string)
	    string)
	font))))

(DEFUN get-real-edge (edge-struct)
  "1Same idea as get-real-vertex.0"
  1;;All graph edges have lists as EDGE-DATA, so that multiple edge from V1 to V2
0  1;;can be displayed as one edge.  This feature is also true of metagraph edges.
0  1;;Metagraph edges hold edge structs of the displayed graph in their EDGE-DATA
0  1;;field.  But since the displayed graph's multiple edges have already been merged
0  1;;together, we are guaranteed that (CDR (edge-data edge-struct)) is NIL.
0  (DO ((data (edge-data edge-struct) (edge-data edge-struct)))
      ((NOT (TYPEP (CAR data) 'edge))
       edge-struct)
    (SETQ edge-struct (CAR data))))

(DEFUN metagraph-edge-print-string-function (edge-struct)
  1;;Works just like METAGRAPH-VERTEX-PRINT-STRING-FUNCTION.
0  (FUNCALL (SEND (SEND (CAR (LAST *graph-stack*)) :type) :edge-print-string-function)
	   (get-real-edge edge-struct)))

(DEFUN metagraph-vertex/edge-selection-handler (thing window)
  1;;Similar to METAGRAPH-VERTEX-PRINT-STRING-FUNCTION.
0  (FUNCALL (SEND (SEND (CAR (LAST *graph-stack*)) :type)
		 :vertex/edge-selection-handler)
	   (IF (TYPEP thing 'vertex)
	       (get-real-vertex thing)
	       (get-real-edge thing))
	   window))

(DEFUN metagraph-vertex/edge-description-function (thing scroll-window)
  1;;Similar to METAGRAPH-VERTEX-PRINT-STRING-FUNCTION.
0  (FUNCALL (SEND (SEND (CAR (LAST *graph-stack*)) :type)
		 :vertex/edge-description-function)
	   (IF (TYPEP thing 'vertex)
	       (get-real-vertex thing)
	       (get-real-edge thing))
	   scroll-window))


(DEFMETHOD (graph-display-pane :recursively-display-vertices)
	   (vertex-structs &OPTIONAL *vertices-to-highlight* (update-afterwords? T))
  "1Displays VERTEX-STRUCTS and edges between them.
Allows the user to interact with the displayed graph as normal through the use of a
recursive call to the command loop.  
0  1The first optional argument specifies a subset of
VERTEX-STRUCTS, the first argument, containing vertices which are to be highlighted
by printing pointers to their labels.
  The second optional argument, if NIL, inhibits redisplay of the original graph once
we've finished the recursive call to the command loop.  This can be used to display a
series of subgraphs in succession, for instance.0"
  (DECLARE (SPECIAL *zgraph-frame-keystroke-commands*))
  (LET ((metagraph (MAKE-INSTANCE 'graph
				  :type (get-metagraph-type)
				  :root-vertices vertex-structs))
	1;;Bind this global so that the metagraph traversal function finds only the
0	1;;vertices and edges comprising the subset of the graph we want to display.
0	(*vertices-to-display* vertex-structs)
	1;;Bind this global so that the metagraph vertex/edge print string functions
0	1;;can make use of the previous graph's print string functions.
0	(*graph-stack* (CONS (SEND SELF :graph) *graph-stack*)))
    1;;Traverse the graph given the root vertices
0    (SEND metagraph :construct)
    (SEND metagraph :plot-vertices)
    1;;Store GRAPH in the display pane, which automatically draws it.
0    (debug-print T "~%Drawing graph on hidden bit-array...")
    1;;Call the command loop recursively, providing a means to jump out of the recursive
0    1;;level.
0    (UNWIND-PROTECT
	(CATCH 'exit-recursive-command-loop
	  (SEND SELF :set-keystroke-commands 
		(CONS '(148. (THROW 'exit-recursive-command-loop NIL))
		      (SEND SELF :keystroke-commands)))
	  (SEND SELF :set-graph metagraph)
	  1;;Print done and indicate how to exit.
0	  (debug-print T "done. {Recursive call to command loop--press the END key to pop}")
	  1;;When we're part of a Zgraph user interface (or any application interface who
0	  1;;has bound *GD* to itself which can handle :COMMAND-LOOP), fire up its command
0	  1;;loop.  Otherwise fire ours up.  (Ours handles mouse blips and keystrokes, but
0	  1;;not menu blips or Lisp read/eval/print.)
0	  (IF (AND (EQ *gd* TV:SUPERIOR) (SEND *gd* :GET-HANDLER-FOR :command-loop))
	      (SEND *gd* :command-loop)
	      (SEND SELF :command-loop)))
      (SEND SELF :set-keystroke-commands
	    (CDR (SEND SELF :keystroke-commands)))
      1;;Remove metagraph from history.
0      (SETQ graph-history (REMOVE metagraph graph-history))
      1;;Restore previous graph.
0      (SEND SELF :set-graph (CAR *graph-stack*) update-afterwords?))))


(DEFMETHOD (graph-display-pane :command-loop) ()
  "1Command loop used for recursive display of portions of a graph.0"
  (CATCH 'exit-command-loop
    (DO ((input (SEND SELF :ANY-TYI)
		(SEND SELF :ANY-TYI)))
	(NIL)
      (IF (EQ input #+Symbolics 148. #+Explorer #\END)
	  (RETURN T)
	  (SEND SELF :handle-input input)))))


1;;
;; Zgraph user interface: one constraint frame with four window panes
;;

0(DEFFLAVOR graph-display-menu-pane ()
	   (TV:DONT-SELECT-WITH-MOUSE-MIXIN
	    TV:COMMAND-MENU-PANE)
  (:DEFAULT-INIT-PLIST
   :LABEL NIL
   :ITEM-LIST *zgraph-frame-menu-commands*))

(DEFFLAVOR graph-display-interaction-pane ()
	   (#+Explorer
	    TV:DONT-SELECT-WITH-MOUSE-MIXIN	;any problem using this on Symbolics too?
	    TV:PANE-MIXIN
	    TV:NOTIFICATION-MIXIN
	    TV:WINDOW)
  (:DEFAULT-INIT-PLIST
   :MORE-P NIL
   :LABEL NIL
   :DEEXPOSED-TYPEOUT-ACTION :PERMIT))


(DEFVAR *graph-display-window-width* 2500.)
(DEFVAR *graph-display-window-height* 2500.)

(DEFFLAVOR zgraph-graph-display-pane ()
	   (TV:PANE-MIXIN
	    graph-display-pane)
  (:DEFAULT-INIT-PLIST
   :extended-width (- *graph-display-window-width* TV:MAIN-SCREEN-WIDTH)
   :extended-height (- *graph-display-window-height* TV:MAIN-SCREEN-HEIGHT)
   :real-window-flavor 'TV:WINDOW
   1;;Give the display pane (which by default interprets its own set of
0   1;;commands and allows them to be selected with MOUSE-R) our set of
0   1;;commands (which include its own and some of our own).
0   :menu-commands *zgraph-frame-menu-commands*
   :keystroke-commands *zgraph-frame-keystroke-commands*)
  (:DOCUMENTATION :SPECIAL-PURPOSE "
1This flavor is not useful for applications, since it redefines some of the normal
ZG:GRAPH-DISPLAY-PANE commands appropriate for the Zgraph user interface.  Use
ZG:GRAPH-DISPLAY-PANE instead.0"))

1;;
;;Tailor the following ZG:GRAPH-DISPLAY-PANE methods so that they work better for Zgraph.
;;


0(DEFWHOPPER (zgraph-graph-display-pane :change-status) ()
  "1Adds Zgraph options to the normal graph-display status command.0"
  (LET* ((configuration (SEND TV:SUPERIOR :configuration))
	 (constraints (LOOP FOR entry IN (SEND TV:SUPERIOR :CONSTRAINTS)
			    COLLECT (CAR entry)))
	 (sample-graph-type (FIND-IF #'(LAMBDA (graph-type)
				       (EQ (SEND graph-type :name) 'sample-graph-type))
				     *graph-types*))
	 (sample-graph-roots (SEND sample-graph-type :default-root-finding-form)))
    (DECLARE (SPECIAL configuration constraints sample-graph-roots *sample-root-list*))
    1;;Pass extra CVV items to the display pane's :CHANGE-STATUS command.
0    (CONTINUE-WHOPPER 
      `((configuration ,(si:string "Frame Configuration")
		       :DOCUMENTATION
		       ,(si:string "Configuration of window panes (MAIN by default).  Also see variable ZG:*DEFAULT-PANE-CONFIGURATION*")
		       :CHOOSE ,constraints)
	(sample-graph-roots ,(si:string "Roots for the SAMPLE-GRAPH-TYPE")
			    :DOCUMENTATION ,(si:string "By changing the selection, you'll get different graphs when creating SAMPLE-GRAPHs.")
			    :CHOOSE ,*sample-root-list*)
	(*graph-debug-actions* ,(si:string "Debugging Options")
			       :DOCUMENTATION ,(si:string "Each option shows particular facets of the graph computation")
			       #+Symbolics :CHOOSE-MULTIPLE #+Symbolics ,*all-graph-debug-actions*
			       #+Explorer :MULTIPLE-MENU #+Explorer *all-graph-debug-actions*)
	(*graph-plotting-style* ,(si:string "Graph Plotting Style")
				:DOCUMENTATION ,(si:string "Select the method to use in plotting graphs on the cartegion plane.")
				:ASSOC ,*all-graph-plotting-styles*)
	(*too-many-non-leaves* ,(si:string "Max vertices for crossover minimization")
			       :DOCUMENTATION ,(si:string "If the graph has more non-leaf vertices than this, don't try to minimize crossovers.")
			       :NUMBER)
	(*too-many-edges* ,(si:string "Max edges for crossover minimization")
			  :DOCUMENTATION ,(si:string "If the graph has more edges than this, don't try to minimize crossovers.")
			  :NUMBER)
	(*number-vertices-over-which-graphs-are-fit-onto-hidden-bit-array*
	  ,(si:string "Max vertices to fit in window pane")
	  :DOCUMENTATION ,(si:string "If the graph has more vertices than this, fit the graph onto the hidden bit array--else fit onto the display pane.")
	  :NUMBER)))
    (UNLESS (EQ configuration (SEND TV:SUPERIOR :CONFIGURATION))
1      ;;Don't worry about recovering the label.  This will be a convenient way to eliminate it
      ;;accross all configurations.
0      (WHEN (EQ configuration 'trimmed)
	(SEND TV:SUPERIOR :SET-LABEL NIL))
      (SEND TV:SUPERIOR :set-configuration configuration)
      (SEND SELF :update-image)
      (SEND *TERMINAL-IO* :CLEAR-WINDOW)
      (FORMAT *TERMINAL-IO* "> ")
      (WHEN graph
	(SEND graph :description (SEND TV:SUPERIOR :status-pane))))
    (SEND sample-graph-type :set-default-root-finding-form sample-graph-roots)))


(DEFMETHOD (zgraph-graph-display-pane :describe-graph) (&OPTIONAL (graph-instance graph))
  (LET ((status-pane (SEND TV:SUPERIOR :status-pane)))
    (SEND status-pane :CLEAR-WINDOW)
    (SEND graph-instance :description status-pane)))

(DEFMETHOD (zgraph-graph-display-pane :BEFORE :SET-GRAPH) (new-graph &OPTIONAL IGNORE)
  "1Has graph describe0 1itself just before being drawn. 
It is important to do this so that the user gets an indication of how large the graph
is, since large graphs can take a long time to be drawn.0"
  (WHEN new-graph
    (SEND SELF :describe-graph new-graph)))


(DEFMETHOD (zgraph-graph-display-pane :AFTER :pan) (&REST IGNORE)
  1;;Regenerate a description to show possible change in number of clipped vertices.
0  (SEND SELF :describe-graph graph))
(DEFMETHOD (zgraph-graph-display-pane :AFTER :zoom) ()
  (SEND SELF :describe-graph graph))
(DEFMETHOD (zgraph-graph-display-pane :AFTER :keyboard-zoom) ()
  (SEND SELF :describe-graph graph))
(DEFMETHOD (zgraph-graph-display-pane :AFTER :reset-scale) ()
  (SEND SELF :describe-graph graph))


(DEFFLAVOR description-scroll-pane ()
	   1;;Blow off TV:SCROLL-WINDOW which scrolls too slowly.
0#+Symbolics
	   (TV:DONT-SELECT-WITH-MOUSE-MIXIN
	    TV:PANE-MIXIN
	    TV:MARGIN-SCROLLING-WITH-FLASHY-SCROLLING-MIXIN
	    TV:MARGIN-SCROLL-MIXIN
	    TV:MARGIN-REGION-MIXIN
	    TV:MOUSE-SENSITIVE-TEXT-SCROLL-WINDOW-WITHOUT-CLICK
	    TV:MARGIN-SPACE-MIXIN
	    TV:BASIC-MOUSE-SENSITIVE-ITEMS)
	   1;;Blow off TV:SCROLL-WINDOW which scrolls too slowly.
0#+Explorer
	   (TV:DONT-SELECT-WITH-MOUSE-MIXIN
	    TV:PANE-MIXIN
	    TV:MOUSE-SENSITIVE-TEXT-SCROLL-WINDOW
	    TV:MARGIN-SCROLLING-WITH-FLASHY-SCROLLING-MIXIN
	    TV:FLASHY-SCROLLING-MIXIN
	    TV:BORDERS-MIXIN
	    TV:MARGIN-SCROLL-MIXIN
	    TV:MARGIN-REGION-MIXIN
	    TV:BASIC-SCROLL-BAR
            TV:WINDOW)
  (:DEFAULT-INIT-PLIST
    :SENSITIVE-ITEM-TYPES T
    :SCROLL-BAR-ALWAYS-DISPLAYED T
#+Symbolics   :MARGIN-SPACE
#+Symbolics   '(2 0 0 0)
    :LABEL NIL
   :MARGIN-SCROLL-REGIONS `((:TOP ,(si:string "Top"))
			    (:BOTTOM ,(si:string "Bottom")))
   :FLASHY-SCROLLING-REGION '((20 .4s0 .6s0)
			      (20 .4s0 .6s0))))

(DEFMETHOD (description-scroll-pane :ADJUSTABLE-SIZE-P) ()		       
  NIL)

(DEFMETHOD (description-scroll-pane :PRINT-ITEM) (ITEM LINE-NO ITEM-NO)
  1"A customization of TV:TEXT-SCROLL-WINDOW's method."
0  LINE-NO ITEM-NO				1;Ignore these
0  (APPLY #'FORMAT SELF (CAR item) (CDR item)))

1;To clear, (SEND window :SET-ITEMS NIL)
;To add, (SEND window :append-new-item (cons format-string format-args))
;To set, (SEND window :SET-ITEMS (LIST (cons format-string format-args)...))
;To refresh, (SEND window :REFRESH)

;0#|1 ATSIGN was barfing, so I put these quotes in. -- KDF
;0(send tv:mouse-window :set-items
1;0      (LOOP FOR f IN (nthcdr 1500. *all-flavor-names*)
1;0	    COLLECT (LIST "~s" f)))
1;0|#



(DEFFLAVOR graph-display-frame
	(1;;Window panes
0	 menu-pane
	 display-pane
	 interaction-pane
	 status-pane
	 description-pane
	1 ;;Bind this locally so each frame has it's own state of debug actions.
0	 (*graph-debug-actions* *graph-debug-actions*)
	 input)
	(TV:BORDERED-CONSTRAINT-FRAME-WITH-SHARED-IO-BUFFER
	 TV:PROCESS-MIXIN
	 TV:STREAM-MIXIN
	 TV:WINDOW)
  :SETTABLE-INSTANCE-VARIABLES
  :SPECIAL-INSTANCE-VARIABLES 1;;so they can be referenced in the Lisp interaction pane
0  (:DEFAULT-INIT-PLIST
   :PANES
   `((menu-pane graph-display-menu-pane :FONT-MAP ,(LIST fonts:cptfont))
     (display-pane zgraph-graph-display-pane)
     (description-pane description-scroll-pane)
     (interaction-pane graph-display-interaction-pane)
1     ;;We can reuse this flavor--nothing functionally special about it.
0     (status-pane graph-display-interaction-pane
		  :BLINKER-P NIL
		  :FONT-MAP ,(LIST FONTS:TR8B)))
   1;;Default is MAIN; TRIMMED is available for increasing display space.
0   :CONSTRAINTS '((main
		    . ((menu-pane display-pane interaction-and-status)
		       ((menu-pane :ASK :PANE-SIZE))
		       ((display-pane .9 :LINES))
		       1;;Oh no, another one of these...
0		       ((interaction-and-status
			  :HORIZONTAL (:EVEN)
			  (left-half right-half)
			  ((left-half :vertical (.70)
				      (interaction-pane)
				      ((interaction-pane
					 :EVAL TV:**CONSTRAINT-REMAINING-HEIGHT**)))
			   (right-half :vertical (.30)
				       (status-pane)
				       ((status-pane :EVEN))))))))
		  (with-description-pane
		    . ((menu-pane description-and-display-panes interaction-and-status)
		       ((menu-pane :ASK :PANE-SIZE))
		       ((description-and-display-panes
			  :HORIZONTAL (:LIMIT (10. NIL :LINES description-pane)
					      .87 :LINES description-pane)
			  (description display)
			  ((description :vertical (.50)
					(description-pane)
					((description-pane
					   :EVAL TV:**CONSTRAINT-REMAINING-HEIGHT**)))
			   (display :vertical (.50)
				    (display-pane)
				    ((display-pane
				       :EVAL TV:**CONSTRAINT-REMAINING-HEIGHT**))))))
		       ((interaction-and-status
			  :HORIZONTAL (:EVEN)
			  (left-half right-half)
			  ((left-half :vertical (.70)
				      (interaction-pane)
				      ((interaction-pane
					 :EVAL TV:**CONSTRAINT-REMAINING-HEIGHT**)))
			   (right-half :vertical (.30)
				       (status-pane)
				       ((status-pane :EVEN))))))))
		  (description-pane-vertical
		    . ((menu-pane description-pane display-pane interaction-and-status)
		       ((menu-pane :ASK :PANE-SIZE))
		       ((description-pane .40 :LINES))
		       ((display-pane .85 :LINES))
		       ((interaction-and-status
			  :HORIZONTAL (:EVEN)
			  (left-half right-half)
			  ((left-half :vertical (.70)
				      (interaction-pane)
				      ((interaction-pane
					 :EVAL TV:**CONSTRAINT-REMAINING-HEIGHT**)))
			   (right-half :vertical (.30)
				       (status-pane)
				       ((status-pane :EVEN))))))))
		  (trimmed . ((display-pane interaction-and-status)
			      ((display-pane .95 :LINES))
			      ((interaction-and-status
				 :HORIZONTAL (:EVEN)
				 (left-half right-half)
				 ((left-half :vertical (.70)
					     (interaction-pane)
					     ((interaction-pane
						:EVAL TV:**CONSTRAINT-REMAINING-HEIGHT**)))
				  (right-half :vertical (.30)
					      (status-pane)
					      ((status-pane :EVEN)))))))))
   :PROCESS '(graph-display-command-loop :special-pdl-size 6000. :regular-pdl-size 20000.))
  (:DOCUMENTATION :COMBINATION "1General Graph/Data Structure Displayer0"))

(DEFMETHOD (graph-display-frame :AFTER :INIT) (IGNORE)
  (SEND SELF :SEND-ALL-PANES :set-save-bits T)
  (SEND SELF :SEND-ALL-PANES :set-deexposed-typeout-action :PERMIT)
1  ;;Store the panes in instance variables for quick access.
0  (SETQ menu-pane (SEND SELF :GET-PANE 'menu-pane)
	display-pane (SEND SELF :GET-PANE 'display-pane)
	interaction-pane (SEND SELF :GET-PANE 'interaction-pane)
	status-pane (SEND SELF :GET-PANE 'status-pane)
	description-pane (SEND SELF :GET-PANE 'description-pane))
  (SEND SELF :SET-CONFIGURATION *default-pane-configuration*))

(DEFMETHOD (graph-display-frame :NAME-FOR-SELECTION) () (si:string "Graph Displayer"))


(DEFUN graph-display-command-loop (graph-display-frame)
  (SEND graph-display-frame :command-loop))

(DEFMETHOD (graph-display-frame :command-loop) ()
  (LET ((*TERMINAL-IO* interaction-pane)
	(*DEBUG-IO* display-pane)1 ;;since it's big and can be refreshed.
0	1;;This is the window the graph code expects to output to.
0	1;;We bind it to our display panes's "real" window.  (The display pane is a meta
0	1;;window.)
0	(*graph-output* (SEND display-pane :real-window))
	1;;This is the meta-window the graph code expects us to be displaying whats on
0	1;;*GRAPH-OUTPUT*
0	(*display-io* display-pane)
	1;;Bind this so that descriptions will go to the scroll window.  We don't worry
0	1;;here whether or not the scroll window is present in the current configuration.
0	(*description-output* description-pane)
	1;;Bind this for use in the Lisp interaction pane.
0	(*gd* SELF)
#+Symbolics
	1;;Make this a CL top level.
0	(SI:*INPUT-EDITOR-PACKAGE* (PACKAGE 'cl))
#+Symbolics
	(SI:READTABLE SI:*COMMON-LISP-READTABLE*)
	inhibit-prompt?
	1;;Similar to \ in the Inspector.
0	v)
    (DECLARE (SPECIAL v))
    (CATCH 'exit-command-loop
      (LOOP DO
	1    ;;Prompt the user if we're supposed to.  Otherwise set prompt to be printed
0	1    ;;next iteration.
0	    (IF inhibit-prompt?
		(SETQ inhibit-prompt? NIL)
		(SEND *TERMINAL-IO* :FRESH-LINE)
		(FORMAT T "> "))
	    1;;Read mouse or keystroke
0	    (SETQ input (SEND SELF :ANY-TYI))
	    (COND
	      ((LISTP input)
	       (SETQ inhibit-prompt? T)
	       (SEND display-pane :handle-input input))
	      1;;Else key input.  Let the display-pane try to interpret the keystroke--if
0	      1;;knows about Zgraph keystroke commands as well as its own.  If unrecognized,
0	      1;;put the keystroke back into the buffer and read in a Lisp expression to EVAL.
0	      1;;If user rubs out expression completely, abort the read.
0	      ((EQ (SEND display-pane :handle-input input 'unrecognized)
		   'unrecognized)
	       1;;Read and EVAL a Lisp expression.
0	       (SEND SELF :UNTYI input)
	       (MULTIPLE-VALUE-BIND (read-sexp read-flag)
#+Symbolics
		   (TV:WITH-INPUT-EDITING-OPTIONS ((:FULL-RUBOUT :FULL-RUBOUT))
		     (TV:WITH-INPUT-EDITING (*TERMINAL-IO*)
		       1;;Common Lisp reader.
0		       (SI:CP-COMMAND-LOOP-READ-FUNCTION)))
#+Explorer
		   (SEND terminal-io :preemptable-read '((:FULL-RUBOUT :FULL-RUBOUT))
			 #'READ-FOR-TOP-LEVEL)
		 (IF (EQ read-flag :FULL-RUBOUT)
		     (SETQ inhibit-prompt? T)
		     1;;Else
0		     (SETQ / (MULTIPLE-VALUE-LIST (EVAL read-sexp)))
		     (DOLIST (value / )
		       (SEND *TERMINAL-IO* :FRESH-LINE)
		       (PRIN1 value *TERMINAL-IO*))
		     (SEND *TERMINAL-IO* :FRESH-LINE)
		     (PSETQ *** ** ** * * (CAR / ) +++ ++ ++ + + read-sexp))))
	      1;;Display pane handled the keystroke, so suppress prompt.
0	      (T
	       (SETQ inhibit-prompt? T)))))))


1;;
;; Additional Zgraph command definitions
;;

0(DEFUN define-graph-type ()
  1;;If user aborts, pop the created graph type off of the global list.
0  (LET ((aborted? T))
    (UNWIND-PROTECT
	(PROGN
	  (SEND (MAKE-INSTANCE 'graph-type) :edit)
	  (SETQ aborted? NIL))
      (POP *graph-types*))))

#+Symbolics
(DEFUN edit-graph-type ()
  (MULTIPLE-VALUE-BIND (IGNORE button)
      (TV:MENU-CHOOSE
	(LOOP FOR graph-type IN *graph-types*
	      COLLECT `(,(SEND graph-type :name)
			1;;Returns graph type as first value, and the 1st or
0			1;;second cons (depending on mouse button) as 2nd value
0			:BUTTONS ((,graph-type . 1) (,graph-type . 2))
			:DOCUMENTATION
			,(SI:STRING "L: Edit Graph type,   M: Delete Graph Type")))
	(si:string "Select a Graph Type to Edit"))
    (WHEN button
      (CASE (CDR button)
	(1
	  (SEND (CAR button) :edit))
	(2
	  (SETQ *graph-types* (DELETE (CAR button) *graph-types*)))))))
#+Explorer
(DEFUN edit-graph-type ()
  (LET ((button (TV:MENU-CHOOSE
		  (LOOP FOR graph-type IN *graph-types*
			COLLECT `(,(SEND graph-type :name)
				  1;;Returns graph type as first value, and the 1st or
0				  1;;second cons (depending on mouse button) as 2nd value
0				  :BUTTONS ((NIL :VALUE (,graph-type . 1)) (NIL :VALUE (,graph-type . 2)))
				  :DOCUMENTATION
				  ,(SI:STRING "L: Edit Graph type,   M: Delete Graph Type")))
		  (si:string "Select a Graph Type to Edit"))))
    (WHEN button
      (CASE (CDR button)
	(1
	 (SEND (CAR button) :edit))
	(2
	 (SETQ *graph-types* (DELETE (CAR button) *graph-types*)))))))

(DEFMETHOD (graph-display-frame :create-graph) ()
  (LET ((graph-type (TV:MENU-CHOOSE (MAPCAR #'(LAMBDA (graph-type)
					      (CONS (SEND graph-type :name) graph-type))
					    *graph-types*)
				    (si:string "Select a Graph Type to Instantiate")))
	graph)
    (WHEN graph-type
      (SETQ graph (FUNCALL (SEND graph-type :instantiation-function) graph-type))
      1;;Traverse the graph given the root vertices
0      (SEND graph :construct)
1      ;;Plot the graph
0      (SEND graph :plot-vertices)
      1;;Store GRAPH in the display pane, which automatically draws it.
0      (debug-print T "~%Drawing graph on hidden bit-array...")
      (SEND display-pane :set-graph graph)
      1;;Print done and reprompt.
0      (debug-print T "done.~%> "))))

(DEFUN read-file-name (label near-window)
  1;;If the ZWEI thing ever breaks, use this:
;0  (LET ((pathname (SI:STRING "machine:>user>foo.lisp")))
1;0    (DECLARE (SPECIAL pathname))
1;0    (TV:CHOOSE-VARIABLE-VALUES
1;0      `((pathname ,(si:string "File to save Graph Types into") :pathname)))
1;0     pathname)
  (LET ((ZWEI:*INTERVAL* 'IGNORE)
	#+Explorer
	(ZWEI:*MAJOR-MODE* 'zwei:common-lisp-mode))
    (SEND (ZWEI:READ-DEFAULTED-PATHNAME-NEAR-WINDOW near-window label)
	  :STRING-FOR-PRINTING)))

  
(DEFMETHOD (graph-display-frame :save-scroll-window) ()
  1;;This binding is a kludge to satisfy the ZWEI function.  IGNORE
0  1;;is a function which does nothing.
0  (WITH-OPEN-FILE (stream (STRING (read-file-name
				    "Save scroll window text to what2 0file" SELF))
			  :DIRECTION :OUTPUT)
    (LET ((items (SEND *description-output* :items)))
      (DOTIMES (line-number (LENGTH items))
	(APPLY #'FORMAT stream (AREF items line-number))
	(TERPRI stream)))))

(DEFUN save-graph-types ()
  (LET ((pathname (read-file-name "File to save Graph Types into" *gd*)))
    (WITH-OPEN-FILE (file (STRING pathname) :DIRECTION :OUTPUT)
      (FORMAT file ";;; -*- Mode: LISP; Syntax: Common-lisp; -*-")
      (FORMAT file "~2%(SETQ zg:*graph-types* `(")
      1;;Bind package to NIL so package prefixes are explicitly printed.
0      (LET ((*package* NIL))
	(DOLIST (graph-type *graph-types*)
	  (FORMAT file "~%," (SEND graph-type :name))
	  (SEND graph-type :definition-form file)))
      (FORMAT file "))"))
    (FORMAT T "~%Default root-finding forms and the names of graph type traversal functions ~
               ~%and print string functions have been saved.  You should save the function ~
               ~%definitions yourself, perhaps in your LISPM-INIT.LISP file.")))

(DEFUN load-graph-types ()
  (LET ((pathname (read-file-name "File to retrieve Graph Types from" *gd*)))
    (LOAD (string pathname) :PACKAGE *package*)))


(DEFMETHOD (graph-display-frame :help) ()
  "1Prints help message.0"
  (SEND display-pane :CLEAR-WINDOW)
  (FORMAT display-pane "
This window frame is Zgraph, a general directed graph displayer constructed as a tool for
examining and debugging data structures.  Zgraph was written by John Hogge at The University
of Illinois.2 

0Zgraph frames can be selected by pressing SELECT G.  After the first frame is created,
SELECT CONTROL-G can be used to create other frames, which can be operated independently.

The commands in the top menu may be invoked using keystrokes, as well as by selecting
them with the mouse.  To see a keystroke and short description of a command, place
the mouse over it and read the who-line.  The following are explanations of each command,
followed by comments on using the mouse to pan a displayed graph.

COMMAND
=======
Graph Types
 -- Brings up a submenu of the following four commands:

2  0Define Graph Type
2  0 -- Defines a new type of graph to display.  Typically the user defines a graph type for 
2  0    use in examining each type of data structure he is working with.  Several graph types
2  0    could also be defined for displaying the same data structure in different ways.

2  0    This command prompts you for an identifying name for the graph type, a graph traversal
2  0    function for finding the vertices and edges in a graph (given a set of root vertices),
2  0    and optionally a default root-finding form.  The default root finding form, if supplied, 
2  0    is a form which is evaluated whenever a graph of the given type is created (using the 
2  0    Create Graph command).  This form should return a list of root vertices to use
2  0    to get at the rest of the vertices and edges in the graph.  Specifically, the graph
2  0    traversal function is applied to each of the root vertices to get a set of outgoing
2  0    edges to other vertices.  It is recursively applied to each of these new vertices to get
2  0    their outgoing edges and a further set of vertices, etc., in a depth-first manner.

2  0    If you don't supply a default root-finding form for your graph type, then you will have
2  0    to supply a form whenever you create a graph of the type (via Create Graph).  Even if
2  0    you do supply one, Create Graph allows you to override it.  The idea behind having a 
2  0    root-finding form is that you might only want to see a portion of your data structure.
2  0    Alternatively, you might have a convenient handle on only a few vertices (data objects)
2  0    which, with a properly written graph traversal function, will lead to the other vertices 
2  0    (data objects) in the graph.  By all means, if you have stored a list of all the
2      0vertices in the graph, you might as well use it as the default root-finding form.  In
2  0    that case, the traversal function will simply provide the edges between these vertices.

2  0    This explanation doesn't provide enough information to define a graph.  For an example
2      0of2 0how to write a graph traversal function and a default root-finding form, press
2      0META-.  For a further example of how to integrate Zgraph into an application, press
2      0SUPER-.2 

  0Edit Graph Type
2  0 -- Edits the attributes of a previously defined graph type.  For an example of the process,
2  0    execute this command and edit the sample graph type provided.

2  0Save Graph Types
2  0 --Saves currently defined graph types into a file.  Note that *graphs* are *not* saved.
2  0   Only the graph types (their identifying names, names of traversal functions, default 
2  0   forms, and names of print-string functions) are saved.  This is useful for offloading
2  0   a graph type which you want to use between Lisp sessions.

2  0Load Graph Types
2  0 --Loads previously saved graph types from a file.


Create Graph
2  0 --Creates and displays a graph of one of the currently defined graph types.  For an example
2  0   graph, execute this command and select the SAMPLE-GRAPH-TYPE provided.

Display Graph
 --Displays a previously created graph.  If you create several graphs, you can look at the
   previously displayed graphs through this command.  This command can also be used to delete
   graphs, which may be desireable in order to free up memory eaten by large graphs.

Status
 --Alters the operation of the Graph Displayer through a window interface.  For short
   descriptions of each option, place the mouse over the displayed values and read the
   who-line.  Important options:

   Graph Plotting Style - provides a choice of algorithms for positioning graph
   vertices.  The default lays out (roughly) biconnected components of the graph in
   a hexagonal grid arrangement, with the vertices of each biconnected component
   arranged in a circle.  Another option is designed for trees and lattices.
   >>WARNING<< The lattice option has the following bugs:
   1. Loops infinitely on any graph with cycles (i.e. not a lattice or tree)
   2. Sometimes gets confused on large trees.
   Sorry about these bugs, but time constraints have kept me from working on them.

   Frame Configuration2 -0 provides three different window pane configurations for Zgraph.
   Besides the default, the TRIMMED configuration allocates more space to the display pane
   and the WITH-DESCRIPTION-PANE provides a scroll window for viewing descriptions of vertex
   and edge data (discussed under Mouse-M below). 

   >>WARNING<<  One option allows you to modify the output plane width and height.
   These dimensions are the dimensions of a screen array which is BITBLTed onto the
   graph display pane.  If you set these real large, you can pan around (by rubbing
   the mouse on the display pane) much further than with the default setting.  However,
   a huge screen array can take up LOTS of memory.  I haven't placed any safeguards on
   these options.  You have been warned.

Menu Zoom
 --Zooms the currently displayed graph in or out.  >>WARNING<< this operation can take
   a long time on large graphs.  If you abort in the middle of it, the locations of
   vertices will be scattered in a weird manner (some zoomed, some not zoomed) and
   you'll have to explicitly redraw the graph using Reset Scale or another Zoom.
   There's no way to restore the vertices' original positions other than by recreating
   the graph via  Create Graph.

   Comment:  large graphs take a long time to draw.  Their density also makes it difficult
   to read vertex and edge labels.  After the initial drawing, you might want to zoom in
   to view small portions at a time.  Furthermore, panning (discussed below) speeds up alot
   when most of the graph has been clipped from view. 

Reset Scale
 --Resets the scale to effectively undo all previous Zooms and Pans.
  
Help
 --Prints this message.

Exit
 --Buries the Zgraph Display Frame.


MOUSE USAGE
===========
  Mouse-Left
  ----------
  The large center pane is used to display graphs.  In order for large graphs to be easily
  viewed, graphs are drawn onto a large (2500 by 2500) separate pixel array.  The center pane 
  shows a portion of this pixel array, and you can pan this portion by using the left mouse button.
  Simply hold down the left mouse button and move it around.  The direction panned is the same
  as the direction the mouse is moved, mimicking the action of moving a sheet of paper out
  from under a book with your finger.  This panning is very fast, since it just causes a
  copying of a different portion of the hidden bit array onto the display pane.

  But the plot thickens.  Initially, graphs are drawn to fit entirely onto either the center
  pane (on small graphs) or the hidden bit array (on large graphs).  In the latter case,
  you can see all of the graph through the fast panning discussed above.  However, after
  ZOOMing in on a small portion of the graph, some portions are clipped from view.  To pan
  them into view, use the left mouse button, as described above, as if the hidden bit array
  were infinitely large (and the entire graph drawn on it).  When you come to an edge of what
  is displayed on the hidden bit array (indicated by a dashed line), a \"real\" pan will be
2  0performed so that you can view what lies beyond.  Real panning causes the entire graph to be
2  0redrawn, so it is much slower2 0than the fast panning described above.  Whenever real panning
2  0occurs, a message will be displayed indicating the direction of the real pan.  If you ever
2  0pan off of the graph and2 0get lost, execute the Reset Scale command.

  Mouse-Middle
  ------------
  The middle mouse button can be used to select vertices and edges of the displayed graph.
  To select a vertex, move the mouse to the vertex's displayed circle.  When the circle is
  highlited, click MOUSE-M.  To select an edge, move the mouse to the edge's directional
  arrow.  When it is highlited, click MOUSE-M.  (Please note that the who-line displays the
  participating vertices, which is helpful when viewing large graphs.)

  By selecting a vertex or edge, you are given various options for inspecting their associated
  data and one option for recursively displaying their immediate neighbors (also helpful when
  viewing large graphs).  Additionally, if you have selected the WITH-DESCRIPTION-PANE frame
  configuration via the Status command, you are given an option to view a description of the
  vertex or edge in a scroll window pane.  The default description is taken from the output of
  the Common Lisp DESCRIBE function, but this is programmable.

  Mouse-Right
  -----------
  Click MOUSE-R to get a command menu identical to that displayed in the command menu pane.

  Mouse-Control-Middle
2  --------------------
0  While holding down the CONTROL key, click MOUSE-M on a highlited vertex in order to start
  moving it.  You will then be prompted to once again click CONTROL-MOUSE-M wherever you wish
  the vertex to be moved.  The graph is redrawn.  This feature can be of use when viewing
  large graphs or when generating a nice screen image of the graph.

FURTHER HELP
============
For an example of how to write a graph traversal function and a default root-finding form,
press META-.  For a further example of how to integrate Zgraph into an application, 
press SUPER-.

    {press any key to continue}")
  (LET ((command? (SEND *TERMINAL-IO* :TYI)))
    (SEND display-pane :CLEAR-WINDOW)
    (WHEN (SEND display-pane :graph)
      (SEND display-pane :update-image))
    (WHEN (OR (ASSOC command? *zgraph-frame-keystroke-commands*
		     #+Explorer :TEST #+Explorer #'=)
	      (ASSOC command? (SEND display-pane :keystroke-commands)
		     #+Explorer :TEST #+Explorer #'=))
      (SEND *TERMINAL-IO* :FORCE-KBD-INPUT command?))))


(DEFMETHOD (graph-display-frame :exit-command-loop) ()
  "1Used as a temporary command to exit recursive command loop calls.0"
  (THROW 'exit-command-loop NIL))



(COMPILE-FLAVOR-METHODS graph-display-frame 
			graph-display-menu-pane
			graph-display-pane
			zgraph-graph-display-pane
			graph-display-interaction-pane
			description-scroll-pane)

1;;Assign SELECT G to select graph display frames and SELECT CONTROL-G to create new ones.
;;Use #\g when window system => Common Lisp
0#+Symbolics
(TV:ADD-SELECT-KEY (AREF (SI:STRING "g") 0) 'graph-display-frame "Graph Displayer" T)
#+Explorer
(TV:ADD-SYSTEM-KEY #\g 'graph-display-frame "Graph Displayer" T)


1;;Arguments compatible with Symbolics :DRAW-STRING, but behavior is very different. 
0#+Explorer
tv:
(DEFMETHOD (graphics-mixin :draw-string) (STRING x y towards-x towards-y &OPTIONAL IGNORE font (alu TV:ALU-IOR)
					  backwards?)
  "1ANGLE is in radians.  It is a value on the upside-down unit circle.  Examples:
Intended Direction for Text    Value for ANGLE
===========================    ===============
left to right (the default)      0
straight down                    (* PI .5)
right to left                    PI
straight up                      (* PI 1.5)


X,Y is the pixel location to place the first character in STRING at.0"
1  ;;Modeled after :DRAW-CHAR.
0  (PREPARE-SHEET (SELF)
    (LET* ((rise (- towards-y y))
	   (run (- towards-x x)))
      (COND
	1;;If angle is less than 45 degrees, place chars next to each other
0	((< (ABS rise) (ABS run))
	 (COERCE-FONT FONT SELF)
	 (LET ((slope (/ rise run))
	       (x-direction (IF (MINUSP run) -1 1))
	       (fixed-width (FONT-CHAR-WIDTH font))
	       (width-table (font-char-width-table font))
	       (width-of-previous-chars 0))
	   (LABELS ((print-char (char)
				(%DRAW-CHAR FONT char
					    (+ x (* width-of-previous-chars x-direction))
					    1;;Right triangle: height = width * sin(angle)
0					    (+ y (ROUND (* slope width-of-previous-chars x-direction)))
					    ALU SELF)
				(INCF width-of-previous-chars (IF width-table
								  (AREF width-table char)
								  fixed-width))))
	     (IF backwards?
		 (DO ((i (1- (LENGTH string)) (1- i)))
		     ((MINUSP i))
		   (print-char (AREF string i)))
		 (DOTIMES (i (LENGTH string))
		   (print-char (AREF string i)))))))
	1;;Else stack chars on top of each other.
0	(T
	 (COERCE-FONT FONT SELF)
	1 ;;No variable length fonts to worry about.
0	 (LET ((anti-slope (/ run rise))
	       (fixed-height (FONT-CHAR-HEIGHT font))
	       (y-direction (IF (MINUSP rise) -1 1))
	       (height-of-previous-chars 0))
	   (LABELS ((print-char (char)
				(%DRAW-CHAR FONT char
					    (+ x (ROUND (* anti-slope height-of-previous-chars y-direction)))
					    (+ y (* height-of-previous-chars y-direction))
					    ALU SELF)
				(INCF height-of-previous-chars fixed-height)))
	     (IF backwards?
		 (DO ((i (1- (LENGTH string)) (1- i)))
		     ((MINUSP i))
		   (print-char (AREF string i)))
		 (DOTIMES (i (LENGTH string))
		   (print-char (AREF string i)))))))))))