;;; -*- Mode: LISP; Syntax: Common-lisp; Base: 10; Fonts: CPTFONT,CPTFONTB; Package: ZG -*-

1;;Copyright (c) 1986 by John C. Hogge, The University of Illinois.
;;
;;File DEFINITIONS.LISP of system Zgraph.


;;==========================================================================================
;;
;;;; Some graphics support code
;;
;;==========================================================================================

;;
;;Constants are used for efficiency:
;;
0(DEFCONSTANT *1/8PI* (* PI 1/8))
(DEFCONSTANT *1/4PI* (* PI 1/4))
(DEFCONSTANT *1/2PI* (* PI 1/2))
(DEFCONSTANT *3/4PI* (* PI 3/4))
(DEFCONSTANT *5/4PI* (* PI 5/4))
(DEFCONSTANT *3/2PI* (* PI 3/2))
(DEFCONSTANT *7/4PI* (* PI 7/4))
(DEFCONSTANT *2PI* (* PI 2))

(DEFUN angle-for-point (point &OPTIONAL (origin '(0 . 0)))
  "1POINT is a cons (x . y).  Angle returned in radians, from ORIGIN to POINT.0"
  (LET* ((x (- (CAR point) (CAR origin)))
	 (y (- (CDR point) (CDR origin)))
	 (arctan (IF (ZEROP x)
		   *1/2PI*
		   (ATAN (/ (ABS y) (ABS x))))))
    (IF (MINUSP x)
	(IF (MINUSP y)
	    (+ PI arctan)  1;third quadrant
0	    (- PI arctan)) 1;second quadrant
0	(IF (MINUSP y)
	    (- *2PI* arctan) 1;fourth quadrant
0	    arctan))))

(DEFUN find-point (from-point distance angle)
  "1Locates a coordinate at a DISTANCE and ANGLE from FROM-POINT.  ANGLE is in radians.
Distance is in pixels.  Returns the located point: (x . y).0"
  (CONS (+ (ROUND (* (COS angle) distance)) (CAR from-point))
	(+ (ROUND (* (SIN angle) distance)) (CDR from-point))))


1;;
;; Code to set up super-fast panning, using two windows.  The META-WINDOW flavor is used to
;; window into a bigger window which isn't ever exposed.
;;

0(DEFFLAVOR meta-window
	((real-window)
	 (currently-panning? NIL)
	 (who-line-documentation-when-not-panning
	   (SI:STRING "LHold: Pann by holding down Mouse-L while moving mouse around window"))
	 (who-line-documentation-when-panning
	   (SI:STRING "Rub mouse on window to pan.  Release mouse button to stop panning."))
	 extended-width
	 extended-height
	 (x-displacement 0)
	 (y-displacement 0)
	 (acceleration 2)
	 1;;Determines how sensitive we are to the user panning against the edge of the real
0	1 ;;window.
0	 (window-edge-tolerance 10)
	 1;;These are used internally as efficiency hacks.
0	 1;;Keeping these values around is much faster during panning then using message
0	1 ;;sending to get them.
0	 meta-inside-width
	 meta-inside-height
	 meta-x-origin
	 meta-y-origin
	 real-x-origin
	 real-y-origin)
	(TV:WINDOW)
  :SETTABLE-INSTANCE-VARIABLES
1  ;;Must have save bits so we know where the screen array is stored. (TV:SCREEN-ARRAY)
0  (:DEFAULT-INIT-PLIST :SAVE-BITS T)
  (:REQUIRED-INIT-KEYWORDS :extended-width :extended-height)
  (:DOCUMENTATION :COMBINATION "
1Instance variables:
 REAL-WINDOW --Window instance you should send all output to.  The meta-window is used to 
   display a portion of this window.  It is initialized automatically by META-WINDOW's
   :AFTER :INIT as an instance of TV:WINDOW.  You can use the :REAL-WINDOW-FLAVOR init
   keyword to specify a different flavor.  Note: you will probably never need to reset the
    margin size of REAL-WINDOW.  If you do for some reason, you must send 
   :STORE-QUICK-ACCESS-VARIABLES to the meta-window.
 CURRENTLY-PANNING? --used internally.  Set to NIL when user is not currently panning.
 WHO-LINE-DOCUMENTATION-WHEN-NOT-PANNING --Wholine doc string to display when not panning.
 WHO-LINE-DOCUMENTATION-WHEN-PANNING --Wholine doc string to display when panning.
 EXTENDED-WIDTH --Extra width in pixels to make REAL-WINDOW.  REAL-WINDOW's width is this + 
   the meta-window's width.
 EXTENDED-HEIGHT --Extra height in pixels to make REAL-WINDOW.  REAL-WINDOW's height is 
   this + the meta-window's height.
 X-DISPLACEMENT --Number of pixels the currently displayed portion of REAL-WINDOW is 
   displaced in X from coordinate 0,0
 Y-DISPLACEMENT --Number of pixels the currently displayed portion of REAL-WINDOW is
   displaced in Y from coordinate 0,0
 ACCELERATION --During panning when the user moves the mouse N pixels, we pan N x 
   ACCELERATION pixels."0))

#+Symbolics
(DEFMETHOD (meta-window :AFTER :INIT) (options)
  (SETQ real-window
	(TV:MAKE-WINDOW (OR (GETF (CDR options) :real-window-flavor)
			    'TV:WINDOW)
			  :WIDTH (+ TV:WIDTH extended-width)
			  :HEIGHT (+ TV:HEIGHT extended-height)
			  :DEEXPOSED-TYPEOUT-ACTION :PERMIT
			  :SAVE-BITS T))
1  ;;Send this so that our :AFTER :SET-SIZE demon fires, to initialize various instance 
  ;;variables.
0  (SEND SELF :store-quick-access-variables))
#+Explorer
(DEFMETHOD (meta-window :AFTER :INIT) (options)
1  ;;Sometimes the meta-window's screen array is bigger than itself.  The MAX calls make sure
  ;;the real window is no smaller than this screen array.  In addition, BITBLT requires that
  ;;the width of the screen array be a multiple of 32.
0  (LET ((w (multiple-of (MAX (+ TV:WIDTH extended-width)
			     (OR (CADR (ARRAY-DIMENSIONS TV:SCREEN-ARRAY)) 0))
			32))
	(h (MAX (+ TV:HEIGHT extended-height) (OR (CAR (ARRAY-DIMENSIONS TV:SCREEN-ARRAY)) 0))))
    (SETQ real-window
	  (TV:MAKE-instance (OR (GETF (CDR options) :real-window-flavor)
				'TV:WINDOW)
			    :WIDTH w
			    :HEIGHT h
			    :DEEXPOSED-TYPEOUT-ACTION :PERMIT))
1    ;;I don't know why the bleep this is required, but guessed at it after staring at
    ;;:METHOD (tv:SHEET :SET-SAVE-BITS).  Without it, :SET-SAVE-BITS barfs.  Warning:  only 
    ;;tested on the TV:LOCATIONS-PER-LINE=2 case.
0    (WHEN (> w 1024)
      (set-in-instance real-window 'tv:locations-per-line (* 32 (/ (multiple-of w 1024) 1024))))

    (SEND real-window :SET-SAVE-BITS T)
    1;;Send this so that our :AFTER :SET-SIZE demon fires, to initialize various instance 
0    1;;variables.
0    (SEND SELF :store-quick-access-variables)))

(DEFUN multiple-of (x mult)
  (LET ((mod (mod x mult)))
    (IF (ZEROP mod)
	x
	(+ x (- mult mod)))))

(DEFMETHOD (meta-window :AFTER :SET-SIZE) (&REST IGNORE)
  "1Whenever our window size changes, update size-related instance variables.0"
  (SEND SELF :store-quick-access-variables))

(DEFMETHOD (meta-window :AFTER :CHANGE-OF-SIZE-OR-MARGINS) (&REST IGNORE)
  "1Whenever our window size changes, update size-related instance variables.0"
  (SEND SELF :store-quick-access-variables))


(DEFMETHOD (meta-window :store-quick-access-variables) ()
  "1Stores size-related info in instance variables for fast reference.0"
1  ;;These two are the inside window width and height, the size of the window area
  ;;inside of the window borders.
0  (SETQ meta-inside-width (SEND SELF :INSIDE-WIDTH)
	meta-inside-height (SEND SELF :INSIDE-HEIGHT))
  1;;These are the pixel location (relative to SELF) of the upper-left-most pixel.
0  (MULTIPLE-VALUE-SETQ (meta-x-origin meta-y-origin)
      (SEND SELF :INSIDE-EDGES))
  (MULTIPLE-VALUE-SETQ (real-x-origin real-y-origin)
      (SEND real-window :INSIDE-EDGES)))

(DEFMETHOD (meta-window :update-image) ()
  "1Redraw the image stored on REAL-WINDOW onto SELF.0"
  1;;Copy from REAL-WINDOW to SELF.  We copy a portion of REAL-WINDOW's screen array
0  1;;equal to the inside size of SELF, starting at the specified displacement.
0  (BITBLT TV:ALU-SETA meta-inside-width meta-inside-height
	  (SEND real-window :SCREEN-ARRAY)
	  (+ x-displacement real-x-origin)
	  (+ y-displacement real-y-origin)
	  TV:SCREEN-ARRAY meta-x-origin meta-y-origin))

(DEFMETHOD (meta-window :update-displacement) (x-delta y-delta &OPTIONAL absolute?)
  "1Modify the stored displacement values by X-DELTA, Y-DELTA.
Regardless of what X-DELTA and Y-DELTA are, error checking is done to make sure
we don't displace beyond the window boundaries of our REAL-WINDOW; in this case, we
return non-NIL so that the handler can decide what to do (for instance, he might
want to do a real pan).  The value we return is either :LEFT, :RIGHT, :UP, or :DOWN,
cooresponding to the direction of the window edge the user was banging against.  (The
top of the window is :UP, etc.  This may not coorespond with your coordinate system!)
ABSOLUTE? may be specified non-nil in order to interpret the
first two args as absolute x,y displacements.0"
1  ;;Make sure the deltas are integers.
0  (SETQ x-delta (ROUND x-delta)
	y-delta (ROUND y-delta))
  (MULTIPLE-VALUE-BIND (max-real-window-width max-real-window-height)
      (SEND real-window :inside-size)
1    ;;Modify the values.
0    (COND (absolute?
	   (SETQ x-displacement x-delta
		 y-displacement y-delta))
	  (T
	   (INCF x-displacement x-delta)
	   (INCF y-displacement y-delta)))
    (LET ((lose? NIL)
	  (max-meta-window-x (- max-real-window-width meta-inside-width))
	  (max-meta-window-y (- max-real-window-height meta-inside-height)))
      (COND-EVERY
	((MINUSP x-displacement)
	 (SETQ x-displacement 0
	       lose? :LEFT))
	((MINUSP y-displacement)
	 (SETQ y-displacement 0
	       lose? :UP))
	((> x-displacement max-meta-window-x)
	 (SETQ x-displacement max-meta-window-x
	       lose? :RIGHT))
	((> y-displacement max-meta-window-y)
	 (SETQ y-displacement max-meta-window-y
	       lose? :DOWN)))
      lose?)))

(DEFMETHOD (meta-window :pan-around) ()
  "1Lets the user dynamically pan the meta-window.
It's assumed that the left mouse button is being held down upon call to this method,
since the dynamic pan is aborted upon release of the mouse button.  We return as soon
as the user releases the mouse button, or when the user gets to the edge of the real
window.  If he tries to pan past it, we return either :LEFT, :RIGHT, :UP, or :DOWN.
Otherwise we return NIL.0"
  (LET ((previous-mouse-x TV:MOUSE-X)
	(previous-mouse-y TV:MOUSE-Y)
	hitting-window-edge?)

1    ;;Set this so that SELF's who-line message concerning how to stop panning is activated.
    ;;Use an UNWIND-PROTECT to make sure it's set back to NIL, incase user aborts.
0    (SETQ currently-panning? T)
    (CATCH 'panning-past-window-edge
      (UNWIND-PROTECT
	  (LOOP DO
		(WHEN (AND hitting-window-edge?
			   (EQ (LENGTH hitting-window-edge?) window-edge-tolerance))
		  (THROW 'panning-past-window-edge (CAR hitting-window-edge?)))
		(COND
		1  ;;If user releases the mouse button, exit loop.
0		  ((ZEROP (TV:MOUSE-BUTTONS))
		   (RETURN T))
		  (T
		   1;;Save these incase the mouse moves during computation of
0		1   ;;:UPDATE-DISPLACEMENT.
0		   (LET* ((x TV:MOUSE-X)
			  (y TV:MOUSE-Y)
			  1;;Set SELF's displacement to the delta between the current mouse
0			1  ;;position and the position
0			  1;;it was in when we were called, times the acceleration factor.
0			  (result (SEND SELF :update-displacement
					(* acceleration (- previous-mouse-x x))
					(* acceleration (- previous-mouse-y y)))))
		     (WHEN result
		       (PUSH result hitting-window-edge?))
		     (SETQ previous-mouse-x x
			   previous-mouse-y y)
		     1;;Copy image from REAL-WINDOW to SELF, given the new displacement.
0		     (SEND SELF :update-image)))))
	(SETQ currently-panning? NIL))
1      ;;Return NIL, unless the THROW passes a value for the CATCH to return.
0      NIL)))


(DEFMETHOD (meta-window :who-line-documentation-string) ()
  "1When not panning, show a panning prompt message in the wholine.
When in the middle of panning, show a message about how to quit panning.0"
  (IF currently-panning?
      who-line-documentation-when-panning
      who-line-documentation-when-not-panning))


(DEFMETHOD (meta-window :translate-mouse-blip) (mouse-blip)
  "1Modifies MOUSE-BLIP, which is presumed to be a blip input on SELF, such that
the blip's x and y coordinates are set to what they would be if the blip had been input
from REAL-WINDOW.  In other words, we use the current x and y displacement to figure out
the REAL-WINDOW mouse location which cooresponds to the location of MOUSE-BLIP.0"
  (SETF (THIRD mouse-blip) real-window
	(FOURTH mouse-blip) (+ (FOURTH mouse-blip) x-displacement)
	(FIFTH mouse-blip) (+ (FIFTH mouse-blip) y-displacement))
  mouse-blip)

(DEFMETHOD (meta-window :send-to-real) (message &REST args)
  "1Sends MESSAGE and ARGS to SELF's REAL-WINDOW.0"
  (APPLY real-window message args))

(COMPILE-FLAVOR-METHODS meta-window)

#| 1Example of how to use the meta-window:

0(DEFFLAVOR my-window () (meta-window)) 
(DEFVAR my-window (TV:MAKE-WINDOW 'my-window
				  :width 700
				  :height 700
				  :extended-width 1000
				  :extended-height 1000
				  :deexposed-typeout-action :permit))

(PROGN
  (SETQ real-window (send my-window :real-window))
1  ;;These four messages just reset things inbetween test evaluations.
0  (send real-window :clear-window)
  (send my-window :clear-window)
  (send my-window :set-x-displacement 0)
  (send my-window :set-y-displacement 0)
  (tv:window-call (my-window)
1    ;;Draw some random lines on the real window.
0    (MULTIPLE-VALUE-BIND (w h)
	(send real-window :inside-size)
      (print-disk-label 0 real-window)
      (dotimes (x 500)
	(SEND real-window :draw-line (random w) (random h) (random w) (random h))))
1    ;;Initial update so user sees the image before panning begins.
0    (SEND my-window :update-image)
    (LOOP DO
	1  ;;Prompt for left mouse
0	  (WHEN (= (cadr (SEND my-window :any-tyi)) #\mouse-l) ;mouse-l
	    (SEND my-window :pan-around)))))					    
|#


1;;
;; Code for having mouse-sensitive graphic objects.
;;

0(DEFFLAVOR mouse-sensitive-graphics-mixin ((current-item NIL))
	   ()
  :settable-instance-variables
  :initable-instance-variables
#+Symbolics
  (:REQUIRED-METHODS :item-at? :box-item-at)
#+Symbolics
  (:REQUIRED-FLAVORS TV:GRAPHICS-MIXIN)
  (:DOCUMENTATION :MIXIN "
1This flavor provides the basic framework for setting up complex mouse sensitivity
in which mouse sensitive items do not have to be detected based on rectangular 
areas0 1of the screen.  The boxing of mouse sensitive items is left entirely up to 
the user,0 1so you aren't forced to draw an actual box around items.  (For instance,
you could outline0 1it, draw it in reverse video, etc.)

These are the steps for using this flavor:
1. Mix it into a window flavor, say FOO.

2. Define for FOO an :ITEM-AT? method which takes two arguments, X and Y screen 
locations.0  1(X,Y) is where the mouse is at the time of the call.  Return non-NIL 
if there is an item to0 1be boxed at (X,Y).  The value returned should be whatever 
data corresponds to the item.  If0 1the user clicks the mouse while an item is being
boxed, this value is inserted into the mouse0 1blip, as the sixth item.  There are 
usually only five elements in any mouse blip, so by0 1checking for a non-NIL sixth 
element, you can detect when the user has clicked on a mouse0 1sensitive item.

Note that your :ITEM-AT? method has to be fairly efficient, otherwise mouse 
movement will0 1slow down tremendously.

3. Define for FOO a :BOX-ITEM-AT method which takes three arguments: an item returned
by :ITEM-AT?, and the same X and Y args as above.  This method should draw some sort of
box around whatever the item.  Your draw routine MUST use TV:ALU-XOR, since :BOX-ITEM-AT is 
subsequently used to erase the box when0 1the mouse moves away from (X,Y).

4. Optionally, if you want the who-line to provide some documentation when the 
mouse is0 1boxing an item, define an :ITEM-WHO-LINE-DOCUMENTATION method for FOO 
which takes an0 1item (the thing returned by :ITEM-AT?) and returns an appropriate 
documentation string.0  1For instance, you could say what the mouse buttons do, or 
you could further describe the0 1item, or both.0"))


(DEFMETHOD (mouse-sensitive-graphics-mixin :mouse-moves) (x y)
1  0"1If within an item, box it unless we have already.0"
  (LET ((were-we-over-a-mouse-sensitive-item? current-item))
    (COND
      ((SETQ current-item (SEND SELF :item-at? x y))
       (COND
	 ((NULL were-we-over-a-mouse-sensitive-item?)
	  (SEND SELF :box-item-at current-item x y))
	 1;;If we moved from one item to another...
0	 ((NEQ were-we-over-a-mouse-sensitive-item? current-item)
	  1;;Blank out box around previous mouse-sensitive item.
0	1  ;;x and  y are slightly bogus here
0	  (SEND SELF :box-item-at were-we-over-a-mouse-sensitive-item? x y) 
	  1;;Draw box around new item.
0	  (SEND SELF :box-item-at current-item x y))))
      1;;When we've left an item, un-box it.
0      (were-we-over-a-mouse-sensitive-item?
       (SEND SELF :box-item-at were-we-over-a-mouse-sensitive-item? x y)))))


(DEFWRAPPER (mouse-sensitive-graphics-mixin :WHO-LINE-DOCUMENTATION-STRING) ((IGNORE) . body)
  "1If the mouse is over an item, try providing any user-written who-line doc
on that item.  Otherwise return the usual string (whatever that may be).0"
  `(OR (WHEN current-item
	 (SEND SELF :item-who-line-documentation current-item))
       (PROGN . ,body)))

(DEFMETHOD (mouse-sensitive-graphics-mixin :item-who-line-documentation) (item)
  "1This is to be redefined by the flavor user.  Should return a who-line doc string,
given an ITEM as argument.  ITEM is whatever the user returns in his :ITEM-AT? method.0"
  item
  NIL)


(DEFMETHOD (mouse-sensitive-graphics-mixin :MOUSE-CLICK) (button x y)
  "1This adds the mouse sensitive item as the sixth element of the mouse blip,
whenever the mouse is clicked over a mouse-sensitive item.0"
  (WHEN current-item
    (SEND SELF :FORCE-KBD-INPUT `(:MOUSE-BUTTON ,button ,self ,x ,y ,current-item))
    T))

(DEFMETHOD (mouse-sensitive-graphics-mixin :AFTER :HANDLE-MOUSE) ()
1  0"1If the mouse is over a boxed item when we lose control of the mouse (as
when window is exposed or mouse is held down,0 1undraw the box and set the flag off.
Otherwise the item item is often not boxed unless the mouse is NOT over it (the
behavior inverts).0"
  (WHEN current-item
    (MULTIPLE-VALUE-BIND (x-offset y-offset)
	(TV:SHEET-CALCULATE-OFFSETS SELF TV:MOUSE-SHEET)
      (SEND SELF :box-item-at current-item (- TV:MOUSE-X x-offset) (- TV:MOUSE-Y y-offset)))
    (SETQ current-item NIL)))


(COMPILE-FLAVOR-METHODS mouse-sensitive-graphics-mixin)

#| 1Example:

0(DEFFLAVOR hogge-window () (mouse-sensitive-graphics-mixin TV:WINDOW))

(DEFMETHOD (hogge-window :item-at?) (x y)
  (WHEN (AND (< 480 x 510) (< 480 y 510))
    'center-object))

(DEFMETHOD (hogge-window :box-item-at) (item x y)
  (SEND SELF :draw-box 480 480 510 510 TV:ALU-XOR))

(DEFMETHOD (mouse-sensitive-graphics-mixin :item-who-line-documentation) (item)
  (si:string (FORMAT NIL "~S" item)))

(DEFMETHOD (hogge-window :draw-box) (lower-left-x lower-left-y upper-right-x upper-right-y
						  alu)
  (SEND SELF :DRAW-LINE lower-left-x lower-left-y lower-left-x upper-right-y alu)
  (SEND SELF :DRAW-LINE lower-left-x upper-right-y upper-right-x upper-right-y alu)
  (SEND SELF :DRAW-LINE upper-right-x upper-right-y upper-right-x lower-left-y alu)
  (SEND SELF :DRAW-LINE upper-right-x lower-left-y lower-left-x lower-left-y alu))


(LET ((hogge-window (TV:MAKE-WINDOW 'hogge-window
				    :MORE-P NIL
				    :DEEXPOSED-TYPEOUT-ACTION :PERMIT
				    :WIDTH 700)))
 (SEND hogge-window :expose)
 (SEND hogge-window :draw-circle 500 500 5)
 (LOOP FOR blip = (SEND hogge-window :any-tyi)
       DO
   (WHEN (SIXTH blip)
     (TV:MENU-CHOOSE (LIST (SIXTH blip))))))

|#


1;;==========================================================================================
;;
;;;; Global variables and constants.
;;
;;  Most of these are used by both the graph plotting code
;;  and the user interface code, so they appear here.
;;
;;==========================================================================================

0(DEFVAR *graph-types* NIL
  "1List of the types of graphs the Graph Displayer can display0.
1List entries are GRAPH-TYPE structs.0")

(DEFVAR *graph-output* NIL
  "1The window to draw graphs on.
The way the Graph Display Frame code is defined, this is bound to a large,
hidden graphics window.  Its meta-window displays a portion of the graph at any given time.0")

(DEFVAR *display-io* NIL
  "1The meta-window which displays what is drawn on *GRAPH-OUTPUT*.
The scaling code uses this to make sure that *GRAPH-OUTPUT* isn't shrunk smaller than the
meta window.0")

(DEFVAR *description-output* NIL
  "1The scroll window (if present) which should be used to display a description of
a selected vertex or edge.  If NIL, no such scroll window is available.0")

(DEFVAR *graph-debug-actions* '(:print-graph-computation-messages)
  "1List of extra things to do while computing a graph.  Refer to *ALL-GRAPH-DEBUG-ACTIONS*
for possibilities.0")

(DEFVAR *all-graph-debug-actions*
	'(("Graph Computation Messages" . :print-graph-computation-messages)
	  ("Step Thru Placement" . :step-through-placement))
  "1Possible elements *GRAPH-DEBUG-ACTIONS*. These are actions to do while computing graphs.
:PRINT-GRAPH-COMPUTATION-MESSAGES = print which vertices are reached from each vertex.
:STEP-THROUGH-PLACEMENT = display the graph at each stage of the placement algorythm.0")

(DEFMACRO debug-print (destination control-string &REST args)
  `(WHEN (MEMQ :print-graph-computation-messages *graph-debug-actions*)
     (FORMAT ,destination ,control-string . ,args)))



(DEFVAR *all-graph-plotting-styles*
	'(("One Big Circle"
	   . :plot-in-one-big-circle)
	  ("Circles For Each Bi-Connected Component"
	   . :plot-in-circles-for-bi-connected-components)
	  ("Plot as a Lattice (use only for lattices!!)"
	   . :plot-lattice))
  "1Alist of methods for plotting graphs in world coordinates.0")

(DEFVAR *graph-plotting-style* :plot-in-circles-for-bi-connected-components
  "1Current method to use of those in *all-graph-plotting-styles* in plotting graphs.0")


(DEFVAR *smallest-edge-length-to-bother-labelling* 100.
  "1Any edges with length less than this won't be labelled.  This reduces clutter in the
graph display.0")


(DEFVAR *too-many-non-leaves* 80.
  "1If the graph has more leaves than this, don't try to minimize crossovers.0")

(DEFVAR *too-many-edges* 400.
  "1If the graph has more edges than this, don't try to minimize crossovers.0")


(DEFVAR *percentage-radius-for-leaves-in-circular-arrangement* .30
  "1Given a circle of radius R in which to place a given set of non-leaf
vertices and the leaves to which they are connected, this var determines
the radius of the inner circle on which to place the non-leaves.  The
leaves are placed outside of this inner circle.  INNER-CIRCLE-RADIUS =
OUTER * this variable0")

(DEFVAR *gd* NIL "1Bound in Graph Displayer command loop to the Graph Display Frame.0")

(DEFVAR *default-pane-configuration* 'main
  "1Specifies the configuration of window panes to use when instantiating ZGRAPH-DISPLAY-FRAME.
Can be either ZG:MAIN or ZG:TRIMMED.  The former is the default.  If you prefer TRIMMED,
feel free to SETQ this in your init file.0")

(DEFVAR *number-vertices-over-which-graphs-are-fit-onto-hidden-bit-array* 18
  "1If a graph has more than this number of vertices, it is fit onto the hidden bit array
when first displayed.  Otherwise, it is fit onto the display pane.0")

(DEFVAR *dashed-line-margin* 10
  2"Gap in pixels between inside edge of *GRAPH-OUTPUT* and the dashed line drawn to indicate
the viewing boundary.  If NIL, the dashed line isn't drawn at all."0)


1;;==========================================================================================
;;
;;;; Organization of vertices of a graph on a cartegion plane.
;;
;;==========================================================================================


0(DEFSTRUCT (vertex
	     #+Explorer
	     (:PRINT-FUNCTION 
	       (LAMBDA (v stream IGNORE)
		 (FORMAT stream "#<VERTEX ~a>" (vertex-data v)))))
 "1Holds a graph vertex.
0  1Slot LOCATION is a cons: ({x coordinate} . {y coordinate})
0  1Slot EDGES is a list of EDGE structs of edges incident from this vertex.
0  1Slot CONNECTED-TO is a list of vertices we are connected to via In and Out links (edges).
0  1DATA is LISP object associated with the vertex.
0  1MISC is a slot used internally for performing searches on the graph.  Currently it is used
only for the finding of bi-connected components.0"
  location
  edges
  connected-to
  data
  misc)

(DEFSTRUCT (edge
	     #+Explorer
	     (:PRINT-FUNCTION
	       (LAMBDA (e stream IGNORE)
		 (FORMAT stream "#<EDGE ~a->~a by ~a>"
			 (vertex-data (edge-vertex-1 e))
			 (vertex-data (edge-vertex-2 e))
			 (IF (CDR (vertex-data e)) (vertex-data e) (CAR (vertex-data e)))))))
  "1Holds a graph edge.
0  1Slots VERTEX-1 and VERTEX-2 hold the endpoint vertices of this edge.
0  1SLOPE is used internally as the angle (in radians) of the vector from VERTEX-1 to VERTEX-2.
0  1DATA is a list of LISP objects associated with the edge, usually labels expressing any
relationships between the LISP objects (data) of the vertices the edge connects.  It is
important to note that this is a list of labels.  Zgraph combines any edges leading from one
vertex to another and displays a composite label.  In this way, the edge labels do not 
overwrite each other.  (It also makes edge-crossover minimization more accurate.)
  MISC is a slot used internally for storing label locations for doing mouse-sensitivity.0"
  vertex-1
  vertex-2
  slope
  data
  misc)

1;;Accessor macros which the following algorithm uses to store data in the MISC slot.
0(DEFSUBST vertex-mark (vertex)
  (CAR (vertex-misc vertex)))

(DEFSUBST vertex-dfnumber (vertex)
  (CADR (vertex-misc vertex)))

(DEFSUBST vertex-low (vertex)
  (CADDR (vertex-misc vertex)))

(DEFSUBST vertex-father (vertex)
  (CADDDR (vertex-misc vertex)))


(DEFUN get-compiled-function-or-die-trying (object)
  "1OBJECT is either a compiled function or a symbol with a function definition.
If the latter and the definition is compiled, it is returned.  If it isn't compiled,
we compile it and return.0"
  (IF (TYPEP object 'COMPILED-FUNCTION)
      object
      (LET ((cf? (SYMBOL-FUNCTION object)))
	(IF (TYPEP cf? 'COMPILED-FUNCTION)
	    cf?
	    1;else
0	    (COMPILE object)
	    (SYMBOL-FUNCTION object)))))

