;;; -*- Mode:  Lisp; Syntax: Common-lisp; Package: USER -*-

;;; Character-oriented menu system

;; "Pure" CL for portability.

;;; The idea is to duplicate some reasonable subset of
;;; the old Symbolics menu system, w/o all the machine-specific
;;; aspects.

(defvar *line-length* 80)
(defvar *help-key* '?) ;; Can be re-bound
(defvar *abort-key* 'q)


;;;; Basic choice menu

;; When there is a discrete choice to make, this is the menu to use.
;; Items are specified as (<printed form> <value returned>). 

(defun cmenu-choose (item-list &optional (header "Choose one of")
			       (help "~% Sorry, no help available."))
  (cond ((null item-list) nil)
	;; dangerous option((null (cdr item-list)) (cadar item-list)) ;; no choice, really!
	(t 
  (do ((len (length item-list))
       (answer nil)
       (result nil)
       (done? nil))
      (done? result)
    (show-basic-menu-choices header item-list)
    (setq answer (read-stuff))
    (cond ((integerp answer)
	   (cond ((or (< answer 1) (> answer len))
		  (format *standard-output* "~%  Must be between 1 and ~D." len))
		 (t (setq result (cadr (nth (1- answer) item-list))
			  done? t))))
	  ((eq answer :PUNT) (setq done? t))
	  ((eq answer :HELP) (format *standard-output* "~A" help))
	  (t (format *standard-output* "~%  Must be an integer between 1 and ~D."
		     len)))))))

(defun choice-aborted? (choice) (or (null choice) (eq choice :punt)))

(defun show-basic-menu-choices (header items)
  ;; This version tries to put as many things on a line as will fit
  (format *standard-output* "~%~A:~%" header)
  (do ((i 1 (1+ i))
       (room-left *line-length*)
       (item "") (len 0)
       (counter 0 (1+ counter))
       (choice (car items) (car rest))
       (rest (cdr items) (cdr rest)))
      ((null choice))
    (setq item (format nil "~D: ~A " i (car choice))
	  len (length item))
    (when (or (> len room-left) (< room-left 0)) ;; 2nd case is something being too long
      (format *standard-output* "~%")
      (setq room-left *line-length*))
    (format *standard-output* "~A" item)
    (setq room-left (- room-left len))))

(defun show-basic-menu-choices-old (header items)
  (format *standard-output* "~%~A:" header)
  (do ((i 1 (1+ i))
       (counter 0 (1+ counter))
       (choice (car items) (car rest))
       (rest (cdr items) (cdr rest)))
      ((null choice))
    (when (= 0 (mod counter 4)) (format *standard-output* "~%"))
    (format *standard-output* "~20@<~D: ~A~>" i (car choice))))

(proclaim '(special *cmenu-prompt*))
(defvar *cmenu-prompt* ">->") ;; default, can be lambda-bound

(defun print-cmenu-prompt (&optional (header ""))
  (if (string= header "") (format t "~%~A" *cmenu-prompt*)
      (format t "~%~A" header)))

(defun read-stuff (&optional (header ""))
  (print-cmenu-prompt header)
  (let ((thing (read *standard-input*)))
    (cond ((symbolp thing)
	   ;; get rid of package dependence for special
	   ;; cases, so far just Q and ?.
	   (let ((name (symbol-name thing)))
	     (cond ((string= name "Q") :PUNT)
		   ((string= name "?") :HELP)
		   (t thing))))
	  (t thing))))

;;;; Choosing variable values
;;
;; Once a variable has been chosen, select a value for it.
;; Item lists here are (<printed version> <variable> <value-type>).
;; It is assumed that all variables are bound.

;; Supported datatypes: 
;;
;; :anything
;; :integer
;; :float
;; :string
;; :list-of-strings (for lists of files)
;;  This facility is data-driven, so new types may be added.

(defun cmenu-choose-values (item-list &optional (header "Choose values for")
			    (help ""))
  (do ((len (length item-list))
       (answer nil)
       (done? nil))
      (done? nil)
    (show-cvalues-menu-choices header item-list)
    (setq answer (read-stuff))
    (cond ((integerp answer)
	   (cond ((or (< answer 1) (> answer len))
		  (format *standard-output* "~%  Must be between 1 and ~D." len))
		 (t (twiddle-variable-value (nth (1- answer) item-list)))))
	  ((eq answer ':punt) (setq done? t))
	  ((eq answer ':help) (basic-menu-documentation item-list len help))
	  (t (format *standard-output* "~%  Must be an integer between 1 and ~D."
		     len)))))

(defun show-cvalues-menu-choices (header items)
  (format *standard-output* "~%~A:" header)
  (do ((i 1 (1+ i))
       (choice (car items) (car rest))
       (rest (cdr items) (cdr rest)))
      ((null choice))
    (format *standard-output* "~%  ~D: ~A [~A]" i (car choice) 
	    (symbol-value (cadr choice)))))

(defun twiddle-variable-value (var-spec)
  (let ((val (funcall (lookup-cvalues-type-reader (caddr var-spec)) var-spec)))
    (unless (eq val ':NO-VALUE-PROVIDED)
      (set (cadr var-spec) val))))

(defvar *cvalues-types* nil)

(defun lookup-cvalues-type-reader (type)
  (let ((entry (assoc type *cvalues-types*)))
    (if entry (cdr entry) (function read))))

(defmacro Define-cvalues-type (type function)
  `(let ((entry (assoc ',type *cvalues-types*)))
     (unless entry
       (push (setq entry (cons ',type nil)) *cvalues-types*))
     (setf (cdr entry) (function ,function))))

(defun read-list-of-unempty-strings ()
  (do ((answer nil)
       (current nil))
      (nil)
    (setq current (read-line))
    (when (string= current "")
      (return (nreverse answer)))
    (push current answer)))

(defun read-integer ()
  (do ((answer nil))
      (nil)
    (setq answer (read))
    (cond ((integerp answer) (return answer))
	  (t (print-cmenu-prompt " Must be an integer...try again.:")))))

(defun read-float ()
  (do ((answer nil))
      (nil)
    (setq answer (read))
    (cond ((floatp  answer) (return answer))
	  (t (print-cmenu-prompt " Must be a flonum...try again.")))))

(Define-cvalues-type :anything
		     (lambda (var-spec)
		       (print-cmenu-prompt
			(format nil " New value for ~A:" (car var-spec)))
		       (read)))
(Define-cvalues-type :string
		     (lambda (var-spec)
		       (print-cmenu-prompt
			(format nil " New value for ~A:" (car var-spec)))
		       (read-line)))
(Define-cvalues-type :list-of-strings
		     (lambda (var-spec)
		       (print-cmenu-prompt
			(format nil " New value for ~A:" (car var-spec)))
		       (read-list-of-unempty-strings)))
(Define-cvalues-type :integer
		     (lambda (var-spec)
		       (print-cmenu-prompt 
			(format nil " New value for ~A:" (car var-spec)))
		       (read-integer)))
(Define-cvalues-type :float
		     (lambda (var-spec)
		       (print-cmenu-prompt 
			(format nil " New value for ~A:" (car var-spec)))
		       (read-float)))

(Define-cvalues-type :one-of (lambda (var-spec)
			       (let ((choices (cadr (member :one-of var-spec))))
				 (cmenu-choose
				   choices 
				   (format nil "~% Select new value for ~A:"
					   (car var-spec))))))
(Define-cvalues-type :boolean (lambda (var-spec)
				(yes-or-no-p (car var-spec))))

;;;; Command menus
;;
;; A minimalist implementation of a command processor.
;; An entry is:
;; (<key> <code> &optional :documentation <doc string>)
;;
;; For all command menus, Q means quit, 0 means re-display options.
;; Some actions will in turn be command menus, of course. 

(defun run-command-menu (cmenu header &aux len hprompt)
  (setq len (length cmenu))
  (setq hprompt (concatenate 'string header ">>"))
  ;; Runs command menu until told otherwise
  (show-basic-menu-choices header cmenu)
  (do ((answer (read-stuff hprompt)))
      (nil)
      (case answer
	    (:PUNT (return t))
	    (:HELP (basic-menu-documentation cmenu len ""))
	    (0 (show-basic-menu-choices header cmenu))
	    (t (cond ((integerp answer)
		      (cond ((or (< answer 1) (> answer len))
			     (format t
				     "~%  Must be between 1 and ~D." len))
			    (t (eval (cadr (nth (1- answer) cmenu))))))
		     (t (format t "<-  Must be 0,1...~D, Q, or ?." len)))))
      (setq answer (read-stuff hprompt))))

(defvar *basic-menu-help*
"  0: Redisplay choices.
  Q: Quits
  ?: Help (if available)
  #: That choice, or help concerning it.") 

(defun basic-menu-documentation (cmenu len &optional (global-help ""))
  (unless (string= global-help "")
    (format t "~%~A" global-help))
  (format t "Either ?,Q, or 1-~D:" len)
  (let ((answer (read-stuff)))
    (cond ((eq answer ':HELP)
	   (format t *basic-Menu-Help*))
	  ((eq answer ':PUNT))
	  ((integerp answer)
	   (cond ((or (< answer 0) (> answer len))
		  (basic-menu-documentation cmenu len global-help))
		 (t (let ((answer (cadr (member :documentation (nth (1- answer) cmenu)))))
		      (cond (answer (format t "~%~A" answer))
			    (t (format t "~% Sorry, no help available for item #~D" len)))))))
	  (t (basic-menu-documentation cmenu len)))))

