;;; -*- Mode: LISP; Syntax: Common-lisp; Fonts: MEDFNT; Package: USER -*-


;;;; Bibliography checker for LaTeX
;;  Two problems continually crop up with papers:
;;  (1) what cite's are unbound, and (2) what 
;;  bibitems are not referenced.  This program
;;  produces a report of such things.

;; There are alot of similar dumb, text-based problems
;; so this system is written to be very modular.  To wit,
;; generic "string recognizers" are passed over the file.  Each
;; string recognizer carries its own state.  

(defvar *string-recognizers* nil)

(defun check-latex-references (file)
  (setq *string-recognizers* (list (function glom-cite)
				   (function glom-bibitem)))
  (clear-biblio-index)
  (run-recognizers-on-file file)
  (show-biblio-index))

(defun run-recognizers-on-file (file)
  (run-recognizers-on :START)
  (with-open-file (in file :direction :input)
    (do ((char (read-char in nil nil)
	       (read-char in nil nil)))
	((null char)
	 (run-recognizers-on :END))
;      (format t "~% Input = ~A" char)
      (run-recognizers-on char))))

(defun run-recognizers-on (data)
  (dolist (recognizer *string-recognizers*)
    (funcall recognizer data)))

;;; Biblio system

(defvar *citations* nil) ;; alist of (<item name> <number of cites>
                         ;;           <number of defs>)

(defun clear-biblio-index ()
  (setq *citations* nil))

(defun add-bibitem (item) (incf (caddr (get-biblio-entry item))))

(defun add-cite (item) (incf (cadr (get-biblio-entry item))))

(defun get-biblio-entry (item)
  (let ((entry (assoc item *citations* :test #'string=)))
    (unless entry (setq entry (list item 0 0))
	    (push entry *citations*))
    entry))

(defun show-biblio-index ()
  (let ((def-not-cite nil)
	(cite-not-def nil)
	(cite-count 0)
	(def-count 0))
    (dolist (citation *citations*)
      (incf cite-count (cadr citation))
      (incf def-count (caddr citation))
      (if (= (cadr citation) 0) ;uncited
	  (if (= (caddr citation) 0) ;undefined
	      (error "~% ~A neither cited nor defined!" (car citation))
	      (push citation def-not-cite))
	  (if (= (caddr citation) 0) ;undefined
	      (push citation cite-not-def))))
    (cond (cite-not-def
	   (format t "~% ~D citations not defined:" (length cite-not-def))
	   (pprint (sort (mapcar #'car cite-not-def) #'string<)))
	  (t (format t "~% All citations are defined.")))
    (cond (def-not-cite
	   (format t "~% ~D references not cited:" (length def-not-cite))
	   (pprint (sort (mapcar #'car def-not-cite) #'string<)))
	  (t (format t "~% All citations are referenced.")))
    (format t "~% ~D citations, ~D references total." cite-count def-count)))


;;;; Recognizer for cite

;; States:
;; :WAIT -- looks for "\", transitions to :SNIFF.
;; :SNIFF -- looks for "cite{", if not found, back to :WAIT.
;;           otherwise, goes to :READ-ITEMS.
;; :READ-ITEMS -- if "," emits then clears *bib-key*.
;;             if "}", same, but also goes to :WAIT.
;;             all other chars are added to *bib-key*.

(defvar *cite-state* nil) ;; internal state flag
(defvar *cite-key* "")
(defvar *cite-table* nil)

(defun glom-cite (input)
  (cond ((characterp input)
	 (case *cite-state*
	   (:WAIT
	     (case input
	       (#\\ (setq *cite-state* :SNIFF
			 *cite-table* '( #\c #\i #\t #\e #\{)))))
	   (:SNIFF
	     (cond ((char= input (car *cite-table*))
		    (setq *cite-table* (cdr *cite-table*))
		    (unless *cite-table* (setq *cite-state* :READ-ITEMS)))
		   (t (setq *cite-state* :WAIT))))
	   (:READ-ITEMS
	     (case input
	       (#\} (add-cite *cite-key*)
		(setq *cite-key* "" *cite-state* :WAIT))
	       (#\, (add-cite *cite-key*)
		(setq *cite-key* ""))
	       (t (setq *cite-key* (format nil "~A~A" *cite-key* input)))))))
	(t (setq *cite-key* "" *cite-state* :WAIT)))
;  (format t "~% Cite state = ~A" *cite-state*)
  )

;;;; Recognizer for bibitem
;; Same as cite, just no comma option

;; States:
;; :WAIT -- looks for "\", transitions to :SNIFF.
;; :SNIFF -- looks for "cite{", if not found, back to :WAIT.
;;           otherwise, goes to :READ-ITEMS.
;; :READ-ITEMS -- 
;;             if "}", emits key and goes to :WAIT.
;;             all other chars are added to *bib-key*.

(defvar *bib-state* nil) ;; internal state flag
(defvar *bib-key* "")
(defvar *bib-table* nil)

(defun glom-bibitem (input)
  (cond ((characterp input)
	 (case *bib-state*
	   (:WAIT
	     (case input
	       (#\\ (setq *bib-state* :SNIFF
			 *bib-table* '( #\b #\i #\b #\i #\t #\e #\m #\{)))))
	   (:SNIFF
	     (cond ((char= input (car *bib-table*))
		    (setq *bib-table* (cdr *bib-table*))
		    (unless *bib-table* (setq *bib-state* :READ-ITEMS)))
		   (t (setq *bib-state* :WAIT))))
	   (:READ-ITEMS
	     (case input
	       (#\} (add-bibitem *bib-key*)
		(setq *bib-key* "" *bib-state* :WAIT))
	       (t (setq *bib-key* (format nil "~A~A" *bib-key* input)))))))
	(t (setq *bib-key* "" *bib-state* :WAIT)))
;  (format t "~% Bibitem state = ~A" *bib-state*)
  )


