;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Agnar Aamodt and Daniel L. Dvorak.

(in-package 'protos)



;;;=============================================================================
;;;
;;;                        T R A N S F O R M A T I O N S
;;;  ---------------------------------------------------------------------------
;;;
;;;  Overview:  This file contains functions for transforming features.
;;;=============================================================================



;;;----------------------------------------------------------------------------
;;;  Function: (check-and-transform-feature (fname flist))
;;;
;;;  Purpose:  To check whether a transformation function is defined for this
;;;            numerical feature, and if so, transform the feature into a
;;;            symbolic one.
;;;
;;;  Design:   If a transformation is defined, the type is checked, and the
;;             lisp-function is called. See description of transformation
;;;            mechanism within Protos under the 'transformation' structure.
;;;----------------------------------------------------------------------------

(defun check-and-transform-feature (fname flist)
 (let (term) 

  ;; The feature has numeric argument(s), an attempt is made to transform the
  ;; numerical (quantitative) feature-values into symbolic (qualitative) values 
  ;; before the feature is pushed onto the feature list. First, we check
  ;; whether a transformation is defined for this predicate.
  (cond ((or (not (boundp (car fname))) (null (car fname)))
	 (set (car fname) (make-predicate :name (car fname)))))
  (cond ((null (predicate-input-xforms (eval (car fname))))
	 
	 ;; No transform exists, the feature is kept with its numeric argument,
	 ;; e.g. (number-of-wheels 4)
	 (setq term (check-term-name fname 'ask))
	 (if term (pushnew term flist))
	 (if *trace-transformations*
	     (format t "~%No transformation of numerical feature ~A." fname)))
	
	;; We have a transformation.
	;; The numeric input feature may be involved in several transformations
	;; in a forward-propagating way. A numeric feature may be input to several
	;; transformations, in this case the input-xforms slot of the predicate
	;; has more than one element. This loop handles all possible transforms in
	;; a 'spreading activation' manner:

	(t (do* ((xforms-list (predicate-input-xforms (eval (car fname))) (cdr xforms-list))
		 (xform (eval (car xforms-list)))
		 (xformed-feature nil))
		((endp xforms-list))
	     
	     ;; Do we have a quantitative-to-qualitative transformation?
	     (cond ((equal (transformation-type xform) 'qq-transformation)
		 (prog ()   
		   retry    
		    (setq xformed-feature (qq-transform-feature fname xform))
		    (cond (xformed-feature
			   (pushnew (check-term-name xformed-feature 'ask) flist)
			   
			   ;; Save numerical input feature in "xformed-from" slot of the
			   ;; transformed feature. 
			   ;; Remember that if a feature is transformed, no
			   ;; feature structure is created for the numerical feature.
			   (push fname (feature-xformed-from
				   (cdar (predicate-args (eval (car xformed-feature))))))

			   ;; Remove the predicate from the list of uninstantiated predicates,
                           ;; if it is there.
			   (setq *uninstantiated-predicates*
				 (remove (eval (car xformed-feature)) *uninstantiated-predicates*)))
			  (t (format t "~2% An error results from trying to transform ~A.~
                                         ~% You probably have given a wrong numer of arguments, ~
                                         ~% or there is an error in the transformation function."
				     fname)
			     (format t "~% Please try to input the feature again:~
                                        ~% >")
			     (force-output *query-io*)
			     (setq fname (read *query-io* nil nil nil))
			     (go retry)))
		    (if *trace-transformations*
			(format t "~%Feature ~A transformed to ~A" fname xformed-feature))))
		   
		   ;; OK, so we have a numeric computation?
		   ((equal (transformation-type xform) 'computation)
		    (setq xformed-feature (compute-feature fname xform))
		    (cond (xformed-feature
			   (if *trace-transformations*
			       (format t "~%Features ~A ~%  transformed to ~A."
				       (transformation-in-features xform) xformed-feature))
			   
			   ;; The computation has most likely changed the name of the feature
			   ;; i.e the predicate. If this new predicate has some numerical
			   ;; argument, check to see if it can be transformed, by recursively
			   ;; calling this function:

			   (cond ((and (not (equal (car fname) (car xformed-feature)))
				     (dolist (value (cdr xformed-feature) nil)
				       (cond ((numberp value) (return t)))))
				  (setf flist (append (check-and-transform-feature xformed-feature nil)
						      flist)))
				(t (pushnew (check-term-name xformed-feature 'ask) flist)
				
				   ;; Updates the computed feature structure's xformed-from slot, by saving
				   ;; the list of input features involved in the computation
				   
				   (setf (feature-xformed-from (cdar (predicate-args (eval (car xformed-feature)))))
					 (transformation-in-features xform))))
			   

			   ;; Resets the transformation structure for this predicate, making it ready for
			   ;; successive transformations.
			   (setf (transformation-in-features xform) nil))))
		   
		   
		   ;; Hopefully, we never end up here
		   (t (format t "~2% ERROR. Exit from CHECK-AND-TRANSFORM-FEATURE. ~
                               ~%Wrong or missing TYPE slot on transformation structure~
                               for out-predicate ~A" (car fname))
		      (error nil))))))
  flist))
		   
		   
  
;;;----------------------------------------------------------------------------
;;;  Function:  qq-transform-feature (input-feature)
;;;
;;;  Purpose:   To take a feature of the form
;;;             (<predicate-1> [<numerical-value>] {[<numeric-or-symbolic-value>]}) and return
;;;             (<predicate-2> [<symbolic-value>] {[<numeric-or-symbolic-value>]}),
;;;             i.e transform all transformable quantitative (numeric) values in a feature 
;;;             into qualitative (symbolic) values. The feature name may change, too.
;;;
;;;  Input-arg:    A feature as a predicate with at least one numerical value.
;;;
;;;  Returns:      The transformed feature
;;;
;;;  Side-effects: Adds the numerical value to the feature-structure's xformed-from
;;;                slot.
;;;                
;;;                Updates the transformation structure, i.e the slots:
;;;                in-features and out-feature.
;;;
;;;   The lisp-function on the transform structure defines the entire transformation
;;;   operation. When defining a transformation, the expert is recommended to write
;;;   the definitions in a pattern similar to all transformation. This ensures that
;;;   only the numerical arguments are treated, the symbolic values in the input
;;;   feature are just  returned as given (see guiding text in the defun-transformation
;;;   function)
;;;
;;;----------------------------------------------------------------------------

(defun qq-transform-feature (feature xform)
  (push feature (transformation-in-features xform))
  (substitute (transformation-out-predicate xform)
	      (car feature)
	      (funcall (transformation-lisp-function xform) feature)))



;;;----------------------------------------------------------------------------
;;;  Function:  compute-feature (feature xform)
;;;
;;;  Purpose:   To compute a numerical feature on the basis of other
;;;             numeric features.
;;;
;;;  Input-args: A list describing the feature a predicate name with arguments.
;;;              The transformation structure defining the computation.
;;;
;;;  Returns:    If the feature given as input argument completes the features
;;;              necessary for performing the computation, the computed feature
;;;              is returned.
;;;              Otherwise, the feature is saved (on the transformation structure)
;;;              and NIL is returned.
;;;
;;;  Procedure:  1. The input feature is appended to the 'input-features' slot of
;;;                 the transformation structure.
;;;              2. If all the features necessary for performing the computation
;;;                 have values, the computation is done. The test is done by
;;;                 comparing the number of features in the 'input-features' slot
;;;                 of the transfromation with the number of elements in the 'in-
;;;                 predicates' slot. The latter contains the neames of all the
;;;                 predicates given as input to the computation, and is created
;;;                 when the lisp-function is defined.
;;;              3. If not all features present, no computation is possible and
;;;                 there's nothing else to do but to wait for the remaining
;;;                 feature values to appear. NIL is returned.
;;;
;;;----------------------------------------------------------------------------

(defun compute-feature (feature xform)
  (push feature (transformation-in-features xform))
  (cond ((= (length (transformation-in-features xform))
	    (length (transformation-in-predicates xform)))
	 (funcall (transformation-lisp-function xform)
		  (transformation-in-features xform)))))



;;;----------------------------------------------------------------------------
;;;  Function:  (define-transformation  stream)
;;;
;;;  Purpose:   Aids the user in defining a transformation for a feature.
;;;
;;;  Note:      This code written by Agnar Aamodt; modified somewhat by
;;;             Dan Dvorak.
;;;----------------------------------------------------------------------------

(defparameter *transform-menu* (make-menu
      :label  "~%Please select the type of transformation:"
      :items  '((#\Q . ("Quantitative-to-qualitative"   return  'qq-transformation))
		(#\C . ("numeric Computation"           return  'computation)))))

(defun define-transformation (stream)
  (let (xform-type defined-preds in-pred-names out-pred-name
	xform-text xform-name    file-name     file-stream)

    ;; Ask teacher for type of transformation.
    (setq xform-type (menu-select *transform-menu*))
    
    (format stream "~%Give the feature-name (the predicate) which will be the output from~
                    ~%the transformation~
                    ~2%>")
    (setf out-pred-name (read *query-io* nil nil nil))
    (setf xform-name (format nil "TRANSFORM-~A" (string out-pred-name)))
    ;;(setf xform-name (read-from-string (string-append "TRANSFORM-" (string out-pred-name))))    
    (format stream "~%Give the feature-name(s)  which will be given as input to~
                    ~%the transformation (wrapped within a list)~
                    ~2%>")
    (setf in-pred-names (read *query-io* nil nil nil))
    (format stream "~2%Now, give an English description of the transformation you are about to define:~
                    ~2%>")
    (setf xform-text (read *query-io* nil nil nil))

    (if (y-or-n-p "~2%You are now supposed to define the lisp-expression for the transformation.~
                    ~2%Next, you will be given a buffer of your favorite editor.~
                    ~%You have to complete the definition.~
                    ~%A pattern of a transformation, with a simple example,~
                    ~%is given at the top of the screen.~
                    ~2%Would you like to define the transformation now?~
                    ~%(If NO, this transformation will be deleted)
                    ~%> ")
	(progn
	  (setf file-stream (open "XDEF.lisp" :direction :output :if-exists :new-version
				  :if-does-not-exist :create))
	  (print-xform-info-on-file file-stream)
	  (format file-stream "~2%(setf ~A ~
                  ~%   (make-transformation :name '~A~
                  ~%                        :type '~A~
                  ~%                        :out-predicate '~A~
                  ~%                        :in-predicates '~A~
                  ~%                        :description ~S~
                  ~%                        :lisp-function~
                  ~%                         '(lambda (f) )))"
		     xform-name xform-name xform-type out-pred-name in-pred-names xform-text)
	  (close file-stream)
	  (ed file-name)
	  (load file-name)
	  (pushnew (eval xform-name) *transformations*)
	  (cond ((not (member out-pred-name defined-preds))
		 (set out-pred-name (make-predicate :name out-pred-name
						    :output-xform xform-name))
		 (push out-pred-name *history*)
		 (push out-pred-name defined-preds)
		 (pushnew (eval out-pred-name) *uninstantiated-predicates*))
		(t (setf (predicate-output-xform (eval out-pred-name)) xform-name)))
	  (dolist (in-pred in-pred-names)
	    (cond ((not (member in-pred defined-preds))
		   (set in-pred (make-predicate :name in-pred
						:input-xforms (list in-pred)))
		   (push in-pred *history*)
		   (push in-pred defined-preds)
		   (pushnew (eval in-pred) *uninstantiated-predicates*))
		  (t (push xform-name (predicate-input-xforms (eval in-pred))))))
	  (format stream "~%The transformation ~A is now a part of the knowledge base.~%" xform-name)
	  (delete-file file-name)))))



(defun print-xform-info-on-file (file)
  (format file
  "~%;                    DEFINING A TRANSFORMATION WITHIN PROTOS~
   ~%; -----------------------------------------------------------------------~
   ~%;~
   ~%;    Please complete the LAMBDA function in the LISP-FUNCTION slot~
   ~%;    of the transformation structure.~
   ~%;~
   ~%;    Any valid lambda expression will do, but in order to achieve~
   ~%;    some similarity in the definitions, variations of the following~
   ~%;    patterns are recommended:~
   ~%;~
   ~%;       QQ-TRANSFORMATION.~
   ~%;~
   ~%;           E.g:  f = (seat-height 1.5) --> (seat-height 'medium)~
   ~%;~
   ~%;           (lambda (f)~
   ~%;             (let ((h (nth 1 f)))~
   ~%;               (if ((numberp h))~
   ~%;  		       (cond ((< h 1) (substitute 'small h f))~
   ~%;  		    ((and (>= h 1) (< h 2)) (substitute 'medium h f))~
   ~%;  		    ((> h 2) (substitute 'large h f))))))~
   ~%;~
   ~%;         E.g:  f = (seat-height 1.5) --> (seat-height 'medium)~
   ~%;~
   ~%;         By using the general construct (NTH n F) you may access feature~
   ~%;         value number n in any feature-vector, as in transforming~
   ~%;         (weather sunny 68 dry 30) --> (weather sunny mild dry strong)~
   ~%;~
   ~%;~
   ~%;       COMPUTATION:~
   ~%;~
   ~%;       e.g:  f = ((seat-width 2) (seat-depth 1.25))  --> (seat-area 2.50)~
   ~%;~
   ~%;           (lambda (f)~
   ~%;             (let ((sw (nth 1 (assoc 'seat-width f)))~
   ~%;                   (sd (nth 1 (assoc 'seat-depth f))))~
   ~%;               (if ((and (numberp sw) (numberp sd))~
   ~%;                   (list 'seat-area (* sw sd))))~
   ~%;~
   ~%;~
   ~%;    --------------------------------------------------------------------~
   ~%;    Before exiting the editor, remember to:~
   ~%;     - CLOSE THE ENTIRE SETF EXPRESSION WITH THE APPROPRIATE NUMBER OF~
   ~%;       RIGHT-PARENTHESES~
   ~%;     - SAVE THE DEFINITION TO THE CURRENT FILE~
   ~%;    --------------------------------------------------------------------"))

