;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Hilel Swerdlin

(in-package "PROTOS")




;;;=============================================================================
;;;
;;;                      E X E M P L A R    M E R G I N G
;;;   ------------------------------------------------------------------------
;;;
;;;  Overview:	 This file contains the functions for testing to see if a
;;;              case can be merged with an exemplar and for actually merging
;;;              the two.
;;;
;;;  Functions:	 mergeable
;;;              merge-together
;;;=============================================================================



(defparameter *high-bound-importance* .75)  
                                      ; a feature with an importance value
                                      ; higher than this value is considered
                                      ; to be highly important.

(defparameter *low-bound-importance* .26) 
                                       ; a feature with an importance value
                                       ; lower than this value is considered 
                                       ; to be not important.

(defparameter *match-ratio-threshold* 2/3)
                                       ; lower bound for merging.
             


;;;----------------------------------------------------------------------------
;;; Function:  find-matches-and-non-matches
;;;
;;; Given:     The list of results of a match.
;;;
;;; Returns:   Two values: the number of matched features, and the number of
;;;            unmatched features.  Note that "spurious" and "excluded"
;;;            features are ignored.
;;; 
;;; Callers:   merge-majority-test, merge-match-ratio-test
;;;----------------------------------------------------------------------------

(defun find-matches-and-non-matches (results)
  (let ((matched    0)
	(unmatched  0))
    (dolist (result results)
      (case (result-type result)
	(identical    (incf matched))
	(explained    (incf matched))
	(unmatched    (incf unmatched))
	(spurious     nil)
	(excluded     nil)
	(otherwise    (format t "~%Unknown result type ~A !!" (result-type result)))))
    (values matched unmatched)))


;;;----------------------------------------------------------------------------
;;; Function:   merge-majority-test
;;; 
;;; Given:      The list of results of a match.
;;;
;;; Returns:    True if the majority of the features are matched,
;;;             otherwise nil (false).
;;;
;;; Callers:    merge-test-1
;;;----------------------------------------------------------------------------

(defun merge-majority-test (results)
  (multiple-value-bind (matched unmatched)
      (find-matches-and-non-matches results)
    (> matched unmatched)))



;;;-----------------------------------------------------------------------------
; Function:  merge-match-importance-test
;
; Given:     The results list of a match structure
;
; Returns:   True if all the important features matched (an important feature 
;            is one which has an importance above or equal to the high-bound-
;            importance parameter, nil (false) otherwise.
;
; Callers:   merge-test-1
;;;-----------------------------------------------------------------------------

(defun merge-match-importance-test (results)
  (dolist (result results)
    (if (>= (result-importance result) *high-bound-importance*)
	(if (member (result-type result) '(unmatched excluded))
	    (return-from merge-match-importance-test nil))))
  t)


;;;-----------------------------------------------------------------------------
; Function:   merge-match-ratio-test
;
; Given:      The results list of a match structure.
;
; Returns:    True if the ratio of matched (explained or identical) over the 
;             total number of non-spurious features, is greater or equal to the
;             *match-ratio-threshold* parameter; it returns nil otherwise.
;
; Callers:    merge-test-2
;;;-----------------------------------------------------------------------------

(defun merge-match-ratio-test (results)
  (multiple-value-bind (matched unmatched)
      (find-matches-and-non-matches results)
    (if (zerop unmatched) (setq unmatched 1))
    (>= (/ matched unmatched) *match-ratio-threshold*)))



;;;-----------------------------------------------------------------------------
; Function:   merge-non-match-importance-test
;
; Given:      The results list of a match structure
;
; Returns:    True if all the unmatched features had importance values less 
;             than the *low-bound-imporance* parameter; it returns nil otherwise.
;
; Callers:    merge-test-2
;;;-----------------------------------------------------------------------------

(defun merge-non-match-importance-test (results)
  (dolist (result results)
    (if (member (result-type result) '(unmatched excluded))
	(if (>= (result-importance result) *low-bound-importance*)
	    (return-from merge-non-match-importance-test nil))))
  t)


;;;-----------------------------------------------------------------------------
;;; Function:  mergeable
;;;
;;; Given:     The results list of a match-structure
;;;
;;; Returns:   True if either of the following two tests are true:
;;;            -- Test1.- If the number of matches is strictly greater than the;;;               number of non-matches, and if all the important features were
;;;               matched. NOTE:  A feature with an importance value greater than 
;;;               the *high-bound-importance* parameter is considered to be  
;;;               important; A match can be identical or explained; Finally, 
;;;               spurious features are ignored.
;;;            -- Test2.- If there is a match ratio above or equal to the 
;;;               *match-ratio-threshold*; and all the unmatched features have
;;;               importance values smaller than the *low-bound-importance* parameter.
;;;               NOTE: Again, spurious features are ignored
;;;
;;;            Nil otherwise.
;;;
;;; Callers:   add-new-exemplar
;;;-----------------------------------------------------------------------------

(defun mergeable (match)
  (or (merge-test-1 match) (merge-test-2 match)))

(defun merge-test-1 (match)
  (let ((results  (match-results match)))
    (and (merge-majority-test results)
	 (merge-match-importance-test results))))

(defun merge-test-2 (match)
  (let ((results  (match-results match)))
    (and (merge-match-ratio-test results) 
	 (merge-non-match-importance-test results))))



;;;-----------------------------------------------------------------------------
;;;  Function:   (merge-exemplar  match)
;;;
;;;  Purpose:    This function is called when Protos has decided that the new
;;;              case is mergeable with its matching exemplar (and the teacher
;;;              has agreed to the merge).  This function replaces an exemplar
;;;              feature with a more abstract feature when the exemplar and
;;;              new case share a common generalization, function, or cause.
;;;
;;;  Called by:  discuss-success
;;;
;;;  Note:       THIS FUNCTION CURRENTLY DOES NOTHING EXCEPT CALL THE FUNCTION
;;;              TO DISCUSS NEAR MISSES.
;;;              Ray now feels that the abstraction of exemplar features is
;;;              unwise since (a) it causes the exemplar to have "non operational"
;;;              features, and (b) it doesn't improve Protos' ability to match
;;;              future cases to the exemplar, anyway.
;;;-----------------------------------------------------------------------------

(defun merge-exemplar (match)
  (discuss-near-misses (match-exemplar match)))




;;;  ALL OF THE FOLLOWING CODE & COMMENTS WAS WRITTEN BY HILEL AS SUBORDINATE
;;;  FUNCTIONS FOR MERGING AN EXEMPLAR.  THIS WAS NEVER DEBUGGED AND TESTED
;;;  BECAUSE OF RAY'S POSITION EXPRESSED IN THE ABOVE NOTE.


;;;-----------------------------------------------------------------------------
; Function:    find-common-ancestor
;
; Given:       An explanation list.
;
; Returns:     A feature which a common ancestor of the two features linked by
;              the given explanation list.  It does so by traversing the
;              explanation list until it finds two adjacent inverse relations,
;              and by returning the second argument of the leftmost relation
;
; Callers:     generalize-feature
;;;-----------------------------------------------------------------------------

;(defun find-common-ancestor (expl)
;  (if (null expl)
;      (return-from find-common-ancestor nil))   ;;; SIGNAL ERROR?  Why?
;  
;  (let* ((rel1  (explanation-relation expl))
;	 (verb1 (relation-verb rel1))
;	 (to-nodes (relation-to-nodes rel1)))
;    (if (and (= (length to-nodes) 1)
;	     (typep (car to-nodes) 'explanation))
;	(let* ((expl2 (car to-nodes))
;	       (rel2  (explanation-relation expl2))
;	       (verb2 (relation-verb rel2)))
;	  (if (eq verb1 (verb-inverse verb2))
;	      (explanation-start-term expl2)
;	      (find-common-ancestor expl2))))))


;;;-----------------------------------------------------------------------------
; Function:    find-most-general
;
; Given:       A unidirectional  explanation list - by unidirectional I mean a
;              list which goes from general to specific or vice versa; this 
;              assumption is valid since this function is mutually exclusively
;              called after the find-common-ancestor function ,which takes
;              care of bidirectional explanations (explanations with inverses).
;
; Returns:     The most general or common feature of the given explanation.
;              Namely, one of the arguments (features) of the first or last 
;              entry of the explanation list depending on which relation is
;              the most general.   For this purpose I am assuming that the 
;              explanation list contains unidirectional relations (i.e, 
;              relations that go from general to specific or viceversa).  
;              With this assumption in mind, it is easy to see why this 
;              function returns the first or last entry of the explanation
;              list.  Finally, in order to decide what feature to return of
;              a given relation,  I noticed that the relations used could
;              be partitioned into two groups: One group (arg-1-relations),
;              has the peculiarity that its first argument is always more
;              general or common to the second; thus if we encounter one of
;              the functions belonging to this group at the beginning of the 
;              explanation list, we can deduce two things:
;              -- 1.-  The directionality of the overall explanation is from
;                      left to right general to specific.
;              -- 2.-  Thus, the most general argument is the first argument
;                      of the first relation.
;              Similarily, the second group of relations (arg-2-relations),
;              has the peculiarity that its second argument is always more
;              general than the first one; thus, if we found such a relation
;              at the beginning of the explanation list, we could again deduce
;              two things:
;              -- 1.-  The directionality of the overall explanation is from
;                      left to right specific to general.
;              -- 2.-  Thus, the most general argument is th second argument
;                      of the second relation.
;              Using the above observations, this function returns the most 
;              common or general feature of an explanation path that does not 
;              have inverse relations (this is from where the unidirectionality
;              of the explanation comes from).
;;;-----------------------------------------------------------------------------

;(defconstant *arg-1-relations* 
;  '((is-function-of) (causes) (has-typical-specialization)
;    (is-enabled-by)))
;
;(defconstant *arg-2-relations*
;  '((has-function) (is-caused-by) (has-typical-generalization)
;    (enables)))
;
;(defun find-most-general (explanation)
;  (let ((rel1 (first explanation)) (rel2 (car(last explanation))) )
;    (cond
;      ((member (list(second rel1)) *arg-1-relations* :test #'equalp)
;       (first rel1))
;      ((member (list(second rel1)) *arg-2-relations* :test #'equalp)
;       (third rel2))
;      (T nil) )))


;;;-----------------------------------------------------------------------------
; Function:     generalize-feature
;
; Given:        An entry of the match structure of the form:
;               (feature importance 'explained explanation)
;
; Returns:
;
; Side effects: It modifies the feature value with the expert's supervision
;
; Callers:      merge-exemplar
;;;-----------------------------------------------------------------------------

;(defun generalize-feature (feat)
;  (let ((expl (fourth feat))
;	(common-ancestor (find-common-ancestor (expl)))
;	(most-general-node (find-most-general (expl)))
;	() )
;    (cond
;      ((common-ancestor) (ask-expert-if-merge-feature common-ancestor))
;      ((most-general-node) (ask-expert-if-merge-feature most-general))
;      (T (ask-expert-if-merge-feature nil)) );;PENDING  
;    ))



