;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak and Agnar Aamodt.

(in-package 'protos)


;;;===============================================================================
;;;
;;;                E X A M P L E    K N O W L E D G E    B A S E
;;;   -------------------------------------------------------------------------
;;;
;;;  Overview:	This file contains the function load-example which creates an
;;;		example category network with remindings.  This example was
;;;		published in "Protos: An Exemplar-Based Learning Apprentice",
;;;		E. Ray Bareiss, Bruce W. Porter, and Craig C. Wier, Proceedings
;;;		of the Fourth International Machine Learning Workshop (held at
;;;		UC-Irvine, 1987).  Morgan-Kaufman Publishers.
;;;
;;;		The example contains the single category "chairs", which has
;;;		exemplars "chair1" and "chair2", and observed features
;;;		"armrests", "pedestal", "seat", "backrest", "legs-4".
;;;		The function load-example directly uses the primitive 
;;;		structure-making functions to build the category network;
;;;		it does not exercise any of the Protos knowledge acquisition
;;;		procedures.
;;;
;;;		This example category network serves three purposes:
;;;		-- it aided in the early testing and debugging of CL-Protos;
;;;		-- it is a built-in example of a category network, useful for
;;;		   demonstrating CL-Protos;
;;;		-- it illustrates how the category network is represented in
;;;		   CL-Protos.
;;;
;;;  Design:    The category network is a tightly-connected network of nodes,
;;;             so the load-example function has to create things in just the
;;;             right order lest it refer to something that hasn't been created
;;;             yet (such as a node or a relation).  The following sequence of
;;;             creation is guaranteed to work for all category networks (it is
;;;             one of several valid sequences):
;;;
;;;             1.  Make all term nodes, but don't set its relations, remindings
;;;                 or exemplars.
;;;             2.  Make all exemplar nodes, setting its features, category,
;;;                 and typicality, but not its difference links.
;;;             3.  Install list of exemplars in each category.
;;;             4.  Make each difference link and install it in its associated
;;;                 exemplar.
;;;             5.  Make each reminding and install it in its associated term.
;;;             6.  Make each relation (and any condition it has) and install
;;;                 it in its associated term or exemplar.
;;;
;;;  Function:	load-example
;;;=============================================================================



(defun load-example ()
  (declare (special  wheels  armrests  metal  pedestal  seat  backrest  legs
	             legs-4  wood  lateral-support  person holds  holds-person
	             seat-support  rigid-material  chairs  chair1  chair2))
  (declare (special  seat-height  transform-seat-height  object-size
		     transform-object-size  object-floor-space
		     transform-object-floor-space  object-space
		     transform-object-space object-width object-depth))

  ;; First, check to see if there is an existing knowledge base.
  (if *history*
      (if (not (prompt "~%There is an existing knowledge base.~
                        ~%Do you wish to overwrite it? "
		       nil 'y-or-n nil nil))
	  (progn
	    (format t "~%Then this load operation is cancelled.~%")
	    (return-from load-example (values)))))

  ;; Forget the current knowledge base.
  (forget-kb t)


  ;;--------------------------------------------------------------------------
  ;; Make term nodes that are predicates-with-arguments.
  ;;--------------------------------------------------------------------------
  (setq legs   (make-predicate :name 'legs))
  (push legs *history*)
  (setq legs-4 (make-term :name      '(legs 4)
			  :predicate legs
			  :creator   'dvorak))
  (push (cons 4 legs-4) (predicate-args legs))
  (push legs-4 *history*)

  (setq holds (make-predicate :name 'holds))
  (push holds *history*)
  (setq holds-person (make-term :name      '(holds person)
				:predicate holds
				:creator   'dvorak))
  (push (cons 'person holds-person) (predicate-args holds))
  (push holds-person *history*)


  ;;--------------------------------------------------------------------------
  ;; Code for testing transformation mechanism.  Note: all of the code from
  ;; here until the code for making term nodes deals strictly with the
  ;; numeric-to-qualitative transformation mechanism (and may be safely
  ;; ignored if you're not interested in that aspect).
  ;;--------------------------------------------------------------------------
  
  (setf transform-seat-height
	(make-transformation :name 'transform-seat-height
			     :type 'qq-transformation
			     :in-predicates '(seat-height)
			     :out-predicate 'seat-height
			     :description
			     "A seat height smaller than 12 inches is low,
                              from 12 to 24 inches is medium, and above
                              24 inches is high."
			     :lisp-function
			     '(lambda (f)
				(if (numberp (nth 1 f))
				    (cond ((< (nth 1 f) 12) (substitute 'low (nth 1 f) f))
					  ((> (nth 1 f) 24) (substitute 'high (nth 1 f) f))
					  (t (substitute 'medium (nth 1 f) f)))))))
			     
  (setq seat-height
	(make-predicate :name 'seat-height
			:input-xforms '(transform-seat-height)
			:output-xform '(transform-seat-height)))

  (push transform-seat-height *transformations*)
  (push seat-height *uninstantiated-predicates*)

  (setf transform-object-size
	(make-transformation :name 'transform-object-size
			     :type 'qq-transformation
			     :in-predicates '(object-size)
			     :out-predicate 'object-size
			     :description
			     "A object's size is measured in height (inches)
                              and floor space occupied, i.e horizontally projected
                              surface (square feet)."
			     :lisp-function
			     '(lambda (f)
				(if (and (numberp (nth 1 f)) (numberp (nth 2 f)))
                                    (cond ((and (< (nth 1 f) 24) (< (nth 2 f) 5))
					   (substitute 'small (nth 1 f) (remove (nth 2 f) f)))
					  ((and (> (nth 1 f) 48) (> (nth 2 f) 12))
					   (substitute 'large (nth 1 f) (remove (nth 2 f) f)))
					  (t (substitute 'medium (nth 1 f) (remove (nth 2 f) f))))))))
  (setq object-size
	(make-predicate :name 'object-size
			:input-xforms '(transform-object-size)
			:output-xform '(transform-object-size)))

  (push transform-object-size *transformations*)
  (push object-size *uninstantiated-predicates*)
  
  (setf transform-object-floor-space
	(make-transformation :name 'transform-object-floor-space
			     :type 'computation
			     :in-predicates '(object-width object-depth)
			     :out-predicate 'object-floor-space
			     :description
			     "The floor space occupied by a object, i.e. the size of a projection
                              of the object along a vertical axis onto the floor."
			     :lisp-function
			     '(lambda (flist)
				(list 'object-floor-space (* (cadr (assoc 'object-width flist))
							    (cadr (assoc 'object-depth flist)))))))

  (setf transform-object-space
	(make-transformation :name 'transform-object-space
			     :type 'qq-transformation
			     :in-predicates '(object-floor-space)
			     :out-predicate 'object-space
			     :lisp-function
			     '(lambda (f)
				(if (numberp (nth 1 f))
				    (cond ((< (nth 1 f) 6) (substitute 'small (nth 1 f) f))
					  ((> (nth 1 f) 20) (substitute 'large (nth 1 f) f))
					  (t (substitute 'medium (nth 1 f) f)))))))


  
  (setq object-floor-space
	(make-predicate :name 'object-floor-space
			:output-xform '(transform-object-floor-space)
			:input-xforms '(transform-object-space)))

  (push object-floor-space *uninstantiated-predicates*)

  (setq object-space
	(make-predicate :name 'object-space
			:output-xform '(transform-object-space)
			:input-xforms '(transform-object-floor-space)))

  (push object-space *uninstantiated-predicates*)

  (push transform-object-floor-space *transformations*)
  (push transform-object-space *transformations*)
  
  (setq object-width
	(make-predicate :name 'object-width
			:input-xforms '(transform-object-floor-space)))
  (push object-width *uninstantiated-predicates*)

  (setq object-depth
       (make-predicate :name 'object-depth
			:input-xforms '(transform-object-floor-space)))
  (push object-depth *uninstantiated-predicates*)


  ;;--------------------------------------------------------------------------
  ;; Make term nodes that are propositions.
  ;;--------------------------------------------------------------------------
  (let ((term-names  '(wheels armrests metal pedestal seat backrest wood
			lateral-support seat-support rigid-material chairs
			person)))
    (dolist (term term-names)
      (let ((pointer (make-term :name    term
				:creator 'dvorak)))
	(set term pointer)
	(push pointer *history*))))

  (setq *number-of-ec-categories* 2)

  ;;--------------------------------------------------------------------------
  ;; Make exemplar nodes.
  ;;--------------------------------------------------------------------------
  (setq chair1 (make-term :name 'chair1
			  :features   (list wheels armrests metal pedestal
					    seat backrest)
			  :category   chairs
			  :typicality 0.8
			  :creator    'dvorak))

  (push chair1 (feature-of-exemplars wheels))
  (push chair1 (feature-of-exemplars armrests))
  (push chair1 (feature-of-exemplars metal))
  (push chair1 (feature-of-exemplars pedestal))
  (push chair1 (feature-of-exemplars seat))
  (push chair1 (feature-of-exemplars backrest))

  (setq chair2 (make-term :name 'chair2
			  :features   (list seat backrest legs-4 wood)
			  :category   chairs
			  :typicality 0.4
			  :creator    'dvorak))

  (push chair2 (feature-of-exemplars seat))
  (push chair2 (feature-of-exemplars backrest))
  (push chair2 (feature-of-exemplars legs-4))
  (push chair2 (feature-of-exemplars wood))

  (setq *history* (nconc (list chair1 chair2) *history*))

  ;;--------------------------------------------------------------------------
  ;; Install list of exemplars in each category.
  ;;--------------------------------------------------------------------------
  (setf (category-exemplars chairs) (list chair1 chair2))


  ;;--------------------------------------------------------------------------
  ;; Install difference links in the exemplars.
  ;;--------------------------------------------------------------------------
  (let ((diff1 (make-difference :node chair2
				:features (list legs-4))))
    (setf (exemplar-differences chair1) (list diff1)))
  (let ((diff1 (make-difference :node chair1
				:features (list armrests pedestal))))
    (setf (exemplar-differences chair2) (list diff1)))


  ;;--------------------------------------------------------------------------
  ;; Install remindings in the feature nodes.
  ;;--------------------------------------------------------------------------
  (setf (feature-remindings armrests) (list (cons chair1 0.4)))
  (setf (feature-remindings pedestal) (list (cons chair1 0.8)))
  (setf (feature-remindings seat    ) (list (cons chairs 0.9)))
  (setf (feature-remindings backrest) (list (cons chairs 0.5)))
  (setf (feature-remindings legs-4  ) (list (cons chair2 0.7)))


  ;;--------------------------------------------------------------------------
  ;; Install importances in the category nodes.
  ;;--------------------------------------------------------------------------
  (setf (category-importances chairs) (list (cons seat 0.7)
					    (cons backrest 0.6)
					    (cons armrests 0.7)
					    (cons pedestal 0.3)
					    (cons legs-4 0.3)
					    (cons metal 0.2)
					    (cons wood 0.2)
					    (cons wheels 0.0)))

  ;;--------------------------------------------------------------------------
  ;; Install relational links in their respective nodes.
  ;;--------------------------------------------------------------------------
  
  (let (rel1 rel2 rel3 rel4 rel5 rel6 rel7 rel8 rel9 rel10
	rel11 rel12 rel13 rel14 rel15 rel16 rel17 rel18 rel19 rel20
	rel21 rel22 rel23 rel24 rel25)
    
    (setq  rel1  (make-relation	 :verb *verb-spurious*
				 :strength 1.0
				 :from-nodes (list wheels)
				 :to-nodes nil))
    (setf (node-relations wheels) (list rel1))
    
    (setq  rel2  (make-relation	 :verb *verb-hasFunction*
				 :strength 1.0
				 :from-nodes (list armrests)
				 :to-nodes (list lateral-support)))
    (setf (node-relations armrests) (list rel2))
    
    (setq  rel3  (make-relation	 :verb *verb-partOf*
				 :strength 0.7
				 :from-nodes (list metal)
				 :to-nodes (list chairs)))
    (setq  rel4  (make-relation  :verb *verb-hasTypicalGen*
				 :strength 0.9
				 :from-nodes (list metal)
				 :to-nodes (list rigid-material)))
    (setf (node-relations metal) (list rel3 rel4))
    
    (setq  rel5  (make-relation  :verb *verb-hasTypicalGen*
				 :strength 0.9
				 :from-nodes (list pedestal)
				 :to-nodes (list seat-support)))
    (setf (node-relations pedestal) (list rel5))
    
    (setq  rel6  (make-relation  :verb *verb-enables*
				 :strength 0.9
				 :from-nodes (list seat)
				 :to-nodes (list holds-person)))
    (setf (node-relations seat) (list rel6))
    
    (setq  rel7  (make-relation  :verb *verb-enables*
				 :strength 0.9
				 :from-nodes (list backrest)
				 :to-nodes (list holds-person)))
    (setf (node-relations backrest) (list rel7))
    
    (setq  rel8  (make-relation	 :verb *verb-hasTypicalGen*
				 :strength 0.9
				 :from-nodes (list legs-4)
				 :to-nodes (list seat-support)))
    (setf (node-relations legs-4) (list rel8))
    
    (setq  rel9  (make-relation  :verb *verb-partOf*
				 :strength 0.7
				 :from-nodes (list wood)
				 :to-nodes (list chairs)))
    (setq  rel10  (make-relation  :verb *verb-hasTypicalGen*
				  :strength 0.9
				  :from-nodes (list wood)
				  :to-nodes (list rigid-material)))
    (setf (node-relations wood) (list rel9 rel10))
    
    (setq  rel11  (make-relation  :verb *verb-isFunctionOf*
				  :strength 1.0
				  :from-nodes (list lateral-support)
				  :to-nodes (list armrests)))
    (setq  rel12  (make-relation  :verb *verb-cooccurs*
				  :strength 0.8
				  :from-nodes (list lateral-support)
				  :to-nodes (list holds-person)))
    (setf (node-relations lateral-support) (list rel11 rel12))
    
    (setq  rel13  (make-relation  :verb *verb-cooccurs*
				  :strength 0.8
				  :from-nodes (list holds-person)
				  :to-nodes (list lateral-support)))
    (setq  rel14  (make-relation  :verb *verb-isEnabledBy*
				  :strength 0.9
				  :from-nodes (list holds-person)
				  :to-nodes (list seat)))
    (setq  rel15  (make-relation  :verb *verb-isEnabledBy*
				  :strength 0.9
				  :from-nodes (list holds-person)
				  :to-nodes (list backrest)))
    (setq  rel16  (make-relation  :verb *verb-isEnabledBy*
				  :strength 0.9
				  :from-nodes (list holds-person)
				  :to-nodes (list seat-support)))
    (setq  rel17  (make-relation  :verb *verb-isFunctionOf*
				  :strength 1.0
				  :from-nodes (list holds-person)
				  :to-nodes (list chairs)))
    (setf (node-relations holds-person) (list rel13 rel14 rel15 rel16 rel17))
    
    (setq  rel18  (make-relation  :verb *verb-hasTypicalSpec*
				  :strength 0.9
				  :from-nodes (list seat-support)
				  :to-nodes (list pedestal)))
    (setq  rel19  (make-relation  :verb *verb-hasTypicalSpec*
				  :strength 0.9
				  :from-nodes (list seat-support)
				  :to-nodes (list legs-4)))
    (setq  rel20  (make-relation  :verb *verb-enables*
				  :strength 0.9
				  :from-nodes (list seat-support)
				  :to-nodes (list holds-person)))
    (setf (node-relations seat-support) (list rel20 rel19 rel18))
    
    (setq  rel21  (make-relation  :verb *verb-hasTypicalSpec*
				  :strength 0.9
				  :from-nodes (list rigid-material)
				  :to-nodes (list metal)))
    (setq  rel22  (make-relation  :verb *verb-hasTypicalSpec*
				  :strength 0.9
				  :from-nodes (list rigid-material)
				  :to-nodes (list wood)))
    (setf (node-relations rigid-material) (list rel21 rel22))
    
    (setq  rel23  (make-relation  :verb *verb-hasPart*
				  :strength 0.7
				  :from-nodes (list chairs)
				  :to-nodes (list metal)))
    (setq  rel24  (make-relation  :verb *verb-hasPart*
				  :strength 0.7
				  :from-nodes (list chairs)
				  :to-nodes (list wood)))
    (setq  rel25 (make-relation  :verb *verb-hasFunction*
				 :strength 1.0
				 :from-nodes (list chairs)
				 :to-nodes (list holds-person)))
    (setf (node-relations chairs) (list rel23 rel24 rel25))
    
    ;; Set all relation-inverse slots.
    (setf (relation-inverse rel1)  rel1)
    (setf (relation-inverse rel2)  rel11)
    (setf (relation-inverse rel11) rel2)
    (setf (relation-inverse rel3)  rel23)
    (setf (relation-inverse rel23) rel3)
    (setf (relation-inverse rel4)  rel21)
    (setf (relation-inverse rel21) rel4)
    (setf (relation-inverse rel5)  rel18)
    (setf (relation-inverse rel18) rel5)
    (setf (relation-inverse rel6)  rel14)
    (setf (relation-inverse rel14) rel6)
    (setf (relation-inverse rel7)  rel15)
    (setf (relation-inverse rel15) rel7)
    (setf (relation-inverse rel8)  rel19)
    (setf (relation-inverse rel19) rel8)
    (setf (relation-inverse rel9)  rel24)
    (setf (relation-inverse rel24) rel9)
    (setf (relation-inverse rel10) rel22)
    (setf (relation-inverse rel22) rel10)
    (setf (relation-inverse rel12) rel13)
    (setf (relation-inverse rel13) rel12)
    (setf (relation-inverse rel16) rel20)
    (setf (relation-inverse rel20) rel16)
    (setf (relation-inverse rel17) rel25)
    (setf (relation-inverse rel25) rel17)
  )
  
  (format t
	  "~%~%EXAMPLE KNOWLEDGE BASE LOADED.~
           ~%---------------------------------------------------------------~
           ~%This is the simple category network for classifying chairs,~
           ~%as described in the paper \"Protos: An Exemplar-Based Learning~
           ~%Apprentice\".  It contains a single category 'chairs' plus~
           ~%exemplars 'chair1' and 'chair2'.  You may wish to display~
           ~%portions of this category network.")
  )

