;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)



;;;=============================================================================
;;;
;;;                             C L - P R O T O S
;;;  ---------------------------------------------------------------------------
;;;
;;;  Overview:   This file contains the functions and menus that create the
;;;              top-level user interface to CL-Protos.  Some of these are:
;;;              -- top-level menu, overview menu, control menu, display menu,
;;;                 case-entry menu;
;;;              -- protos: the main function for invoking CL-Protos;
;;;              -- enter-case-and-classify: the top-level function for
;;;                 entering a case, optionally transforming its features, and
;;;                 classifying it;
;;;              -- display-classification: for displaying the resulting
;;;                 classification a new case.
;;;  ---------------------------------------------------------------------------
;;;
;;;  Functions:   protos
;;;               toggle
;;;               overview
;;;               enter-case-and-classify
;;;               get-case-name
;;;               get-case-features
;;;               get-case-category
;;;               get-sample-case
;;;               check-case-features
;;;=============================================================================



(defparameter *top-menu* (make-menu
  :label  "~%~%TOP-LEVEL MENU~
             ~%==============================================================================="
  :repeat t
  :redisplay t
  :twocol t
  :items  `(
	    (#\O . ("go to Overview menu"               menu-select-2  *overview-menu*))
	    (#\X . ("load eXample knowledge base"       load-example))
  	    (#\D . ("go to Display menu"                menu-select-2  *display-menu*))
	    (#\L . ("Load knowledge base from a file"   load-kb))
	    (#\C . ("go to Control menu"                menu-select-2  *control-menu*))
	    (#\S . ("Save knowledge base to a file"     save-kb))
	    (#\U . ("go to Unfocused menu"              menu-select-2  *unfocused-instruction-menu*))
	    (#\F . ("Forget current knowledge base"     forget-kb ,nil))
	    (#\E . ("go to case Entry menu"             enter-case-and-classify))
	    (#\Q . ("Quit Protos"                       return)))))



(defparameter *overview-menu* (make-menu
  :label  "~%~%What would you like an overview of?~
             ~%==============================================================================="
  :repeat t
  :redisplay t
  :twocol t
  :items  '((#\G . ("General concept behind Protos"     overview-general))
	    (#\N . ("Node (term) syntax"                print-node-syn))
	    (#\S . ("Suggestions for getting started"   overview-suggestions))
	    (#\R . ("Relation syntax"                   print-rel-syn))
  	    (#\T . ("Tips for a teacher of Protos"      overview-tips))
	    (#\K . ("Quantifier syntax"                 print-quan-syn))
	    (#\E . ("Explanation language"              overview-language))
	    (#\C . ("Condition syntax"                  print-cond-syn))
	    (nil . (""))
	    (#\Q . ("Quit this menu"                    return)))))



(defparameter *display-menu* (make-menu
  :label  "~%DISPLAY MENU  (select something to display)~
           ~%================================================================================"
  :repeat t
  :redisplay t
  :twocol t
  :items  '((#\G . ("Generalization hierarchy"        print-hierarchy *verb-hasTypicalGen*))
	    (#\T . ("names of all Terms"              print-names *history*))
	    (#\F . ("Functional hierarchy"            print-hierarchy *verb-hasFunction*))
	    (#\D . ("Details about a given name"      print-it))
	    (#\K . ("causal hierarchy"                print-hierarchy *verb-causedBy*))
	    (#\C . ("exemplar-containing Categories"  print-e-categories))
	    (#\P . ("Part-to-whole hierarchy"         print-hierarchy *verb-partOf*))
	    (#\1 . ("1-step relations from x"         show-relations))
	    (#\W . ("Whole knowledge base"            print-whole-kb))
	    (#\R . ("Relationship between 2 terms"    print-relationship))
	    (#\X . ("Show transformation definitions" print-all-transformations))
	    (#\Q . ("Quit this menu"                  return)))))



(defparameter *control-menu* (make-menu
  :label  "~%CONTROL MENU   (select something to toggle)~
           ~%=============================================="
  :repeat t
  :displayvar t
  :items  '((nil . ("Modes:"))
	    (nil . ("----------------------------------"))
	    (#\L . ("  Learning (vs. performance)"       toggle *learning-mode*))
	    (#\M . ("  Maximize learning"                toggle *maximize-mode*))
	    (#\Z . ("  Heuristics"                       toggle *heuristic-mode*))
	    (#\D . ("  Medical diagnosis model"          toggle *diagnostic-model*))
	    (#\X . ("  critique eXplanations"            toggle *switch-critiques*))
	    (#\N . ("  Nth root of similarity"           toggle *switch-nth-root*))
	    (#\A . ("  Abbreviated names"                toggle *abbreviation-mode*))
	    (nil . (""))
            (nil . ("Tracing:"))
	    (nil . ("----------------------------------"))
	    (#\R . ("  Remindings & censors"             toggle *trace-remindings*))
	    (#\I . ("  Importances"                      toggle *trace-importances*))
	    (#\H . ("  Hypotheses"                       toggle *trace-hypotheses*))
	    (#\K . ("  Knowledge-based pattern matching" toggle *trace-kbpm*))
	    (#\Y . ("  KBPM relations traversed"         toggle *trace-kbpm2*))
	    (#\U . ("  heUristics (inside kbpm)"         toggle *trace-heuristics*))
	    (#\C . ("  Case matching"                    toggle *trace-case-matching*))
	    (#\S . ("  tranSformations & computations"   toggle *trace-transformations*))
	    (nil . (""))
	    (#\Q . ("Quit this menu"                     return)))))



(defparameter *unfocused-instruction-menu* (make-menu
  :label  "~%UNFOCUSED INSTRUCTION MENU~
           ~%================================================================================"
  :repeat t
  :redisplay t
  :twocol t
  :items  '((#\T . ("enter new Terms/abbrev/synonyms"          enter-new-terms))
	    (#\R . ("Remove a Relation"                        remove-relation))
	    (#\E . ("Enter new explanations"                   enter-explanation))
	    (#\X . ("define a transformation"                  define-transformation t))
	    (#\C . ("add Censor"                               add-censor))
	    (#\D . ("go to Display menu"                       menu-select-2 *display-menu*))
	    (#\I . ("add/change an Importance"                 change-importance))
	    (#\Q . ("Quit this menu"                           return)))))



;;;---------------------------------------------------------------------------
;;;  Function:	(toggle  switch)
;;;
;;;  Given:	switch, an on/off switch variable
;;;
;;;  Returns:	the negated value of switch which it has stored into switch.
;;;---------------------------------------------------------------------------

(defun toggle (switch)
  (set switch (not (eval switch)))
  (format t "~A set to ~A" switch (eval switch)))



;;;----------------------------------------------------------------------------
;;;  Function:  (overview)
;;;
;;;  Purpose:   This function presents a brief overview of what Protos is
;;;             and how to use it.
;;;----------------------------------------------------------------------------

(defun overview-general ()
  (format t "~%~66,1,0,'\~A" '\~)
  (format t
	  "~%Protos is a machine learning program which takes a new approach~
           ~%to concept acquisition.  It is a domain-independent system which~
           ~%interacts with a human teacher to acquire knowledge.  The teacher~
           ~%trains Protos by giving it cases to classify and by correcting its~
           ~%failures.  Protos learns by selectively retaining cases and other~
           ~%domain knowledge and by compiling appropriate indices for future~
           ~%problem solving.  There is no distinction between learning and~
           ~%problem solving; Protos learns as a natural consequence of~
           ~%performing the task for which the classification system is~
           ~%intended.")
  (format t "~%~66,1,0,'\~A" '\~))


(defun overview-suggestions ()
  (format t "~%~66,1,0,'\~A" '\~)
  (format t
	  "~%The best way to learn how to use Protos is to play with it using~
           ~%an example knowledge base.  So, do the following:~
           ~%-- At the top-level menu, load the example knowledge base.~
           ~%-- Then, go to the display menu and examine the knowledge base~
           ~%   in some detail.~
           ~%-- Then return to the top-level, go to the case-entry menu,~
           ~%   and get the sample case.~
           ~%-- After that case is processed, think up a new case of your own.")
  (format t "~%~66,1,0,'\~A" '\~)
  (terpri))


(defun overview-tips ()
  (format t "~%~66,1,0,'\~A" '\~)
  (format t
	  "~%The following tips are condensed from the chapter~
           ~%\"Tips for a Teacher of Protos\":~
           ~%-- Organize your category network on paper first.~
           ~%-- Decide what features you will use to describe cases.~
           ~%-- Use appropriate level of detail, neither too much nor too little.~
           ~%-- Present the most typical examples of a category first (ideally).~
           ~%-- Give \"educational\" rather than correlational explanations.~
           ~%-- Feel free to add new verbs and quantifiers to the language.~
           ~%-- Run a small 10-case experiment first, before production usage.")
  (format t "~%~66,1,0,'\~A" '\~))




;;;=============================================================================
;;;  Function:  protos
;;;
;;;  Purpose:   This is the top-level function for invoking CL-Protos by a
;;;             user.
;;;
;;;  Design:    1. Initialization (get user's name, open log file);
;;;             2. Invoke top-level menu (repeated until user quits the menu);
;;;             3. If a KB exists, ask user if it should be saved.
;;;=============================================================================

(defun protos ()
  (consume-white-space)
  (format t "~%~22T============================================~
             ~%~22T||        CL-Protos version ~3,1F           ||~
             ~%~22T||   Artificial Intelligence Laboratory   ||~
             ~%~22T||   The University of Texas at Austin    ||~
             ~%~22T============================================"
	  *protos-version*)

  (if (string-not-equal "protos" (package-name *package*))
      (progn
	(format t "~%~%ERROR:  You need to be in package PROTOS,~
                     ~% but you currently are in package ~A.~
                     ~% Please reinvoke by typing:~
                     ~%       (in-package 'protos)~
                     ~%       (protos)"
		(package-name *package*))
	(return-from protos (values))))

  (format *query-io* "~%~%Welcome to Protos.~
               	      ~%Please enter your name or initials ---> ")
  (force-output *query-io*)
  (setq *username* (read-line *query-io* nil nil))

  (if (y-or-n-p "~%~%Do you wish to maintain a log of this session? ")
      (progn
	;; Open log file for explanations, named "protos-log".
	(format t "~%~%Opening log file ~A ..." *log-filename*)
	(with-open-file (*logstream* *log-filename*
				     :direction :output
				     :if-exists :append
				     :if-does-not-exist :create)
	  (multiple-value-bind (second minute hour date month year day dstp tz)
	      (get-decoded-time)
	    (declare (ignore second day dstp tz))
	    (format *logstream*
		    "~%~%====================================================~
                   ~%NEW SESSION STARTED BY ~A: ~D/~D/~D  ~D:~2,'0D~%~%"
		    *username* month date year hour minute))
    
	  (menu-select *top-menu*)))
      (progn
	(setq *logstream* nil)
	(menu-select *top-menu*)))
  
  (if *history*
      (if (prompt "~%~%Save the current knowledge base? " nil 'y-or-n nil nil)
	  (save-kb)))
  (format t "~%~%End of Protos session."))



;;;----------------------------------------------------------------------------
;;;  Function:  enter-case-and-classify
;;;
;;;  Purpose:   This function is called from the top-level menu and presents
;;;             the case-entry menu offering the teacher different choices on
;;;             how to enter the next case.  After the case is entered, it is
;;;             classified and the results are displayed.  This is performed
;;;             repeatedly until the teacher quits the case-entry menu.
;;;
;;;  Note:      This function embodies the main loop of case-entry and 
;;;             classification, i.e.,
;;;
;;;                  {LOOP
;;;                         Enter new case.
;;;                         Transform features (if needed).
;;;                         Classify it.
;;;                         Display results.
;;;                  }
;;;----------------------------------------------------------------------------

(defparameter *enter-case-menu* (make-menu
  :label  "~%CASE-ENTRY MENU~
           ~%=============================================="
  :repeat t
  :redisplay t
  :items  '((#\E . ("Enter case as a list of features"  return  enter-new-case))
	    (#\C . ("Copy-and-edit a previous case"     return  copy-and-edit-case))
	    (#\S . ("get Sample case for Chairs KB"     return  get-sample-case))
	    (#\D . ("Display a selected term"           return  print-it))
	    (#\Q . ("Quit, return to top-level"         return  quit)))))


(defun enter-case-and-classify ()
  (let (newcase)
    (loop
      (case (menu-select *enter-case-menu*)
	
	(enter-new-case      (setq newcase (enter-new-case))
			     (if newcase
				 (classify-and-display newcase)))
	
	(copy-and-edit-case  (setq newcase (copy-and-edit-case))
			     (if newcase
				 (classify-and-display newcase)))
	
	(get-sample-case     (classify-and-display (get-sample-case)))
	
	(print-it            (print-it))
	
	(quit                (return (values)))))))


(defun classify-and-display (newcase)
  (let (matches)
    (if newcase
	(progn
	  (if (case-category newcase)
	      (setf (case-preclassify newcase) t))
	  (setq matches (classify newcase))
	  (display-classification newcase matches)))))

;;;----------------------------------------------------------------------------
;;;  Function:  enter-new-case
;;;
;;;  Purpose:   This function prompts the teacher for the name, features, and
;;;             category of a new case.
;;;
;;;  Returns:   the new case structure, or NIL if the case was abandoned.
;;;----------------------------------------------------------------------------

(defun enter-new-case ()

  ;; Upon starting a new case, force output to the explanation log file
  ;; from the previous case (if any).  This is just a precautionary measure
  ;; to avoid losing any significant amount of log file text in the event
  ;; of a crash or abort.
  (if *logstream*
      (force-output *logstream*))

  (let ((newcase  (make-case))
	(casename (get-case-name))
	features)

    (if (null casename)
	nil
	(progn
	  (setf (case-name     newcase) casename)
	  (setf (case-comment  newcase) (get-comment newcase 'case))
	  (setq features (get-case-features nil))
	  (if (null features)
	      (progn
		(format t "~%No features, so this case is abandoned.")
		nil)
	      (progn
		(setf (case-features newcase) (check-case-features features))
		(setf (case-category newcase) (get-case-category nil))
		(if (case-category newcase)
		    (setf (case-preclassify newcase) t))
		(push newcase *history*)
		newcase))))))



;;;----------------------------------------------------------------------------
;;;  Function:  get-sample-case
;;;
;;;  Purpose:   This function automatically enters a sample case intended for
;;;             use with the "chairs" knowledge base.  This function allows
;;;             for easy demonstration of CL-Protos.
;;;----------------------------------------------------------------------------

(defun get-sample-case ()
  (let ((features '(seat pedestal rigid-material swivel))
	(newcase  (make-case)))

    (format t "~%Entering a new case consisting of these features:")
    (dolist (featurename features)
      (format t "~%---> ~(~A~)"  featurename))

    (setf (case-creator  newcase) 'dvorak)
    (setf (case-name     newcase) 'chair3)
    (setf (case-comment  newcase) nil)
    (setf (case-features newcase) (check-case-features features))
    (setf (case-category newcase) nil)
    (if (case-category newcase)
	(setf (case-preclassify newcase) t))
    (push newcase *history*)
    newcase))



;;;----------------------------------------------------------------------------
;;;  Function:  copy-and-edit-case
;;
;;;  Purpose:   This function assists the user in constructing a new case by
;;;             copying and then editing an existing case.  
;;;----------------------------------------------------------------------------

(defun copy-and-edit-case ()
  (prog ((newcase (make-case))
	 input object exemplars exemplar category features)

     RETRY
	(format *query-io*
		"~%~%Enter the name of an existing exemplar or category~
                   ~%(or type 'Q' to return to menu) ------->  ")
	(force-output *query-io*)
	(setq input (read *query-io* nil 'Q))

	(if (eql 'Q input)
	    (return-from copy-and-edit-case nil))

     TESTIT
	(if (not (boundp input))
	    (progn
	      (format *query-io* "~%Sorry, but ~A is unrecognized.~%" input)
	      (go RETRY)))

	(setq object (eval input))
	(if (not (term-p object))
	    (progn
	      (format t "~%Sorry, but ~A is neither an exemplar nor a category~%"
		      input)
	      (go RETRY)))

	(if (my-exemplar-p object)
	    ;; this is an exemplar, so go copy and edit it.
	    (progn
	      (setq exemplar object)
	      (go PROCESS-IT))
	    ;; this is a category, so let user pick one of its exemplars.
	    (progn
	      (setq category object)
	      (setq exemplars (category-exemplars category))
	      (if (null exemplars)
		  (progn
		    (format t "~%Sorry, this category has no exemplars.~%")
		    (go RETRY)))
	      (format t "~%~A has the following exemplars, in order of typicality:~%"
		      (getname category))
	      (print-node-names exemplars t ", ")
	      (format *query-io* "~%~%Type 'p' for the prototype, else enter a name: ")
	      (force-output *query-io*)
	      (setq input (read *query-io* nil 'P))
	      (if (eql 'P input)
		  (progn
		    (setq exemplar (first exemplars))
		    (go PROCESS-IT)))
	      (go TESTIT)))
	
     PROCESS-IT
	(setq features (exemplar-features exemplar))
	(format t "~%This exemplar has the following features:~%")
	(print-node-names features t ", ")
	(format t "~%~%Which features should be included in the new case?~%")
	(let ((flist1 nil) flist2)
	  (dolist (feature features)
	    (if (y-or-n-p "~&--> ~(~A~)?~26T" (getname feature))
		(push (getname feature) flist1)))
	  (setq flist1 (nreverse flist1))
	  (format t "~%~%OK, the case has these features:~%~(~A~)~%" flist1)
	  ;; ask about additional new features
	  (setq flist2 (get-case-features t))
	  (setf (case-features newcase) (check-case-features (append flist1 flist2)))
	  (setf (case-name     newcase) (get-case-name))
	  (setf (case-category newcase) (get-case-category (exemplar-category exemplar)))
	  (if (case-category newcase)
	      (setf (case-preclassify newcase) t))
	  (format t "~%~%OK, the final case features are:~%")
	  (print-node-names (case-features newcase) t ", ")
	  (push newcase *history*)
	  (return-from copy-and-edit-case newcase))))





;;;----------------------------------------------------------------------------
;;;  Function:  (display-classification  newcase  matches)
;;;
;;;  Given:     -- newcase, a case which has been classified; and
;;;             -- matches, the resulting newcase-to-exemplar matches;
;;;
;;;  Do:        Display the classification results in a form suitable for the
;;;             teacher/user.
;;;----------------------------------------------------------------------------

(defun display-classification (newcase matches)

  (format t "~%~%Final Results of Classifying ~A:~
               ~%----------------------------------------------------------------"
	  (case-name newcase))

  (cond (;; NIL means no match was found and approved.
	 (null matches)
	 (format t "~%   No match was found and approved."))
	      
	(;; An exemplar means the new case was made into an exemplar.
	 (exemplar-p matches)
	 (format t "~%   The new case became an exemplar of ~A"
		 (getname (exemplar-category matches)))
	 (setf (case-disposition newcase) (list 'became (getname matches))))

	(;; A list means the ordered list of successful matches.
	 (listp matches)
	 (format t "~%    Category  ~29TExemplar  ~54TSimilarity~
                    ~%    --------  ~29T--------  ~54T----------")
	 (let ((n 0))
	   (dolist (match matches)
	     (let* ((exemplar (match-exemplar match))
		    (category (exemplar-category exemplar))
		    (similarity (if *switch-nth-root*
				    (match-nth-root-of-similarity match)
				    (match-similarity match))))
	       (format t "~%~2D  ~24A ~24A ~6,2F"
		       (incf n) (getname category) (node-name exemplar) similarity)))

	   (cond ((null (cdr matches))
		  (if (y-or-n-p "~%~%Do you wish to see this match? ")
		      (print-match (car matches))))
		 (t
		  (if (y-or-n-p "~%~%Do you wish to see any of these matches? ")
		      (loop
			(let* ((prompt1 (format nil "~%Please enter a number between 1 and ~D (q to quit): "
						n))
			       (m (prompt prompt1 nil 'integer nil nil)))
			  (if (null m) (return (values)))
			  (if (and (>= m 1) (<= m n))
			      (print-match (nth (1- m) matches))))))))))

	(;; Anything else is an error.
	 t
	 (format t "~%   ERROR: display-classification: unknown value of 'matches'"))))




;;;----------------------------------------------------------------------------
;;;  Function:  get-case-name
;;;
;;;  Purpose:   This function prompts the user to enter the name of a new case,
;;;             then checks the to be sure the name isn't already in use.
;;;----------------------------------------------------------------------------

(defun get-case-name ()
  (let (input)
    (loop
	(setq input (prompt "~%~%Enter name of this case (a single word, or q to quit): "
			    nil 'symbol nil nil))

	(cond ((or (null input) (eql 'q input))
	       (return nil))

	      ((boundp input)
	       (format *query-io* "~%Sorry, ~A is already in use as a ~A.~
                                   ~%Please choose another name."
		       input (type-of (eval input))))
	      (t    (return input))))))



;;;----------------------------------------------------------------------------
;;;  Function:  (get-case-features  additional)
;;;
;;;  Purpose:   This function prompts the user to enter the features (or
;;;             additional features) of a case one feature at a time.
;;;----------------------------------------------------------------------------

(defun get-case-features (additional)
   (if additional
       (prompt "~%Please enter any additional features, ~
                ~%one feature per line (terminate with blank line):~%"
               "~&---> " 'feature nil nil)
       (prompt "~%Please enter the features of this case, ~
                ~%one feature per line (terminate with blank line):~%"
               "~&---> " 'feature nil nil)))


;;;----------------------------------------------------------------------------
;;;  Function:  (get-comment  object  view)
;;;
;;;  Purpose:   This function prompts the user to enter any comments about this
;;;             case/term/explanation as free-form text.  This text may be
;;;             useful later in verifying why some piece of information is in
;;;             the knowledge base.
;;;----------------------------------------------------------------------------

(defun get-comment (object view)
  (let ((prompt1 (format nil "~%Please enter any comments on this ~(~A~):"
			 (if view view 'item))))
    (prompt prompt1 "~&---> " 'string object view)))



;;;----------------------------------------------------------------------------
;;;  Function:  (check-case-features  fnames)
;;;
;;;  Given:     fnames, a list of feature names;
;;;
;;;  Returns:   a list of terms.
;;;
;;;  Purpose:   Check each feature of the new case to ensure that it is a
;;;             valid feature. Transform numerical features if possible.
;;;----------------------------------------------------------------------------

(defun check-case-features (fnames)
  (let ((flist nil) term)
    (dolist (fname fnames)
      ;; Check whether some of the features contain numerical values.
      ;; Check for new terms in the feature list by calling 'check-term-name'.
      ;; This fuunction also creates the appropriate feature- and predicate-
      ;; structures.
      (cond ((and (listp fname)
		  (dolist (value (cdr fname) nil)
		    (cond ((numberp value) (return t)))))
	     (setf flist (check-and-transform-feature fname flist)))
	    
	    ;; If no numeric values, the feature is pushed onto the feature list as it is.
	    ;; Pushnew is used instead of push in case the teacher accidentally
	    ;; enters the same feature name twice.
	    (t (setq term (check-term-name fname 'ask))
	       (if term
		   ;; pushnew is used instead of push in case the teacher
		   ;; accidentally enters the same feature name twice.
		   (pushnew term flist)))))
    (nreverse flist)))




(defparameter *category-menu1* (make-menu
  :label  "~%~%Do you wish to pre-classify this case?~
             ~%---------------------------------------"
  :repeat nil
  :redisplay nil
  :items  `((#\N . ("No."                           return N))
  	    (#\Y . ("Yes, prompt for a category."   return Y)))))


(defparameter *category-menu2* (make-menu
  :label  "~%~%Do you wish to pre-classify this case?~
             ~%---------------------------------------"
  :repeat nil
  :redisplay nil
  :items  `((#\N . ("No."                           return N))
  	    (#\Y . ("Yes, prompt for a category."   return Y))
	    (#\C . ("yes, Classify as XXX"          return C)))))



;;;----------------------------------------------------------------------------
;;;  Function:  get-case-category
;;;
;;;  Purpose:   This function prompts the user for the classification of the
;;;             the new case.
;;;----------------------------------------------------------------------------

(defun get-case-category (thiscat)
  (let (menu)
    
    (if (null thiscat)
	(setq menu *category-menu1*)
	(progn
	  (setf (cadr (third (menu-items *category-menu2*)))
		(format nil "yes, Classify as ~A" (getname thiscat)))
	  (setq menu *category-menu2*)))
    
    (let ((input (menu-select menu)))
      (case input
	(N  nil)
	(C  thiscat)
	(Y  (let ((cname (prompt "~%~%Enter classification: " nil 'symbol nil nil)))
	      (if cname
		  (check-term-name cname 'ask)
		  nil)))
	(t  (format t "~%ERROR: get-case-category: invalid input."))))))
    

