;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Ben Tso and Daniel L. Dvorak

(in-package 'protos)



;;;=============================================================================
;;;
;;;                  E X P L A N A T I O N    P A R S E R
;;;   ------------------------------------------------------------------------
;;;
;;;  Overview:	 This file contains the vocabulary of the Protos explanation
;;;              language and the functions for parsing explanations entered
;;;              by the user.
;;;
;;;  Functions:	 first-pass
;;;              get-explanation
;;;              get-name
;;;              get-condition
;;;              get-quantifiers
;;;              get-more-quanitifiers
;;;              process-phrase
;;;              set-term-list
;;;
;;;  Future:     Ideally, the rel-list and quan-list should be built
;;;              automatically from the information in the verb and 
;;;              quantifier structures defined in verb.lisp.  It would
;;;              require the addition of a "synonyms" slot in these
;;;              structures so that all of the equivalent ways of expressing
;;;              a quantifier or verb would be stored in its structure.
;;;=============================================================================



;;;-----------------------------------------------------------------------------
;;; Parameter: verb-list
;;;
;;; Purpose:   This is an association list of the possible relation names
;;;            and thier verb name counterparts.
;;;
;;;-----------------------------------------------------------------------------


(defparameter verb-list
	   `((implies             . ,*verb-implies*)
	     (isImpliedBy         . ,*verb-isImpliedBy*)
	     (causes              . ,*verb-causes*)
	     (causedBy            . ,*verb-causedBy*)
	     (enables             . ,*verb-enables*)
	     (isEnabledBy         . ,*verb-isEnabledBy*)
	     (hasFunction         . ,*verb-hasFunction*)
	     (isFunctionOf        . ,*verb-isFunctionOf*)
	     (hasPart             . ,*verb-hasPart*)
	     (partOf              . ,*verb-partOf*)
	     (suggests            . ,*verb-suggests*)
	     (isInferredFrom      . ,*verb-isInferredFrom*)
	     (isRequiredBy        . ,*verb-isRequiredBy*)
	     (requires            . ,*verb-requires*)
	     (exhibits            . ,*verb-exhibits*)
	     (isConsistentWith    . ,*verb-isConsistentWith*)
	     (actsOn              . ,*verb-actsOn*)
	     (isActedOnBy         . ,*verb-isActedOnBy*)
	     (affects             . ,*verb-affects*)
	     (isAffectedBy        . ,*verb-isAffectedBy*)
	     (m+                  . ,*verb-m+*)
	     (m-                  . ,*verb-m-*)
	     (fm+                 . ,*verb-fm+*)
	     (rm+                 . ,*verb-rm+*)
	     (fm-                 . ,*verb-fm-*)
	     (rm-                 . ,*verb-rm-*)
	     (iff                 . ,*verb-iff*)
	     (equivalent          . ,*verb-equivalent*)
	     (defimplies          . ,*verb-defimplies*)
	     (cooccurs            . ,*verb-cooccurs*)
	     (mutualExclusion     . ,*verb-MEx*)
	     (hasTypicalSpec      . ,*verb-hasTypicalSpec*)
	     (hasTypicalGen       . ,*verb-hasTypicalGen*)
	     (spurious            . ,*verb-spurious*)))


;;;-----------------------------------------------------------------------------
;;; Parameter: rel-list
;;; 
;;; Purpose:   The relation list is a "tree" of association lists
;;;            that allows for the parsing of that part of the 
;;;            sentence that constitutes as an acceptable relation
;;;            name.  An atomic key (i.e. a leaf node) is the
;;;            key word of the relation that a word or phrase
;;;            represents.
;;;-----------------------------------------------------------------------------

(defparameter rel-list
    '((consistent  . ((with       . isConsistentWith)))
      (implied     . ((by         . isImpliedBy     )))
      (required    . ((by         . isRequiredBy    )))
      (definition  . ((implies    . defimplies      )
		      (imp        . defimplies      )
		      (suggests   . defimplies      )))
      (defn        . ((implies    . defimplies      )
		      (imp        . defimplies      )
		      (suggests   . defimplies      )))
      (def         . ((implies    . defimplies      )
		      (imp        . defimplies      )
		      (suggests   . defimplies      )))
      (enabled     . ((by         . isEnabledBy     )))
      (act         . ((on         . actsOn)))
      (acts        . ((on         . actsOn)))
      (cooccur     . ((with       . cooccurs)))
      (cooccurs    . ((with       . cooccurs)))
      (if          . ((and        . ((only  . ((if . iff)))))))
      (caused      . ((by         . causedBy)))
      (inferred    . ((from       . isInferredFrom    )))
      (part        . ((of         . partOf         )))
      (have        . ((parts      . hasPart)
		      (functions  . hasFunction)
		      (Functions  . hasFunction)
		      (func       . hasFunction)))
      (has         . (( part      . hasPart         )
		      (function   . hasFunction     )
		      (Function   . hasFunction     )
		      (func       . hasFunction     )
		      (typical    . ((generalization   . hasTypicalGen)
				     (genl             . hasTypicalGen)
				     (gen              . hasTypicalGen)
				     (specialization   . hasTypicalSpec)
				     (specl            . hasTypicalSpec)
				     (spec             . hasTypicalSpec))) ))
      (is          . (( consistent . ((with             . isConsistentWith)))
                      ( part       . ((of              . partOf)))
		      ( equivalent . ((to              . equivalent)))
		      ( implied   . ((by               . isImpliedBy)))
		      ( required  . ((by               . isRequiredBy)))
		      ( function  . ((of               . isFunctionOf)))
		      ( func      . ((of               . isFunctionOf)))
		      ( enabled   . ((by               . isEnabledBy)))
		      ( inferred  . ((from             . isInferredFrom)))
                      (caused     . ((by               . causedBy)))
                      (mutually   . ((exclusive        . ((with . mutualExclusion))))) 							
		      (acted      . ((on               . ((by  . isActedOnBy)))))
                      (effected   . ((by               . isAffectedBy)))
		      (affected   . ((by               . isAffectedBy))) ))
       (are         . ((consistent  . ((with            . isConsistentWith)))
		      (implied    . ((by               . isImpliedBy)))
		      (required   . ((by               . isRequiredBy)))
		      (function   . ((of               . isFunctionOf)))
		      (func       . ((of               . isFunctionOf)))
		      (enabled    . ((by               . isEnabledBy)))
		      (caused     . ((by               . causedBy)))
		      (acted      . ((on               . ((by  . isActedOnBy)))))
		      (effected   . ((by               . isAffectedBy)))
		      (affected   . ((by               . isAffectedBy))) ))
		      

      (co  . cooccurs)            (cooccurs        . cooccurs)
                                  (cooccur         . cooccurs)
      (cw  . isConsistentWith)    (isConsistentWith . isConsistentWith)
      (ex  . exhibits)            (exhibits        . exhibits)
                                  (exhibit         . exhibits)
      (ib  . isImpliedBy)         (isImpliedBy     . isImpliedBy)
      (imp . implies)             (implies         . implies)
                                  (imply           . implies)
      (rq  . requires)            (requires        . requires)
                                  (require         . requires)
      (rb  . isRequiredBy)        (isRequiredBy    . isRequiredBy)
      (sp  . spurious)            (spurious        . spurious)
      (eq  . equivalent)          (equivalent      . equivalent)
      (di  . defimplies)          (defimplies      . defimplies)
      (if  . iff)                 (iff             . iff)
      (sg  . suggests)            (suggests        . suggests)
				  (suggest         . suggests)			     
      (ifr . isInferredFrom)      (isInferredFrom  . isInferredFrom)
      (po  . partOf)              (partOf          . partOf)
      (hp  . hasPart)             (hasPart         . hasPart)
      (hf  . hasFunction)         (hasFunction     . hasFunction)
      (fo  . isFunctionOf)        (isFunctionOf    . isFunctionOf)
      (en  . enables)             (enables         . enables)
      (eb  . isEnabledBy)         (isEnabledBy     . isEnabledBy)
      (cs  . causes)              (causes          . causes)
                                  (cause           . causes)
      (cb  . causedBy)            (causedBy        . causedBy)
      (me  . mutualExclusion)     (mutualExclusion . mutualExclusion)
      (ao  . actsOn)              (actsOn          . actsOn)
      (ia  . isActedOnBy)         (isActedOnBy     . isActedOnBy)
      (af  . affects)             (affects         . affects)
				  (affect          . affects)
      (ab  . isAffectedBy)        (isAffectedBy    . isAffectedBy) 
      (tg  . hasTypicalGen)       (hasTypicalGen   . hasTypicalGen)
      (ts  . hasTypicalSpec)      (hasTypicalSpec  . hasTypicalSpec)
      (m+  . m+)
      (m-  . m-)
      (fm+ . fm+)
      (rm+ . rm+)
      (fm- . fm-)
      (rm- . rm-)
      (cooccurs   . cooccurs)
      (co-occurs  . cooccurs)
      (imply      . implies)
      (req        . requires)
      (irrelevant . spurious)
      (spur       . spurious)
      (equiv      . equivalent)
      (partof     . partOf)
      (haspart    . hasPart)
      (effect     . affects)
      (effects    . affects) ))

;;;-----------------------------------------------------------------------------
(defparameter cond-words '(if when category cat case exem exemplar))

;;;-----------------------------------------------------------------------------
;;;  Parameter:  quan-list
;;;  
;;;  Purpose:    The quantifier list is a tree of association lists 
;;;              used for retrieving the key word that a quantifier
;;;              may represent. 
;;;-----------------------------------------------------------------------------

(defparameter quan-list 
	      '(( occasionally . occasionally)
		( occ          . occasionally)
		( oc           . occasionally)
		( sometimes    . sometimes   )
		( st           . sometimes   )
		( moderately   . moderately  )
		( moderate     . moderately  )
		( mod          . moderately  )
		( mo           . moderately  )
		( strongly     . strongly    )
		( strong       . strongly    )
		( str          . strongly    )
		( usually      . usually     )
		( usual        . usually     )
		( us           . usually     )
		( always       . always      )
		( al           . always      )
		( weakly       . weakly      )
		( weak         . weakly      )
		( we           . weakly      )
		( gradually    . gradually   )
		( gradual      . gradually   )
		( grad         . gradually   )
		( gr           . gradually   )
		( quickly      . quickly     )
		( quick        . quickly     )
		( qu           . quickly     )
		( instantly    . instantly   )
		( instant      . instantly   )
		( ins          . instantly   )
		( in           . instantly   )
		( possibly     . possibly    )
		( possible     . possibly    )
		( poss         . possibly    )
		( pos          . possibly    )
		( probably     . probably    )
		( probable     . probably    )
		( prob         . probably    )
		( certainly    . certainly   )
		( certain      . certainly   )
		( cert         . certainly   )
		( ce           . certainly   )))


(defparameter quant-alist
	      `((always       . ,*quant-always*)           ; CONTEXT
		(usually      . ,*quant-usually*)
		(sometimes    . ,*quant-sometimes*)
		(occasionally . ,*quant-occasionally*)
		(strongly     . ,*quant-strongly*)         ; STRENGTH
		(moderately   . ,*quant-moderately*)
		(weakly       . ,*quant-weakly*)
		(instantly    . ,*quant-instantly*)        ; TEMPORAL
		(quickly      . ,*quant-quickly*)
		(gradually    . ,*quant-gradually*)
		(certainly    . ,*quant-certainly*)        ; BELIEF
		(probably     . ,*quant-probably*)
		(possibly     . ,*quant-possibly*)))

;;;-----------------------------------------------------------------------------
;;;   Parameter:        condition-list
;;;
;;;   Purpose:          This tree of association lists is used to search through
;;;                     for a key that represents the type of a condition 
;;;
;;;-----------------------------------------------------------------------------
(defparameter condition-list
	     '((if   . ((category    . ((is       . category)))
		        (cat         . ((is       . category)))
			(case        . ((has      . ((features     . newcase)))
				        (hf       . newcase)))
			(exem        . ((has      . ((features     . exemplar)))
				        (hf       . exemplar)))
			(exemplar    . ((has      . ((features     . exemplar)))
				        (hf       . exemplar)))))
	       (when . ((category    . ((is       . category)))
			(cat         . ((is       . category)))
			(case        . ((has      . ((features     . newcase)))
				        (hf       . newcase)))
			(exem        . ((has      . ((features     . exemplar)))
				        (hf       . exemplar)))
			(exemplar    . ((has      . ((features     . exemplar)))
				        (hf       . exemplar))) )) ))
;;;-----------------------------------------------------------------------------

(defparameter rel-print-list
    '(("imp"     "implies"                       "ib"     "is implied by")
      ("cs"      "causes"                        "cb"     "is caused by")
      ("en"      "enables"                       "eb"     "is enabled by")
      ("hf"      "has function"                  "fo"     "is function of")
      ("hp"      "has part"                      "po"     "part of")
      ("sg"      "suggests"                      "ifr"    "is inferred from")
      ("req"     "requires"                      "rb"     "is required by")
      ("ex"      "exhibits"                      "cw"     "is consistent with")
      ("ao"      "acts on"                       "ia"     "is acted on by")
      ("af"      "affects"                       "ab"     "is affected by")
      ("tg"      "has typical generalization"    "ts"     "has typical specialization")
;      ("m+"      "m+"                            "(same)")
;      ("m-"      "m-"                            "(same)")
;      ("fm+"     "fm+"                           "rm+"    "rm+")
;      ("fm-"     "fm-"                           "rm-"    "rm-")
      ("iff"     "if and only if"                "(same)")
      ("eq"      "equivalent"                    "(same)")
      ("di"      "definition implies"            "(same)")
      ("co"      "cooccurs with"                 "(same)")
      ("me"      "is mutually exclusive with"    "(same)")
      ("spur"    "spurious"                      "(same)") ))

;;;-----------------------------------------------------------------------------


;;;-----------------------------------------------------------------------------									 
;;; Function:  (get-name s name-alist)
;;; 
;;; Given:     s, a sentence expressed as a list of symbols;
;;;            name-alist, an alist to be searched for the longest phrase
;;;                        matching the beginning of the sentence;
;;;      		      
;;; Returns:   two values:
;;;            -- a keyword representing the phrase that was found, or nil if
;;;               no valid phrase was found;
;;;            -- the number of words matched from the beginning of the sentence.
;;;-----------------------------------------------------------------------------

(defun get-name (s name-alist ctr)
  (let ((cur-state (cdr (assoc (car s) name-alist))))
    (cond
      ;; if at end of list then key doesn't exist
      ((null cur-state) (values nil 0))
      ;; if the key of the association is an atom
      ;; then the key has been found.
      ((atom cur-state)  (values cur-state ctr))
      ;; otherwise continue searching list
      (t (get-name (cdr s) cur-state (+ 1 ctr))))) )
      
;; get-name2 is the same as get name, except it returns
;; the number of words searched through regardless of
;; whether a keyword was found.

(defun get-name2 (s name-alist ctr)
  (let ((cur-state (cdr (assoc (car s) name-alist))))
	(cond
	  ((null cur-state) (values nil ctr))
	  ((atom cur-state) (values cur-state ctr))
	  (t (get-name2 (cdr s) cur-state (+ 1 ctr))) )))

;;;----------------------------------------------------------------------------
;;; Function:  (process-phrase s name-alist)
;;;
;;; Given:     s, a sentence expressed as a list of symbols
;;;            name-alist, an alist to be searched for the longest phrase
;;;                        matching the beginning of the sentence
;;; Returns:   two values:
;;;            -- a keyword representing the phrase that was found , or nil if
;;;               no valid phrase was found.
;;;            -- The sentence minus the words that had been searched through if
;;;               the keyword was found.  The original sentence if no keyword was 
;;;               found.
;;;----------------------------------------------------------------------------

(defun process-phrase  (s name-alist)
  (multiple-value-bind (name counter) (get-name s name-alist 1)
    (if name
	(values (nthcdr counter s) name)
	(values s name))))

;;;----------------------------------------------------------------------------
;;;  Function   (get-condition s)
;;;  
;;;  Given:     s, a sentence expressed as a list of symbols, the beginning of 
;;;             which a condition is thought to occur.
;;;  Returns:   two values:
;;;             -- a keyword representing the condition type that was found, if 
;;;                any, or nil if no valid condition phrase was found.
;;;             -- the sentence minus all the words that had been searched through
;;;                as a part of the conditional, if a conditional was found.
;;;                (The keyword "then" is also skipped over). If a conditonal
;;;                was not found, the original sentence is returned.
;;;----------------------------------------------------------------------------
                               
(defun get-condition (s)
  (let ((features nil)
	conditional
	feature)
    ;; try to find condition type in the alist condition-list
    (multiple-value-bind (new-sentence condition-type) (process-phrase s condition-list)
    (cond 
      ;; if a category was found, make appropriate conditional structure.
      ((eq condition-type 'category)
       (setq conditional (build-condition
			 :type        'category
                         :category    (check-term-name (car new-sentence) 'ask))))
      ;; if an exemplar or newcase was found, make appropriate condtional structure.
      ((or (eq condition-type 'newcase) (eq condition-type 'exemplar))
       (dolist (fname (car new-sentence))
	 (and (not (eql 'and fname))
	      (setq feature (check-term-name fname 'ask))
	      (push feature features)))
       (setq features (nreverse features))
       (setq conditional (build-condition :type      condition-type
					  :features  features)))
      ;; otherwise, no conditonal was found.			 
      (t
       (setq conditional nil)) )
    (cond
      ;; if no conditional found, return nil and original sentence.
      ((null conditional)
       (values nil s))
      ;; if the keyword "then" exsists after the conditional,
      ;; skip over it and return conditional and rest of sentence.
      ((eq (cadr new-sentence) 'then)
       (values conditional (cddr new-sentence)))
      ;; otherwise, return conditional and rest of sentence.
      (t (values conditional (cdr new-sentence))) ) )))
      


;;;-----------------------------------------------------------------------------
;;;
;;;  Function:  (get-quantifiers s)
;;;  
;;;  Given:     s, a sentence expressed as a list of symbols, at the 
;;;             beginning of which quantifier(s) may occur.
;;;
;;;  Returns:   TWO VALUES:
;;;             --a list containing all the quantifiers (that is, a pointer
;;;               to the structure of each quantifier) if any quantifiers
;;;               are found. Otherwise, return nil. 
;;;             --the sentence minus all the words that had been searched 
;;;               through to find the quantifiers (If at least one quantifier
;;;               was found).  Otherwise, the original sentence is returned.
;;;
;;;-----------------------------------------------------------------------------

(defun get-quantifiers (s)
  (let (new-sentence quantifiers)
    ;; get the first quantifier, if any
    (multiple-value-bind (new-lis quan) (process-phrase s quan-list)
      (cond 
	;; if no quantifiers, return original sentence and nil for quantifiers
	((null quan) (values s nil))
	;; otherwise, try to get more quantifiers.
	(t (multiple-value-setq (new-sentence quantifiers) 
                                (get-more-quantifiers new-lis (list quan)))
	   ;; return new sentence
	   (values new-sentence
            ;; the mapcar function returns for each quantifier keyword,
            ;; the pointer to the corresponding quantifier structure.
	    ;; therefore, a list of pointers to each quantifier is returned.
	    (mapcar #'(lambda (q) (cdr (assoc q quant-alist)))  quantifiers) ))))))

;;;-----------------------------------------------------------------------------
;;;  FUNCTION:  (get-more-quantifiers sen quantifiers)
;;;
;;;  GIVEN:     1) sen: a list representing the sentence to be searched.
;;;             2) quantifiers: a list onto which quantifier keywords
;;;                are to be added.
;;;  RETURNS:   a list of quantifier keywords.
;;;-----------------------------------------------------------------------------

(defun get-more-quantifiers (sen quantifiers)
  ;;try to get a quantifier
  (multiple-value-bind (sen cur-quan) (process-phrase sen quan-list)
    (cond 
      ;; if no quantifier was found, then return
      ;; the reverse of the list of quantifiers.
      ((null cur-quan) (values sen (reverse quantifiers)))
      ;; otherwise, push the quantifier found onto the
      ;; list of quantifiers. Try to get another quantifier.
      (t (push cur-quan quantifiers)
	 (get-more-quantifiers sen quantifiers))) ))

    
;;;----------------------------------------------------------------------------
;;;  Function:  (set-term-list  names  antecedents)
;;;
;;;  Given:     -- names, a list of names of terms or a single name of a term;
;;;             -- antecedents, which, if non-nil, means that the terms are to
;;;                     be installed in (explanation-from-terms expl), other-
;;;                     wise in (explanation-to-terms expl);
;;;             -- expl, the explanation structure into which the terms are
;;;                     to be placed.
;;;
;;;  Returns:   nil if an error encountered, otherwise t.
;;;----------------------------------------------------------------------------

(defun set-term-list (names antecedents expl)
  (let ((termlist nil) term)
    ;; 'names' may be null in the case of a spurious relation.
    (if (null names)
	(if antecedents
	    (progn
	      (format t "~%Error: antecedents are missing!")
	      (return-from set-term-list nil))
	    (progn
	      (setf (explanation-to-terms expl) nil)
	      (return-from set-term-list t))))

    (if (atom names)
	(return-from set-term-list
	  (set-term-list (list names) antecedents expl)))

    ;; If this is of the form "(A)" or "(A and B ...)" ...
    (if (or (not (cdr names))
	    (and (>= (length names) 3)
		 (eql 'and (second names))))
	;; then this is a conjunction of terms to be checked
	(dolist (name names)
         (progn
	  (if (numberp name)
	      (progn
		(format t "~%Error: ~A is not a legal term name." name)
		(return-from set-term-list nil)))
	  ;; If the conjunct word "and" appears, skip over it.
	  (if (not (eql 'and name))
	      (progn                ;; removed a hack for variables
		(setq term (check-term-name name 'ask))
		(if term
		    (push term termlist))))))
	;; else this is a predicate with arguments.
	(progn
	  (setq term (check-term-name names 'ask))
	  (if term
	      (push term termlist))))
    
    ;; If no terms, report error.
    (if (null termlist)
	(progn
	  (format t "~%Syntax error in ~A." names)
	  (return-from set-term-list nil)))
    
    ;; Install list of terms in the explanation structure.
    (setq termlist (nreverse termlist))
    (if antecedents
	(setf (explanation-from-terms expl) termlist)
	(setf (explanation-to-terms expl)   termlist))
    
    ;; Install the start-term, if these are antecedents.
    (if antecedents
	(setf (explanation-start-term expl) (car termlist)))
    t))



;;;----------------------------------------------------------------------------------------
;;;  Function:  (get-explanation  e-list)
;;;
;;;  Given:     e-list the list of symbols that make up the sentence of an explantion
;;;  
;;;  Returns:   two values:
;;;             -- the explanation structure, and
;;;             -- the first relation installed.
;;;
;;;-----------------------------------------------------------------------------------------
  
(defun get-explanation (e-list)
  (get-explanation2 e-list nil nil))


;;;  get-explanation2 is like get-explanation except that its one additional
;;;  argument (from-terms), if non-nil, is a list of from-terms to be used
;;;  in this relation (and not extracted from the e-list).

(defun get-explanation2 (e-list from-terms condition)
   (let (rel quan  verbtype verb 
	     (expl (make-explanation)))
    
    ;; GET CONDITIONAL
    ;; e-list = e-list minus the words that were part of the condition
    (if (null condition)
       (multiple-value-setq (condition e-list) (get-condition e-list)))

    ;; GET ANTECEDENTS.
    ;; If from-terms have already been extracted (as the to-terms of a previous relation) ...
    (if from-terms
	(progn
	  (setf (explanation-from-terms expl) from-terms)
	  (setf (explanation-start-term expl) (car from-terms)))
	;; Check and install each antecedent term.
	(if (set-term-list (car e-list) t expl)
	    (setq e-list (cdr e-list))
	    (return-from get-explanation2 (values nil nil))))

    ;; GET QUANTIFIERS (if any).
    ;; If there is a quantifier, then
    ;; e-list = e-list  minus words that were part of the quantifier.
    (multiple-value-setq (e-list quan) (get-quantifiers e-list ))

    ;; GET VERB/RELATION.
    ;; e-list = e-list minus words that were part of the relation.
    (multiple-value-setq (e-list verbtype)  (process-phrase e-list rel-list))
    (setq verb (cdr (assoc verbtype verb-list)))

    ;; If no relation found then report error.
    (if (null verb)
	(progn
	  (format t "~%Syntax error: a verb was expected a this point: ~A" e-list)
	  (return-from get-explanation2 (values nil nil))))

    ;; GET CONSEQUENTS.
    (if (null (set-term-list (car e-list) nil expl))
	(return-from get-explanation2 (values nil nil)))

    ;; INSTALL RELATION.
    (setq rel (install-relation (explanation-from-terms expl)
                                (explanation-to-terms expl) 
				quan
                                verb
				condition))

    (setf (explanation-relation expl) rel)

    ;; If there is another step to this explanation, then process it.
    (if (cdr e-list)
	(let ((expl2  (get-explanation2 (cdr e-list) (explanation-to-terms expl) condition)))
	  (if expl2
	      (setf (explanation-to-terms expl) (list expl2)))))

    (values expl rel)))

;;;-----------------------------------------------------------------------

(defun print-cond-syn ()
  (format t "~%SYNTAX OF CONDITIONS:~
             ~%~
             ~%   1.  (if category is X then ...)~
             ~%       (if cat is X then ...)                         (Abbreviated Form)~
             ~%~
             ~%   2.  (if case has features (X and Y) then ...)~
             ~%       (if case hf (X and Y) then ...)                (Abbreviated Form)~
             ~%~
             ~%   3.  (if exemplar has features (X and Y) then ...)~
             ~%       (if exemplar hf (X and Y) then ...)            (Abbreviated Form)"))

;;;-------------------------------------------------------------------

(defun print-node-syn ()
  (format t "~%SYNTAX OF TERMS:~
             ~%~
             ~%-- For a single term, type the term itself, as in:~
             ~%       (BOILING enables EVAPORATION)~
             ~%~
             ~%-- For multiple conjoined terms, type them in an \"and\" list, as in:~
             ~%       ((HEAT and HUMIDITY) causes MUGGY-WEATHER)~
             ~%~
             ~%-- Terms may also be expressed as a predicate with arguments, as in:~
             ~%       (INFECTION usually causes (FEVER MODERATE))"))

;;;-------------------------------------------------------------------

(defun print-quan-syn ()
  (format t "~%SYNTAX OF QUANTIFIERS:~
             ~%~
             ~%    Quantifiers are adjectives that modify the strength of~
             ~%    the verb that follows.  Quantifiers are optional, and~
             ~%    any number of them may precede a verb.  The quantifiers~
             ~%    and their abbreviations are shown below:~
             ~%~
             ~%    Type         Abbr.      Full name~
             ~%    ---------    -----    --------------~
             ~%    Context:      al       always~
             ~%                  us       usually~
             ~%                  st       sometimes~
             ~%                  occ      occasionally~
             ~%~
             ~%    Strength:     str      strongly~
             ~%                  mod      moderately~
             ~%                  we       weakly~
             ~%~
             ~%    Belief:       cert     certainly~
             ~%                  prob     probably~
             ~%                  poss     possibly~
             ~%~
             ~%    Temporal:     ins      instantly~
             ~%                  qu       quickly~
             ~%                  grad     gradually"))

;;;-------------------------------------------------------------------

(defun print-rel-syn ()
  (format t "~%SYNTAX OF RELATIONS:~
             ~%   A relation links a set of from-terms with a set of to-terms,~
             ~%   as in \"(fire causes heat)\".  The relation may be entered by~
             ~%   typing its full name or abbreviation.  The list below shows~
             ~%   all legal relations:~
           ~%~%                        RELATIONS~
             ~%                     --------------~
           ~%~%ABBR      VERB                        ABBR      INVERSE~
             ~%----      -------------               ----      -------------~
             ~%")
  (print-listx rel-print-list)
  (terpri))

;;;--------------------------------------------------------

(defun print-line (line size)
  (cond ((null (car line)) (terpri))
    (t (cond
	 ((= 0 (mod size 2))
	  (format t "~28,1,1A" (car line)))
	 (t 
	  (format t "~10,1,1A" (car line))) )
       (print-line (cdr line) (+ 1 size)) )))

;;;--------------------------------------------------------------------

(defun print-listx (lis)
  (cond 
    ((null lis) nil)
    (t (print-line (car lis) 1)
       (print-listx (cdr lis))) ))

;;;---------------------------------------------------------------------

(defun overview-language ()
  (format t "~%~%~66,1,0,'\~A" '\~)
  (format t "~%GENERAL SYNTAX OF EXPLANATIONS:~
             ~%~
             ~%   CONDITION    TERM(s)    QUANTIFIER(s)    RELATION    TERM(s)~
             ~%   (optional)              (optional)~
             ~%~
             ~%~
             ~%EXAMPLES:~
             ~%~
             ~%    (A causes B)~
             ~%    (A usually causes B)~
             ~%    (A sometimes weakly suggests B)~
             ~%    ((A and B) causes C)~
             ~%    (if category is CARS then RUBBER part of TIRES)~
             ~%    (A enables B affects C has function D)")
  (format t "~%~66,1,0,'\~A" '\~))


;;;---------------------------------------------------------------------
;;;     Function:     (nthcar ctr lis)
;;;     
;;;     Given:        two values:
;;;                   1)  the number of cars to take off.
;;;                   2)  the list on which to take them off.
;;;
;;;     Returns:      a list of the "ctr" number of cars taken off "lis"
;;;---------------------------------------------------------------------  
(defun nthcar (ctr lis)
  (cond ((= ctr 0) nil)
	(t         (cons (car lis) (nthcar (- ctr 1) (cdr lis))))))
    
;;;---------------------------------------------------------------------
;;;
;;;  Function:    (test-term-list objects)
;;;  
;;;  Given:       a list of objects
;;; 
;;;  Returns:     T if list of objects is syntactically valid.
;;;               NIL otherwise.
;;;
;;;---------------------------------------------------------------------

(defun test-term-list (terms)
  (cond ((numberp terms)   nil)
	((symbolp terms)   t)
	((listp   terms)   (test-arg-list terms))))   ;;changed predicate to arg

(defun test-predicate-list (pred)
  (if (not (symbolp (car pred)))
      nil
      (test-arg-list (cdr pred))))

(defun test-arg-list (args)
  (cond ((null args)           t)
	((listp (car args))    (and (test-predicate-list (car args))
				    (test-arg-list (cdr args))))
	(t                     (test-arg-list (cdr args)))))
	  

;;;---------------------------------------------------------------------
;;;	      
;;;  Function:  (test-quan e-list)
;;;
;;;  Given:     a list of explanations, beginning with a quantifier
;;;             or a relation.
;;;  
;;;  Returns:   e-list minus all words that were used to form valid
;;;             quantifiers if valid quantifiers were found.
;;;             The original e-list is returned if no valid quantifiers 
;;;             were found.
;;;
;;;---------------------------------------------------------------------

(defun test-quan (e-list)
  ;; try to get a quantifier.
  ;; new-list = e-list minus words used to form the quantifier.
  (multiple-value-bind (new-list quan) (process-phrase e-list quan-list)
    (cond
      ;; if no quantifier was found retun the explanation list.
      ((null quan) e-list) 
      ;; otherwise, try to find more quantifiers.
      (t (test-quan new-list)) )))
      
;;;--------------------------------------------------------------------
;;;
;;;  Function:   (test-keyword e-list a-list)
;;;
;;;  Given:      1) the list of explanations on which to search 
;;;                 for a keyword.
;;;              2) the a-list on which the keyword may be found
;;;
;;;  Returns:    three values:
;;;              1)  The e-list minus all words that formed the
;;;                  keyword, if a keyword was found.  Otherwise,
;;;                  the original e-list is returned.
;;;              2)  The keyword if it was found. Otherwise,
;;;                  NIL is returned.
;;;              3)  All the words searched through if the
;;;                  keyword was NOT found. If the keyword was
;;;                  found , NIL is returned.
;;;
;;;--------------------------------------------------------------------

(defun test-keyword (e-list a-list)
  ;; try to get a keyword
  (multiple-value-bind (keyword ctr )
    (get-name2 e-list a-list 1)
  
  (if (null keyword)
         ;; if no keyword found:
         (values e-list nil (nthcar  ctr  e-list))
         ;; if keyword found:			 
	 (values (nthcdr ctr e-list) keyword nil)) ))



;;;--------------------------------------------------------------------
;;;  Function:    (term-error terms)
;;;  
;;;  Given:       a list of terms
;;;
;;;  Does:        prints a message that the term syntax is wrong 
;;;--------------------------------------------------------------------
(defun term-error (terms)
  (format t "~%   ERROR -- possible illegal term or set of terms.~
             ~%   Check phrase: ~A" terms))
  


;;;----------------------------------------------------------------------
;;;  Function:  (first-pass e-list)
;;;
;;;  Given:     a list of words that comprises the explanation;
;;;
;;;  Returns:   -- nil if syntax error in the list, else
;;;             -- non-nil if syntax OK.
;;;  
;;;  Does:      prints a message indicating the first error encountered,
;;;             if any.
;;;----------------------------------------------------------------------
  
(defun first-pass (e-list)
  ;; test if there is any reasonable possibility that
  ;; a conditional was intended to be created.
  (cond ((or (member (car  e-list) cond-words) 
             (member (cadr e-list) cond-words))
	 
	 ;; if a conditional was intended, try to find
	 ;; a keyword.	 
	 (multiple-value-bind (new-list condtype error)
	     (test-keyword e-list condition-list)
	   (declare (ignore condtype))
	   
	   ;; if not found, then print error.			 
	   (cond (error
		  (format t "~%   ERROR -- possible misspelled conditional.~
		             ~%   Check phrase: ~A" error)
		  nil)
		 (t
		  ;; if conditional keyword found, test if its
		  ;; features or category is syntactically valid.
		  (cond ((test-term-list (car new-list))
			 ;; test whether to skip the word "then"
			 (if (equal (cadr new-list) 'then)
			     (first-pass2 (cddr new-list))
			     (first-pass2 (cdr new-list))))
			(t 
			 ;; if features or category not 
			 ;; syntactically valid, return nil.
			 (term-error (car new-list))
			 nil)) ))))
	(t
	 ;; if no conditional was intended, then 
	 ;; go on to first-pass2.
	 (first-pass2 e-list))))

;;;-----------------------------------------------------------
;;; 
;;;  Function:   (first-pass2 e-list)
;;;			
;;;  Given:      a list which comprises the explanation, 
;;;              minus any words that may be used to form
;;;              a condition.
;;;
;;;  Does:       prints a message indicating the first error 
;;;              encountered, or message indicating
;;;              no error was found.
;;;
;;;------------------------------------------------------------

(defun first-pass2 (e-list)
  ;; test syntactic validity of predicate nodes.
  (cond ((null (test-term-list (car e-list)))
         (term-error (car e-list))
	 nil)
	(t
	 ;; remove words that comprise quantifiers,
	 ;; and test for a valid relation.
	 (multiple-value-bind (new-list rel-type error)
	     (test-keyword (test-quan (cdr e-list)) rel-list)
	   (declare (ignore rel-type))
	   
	   (cond
	     ;; if no relation was found, print error message.
	     (error
	      (format t "~%   ERROR -- relation or quantifier unrecognized.~
                         ~%   Check phrase: ~A" error)
	      nil)
	     (t 
	      
	      (cond
		;; if relation was found, test syntactic
		;; validity of consequent nodes
		((null (test-term-list (car new-list)))
		 (term-error (car new-list))
		 nil)
		(t 
		 ;; test if at end of sentence.
		 ;; if so, then sentence has passed first-pass successfully.
		 ;; otherwise, keep testing.
		 (if (endp (cdr new-list))
		     t
		     (first-pass2 new-list) )) )) )) )))


