;;; -*- Syntax: Common-lisp; Package: USER -*-
;;;     Copyright (c) Daniel L. Dvorak


(defconstant  *hostname*  "maya")		       			   ; <==== EDIT
(defconstant  *dirname*   "/v/maya/v0/eilerts/user/")   ; <==== EDIT
(defconstant  *srctype*   ".lisp")						   ; <==== EDIT
(defconstant  *bintype*   ".o")                              ; <==== EDIT

;;;----------------------------------------------------------------------------
;;;  Notes:      -- First, edit the above constants, as follows:
;;;
;;;		    *hostname* is a string containing the name of the host
;;;		    on which the CL-Protos directory resides.  If it is the
;;;		    same as the host on which you are running, then it can
;;;		    probably be nil.
;;;
;;;		    *dirname* is a string containing the name of the CL-Protos
;;;		    directory, complete with any trailing delimiter such as
;;;		    "/" or ">" that must precede a file name.
;;;
;;;		    *srctype* is the type suffix for source files, typically
;;;		    ".lisp",".lsp", or ".l", depending on the conventions of
;;;		    your system.
;;;
;;;                 *bintype* is the type suffix for binary files, typically
;;;                 ".bin" or ".b", depending on the conventions of your
;;;                 system.
;;;
;;;		-- To load CL-Protos, load this file and it will do the rest
;;;                 (i.e., it will invoke the 'load-protos' function that is
;;;                  defined here, and it will do the rest).
;;;
;;;		-- If your Lisp environment supports a "system" facility (such
;;;		   as 'defsystem', as provided on Symbolics and TI Explorer),
;;;		   then you may wish to use that instead of this portable
;;;		   loading facility.
;;;----------------------------------------------------------------------------


(defconstant  *protos-filenames*
	      '("global"
		"defs"
		"verb"
		"kbio"
		"io"
		"example"
		"print"
		"parse"
		"kbpm"
		"heuristics"
		"merging"
		"reminding"
		"explanation"
		"discuss"
		"classify"
		"transform"
		"protos"))

;;;----------------------------------------------------------------------------
;;;  Function:   (load-protos)
;;;
;;;  Purpose:    This function loads the various files of CL-Protos in a
;;;              deliberate order.  The files are expected to be on host
;;;		 *hostname* in directory *dirname*.
;;;
;;;  Execution:  This function is invoked automatically when this file is
;;;              loaded (see the last line of this file).
;;;----------------------------------------------------------------------------

(defun load-protos ()
  (let (action typename fullname first-pathname)

    ;; Clear all buffered input.  Since clear-input doesn't always seem to
    ;; work, the loop below should always work.
    (clear-input *query-io*)
    (loop
      (if (listen *query-io*)
	  (read-char *query-io* nil nil nil)
	  (return (values))))

    (format *query-io*
	"~%~%Type the single character corresponding to what you want to do:~
           ~%---------------------------------------------------------------~
           ~%C   Compile the CL-Protos source files.~
           ~%S   load the Source files for interpretative execution.~
           ~%B   load the Binary files for execution.~
           ~%Q   Quit (do nothing).~
         ~%~%---> ")

    (setq action (char-upcase (read-char *query-io* nil nil nil)))
    (case action
      ((#\C #\S)       (setq typename *srctype*))
      (#\B             (setq typename *bintype*))
      (#\Q             (return-from load-protos (values)))
      (otherwise       (format *query-io* "~%Unrecognized action; quitting.")
		       (return-from load-protos (values))))

    ;; Generate pathname of first file.
    (setq fullname (format nil "~A~A~A"
			   *dirname* (first *protos-filenames*) typename))
    (format t "~%Probing ~A on host ~A" fullname *hostname*)
    (setq first-pathname  (parse-namestring fullname *hostname*))

    ;; If the first file is not found in the given directory ...
    (if (not (probe-file first-pathname))
	;; else ask user to enter new directory and host.
	(progn
	  (format *query-io*
		  "~%ERROR: Cannot find file ~A~
		   ~%The files are expected to be on host ~A~
		   ~%in directory ~A~
		   ~%with type suffix ~A~
		   ~%If this is incorrect, please edit load-protos.lisp"
		   fullname *hostname* *dirname* typename)
	  (return-from load-protos nil)))

    (format t "~%~%~D files will be ~:[loaded~;compiled~] from host ~:@(~A~), directory ~A~
                 ~%This may take a few minutes.~%"
	    (length *protos-filenames*) (char= #\C action) *hostname* *dirname*)

    ;; Load/compile each file of CL-Protos.
    (dolist (name *protos-filenames*)
      (let* ((fullname   (format nil "~A~A~A"
				*dirname* name typename))
	     (pathname2  (parse-namestring fullname *hostname*))
	     (binname    (format nil "~A~A~A" *dirname* name *bintype*))
	     (binpath    (parse-namestring binname *hostname*)))

	;; The reason for the following format statement, rather than
	;; just using ":verbose t" in the load call, is that on HP 9000
	;; systems that causes everything in the file to be printed.
	;; All I want here is a 1-line note that the file is being loaded.
	(format t "~%~:[Loading~;Compiling~] ~A"
		(char= #\C action)
		(namestring pathname2))

	(case action
	  (#\C        (compile-file pathname2)
	              (load binpath    :verbose nil  :print nil))
	  ((#\S #\B)  (load pathname2  :verbose nil  :print nil))
	  (otherwise  (format t "~%Unknown action!")))))

    (format t "~%~%CL-Protos load completed.  To run CL-Protos, enter:~
                 ~%   (in-package 'protos)~
                 ~%   (protos)~%")
  )
)


(load-protos)       ; execute the function to load all Protos files.
