;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Rita T. Duran and Daniel L. Dvorak

(in-package 'protos)


;;;-----------------------------------------------------------------------------
;;; Version 22d (as modified by Dan & Rita)
;;;    This version divided generate-successors into 2 main subordinate 
;;;    functions (add-related-nodes, add-transformed-nodes) and one utility
;;;    function (add-child-node).  Also, add-to-open was rewritten.
;;;
;;; Problems that still may need fixing are the following:
;;;    1.  What to do about successors which already appear on limbo
;;;    2.  Generate backsiblings
;;;
;;;-----------------------------------------------------------------------------



;;;-----------------------------------------------------------------------------
;;;
;;; Global variables used by the knowledge-based pattern matcher
;;;
;;;-----------------------------------------------------------------------------

(defvar *kbpm-mode*)
(defvar *category*)
(defvar *exemplar*)
(defvar *target*)
(defvar *case-features*)
(defvar *open*              nil)
(defvar *closed*            nil)
(defvar *limbo*             nil)
(defvar *current-tree*      0)
(defvar *tree-counter*      0)
(defvar *category-siblings* nil)
(defvar *fault-variables*   nil)
(defvar *ignore-conditions* nil)


;;;-----------------------------------------------------------------------------
;;; Structure:  graphnode
;;;
;;; Purpose:    The kbpm algorithm constructs a search graph as it searches for
;;;             a match between a feature and the newcase or between a feature 
;;;             and a category.  This data structure provides a place to store
;;;             information needed by the search algorithm.
;;;
;;;-----------------------------------------------------------------------------

(defstruct graphnode 
   "structure for nodes on the open and closed lists" 
   name               ; pointer to node in the category network
   (downstrength 1.0) ; strength of path from start node to this node
   (certainty 1.0)    ; strength of path propagated upwards from goal node
                      ;    to this node
   (parent nil)       ; parent node (if any)
   (status 'unknown)  ; current status of the node:  
                      ;   'solved       if the node is solved
                      ;   'spurious     if the node is solved, but on a spurious
                      ;                 path
		      ;   'failed       if the node does not lead to a solution
                      ;   'unknown      if the node's status not yet determined
   (tree   0)         ; number of search tree to which this node belongs
   (num-relations 0)  ; number of relations which emanate from this node
   (forward t)        ; direction of search from this node
                      ;        t   --> forward  (toward newcase in 'FtoF)
                      ;        nil --> backward (toward exemplar in 'FtoF)
   (direction nil)    ; Direction of reasoning leading up to this node:
                      ;   nil        -- all verbs so far are non-directional;
                      ;   'forward   -- verbs are going from cause to effect;
                      ;   'backward  -- verbs are going from effect to cause.
   (limbo  nil)       ; list of graphnode which is on *limbo* until this node is solved
   (marked nil)       ; set to t if this node on solution path
   (link nil)         ; graphlink from parent to this node
   (best-child-link nil) ; best outgoing link 
)


(defun print-graphnode (gnode stream depth)
  (declare (ignore depth))
  (let ((gname  (graphnode-name gnode))
	(parent (graphnode-parent gnode))
	(link   (graphnode-link gnode)))
    
   (format stream "~%   ~(~24A~) ~2A ~(~7A~) ~(~24A~) ~(~4A~) ~4,2F ~4,2F ~2A ~3A ~3A"
	   (if (node-p gname)
	       (getname gname)
	       gname)
	   (graphnode-tree gnode)
	   (graphnode-status gnode)
	   (if parent
	       (getname (graphnode-name parent))
	       "")
	   (if link
	       (getname (relation-verb (graphlink-relation link)))
	       "")
	   (graphnode-downstrength gnode)
	   (graphnode-certainty gnode)
	   (graphnode-num-relations gnode)
	   (if (graphnode-forward gnode)
	       "-->"
	       "<--")
	   (cond ((null (graphnode-direction gnode))          "")
		 ((eql (graphnode-direction gnode) 'forward)  "-->")
		 ((eql (graphnode-direction gnode) 'backward) "<--")
		 (t (graphnode-direction gnode))))
     
    (if (graphnode-marked gnode)
	(princ " Marked" stream))
    
    (if (graphnode-limbo gnode)
      (progn (format stream "~%   Limbo Nodes:   ")
	     (dolist (l (graphnode-limbo gnode))
	       (format stream "~@(~A~) " (getname (graphnode-name l)))
	       )))

     ))


;;;-----------------------------------------------------------------------------
;;; Structure:  graphlink
;;;
;;; Note:       You may wonder why this structure contains slots for
;;;             "from-nodes" and "to-nodes" when these could be obtained from
;;;             the relation that is pointed to by the "relation" slot.  The
;;;             reason is that, in the case of transforming the arguments of
;;;             a predicate, the nodes will be different than the nodes in the
;;;             relation.  For example, the graphnode "(fever mild)" can be
;;;             transformed to "(fever moderate)" by the relation
;;;             "mild equivalent to moderate".  The relation used here has 
;;;             the to-node "moderate" when we really want the to-node
;;;             "(fever moderate)".  Thus, we explicitly store the appropriate
;;;             from-nodes and to-nodes in this graphlink structure.
;;;-----------------------------------------------------------------------------

(defstruct graphlink
    "structure for links within the search graph created by kbpm"
    relation          ; ptr to relation within category network
    from-nodes        ; from-nodes of this link
    to-nodes          ; to-nodes of this link
    (is-and nil)      ; t for and-relations
    (num-siblings 0)  ; number of sibling nodes within an and-relation
    (parent nil)      ; ptr to parent node

)

(defun print-graphlink (glink stream depth)
  (declare (ignore depth))
  (prog (parent)
    
    (format stream "~&Name: ~@(~A~)"
      (getname (relation-verb (graphlink-relation glink))))
    
    (setq parent (graphlink-parent glink))
    (if parent
      (format stream "~%   Parent:   ~A"(getname (graphnode-name parent)))
      (format stream "~%   Parent:  None"))
    
    (if (graphlink-is-and glink)
        (progn (format stream "~%   Type of Link:  And")
	       (format stream "~%   Number of Siblings: ~A"
                    (graphlink-num-siblings glink)))

	(format stream "~%   Type of Node:  Or"))
    
    ))


;;;-----------------------------------------------------------------------------
;;;
;;; Function:  (kbpm  mode  feature  importance  object  case-features)
;;;
;;; Given:     -- mode, whose value is either:
;;;               'FtoF  (search from feature of the exemplar to feature of
;;;                         the new case), or
;;;               'FtoT  (search from feature of new case to target category
;;;                         or exemplar).
;;;            -- feature, the feature that is the starting point of this search.
;;;            -- importance, the importance of this feature (which controls the
;;;               amount of effort to be expended while attempting a match).
;;;            -- object, which is an exemplar if mode = 'FtoF and is a
;;;               category or exemplar if mode = 'FtoT.
;;;            -- case-features, the list of features of the new case.
;;;
;;; Returns:   A 'result' structure containing feature, importance, type of
;;;            match, quality of match, and explanation (if any).
;;; 
;;;            where type of match can be one of the following:
;;;               'identical      this feature exists in the new case
;;;               'explained      the two features are related by an explanation
;;;               'spurious       the feature is spurious
;;;               'excluded       this feature is mutually excluded from newcase
;;;               'unmatched      the feature couldn't be matched to newcase
;;;
;;;            similarity is the strength of the match
;;;            explanation is an instance of an explanation structure
;;;
;;; Called-by: compare-feature
;;;
;;; Notes:     1.  If mode is 'FtoT the resulting explanation links the feature
;;;                of the newcase to the target category or exemplar.
;;;                If mode is 'FtoF, the resulting explanation links a feature
;;;                of the newcase to the feature of the exemplar.  Since
;;;                uniform-cost-search returns an explanation from the exemplar
;;;                to the newcase, the returned explanation must be inverted
;;;                before it is stored in the 'result' structure.
;;;
;;; ----------------------------------------------------------------------------

(defun kbpm  (mode  feature  importance  object  case-features)

  ;; Set the global parameters that the kbpm routines depend upon.
  (setq *kbpm-mode*     mode
        *target*        object
	*case-features* case-features)

  (if (eql *kbpm-mode* 'FtoF)
      ;; For feature-to-feature search, make exemplar and category global.
      (setq *exemplar* object
	    *category* (exemplar-category *exemplar*))
      ;; For feature-to-target search, set exemplar and category depending
      ;; on type of target
      (if (my-exemplar-p object)
	(setq *exemplar* object
	      *category* (exemplar-category *exemplar*))
	(setq *exemplar* nil
	      *category* object)))
  
  ;; Check for identical match.
  (if (goal-node? feature 'forward)
      (return-from kbpm (make-result :feature    feature
				     :importance importance
				     :type       'identical
				     :quality    1.0)))
  
  ;; Check for spurious feature.
  (if (<= importance 0.0)
      (return-from kbpm (make-result :feature    feature
				     :importance importance
				     :type       'spurious
				     :quality    1.0)))
  
  ;; Initialize search.
  (setq *open*     (list (make-graphnode :name feature))
	*closed*   nil
	*limbo*    nil
	*category-siblings* (get-sibling-categories)
	*current-tree* 0
	*tree-counter* 0)

  (let ((threshold  (calculate-threshold importance))
	(success    nil)
 	(start      (car *open*)))
    
    ;; Call recursive algorithm to search the graph.
    (setq success (uniform-cost-search start threshold))
    

    ;; If search was successful ...
    (if success
      
        ;; then retrieve explanation and quality
        (multiple-value-bind (etype explain strength)
	  (retrieve-explanation start 1.0)

	  (if *trace-kbpm*
	    (progn (terpri)
		   (print-explanation explain)
		   (format t ", strength = ~4,2F" (explanation-strength explain))
		   (terpri)
		   (if (eql mode 'FtoF)
		     (print-explanation (invert-explanation-2 explain
							      nil nil nil)))))
          (if (eql mode 'FtoF)
	    (let* ((inv-explanation (invert-explanation-2 explain nil nil nil))
		   (i-strength (compute-explanation-strength inv-explanation)))
	      (make-result :feature     feature
			   :importance  importance
			   :type        etype
			   :quality     i-strength
			   :explanation inv-explanation))
	    ;;else FtoT
	    (make-result :feature     feature
			 :importance  importance
			 :type        etype
			 :quality     strength
			 :explanation explain))

	  )

	;; else report the feature as being unmatched.
	(make-result :feature    feature
		     :importance importance
		     :type       'unmatched
		     :quality    (- 1.0 importance)))
    ))

;;;-----------------------------------------------------------------------------
;;; Function: (get-sibling-categories)
;;; 
;;; Given:    *category*, global variable
;;;
;;; Returns:  a list of nodes in the category network which are siblings of
;;;           category. 
;;;
;;; Design:   First, find the parent(s) of *category*, that is, all nodes which
;;;           are generalizations of *category*.  Then, find the children of
;;;           each parent, that is, nodes which are specializations of the 
;;;           parent.  These children (except for *category* itself) are the
;;;           siblings of category.
;;;           For example, if both dog and wolf have typical gen canine, then
;;;           wolf is a sibling of the category dog.
;;;            
;;;-----------------------------------------------------------------------------
    
(defun get-sibling-categories ()
  (if *category*
      (let ((relations (category-relations *category*))
	    (parents   nil)
	    (siblings  nil))
	
	;; Find parent(s) of category (may be 0 or more)
	(dolist (rel relations)
	  (if (eq *verb-hasTypicalGen* (relation-verb rel))
	      (setq parents (append (relation-to-nodes rel) parents))))
	
	;; For each parent, find all its children.
	;; these are the siblings of *category*
	(dolist (p parents)
	  (dolist (rel (category-relations p))
	    (if (eq *verb-hasTypicalSpec* (relation-verb rel))
		(setq siblings (append (relation-to-nodes rel) siblings)))))
	
	;; Delete the category itself from the list of siblings.
	(delete *category* siblings))))


;;; ----------------------------------------------------------------------------
;;; Function:  (calculate-threshold importance)
;;;
;;; Given:     importance,  the importance of an exemplar feature
;;;  
;;; Returns:   a floating point value in the range .05 to 1.0, representing
;;;            a threshold at which to cut off search.  The more important a
;;;            feature is, the lower the threshold for cutting off the search.
;;;
;;; Called-by: kbpm
;;;
;;; Design:    The raw threshold is calculated to be (1.0 - importance + .05). 
;;;            If the raw threshold is greater than 1.0, it is set back to 1.
;;;
;;; ----------------------------------------------------------------------------

(defconstant *minimum-feature-threshold* 0.05)

(defun calculate-threshold (importance)
  (let ((raw (- 1.0 importance)))
    (if (<= raw *minimum-feature-threshold*)
	*minimum-feature-threshold*
	raw)))


;;; ----------------------------------------------------------------------------
;;; Function:  (uniform-cost-search  start  threshold)
;;;
;;; Given:     start    ,  start node
;;;            threshold,  value at which the search will be cut off
;;;            *open*,     global variable which contains a list of nodes on
;;;                        the frontier of the search graph
;;;            *closed*,   global variable containing nodes which have already
;;;                        been explored
;;;
;;; Returns:   a flag indicating whether or not the search was successful,
;;;            either 'explained or NIL.
;;;
;;; Called-by: kbpm
;;;
;;; Design:    The algorithm is a uniform-cost search of an and-or graph.
;;;            It uses the strength of the explanation path to a node to
;;;            evaluate the node.
;;;
;;; -----------------------------------------------------------------------------

(defun uniform-cost-search (start threshold)

  (let ((status  (graphnode-status start)))

    (cond

      ;; If start node solved (or solved but spurious), or if start node is
      ;; spurious and no outgoing relations left, then we are done.
      ((or (eql status 'solved)
	   (and (eql status 'spurious)
		(< (graphnode-num-relations start) 1)))
       'explained)
    
      ;; if start node has failed, or if open list is empty, then fail.
      ((or (eql status 'failed)
	   (null *open*))
       nil)
    
      ;; otherwise
      (t
       (if *trace-kbpm*
	   (progn (terpri)
		  (print-open-list)
		  (print-closed-list)
		  (print-limbo-list)))

       ;; Remove graphnode n from open list.
       (let* ((n          (pop *open*))
	      (strength   (graphnode-downstrength n))
	      (direction  (graphnode-forward n)))

	 (setq *current-tree* (graphnode-tree n))

	 (cond
	   ;; is n a dummy-node?
	   ((eql (graphnode-name n) 'dummy-node)
	    (setf (graphnode-certainty n) strength)
	    (if (> strength threshold)
		(progn (setf (graphnode-status n) 'spurious)
		       (add-to-closed n)
		       (propagate-spurious n direction)
		       (uniform-cost-search start threshold))
		(progn (setf (graphnode-status n) 'failed) 
		       (add-to-closed n)
		       (propagate-failure n direction)
		       (uniform-cost-search start threshold))))

	   ;; is n the goal node?
	   ((goal-node?  (graphnode-name n) direction)
	    (setf (graphnode-certainty n) strength)
	    (if (>= strength threshold)
		(progn
						;(print 'foundgoalnode)
		  (setf (graphnode-status n) 'solved)
		  (add-to-closed n )
		  (propagate-success n direction)
		  (uniform-cost-search start threshold))

		(progn (print 'foundgoal--too--weak)
		       (format t "~%strength=~A, threshold=~A"
			       strength threshold)
		       (setf (graphnode-status n) 'failed)
		       (add-to-closed n)
		       (propagate-failure n direction)
		       (uniform-cost-search start threshold))))
	      
	   ;; generate the successors of n
	   (t (let ((children  (generate-successors n threshold)))
		(if (null children)		;if no successors then fail
		    (progn (setf (graphnode-status n) 'failed)
			   (add-to-closed n)
			   (propagate-failure n direction)
			   (uniform-cost-search start threshold))

		    (progn (add-to-closed n)
			   (uniform-cost-search start threshold)
			   ))))
	   ))
       ))))

;;; ----------------------------------------------------------------------------
;;; Function:  (goal-node?  node  forward-search)
;;;
;;; Given:     node,            a node of the category network;
;;;            forward-search,  flag indicating direction of the search;
;;;
;;; Returns:   non-NIL    if node is a goal node, or
;;;            NIL        otherwise.
;;;
;;; Algorithm:  If node is a dummy node, then search was successful.
;;;             For forward search, return t if n is a feature of the 
;;;             new case.
;;;             For backward search, return t if n is a feature of the
;;;             exemplar.
;;; ----------------------------------------------------------------------------

(defun goal-node? (node forward-search)

  (if *trace-kbpm*
      (format t "~%goal-node?: ~A" (getname node)))

  (case *kbpm-mode*
    (FtoF          (if forward-search
		       (member node *case-features*)
		       (member node (exemplar-features *exemplar*))))
    
    ((FtoT FtoC)   (if forward-search
                       (eq node *target*)
                       (member node *case-features*)))

    (otherwise     (cerror "treat node as unmatched"
			   "goal-node?: invalid *kbpm-mode*")
		   nil)))



;;;-----------------------------------------------------------------------------
;;; Function: (generate-successors  n  threshold)
;;;
;;; Given:    n, node on the search graph.  Type is graphnode.
;;;           threshold
;;;
;;; Returns:  t if any successors or back-siblings generated.
;;;           nil, otherwise
;;;-----------------------------------------------------------------------------

(defun generate-successors (n threshold)
  (let* ((num-relations  0)
	 (success        nil))
    (declare (special num-relations success))

    ;; Look for successors via direct relations from node n.
    (add-related-nodes     n threshold)

    ;; Look for successors via transformed arguments of node n,
    ;; if it is a predicate-with-arguments node.
    (add-transformed-nodes n threshold)
 ;(format t "~%JUST AFTER add-transformed-nodes")
 ;(print-open-list)

    ;; Look for successors via mathematical relations on variables.
    (add-variable-nodes n threshold)

    ;; Set count of number of relations found from graphnode n.
    (setf (graphnode-num-relations n) num-relations)

    ;; Return success if any siblings or children were generated.
    success))
    

    
;;;-----------------------------------------------------------------------------
;;; Function: (add-related-nodes  n  threshold)
;;;
;;; Given:    n, node on the search graph.  Type is graphnode.
;;;           threshold
;;;
;;; Returns:  nothing of significance, but if it adds any nodes to the search
;;;           graph, it sets the special parameter "success" to t and increments
;;;           the special parameter "num-relations".
;;;
;;; Purpose:  This function looks for successor nodes to the current node n
;;;           by traversing relations emanating from n.
;;;
;;; Design:   In the case of a spurious relation, create a node with
;;;           name 'dummy-node to be the child node.  This node must
;;;           be put on *open* in the correct position so that a spurious
;;;           explanation is only found if a better explanation does not
;;;           exist.
;;; Notes:
;;;     1.  This assumes that all from nodes and all to-nodes must be
;;;         proven.  Need to modify to check for active and passive verbs.
;;;     2.  This assumes that the spurious relation is the only relation
;;;         in the explanation which has no to-nodes.
;;;-----------------------------------------------------------------------------

(defun add-related-nodes (n threshold)
  (declare (special num-relations success))
  (let* ((node               (graphnode-name n))
	 (relations          (copy-list (node-relations node)))
	 (downstrength       (graphnode-certainty n))
	 (link-from-parent   (graphnode-link n)))	 ; how i got here


    ;; If certainty of this node has not already been determined,
    ;; then use graphnode-downstrength as strength of path to this node.
    (if (= downstrength 1.0)
	(setq downstrength (graphnode-downstrength n)))

    ;; If we get to node N from some "parent" node via g-relation, then we
    ;; must delete the inverse of g-relation from node N's relations to avoid
    ;; going back to that parent node.  That explains step 1 below.
    ;; Step 2 is relevant only when dealing with multiple antecedents.
    ;; For example, if we have the relation "(A and B) implies C", then step 1
    ;; will delete the inverse relation "C is implied by A and B".  When B
    ;; is processed as a backsibling of A, step 2 will delete the relation
    ;; "A and B implies C" so that B won't find C as a successor.

    (if link-from-parent
	(let ((g-relation (graphlink-relation link-from-parent)))
	  ;; Step 1.
	  (setq relations (delete (relation-inverse g-relation) relations))
	  ;; Step 2.
	  (setq relations (delete g-relation relations))))


    ;; LOOK FOR SUCCESSOR NODES BY TRAVERSING RELATIONS FROM THIS NODE.
    (dolist (rel relations)
       (if *trace-kbpm2*
           (progn
              (format t "~%Examining the relation:  ")
              (print-relation rel t 1)))
      ;; condition must be met
      (if (check-condition (relation-condition rel))
	  
	  (let* ((to-nodes        (relation-to-nodes rel))
		 (num-to          (length to-nodes))
		 (strength-vector nil)
		 (direction       (graphnode-forward n))
		 (num-from        (length (relation-from-nodes rel)))
		 (prove-all-to    (not (verb-to-proof (relation-verb rel))))
		 (prove-all-from  (not (verb-from-proof (relation-verb rel))))
		 )
	    
           ;;(format t "~% in add-related-nodes, to nodes:~%")
           ;;(dolist (x to-nodes) (print-node x t 1))
           ;;(format t "~%direction = ~A" direction)
	    ;; while debugging set prove-all variables to true
	    ;;(setq prove-all-to t)
	    ;;(setq prove-all-from t)
	    
	    ;; heuristically evaluate strength of childnodes
	    (setq strength-vector (evaluate-strength n to-nodes
						     rel downstrength))
	    
	    ;; if all children have same strength and it is less than
	    ;; threshold, then path too weak. (if not all the same,
	    ;; all-same returns infinity.
	   
	    ;; is path too weak?
	    (if (< (all-same strength-vector) threshold)
		;; go on to next relation
                (if *trace-kbpm2*
                    (format t "~%  Relation's strength (~A) below threshold (~A)" 
                              (all-same strength-vector) threshold))
		
		;; otherwise, strengths must be checked individually
		(let ((children  nil)
		      (newlink   nil))
		  (declare (special children))    ; make visible to add-child-node
		  
		  ;; create a graphlink for this relation
		  (if (and (>  num-to 1) prove-all-to)
		      ;; AND-link
		      (setq newlink (make-graphlink :relation rel
						    :from-nodes (relation-from-nodes rel)
						    :to-nodes (relation-to-nodes rel)
						    :parent n
						    :is-and t
						    :num-siblings num-to))
		      ;;OR-link
		      (setq newlink (make-graphlink :relation rel
						    :from-nodes (relation-from-nodes rel)
						    :to-nodes (relation-to-nodes rel)
						    :parent n
						    :num-siblings 0)))
		  
		  ;; If this relation has no consequents, i.e., a spurious relation ...
		  (if (null to-nodes)
		      ;; Then create a dummy child node
		      (progn
			(setq success t)
			(setq children (list (make-graphnode
					       :name 'dummy-node
					       :downstrength (car strength-vector)
					       :link newlink
					       :num-relations 0
					       :forward direction
					       :parent n
					       :tree *current-tree*))))

		      ;; Else create a child graphnode for each member of
		      ;; to-nodes whose strength is above the threshold.
		      (do ((child  to-nodes         (cdr child))
			   (str    strength-vector  (cdr str)))
			  
			  ((endp child))

			;; If strength of this child is above threshold ...
			(if (>= (car str) threshold)

			    ;; Then add it to the search graph.
			    ;; Note: add-child-node adds to children.
			    (add-child-node  (car child) (car str) newlink direction n)

			    ;; Else if this is one child of an and-link then
			    ;; we must abandon the link entirely.
			    (if (graphlink-is-and newlink) 
				(setq child    nil
				      children nil)))))

		  ;; If any children were generated from this relation ...
		  (if children
		      ;; Then report success, increment number of relations,
		      ;; and either add the children to the open list or,
		      ;; if there are backsiblings, put the children in limbo
		      ;; and put the backsiblings on the open list.
		      (progn
			(setq success t)
			(incf num-relations)
			(if (and (> num-from 1) prove-all-from)
			    
			    (generate-backsiblings n rel children)
			    (add-to-open children)))))))))))


;;;-----------------------------------------------------------------------------
;;; Function:  (add-child-node  node  strength  newlink  forward  n)
;;;
;;; Given:     -- node, a node of the category network to be added as a 
;;;               successor (a child) of the graphnode n;
;;;            -- strength, the strength of the path to this node;
;;;            -- newlink, the graphlink from the parent node n;
;;;            -- forward, T if searching forward toward goal node;
;;;            -- n, the graphnode that is the parent of this child.
;;;
;;; Returns:   nothing of significance, but it will add the new child node to
;;;            the list of child nodes (the special parameter "children"),
;;;            unless the child already exists on limbo.
;;;
;;; Purpose:   This function adds the given node as a child (a successor) of
;;;            the graphnode n.
;;;-----------------------------------------------------------------------------

(defun add-child-node (node strength newlink forward n)
  (declare (special children))   ;; 'children' is supplied by the caller.

  ;; Determine what direction of cause-and-effect must apply to this
  ;; new child node based on the direction of its parent and the
  ;; direction of newlink.
  (prog* ((relation   (graphlink-relation newlink))
	  (verb       (relation-verb relation))
	  (verbdir    (direction verb))
	  (parentnode (graphnode-name n))
	  (parentpred (term-predicate parentnode))
	  (parentdir  (graphnode-direction n))
	  (childdir   (or verbdir parentdir))
	  (pred       (term-predicate node))
	  old)

 ;(format t "~%add-child-node: ")
 ;(print-relation relation t 1)
 ;(format t " vdir= ~A  pdir= ~A  cdir=~A" verbdir parentdir childdir)

         (if (and parentpred
		  (member parentpred *fault-variables*)
		  (equal verbdir 'backward))
	     (return-from add-child-node (values)))

	 ;; A child node may be reached through a forward or a backward
	 ;; direction of reasoning.  For example, in "A causes B", B is
	 ;; reached in a forward direction, whereas in "C caused by D",
	 ;; D is reached in a backward direction.  The direction of the
	 ;; the child node is determined from the verb that got us to the
	 ;; child node.  If the verb is non-directional (such as "eq")
	 ;; then the direction is taken from the parent node.  This
	 ;; explains the above binding: (childdir (or verbdir parentdir)).
    
	 ;; If this node to be added is one of the current "fault variables",
	 ;; then make sure we didn't get here in the forward direction since,
	 ;; by definition, a fault variable cannot respond to its inputs.
	 (if (and pred
		  (member pred *fault-variables*)
		  (equal childdir 'forward))
	     (return-from add-child-node (values)))

	 (if (and relation (or *trace-kbpm* *trace-kbpm2*))
	     (progn
	       (terpri)
	       (princ "   ")
	       (print-relation relation t 1)))


	 ;; need to add case statement here
	 ;; if child already in closed
	 ;;     same tree --> ignore
	 ;;     different tree --> search it
	 ;; if child already on open
	 ;;     same tree
	 ;;         and-link --> search it
	 ;;         or-links -->
	 ;;           if stronger path found, reset parent
	 ;;                  else ignore
	 ;;     different trees
	 ;;         search it
	 ;; if child already on limbo
	 ;;     discard it
    
	 (cond 
      
	   ;; IS CHILD ALREADY IN CLOSED?
	   ((check-gnodes *closed* node childdir)
	    ;; If so, then just return.
	    (return-from add-child-node (values)))

      
	   ;; IS CHILD ALREADY IN OPEN?
	   ((setq old (check-gnodes *open* node childdir))
	    ;; If so, then 
	    (if (or (graphlink-is-and (graphnode-link old))
		    (graphlink-is-and newlink))
		(go ADD-CHILD)
		(if (> strength (graphnode-downstrength old))
		    (progn 
		      (setq *open* (delete old *open*))
		      (go ADD-CHILD))))
	    (return-from add-child-node (values)))
      

	   ;; IS CHILD ALREADY IN LIMBO?
	   ((check-gnodes *limbo* node childdir)
	    ;; If so, just return.
	    (return-from add-child-node (values)))
	   )

	 ;; The child node is not in *closed*, *open*, or *limbo*, so we want
	 ;; to create the child graphnode and add it to 'children'.

      ADD-CHILD
	 ;; Make graphnode for this child node.
	 (let ((childnode (make-graphnode :name          node
					  :downstrength  strength
					  :link          newlink
					  :num-relations 0
					  :forward       forward
					  :direction     childdir
					  :parent        n
					  :tree          *current-tree*)))
	   (push childnode children))
	 ))

;;;-----------------------------------------------------------------------------
;;;  Function:  (check-gnodes  gnodelist  childname  childir)
;;;
;;;  Purpose:   This function searches the given list of graphnodes to see if
;;;             any of those graphnodes matches the given child in name, in
;;;             search tree, and in direction of search.  If so, the function
;;;             returns the found node, otherwise nil.
;;;
;;;  Callers:   add-child-node
;;;-----------------------------------------------------------------------------

(defun check-gnodes (gnodelist child childdir)
  (let ((gnode nil))

    (do ((gnodes (member child gnodelist    :key #'graphnode-name)
		 (member child (cdr gnodes) :key #'graphnode-name)))
	((endp gnodes) nil)

      (setq gnode (car gnodes))
      (if (and (=   (graphnode-tree gnode) *current-tree*)
	       (or  (null (graphnode-direction gnode))
		    (eql  (graphnode-direction gnode) childdir)))
	  (return-from check-gnodes gnode)))))


;  Below is an older version of add-child-node that I haven't had the courage
;  to delete until I'm sure the new version is flawless...
;
;(defun add-child-node (node strength newlink direction n)
;  (declare (special children))
;
;  (let (childnode
;	oldnodes)
;
;    ;; Make graphnode for this child node.
;    (setq childnode (make-graphnode :name          node
;				    :downstrength  strength
;				    :link          newlink
;				    :num-relations 0
;				    :forward       direction
;				    :parent        n
;				    :tree          *current-tree*))
;    
;    ;; need to add case statement here
;    ;; if child already in closed
;    ;;     same tree --> ignore
;    ;;     different tree --> search it
;    ;; if child already on open
;    ;;     same tree
;    ;;         and-link --> search it
;    ;;         or-links -->
;    ;;           if stronger path found, reset parent
;    ;;                  else ignore
;    ;;     different trees
;    ;;         search it
;    ;; if child already on limbo
;    ;;     discard it
;    
;    (cond 
;      
;      ;; IS CHILD ALREADY IN CLOSED?
;      ((setq oldnodes (already-in-closed node))
;       ;;(print 'already-in-closed)
;       ;;
;       ;; Hmmm.  Shouldn't the dolist below push the childnode ONLY if
;       ;; NONE of the oldnodes match *current-tree* ?  Need to ask Rita.
;       (dolist (old oldnodes)
;	 (if (/= (graphnode-tree old) *current-tree*)
;	     (push childnode children))))
;      
;      ;; IS CHILD ALREADY IN OPEN?
;      ((setq oldnodes (already-in-open node))
;       ;(print 'already-in-open)
;       (dolist (old oldnodes)
;	 ;; if same tree
;	 (if (= (graphnode-tree old) *current-tree*)
;	     (if (or (graphlink-is-and (graphnode-link old))
;		     (graphlink-is-and newlink))
;		 (push childnode children)
;		 (if ( > (graphnode-downstrength childnode)
;		      (graphnode-downstrength old))
;		     (progn 
;		       (push childnode children)
;		       (setq *open* (remove old *open*)))))
;	     
;	     ;; different trees
;	     (push childnode children))))
;      
;      ;; IS CHILD ALREADY IN LIMBO?
;      ((setq oldnodes (already-in-limbo node))
;       ;(print 'already-in-limbo)
;       )
;
;      ;; OTHERWISE, ADD CHILD TO CHILDREN.
;      (t
;       (push childnode children)))
;    ))



;;;-----------------------------------------------------------------------------
;;; Function:  (add-transformed-nodes  node  threshold)
;;;
;;; Given:     node, a predicate-with-arguments node, and
;;;            threshold, a cutoff limit for search;
;;;
;;; Returns:   t if any nodes were added to the open list, else nil.
;;;
;;; Purpose:   The purpose of this function is to attempt transform the
;;;            arguments of a predicate-with-arguments node in order to add
;;;            successor nodes to the search.  For example, if we are given
;;;            the node "(fever mild)" and the knowledge base happens to have
;;;            the relation "mild is equivalent to moderate", then we can add
;;;            the successor node "(fever moderate)".
;;;
;;; Note:      This function will silently create new nodes in the knowledge
;;;            base as a result of transforming arguments.  For instance, in
;;;            the above example, if "(fever moderate)" did not exist in the
;;;            knowledge base before the call to this function, it WOULD exist
;;;            after the call.  Thus, a side-effect of calling this function
;;;            is that it creates nodes for all possible transformations of the
;;;            given node.  This was done because the existing kbpm functions
;;;            expect to point to "real" nodes.  This side-effect is 
;;;            undesirable only in the sense of memory usage.
;;;-----------------------------------------------------------------------------

(defun add-transformed-nodes (n threshold)
  (declare (special num-relations success))
  (let ((node          (graphnode-name n))
	(direction     (graphnode-forward n))
	(downstrength  (graphnode-certainty n)))
    
    ;; If this node is not a predicate-with-arguments term, then just return nil.
    ;(format t "~% add-transformed-nodes:")
    ;(format t "~%    node = ~A" (getname node))
    (if (not (and (term-p node) (term-predicate node)))
	(return-from add-transformed-nodes nil))
    
    ;; If certainty of this node has not already been determined,
    ;; then use graphnode-downstrength as strength of path to this node.
    (if (= downstrength 1.0)
	(setq downstrength (graphnode-downstrength n)))
    
    ;; For each argument of this predicate-with-arguments, find all 1-step
    ;; transformations of that argument.  For each such transformation,
    ;; create a new predicate-with-arguments node with the corresponding
    ;; transformed argument (if the node doesn't already exist) and add that
    ;; node as a new child node.
    (let ((termname  (term-name node)))
      
      ;; For each argument in the predicate-with-arguments ...
      (do ((i  1 (1+ i)))
	  ((>= i (length (term-name node))))
	(let ((argname  (nth i termname)))
	  ;; Skip arg if it is not a symbol or list (e.g., a number).
	  (if (or (symbolp argname) (listp argname))
	      (let ((argnode  (check-term-name argname 'create)))
		
		;; For each relation emanating from this argument node ...
		(dolist (rel (node-relations argnode))
		  (let* ((condition       (relation-condition rel))
			 (to-nodes        (relation-to-nodes rel))
			 (to-node         (car to-nodes))
			 (child-strength  (* downstrength (relation-strength rel))))
		    ;; If this relation either has no condition or a matched predicate
		    ;; condition, AND there is only one to-node, AND the strength of
		    ;; that node is above the threshold ...
		    (if (and (or (null condition)
				 (eq (term-predicate node) (condition-predicate condition)))
			     (null (cdr to-nodes))
			     (>= child-strength threshold))
			
			;; Then build the new "transformed" node and add it as a child.
			(let ((child-termname  (copy-list termname))
			      (children        nil)
			      child-node
			      newlink)
			  (declare (special children))	; make visible to add-child-node
			  (setf (nth i child-termname) (term-name to-node))
   ;;(format t "~%Add-transformed-nodes:  ~A ---> ~A" termname child-termname)
			  (setq child-node (check-term-name child-termname 'create))
			  (setq newlink (make-graphlink :relation rel
	;;;;;						:from-nodes (relation-from-nodes rel)
	;;;;;						:to-nodes to-nodes
							:from-nodes (list node)
							:to-nodes (list child-node)
        ;;;;;; really not sure about above lines
							:parent n
							:num-siblings 0))
			  ;; add-child-node may add to children.
			  (add-child-node child-node child-strength newlink direction n)
			  
			  ;; If any children were generated from this relation ...
			  (if children
			      ;; Then report success, increment number of relations,
			      ;; and add the child to the open list.
			      (progn
				(setq success t)
				(incf num-relations)
				(add-to-open children))))))))))))))



;;;-----------------------------------------------------------------------------
;;; Function:  (add-variable-nodes  node  threshold)
;;;
;;; Given:     node, a predicate-with-arguments node, and
;;;            threshold, a cutoff limit for search;
;;;
;;; Returns:   t if any nodes were added to the open list, else nil.
;;;
;;; Purpose:   The purpose of this function is to add successor nodes to the
;;;            search that are "mathematically" related to the given node.
;;;            For example, if we are given the node "(pulse elevated)" and
;;;            we have the mathematical relation "pulse fm+ aortic-flow",
;;;            then we can add the successor node "(aortic-flow elevated)".
;;;
;;; Notes:     The word "variable" is used in this function to refer to a
;;;            predicate that has mathematical relations to other variables.
;;;            For example, "pulse" is a variable, and its value is
;;;            "elevated" in the term "(pulse elevated)".
;;;
;;;            This function will silently create new nodes in the knowledge
;;;            base as it instantiates mathematical relations.  For example,
;;;            in the above example the node "(aortic-flow elevated)" would
;;;            be created if it did not already exist.  This was done because
;;;            the existing kbpm functions expect to point to "real" nodes.
;;;            This side-effect is undesirable only in the sense of memory
;;;            usage.
;;;-----------------------------------------------------------------------------

(defun add-variable-nodes (n threshold)
  (declare (special num-relations success))
  (let ((node          (graphnode-name n))
	(direction     (graphnode-forward n))
	(downstrength  (graphnode-certainty n))
	predicate)
    
    ;; If this node is not a predicate, then just return nil.
    (if (not (and (term-p node)
		  (setq predicate (term-predicate node))))
	(return-from add-variable-nodes nil))
    
    ;; If certainty of this node has not already been determined,
    ;; then use graphnode-downstrength as strength of path to this node.
    (if (= downstrength 1.0)
	(setq downstrength (graphnode-downstrength n)))
    
    ;; For each relation connected to this variable, make a child node of
    ;; the related variable, instantiated with a value consistent with the
    ;; relation.  If necessary, create the needed term node.
    (let* ((termname  (term-name node))
	   (qmag    (second termname))
	   (qdir    (third  termname))
	   termname2
	   varname2
	   qmag2
	   qdir2
	   child-strength)

      ;; For each relation connected to this variable ...
      (dolist (rel (predicate-relations predicate))
	(setq child-strength (* downstrength (relation-strength rel)))
	(if (>= child-strength threshold)
	    (let ((verb    (relation-verb rel)))
	      (if (and (consistent-direction verb n)
		       (not (repeated-predicate (car (relation-to-nodes rel)) n)))
		  (progn
		    (cond
		      ((member verb (list *verb-fm+* *verb-rm+* *verb-m+*))
		       (setq qmag2 qmag
			     qdir2 qdir))
		      ((member verb (list *verb-fm-* *verb-rm-* *verb-m-*))
		       (setq qmag2 (qmag-inverse qmag)
			     qdir2 (qdir-inverse qdir)))
		      (t (format t "~%add-variable-nodes: Unrecognized verb-type ~A !"
				 (verb-name verb))
			 (return-from add-variable-nodes nil)))
		    (setq varname2 (node-name (car (relation-to-nodes rel))))
		    (setq termname2 (list varname2 qmag2 qdir2))
		    ;;(format t "~%Doing:  ~A ----- ~A ----- ~A"
		    ;;	      termname (verb-name verb) termname2)
	      
	      
		    ;; Then build the new "transformed" node and add it as a child.
		    (let ((child-termname  termname2)
			  (children        nil)
			  child-node
			  newlink)
		      (declare (special children))	; make visible to add-child-node
		      (setq child-node (check-term-name child-termname 'create))
		      (setq newlink (make-graphlink :relation rel
						    :from-nodes (list node)
						    :to-nodes   (list child-node)
						    :parent n
						    :num-siblings 0))
		      ;; add-child-node may add to children.
		      (add-child-node child-node child-strength newlink direction n)
		
		      ;; If any children were generated from this relation ...
		      (if children
			  ;; Then report success, increment number of relations,
			  ;; and add the child to the open list.
			  (progn
			    (setq success t)
			    (incf num-relations)
			    (add-to-open children))))))))))))


(defconstant *qmag-inverse* '((normal              . normal)
			      (mildly-elevated     . mildly-depressed)
			      (moderately-elevated . moderately-depressed)
			      (severely-elevated   . severely-depressed)
			      (profoundly-elevated . profoundly-depressed)))

(defconstant *qdir-inverse* '((std . std)
			      (inc . dec)
			      (dec . inc)
			      (xxx . xxx)))

(defun qmag-inverse (qmag)
  (let* ((term (check-term-name qmag 'create))     ; done in case of synonyms
	 (name (term-name term)))
    (or (cdr (assoc  name *qmag-inverse*))
	(car (rassoc name *qmag-inverse*)))))

(defun qdir-inverse (qdir)
  (let* ((term (check-term-name qdir 'create))     ; done in case of synonyms
	 (name (term-name term)))
    (cdr (assoc name *qdir-inverse*))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (consistent-direction  verb  graphnode)
;;;
;;;  Returns:   T if the "direction of effect" of the given verb is consistent
;;;             with previous verbs on the path leading to this graphnode,
;;;             otherwise NIL is returned.
;;;
;;;  Note:      Verbs that have a "forward" direction of effect are "fm+", 
;;;             "fm-", "causes", "enables", "implies", "acts on", and "affects".
;;;             Verbs that have a "backward" direction of effect are the
;;;             inverses of these verbs.  All other verbs are considered to be
;;;             bi-directional and are therefore consistent with any verb.
;;;-----------------------------------------------------------------------------

(defun consistent-direction (verb gnode1)
  (let ((verb-dir (direction verb)))

    ;; If this verb is directional ...
    (if verb-dir

	;; then check all preceding verbs.
	(do* ((gnode  gnode1  (graphnode-parent gnode)))
	     ((null gnode) t)
	  (if (graphnode-link gnode)
	      (let* ((prev-verb (relation-verb
				  (graphlink-relation
				    (graphnode-link gnode))))
		     (prev-dir  (direction prev-verb)))
		;; If this preceding verb was directional, then see if it's
		;; direction is consistent with the current verb.
		(if (and prev-dir
			 (not (eql verb-dir prev-dir)))
		    (return-from consistent-direction NIL))))))
    (return-from consistent-direction T)))


;; Returns 'forward, 'backward, or NIL depending on "direction" of verb.
(defun direction (verb)
  (cond ((member verb (list *verb-fm+* *verb-fm-* *verb-enables* *verb-causes*
			    *verb-actsOn* *verb-affects*))
	 'forward)
	((member verb (list *verb-rm+* *verb-rm-* *verb-isEnabledBy* *verb-causedBy*
			    *verb-isActedOnBy* *verb-isAffectedBy*))
	 'backward)
	(t  nil)))



;;;-----------------------------------------------------------------------------
;;; Function:  (repeated-predicate  predicate gnode)
;;;
;;; Returns:   non-nil if the given predicate appears anywhere in the path
;;;            from gnode back to the root.
;;;-----------------------------------------------------------------------------

(defun repeated-predicate (pred1 gnode1)
  (do ((gnode  gnode1  (graphnode-parent gnode)))
      ((null gnode) NIL)
    (let* ((term (graphnode-name gnode))
	   (pred2 (term-predicate term)))
      (if (eq pred1 pred2)
	  (return-from repeated-predicate T)))))


;(defun evaluate-strength (parent children relation dstrength)
;  (let ((link-strength (relation-strength (relation-inverse relation)))
;	(num-children (length children)))
;    (if (zerop num-children)
;        (setq num-children 1))
;    (make-list num-children :initial-element (/ (* dstrength link-strength)
;						num-children))))

;;;-----------------------------------------------------------------------------
;;; Function:  (all-same vector)
;;;
;;; Given:     vector, an array of numbers
;;;
;;; Returns:   an element of vector, if they are all the same
;;;            infinity, otherwise
;;;-----------------------------------------------------------------------------

(defun all-same (vector)
  (let ((infinity 1.0e6)
	(answer (car vector)))
    (dolist (v vector)
      (if (not (= v answer))
	(return-from all-same infinity)))
    (return-from all-same answer)))

  
;;;-----------------------------------------------------------------------------
;;; Function:  (generate-backsiblings n rel parents)
;;;
;;; Given:     n,       a graphnode
;;;            rel,     the relation with from-nodes (n ... )
;;;            parents, a list of graphnodes which can't be put on open
;;;                     until all backsiblings solved
;;;
;;; Returns:   a list of siblings that must be proven in the reverse direction
;;;            before any children of n can be proved.  
;;;
;;; Called by: generate-successors
;;;
;;; Design:    Meant to handle the problem of multiple from nodes
;;;-----------------------------------------------------------------------------

(defun generate-backsiblings (n rel parents)
   (let*  ((siblings (relation-from-nodes rel))
	   (num-siblings (1- (length siblings)))
	   (sibnode nil)
	   (backsiblings nil)
	   (oldnodes nil)
	   (newlink nil)
	   (direction (graphnode-forward n)))


     ;; take out the node itself
     (setq siblings (remove  (graphnode-name n) siblings))

     ;; add parents to limbo
     (add-to-limbo parents)

     ;; for each parent mark how many siblings must be 
     ;; solved before parent can be removed from limbo
     (dolist (p parents)
       (setf (graphnode-num-relations p) num-siblings))

     ;;make graphlink
     (setq newlink (make-graphlink :relation     rel
				   :from-nodes   (relation-from-nodes rel)
				   :to-nodes     (relation-to-nodes rel)
				   :parent       parents
				   :num-siblings num-siblings
				   :is-and       t))

     ;;for each backsibling
     (dolist (sibling siblings)
       
       (incf *tree-counter*)
       
       ;; make graphnode
       (setq sibnode (make-graphnode :name sibling
				     :downstrength 1.0
				     :link newlink
				     :num-relations 0
				     :parent nil
				     :limbo parents
				     :forward (null direction)
				     :tree *tree-counter*))

       ;; now need to check what lists this graphnode may
       ;; already be on
       ;; if sibling already in closed -->
       ;;     path to this node already found, propagate success
       ;;         (take parent off limbo)
       ;; if sibling already in open
       ;;    path to this node already found, propagate success
       ;;          (take parent off limbo, if possible)
       ;; if sibling already in limbo?
       ;;


       (cond   ;; is this key right?
	 ((setq oldnodes (already-in-closed sibling))
	  ;(print 'sibnode-already-in-closed)
	  ;(add-to-closed sibnode)  ;; why
	  (propagate-success sibnode (null direction)))
	 

	 ((setq oldnodes (already-in-open sibling))
	  ;(print 'sibnode-already-in-open)
	  (add-to-closed sibnode)
	  (propagate-success sibnode (null direction)))

	 ((setq oldnodes (already-in-limbo sibling))
	  ;(print 'backsibling-in-limbo )
	  )

	 (t 
	    (push sibnode backsiblings)))
       )
     (if backsiblings
         (add-to-open backsiblings))
	 
     
     ))


;;;-----------------------------------------------------------------------------
;;; Function:  (check-condition  condition)
;;;
;;; Given:     condition,  a condition on a relation.  May be one of four
;;;                        types: category, newcase, exemplar, or
;;;                        predicate.
;;;
;;; Returns:   t,    if condition is met
;;;            nil,  otherwise
;;;
;;; Note:      Conditions of type predicate not yet implemented
;;;            This has not been tested!
;;;-----------------------------------------------------------------------------


(defun check-condition (condition)
  
  ;; If there is no condition ...
  (if (or (null condition) *ignore-conditions*)

      ;; then just return t since "no condition" is always satisfied
      t
      
      ;; else check the conditions appropriate to the search mode.
      (case (condition-type condition)
	
	;; condition on category (applies in FtoF and FtoT)
	(category    (eq *category* (condition-category condition)))
	
	;; condition on newcase  (applies in FtoF and FtoT)
	(newcase     (same-features *case-features*
				    (condition-features condition)))
	
	;; condition on exemplar (applies only to FtoF)
	(exemplar    (if (eql *kbpm-mode* 'FtoF)
			 (same-features (exemplar-features *exemplar*)
				    (condition-features condition))))
	
	(otherwise   (cerror "condition will be treated as NOT met"
			     "check-condition: bad condition type")
		     nil))))


;;;-----------------------------------------------------------------------------
;;; Function:  (same-features features clist)
;;;
;;; Given:     features,  a list of features
;;;            clist,     a list of features specified by a condition
;;;
;;; Returns:   t,   if all features in clist are contained in features
;;;            nil, otherwise
;;;
;;;----------------------------------------------------------------------------

(defun same-features (features clist)
  (null (set-difference clist features)))




;;;-----------------------------------------------------------------------------
;;; Function: (retrieve-explanation gnode dstrength)
;;;
;;; Given:    gnode, the graphnode from which explanation begins
;;;           dstrength, strength of path from start to this node
;;;
;;; Returns:  etype,       the type of explanation
;;;           explanation, the explanation of gnode
;;;           certainty,   the strength of the explanation
;;;
;;;-----------------------------------------------------------------------------

(defun retrieve-explanation (gnode dstrength )
  (prog (vtype downlink start from-ex to-nodes etype tree)
    
	(setq downlink  (graphnode-best-child-link gnode)
	      start     (graphnode-name gnode)
	      tree      (graphnode-tree gnode)
	      etype     'explained)
    
    
	;; if no downlink -- gnode is goal node.
	(if (null downlink)
	    (return (values 'explained start dstrength)))
    
	;; else need to construct explanation
	(let ((from-nodes    (remove start (graphlink-from-nodes downlink)))
	      (from-type     'explained)
	      (from-strength 1.0))

	  ;; If more than 1 from-node ...
	  (if (> (length from-nodes) 0)
	      ;; then get explanation for each
	      (progn
		(setq from-nodes (get-from-closed from-nodes (1+ tree)))
		(multiple-value-setq (from-type from-ex from-strength)
		  (retrieve-explanations from-nodes 1.0))

		(push start from-ex)		; is this right?
		;; need to check type of explanation of from-nodes
		(setq etype (get-explanation-type etype from-type)))

	      ;; else, for a single from node, the start term is the only one.
	      (setq from-ex (list start))))
      
    
	(setq vtype (relation-verb (graphlink-relation downlink)))
	;; check this relation for type of verb
	(cond 

	  ;; relation is spurious -- there will be no to-explanations
	  ((eql vtype *verb-spurious*)
	   (let ((sim (* dstrength (relation-strength (graphlink-relation downlink)))))
	     (return (values 'spurious
			     (make-explanation :start-term start
					       :from-terms from-ex
					       :to-terms nil
					       :relation (graphlink-relation
							   downlink)
					       :strength sim)
			     sim))))

	  ;; relation is mutual exclusion, set explanation type
	  ;; and continue
	  ((eql vtype *verb-MEx*)
	   (setq etype (get-explanation-type etype 'excluded)))

	  ;; any other relation - do nothing
	  )

	;; get to-explanations
	(setq to-nodes (get-from-closed (graphlink-to-nodes downlink) tree))
       
	(multiple-value-bind (to-type to-ex sim)
	    (retrieve-explanations to-nodes
				   (* dstrength (relation-strength (graphlink-relation downlink))))

	  ;; need to check type of explanation of to-nodes
	  (setq etype (get-explanation-type etype to-type))

	  (if to-ex
	      (return (values etype
			      (make-explanation :start-term start
						:from-terms from-ex
						:to-terms   to-ex
						:relation   (graphlink-relation downlink)
						:strength   sim)
			      sim))))
	))

;;;-----------------------------------------------------------------------------
;;; Function:  (retrieve-explanations gnodes)
;;; 
;;; Given:     gnodes,   a list of graphnodes
;;;      
;;; Return:    etype,    the type of the explanations
;;;            explains, a list of explanations
;;;            sim,      the strength of the explanations
;;;
;;; Design:    For each node in gnodes, retrieve its explanation. Keep track
;;;            of the minimum explanation strength returned.  Also keep track
;;;            of the explanation type of the explanations because a spurious
;;;            explanation should override an 'explained or 'excluded
;;;            explanation.
;;;-----------------------------------------------------------------------------

(defun retrieve-explanations (gnodes dstrength)
  (do ((g        gnodes (cdr g))
       (etype    'explained)
       (sim      1e10)
       (explains nil))
      ((endp g)  (values etype explains sim))
    
    (if (graphnode-marked (car g))
        (multiple-value-bind (et ex s)
	    (retrieve-explanation (car g) dstrength)
	  (if ex
	      (progn
	       (push ex explains)
		(if (< s sim)
		    (setq sim s))
		(setq etype (get-explanation-type etype et))))
	  ))))

;;;-----------------------------------------------------------------------------
;;; Function:  (get-explanation-type oldtype newtype)
;;;
;;; Given:     oldtype,   an explanation type
;;;            newtype,
;;;
;;; Returns:   the stronger of the explanation types
;;;            spurious overrides mex
;;;            mex overrides explained
;;;-----------------------------------------------------------------------------

(defun get-explanation-type (oldtype newtype)
  (cond
    ((eql oldtype 'explained) newtype)
    ((eql oldtype 'excluded)
     (if (not (eql newtype 'explained))
       newtype
       oldtype))
    ((eql oldtype 'spurious) oldtype)
    (t (cerror "invalid explanation type"))))


;;;-----------------------------------------------------------------------------
;;; Function:  (get-from-closed features)
;;;
;;; Given:     features,   a list of terms in the category network
;;;
;;; Returns:   glist,      a list of graphnodes corresponding to features  
;;;
;;; Algorithm: For each features in features, find the graphnode in *closed*
;;;            which corresponds to it. 
;;;-----------------------------------------------------------------------------

(defun get-from-closed (features tree)
  (let ((glist nil))

    ;; For each feature ...
    (dolist (f features glist)

      ;; Find the marked graphnode corresponding to the feature.
      (do ((gnodes (member f *closed*     :key #'graphnode-name)
		   (member f (cdr gnodes) :key #'graphnode-name)))
	  ((endp gnodes))

	(if (and (= tree (graphnode-tree (car gnodes)))
		 (graphnode-marked (car gnodes)))
	    (push (car gnodes) glist))))))

      
;  (do ((f features (cdr f))
;       (gnodes nil)
;       (g nil))
;      ((endp f) gnodes)
;      
;      (setq g 
;	(find (car f) *closed* :key #'graphnode-name))
;      (if (and g (= (graphnode-tree g) tree)) 
;	       (push g gnodes))
;;;   (format t "~%get-from-closed: ~A ~:[not found~;found~]"
;;;	   (node-name (car f)) g)
;))


  
;;;-----------------------------------------------------------------------------
;;; Function: (add-to-closed n)
;;;
;;; Given:    n, a graphnode
;;;
;;; Returns:  adds the node n to the closed list
;;;
;;;-----------------------------------------------------------------------------

(defun add-to-closed (n)
  (push n *closed*)
  (if *trace-kbpm2*
      (progn
	(print "Add to closed:")
	(print-graphnode n t 1))))


;;;-----------------------------------------------------------------------------
;;; Function:  (add-to-open children)
;;;
;;; Given:     children, a list of graphnodes
;;;
;;; Returns:   the *open* list with children added to it such that *open*
;;;            remains ordered according to decreasing strength.
;;;-----------------------------------------------------------------------------

(defun add-to-open (children)
  (let ((sorted-children (sort children #'>= :key #'graphnode-downstrength)))
    (if *trace-kbpm2*
	(progn
	  (print "Add to open:")
	  (dolist (child sorted-children)
	    (print-graphnode child t 1))))
    (setq *open*
	  (merge 'list sorted-children *open* #'>= :key #'graphnode-downstrength))))


;;;-----------------------------------------------------------------------------
;;; Function:  (add-to-limbo siblings)
;;;
;;; Given:     siblings, a list of graphnodes
;;;            *limbo*, unordered list of graphnodes which cannot be added to
;;;            *open* until all their from nodes are solved
;;;
;;; Returns:   *limbo*
;;;
;;;-----------------------------------------------------------------------------

(defun add-to-limbo (siblings)
  (dolist (sib siblings *limbo*)
    (push sib *limbo*)))


;;;-----------------------------------------------------------------------------
;;; Function:  (remove-from-limbo gnodes)
;;;
;;; Given:     gnodes, a list of graphnodes
;;;            *limbo*, global variable
;;;
;;; Returns:   *limbo* with all the nodes in gnodes removed
;;;-----------------------------------------------------------------------------

(defun remove-from-limbo (gnodes)
  (setq *limbo* (set-difference *limbo* gnodes)))


;;;-----------------------------------------------------------------------------
;;; Function:  (propagate-success  n  direction)
;;;
;;; Given:     n, a solved graphnode
;;;            direction, direction of search
;;;
;;; Returns:   traverses graph from n to all its ancestors determining
;;;            if they are solved, given that n is solved
;;;
;;; Note:  Doesn't do anything if path to limbo node is spurious
;;;-----------------------------------------------------------------------------

(defun propagate-success (n direction)
  (prog (p link numr numleft)

    (setq p (graphnode-parent n))
    (if p   (setq numr (1- (graphnode-num-relations p))))
    (setq link (graphnode-link n))
    
    (cond
        ;; no parent
	((null p)  (if (graphnode-limbo n)
		     
                       (progn
			(setq numleft (1- (graphlink-num-siblings link)))
			(setf (graphlink-num-siblings link) numleft)
			(setf (graphnode-marked n) t)
			(dolist (l (graphnode-limbo n))
			     (setf (graphnode-certainty l) 
				   (min (graphnode-downstrength l)
					(graphnode-certainty n))))
			(if (zerop numleft)
			  (progn 
			   (add-to-open (copy-list (graphnode-limbo n)))
			   (remove-from-limbo (graphnode-limbo n)))))
                        ))
	
	
	
	;; and link
	((graphlink-is-and link)
	        ;; solved one of siblings
	        (setq numleft (1- (graphlink-num-siblings link)))
	        (setf (graphlink-num-siblings link) numleft)
		(setf (graphnode-marked n) t)
		(setf (graphnode-certainty p)
		      (min (graphnode-certainty p)
			   (graphnode-certainty n)))

		(if (= numleft 0)
		  ;; solved whole relation
		  (if (eql (graphnode-status p) 'spurious)
		      (progn (setf (graphnode-best-child-link p) link)
			     (setf (graphnode-num-relations p) numr)
			     (propagate-spurious p direction))

		      (progn (setf (graphnode-status p) 'solved)
			     (setf (graphnode-best-child-link p) link)
			     (setf (graphnode-num-relations p) numr)
			     (propagate-success p direction)))))
	      
	;; or link
	(t   
	        (setf (graphnode-status p) 'solved)
		(setf (graphnode-best-child-link p) link)
		(setf (graphnode-num-relations p) numr)
		(setf (graphnode-marked n) t)
		(setf (graphnode-certainty p) (graphnode-certainty n))
		(unless (zerop numr)
		  (remove-siblings-from-open link)) 
		(propagate-success p direction)))
    
))

;;;-----------------------------------------------------------------------------
;;; Function:  (propagate-failure n direction)
;;;
;;; Given:     n, a failed graphnode
;;;            direction, the direction of search
;;;
;;; Returns:   traverses graph from n up to its ancestors marking
;;;            those that have failed, given that n has failed
;;;
;;;-----------------------------------------------------------------------------

(defun propagate-failure (n direction)
  (prog (p link numr)
    (setq p (graphnode-parent n))
    (if p (setq numr (1- (graphnode-num-relations p))))
    (setq link (graphnode-link n))
    (cond 
      
         ;;no parent
         ((null p)  (if (graphnode-limbo n)
			 (dolist (p (graphnode-limbo n))
			     (setf (graphnode-status p) 'failed)
			     (propagate-failure p (null direction))))
			
			     )

	 ;; and link
	 ((graphlink-is-and link) (setf (graphnode-status p) 'failed)
	                          (unless (zerop (graphlink-num-siblings link))
	                            (remove-siblings-from-open link))
	                          (setf (graphnode-num-relations p) numr)
				  (setf (graphnode-certainty p)
					(graphnode-downstrength p))
	                          (propagate-failure p direction))
	 
	 ;; or node
	 (t
	         (setf (graphnode-num-relations p) numr)
		 (if (< numr 1)
		   (if (eql (graphnode-status p) 'spurious)
		     (propagate-spurious p direction)
		     (progn
	 	      (setf (graphnode-status p) 'failed)
 		      (propagate-failure p direction)))))
	 
	 )))
;;;-----------------------------------------------------------------------------
;;; Function:  (propagate-spurious n direction)
;;;
;;; Given:     n, a solved, but spurious  graphnode
;;;            direction, the direction of search
;;;
;;; Returns:   traverses graph from n up to its ancestors marking
;;;            those that have succeed, given that n has succeed along
;;;            a spurious path.
;;;
;;;-----------------------------------------------------------------------------

(defun propagate-spurious (n direction)
  (prog (p link numleft numr)
    (setq p (graphnode-parent n))
    (if p (setq numr (1- (graphnode-num-relations p))))
    (setq link (graphnode-link n))
    (cond 

      ;;no parent
         ((null p)  (if (graphnode-limbo n)
		     
                       (progn
			(setq numleft (1- (graphlink-num-siblings link)))
			(setf (graphlink-num-siblings link) numleft)
			(setf (graphnode-marked n) t)
			(dolist (l (graphnode-limbo n))
			     (setf (graphnode-certainty l) 
				   (min (graphnode-downstrength l)
					(graphnode-certainty n))))
			(if (zerop numleft)
			  (progn 
			   (add-to-open (copy-list (graphnode-limbo n)))
			   (remove-from-limbo (graphnode-limbo n)))))
                        ))

	 ;; and link
	 ((graphlink-is-and link) 
	  (setq numleft (1- (graphlink-num-siblings link)))
	  (setf (graphlink-num-siblings link) numleft)
	  (setf (graphnode-marked n) t)
	  (setf (graphnode-status p) 'spurious)
	  (setf (graphnode-certainty p)
		(min (graphnode-certainty p)
		     (graphnode-certainty n)))
	  (if (< numleft 1)
	    (progn (setf (graphnode-best-child-link p) link)
		   (setf (graphnode-num-relations p) numr)
		   (propagate-spurious p direction))))
	 
	 
	 ;; or node
	 (t
	  (setf (graphnode-status p) 'spurious)
	  (setf (graphnode-best-child-link p) link)
	  (setf (graphnode-num-relations p) numr)
	  (setf (graphnode-certainty p)
		(graphnode-certainty n))
	  (if (< numr 1)
	    (propagate-spurious p direction)))
	 
	 
	 )))

;;;-----------------------------------------------------------------------------
;;; Function:  (remove-siblings-from-open link)
;;;
;;; Given:     link, a graphlink
;;;
;;; Returns:   The *open* list with all children of link removed.
;;;
;;;-----------------------------------------------------------------------------

(defun remove-siblings-from-open (link)
  (if (null *open*)
    *open*
    
  (cond 
    ((graphlink-is-and link)
      (setq *open* (remove  link *open* :key #'graphnode-link)))
    ; or-link
    (t (setq *open*
      (let ((parent (graphlink-parent link)))
       
        (do ((nodes *open* (cdr nodes))
	     (result nil))
         ((endp nodes) (reverse result))
	 (if (not (eql  parent (graphnode-parent (car nodes))))
	   (push (car nodes) result)))))))
	 
    ))
;; check tree too?
;;;-----------------------------------------------------------------------------
;;; Function:  (already-in-closed node)
;;; 
;;; Given:     node, the name of node in the category network
;;;
;;; Returns:   list of graphnodes with the same name on *closed*
;;;
;;; Design:    uses the built-in member function
;;;-----------------------------------------------------------------------------

(defun already-in-closed (node)
  (do* ((gnodes *closed* (cdr l))
	(l (member node gnodes :key #'graphnode-name)
	   (member node gnodes :key #'graphnode-name))
	(result nil))
    ((null l) result)
    (push (car l) result)))

(defun already-in-open (node)
  (do* ((gnodes *open* (cdr l))
	(l (member node gnodes :key #'graphnode-name)
	   (member node gnodes :key #'graphnode-name))
	(result nil))
    ((null l) result)
    (push (car l) result)))

(defun already-in-limbo (node)
  (do* ((gnodes *limbo* (cdr l))
	(l (member node gnodes :key #'graphnode-name)
	   (member node gnodes :key #'graphnode-name))
	(result nil))
    ((null l) result)
    (push (car l) result)))

;;;-----------------------------------------------------------------------------
;;; Function:  (print-open-list)
;;;
;;; Given: nil
;;;
;;; Returns: 'Open-is-empty if *open* is empty
;;;          Otherwise, prints out each graphnode on the *open* list
;;;-----------------------------------------------------------------------------


(defun print-open-list ()
  (if *open*
    (progn  (print 'open-list)
            (print-list *open*))
    (print 'Open-is-empty)))


;;;-----------------------------------------------------------------------------
;;; Function:  (print-closed-list)
;;;
;;; Given: nil
;;;
;;; Returns: 'Closed-is-empty if the *closed* list is empty
;;;          Otherwise, prints out all the graphnodes on *closed*
;;;
;;;-----------------------------------------------------------------------------

(defun print-closed-list ()
  (if *closed*
    (progn  (print 'closed-list)
            (print-list *closed*))
    (print 'Closed-is-empty)))

;;;-----------------------------------------------------------------------------
;;; Function:  (print-limbo-list)
;;;
;;; Given: nil
;;;
;;; Returns: 'Limbo-is-empty if the *limbo* is empty
;;;          Otherwise, prints out all the graphnodes on *limbo*
;;;
;;;-----------------------------------------------------------------------------

(defun print-limbo-list ()
  (if *limbo*
    (progn  (print 'limbo)
            (print-list *limbo*))
    (print 'limbo-is-empty)))


;;;-----------------------------------------------------------------------------
;;; Function: (print-list gnodes)
;;;
;;; Given:    gnodes, a list of graphnodes
;;;
;;; Design:   Sequentially prints out all the graphnodes in the list
;;;           gnodes
;;;-----------------------------------------------------------------------------

(defun print-list (gnodes)
  (format t "~%   ~24A ~2A ~7A ~24A ~4A ~4A ~4A ~2A ~3A ~3A"
	  "Name" "Tr" "Status" "Parent" "Link" "Str" "Cert" "NR" "Fwd" "Dir")
  (dolist (gnode gnodes)
    (print-graphnode gnode t 1))
  'done)

;  (cond
;    ((null gnodes) 'done)
;    (t (print-graphnode (car gnodes) t 1)
;       (print-list (cdr gnodes)))))

;;;-----------------------------------------------------------------------------
;;; Function:  (trace-print flag cstring  args)
;;;
;;; Given:
;;;
;;;
;;;-----------------------------------------------------------------------------

(defun trace-print (flag cstring &rest args)
  (if flag
    (format t "~?" cstring args)))

