;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.


(in-package 'protos)


;;;=============================================================================
;;;             I N P U T  /  O U T P U T    F U N C T I O N S
;;;
;;;  Contents:  This file contains functions related to the user interface of
;;;		CL-Protos.
;;;
;;;  Functions: consume-white-space
;;;             escape
;;;             menu-select
;;;             prompt
;;;		show-escape-menu
;;;=============================================================================



;;;-----------------------------------------------------------------------------
;;;  Function:  consume-white-space
;;;
;;;  Purpose:   This function removes any "white space" characters currently
;;;		buffered in the input stream *query-io*.  "White space" is
;;;		defined as any of the following characters: #\Newline, 
;;;		#\Space, #\Backspace, #\Tab, #\Linefeed, #\Page, #\Return,
;;;		and #\Rubout.
;;;
;;;  Note:	This function was found to be necessary on UNIX systems running
;;;		Kyoto Common Lisp; it may be necessary for other Common Lisps
;;;		running on UNIX.  The problem arises from the different ways
;;;		that input is delimited by UNIX terminal I/O versus Common
;;;		Lisp's input from character streams.  Using normal buffered
;;;		input on UNIX, each input operation is terminated by a
;;;		#\Newline character, but the KCL (read) function does not
;;;		consume the #\Newline if it can determine the end of the
;;;		expression before reaching the #\Newline, such as when it
;;;		reads a list bounded by parentheses.  In this case it leaves
;;;		the #\Newline in the input stream.  If the next input
;;;		operation was also a (read), there would be no problem since
;;;		(read) will skip over any initial white space.  However, if
;;;		the next input operation is (read-line) or (read-char), the
;;;		buffered #\Newline will immediately terminate those input
;;;		operations (highly undesirable!).
;;;
;;;		The solution taken here is to call (consume-white-space)
;;;		prior to calling (read-line) or (read-char) whenever there
;;;		may be buffered white space (typically a #\Newline) remaining
;;;		from an immediately preceding (read).  Consume-white-space
;;;		is not needed on Lisp machines, but it doesn't hurt either.
;;;
;;;             Incidentally, the Common Lisp function 'clear-input' cannot
;;;             be used here because it will delete any legitimate type-ahead
;;;             input, whereas we only want to skip over the white space.
;;;-----------------------------------------------------------------------------

(defun consume-white-space ()
  (let (c)
    (loop
      ;; If there is a "white space" character in the input stream ...
      (if (and (listen *query-io*)
	       (setq c (peek-char nil *query-io* nil nil))
	       (member c (list #\Newline #\Space #\Backspace #\Tab
			       #\Linefeed #\Page #\Return #\Rubout)))
	  ;; then consume it ...
	  (read-char *query-io* nil nil nil)
	  ;; else return.
	  (return (values))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (prompt  prompt1 prompt2 type object view)
;;;
;;;  Given:     -- prompt1, the primary prompt string or menu;
;;;             -- prompt2, the secondary prompt string for repeated input;
;;;             -- type, the type of Lisp or Protos object to be read
;;;                (symbol string integer y-or-n termname explanation);
;;;             -- object, the object (if any) referred to by this prompt;
;;;             -- view, the way this object is currently being viewed, i.e.,
;;;                  as a category, an exemplar, or feature.
;;;
;;;  Returns:   An object of the specified type or, if repeated input is
;;;             expected, a list of objects of the specified type.
;;;
;;;  Reason:    The reason for this function is three-fold:
;;;             (1)  to perform elementary syntax checking on input, with
;;;                  reprompting in case of error, and
;;;             (2)  to handle the "?" escapes, and
;;;             (3)  this function may be the obvious place to handle much of
;;;                  the "replay" capability, if I ever decide I need to
;;;                  implement it.
;;;
;;;             The "?" escapes allow the user to momentarily escape out of a
;;;             query and go display something or set something before answering
;;;             the query.
;;;-----------------------------------------------------------------------------

(defun prompt (prompt1 prompt2 type object view)
  (prog ((inputs nil)
	 input)
     REPROMPT
	(clear-input *query-io*)
	(format *query-io* prompt1)
	(if prompt2 (format *query-io* prompt2))
	(if (eql type 'y-or-n)
	    (princ "(Y or N) " *query-io*))

	(prog (c)
	   TOP
              ;; Make sure no extraneous input is left over.
	      (consume-white-space)
	      ;; If next character is the "?" escape character ...
	      (setq c (peek-char nil *query-io* nil 'eof))
	      (if (char= #\? c)
		  ;; then consume it and examine next character.
		  (prog (selection)
			(read-char *query-io* nil 'eof)
			(setq selection (read-char *query-io* nil 'eof))
			(if (or (char= #\Rubout selection)
				(char= #\Backspace selection))
			    (go TOP))
			(escape selection object view)
			(go REPROMPT)))
	      
	      (case type
		(string
		  (setq input (read-line *query-io* nil nil nil))
		  (if (or (null input) (string= "" input))
		      (go OUT)))
	
                (feature	
	          (if (char= #\Newline (peek-char nil *query-io* nil nil))
	              (progn
		        (read-char *query-io* nil nil)	; consume the newline
		        (go OUT)))
	          (setq input (read *query-io* nil nil)))

		(integer
		  (if (char= #\Newline c)
		      (progn (read-char *query-io* nil nil) (go OUT)))
		  (setq input (read *query-io* nil nil nil))
		  (if (or (null input) (eql input 'q))
		      (progn (setq input nil) (go OUT)))
		  (if (not (integerp input))
		      (progn
			(format *query-io* "~%Error: you must enter an integer: ")
			(go REPROMPT))))
		
		(symbol
		  (if (char= #\Newline c)
		      (go OUT))
		  (setq input (read *query-io* nil nil nil))
		  (if (not (symbolp input))
		      (progn
			(format *query-io* "~%Error: you must enter a symbol: ")
			(go REPROMPT))))
		
		(termname
		  (if (char= #\Newline c)
		      (go OUT))
		  (setq input (read *query-io* nil nil nil))
		  (if (not (or (symbolp input) (consp input)))
		      (progn
			(format *query-io* "~%Error: you must enter a term name,~
                                            ~%either proposition or predicate: ")
			(go REPROMPT))))
		
		(y-or-n
		  (setq input (read-char *query-io* nil nil nil))
		  (case input
		    ((#\Y #\y)   (return-from prompt t))
		    ((#\N #\n)   (return-from prompt nil))
		    (otherwise   (progn
				   (format *query-io* "~%Error: you must enter y or n: ")
				   (go REPROMPT)))))
		
		(explanation
		  (if (char= #\Newline c)
		      (go OUT))
		  (setq input (read *query-io* nil nil nil))
		  (cond
		    ((null input)
		     (go OUT))
		    ((not (listp input))
		     (format *query-io* "~%Error: an explanation must be a list,~
                                         ~%enclosed in parentheses: ")
		     (go REPROMPT))
		    ((null (first-pass input))
		     (go REPROMPT))))

		(otherwise
		  (if (char= #\Newline c)
		      (go OUT))
		  (format *query-io* "~%Illegal type \"~A\" given to prompt.~
                                      ~%Go ahead and enter input anyway.~%"
			  type)
		  (setq input (read *query-io* nil nil nil))))

	      (if (null prompt2)
		  (return-from prompt input))
	      (push input inputs)
	      (format *query-io* prompt2)
	      (go TOP))
	      
     OUT
	(return-from prompt (nreverse inputs))))

;;;-----------------------------------------------------------------------------
;;;  Function:  (escape  selection object view)
;;;
;;;  Given:     -- selection, the character designating the selected escape
;;;                action;
;;;             -- object, the object (if any) to which an escape action might
;;;                be applied;
;;;             -- view, the way this object is currently being viewed, i.e.,
;;;                as a category, an exemplar, or a feature.
;;;-----------------------------------------------------------------------------

(defun escape (selection object view)
  (declare (special *display-menu* *control-menu* *overview-menu*
		    *unfocused-instruction-menu*))
  
  (case (char-upcase selection)
    (#\?    (show-escape-menu object view))
    
    (#\D    (menu-select *display-menu*))
    
    (#\C    (menu-select *control-menu*))
    
    (#\O    (menu-select *overview-menu*))

    (#\U    (menu-select *unfocused-instruction-menu*))
    
    (#\B    (break))

    (#\N    (let ((name nil) (abbrev nil) (synonyms nil))
		  (typecase object
		    (node        (setq name     (node-name object)
				       abbrev   (node-abbrev object)
				       synonyms (node-synonyms object)))
		    (case        (setq name     (case-name object)))
		    (otherwise   (setq name     "Unrecognized type in 'escape'")))
		  (format *query-io* "~%   Name: ~A" name)
		  (if synonyms
		      (format *query-io* "~%   Synonyms: ~A" synonyms))
		  (if abbrev
		      (format *query-io* "~%   Abbrev: ~A" abbrev))))
    
    (#\I    (typecase object
		  (term      (print-term object *query-io* 1))
		  (predicate (print-predicate object *query-io* 1))
		  (case      (print-case object *query-io* 1))
		  (otherwise (format *query-io* "~%Unknown object type."))))
     
    (#\F    (let ((pathnames (directory "*")))
		  (dolist (pathname pathnames)
		    (format *query-io* "~%   ~A" (file-namestring pathname)))))
     
    (#\Q    nil)

    (otherwise
            (format *query-io* "~%Invalid escape code (type \"??\" for a menu of escapes)."))))



(defun show-escape-menu (object view)
  
  ;; Consume any accidental/erroneous type-ahead.
  (clear-input *query-io*)
  (consume-white-space)

  (format *query-io*
	  "~%~%ESCAPE OPTIONS~
             ~%========================================================================~
             ~%  D   go to Display menu.         ?   Show these options.~
             ~%  C   go to Control menu.         B   Break (enter debugger).~
             ~%  U   go to Unfocused menu.       F   show File names in directory.~
             ~%  O   go to Overview menu.        Q   Quit (return to original prompt).")
  (if (and object (term-p object))
      (format *query-io*
	    "~%  N   Names (synonyms) of ~A" (getname object)))
  (if object
      (format *query-io*
	    "~%  I   Inspect ~A" (getname object)))
  (format *query-io*
          "~%~%Enter option ---> ")
  
  ;; Read an escape option, either as "?x" or just "x".
  (let ((selection (read-char *query-io* nil nil nil)))
    (if (char= #\? selection)
	(setq selection (read-char *query-io* nil nil nil)))
    (escape selection object view)))



;;;-----------------------------------------------------------------------------
;;;  Function:  (menu-select  menu)
;;;
;;;  Given:     menu, an instance of the 'menu' structure
;;;
;;;  Does:      This function presents a menu of choices to the user, reads
;;;             the user's choice, then performs the action associated with
;;;             the selected menu item.
;;;
;;;  Details:   This function provides the following features:
;;;
;;;             -- A menu has a title line.  It may also have embedded
;;;                sub-headings.  For example, Protos' control menu has sub-
;;;                headings for trace switches and mode switches.
;;;
;;;		-- Each selectable menu item is displayed with a key character
;;;	           in the left margin.  The user selects the item by typing
;;;	           the related key.
;;;
;;;		-- A selection causes one of two actions.  If the associated
;;;		   function is 'return then menu-select returns the associated
;;;		   argument.  Otherwise, menu-select calls the associated
;;;		   function with the associated arguments.  In that case,
;;;		   menu-select returns only when the called function returns,
;;;                and then only if menu-repeat is nil (see next point).
;;;
;;;             -- A menu may be displayed for a single selection or it may
;;;                allow multiple selections until a 'return is selected.
;;;                For example, Protos' control menu allows repeated selection
;;;                of various trace switches and mode switches.
;;;  
;;;             -- A menu that allows repeated selection may optionally
;;;                redisplay the entire menu each time.  This option is used
;;;                by Protos' top-level menu since the menu usually gets
;;;                scrolled off the screen between selections.
;;;
;;;             -- A menu may optionally display the value of a variable
;;;                associated with each item.  For example, Protos' control
;;;                menu displays the current value of each switch variable.
;;;
;;;  Note:      This function is system-independent and therefore portable,
;;;             but it handles an important part of the user interface which
;;;             might work better with a system-dependent implementation.
;;;             For example, a mouse-selectable menu would probably be 
;;;             preferable.
;;;-----------------------------------------------------------------------------



(defun menu-select (menu)
  
  (do ((repeat    t (menu-repeat menu))
       (redisplay t (menu-redisplay menu))
       (items     (menu-items menu))
       (val       nil)
       (input     nil)
       (selection nil))
      ((null repeat) val)

    REDISPLAY
    
    (if redisplay    ;; (Re-)Display the menu, if desired.
	(progn
	  ;; Print menu title if present.
	  (if (menu-label menu) 
	      (format t "~%~?" (menu-label menu) nil))

	  ;; If this is a 2-column menu, do a 2-column display.
	  (if (menu-twocol menu)
	      (do ((nextitems items (cddr nextitems)))
		  ((endp nextitems))
		(let* ((item1 (first nextitems))
		       (item2 (second nextitems))
		       (key1  (car item1))
		       (key2  (car item2)))
		  (if (null key1) (setq key1 #\Space))
		  (if (null key2) (setq key2 #\Space))
		  (format t "~%  ~2A  ~A~42T~2A  ~A"
			  key1 (item-label item1)
			  key2 (item-label item2))))

	      ;; else do a 1-column display.
	      (dolist (item items)
		(let ((key (car item)))
		  ;;(format t "~%menu-select: key = ~A" key)
		  ;; If this is a selectable item of the menu ...
		  (if key
		      ;; then display the key and its label
		      ;; and if it has argument values to be displayed ...
		      (if (and (menu-displayvar menu)
			       (not (equal 'return (item-function item))))
			  ;; then display them
			  (format t "~%  ~2A  ~A~42T~A"
				  key  (item-label item)  (mapcar #'eval (item-args item)))
			  ;; else just display the menu item
			  (format t "~%  ~2A  ~A" key (item-label item)))
		  
		      ;; else display a sub-heading (which is not selectable).
		      (format t "~%  ~A" (item-label item))))))))

    (consume-white-space)
    (format *query-io* "~%~%Select an item ---> ")

    ;; This tagbody handles escapes and invalid menu selections, and 
    ;; doesn't exit until it gets a valid menu selector character.
    (tagbody
      TOP
	 (force-output *query-io*)
	 ;; If next character is the "?" escape character ...
	 (if (char= #\? (peek-char nil *query-io* nil 'eof))
	     ;; then consume it and examine next character.
	     (let ((junk        (read-char *query-io* nil 'eof))
		   (e-selection (read-char *query-io* nil 'eof)))
	       (declare (ignore junk))
	       (if (or (char= #\Rubout e-selection)
		       (char= #\Backspace e-selection))
		   (go TOP))
	       (escape e-selection nil nil)
	       (setq redisplay t)
	       (go REDISPLAY)))

	 ;; else read the user's menu selection, looping until we
	 ;; have a non-white-space character.
	 (loop
	   (setq input (char-upcase (read-char *query-io* nil #\?)))
	   (if (not (member input (list #\Newline #\Space #\Return
					#\Backspace #\Tab #\Page
					#\Linefeed #\Rubout)))
	       (return)))

	 (setq selection (assoc input items))
	 (if (null selection)
	     (progn
	       (terpri *query-io*)
	       (write-string "Error: Please type one of the characters:" *query-io*)
	       (dolist (pair items)
		 (if (car pair)      ;; eliminate NILs in output.
		     (progn
		       (write-char #\Space *query-io*)
		       (write-char (car pair) *query-io*))))
	       (terpri *query-io*)
               (write-string "- - - - - - - - - - - - - - - - - - - - -> " *query-io*)
	       (clear-input)
	       (go TOP))))

    (format *query-io* " {~A} " (item-label selection))
    ;; Call the function associated with the selected menu item.
    (if (equal 'return (item-function selection))
	;; then exit from this menu
	(if (item-args selection)
	    (return-from menu-select (car (item-args selection)))
	    (return-from menu-select (values)))
	;; else call the designated function and then continue the do loop.
	(progn
	  (if (null (item-args selection))
	      (setq val (funcall (item-function selection)))
	      (setq val (apply (item-function selection) (item-args selection))))))))



(defun menu-select-2 (menusymbol)
  (menu-select (eval menusymbol)))

