functor SearchFUN (Graph : GRAPH) : SEARCH =
struct
 structure Graph = Graph

 (* Consists primarily of the shortestPath function for finding the 
    best path through a graph at one particular level of the
    abstraction hierarchy. Using flags indicating the direction
    of this graph and its next superior graph, if any, it 
    detemines the direction of search. The function initialises the
    flags for the goal and start nodes and determines and executes
    the appropriate search method dependent on the search type
    selected *)

 (* Define structures for different search function *)
 structure AlternateOp = 
   BFSAllVisitedFUN  (structure Graph = Graph
               fun incFn x = x + 1
 	       fun queueFn x = x) 

 structure AllMark = 
   BFSAllVisitedFUN  (structure Graph = Graph
               fun incFn x = 1
 	       fun queueFn x = 1) 

 structure PathMark = 
  BFSPathFUN  (structure Graph = Graph
	       fun incFn x = 1
 	       fun queueFn x = 1) 

 structure PathOp = 
 BFSPathFUN  (structure Graph = Graph
	      fun incFn x = x + 1
 	      fun queueFn x = x)  

 structure Classic = 
   ClassicRefFUN (structure Graph = Graph) 
 structure NoAbs = 
   NoAbsFUN  (structure Graph = Graph) 

 (* Define structure for revising start and goal nodes *)
 structure Revise = 
   ReviseFUN  (structure Graph = Graph) 

local open WorkingMemory Priority_Queue Graph Revise in

 exception Trivial of node list

fun shortestPath (subFrom,subTo,from,to,(graph as Graph{nodes,graphType,super,...}),absPath) =
 let val superFore =  case (!super) of 
		 SOME (Graph {graphType = ref SearchForward,...}) => true | _ => false
     val fore =  ((!graphType) = SearchForward)
     val absFore  = (fore = superFore)
     val down =  (fn Node{succs,...} => !succs)
     val up = (fn Node{preds,...} => !preds)
     val searchFn = if (fore) then down else up
     val absPathLength = length absPath
     val WM {searchType,...} = Wm
     val [(startPath,newStart), (endPath,newGoal)] =   
		revise fore subFrom from subTo to 

     fun getPathFor (SOME node) = getPath (node::startPath)
       | getPathFor NONE = NONE
     and getPath (path as ((Node{flag=ref(Back (0, n)),...})::_)) =
          	SOME (endPath @ path) 
       | getPath (path as ((Node{flag=ref(Back (_, n)),...})::_)) =
          	getPath (n::path)

     fun setFlags init  =
    	(init absPath absFore fore;
	app (fn (node as Node{flag,...}) => flag := Back (0,node)) newStart;
    	app (fn (node as Node{flag,...}) =>
        case !flag of Back _ => raise Trivial (endPath @ (node::startPath))
                      | _    => flag := Stop) newGoal;
        allNodes := newStart @ newGoal @ (!allNodes))

   in   (case (!searchType) of 
           PM => (setFlags PathMark.init; 
		    getPathFor (PathMark.bfs searchFn absPathLength newStart))
         | AO => (setFlags AlternateOp.init; 
		    getPathFor (AlternateOp.bfs searchFn absPathLength newStart))
         | AM => (setFlags AllMark.init; 
		    getPathFor (AllMark.bfs searchFn absPathLength newStart))
         | PO _ => (setFlags PathOp.init; 
		    getPathFor (PathOp.bfs searchFn absPathLength newStart))
         | CR => (setFlags Classic.init; 
		    getPathFor (Classic.bfs searchFn absPathLength newStart))
         | _  => (setFlags NoAbs.init; 
		    getPathFor (NoAbs.bfs searchFn absPathLength newStart))
         ) end handle Trivial list => SOME list

end
end
