structure BinPack : GENERIC_PUZZLE =
struct
local open Random
in

(*  A state is represented by two lists of pairs.  In each pair the
    first number is a size (remaining capacity of a bin, or, size of
    the block to put into a bin), the second is the number of things
    of this size.  The first list describes the bins, the second
    describes the blocks to be put into the bins.
    The lists are sorted according to size (decreasing order).
       For example, the state  ( [(5,2),(4,1)] , [(3,2),(2,4)] )
    means there are 2 bins of size 5 and 1 bin of size 4, and into
    these bins we'd like to place 2 blocks of size 3 and 4 blocks
    of size 2.

       The moves available at any state are:
    if the largest block will not fit into any bin, then there are
       NO legal moves (this speeds up failure detection)
    otherwise you may put any block into any bin in which it will fit

    A variation would have the legal moves be just those involving the
    largest block, since this must eventually be placed somewhere.
    This reduces the branching factor of the space without affecting
    reachability ;  abstraction gives best speedup over no-abstraction
    with high branching factor, so I implemented the first version.
    (also I wanted to avoid creating the impression that I had sneaked
     in some sort of largest-first heuristic)

*)

 type external_state =  ( (int*int) list ) * ( (int*int) list )
 type internal_state = external_state
 type puzzle_parameter =   int * int * int * int
                           (* size of the bins ,
                              number of bins,
                              max size of a block ,
                              max number of blocks *)

 fun sameState (x,y) = (x = y)

     local fun listpairhash [] = 0
            |  listpairhash ((a,b)::t) = ( a*b+(listpairhash t) ) mod 1000
     in
 fun hashState (binL,blockL) =
     1000*(listpairhash binL) + (listpairhash blockL)
     end

fun make_pair a b = (a,b)


fun make_pair_rev a b = (b,a)   (* N.B. reverses order of arguments  *)

fun update_blocks (_,1) x = x
 |  update_blocks (size,blockcount) others = (size,blockcount-1)::others

fun binsert  x [] = [x]
 |  binsert (x    as (xsize:int,xcount:int))
            (bins as ((head as (hsize,hcount))::tail)) =
 if xsize > hsize
 then x::bins
 else if xsize = hsize
      then (hsize,hcount+xcount)::tail
      else head::(binsert x tail)

fun update_bins blocksize (binsize, 1 ) others =
       if blocksize = binsize
       then others
       else binsert (binsize - blocksize , 1) others
 |  update_bins blocksize (binsize,bincount) others =
       if blocksize = binsize
       then (binsize,bincount-1)::others
       else (binsize,bincount-1)::(binsert (binsize - blocksize , 1) others)

fun this_block_in_any_bin   _   [] _ = []
 |  this_block_in_any_bin blocksize ((bin1 as (binsize,_))::others) biggerbins =
    if blocksize > binsize
    then []
    else
        (biggerbins @ (update_bins blocksize bin1 others)) ::
        ( this_block_in_any_bin blocksize others (biggerbins @ [bin1] ) )

fun any_block_in_any_bin  _  []  _  = []
 |  any_block_in_any_bin   bins
                           ((block as (bsize,_))::smallerblocks)
                           biggerblocks =
        any_block_in_any_bin bins smallerblocks (biggerblocks @ [block] )
      @ (let val newblocks = biggerblocks @ (update_blocks block smallerblocks)
             val newbins_list = this_block_in_any_bin bsize bins []
         in map (make_pair_rev newblocks) newbins_list
         end)

 fun successors ([], _)= []
  |  successors ( _,[])= []
  |  successors (bins   as ((binsize1,  _)::_),
                 blocks as ((blocksize1,_)::_) ) =
     if binsize1 < blocksize1
     then []
     else any_block_in_any_bin bins blocks []

fun make_initial_states (binsize,bincount,maxblocksize,maxblockcount) =
    let val maxarea = binsize * bincount
        val minarea = 1 + maxarea - binsize
        fun repeat F min max = if min > max
                               then []
                               else (F min) @ (repeat F (min+1) max)
        fun partition_exact_area nblocks thisblocksize area =
         if area = 0
         then [ [] ]
         else if (area < 0) orelse ((nblocks*thisblocksize) < area)
              then []
              else (partition_exact_area nblocks (thisblocksize-1) area)
                  @ (   map (binsert (thisblocksize,1))
                      (partition_exact_area
                            (nblocks-1) thisblocksize (area-thisblocksize) ) )
     in map (make_pair [(binsize,bincount)])
        (
        repeat (partition_exact_area maxblockcount maxblocksize) minarea maxarea
        )
    end



fun random_element L =
    let val pos = (newrandom (length L))-1
    in  (nth (L,pos))
    end


(* the following takes a state and returns a random successor of that state,
   successors with low outdegree have higher probability of being chosen
   than those with a high outdegree.  The chosen state will be at most
   distance D from the initial state.
   (D is really only needed to rpevent infinite loops in cyclic spaces.
    I have set it large enough here that it will never cause termination)
*)

 fun randState 0 state = state
  |  randState D state =
     (
     case (successors state)
      of   []  =>  state
       |   ss  =>  randState (D-1)  (random_element ss)
     )

fun identity x = x

 fun mkInstance pp  =
 let val istates = make_initial_states pp
     fun have_succs [] = []
      |  have_succs (h::t) = ( case (successors h)
                              of  [] => (have_succs t)
                               |  _  => h::(have_succs t)
                             )
     val sstates = have_succs istates

 in
     { decode   = identity ,
       encode   = identity ,
       successor= successors,
       initialStates = istates,
       randomStatePair = (fn () => let val istate = random_element sstates
                                    in ( istate, randState 300 istate) 
                                   end
                 )
      }
 end
end
end


