functor AbsAbstractionGeneratorFUN 
  (structure Graph : GRAPH 
	 and AbsFunctions : ABS_FUNCTIONS
   sharing Graph = AbsFunctions.Graph) : ABSTRACTION_GENERATOR = 

(* Builds levels of abstraction upwards starting from the ground level. The
   process first builds a "copy" of the level to be abstracted - one abstract
   class per base node with the same connecting arcs. A list of functions
   is applied to graph to determine which nodes should be combined. The classes 
   asociated with the nodes are then merged into a single new class and the old
  ones removed *) 

struct
 structure Graph = Graph 
 structure AbsFunctions = AbsFunctions
local
 open AbsFunctions Graph WorkingMemory

 fun abstract (baseGraph as (Graph{nodes,super,size,height,...})) method  =
 if size<2 then reset Wm else
 let val classes =  copyGraph nodes 

   (* loops through a list of abstraction functions *)
     fun applyFunctions [] graph newClasses = newClasses
       | applyFunctions ({testForGen,nextGroup}::fs) graph newClasses =
	   if (testForGen graph)
	       	then applyFunctions fs graph 
			(removeOldClasses (reduce nextGroup nodes newClasses))
		else applyFunctions fs graph newClasses
     val newClasses = applyFunctions (absFunctions method) baseGraph classes
     val size' = (length newClasses)
     val graph = mkGraph (size',(height+1), newClasses)
     val _ = app (fn Node{flag,...} => flag:= Start) nodes
 in if (size = size') then reset Wm
       else 
   	(super := SOME graph;
    	abstract graph method) end

 (* forms a new graph of classes which is a copy of the base graph *)
 and  copyGraph nodes =
 let fun mkClass (node as Node{class,flagP,...}) =
         let val  (class' as Node{flag,...}) = mkNode (Abstract [node])
         in (class := SOME class'; flagP:= flag; class') end
     and !! (ref(SOME x)) = x
     and getClass (Node{class,...}) = !! class
     val classes =  map (fn node' => mkClass node') nodes
     val _ = app (fn Node{class,succs,preds,...} =>
             let val Node{succs=succs',preds=preds',flag,...} = !! class    
             in succs' := map getClass (!succs);
                preds' := map getClass (!preds);
		flag := (AbsInfo (Degree (length (!preds'),length (!succs')))) end) nodes
 in classes end 

  (* removes classes that where merged and so are no longer valid *)

 and removeOldClasses []  = []
   | removeOldClasses (Node{flag=ref Stop,...}::cs) =  removeOldClasses cs
   | removeOldClasses (c::cs) = c ::  removeOldClasses cs

 (* successively reduces the new class graph by merging the classes *)
 and reduce nextGroup [] newClasses  = newClasses
   | reduce nextGroup (Node{flag=ref Stop,...}::ns) newClasses =
	 reduce nextGroup ns newClasses
   | reduce nextGroup nodes newClasses =
    	let val (newGroup,ns,absType) = nextGroup nodes
     	in case newGroup of
             [_]    => reduce nextGroup ns newClasses
             | ns' => reduce nextGroup ns ((merge ns' absType):: newClasses) end

 and getSuccs (Node{succs,...}) = (!succs)
 and getPreds (Node{preds,...}) = (!preds)
 and getClass (Node{class = ref (SOME c),...}) = c
 and updateDegree (Node{succs,preds,flag,...}) = 
 		flag := (AbsInfo (Degree (length (!preds),length (!succs))))

  (* merges the classes. The input argument is a list of nodes to be merged.
     It retreives the present classes of these nodes and then all their children
     a superset of the original nodes. These become the children of the new
     class. The neigbours of the new class are the union of the neighbours of
     the old classes minus the old classes. Also, all neigbouring nodes have to be
     diconnected from the old classes and connected to the new one. *)

 and merge (nodes  as (n::ns)) absType' =
     let   fun getChildren (Node{state=Abstract ns,...}) = ns
           val (oldClasses as (c::cs)) = map getClass nodes 
           val allNodes = (fold (op @) (map getChildren cs) (getChildren c))
           val cl as Node{succs,preds,absType,flag,...} = 
                 mkNode (Abstract allNodes)
     in
        succs := abstractLinks oldClasses [] oldClasses  getSuccs;
        preds := abstractLinks oldClasses [] oldClasses  getPreds;
        app (fn Node{preds=preds',...} =>
             preds' :=cl :: removeEdges oldClasses (!preds'))  (!succs);
        app (fn Node{succs=succs',...} =>
             succs' :=cl :: removeEdges oldClasses (!succs')) (!preds);
        app (fn Node{class,flagP,...} => (class := SOME cl; flagP := flag) ) allNodes;
        app updateDegree (!succs);
        app updateDegree (!preds);
        app (fn Node{flag,...} => flag := Stop) oldClasses;        
        absType := absType' ; cl
     end 
 (* builds the new neighbour lists *)
 and abstractLinks [] [] _ linkFn = []
   | abstractLinks (c::cs)  []  excludeList linkFn = 
	abstractLinks cs  (linkFn c)  excludeList linkFn
   | abstractLinks cs (n::ns)  excludeList linkFn = 
      if (exists (sameNode n) excludeList) 
     	 then abstractLinks cs  ns  excludeList linkFn
	 else n :: abstractLinks cs ns  (n :: excludeList)  linkFn

 (* removes references to the old classes from the neighbours'
    neighbours lists *)
 and removeEdges cs [] = []
   | removeEdges cs (c'::cs') =
       if (exists (sameNode c') cs) 
         then removeEdges cs cs' 
         else c' :: removeEdges cs cs'

 and sameNode (Node{flag=x,...}) (Node{flag=y,...}) = x=y

in type abstraction_parameter = absSelect
   val abstractionGenerator = abstract end
end


