

functor AbsGeneratorBypassFUN (structure Graph : GRAPH) : ABSGEN =

  (* BYPASS ABSTRACTION GENERATOR *)

(**** final version.
       In this version, a node can be bypassed if
       it has only one successor or only one predecessor ;
    OR if all its successors are also reachable with a path of length D
       (user-specified) from all its predecessors
    OR if there is one successor for which this isn't true, in which case it
       is that ONE special successor with which the node is merged.

   This applies only to nodes that have some successors and some predecessors.

   A special case of this test, which is treated separately ("alternatives"),
   is when two nodes are alternatives -- you need one or the other but not
   both.  If the current test is applied in this situation, whichever of these
   nodes is considered first will be regarded as redundant. The difference
   between the two is that if "alternatives" finds that N1 and N2 are
   alternatives it merges them; but if "bypass" is applied it will find that
   one of them is redundant and "delete" it (by merging it with one of its
   successors or predecessors, NOT with the node it's equivalent to).


      Ideally, we would like DELETE a node that can be bypassed, but
      I am not sure at present what the ramifications would be of
      doing this.  So the present version just merges the bypassed node
      with ONE of its successors or predecessors.  If there are
      multiple predecessors and multiple successors, the options
      are either to merge it with one of them (which one ?), or to
      merge it with all of them (but then the abstract classes would
      overlap -- I don't know if this is meaningful (it's something Chris
      thought would be a good idea) )

    As with the other abstraction methods, returns a triple of:
            the set of nodes to be combined,
            the list without the first node of this set and
            the abstract type is returned.
****)
  

struct

structure Graph = Graph

local open Graph Random in

type CONDITIONparameter = unit
type ACTIONparameter    = int  (* how far away from the predecessors of N
                                  may the successors of N be and still have
                                  N regarded as redundant ?
                                  A value of 1 means N's successors must also
                                  be immediate successors of N's predecessors
                               *)
(*
       later we might wish to make the ACTIONparameter more complex,
       e.g. to give run-time control over how "weak/strong" the test is
*)


 fun CONDITION _ _ = true           (* Always apply this generator *)


(* the node we want to merge the given one, N, is selected according to
   these preferences (first is the highest preference):
     (1) the unique successor of N
     (2) the unique predecessor of N
     (3) a predecessor or successor that is not in any other class
     (4) a randomly chosen successor or predecessor
*)

datatype answer_type = NoNode | StopNode of node | NonStopNode of node

  fun nth_prefer_Stop [] NoNode _ = NONE
   |  nth_prefer_Stop [] (StopNode n) _ = SOME n
   |  nth_prefer_Stop [] (NonStopNode n) _ = SOME n

   | nth_prefer_Stop ((n as (Node{flag= ref Stop,...}))::tail) _ count =
           if (count <= 1)
           then  (SOME n)
           else nth_prefer_Stop tail (StopNode n) (count-1)

   | nth_prefer_Stop (_::tail)  (sofar as (StopNode _))  count  =
                       nth_prefer_Stop tail sofar (count-1)
   | nth_prefer_Stop (n::tail)  sofar  count  =
           if (count < 1)
           then nth_prefer_Stop tail sofar (count-1)
           else nth_prefer_Stop tail (NonStopNode n) (count-1)

fun remove _ [] = []
 |  remove x (h::t) = if (sameNode x h) then t else (h::(remove x t))

fun remove_all _ [] = []
 |  remove_all x (h::t) = if (sameNode x h)
                          then (remove_all x t)
                          else h::(remove_all x t)

fun choose_a_node (node as (Node {preds,succs,...})) =
    case (remove node (!succs)) of
      [only_one] => SOME only_one
    |      s     =>( case (remove node (!preds)) of
                       [only_one] => SOME only_one
                     |    p       => let val x = s@p
                                      in nth_prefer_Stop x NoNode
                                                (newrandom (length x))
                                     end
                   )

  fun bypass node  =
      case (choose_a_node node) of
           NONE => NONE
       |   SOME (chosen_node as (Node {flag,...})) =>
                              ( flag := Stop ; SOME [node,chosen_node] )


  fun member _ [] = false
   |  member x (h::t) = (sameNode x h) orelse (member x t)

  fun nonmembers _ [] = []
   |  nonmembers MasterList (h::t) =
      if (member h MasterList)
      then nonmembers MasterList t
      else h::(nonmembers MasterList t)

  fun union [] X = X
   |  union X [] = X
   |  union (h::t) X = if (member h X) then (union t X) else (union t (h::X))

  fun extend_by_1 [] sofar = sofar
   |  extend_by_1 ((p as (Node {succs,...}))::t) sofar =
                     extend_by_1 t (union (!succs) sofar)

  fun reachable 0 _ sofar = sofar
   |  reachable d n sofar = sofar @
                            reachable (d-1) n
                                (remove_all n (extend_by_1 sofar []))


  fun redundancy_check _ _ [] _ sofar = sofar
   |  redundancy_check howfar n (p::t) L (sofar as (bad_preds,bad_succs)) =
      (case (nonmembers (reachable howfar n [p]) L)
        of    []  => redundancy_check howfar n t L sofar
        |     nm  => redundancy_check howfar n t L (p::bad_preds, union nm bad_succs)
      )


  fun ACTION (howfar:ACTIONparameter)
             ((node as (Node {preds,succs,flag=nflag,...})) ::others) =
  if   (not ((!nflag) = Stop))
  then
  let val p = remove node (!preds)
      val s = remove node (!succs)
  in
     (case (p,s)
     of  ([] , _ ) => ( [node], others, Another )
      |  ( _ , []) => ( [node], others, Another )

      | _ =>
             (case (redundancy_check howfar node p s ([],[]))
              of  ([],[]) => ( case (bypass node)
                               of SOME x => ( nflag := Stop ;
                                              ( x, others, Bypass ))
                               |    _    => ( [node], others, Another )
                            )
             |   ( _ ,[bads as(Node {flag=xflag,...})]) =>
                         if (not ((!xflag) = Stop))
                         then
                               ( xflag := Stop ; nflag := Stop ;
                               ( [node,bads], others, Bypass ))
                         else  ( [node], others, Another)

             |       _     =>  ( [node], others, Another)
           )
     )
  end
  else  ( [node], others, Another)

end
end
