
type schedule_external_state =  { tasks:  (int * int) list ,
                         deadline: int ,
                         precedence_constraints: (int * int) list ,
                         exclusion_constraints:  (int * int) list
                       }

datatype schedule_puzzle_parameter =
      INITS of schedule_external_state list (* initial states *)
   |  RAND  of (int * int * (int list))
              (* deadline, duration, number in each level *)

structure Schedule : GENERIC_PUZZLE =
struct
local open Random
in


(*  A task is a pair of integers (id,duration).
    The duration is in the same units as a deadline (integer).
    A precedence constraint is a pair of task ids (integers) -- (i1,i2)
          means task i1 must be completed before task i2 can begin.
    An exclusion constraint is a pair of task ids - (i1,i2) means that
          tasks i1 and i2 cannot be done at the same time.
    

   A state is represented by a labelled record
     { tasks, deadline, precedence_constraints, exclusion_constraints }

   A move subtracts one from the deadline and from any subset of the tasks'
   durations, subject to the constraints.

*)


 type external_state =  schedule_external_state
 type internal_state = external_state
 type puzzle_parameter =   schedule_puzzle_parameter

 fun member (_,[]) = false | member (x,h::t) = (x=h) orelse (member (x,t))

 fun subset ([],_) = true
  | subset ((h::t),L) = (member (h,L)) andalso (subset (t,L))

 fun set_equality (L1,L2) = (subset (L1,L2)) andalso (subset (L2,L1))

 fun sameState
    ({tasks=t1,deadline=d1,precedence_constraints=p1, exclusion_constraints=e1},
     {tasks=t2,deadline=d2,precedence_constraints=p2, exclusion_constraints=e2}) =
    (d1=d2) andalso (set_equality (t1,t2)) andalso (set_equality (p1,p2))
           andalso (set_equality (e1,e2))

 fun sum_product [] = 0
  |  sum_product ((a,b)::t) = (a*b + (sum_product t)) mod 9999

 fun hashState {tasks,deadline,precedence_constraints,exclusion_constraints} =
   deadline + 3*( (sum_product tasks)
            + 3*( (sum_product precedence_constraints)
            + 3*( (sum_product exclusion_constraints) )))

fun remove_task _ [] = []
 |  remove_task rid ((h as (hid,_))::t) =
    if (rid = hid) then t else h::(remove_task rid t)

fun unprecedented tasks [] = tasks
 |  unprecedented tasks ((a,b)::t) = unprecedented (remove_task b tasks) t

fun member_task (_,[]) = false
 | member_task (x,(id,_)::t) = (x=id) orelse (member_task (x,t))


(* as this is presently coded it will not delete a task unless doing so is
   necessary to satisfy an exclusion constraint
*)

fun apply_exclusions tasks [] = [tasks]
 |  apply_exclusions tasks ((a,b)::t) =
    if (member_task (a,tasks)) andalso (member_task (b,tasks))
    then    (apply_exclusions (remove_task a tasks) t)
          @ (apply_exclusions (remove_task b tasks) t)
    else  (apply_exclusions tasks t)

fun update_tasks [] _ = ([],[])
 |  update_tasks ((h as (a,b))::t) task_list =
    let val (finished,unfinished) = update_tasks t task_list
    in
        if (member_task (a,task_list))
        then if b=1
             then ( a::finished, unfinished )
             else (  finished, (a,b-1)::unfinished )
        else ( finished, h::unfinished )
    end

fun update_exclusion_constraints [] _ = []
 |  update_exclusion_constraints ((h as (a,b))::t) finished =
    if (member (a,finished)) orelse (member (b,finished))
    then update_exclusion_constraints t finished
    else h :: (update_exclusion_constraints t finished)

fun update_precedence_constraints [] _ = []
 |  update_precedence_constraints ((h as (a,b))::t) finished =
    if (member (a,finished))
    then update_precedence_constraints t finished
    else h :: (update_precedence_constraints t finished)

fun make_move state [] = NONE
 |  make_move  {tasks,deadline,precedence_constraints,exclusion_constraints} 
               task_list =
    let val (finished, unfinished) = update_tasks tasks task_list
    in case unfinished
       of [] => SOME {tasks=[],deadline=0,
                        precedence_constraints=[],exclusion_constraints=[]}
       |   _ => SOME {tasks = unfinished,
                      deadline = (deadline - 1),
                      precedence_constraints =
                                (update_precedence_constraints
                                               precedence_constraints finished),
                      exclusion_constraints =
                                (update_exclusion_constraints
                                               exclusion_constraints finished)
                     } 
    end

fun remove_duplicates [] = []
 |  remove_duplicates (h::t) =
    let val rdt = remove_duplicates t
    in  if member (h,rdt) then rdt else (h::rdt)
    end

 fun cleanup (NONE  ::l) =      cleanup l
  | cleanup (SOME s::l) = s :: cleanup l
  | cleanup []          = []


 fun successors (state as
                {tasks,deadline,precedence_constraints,exclusion_constraints}) =
 if deadline <= 0
 then []
 else case tasks
      of   []  => [{tasks=[],deadline=0,
                    precedence_constraints=[],exclusion_constraints=[]}]
      |    _   =>
                let 
                   val unprecedented_tasks =
                                     unprecedented tasks precedence_constraints
                   val moves =
                        apply_exclusions unprecedented_tasks exclusion_constraints
                in  cleanup (map (make_move state) (remove_duplicates moves))
                end


fun random_element L =
    let val pos = (newrandom (length L))-1
    in  (nth (L,pos))
    end


(* the following takes a state and returns a random successor of that state,
   either a terminal state (one with no successors), or one that is distance
   D from the initially given state.

   (D is really only needed to rpevent infinite loops in cyclic spaces.
    I have set it large enough here that it will never cause termination)
*)

 fun randState 0 state = state
  |  randState D state =
      case (successors state)
      of  [] =>  state
       |  s  => randState (D-1) (random_element s)

fun identity x = x

(* the following function makes a random scheduling problem.
   The user specifies the deadline, duration (of all tasks),
      and the number of tasks to occur at each precedence level
       (a list of integers, the head being the number of tasks that have
        no precence constraints).
   Precedence constraints are created randomly, pairing up tasks in level i
      and level i+1 randomly but with guarantee that every task at level i+1
      is involved in at least one constraint.
   Exclusion constraints are made between task t and t+1 within the same level.
*)


fun make_exclusion_constraints [] = []
 |  make_exclusion_constraints ((a::(b::t1))::t2) =
          (a,b)::(make_exclusion_constraints (t1::t2))
 |  make_exclusion_constraints (_::t) = make_exclusion_constraints t


fun make_pairs (x,[]) = [] | make_pairs (x,(h::t)) = (x,h)::(make_pairs (x,t))

fun all_pairs ([],_) = []
 |  all_pairs ((h::t),L) = (make_pairs (h,L)) @ (all_pairs (t,L))

fun remove (x,[]) = [] | remove (x,h::t) = if x=h then t else (h::(remove (x,t)))

fun Nrandom (0,_) = []
 |  Nrandom (_,[]) = []
 |  Nrandom (n,L) =
    let val x = random_element L
    in  x::(Nrandom ((n-1), remove (x,L)) )
    end

fun pc (L1,L2) =
let val basic_pcs = map (fn x => (random_element L1,x)) L2
    val possible_pcs = all_pairs (L1,L2)
    val number_pcs = newrandom (length possible_pcs)
in remove_duplicates (basic_pcs @ (Nrandom (number_pcs,possible_pcs)))
end

fun make_precedence_constraints (l1::(t as (l2::rest))) =
    (pc (l1,l2)) @  (make_precedence_constraints t)
 |  make_precedence_constraints _ = []

fun upto (start,stop) =
   if (start > stop) then [] else start::(upto (start+1,stop))
fun make_tasks (n,[]) = []
 |  make_tasks (n,(h::t)) = (upto (n,n+h-1))::(make_tasks (n+h,t))

fun flatten [] = [] | flatten (h::t) = h @ (flatten t)

fun randSchedule (deadline, duration, levels) =
let val tasks_at_each_level = make_tasks (1,levels)
in {deadline = deadline ,
    tasks = map (fn x => (x,duration)) (flatten tasks_at_each_level) ,
    precedence_constraints =  make_precedence_constraints tasks_at_each_level ,
    exclusion_constraints  =   make_exclusion_constraints tasks_at_each_level
    }
end

 fun mkInstance parms =
 let val istates =
        case parms
         of  (INITS x) => x
         |   (RAND  x) => [randSchedule x]
 in
     { decode   =  identity ,
       encode   = identity ,
       successor= successors,
       initialStates = istates ,
       randomStatePair = (fn () =>
                           let val xstate =  random_element istates
                            in ( xstate, randState 30 xstate)
                           end )
      }
 end

end
end


(* here's a sample initial state... 
   this generates a space with  881   states in it.
 *)

val istate1 =
  INITS (
    [
    { deadline = 27,
      tasks = [(1,4),(2,4),(3,4),(4,4),(5,4),(6,4),(7,4),(8,4)],
      precedence_constraints = [(1,4),(2,4),(2,5),(3,5),(4,6),(5,7),(5,8)],
      exclusion_constraints  = [(1,2),(4,5),(6,7),(7,8)]
    }
    ]
   )

(* here is the solution (found by AO) -- each step lists the duration remaining
   on each task, I added comments to say what had happened between each:

   [(1,4),(2,4),(3,4),(4,4),(5,4),(6,4),(7,4),(8,4)],    (* start state *)
                                                         (* do tasks: 2, 3  *)
   [(1,4),(2,3),(3,3),(4,4),(5,4),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 2, 3  *)
   [(1,4),(2,2),(3,2),(4,4),(5,4),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 2, 3  *)
   [(1,4),(2,1),(3,1),(4,4),(5,4),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 2, 3  *)
   [(1,4),(4,4),(5,4),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 1, 5  *)
   [(1,3),(4,4),(5,3),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 1, 5  *)
   [(1,2),(4,4),(5,2),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 1, 5  *)
   [(1,1),(4,4),(5,1),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 1, 5  *)
   [(4,4),(6,4),(7,4),(8,4)],
                                                         (* do tasks: 4, 7  *)
   [(4,3),(6,4),(7,3),(8,4)],
                                                         (* do tasks: 4, 7  *)
   [(4,2),(6,4),(7,2),(8,4)],
                                                         (* do tasks: 4, 7  *)
   [(4,1),(6,4),(7,1),(8,4)],
                                                         (* do tasks: 4, 7  *)
   [(6,4),(8,4)],
                                                         (* do tasks: 6, 8  *)
   [(6,3),(8,3)],
                                                         (* do tasks: 6, 8  *)
   [(6,2),(8,2)],
                                                         (* do tasks: 6, 8  *)
   [(6,1),(8,1)],
                                                         (* do tasks: 6, 8  *)
   []                                                    (*** all done  ***)
- 




*)

(* same as above, but with another layer of jobs on it.
   This generates a space with > 2060  nodes in it.
 *)

val istate2 =
  INITS (
    [
    { deadline = 32,
      tasks = [(1,4),(2,4),(3,4),(4,4),(5,4),(6,4),(7,4),(8,4),(9,2),(10,2),
               (11,5), (12,6)],
      precedence_constraints = [(1,4),(2,4),(2,5),(3,5),(4,6),(5,7),(5,8),
                                (6,9),(7,9),(7,11),(8,12),(9,10)],
      exclusion_constraints  = [(1,2),(4,5),(6,7),(7,8)]
    }
    ]
   )

(*    234 states.  2x speedup.  solution length = 11 (optimal)    *)

val rstate1 =   (RAND (15,2,[4,4,4]))



(*   4412 states.  5 mins to build.   35x speedup.
     without abstraction, solution length = 25 (optimal),
     abstraction gives maximal solution length (set by deadline = 27)
*)

val rstate2 =   (RAND (27,2,[6,6,6,6,6,6,6]))  



(*   1030 states.  4 mins to build.   17x speedup.
     solution length = 17 (optimal)
*)

val rstate3 =    (RAND (22,2,[8,8,7,7])) 



