

(*** A state is a list of integers (typically, the integers 1..N).
     There are N operators: operator K reverses the order of the first
     K elements.  For example, applying operator 3 to  [a,b,c,d,e,f,g]
     would produce [c,b,a,d,e,f,g].

     The interesting thing about this space is that it grows exponentially
     with N (all N! permutations are reachable from each other), but the
     furthest distance between 2 states grows linearly: there is a path of
     length 2N or less joining any pair of states.

     Proof:  in two steps you can get the correct element at the end of the
             list -- if the element is in position K to start with, apply
             operator K (this makes it first in the list), and then apply
             operator N (this moves the first element to the last position).

             We then get the correct element into position N-1 by moving it to
             the top and then aplying operator N-1.  Then we do N-2, N-3, etc.

*)

structure Permute : GENERIC_PUZZLE =
struct
local open Random
in

  type external_state = int list ;
  type internal_state =   external_state ;
  type puzzle_parameter = external_state ;  (* any state will do *)

  fun sameState  (S1:internal_state, S2) =  (S1=S2)

  fun hashState [] = 0
   |  hashState (h::t) = h + 8*(hashState t) ;

  fun encode n = n ;
  val decode   = encode ;


(* the rest of this is to define the successor function *)

fun cut _ [] = ([],[])
 |  cut 0 L  = ([],L)
 |  cut n (h::t) =
    let val (front,back) = cut (n-1) t
     in (h::front,back)
    end

fun generator L K  =
let val (front,back) = cut K L
in  (rev front) @ back
end

fun repeat start finish F =
    if start > finish
    then []
    else ((F start)::(repeat (start+1) finish F))


fun mkGenerator state = repeat 2 (length state) (generator state)

fun randState 1 s = s
 |  randState n s = let val new_s = rev (generator s (newrandom n))
                    in randState (n-1) new_s
                    end

fun mkInstance State =
   { decode = decode,
     encode = encode,
     successor = mkGenerator,
     initialStates = [State],
     randomStatePair = (fn () => ( randState (length State) State,
                                   randState (length State) State ))
   }

end
end 
