;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)

(defvar *cur-exemplar*)   ;;global of current exemplar being used.
(defvar *feat*)           ;;global of current feature
(defvar *cat*)            ;;global of current category
(defvar *exp*)            ;;global of current exemplar


;;;=============================================================================
;;;
;;;           D I S C U S S    M A T C H    W I T H    T E A C H E R
;;;   ------------------------------------------------------------------------
;;;
;;;  Overview:	This file contains the functions that Protos uses to discuss
;;;		a proposed exemplar-to-new case match.  The top-level function
;;;		where it all starts is "discuss-match".
;;;
;;;  Functions:	 add-new-exemplar
;;;              get-approximate-importances
;;;              discuss-differences
;;;              discuss-high-importances
;;;		 discuss-match
;;;		 discuss-success
;;;		 discuss-failure
;;;		 discuss-exemplars
;;;		 discuss-relevances
;;;		 discuss-removals
;;;		 discuss-unmatched
;;;              discuss-unmatched-importances
;;;              enter-explanation
;;;		 read-category
;;;=============================================================================



(defparameter *exemplar-menu* (make-menu
  :label  "~%What would you like to do?"
  :items  '((#\T . ("Try the most prototypical remaining exemplar."   return  next-exemplar))
	    (#\S . ("Select a specific exemplar of this category."    return  show-exemplars))
	    (#\C . ("Create new exemplar from this case."	      return  add-exemplar))
	    (#\A . ("Abandon this case."			      return  abandon)))))


;;;----------------------------------------------------------------------------
;;;  Function:  (discuss-match  match)
;;;
;;;  Given:	match, the details of an exemplar-to-newcase match
;;;
;;;  Returns:   2 values: action and match, where:
;;;             -- action is either:
;;;                'done  -- match contains an approved match structure;
;;;                'newex -- match contains a newly created exemplar;
;;;                'retry -- retry finding a match for this case;
;;;                'redo  -- redo the match to this exemplar;
;;;                'next  -- try the next hypothesis.
;;;
;;;  Called by: compare
;;;
;;;  Notes:	Discuss-match presents the match to the teacher and asks
;;;		if the classification is correct.  Regardless of the answer,
;;;		the teacher is presented with other questions which may
;;;		modify the current match.  It is in the subordinate functions
;;;		of discuss-match where Protos learns from the teacher.
;;;----------------------------------------------------------------------------

(defun discuss-match (match)
  (print-match match)
  (examine-explanations match)
  (let ((prompt1 (format nil "~%Is ~A the correct classification for this case? "
			 (getname (exemplar-category (match-exemplar match))))))
    (if (prompt prompt1 nil 'y-or-n nil nil)
	(discuss-success match)
	(discuss-failure match))))


;;;----------------------------------------------------------------------------
;;;  Function:  (discuss-success  match)
;;;
;;;  Given:	match, the results of an exemplar-to-new case match where the
;;;		       the teacher has approved the classification (but hasn't
;;;                    yet approved the exemplar that the case was matched to).
;;;
;;;  Returns:	-- the same match if the teacher agrees that the chosen
;;;		   exemplar is appropriate;
;;;		-- a different match if the teacher chooses a different
;;;		   exemplar; or
;;;		-- nil if no more action is needed on this match (such as when
;;;		   the teacher decides to make a new exemplar from the case or
;;;                the teacher decides to abandon the case).
;;;
;;;  Caller:	discuss-match
;;;----------------------------------------------------------------------------

(defun discuss-success (match)

  (let* ((newcase	(match-newcase match))
	 action)

    ;; Set the classification of the new case.
    (setf (case-category newcase) (exemplar-category (match-exemplar match)))

    (multiple-value-setq (action match) (discuss-exemplars match))

    ;; Return if teacher decided to make exemplar of the new case.
    (if (eql 'newex action)
	(return-from discuss-success (values 'newex match)))

    ;; Return if the case is being abandoned by the user.
    (if (eql 'done action)
	(return-from discuss-success (values 'done match)))

    ;; Ask about unmatched features.
    (discuss-unmatched match)

    ;; Ask for unfocused instruction.
    (discuss-unfocused)

    ;; Update prototypicality of the exemplar.
    (if (mergeable match)
	(progn
	  (if (prompt (format nil
			      "~%~%This case (~A) is similar enough to ~A that Protos~
                                 ~%is prepared to forget the case (and just retain ~A).~
                               ~%~%Will it ever be important to distinguish between these two? "
		  (case-name newcase)
		  (getname (match-exemplar match))
		  (getname (match-exemplar match)))
		      nil 'y-or-n nil nil)

	      (progn
		(update-prototypicality (match-exemplar match) 0.75)
		(return-from discuss-success (values 'newex (add-new-exemplar newcase))))
	      (progn
		(update-prototypicality (match-exemplar match) 1.0)
		(merge-exemplar match)
		(return-from discuss-success (values 'done match)))))

	;; match was not mergeable
	(progn
	  (format t "~%This case is different enough from ~A~
                     ~%that it will be made into an exemplar."
		  (getname (match-exemplar match)))
	  (if (>= (match-similarity match) *strong-match*)
	      (update-prototypicality (match-exemplar match) 0.50)
	      (update-prototypicality (match-exemplar match) 0.25))
	  (return-from discuss-success (values 'newex (add-new-exemplar newcase)))))))



;;;----------------------------------------------------------------------------
;;;  Function:  (update-prototypicality  exemplar  amount)
;;;
;;;  Purpose:   This function increments the given exemplar's typicality value
;;;             and then (re-)sorts the affected category's list of exemplars
;;;             in decreasing order of typicality.
;;;----------------------------------------------------------------------------

(defun update-prototypicality (exemplar amount)
  ;; Add amount to the exemplar's typicality value.
  (incf (exemplar-typicality exemplar) amount)
  ;; Re-sort the category's exemplars by typicality value.
  (let ((category (exemplar-category exemplar)))
    (setf (category-exemplars category)
	  (sort (category-exemplars category) #'>= :key #'exemplar-typicality))))

;;;----------------------------------------------------------------------------
;;;  Function:  (discuss-exemplars match)
;;;
;;;  Given:	match, a match to a Protos-selected exemplar of the correct
;;;		category;
;;;
;;;  Returns:	3 values, action and match, where:
;;;             -- action may be either 'mrgex, 'done or 'newex, and
;;;             -- match is either a teacher-approved exemplar of the same
;;;                category, or is a new exemplar (as when the teacher
;;;                decides to make a new exemplar of the case).
;;;
;;;  Caller:	discuss-success
;;;----------------------------------------------------------------------------

(defun discuss-exemplars (match)
  (declare (special closed-exemplars))
  
  (let* ((exemplar (match-exemplar match))
	 (newcase  (match-newcase  match))
	 (viewed-exemplars (list exemplar))
	 (category (exemplar-category exemplar))
	 (exemplars (category-exemplars category))
	 (exemplist (copy-list exemplars)))
    
    (loop
      (tagbody
	(if (prompt "~%Is this a suitable exemplar for matching the new case? "
		    nil 'y-or-n nil nil)
	    (return-from discuss-exemplars (values 'mrgex match)))

	;; The teacher has rejected this match, saying that this is not a
	;; suitable exemplar.  So, remember this fact so that the function
	;; "test-hypotheses2" will skip over this exemplar if it appears in
	;; the list of hypotheses.  This could happen if we reached this
	;; unsuitable exemplar through a difference link.
	(push (match-exemplar match) closed-exemplars)

	;; If this incorrect match is strong enough (i.e., is a "near miss"),
	;; then remember it for possibly installing a difference link later
	;; when the case finally is correctly classified.
	(if (>= (match-nth-root-of-similarity match) *near-miss-threshold*)
	    (push match *near-misses*))

	;; Teacher has said no, this isn't a suitable exemplar for the new case.
	;; So, Protos now offers the teacher one of 4 choices:
	;;   1.  Take the most prototypical exemplar remaining and match to it.
	;;   2.  Display the names of this category's exemplars and allow teacher
	;;       to select one of them.
	;;   3.  Install this case as a new exemplar of its category.
	;;   4.  Abandon this case.
	
	(ecase (menu-select *exemplar-menu*)
	  (next-exemplar
	    ;; Find the next most prototypical exemplar that hasn't already been viewed.
	    (do ()
		((endp exemplist))
	      (setq exemplar (pop exemplist))
	      (if (not (member exemplar viewed-exemplars))
		  (go SHOW-EXEMPLAR)))
	    
	    ;; No more exemplars left on exemplist.
	    (if (prompt "~%You have rejected all exemplars in this category.~
                         ~%The only choice now is to either make a new exemplar~
                         ~%of this case or abandon the case.~
	                 ~%Shall I make a new exemplar of this case? "
			nil 'y-or-n nil nil)
		(return-from discuss-exemplars (values 'newex (add-new-exemplar newcase)))
		(progn
		  (format t "~%Then this case is being abandoned.")
		  (return-from discuss-exemplars (values 'done nil)))))
	  
	  (show-exemplars
	    ;; Print the names of all the exemplars of this category.
	    (terpri)
	    (print-node-names exemplars t ", ")
	    
	    (loop
	      (let ((input (prompt "~%Please enter one of the above names:  "
				   nil 'termname category 'category)))
		(setq exemplar (find input exemplars :test #'equal :key #'node-name))
		(if exemplar (go SHOW-EXEMPLAR)))))
	  
	  (add-exemplar
	    ;; Make a new exemplar of the new case.
	    (setf (case-category newcase) category)
	    (return-from discuss-exemplars (values 'newex (add-new-exemplar newcase))))
	  
	  (abandon
	    (return-from discuss-exemplars (values 'done nil))))
	
	
     SHOW-EXEMPLAR
	;; This is the bottom of the loop that begins by asking if the exemplar
	;; in the displayed match is a suitable match for newcase.  We arrive
	;; here only if the teacher has selected a different exemplar to examine.
	(push exemplar viewed-exemplars)
	(setq match (compare-new-case exemplar newcase))
	(print-match match)
	(examine-explanations match)))))

;;;----------------------------------------------------------------------------
;;;  Function:  (add-new-exemplar  newcase)
;;;
;;;  Given:	newcase, a case from which an exemplar is to be built and
;;;			 installed;
;;;
;;;  Returns:	This function is executed for its side-effect of creating and
;;;		installing a new exemplar in the category network.
;;;
;;;  Callers:	discuss-success, discuss-failure, discuss-exemplars.
;;;
;;;  Design:	-- If newcase's category is unknown, prompt for it.
;;;		-- Ask if any features of newcase should NOT become part of
;;;		   the exemplar, and then remove them.
;;;		-- Create the exemplar from newcase and install it as a member
;;;		   of the given category in the category network.
;;;		-- For each feature that is newly occurring within the category,
;;;		   ask about its relevance to the category.  For each explana-
;;;		   tion given by the teacher, glean remindings from F-->C
;;;		   explanations and estimate importances from C-->F explana-
;;;		   tions.
;;;		-- Initialize prototypicality of new exemplar to 1.0.
;;;		-- Install pending difference links.
;;;----------------------------------------------------------------------------

(defun add-new-exemplar (newcase)

  (let ((category  (case-category newcase))
	exemplar)
    
    ;; If newcase's category is unknown, then prompt for it.
    (if (null category)
	(progn
	  (setq category (read-category))
	  (setf (case-category newcase) category)))

    ;; Ask if any features of newcase should NOT become features of the
    ;; exemplar.  If so, remove them from the newcase.
    (discuss-removals newcase)

    ;; Create the exemplar from newcase and install it as a member
    ;;   of the given category in the category network.
    (setq exemplar (make-term :name     (case-name     newcase)
			      :comment  (case-comment  newcase)
			      :features (case-features newcase)
			      :category (case-category newcase)
			      :typicality 1.0))
    (push exemplar *history*)

    (setq *cur-exemplar* exemplar)

    ;; Create the symbol that has the print-name of this exemplar.
    ;; (note that this is a 'set', not a 'setf').
    (set (case-name newcase) exemplar)

    ;; Update number of exemplar-containing categories.
    (if (null (category-exemplars category))
	(incf *number-of-ec-categories*))

    ;; Add this new exemplar to the category's list of exemplars (which are
    ;; sorted by typicality).  Since this is a brand new exemplar, it will
    ;; have the lowest possible typicality and can therefore simply be
    ;; added to the end of the list.  Also, install the "has-exemplar" and
    ;; "is-exemplar-of" relations between the category and exemplar.
    (setf (category-exemplars category)
	  (nconc (category-exemplars category) (list exemplar)))
    (install-relation (list category) (list exemplar) nil *verb-hasExemplar* nil)

    ;; For each feature of this exemplar, save a pointer to this exemplar
    ;; in the feature's structure.  This is used in combine-remindings
    ;; (specifically, strengthen-features) so that if featural exemplars
    ;; are present in a match, they will be matched first.
    (dolist (feature (exemplar-features exemplar))
      (push exemplar (feature-of-exemplars feature)))

    ;; Discuss any fault variables that should be added to the category.
    (discuss-faultvars exemplar)

    ;; Discuss relevance of features new to this exemplar's category.
    (discuss-relevances exemplar)

    ;; Discuss category's unmatchable high-importance features.
    (discuss-high-importances category exemplar)

    ;; Install pending difference links.
    (discuss-near-misses exemplar)

    ;; End of function.
    (if *trace-new-exemplar*
	(progn
	  (format t "~%The following new exemplar has been created:~%")
	  (print-exemplar exemplar t 1)))

    exemplar))

;;;---------------------------------------------------------------------------
;;;  Function:	(read-category)
;;;
;;;  Returns:	A category node.
;;;
;;;  Design:	This function prompts the user for a category name.  If the
;;;		input is recognized as an existing category, then a pointer
;;;		to that category node is returned.
;;;
;;;  Callers:	add-new-exemplar
;;;---------------------------------------------------------------------------

(defun read-category ()

  (loop
    (let* ((input  (prompt "~&What is this case's category?  " nil 'termname
			   nil nil))
	   (object (check-term-name input 'ask)))

      ;; If the name entered by the user exists or has been created ...
      (if object

	  ;; then make sure it is a category (as opposed to, say, an exemplar)
	  (if (my-exemplar-p object)
	      (progn
		(format *query-io*
			"~%Error: ~A is an exemplar, not a category.~
	                 ~%Please enter a different category name: "
			input))
	      (return-from read-category object))

	  ;; user apparently said to forget this term.
	  (format *query-io* "~%Sorry, but you HAVE to enter a category name: ")))))

;;;---------------------------------------------------------------------------
;;;  Function:	(discuss-removals  newcase)
;;;
;;;  Given:	newcase, a case to be made into an exemplar
;;;
;;;  Purpose:	The teacher is asked if any of the features of the new case
;;;		are irrelevant to its future role as an exemplar of the
;;;		category.  If so, the teacher is allowed to remove those
;;;		features.
;;;
;;;  Called by:  add-new-exemplar
;;;---------------------------------------------------------------------------

(defun discuss-removals (newcase)

  (let ((features   (case-features newcase))
	(category   (case-category newcase))
	input)

    (format *query-io* "~%Case ~A is composed of the following features:~%"
			(case-name newcase))
    (print-node-names features *query-io* ", ")

    (if (not (prompt (format nil "~%Are any of these features IRRELEVANT~
	 	                  ~%to its role as an exemplar of ~A ? "
			     (getname category))
		     nil 'y-or-n nil nil))
	(return-from discuss-removals (values)))

    (setq input (prompt "~%Then please enter, one per line:" "~&---> "
			'termname newcase 'case))

    ;; If no features to be deleted, then just return.
    (if (null input)
	(return-from discuss-removals (values)))

    ;; Verify that each feature to be deleted was spelled correctly.
    (dolist (fname input)
      (let ((feature (check-term-name fname 'ask)))
	(if feature
	    (if (member feature features)
		(setq features (delete feature features))
		(format *query-io* "~%Skipping ~A -- it wasn't a feature of the case anyway."
			  fname)))))

    ;; Done deleting features -- store results back in newcase.
    (setf (case-features newcase) features)

    (format *query-io* "~%Thank you.  Case ~A now contains only these features:~%"
	    (case-name newcase))
    (print-node-names features *query-io* ", ")
    (terpri)))

;;;-----------------------------------------------------------------------------
;;;  Function:  (discuss-faultvars  exemplar)
;;;
;;;  Purpose:   This functions adds new fault variables to the category, with
;;;             the teachers approval.
;;;
;;;  Design:    1.  Collect all the variables specified as features of this
;;;                 exemplar.
;;;             2.  Remove the variables that are already known to be fault
;;;                 variables of this exemplar's category.
;;;             3.  For each remaining variable, ask the teacher if the
;;;                 variable is a fault variable.
;;;-----------------------------------------------------------------------------

(defun discuss-faultvars (exemplar)
  (let* ((category         (exemplar-category exemplar))
	 (known-faultvars  (category-faultvars category))
	 (maybe-faultvars  nil)
	 pred)

    ;; Check every feature of exemplar for possible fault variables.
    (dolist (feature (exemplar-features exemplar))
      (if (and (setq pred (term-predicate feature))
	       (predicate-relations pred)
	       (not (member pred known-faultvars)))
	  (push pred maybe-faultvars)))

    ;; Ask user about each candidate fault variable.
    (dolist (var maybe-faultvars)
      (if (prompt (format nil "~%Is ~A a fault variable of ~A ? "
			  (getname var) (getname category))
		  nil 'y-or-n nil nil)
	  (push var (category-faultvars category))))))


;;;-----------------------------------------------------------------------------
;;;  Function:  (discuss-near-misses  exemplar)
;;;
;;;  Given:     -- an exemplar which the new case has either been merged with
;;;                or has been made into, and
;;;             -- *near-misses*, a global list of strong matches to exemplars
;;;                that were rejected by the teacher.
;;;
;;;  Does:      For each near miss, the teacher is shown the unmatched features
;;;             of the new case and the near-miss-exemplar, and is asked to
;;;             specify which features are important discriminators.
;;;             If any of the discriminating features currently is of low or
;;;             spurious importance to the target exemplar, then ask the teacher
;;;             for a reassessment of the importance.
;;;-----------------------------------------------------------------------------

(defun discuss-near-misses (chosen-exemplar)

  ;; Consider each near-miss in turn ...
  (dolist (match *near-misses*)

    ;; Guard against the unusual situation where a near-miss was recorded to
    ;; what turned out to be the chosen exemplar.  This can happen in the rare
    ;; case where the teacher changes his/her mind about a match.

    (if (not (eq chosen-exemplar (match-exemplar match)))
	(let* ((exemplar     (match-exemplar match))
	       (c-unmatched  (match-unmatched match))
	       (e-unmatched  nil))

	  ;; Collect non-spurious features of exemplar not matched by current case.
	  (dolist (result (match-results match))
	    (if (and (eql 'unmatched (result-type result))
		     (/=  0.0  (result-importance result)))
		(push (result-feature result) e-unmatched)))
	  
	  (format t "~%~%Protos previously mistook this case for the exemplar ~A."
		  (getname exemplar))
	  
	  (discuss-differences e-unmatched chosen-exemplar exemplar t)
	  (discuss-differences c-unmatched exemplar chosen-exemplar nil)))))

;;;-----------------------------------------------------------------------------
;;;  Function:  (discuss-differences  features  from-exemplar  to-exemplar
;;;                                      from-chosen)
;;;
;;;  Given:     a list of features of the to-exemplar that were not matched by
;;;             the from-exemplar, and a 'from-chosen' flag that is non-nil if
;;;             from-exemplar is the "chosen exemplar" for the new case.
;;;
;;;  Does:      installs a difference link in the from-exemplar pointing to the
;;;             to-exemplar, annotated with a subset of 'features' approved by
;;;             the teacher as important discriminating features.  If the 
;;;             teacher feels that none of 'features' are important discrimina-
;;;             tors, then no difference link is installed.
;;;-----------------------------------------------------------------------------

(defun discuss-differences (features from-exemplar to-exemplar from-chosen)

  (if (null features)
      (return-from  discuss-differences  nil))

  (if from-chosen
      ;; Ask teacher about discriminating features of mistaken exemplar.
      (format t "~%The features of ~A that were not matched by the current case are:~%"
	      (getname to-exemplar))
      ;; Ask teacher about discriminating features of the chosen exemplar.
      (format t "~%The features of this case that were not matched by ~A are:~%"
	      (getname from-exemplar)))
      
  (let ((diff-features nil))
    (dolist (feature features)
      (format t "  ~A"  (getname feature)))

    (format t "~%~%Which of these features (if any) are important discriminators?")
    (dolist (feature features)
      (if (y-or-n-p  "~&--> ~A ?~26T"  (getname feature))
	  (push feature diff-features)))
    
    ;; If no features were specified, then just return.
    (if (null diff-features)
	(progn
	  (format t "~%Since no features were designated as important discriminators,~
                     ~%then no difference link will be created/changed from ~A to ~A"
		  (getname from-exemplar)  (getname to-exemplar))
	  (return-from discuss-differences (values))))
    
    ;; Display the difference link to be installed.
    (let ((newdiff  (make-difference :node to-exemplar
				    :features diff-features))
	  (olddiffs (exemplar-differences from-exemplar)))

      (format t "~%Installing new difference link:~
                 ~%~12@A: "  (getname from-exemplar))
      (print-difference newdiff t 1)
      (terpri t)

      ;; Before installing the new difference link, check to see if there
      ;; is already a difference link between these two exemplars.
      ;; If so, ask the user if it should be retained or deleted.

      (dolist (olddiff olddiffs)
	(if (eq to-exemplar (difference-node olddiff))
	    (progn
	      (format t "~%A previous difference link exists:~
                         ~%~12@A: "  (getname from-exemplar))
	      (print-difference olddiff t 1)
	      (if (y-or-n-p "~&~%Do you want this to be deleted? ")
		  (setf (exemplar-differences from-exemplar) (delete olddiff olddiffs))))))

      (push newdiff (exemplar-differences from-exemplar))

      ;; All of the discriminating features should probably have at least
      ;; moderate importance to the exemplar or category possessing those
      ;; features.  If not, ask teacher to reassess the importance.
      (dolist (feature diff-features)
	(multiple-value-bind (imp node) (get-importance feature to-exemplar)
	  (declare (ignore node))
	  (if (or (null imp) (< (cdr imp) *reassess-importance-threshold*))
	      (progn
		(format t "~%~%~A is a discriminating feature, but it is currently~
                             ~%of ~A importance.  Please reassess its importance."
			(getname feature) (qualitative-value (cdr imp) 'importance))
		(reassess-importance feature to-exemplar))))))))


;;;-----------------------------------------------------------------------------
;;;  Function:	(qualitative-value  strength  type)
;;;
;;;  Returns:	a word giving a qualitative interpretation of the strength of
;;;		an importance or reminding.
;;;
;;;  Note: 	There is a "clinical scale" of normal, mild, moderate,
;;;		severe, profound.
;;;-----------------------------------------------------------------------------

(defun qualitative-value (strength type)
  (if (null strength) (return-from qualitative-value 'unknown))
  (ecase type
    (importance  (cond ((null strength)                        'spurious)
		       ((>= strength *importance-necessary*)   'necessary)
		       ((>= strength *importance-high*)        'high)
		       ((>= strength *importance-moderate*)    'moderate)
		       ((>= strength *importance-low*)         'low)
		       (t		                       'spurious)))

    (reminding   (cond ((=  strength *reminding-absolute*)     'absolute)
		       ((>= strength *reminding-strong*)       'strong)
		       ((>= strength *reminding-moderate*)     'moderate)
		       ((>  strength (- *reminding-moderate*)) 'weak)
		       ((>  strength (- *reminding-strong*))   'moderate)
		       ((>  strength (- *reminding-absolute*)) 'strong)
		       (t                                      'absolute)))))

;; Function to compute the average of a set of numbers.
(defun average (&rest values)
  (/ (apply #'+ values) (length values)))


(defparameter *importance-alist*
	      `((necessary . ,(average *importance-absolute*  *importance-necessary*))
		(high      . ,(average *importance-necessary* *importance-high*))
		(moderate  . ,(average *importance-high*      *importance-moderate*))
		(low       . ,(average *importance-moderate*  *importance-low*))
		(spurious  . ,0.0)))


(defparameter *decide-importance-menu* (make-menu
      :label  "~%Do you agree?"
      :items  '((#\Y . ("Yes."                                   return  yes))
		(#\E . ("no, let me revise the Explanation"      return  revise-expl))
		(#\I . ("no, let me just revise the Importance"  return  revise-imp)))))

(defparameter *importance-menu* (make-menu
      :label  "~%What do you believe its importance to be?"
      :twocol t
      :items  '((#\N . ("Necessary"                 return  necessary))
		(#\L . ("Low"                       return  low))
		(#\H . ("High"                      return  high))
		(#\S . ("Spurious"                  return  spurious))
		(#\M . ("Moderate"                  return  moderate))
		(#\Q . ("Quit (leave it unchanged)" return  nochange)))))


;;;-----------------------------------------------------------------------------
;;;  Function:  (reassess-importance  feature  node)
;;;
;;;  Purpose:   Given a feature and a node (an exemplar or category) for which
;;;             we want to reassess the feature's importance, this function
;;;             shows the user the feature's current importance (if any) in
;;;             qualitative form and asks the user for a revised qualitative
;;;             value.
;;;-----------------------------------------------------------------------------

(defun reassess-importance (feature node1)
  (let (qual-imp1
	qual-imp2
	target)
    
    ;; Get importance of this feature, and node to which it is important.
    (multiple-value-bind (importance node2)
	(get-importance feature node1)

      ;; Set the target node which is to get the revised importance value.
      ;; If no importance already exists, then make the target be a category
      ;; rather than an exemplar.
      (setq target (if importance
		       node2
		       (if (my-exemplar-p node1)
			   (exemplar-category node1)
			   node1)))
      
      ;; Convert importance value to a qualitative form (high, moderate, etc.).
      (setq qual-imp1 (if importance
			  (qualitative-value (cdr importance) 'importance)
			  'unknown))
      (format t "~%~%~A currently is of ~A importance to ~A"
	      (getname feature)
	      qual-imp1
	      (getname target))
      
      ;; Ask teacher for a revised qualitative value of importance.
      (setq qual-imp2 (menu-select *importance-menu*))
      ;; If it's not the same as the old value then modify the importance.
      (if (and (not (equal 'nochange qual-imp2))
	       (not (equal qual-imp1 qual-imp2)))
	  (let ((new-imp  (cdr (assoc qual-imp2 *importance-alist*))))
	    ;; If there was a previous importance cons ...
	    (if importance
		;; Then modify its strength
		(rplacd importance new-imp)
		;; Else create a new importance.
		(push (cons feature new-imp) (category-importances target)))
	    ;; Sort the category's importances.
	    (setf (category-importances target)
		  (sort (category-importances target)  #'>=  :key #'cdr)))))))

;;;-----------------------------------------------------------------------------
;;;  Function:	(get-unknown-features  exemplar)
;;;
;;;  Given:	exemplar, a newly-created exemplar
;;;
;;;  Returns:	a list of features of the new exemplar whose importance is
;;;		unknown to its category or immediate general categories.
;;;
;;;  Design:	For each feature of the new exemplar, this function looks to
;;;		see if the importance of this feature is known:
;;;		-- idiosyncratically to the exemplar,
;;;		-- to the exemplar's category, or
;;;		-- to the next-most-general category(s).
;;;		
;;;		If not found in any of these places, then the feature is
;;;		included in the list of unknowns that is returned.
;;;		If the feature is found in the next-most-general category,
;;;		then its importance is copied into the exemplar's category.
;;;
;;;             !*!*!*!  I need to ask Ray if this last step is wise.  !*!*!*!
;;;
;;;  Called by:  
;;;-----------------------------------------------------------------------------

(defun get-unknown-features (exemplar)

  (let ((category  (exemplar-category exemplar))
	(unknowns  (copy-list (exemplar-features exemplar))))

;    ;; Delete from unknowns all features having idiosyncratic importance.
;    (dolist (imp (exemplar-importances exemplar))
;      (setq unknowns (delete (car imp) unknowns :count 1)))

    ;; Delete from unknowns all features having importance to the category.
    (dolist (imp (category-importances category))
      (setq unknowns (delete (car imp) unknowns :count 1)))

    ;; Return now if all featural importances are known.
    (if (null unknowns) (return-from get-unknown-features nil))

    ;; Delete from unknowns all features having importance to the
    ;;   next-most-general category(s).
    (dolist (rel (node-relations category))
      (if (eq *verb-hasTypicalGen* (relation-verb rel))
	  (dolist (cat (relation-to-nodes rel))
	    (dolist (imp (category-importances cat))
	      (if (member (car imp) unknowns)
		  (push imp (category-importances category)))
	      (setq unknowns (delete (car imp) unknowns :count 1))
	      (if (null unknowns) (return-from get-unknown-features nil))))))
    unknowns))


;;;---------------------------------------------------------------------------
;;;  Function:	(discuss-high-importances  category  exemplar)
;;;
;;;  Given:	-- category, the category of the new exemplar, and
;;;             -- exemplar, the newly-created exemplar
;;;
;;;  Purpose:	If this category already has another exemplar, i.e., a
;;;             prototype, then check every high-importance feature of the
;;;             prototype to see if it can be matched to a feature of the
;;;             new exemplar.  If not, ask the teacher if he/she wants to
;;;             lower the importance of that feature.
;;;
;;;  Notes:	This procedure was part of the original Prolog Protos but
;;;             was not mentioned in Ray Bareiss's dissertation since it
;;;             was considered a detail.
;;;
;;;  Caller:	add-new-exemplar
;;;---------------------------------------------------------------------------

(defun discuss-high-importances (category exemplar)
  (let ((prototype  (first (category-exemplars category)))
	(e-features (exemplar-features exemplar))
	(imps       nil))

    ;; If not in learning mode, then just return.
    (if (null *learning-mode*)
	(return-from discuss-high-importances (values)))

    ;; If the new exemplar IS the prototype, then just return.
    (if (eq prototype exemplar)
	(return-from discuss-high-importances (values)))

    ;; For each feature of known importance to the category ...
    (dolist (imp (category-importances category))
      (let* ((feature  (car imp))
	     (strength (cdr imp)))
	;; If the feature is of high importance and is NOT a feature of
	;; this new exemplar ...
	(if (and (>= strength *importance-high*)
		 (member feature (exemplar-features prototype))
		 (not (member feature e-features)))
	    (push imp imps))))

    (if (null imps)
	(return-from discuss-high-importances (values)))

    (setq imps (nreverse imps))
    (format t "~2%The following are features of the prototype of ~A (~A)~
                ~%that are currently of high importance to the category.~
                ~%Protos is going to see if each of these features matches~
                ~%a feature of the new exemplar."
	    (getname category) (getname prototype))

    (let ((column 0))
      (dolist (imp imps)
	(if (= column 0) (terpri))
	(setq column (mod (1+ column) 3))
	(format t " ~25A" (getname (car imp))))
      (terpri))

    (dolist (imp imps)
      (let ((feature  (car imp))
	    (strength (cdr imp)))

	(format t "~%   ~A ... " (getname feature))
	  
	;; Then see if feature is related to an exemplar feature.
	(let ((result (kbpm 'FtoF feature strength prototype e-features)))

	  (princ (result-type result))

	  ;; If not, then ask teacher to reassess (lower) the importance
	  ;; of this feature to the category.
	  (if (eql 'unmatched (result-type result))
	      (let ((qual-imp1 (qualitative-value strength 'importance))
		    qual-imp2)
		(format t "~2%~A is a feature of ~A importance to category ~A~
                            ~%that does not match any feature of this new exemplar.~
                            ~%The importance of ~A may be set too high."
			(getname feature)
			qual-imp1
			(getname category)
			(getname feature))
      
		;; Ask teacher for a revised qualitative value of importance.
		(setq qual-imp2 (menu-select *importance-menu*))

		;; If it's not the same as the old value then modify the importance.
		(if (and (not (equal 'nochange qual-imp2))
			 (not (equal qual-imp1 qual-imp2)))
		    (let ((new-strength  (cdr (assoc qual-imp2 *importance-alist*))))
		      ;; Then modify its strength ...
		      (rplacd imp new-strength)
		      ;; and re-sort the category's importances.
		      (setf (category-importances category)
			    (sort (category-importances category)  #'>=  :key #'cdr)))))))))))



;;;---------------------------------------------------------------------------
;;;  Function:	(discuss-relevances  exemplar)
;;;
;;;  Given:	exemplar, a newly-created exemplar
;;;
;;;  Purpose:	Given a newly-created exemplar, this function attempts to
;;;		determine the relevance of each	feature whose importance to
;;;		the category (or to the exemplar itself) is unknown.
;;;
;;;  Notes:	This function contains the top-level control for asking about
;;;		the relevance of features to a category.  Details are handled
;;;		in the subordinate functions search-explanation,
;;;		ask-for-explanation, and discuss-explanation.
;;;
;;;		This function gives the teacher the option of submitting an
;;;		explanation before Protos attempts to find one itself.  This can
;;;		save a lot of time in those situations where the teacher KNOWS
;;;		that Protos will not be able to find a feature-to-category
;;;		explanation (unsuccessful searches are time-consuming).
;;;
;;;  Caller:	add-new-exemplar
;;;---------------------------------------------------------------------------

(defparameter *choose-explanation* (make-menu
      :label  nil
      :items  '((#\E . ("Enter an explanation."                return enter))
		(#\L . ("let protos Look for an explanation."  return look))
		(#\S . ("declare the feature Spurious."        return spurious))
		(#\U . ("leave this feature Unexplained."      return unexplain)))))



(defun discuss-relevances (exemplar)

  (let ((category  (exemplar-category exemplar))
	(unknowns  (get-unknown-features exemplar))
	(ask-first nil))

    ;; If all features accounted for, then there is nothing to do.
    (if (null unknowns) (return-from discuss-relevances nil))

    (format t  "~%~%Protos is currently unaware of the relevance~
		  ~%of ~D feature~:P to the category ~A."
	    (length unknowns)  (getname category))

    ;; If several unknowns, see if teacher wants to be asked about each one.
    ;; If only 1 or 2 unknowns, then just go ahead and ask about each one.
    (if (< (length unknowns) 3)
	(setq ask-first t)
	(if (prompt "~%~%Should Protos ask you about each feature~
			 ~%before it tries to find an explanation itself? "
                    nil 'y-or-n nil nil)
	    (setq ask-first t)))

    ;; Main loop -- determine the relevance of each unknown feature.
    (dolist (feature unknowns)

      ;; If teacher is to be prompted about each feature ...
      (if ask-first
	  ;; then prompt the teacher ...
	  (progn
	    (format t "~2%What is the relevance of ~A to ~A ?~
                        ~%----------------------------------------------------"
		    (getname feature)  (getname category))

	    ;; The teacher is given 3 choices:
	    ;;	 1 -- Protos will search for an explanation.
	    ;;	 2 -- Protos will ask for an explanation from the teacher.
	    ;;	 3 -- Teacher declares that the feature is spurious.

	    (case (menu-select *choose-explanation*)
	      (look	  (search-explanation   feature  category t))
	      (enter	  (ask-for-explanation  feature  category))
	      (spurious	  (set-spurious	        feature  category))
	      (unexplain  nil)
	      (otherwise  (format t "Error: discuss-relevances: menu returned unexpected value."))))

	  ;; else search the category network for an explanation.
	  (search-explanation  feature  category t)))))


;;;-----------------------------------------------------------------------------
;;;  Function:	(search-explanation  feature  category  first-time)
;;;
;;;  Purpose:	This function tries to find an explanation relating the given
;;;		feature to the given category.
;;;
;;;  Design:	-- Do search for explanation.
;;;		-- If found, call discuss-explanation,
;;;		   else explain that no explanation was found and ask teacher
;;;		        if he/she wants to give an explanation.
;;;		-- If yes, call ask-for-explanation,
;;;		   else declare the feature to be spurious.
;;;-----------------------------------------------------------------------------

(defun search-explanation  (feature  category  first-time)

  (let ((result (kbpm 'FtoT feature *importance-big* category (exemplar-features *cur-exemplar*))))

    (case (result-type result)

      (explained  (discuss-explanation feature category (result-explanation result)))

      (excluded   nil)  ; don't learn a reminding or importance here.

      (unmatched  (format t "~%Protos ~:[still~;~] could not find an explanation.~
                             ~%What is the relevance of ~A to ~A ?
			     ~%----------------------------------------------------"
			  first-time  (getname feature)  (getname category))

		  (case (menu-select *choose-explanation*)
		    (look	  (search-explanation   feature  category nil))
		    (enter	  (ask-for-explanation  feature  category))
		    (spurious	  (set-spurious	        feature  category))
		    (unexplain    nil)
		    (otherwise    (format t "Error: search-explanation: menu returned unexpected value."))))

      (spurious	  (set-spurious feature category))

      (otherwise  (format t "~%Error: search-explanation: result-type = ~A~%"
				(result-type result))
		  (set-spurious feature category)))))


(defun set-spurious (feature category)
  (push (cons feature 0.0) (category-importances category)))


(defparameter *new-term-menu* (make-menu
  :label  "Is this a:"
  :items  '((#\T . ("New term."                           return new-term))
	    (#\A . ("Abbreviation for an existing term."  return abbrev))
	    (#\S . ("Synonym for an existing term."       return synonym))
	    (#\Q . ("Quit (return to previous menu)."     return quit)))))


(defun enter-new-terms ()
  (let (action termname termname2 node)
    (loop
      (setq termname (prompt "~%~%Enter term name (terminate with blank line) ---> "
			     nil 'termname nil nil))
      (if (null termname) (return (values)))
      (setq action (menu-select *new-term-menu*))
      (case action
	(new-term
	  (check-term-name termname 'ask))
	((abbrev synonym)
	    (setq termname2 (prompt "~%   Enter name of existing term ---> " nil
				    'termname nil nil))
	    (if (null termname2)  (return (values)))
	    (if (not (symbolp termname2))
		(progn
		  (format t "~%   Error: ~A is not a symbol." termname2)
		  (return (values))))
	    (if (not (boundp termname2))
		(progn
		  (format t "~%   Error: ~A is unknown." termname2)
		  (return (values))))
	    (setq node (eval termname2))
	    (if (not (node-p node))
		(progn
		  (format t "~%   Error: ~A is a ~A, not a node."
			  termname2 (type-of node))
		  (return (values))))
	    (if (eql action 'abbrev)
		(setf (node-abbrev node) termname)
		(push termname (node-synonyms node)))
	    (set termname node))
	(quit
	  (return-from enter-new-terms (values)))))))


;;;-----------------------------------------------------------------------------
;;;  Function:	(ask-for-explanation  feature  category)
;;;
;;;  Purpose:	This function prompts the teacher for an explanation relating
;;;		the given feature to the given category (or, the teacher can
;;;		give the missing fragment that will allow Protos to find an
;;;		explanation from feature to category.
;;;
;;;  Design:	-- Get explanation from teacher.
;;;		-- Verify that the given explanation provides a path through
;;;		   the category network from the feature to the category.
;;;		-- If so, call discuss-explanation,
;;;		   else harass the teacher for a correct explanation.
;;;-----------------------------------------------------------------------------

(defun ask-for-explanation (feature category)

  (let (explanation)

    (format t
	    "~%~%Please explain how ~A is related to ~A (preferred)~
	       ~%or how ~A is related to ~A (alternate form).~%"
	    (getname feature)   (getname category)
	    (getname category)  (getname feature))

    (setq explanation (enter-explanation))
    (if (null explanation) (return-from ask-for-explanation nil))

    (setq *feat* feature)
    (setq *cat* category)
    (setq *exp* explanation)
    ;; Try to verify the explanation by the simple syntactic strategy
    ;; of noticing if the explanation begins with 'feature' and ends
    ;; with 'category' (or vice versa).  If so, then we don't have to
    ;; do the more time-consuming search through the category network.

    (let ((to-nodes    (get-leaves explanation t nil))
	  (start-term  (explanation-start-term explanation)))

      (cond  ;; Is this a feature-to-category explanation?
	     ((and (eq feature start-term)  (member category to-nodes))
		(discuss-explanation  feature  category  explanation))

	     ;; Is this a category-to-feature explanation?
	     ((and (eq category start-term)  (member feature to-nodes))
		(discuss-explanation  feature  category  explanation))

	     ;; Apparently this is a fragment, so must search the C.N.
	     (t  (search-explanation feature category t))))))

;;;-----------------------------------------------------------------------------
;;;  Function:	(discuss-explanation  feature  target  explanation)
;;;
;;;  Given:	an explanation which may be either feature-to-target or
;;;		target-to-feature (where "target" is a category or exemplar).
;;;
;;;  Do:	Set both a reminding and an importance from the explanation.
;;;		The reminding comes from the feature-to-target explanation
;;;		and the importance from the target-to-feature explanation.
;;;		Simply invert the given explanation to get the other type.
;;;
;;;  Note:      If the target is related to a more general category, then
;;;             set-reminding will ask the user if the reminding should really
;;;             go to the more general category.  If the user agrees, then
;;;             set-reminding returns the new (reminded) category so that
;;;             set-importance will install the importance in that new category.
;;;
;;;  Callers:   ask-for-explanation, search-explanation
;;;-----------------------------------------------------------------------------

(defun discuss-explanation  (feature  target  explanation)

  (let ((from-leaves  (get-leaves explanation nil nil))
	inverse-explanation)

    ;; If this is a feature-to-[category or exemplar] explanation ...
    (if (member feature from-leaves)
	;; then set reminding, invert explanation, and set importance
	(progn
	  (multiple-value-setq (target explanation)
	    (set-reminding feature target explanation))
	  (if target
	      (progn
		(setq inverse-explanation (invert-explanation explanation))
		(set-importance feature target inverse-explanation))))

	;; else this must be a [category or exemplar]-to-feature explanation,
	;; so invert the explanation, set reminding, then set importance.
	;; The reason why the reminding is always set before the importance
	;; is that 'set-reminding' may truncate the explanation, changing
	;; the target.
	(progn
	  (setq inverse-explanation (invert-explanation explanation))
	  (multiple-value-setq (target explanation)
	    (set-reminding feature target inverse-explanation))
	  (if target
	      (set-importance feature target explanation))))))


    

;;;-----------------------------------------------------------------------------
;;;  Function:	(set-reminding  feature  target  explanation)
;;;
;;;  Given:	-- feature, the feature causing this reminding;
;;;		-- target, the category or exemplar that is the object of
;;;		   the reminding to be installed;
;;;		-- explanation, a feature-to-target explanation.
;;;
;;;  Purpose:	Given a feature-to-category or feature-to-exemplar explanation,
;;;		this function will install a reminding, subject to several
;;;		heuristics.
;;;
;;;  Returns:   Two values:
;;;             -- The target category or exemplar that the reminding really
;;;                points to.  It can be different than the supplied target if
;;;                the target is a category and the teacher agrees to let the
;;;                reminding be to a more general category.
;;;             -- The explanation from feature to actual target.  This may be
;;;                a truncated version of the supplied explanation.
;;;
;;;  Heuristics:  -- If the explanation contains a mutual exclusion relation,
;;;		     then no reminding is installed.
;;;		  -- If the target of the explanation is an exemplar, then
;;;		     install the reminding without applying further heuristics.
;;;		  -- If the explanation contains a "has-specialization",
;;;		     "has-part", or a weak link, then truncate the explanation
;;;		     at that point (thus changing the target of the reminding).
;;;-----------------------------------------------------------------------------
(defun set-reminding  (feature  target  explanation)

  (let (quality)
    ;; If mutual exclusion found, don't install reminding.
    (if (check-mutex explanation)
	(return-from set-reminding (values nil nil)))
    
    ;; If target of explanation is an exemplar,
    ;; install the reminding and return.
    (if (my-exemplar-p target)
	(progn
	  (setq quality (explanation-strength explanation))
	  (set-reminding2 feature target quality)
	  (return-from set-reminding (values target explanation))))
    
    ;; Apply heuristics for truncating the explanation.
    (setq explanation (truncate-explanation explanation))
    
    ;; If explanation truncated down to nothing, just return.
    (if (null explanation)
	(return-from set-reminding (values nil nil)))
    
    ;; Install reminding to target(s) of the possibly-truncated explanation.
    (setq quality (explanation-strength explanation))

    ;; For each target of this explanation ...
    (dolist (e-target (get-leaves explanation t nil))
      (set-reminding2 feature e-target quality))

    (values target explanation)))


;;;-----------------------------------------------------------------------------
;;;  Function:  (set-reminding2  feature  target  quality)
;;;
;;;  Purpose:   This function installs a reminding from 'feature' to 'target'
;;;             with the given 'strength' if there is not already a reminding
;;;             between 'feature' and 'target'.  If there is, then it is updated
;;;             only if this new reminding is stronger.
;;;-----------------------------------------------------------------------------

(defun set-reminding2 (feature target quality)
  (let ((rem (assoc target (feature-remindings feature))))
    
    ;; if there is an existing reminding to this target ...
    (if rem
	
	;; then update the reminding strength only if this one is stronger
	(if (> quality (cdr rem))
	    (progn
	      (rplacd rem quality)
	      (if *trace-new-remindings*
		  (print-reminding feature target quality t))))
	
	;; else create a reminding to it or a more general target.
	(progn
	  (if (not (my-exemplar-p target))
	      (setq target (select-reminding-target feature target quality)))
	  (push (cons target quality) (feature-remindings feature))
	  (if *trace-new-remindings*
	      (print-reminding feature target quality nil))
	  (trim-remindings feature)))))

;;;-----------------------------------------------------------------------------
;;;  Function:  (select-reminding-target  feature  target  quality)
;;;
;;;  Given:     -- a feature that evokes a reminding,
;;;             -- a potential target for that reminding, and
;;;             -- the quality (strength) of the reminding.
;;;
;;;  Returns:   the actual target for the reminding, which may be either the
;;;             given target or a more general category in the generalization,
;;;             functional, or partonomic hierarchies.  If the given target has
;;;             one or more more-general categories, then the teacher is asked
;;;             to select the desired target.
;;;-----------------------------------------------------------------------------

(defun select-reminding-target (feature target quality)
  (let ((relatives (collect-relatives target
				      (list *verb-hasTypicalGen*
					    *verb-hasFunction*
					    *verb-partOf*)
				      nil)))

    ;; If this target has no categories that are more general, then just
    ;; return the original target.
    (if (null relatives)
	(return-from select-reminding-target target))
    
    (format *query-io* "~%Protos is ready to install a ~4,2F reminding from ~A to ~A."
	    quality (getname feature) (getname target))

    ;; IF only one more general category ...
    (if (null (cdr relatives))

	;; THEN ask about it specifically
	(if (y-or-n-p "~%~%Should it instead install it to the more general category ~A ? "
		      (getname (car relatives)))
	    (setq target (car relatives)))

	;; ELSE ask for a selection from the list of more general categories.
	(progn
	  (format *query-io* "~%~%Should it instead install it to one of the more general categories:~%")
	  (print-node-names relatives *query-io* " or ")
	  (format *query-io* " ")
	  (if (y-or-n-p)
	      (loop
		      (format *query-io* "~%Then please specify which one: ")
		      (setq target (check-term-name (read *query-io* nil nil) 'fail))
		      (if (member target relatives) (return (values)))
		      (format *query-io* "~%Error: you must enter one of the above terms.")))))
    target))


;;;-----------------------------------------------------------------------------
;;;  Function:  (check-mutex  explanation)
;;;
;;;  Purpose:   Given an explanation, this function returns T if a mutual-
;;;             exclusion relation is found anywhere within the explanation.
;;;             Otherwise, it returns NIL.
;;;-----------------------------------------------------------------------------

(defun check-mutex (explanation)
  (if (eq *verb-MEx* (relation-verb (explanation-relation explanation)))
      (return-from check-mutex t))
  (dolist (term (explanation-from-terms explanation))
    (if (explanation-p term)
	(if (check-mutex term)
	    (return-from check-mutex t))))
  (dolist (term (explanation-to-terms explanation))
    (if (explanation-p term)
	(if (check-mutex term)
	    (return-from check-mutex t))))
  nil)


;;;-----------------------------------------------------------------------------
;;;  Function:	(set-importance  feature  target  explanation)
;;;
;;;  Given:	-- feature, the feature whose importance value is to be set;
;;;		-- target, the category or exemplar where the importance is 
;;;		   to be stored;
;;;		-- explanation, a category-to-feature or exemplar-to-feature
;;;		   explanation.
;;;
;;;  Purpose:	Given a category-to-feature or exemplar-to-feature explanation,
;;;		this function sets the importance of the feature in the category
;;;		or exemplar, respectively.
;;;
;;;  Notes:	-- The importance value (a number in the range 0 - 1.0) is
;;;		   taken from the strength of the explanation.
;;;
;;;  Callers:	discuss-explanation
;;;-----------------------------------------------------------------------------

(defun set-importance  (feature  target  explanation)
  
  (if (null explanation) (return-from set-importance (values)))
  
  (let* ((quality    (explanation-strength explanation))
	 (qual-imp1  (qualitative-value quality 'importance))
	 (category   (if (my-exemplar-p target)
			 (exemplar-category target)
			 target))
	 qual-imp2)
    
    ;; Show explanation to teacher for possible modification.
    (format t "~%~%Protos believes that ~A is of ~A importance to ~A~
		 ~%based on the explanation:~
                 ~%   "
	    (getname feature)  qual-imp1 (getname category))
    (print-explanation explanation)
    
    ;; See if teacher agrees with this value of importance.
    (case (menu-select *decide-importance-menu*)
      (yes          ;; Teacher agrees with importance, so nothing to do.
	nil)

      (revise-expl  ;; Teacher wants to revise the explanation.
        (prog ()
          TOP
	   (setq explanation 
                (get-specific-explanation feature target 'TtoF (exemplar-features *cur-exemplar*)))
           (if (null explanation)
             (progn
              (format t "~%~%Protos could not find an explanation from ~A to ~A"
                 (getname feature) (getname category))
              (if (y-or-n-p "~%Do you wish to revise your explanation again? ")
                  (go TOP))) 
             (print-explanation explanation))
	   (set-importance feature target explanation)
	   (return-from set-importance (values))))

      (revise-imp   ;; Teacher wants to just revise the importance.
	(setq qual-imp2 (menu-select *importance-menu*))
	(if (and (not (equal 'nochange qual-imp2))
		 (not (equal qual-imp1 qual-imp2)))
	    (setq quality  (cdr (assoc qual-imp2 *importance-alist*))))))
    
    ;; Install the feature's importance in the category.
    (push (cons feature quality) (category-importances category))))


;;;-----------------------------------------------------------------------------
;;;  Function:  (discuss-unmatched-importances  match)
;;;
;;;  Given:     a match of an exemplar of a wrong category to a newcase;
;;;
;;;  Purpose:   This function examines the exemplar's unmatched features to
;;;             determine which ones are believed to be of moderate or low
;;;             importance, since it is possible that the reason the feature is
;;;             unmatched is because the importance was set too low.
;;;             Protos then asks the teacher to reassess the importance
;;;             of each of these features.  The teacher is told the current
;;;             qualitative strength of the feature's importance and then asked
;;;             to revise the importance to be either "necessary", "high",
;;;             "moderate", "low", or "spurious".
;;;-----------------------------------------------------------------------------


(defun discuss-unmatched-importances (match)
  (let ((results    (match-results match))
	(first-time t))
    (dolist (result results)
      (if (and (eql 'unmatched (result-type result))
	       (< (result-importance result) *importance-high*))
          (progn
            (if first-time
	      (progn
		(setq first-time nil)
		(format t
   "~%~%The incorrectly matched exemplar contains unmatched features~
      ~%of moderate or low importance.  A possible reason that this case~
      ~%was classified incorrectly is because one or more of those features~
      ~%is really more important than Protos currently believes.")))
	    (reassess-importance (result-feature result) (match-exemplar match)))))))

;;;-----------------------------------------------------------------------------
;;;  Function:	(get-approximate-importances  exemplar)
;;;
;;;  Given:	exemplar, a newly-created exemplar
;;;
;;;  Returns:	an alist of (feature . importance) with one entry for each
;;;		feature of the exemplar, sorted in decreasing order of
;;;		importance.
;;;
;;;  Caller:	discuss-relevances
;;;
;;;  Purpose:	The decision to create this function was based on the belief
;;;		that when Protos discusses the relevance of each feature to its
;;;		category, it is best to discuss the most important features
;;;		first.  Without this function, Protos would present the features
;;;		in the order in which they occurred in the new case (which may
;;;		bear no resemblance to importances).
;;;
;;;  Design:	For each feature, this function looks for its importance value
;;;		in the following places (in this order):
;;;		-- in the exemplar's category;
;;;		-- in neighboring categories (immediate neighbors only);
;;;		-- otherwise, the value 0 is assigned.
;;;---------------------------------------------------------------------------

(defun get-approximate-importances (exemplar)

  (let* ((features1	 (copy-list (exemplar-features exemplar)))
	 (features2	 (copy-list features1))
	 (category	 (exemplar-category exemplar))
	 (c-importances	 (category-importances category))
	 (importances	 nil))

    ;; ------------- Extract importances from the category. ----------------
    (dolist (feature features1)
      (let ((pair (assoc feature c-importances)))
	(if pair
	    (progn
	      (push pair importances)
	      (setq features2 (delete feature features2))))))

    (if (null features2)
	(return-from  get-approximate-importances
		(sort importances  #'>  :key #'cdr)))

    (setq features1 (copy-list features2))


    ;; --------- Extract importances from neighboring categories. ----------
    (dolist (relation (node-relations category))
      (dolist (to-node (relation-to-nodes relation))
	(if (my-category-p to-node)
	    ;; -------- Extract importances from this neighbor. -----------
	    (let ((c-importances (category-importances to-node)))
	      (dolist (feature features1)
		(let ((pair (assoc feature c-importances)))
		  (if pair
		      (progn
			(push pair importances)
			(setq features2 (delete feature features2))))))

	      (if (null features2)
		  (return-from  get-approximate-importances
			(sort importances  #'>  :key #'cdr)))

	      (setq features1 (copy-list features2))))))


    ;; ------- Assign zero importance to any remaining features. ----------
    (dolist (feature features2)
      (push (cons feature 0.0) importances))

    (return-from  get-approximate-importances
		(sort importances  #'>  :key #'cdr))))

;;;---------------------------------------------------------------------------
;;;  Function:	(discuss-unmatched  match)
;;;
;;;  Given:	match, an approved exemplar-to-newcase match;
;;;
;;;  Does:	For each non-spurious unmatched feature of the exemplar,
;;;		the teacher is asked to explain how a feature of the new case
;;;		is related to the unmatched exemplar feature.  The teacher 
;;;		may either enter an explanation or enter "none", "n" or "no".
;;;
;;;  Caller:	discuss-success
;;;---------------------------------------------------------------------------

(defun discuss-unmatched (match)

  (let ((introduction t)	; Print introduction only on first feature.
	(results  (match-results match)))

    ;; Iterate through the match results looking for unmatched features.
    (do* ((results results (cdr results))
          (result  (car results) (car results)))
         ((endp results))
      (let* ((feature	 (result-feature result))	; pointer to feature node
	     (importance (result-importance result))	; importance of this feature
	     (quality    (result-quality result))
             (unmatched  (match-unmatched match)))      ; unmatched features
	
	(if (eql 'unmatched (result-type result))
	    (progn
	      (if introduction
		  (progn
		    (setq introduction nil)
		    (format t
      "~%~%Protos would like to improve this classification by discussing~
	 ~%some of the exemplar features it could not account for.~%")))
	      
	      (if (prompt (format nil
				  "~%Protos could not account for the exemplar feature ~A.~
	                           ~%Is ~A related to any feature of the new case? "
				  (getname feature)  (getname feature))
			  nil 'y-or-n nil nil)
		  (let (newresult)
		    (format t "~%Please explain how ~A is related to some feature of the new case."
			      (getname feature))
		    (enter-explanation)
		    (setq newresult (compare-feature feature
						     importance
						     (match-exemplar match)
						     (match-newcase match)))
		    (format t "~%~%The exemplar feature ~A now has the following match:"
			    (getname feature))
		    (print-result 0 newresult)
		 
		    ;; Replace the unmatched result with the explained result.
		    (rplaca results newresult)

	            ;; Depending on the type of result, adjust the list
	            ;; of unmatched case features.
	            (ecase (result-type newresult)
	              (identical
		        (setq unmatched (delete feature unmatched)))
	              (explained
		        (setq unmatched (nset-difference
			  unmatched (explanation-from-terms (result-explanation newresult)))))
	              (excluded
		        (setq unmatched (nset-difference
			  unmatched (explanation-from-terms (result-explanation newresult)))))
                      (unmatched nil)
                      (spurious  nil))

                    (setf (match-unmatched match) unmatched)

		    ;; Update the overall similarity by dividing by the old
		    ;; similarity and multiplying by the new similarity.
		    (setf (match-similarity match)
			  (* (/ (match-similarity match) quality) (result-quality newresult)))
		    ))))))))


(defun discuss-unfocused ()
    ;; Ask for unfocused instruction.
    (format t "~%~%If there is any other instruction that you wish to provide,~
                 ~%you may now do so from the following menu:~%")
    (menu-select-2 '*unfocused-instruction-menu*))




;;;----------------------------------------------------------------------------
;;;  Function:  (discuss-failure  match)
;;;
;;;  Given:	match, the results of an exemplar-to-new case match where the
;;;		       the teacher has rejected the match because it is to a
;;;                    wrong category.
;;;
;;;  Returns:	-- the same match with the action slot set to 'next if the
;;;                teacher has said to try the next hypothesis;
;;;		-- the same match with the action slot set to 'done if the
;;;                teacher decided either to create a new exemplar from this
;;;                case or just abandon this case.
;;;
;;;  Caller:	discuss-match
;;;
;;;  Design:	-- Reassess remindings that led to this (incorrect) category.
;;;		-- Ask for any revisions to the new case.
;;;		-- Ask for censors.
;;;		-- Ask for reassessment of importances of low-importance
;;;		   unmatched features.
;;;		-- If strong match then record for later difference links.
;;;		-- Allow unfocused instruction.
;;;		-- Ask teacher whether to install this case as a new exemplar
;;;		   or discard it and go on to the next hypothesis.
;;;----------------------------------------------------------------------------

(defparameter *failure-menu1* (make-menu
  :label  "~%~%What do you want to do with this case now?~
             ~%------------------------------------------"
  :items  '((#\C . ("Create an exemplar from this case"   return  create))
	    (#\R . ("Retry finding a match for this case" return  retry))
	    (#\M . ("reMatch this case to same exemplar"  return  redo))
	    (#\T . ("Try next hypothesis"                 return  next))
	    (#\Q . ("Quit (abandon this case)"            return  abandon)))))


(defun discuss-failure (match)
  (declare (special closed-categories))

  ;; The teacher has rejected this match because it is to the wrong category.
  ;; So, remember that fact so that the function "classify2" will skip over
  ;; any other hypothesized exemplars of the same category.
  (push (exemplar-category (match-exemplar match)) closed-categories)

  (format t "~%~%Protos will now try to learn from this error.")

  ;; If this incorrect match is strong enough (i.e., is a "near miss"),
  ;; then remember it for possibly installing a difference link later
  ;; when the case finally is correctly classified.
  (if (>= (match-nth-root-of-similarity match) *near-miss-threshold*)
      (push match *near-misses*))

  (reassess-remindings match)
  (discuss-censors match)
  (discuss-unmatched-importances match)
  (discuss-unfocused)
  (case (menu-select *failure-menu1*)
    (create    (values 'newex (add-new-exemplar (match-newcase match))))
    (retry     (values 'retry match))
    (redo      (values 'redo  match))
    (next      (values 'next  match))
    (abandon   (values 'done  nil))))



(defparameter *censor-strength-menu* (make-menu
      :label  nil
      :items  `((#\W . ("Weak"       return  ,-0.25))
		(#\M . ("Moderate"   return  ,-0.50))
		(#\S . ("Strong"     return  ,-0.75))
		(#\A . ("Absolute"   return  ,*absolute-censor*)))))


(defun discuss-censors (match)
  (let* ((exemplar  (match-exemplar match))
	 (category  (exemplar-category exemplar))
	 (newcase   (match-newcase match))
	 (features  (case-features newcase))
	 input
	 feature
	 strength)

    (format *query-io* "~2%The features of the new case are:~%")
    (print-node-names features *query-io* ", ")
    (if (prompt (format nil "~2%Are any of these features mutually exclusive~
                              ~%with the category ~A? "
			(getname category))
		nil 'y-or-n nil nil)
	(progn
	  (loop
            (format *query-io*
		    "~%Please enter their name(s), enclosed in parentheses (or type nil).~
                     ~%---> ")
	    (setq input (read *query-io* nil nil))
	    (if (or (null input) (listp input))
		(return (values)))
	    (format *query-io* "~%Input error!"))
	  
	  (dolist (fname input)
	    (setq feature (check-term-name fname 'ask))
	    (if feature
		(progn
		  (format *query-io* "~%Please rate the strength of ~A's negative evidence:"
			  (getname feature))
		  (setq strength (menu-select *censor-strength-menu*))
		  (push (cons category strength) (feature-remindings feature))
		  (if *trace-remindings*
		      (print-reminding feature category strength nil)))))))))



(defun add-censor ()
  (let (name feature target strength)
    (format *query-io*
	    "~%A censor lets a feature \"discount\" a category or exemplar.~
             ~%Please enter a feature name ----------------> ")
    (setq name (read *query-io* nil nil))
    (if (null name)
	(return-from add-censor nil))
    (setq feature (check-term-name name 'ask))
    (if (null feature)
	(return-from add-censor nil))
    (format *query-io* "~%Please enter a category or exemplar name ---> ")
    (setq name (read *query-io* nil nil))
    (if (null name)
	(return-from add-censor nil))
    (setq target (check-term-name name 'ask))
    (if (null target)
	(return-from add-censor nil))
    (format *query-io* "~%Please rate the strength of ~A's negative evidence:"
	    (getname feature))
    (setq strength (menu-select *censor-strength-menu*))
    (push (cons target strength) (feature-remindings feature))
    (if *trace-remindings*
	(print-reminding feature target strength nil))))
    

(defun change-importance ()
  (let (name feature target)
    (format *query-io* "~%Enter name of feature --------> ")
    (setq name (read *query-io* nil nil))
    (if (null name)
	(return-from change-importance (values)))
    (setq feature (check-term-name name 'ask))
    (if (null feature)
	(return-from change-importance (values)))
    (format *query-io* "~%Enter category or exemplar ---> ")
    (setq name (read *query-io* nil nil))
    (if (null name)
	(return-from change-importance (values)))
    (setq target (check-term-name name 'ask))
    (if (null target)
	(return-from change-importance (values)))
    (reassess-importance feature target)))
    

(defun show-relations ()
  (format *query-io* "~%Please enter the name of any term ---> ")
  (let* ((name  (read *query-io* nil nil))
	 (relations nil)
	 node)
    (if (and (symbolp name) (boundp name))
	(setq node (eval name))
	(setq node (check-term-name name 'fail)))
    (if (and node (node-p node))
	(progn
	  (setq relations (node-relations node))
	  (if relations
	      (progn
		(format t "~%Relations that begin with ~A:" name)
		(dolist (rel relations)
		  (format t "~%    ")
		  (print-relation rel t 1)))
	      (format t "~%No relations involve this ~:[term~;predicate~]."
		      (predicate-p node))))
	(format t "~%~A is not a term or predicate!" name))))

