
;;;  This is the MIT Loop package, as described below.  We (Boyer and
;;;  Moore) have replaced all occurrences of "loop" with "iterate" so as
;;;  to avoid collisions with other versions of loop on systems where we
;;;  are running our theorem-prover.  To make this system work where
;;;  fixnums may be few we have suppressed all fixnum declarations.  All of
;;;  our changes, except the loop replacement, may be found on lines with
;;;  the string "Boyer and Moore".

(EVAL-WHEN (LOAD EVAL COMPILE)
    (DEFUN CHK-BASE-AND-PACKAGE (TEN PACKAGE)
      (COND ((NOT (EQUAL TEN (+ 1 1 1 1 1 1 1 1 1 1)))
	     (ERROR "This file is not being read in the decimal base.")))
      (COND ((NOT (EQUAL PACKAGE (FIND-PACKAGE "USER")))
	     (ERROR "This file is not being read in the USER package.")))))

(EVAL-WHEN (LOAD EVAL COMPILE)
	   (CHK-BASE-AND-PACKAGE 10 *PACKAGE*))	    


;;;   *************************************************************************
;;;   ******* Common Lisp ******** ITERATE Iteration Macro ***********************
;;;   *************************************************************************
;;;   ***** (C) COPYRIGHT 1980, 1981 MASSACHUSETTS INSTITUTE OF TECHNOLOGY ****
;;;   ********* THIS IS A READ-ONLY FILE! (ALL WRITES RESERVED) ***************
;;;   *************************************************************************

;;;; ITERATE Iteration Macro

;;; This is the "officially sanctioned" version of ITERATE for running in
;;; Common Lisp.  It is a conversion of ITERATE 829, which is fairly close to
;;; that released with Symbolics Release 6.1 (803).  This conversion was
;;; made by Glenn Burke (one of the original author/maintainers);  the
;;; work was performed at Palladian Software, in Cambridge MA, April 1986.
;;; 
;;; The current version of this file will be maintained at MIT, available
;;; for anonymous FTP on MC.LCS.MIT.EDU from the file "LSB1;CLITERATE >".  This
;;; location will no doubt change sometime in the future.
;;; 
;;; This file was actually taken from ai.ai.mit.edu:gsb;cliterate >.  Boyer and Moore.
;;;
;;; This file, like the ITERATE it is derived from, has unrestricted
;;; distribution -- anyone may take it and use it.  But for the sake of
;;; consistency, bug reporting, compatibility, and users' sanity, PLEASE
;;; PLEASE PLEASE don't go overboard with fixes or changes.  Remember that
;;; this version is supposed to be compatible with the Maclisp/Zetalisp/NIL
;;; ITERATE;  it is NOT intended to be "different" or "better" or "redesigned".
;;; Report bugs and propose fixes to BUG-ITERATE@MC.LCS.MIT.EDU;
;;; announcements about ITERATE will be made to the mailing list
;;; INFO-ITERATE@MC.LCS.MIT.EDU.  Mail concerning those lists (such as requests
;;; to be added) should be sent to the BUG-ITERATE-REQUEST and
;;; INFO-ITERATE-REQUEST lists respectively.  Note the Change History page
;;; below...
;;; 
;;; ITERATE documentation is still probably available from the MIT Laboratory
;;; for Computer Science publications office:
;;; 	LCS Publications
;;; 	545 Technology Square
;;; 	Cambridge, MA 02139
;;; It is Technical Memo 169, "ITERATE Iteration Macro", and is very old.  The
;;; most up-to-date documentation on this version of ITERATE is that in the NIL
;;; Reference Manual (TR-311 from LCS Publications);  while you wouldn't
;;; want to get that (it costs nearly $15) just for ITERATE documentation,
;;; those with access to a NIL manual might photocopy the chapter on ITERATE.
;;; That revised documentation can be reissued as a revised technical memo
;;; if there is sufficient demand.
;;;


;;;; Change History

;;; [gsb@palladian] 30-apr-86 00:26  File Created from NIL's ITERATE version 829
;;; [gsb@palladian] 30-oct-86 18:23  don't generate (type notype var) decls, special-case notype into T.
;;;		    (The NOTYPE type keyword needs to be around for compatibility.)
;;; [gsb@palladian] 30-oct-86 18:48  bogus case clause in iterate-do-collect.  Syntax:common-lisp in file
;;;		    attribute list, for symbolics gratuitousness.
;;;------------------------------------------------------------------------
;;;------- End of official change history -- note local fixes below -------
;;;------------------------------------------------------------------------

;;; [boyer@rascal.ics.utexas.edu] 9-july-87 In define-iterate-macro moved.  Boyer and Moore
;;; &environment and its formal to right after &whole to overcome bug.  Boyer and Moore
;;; in KCL. Boyer and Moore



;;;; Package setup


;(provide "ITERATE")  ;  Made it a string instead of a symbol.  Boyer and Moore.


;;;The following symbols are documented as being available via SI:.  Far be
;;;it for us to define a package by that name, however we can do the
;;;following.  We will create a "iterate-si-kludge" package (sounds like a
;;;fairly safe name), import the SI: symbols from there into ITERATE, export
;;;them, define that people (use-package 'iterate), and if they want to
;;;maintain source compatibility they can add the SI nickname the
;;;iterate-si-kludge package.  How's that?

;(in-package 'iterate-si-kludge)

;(export '(iterate-tequal iterate-tassoc iterate-tmember iterate-use-system-destructuring?
;	  iterate-named-variable iterate-simplep iterate-simplep-1
;	  iterate-sequencer iterate-sequence-elements-path))

;  Commented out by Boyer and Moore
;(in-package "ITERATE")  ; make it a string -- Boyer and Moore

;(use-package '(iterate-si-kludge))

;shadow?

;  Commented out by Boyer and Moore
#| (export '(iterate iterate-finish define-iterate-macro define-iterate-path
	       define-iterate-sequence-path)) |#

;  Commented out by Boyer and Moore
#|(export '(iterate-tequal iterate-tassoc iterate-tmember iterate-use-system-destructuring?
	  iterate-named-variable iterate-simplep iterate-simplep-1
	  iterate-sequencer iterate-sequence-elements-path))|#

;require?


;;;; Macro Environment Setup


; Hack up the stuff for data-types.  DATA-TYPE? will always be a macro
; so that it will not require the data-type package at run time if
; all uses of the other routines are conditionalized upon that value.
(eval-when (eval compile)
  ; Crock for DATA-TYPE? derives from DTDCL.  We just copy it rather
  ; than load it in, which requires knowing where it comes from (sigh).
  ; 
  (defmacro data-type? (frob)
    (let ((foo (gensym)))
      `((lambda (,foo)
	  ;; NIL croaks if nil given to GET...  No it doesn't any more!  But:
	  ;; Every Lisp should (but doesn't) croak if randomness given to GET
	  ;; LISPM croaks (of course) if randomness given to get-pname
	  (and (symbolp ,foo)
	       (or (get ,foo ':data-type)
		   (and (setq ,foo (find-symbol (symbol-name ,foo) 'keyword))
			(get ,foo ':data-type)))))
	,frob))))


;;; The uses of this macro are retained in the CL version of iterate, in case they are
;;; needed in a particular implementation.  Originally dating from the use of the
;;; Zetalisp COPYLIST* function, this is used in situations where, were cdr-coding
;;; in use, having cdr-NIL at the end of the list might be suboptimal because the
;;; end of the list will probably be RPLACDed and so cdr-normal should be used instead.
(defmacro iterate-copylist* (l)
  `(copy-list ,l))


;;;; Random Macros

(defmacro iterate-simple-error (unquoted-message &optional (datum nil datump))
  `(error ,(if datump "ITERATE:  ~S ~A" "ITERATE:  ~A")
	  ',unquoted-message ,@(and datump (list datum))))


(defmacro iterate-warn (unquoted-message &optional (datum nil datump))
  (if datump
      `(warn ,(concatenate 'string "ITERATE: " unquoted-message " -- ~{~S~^ ~}")
	     ,datum)
      `(warn ',(concatenate 'string "ITERATE: " unquoted-message))))


(defmacro iterate-pop-source () '(pop iterate-source-code))

(defmacro iterate-gentemp (&optional (pref ''iteratevar-))
  `(gentemp (symbol-name ,pref)))


;;;; Setq Hackery

; Note:  ITERATE-MAKE-PSETQ is NOT flushable depending on the existence
; of PSETQ, unless PSETQ handles destructuring.  Even then it is
; preferable for the code ITERATE produces to not contain intermediate
; macros, especially in the PDP10 version.

(defun iterate-make-psetq (frobs)
    (and frobs
	 (iterate-make-setq
	    (list (car frobs)
		  (if (null (cddr frobs)) (cadr frobs)
		      `(prog1 ,(cadr frobs)
			      ,(iterate-make-psetq (cddr frobs))))))))


(defvar iterate-use-system-destructuring?
    nil)

(defvar iterate-desetq-temporary)

; Do we want this???  It is, admittedly, useful...
;(defmacro iterate-desetq (&rest x)
;  (let ((iterate-desetq-temporary nil))
;     (let ((setq-form (iterate-make-desetq x)))
;	(if iterate-desetq-temporary
;	    `((lambda (,iterate-desetq-temporary) ,setq-form) nil)
;	    setq-form))))


(defun iterate-make-desetq (x)
   (if iterate-use-system-destructuring?
       (cons (do ((l x (cddr l))) ((null l) 'setq)
	       (or (and (not (null (car l))) (symbolp (car l)))
		   (return 'desetq)))
	     x)
       (do ((x x (cddr x)) (r nil) (var) (val))
	   ((null x) (and r (cons 'setq r)))
	 (setq var (car x) val (cadr x))
	 (cond ((and (not (atom var))
		     (not (atom val))
		     (not (and (member (car val) '(car cdr cadr cddr caar cdar))
			       (atom (cadr val)))))
		  (setq x (list* (or iterate-desetq-temporary
				     (setq iterate-desetq-temporary
					   (iterate-gentemp 'iterate-desetq-)))
				 val var iterate-desetq-temporary (cddr x)))))
	 (setq r (nconc r (iterate-desetq-internal (car x) (cadr x)))))))


(defun iterate-desetq-internal (var val)
  (cond ((null var) nil)
	((atom var) (list var val))
	(t (nconc (iterate-desetq-internal (car var) `(car ,val))
		  (iterate-desetq-internal (cdr var) `(cdr ,val))))))


(defun iterate-make-setq (pairs)
    (and pairs (iterate-make-desetq pairs)))


(defparameter iterate-keyword-alist			;clause introducers
     '(	(named iterate-do-named)
	(initially iterate-do-initially)
	(finally iterate-do-finally)
	(nodeclare iterate-nodeclare)
	(do iterate-do-do)
	(doing iterate-do-do)
	(return iterate-do-return)
	(collect iterate-do-collect list)
	(collecting iterate-do-collect list)
	(append iterate-do-collect append)
	(appending iterate-do-collect append)
	(nconc iterate-do-collect nconc)
	(nconcing iterate-do-collect nconc)
	(count iterate-do-collect count)
	(counting iterate-do-collect count)
	(sum iterate-do-collect sum)
	(summing iterate-do-collect sum)
	(maximize iterate-do-collect max)
	(minimize iterate-do-collect min)
	(always iterate-do-always nil) ;Normal, do always
	(never iterate-do-always t)    ; Negate the test on always.
	(thereis iterate-do-thereis)
	(while iterate-do-while nil while)	    ; Normal, do while
	(until iterate-do-while t until)	    ; Negate the test on while
	(when iterate-do-when nil when)	    ; Normal, do when
	(if iterate-do-when nil if)    ; synonymous
 	(unless iterate-do-when t unless)	    ; Negate the test on when
	(with iterate-do-with)))


(defparameter iterate-iteration-keyword-alist
    `((for iterate-do-for)
      (as iterate-do-for)
      (repeat iterate-do-repeat)))


(defparameter iterate-for-keyword-alist			;Types of FOR
     '( (= iterate-for-equals)
        (first iterate-for-first)
	(in iterate-list-stepper car)
	(on iterate-list-stepper nil)
	(from iterate-for-arithmetic from)
	(downfrom iterate-for-arithmetic downfrom)
	(upfrom iterate-for-arithmetic upfrom)
	(below iterate-for-arithmetic below)
	(to iterate-for-arithmetic to)
	(being iterate-for-being)))

(defvar iterate-prog-names)


(defvar iterate-macro-environment)	;Second arg to macro functions,
					;passed to macroexpand.

(defvar iterate-path-keyword-alist nil)		; PATH functions
(defvar iterate-named-variables)			; see ITERATE-NAMED-VARIABLE
(defvar iterate-variables)			;Variables local to the iterate
(defvar iterate-declarations)			; Local dcls for above
(defvar iterate-nodeclare)			; but don't declare these
(defvar iterate-variable-stack)
(defvar iterate-declaration-stack)
(defvar iterate-desetq-crocks)			; see iterate-make-variable
(defvar iterate-desetq-stack)			; and iterate-translate-1
(defvar iterate-prologue)				;List of forms in reverse order
(defvar iterate-wrappers)				;List of wrapping forms, innermost first
(defvar iterate-before-iterate)
(defvar iterate-body)				;..
(defvar iterate-after-body)			;.. for FOR steppers
(defvar iterate-epilogue)				;..
(defvar iterate-after-epilogue)			;So COLLECT's RETURN comes after FINALLY
(defvar iterate-conditionals)			;If non-NIL, condition for next form in body
  ;The above is actually a list of entries of the form
  ;(cond (condition forms...))
  ;When it is output, each successive condition will get
  ;nested inside the previous one, but it is not built up
  ;that way because you wouldn't be able to tell a WHEN-generated
  ;COND from a user-generated COND.
  ;When ELSE is used, each cond can get a second clause

(defvar iterate-when-it-variable)			;See ITERATE-DO-WHEN
(defvar iterate-never-stepped-variable)		; see ITERATE-FOR-FIRST
(defvar iterate-emitted-body?)			; see ITERATE-EMIT-BODY,
						; and ITERATE-DO-FOR
(defvar iterate-iteration-variables)		; ITERATE-MAKE-ITERATION-VARIABLE
(defvar iterate-iteration-variablep)		; ditto
(defvar iterate-collect-cruft)			; for multiple COLLECTs (etc)
(defvar iterate-source-code)
(defvar iterate-duplicate-code nil)  ; see ITERATE-OPTIMIZE-DUPLICATED-CODE-ETC


;;;; Construct a value return


(defun iterate-construct-return (form)
  (if iterate-prog-names
      `(return-from ,(car iterate-prog-names) ,form)
      `(return ,form)))

;;;; Token Hackery

;Compare two "tokens".  The first is the frob out of ITERATE-SOURCE-CODE,
;the second a symbol to check against.

(defun iterate-tequal (x1 x2)
  (and (symbolp x1) (string= x1 x2)))


(defun iterate-tassoc (kwd alist)
  (and (symbolp kwd) (assoc kwd alist :test #'string=)))


(defun iterate-tmember (kwd list)
  (and (symbolp kwd) (member kwd list :test #'string=)))



(defmacro define-iterate-macro (keyword)
  "Makes KEYWORD, which is a ITERATE keyword, into a Lisp macro that may
introduce a ITERATE form.  This facility exists mostly for diehard users of
a predecessor of ITERATE.  Unconstrained use is not advised, as it tends to
decrease the transportability of the code and needlessly uses up a
function name."
  (or (eq keyword 'iterate)
      (iterate-tassoc keyword iterate-keyword-alist)
      (iterate-tassoc keyword iterate-iteration-keyword-alist)
      (iterate-simple-error "not a iterate keyword - define-iterate-macro" keyword))
  `(defmacro ,keyword (&whole whole-form  &environment env &rest keywords-and-forms) ; Boyer and Moore
     (declare (ignore keywords-and-forms))
     (iterate-translate whole-form env)))


(define-iterate-macro iterate)


(defmacro iterate-finish () 
  "Causes the iteration to terminate normally, the same as implicit
termination by an iteration driving clause, or by use of WHILE or
UNTIL -- the epilogue code (if any) will be run, and any implicitly
collected result will be returned as the value of the ITERATE."
  '(go end-iterate))


(defun iterate-translate (x iterate-macro-environment)
  (iterate-translate-1 x))


(defun iterate-end-testify (list-of-forms)
    (if (null list-of-forms) nil
	`(when ,(if (null (cdr (setq list-of-forms (nreverse list-of-forms))))
		    (car list-of-forms)
		    (cons 'or list-of-forms))
	   (go end-iterate))))

(defun iterate-optimize-duplicated-code-etc (&aux before after groupa groupb a b
					       lastdiff)
    (do ((l1 (nreverse iterate-before-iterate) (cdr l1))
	 (l2 (nreverse iterate-after-body) (cdr l2)))
	((equal l1 l2)
	   (setq iterate-body (nconc (delete nil l1) (nreverse iterate-body))))
      (push (car l1) before) (push (car l2) after))
    (cond ((not (null iterate-duplicate-code))
	     (setq iterate-before-iterate (nreverse (delete nil before))
		   iterate-after-body (nreverse (delete nil after))))
	  (t (setq iterate-before-iterate nil iterate-after-body nil
		   before (nreverse before) after (nreverse after))
	     (do ((bb before (cdr bb)) (aa after (cdr aa)))
		 ((null aa))
	       (cond ((not (equal (car aa) (car bb))) (setq lastdiff aa))
		     ((not (iterate-simplep (car aa)))	;Mustn't duplicate
		      (return nil))))
	     (cond (lastdiff  ;Down through lastdiff should be duplicated
		    (do nil (nil)
		      (and (car before) (push (car before) iterate-before-iterate))
		      (and (car after) (push (car after) iterate-after-body))
		      (setq before (cdr before) after (cdr after))
		      (and (eq after (cdr lastdiff)) (return nil)))
		    (setq iterate-before-iterate (nreverse iterate-before-iterate)
			  iterate-after-body (nreverse iterate-after-body))))
	     (do ((bb (nreverse before) (cdr bb))
		  (aa (nreverse after) (cdr aa)))
		 ((null aa))
	       (setq a (car aa) b (car bb))
	       (cond ((and (null a) (null b)))
		     ((equal a b)
			(iterate-output-group groupb groupa)
			(push a iterate-body)
			(setq groupb nil groupa nil))
		     (t (and a (push a groupa)) (and b (push b groupb)))))
	     (iterate-output-group groupb groupa)))
    (and iterate-never-stepped-variable
	 (push `(setq ,iterate-never-stepped-variable nil) iterate-after-body))
    nil)


(defun iterate-output-group (before after)
    (and (or after before)
	 (let ((v (or iterate-never-stepped-variable
		      (setq iterate-never-stepped-variable
			    (iterate-make-variable
			      (iterate-gentemp 'iterate-iter-flag-) t nil)))))
	    (push (cond ((not before)
			  `(unless ,v (progn ,@after)))
			((not after)
			  `(when ,v (progn ,@before)))
			(t `(cond (,v ,@before) (t ,@after))))
		  iterate-body))))


(defparameter *dcls-permitted* nil)  ; Added by Boyer and Moore

(defun iterate-translate-1 (iterate-source-code)
  (and (eq (car iterate-source-code) 'iterate)
       (setq iterate-source-code (cdr iterate-source-code)))
  (do ((iterate-iteration-variables nil)
       (iterate-iteration-variablep nil)
       (iterate-variables nil)
       (iterate-nodeclare nil)
       (iterate-named-variables nil)
       (iterate-declarations nil)
       (iterate-desetq-crocks nil)
       (iterate-variable-stack nil)
       (iterate-declaration-stack nil)
       (iterate-desetq-stack nil)
       (iterate-prologue nil)
       (iterate-wrappers nil)
       (iterate-before-iterate nil)
       (iterate-body nil)
       (iterate-emitted-body? nil)
       (iterate-after-body nil)
       (iterate-epilogue nil)
       (iterate-after-epilogue nil)
       (iterate-conditionals nil)
       (iterate-when-it-variable nil)
       (iterate-never-stepped-variable nil)
       (iterate-desetq-temporary nil)
       (iterate-prog-names nil)
       (iterate-collect-cruft nil)
       (keyword)
       (tem)
       (progvars))
      ((null iterate-source-code)
       (and iterate-conditionals
	    (iterate-simple-error "Hanging conditional in iterate macro"
			       (caadar iterate-conditionals)))
       (iterate-optimize-duplicated-code-etc)
       (iterate-bind-block)
       (and iterate-desetq-temporary (push iterate-desetq-temporary progvars))
       (setq tem `(block ,(car iterate-prog-names)
		    (let ,progvars
		      (tagbody
			,@(nreverse iterate-prologue)
			,@iterate-before-iterate
		     next-iterate
			,@iterate-body
			,@iterate-after-body
			(go next-iterate)
			(go end-iterate)
		     end-iterate
			,@(nreverse iterate-epilogue)
			,@(nreverse iterate-after-epilogue)))))
       (do ((vars) (dcls) (crocks))
	   ((null iterate-variable-stack))
	 (setq vars (car iterate-variable-stack)
	       iterate-variable-stack (cdr iterate-variable-stack)
	       dcls (car iterate-declaration-stack)
	       iterate-declaration-stack (cdr iterate-declaration-stack)
	       tem (list tem))
	 (and (setq crocks (pop iterate-desetq-stack))
	      (push (iterate-make-desetq crocks) tem))
	 (and *dcls-permitted*  ;  Boyer and Moore
	      dcls (push (cons 'declare dcls) tem))
	 (cond ((do ((l vars (cdr l))) ((null l) nil)
		  (and (not (atom (car l)))
		       (or (null (caar l)) (not (symbolp (caar l))))
		       (return t)))
		  (setq tem `(let ,(nreverse vars) ,@tem)))
	       (t (let ((lambda-vars nil) (lambda-vals nil))
		    (do ((l vars (cdr l)) (v)) ((null l))
		      (cond ((atom (setq v (car l)))
			       (push v lambda-vars)
			       (push nil lambda-vals))
			    (t (push (car v) lambda-vars)
			       (push (cadr v) lambda-vals))))
		    (setq tem `((lambda ,lambda-vars ,@tem)
				,@lambda-vals))))))
       (do ((l iterate-wrappers (cdr l))) ((null l))
	 (setq tem (append (car l) (list tem))))
       tem)
    ;;The following commented-out code is what comes from the newest source
    ;; code in use in NIL.  The code in use following it comes from about version
    ;; 803, that in use in symbolics release 6.1, for instance.  To turn on the
    ;; implicit DO feature, switch them and fix iterate-get-form to just pop the source.
    (if (symbolp (setq keyword (car iterate-source-code)))
    	(iterate-pop-source)
      (setq keyword 'do))
    (if (setq tem (iterate-tassoc keyword iterate-keyword-alist))
    	(apply (cadr tem) (cddr tem))
    	(if (setq tem (iterate-tassoc
    			 keyword iterate-iteration-keyword-alist))
    	    (iterate-hack-iteration tem)
    	    (if (iterate-tmember keyword '(and else))
    		; Alternative is to ignore it, ie let it go around to the
    		; next keyword...
    		(iterate-simple-error
    		   "secondary clause misplaced at top level in ITERATE macro"
    		   (list keyword (car iterate-source-code)
    			 (cadr iterate-source-code)))
    		(iterate-simple-error
    		   "unknown keyword in ITERATE macro" keyword))))
    ;;    (if (symbolp (setq keyword (iterate-pop-source)))
    ;;	(if (setq tem (iterate-tassoc keyword iterate-keyword-alist))
    ;;	    (apply (cadr tem) (cddr tem))
    ;;	    (if (setq tem (iterate-tassoc
    ;;			     keyword iterate-iteration-keyword-alist))
    ;;		(iterate-hack-iteration tem)
    ;;		(if (iterate-tmember keyword '(and else))
    ;;		    ; Alternative is to ignore it, ie let it go around to the
    ;;		    ; next keyword...
    ;;		    (iterate-simple-error
    ;;		       "secondary clause misplaced at top level in ITERATE macro"
    ;;		       (list keyword (car iterate-source-code)
    ;;			     (cadr iterate-source-code)))
    ;;		    (iterate-simple-error
    ;;		       "unknown keyword in ITERATE macro" keyword))))
    ;;	(iterate-simple-error
    ;;	   "found where keyword expected in ITERATE macro" keyword))
))


(defun iterate-bind-block ()
   (cond ((not (null iterate-variables))
	    (push iterate-variables iterate-variable-stack)
	    (push iterate-declarations iterate-declaration-stack)
	    (setq iterate-variables nil iterate-declarations nil)
	    (push iterate-desetq-crocks iterate-desetq-stack)
	    (setq iterate-desetq-crocks nil))))


;Get FORM argument to a keyword.  Read up to atom.  PROGNify if necessary.
(defun iterate-get-progn-1 ()
  (do ((forms (list (iterate-pop-source)) (cons (iterate-pop-source) forms))
       (nextform (car iterate-source-code) (car iterate-source-code)))
      ((atom nextform) (nreverse forms))))

(defun iterate-get-progn ()
  (let ((forms (iterate-get-progn-1)))
    (if (null (cdr forms)) (car forms) (cons 'progn forms))))

(defun iterate-get-form (for)
  ;; Until implicit DO is installed, use the following.  Then, replace it with
  ;; just iterate-pop-source.
  (let ((forms (iterate-get-progn-1)))
    (cond ((null (cdr forms)) (car forms))
	  (t (iterate-warn 
"The use of multiple forms with an implicit PROGN in this context
is considered obsolete, but is still supported for the time being.
If you did not intend to use multiple forms here, you probably omitted a DO.
If the use of multiple forms was intentional, put a PROGN in your code.
The offending clause"
		(if (atom for) (cons for forms) (append for forms)))
	     (cons 'progn forms)))))


;;;This function takes a substitutable expression containing generic arithmetic
;;; of some form or another, and a data type name, and substitutes for the function
;;; any type-specific functions for that type in the implementation.
(defun iterate-typed-arith (substitutable-expression data-type)
  (declare (ignore data-type))
  substitutable-expression)

(defvar iterate-floating-point-types
	'(flonum float short-float single-float double-float long-float))

(defun iterate-typed-init (data-type)
  (let ((tem nil))
    (cond ((data-type? data-type) (initial-value data-type))
	  ((iterate-tmember data-type '(fixnum integer number)) 0)
	  ((setq tem (car (iterate-tmember
			    data-type iterate-floating-point-types)))
	   (cond ((member tem '(flonum float)) 0.0)
		 (t (coerce 0 tem)))))))


(defun iterate-make-variable (name initialization dtype)
  (cond ((null name)
	   (cond ((not (null initialization))
		    (push (list (setq name (iterate-gentemp 'iterate-ignore-))
				initialization)
			  iterate-variables)
		      (push `(ignore ,name) iterate-declarations))))
	((atom name)
	   (cond (iterate-iteration-variablep
		    (if (member name iterate-iteration-variables)
			(iterate-simple-error
			   "Duplicated iteration variable somewhere in ITERATE"
			   name)
			(push name iterate-iteration-variables)))
		 ((assoc name iterate-variables)
		    (iterate-simple-error
		       "Duplicated var in ITERATE bind block" name)))
	   (or (symbolp name)
	       (iterate-simple-error "Bad variable somewhere in ITERATE" name))
	   (iterate-declare-variable name dtype)
	   ; We use ASSOC on this list to check for duplications (above),
	   ; so don't optimize out this list:
	   (push (list name (or initialization (iterate-typed-init dtype)))
		 iterate-variables))
	(initialization
	   (cond (iterate-use-system-destructuring?
		    (iterate-declare-variable name dtype)
		    (push (list name initialization) iterate-variables))
		 (t (let ((newvar (iterate-gentemp 'iterate-destructure-)))
		      (push (list newvar initialization) iterate-variables)
		      ; ITERATE-DESETQ-CROCKS gathered in reverse order.
		      (setq iterate-desetq-crocks
			    (list* name newvar iterate-desetq-crocks))
		      (iterate-make-variable name nil dtype)))))
	(t (let ((tcar nil) (tcdr nil))
	     (if (atom dtype) (setq tcar (setq tcdr dtype))
	       (setq tcar (car dtype) tcdr (cdr dtype)))
	     (iterate-make-variable (car name) nil tcar)
	     (iterate-make-variable (cdr name) nil tcdr))))
  name)


(defun iterate-make-iteration-variable (name initialization dtype)
    (let ((iterate-iteration-variablep t))
       (iterate-make-variable name initialization dtype)))


(defun iterate-declare-variable (name dtype)
    (cond ((or (null name) (null dtype)) nil)
	  ((symbolp name)
	     (cond ((member name iterate-nodeclare))
		   ((data-type? dtype)
		      (setq iterate-declarations
			    (append (variable-declarations dtype name)
				    iterate-declarations)))
		   (t (push `(type ,(if (iterate-tequal dtype 'notype) t dtype) ,name) iterate-declarations))))
	  ((consp name)
	      (cond ((consp dtype)
		       (iterate-declare-variable (car name) (car dtype))
		       (iterate-declare-variable (cdr name) (cdr dtype)))
		    (t (iterate-declare-variable (car name) dtype)
		       (iterate-declare-variable (cdr name) dtype))))
	  (t (iterate-simple-error "can't hack this"
				(list 'iterate-declare-variable name dtype)))))


(defun iterate-constantp (form)
  (constantp form))

(defun iterate-maybe-bind-form (form data-type?)
    ; Consider implementations which will not keep EQ quoted constants
    ; EQ after compilation & loading.
    ; Note FUNCTION is not hacked, multiple occurences might cause the
    ; compiler to break the function off multiple times!
    ; Hacking it probably isn't too important here anyway.  The ones that
    ; matter are the ones that use it as a stepper (or whatever), which
    ; handle it specially.
    (if (iterate-constantp form) form
	(iterate-make-variable (iterate-gentemp 'iterate-bind-) form data-type?)))


(defun iterate-optional-type ()
    (let ((token (car iterate-source-code)))
	(and (not (null token))
	     (or (not (atom token))
		 (data-type? token)
		 (iterate-tmember token '(fixnum integer number notype))
		 (iterate-tmember token iterate-floating-point-types))
	     (iterate-pop-source))))


;Incorporates conditional if necessary
(defun iterate-make-conditionalization (form)
  (cond ((not (null iterate-conditionals))
	   (rplacd (last (car (last (car (last iterate-conditionals)))))
		   (list form))
	   (cond ((iterate-tequal (car iterate-source-code) 'and)
		    (iterate-pop-source)
		    nil)
		 ((iterate-tequal (car iterate-source-code) 'else)
		    (iterate-pop-source)
		    ;; If we are already inside an else clause, close it off
		    ;; and nest it inside the containing when clause
		    (let ((innermost (car (last iterate-conditionals))))
		      (cond ((null (cddr innermost)))	;Now in a WHEN clause, OK
			    ((null (cdr iterate-conditionals))
			     (iterate-simple-error "More ELSEs than WHENs"
						(list 'else (car iterate-source-code)
						      (cadr iterate-source-code))))
			    (t (setq iterate-conditionals (cdr (nreverse iterate-conditionals)))
			       (rplacd (last (car (last (car iterate-conditionals))))
				       (list innermost))
			       (setq iterate-conditionals (nreverse iterate-conditionals)))))
		    ;; Start a new else clause
		    (rplacd (last (car (last iterate-conditionals)))
			    (list (list 't)))
		    nil)
		 (t ;Nest up the conditionals and output them
		     (do ((prev (car iterate-conditionals) (car l))
			  (l (cdr iterate-conditionals) (cdr l)))
			 ((null l))
		       (rplacd (last (car (last prev))) (list (car l))))
		     (prog1 (car iterate-conditionals)
			    (setq iterate-conditionals nil)))))
	(t form)))

(defun iterate-pseudo-body (form &aux (z (iterate-make-conditionalization form)))
   (cond ((not (null z))
	    (cond (iterate-emitted-body? (push z iterate-body))
		  (t (push z iterate-before-iterate) (push z iterate-after-body))))))

(defun iterate-emit-body (form)
  (setq iterate-emitted-body? t)
  (iterate-pseudo-body form))


(defun iterate-do-named ()
  (let ((name (iterate-pop-source)))
    (unless (and name (symbolp name))
      (iterate-simple-error "Bad name for your iterate construct" name))
    ;If this don't come first, ITERATE will be confused about how to return
    ; from the prog when it tries to generate such code
    (when (or iterate-before-iterate iterate-body iterate-after-epilogue)
      (iterate-simple-error "NAMED clause occurs too late" name))
    (when (cdr (setq iterate-prog-names (cons name iterate-prog-names)))
      (iterate-simple-error "Too many names for your iterate construct"
			 iterate-prog-names))))

(defun iterate-do-initially ()
  (push (iterate-get-progn) iterate-prologue))

(defun iterate-nodeclare (&aux (varlist (iterate-pop-source)))
    (or (null varlist)
	(consp varlist)
	(iterate-simple-error "Bad varlist to nodeclare iterate clause" varlist))
    (setq iterate-nodeclare (append varlist iterate-nodeclare)))

(defun iterate-do-finally ()
  (push (iterate-get-progn) iterate-epilogue))

(defun iterate-do-do ()
  (iterate-emit-body (iterate-get-progn)))

(defun iterate-do-return ()
   (iterate-pseudo-body (iterate-construct-return (iterate-get-form 'return))))


(defun iterate-do-collect (type)
  (let ((var nil) (form nil) (tem nil) (tail nil) (dtype nil) (cruft nil) (rvar nil)
	(ctype (case type
		 ((max min) 'maxmin)
		 ((nconc list append) 'list)
		 ((count sum) 'sum)
		 (t (error "ITERATE internal error:  ~S is an unknown collecting keyword."
			   type)))))
    (setq form (iterate-get-form type) dtype (iterate-optional-type))
    (cond ((iterate-tequal (car iterate-source-code) 'into)
	     (iterate-pop-source)
	     (setq rvar (setq var (iterate-pop-source)))))
    ; CRUFT will be (varname ctype dtype var tail (optional tem))
    (cond ((setq cruft (assoc var iterate-collect-cruft))
	     (cond ((not (eq ctype (car (setq cruft (cdr cruft)))))
		      (iterate-simple-error
		         "incompatible ITERATE collection types"
			 (list ctype (car cruft))))
		   ((and dtype (not (eq dtype (cadr cruft))))
		      ;Conditional should be on data-type reality
		    (error "~A and ~A Unequal data types into ~A"
			   dtype (cadr cruft) (car cruft))))
	     (setq dtype (car (setq cruft (cdr cruft)))
		   var (car (setq cruft (cdr cruft)))
		   tail (car (setq cruft (cdr cruft)))
		   tem (cadr cruft))
	     (and (eq ctype 'maxmin)
		  (not (atom form)) (null tem)
		  (rplaca (cdr cruft)
			  (setq tem (iterate-make-variable
				       (iterate-gentemp 'iterate-maxmin-)
				       nil dtype)))))
	  (t (unless dtype
	       (setq dtype (case type
			     (count 'fixnum)
			     ((min max sum) 'number))))
	     (unless var
	       (push (iterate-construct-return (setq var (iterate-gentemp)))
		     iterate-after-epilogue))
	     (iterate-make-iteration-variable var nil dtype)
	     (cond ((eq ctype 'maxmin)
		      ;Make a temporary.
		      (unless (atom form)
			(setq tem (iterate-make-variable
				    (iterate-gentemp) nil dtype)))
		      ;Use the tail slot of the collect database to hold a
		      ; flag which says we have been around once already.
		      (setq tail (iterate-make-variable
				   (iterate-gentemp 'iterate-maxmin-fl-) t nil)))
		   ((eq ctype 'list)
		    ;For dumb collection, we need both a tail and a flag var
		    ; to tell us whether we have iterated.
		    (setq tail (iterate-make-variable (iterate-gentemp) nil nil)
			  tem (iterate-make-variable (iterate-gentemp) nil nil))))
	     (push (list rvar ctype dtype var tail tem)
		   iterate-collect-cruft)))
    (iterate-emit-body
	(case type
	  (count (setq tem `(setq ,var (,(iterate-typed-arith '1+ dtype)
					,var)))
		 (if (or (eq form t) (equal form ''t))
		     tem
		     `(when ,form ,tem)))
	  (sum `(setq ,var (,(iterate-typed-arith '+ dtype) ,form ,var)))
	  ((max min)
	     (let ((forms nil) (arglist nil))
		; TEM is temporary, properly typed.
		(and tem (setq forms `((setq ,tem ,form)) form tem))
		(setq arglist (list var form))
		(push (if (iterate-tmember dtype '(fixnum flonum))
			  ; no contagious arithmetic
			  `(when (or ,tail
				     (,(iterate-typed-arith
				         (if (eq type 'max) '< '>)
					 dtype)
				      ,@arglist))
			     (setq ,tail nil ,@arglist))
			  ; potentially contagious arithmetic -- must use
			  ; MAX or MIN so that var will be contaminated
			  `(setq ,var (cond (,tail (setq ,tail nil) ,form)
					    (t (,type ,@arglist)))))
		      forms)
		(if (cdr forms) (cons 'progn (nreverse forms)) (car forms))))
	  (t (case type
		(list (setq form (list 'list form)))
		(append (or (and (not (atom form)) (eq (car form) 'list))
			    (setq form `(copy-list ,form)))))
	     (let ((q `(if ,tail (cdr (rplacd ,tail ,tem))
			 (setq ,var ,tem))))
		(if (and (not (atom form)) (eq (car form) 'list) (cdr form))
		    `(setq ,tem ,form ,tail ,(iterate-cdrify (cddr form) q))
		    `(when (setq ,tem ,form) (setq ,tail (last ,q))))))))))


(defun iterate-cdrify (arglist form)
    (do ((size (length arglist) (- size 4)))
	((< size 4)
	 (if (zerop size) form
	     (list (cond ((= size 1) 'cdr) ((= size 2) 'cddr) (t 'cdddr))
		   form)))
      ; (declare (type fixnum size))  ;  commented out by Boyer and Moore
      (setq form (list 'cddddr form))))



(defun iterate-do-while (negate? kwd &aux (form (iterate-get-form kwd)))
  (and iterate-conditionals (iterate-simple-error
			   "not allowed inside ITERATE conditional"
			   (list kwd form)))
  (iterate-pseudo-body `(,(if negate? 'when 'unless)
		      ,form (go end-iterate))))


(defun iterate-do-when (negate? kwd)
  (let ((form (iterate-get-form kwd)) (cond nil))
    (cond ((iterate-tequal (cadr iterate-source-code) 'it)
	     ;WHEN foo RETURN IT and the like
	     (setq cond `(setq ,(iterate-when-it-variable) ,form))
	     (setq iterate-source-code		;Plug in variable for IT
		   (list* (car iterate-source-code)
			  iterate-when-it-variable
			  (cddr iterate-source-code))))
	  (t (setq cond form)))
    (and negate? (setq cond `(not ,cond)))
    (setq iterate-conditionals (nconc iterate-conditionals `((cond (,cond)))))))

(defun iterate-do-with ()
  (do ((var) (equals) (val) (dtype)) (nil)
    (setq var (iterate-pop-source) equals (car iterate-source-code))
    (cond ((iterate-tequal equals '=)
	     (iterate-pop-source)
	     (setq val (iterate-get-form (list 'with var '=)) dtype nil))
	  ((or (iterate-tequal equals 'and)
	       (iterate-tassoc equals iterate-keyword-alist)
	       (iterate-tassoc equals iterate-iteration-keyword-alist))
	     (setq val nil dtype nil))
	  (t (setq dtype (iterate-optional-type) equals (car iterate-source-code))
	     (cond ((iterate-tequal equals '=)
		      (iterate-pop-source)
		      (setq val (iterate-get-form (list 'with var dtype '=))))
		   ((and (not (null iterate-source-code))
			 (not (iterate-tassoc equals iterate-keyword-alist))
			 (not (iterate-tassoc
				 equals iterate-iteration-keyword-alist))
			 (not (iterate-tequal equals 'and)))
		      (iterate-simple-error "Garbage where = expected" equals))
		   (t (setq val nil)))))
    (iterate-make-variable var val dtype)
    (if (not (iterate-tequal (car iterate-source-code) 'and)) (return nil)
	(iterate-pop-source)))
  (iterate-bind-block))

(defun iterate-do-always (negate?)
  (let ((form (iterate-get-form 'always)))
    (iterate-emit-body `(,(if negate? 'when 'unless) ,form
		      ,(iterate-construct-return nil)))
    (push (iterate-construct-return t) iterate-after-epilogue)))

;THEREIS expression
;If expression evaluates non-nil, return that value.
(defun iterate-do-thereis ()
   (iterate-emit-body `(when (setq ,(iterate-when-it-variable)
				,(iterate-get-form 'thereis))
		      ,(iterate-construct-return iterate-when-it-variable))))


;;;; Hacks

(defun iterate-simplep (expr)
    (if (null expr) 0
      (catch 'iterate-simplep
	(let ((ans (iterate-simplep-1 expr)))
	  ; (declare (fixnum ans)) ; commented out by Boyer and Moore
	  (and (< ans 20.) ans)))))

(defvar iterate-simplep
	'(> < <= >= /= + - 1+ 1- ash equal atom setq prog1 prog2 and or = aref char schar sbit svref))

(defun iterate-simplep-1 (x)
  (let ((z 0))
    ; (declare (fixnum z)) ; commented out by Boyer and Moore
    (cond ((iterate-constantp x) 0)
	  ((atom x) 1)
	  ((eq (car x) 'cond)
	     (do ((cl (cdr x) (cdr cl))) ((null cl))
	       (do ((f (car cl) (cdr f))) ((null f))
		 (setq z (+ (iterate-simplep-1 (car f)) z 1))))
	     z)
	  ((symbolp (car x))
	     (let ((fn (car x)) (tem nil))
	       (cond ((setq tem (get fn 'iterate-simplep))
		        (if (typep tem 'fixnum) (setq z tem)
			    (setq z (funcall tem x) x nil)))
		     ((member fn '(null not eq go return progn)))
		     ((member fn '(car cdr)) (setq z 1))
		     ((member fn '(caar cadr cdar cddr)) (setq z 2))
		     ((member fn '(caaar caadr cadar caddr
				   cdaar cdadr cddar cdddr))
		        (setq z 3))
		     ((member fn '(caaaar caaadr caadar caaddr
				   cadaar cadadr caddar cadddr
				   cdaaar cdaadr cdadar cdaddr
				   cddaar cddadr cdddar cddddr))
		        (setq z 4))
		     ((member fn iterate-simplep) (setq z 2))
		     (t (multiple-value-bind (new-form expanded-p)
			      (macroexpand-1 x iterate-macro-environment)
			  (if expanded-p
			      (setq z (iterate-simplep-1 new-form) x nil)
			    (throw 'iterate-simplep nil)))))
	       (do ((l (cdr x) (cdr l))) ((null l))
		 (setq z (+ (iterate-simplep-1 (car l)) 1 z)))
	       z))
	  (t (throw 'iterate-simplep nil)))))


;;;; The iteration driver
(defun iterate-hack-iteration (entry)
  (do ((last-entry entry)
       (source iterate-source-code iterate-source-code)
       (pre-step-tests nil)
       (steps nil)
       (post-step-tests nil)
       (pseudo-steps nil)
       (pre-iterate-pre-step-tests nil)
       (pre-iterate-steps nil)
       (pre-iterate-post-step-tests nil)
       (pre-iterate-pseudo-steps nil)
       (tem) (data) (foo) (bar))
      (nil)
    ; Note we collect endtests in reverse order, but steps in correct
    ; order.  ITERATE-END-TESTIFY does the nreverse for us.
    (setq tem (setq data (apply (cadr entry) (cddr entry))))
    (and (car tem) (push (car tem) pre-step-tests))
    (setq steps (nconc steps (iterate-copylist* (car (setq tem (cdr tem))))))
    (and (car (setq tem (cdr tem))) (push (car tem) post-step-tests))
    (setq pseudo-steps
	  (nconc pseudo-steps (iterate-copylist* (car (setq tem (cdr tem))))))
    (setq tem (cdr tem))
    (and (or iterate-conditionals iterate-emitted-body?)
	 (or tem pre-step-tests post-step-tests pseudo-steps)
	 (let ((cruft (list (car entry) (car source)
			    (cadr source) (caddr source))))
	    (if iterate-emitted-body?
		(iterate-simple-error
		   "Iteration is not allowed to follow body code" cruft)
		(iterate-simple-error
		   "Iteration starting inside of conditional in ITERATE"
		   cruft))))
    (or tem (setq tem data))
    (and (car tem) (push (car tem) pre-iterate-pre-step-tests))
    (setq pre-iterate-steps
	  (nconc pre-iterate-steps (iterate-copylist* (car (setq tem (cdr tem))))))
    (and (car (setq tem (cdr tem))) (push (car tem) pre-iterate-post-step-tests))
    (setq pre-iterate-pseudo-steps
	  (nconc pre-iterate-pseudo-steps (iterate-copylist* (cadr tem))))
    (cond ((or (not (iterate-tequal (car iterate-source-code) 'and))
	       (and iterate-conditionals
		    (not (iterate-tassoc (cadr iterate-source-code)
					 iterate-iteration-keyword-alist))))
	     (setq foo (list (iterate-end-testify pre-iterate-pre-step-tests)
			     (iterate-make-psetq pre-iterate-steps)
			     (iterate-end-testify pre-iterate-post-step-tests)
			     (iterate-make-setq pre-iterate-pseudo-steps))
		   bar (list (iterate-end-testify pre-step-tests)
			     (iterate-make-psetq steps)
			     (iterate-end-testify post-step-tests)
			     (iterate-make-setq pseudo-steps)))
	     (cond ((not iterate-conditionals)
		      (setq iterate-before-iterate (nreconc foo iterate-before-iterate)
			    iterate-after-body (nreconc bar iterate-after-body)))
		   (t ((lambda (iterate-conditionals)
			  (push (iterate-make-conditionalization
				   (cons 'progn (delete nil foo)))
				iterate-before-iterate))
		       (mapcar #'(lambda (x)	;Copy parts that will get rplacd'ed
				   (cons (car x)
					 (mapcar #'(lambda (x) (iterate-copylist* x)) (cdr x))))
			       iterate-conditionals))
		      (push (iterate-make-conditionalization
			       (cons 'progn (delete nil bar)))
			    iterate-after-body)))
	     (iterate-bind-block)
	     (return nil)))
    (iterate-pop-source) ; flush the "AND"
    (setq entry (cond ((setq tem (iterate-tassoc
				    (car iterate-source-code)
				    iterate-iteration-keyword-alist))
		         (iterate-pop-source)
			 (setq last-entry tem))
		      (t last-entry)))))


;FOR variable keyword ..args..
(defun iterate-do-for ()
  (let ((var (iterate-pop-source))
	(data-type? (iterate-optional-type))
	(keyword (iterate-pop-source))
	(first-arg nil)
	(tem nil))
    (setq first-arg (iterate-get-form (list 'for var keyword)))
    (or (setq tem (iterate-tassoc keyword iterate-for-keyword-alist))
	(iterate-simple-error
	   "Unknown keyword in FOR or AS clause in ITERATE"
	   (list 'for var keyword)))
    (apply (cadr tem) var first-arg data-type? (cddr tem))))


(defun iterate-do-repeat ()
    (let ((var (iterate-make-variable
		  (iterate-gentemp 'iterate-repeat-)
		  (iterate-get-form 'repeat) 'fixnum)))
       `((not (,(iterate-typed-arith 'plusp 'fixnum) ,var))
         () ()
         (,var (,(iterate-typed-arith '1- 'fixnum) ,var)))))


; Kludge the First
(defun iterate-when-it-variable ()
    (or iterate-when-it-variable
	(setq iterate-when-it-variable
	      (iterate-make-variable (iterate-gentemp 'iterate-it-) nil nil))))



(defun iterate-for-equals (var val data-type?)
  (cond ((iterate-tequal (car iterate-source-code) 'then)
	   ;FOR var = first THEN next
	   (iterate-pop-source)
	   (iterate-make-iteration-variable var val data-type?)
	   `(() (,var ,(iterate-get-form (list 'for var '= val 'then))) () ()
	     () () () ()))
	(t (iterate-make-iteration-variable var nil data-type?)
	   (let ((varval (list var val)))
	     (cond (iterate-emitted-body?
		    (iterate-emit-body (iterate-make-setq varval))
		    '(() () () ()))
		   (`(() ,varval () ())))))))

(defun iterate-for-first (var val data-type?)
    (or (iterate-tequal (car iterate-source-code) 'then)
	(iterate-simple-error "found where THEN expected in FOR ... FIRST"
			   (car iterate-source-code)))
    (iterate-pop-source)
    (iterate-make-iteration-variable var nil data-type?)
    `(() (,var ,(iterate-get-form (list 'for var 'first val 'then))) () ()
      () (,var ,val) () ()))


(defun iterate-list-stepper (var val data-type? fn)
    (let ((stepper (cond ((iterate-tequal (car iterate-source-code) 'by)
			    (iterate-pop-source)
			    (iterate-get-form (list 'for var
						 (if (eq fn 'car) 'in 'on)
						 val 'by)))
			 (t '(function cdr))))
	  (var1 nil) (stepvar nil) (step nil) (et nil) (pseudo nil))
       (setq step (if (or (atom stepper)
			  (not (member (car stepper) '(quote function))))
		      `(funcall ,(setq stepvar (iterate-gentemp 'iterate-fn-)))
		      (list (cadr stepper))))
       (cond ((and (atom var)
		   ;; (eq (car step) 'cdr)
		   (not fn))
	        (setq var1 (iterate-make-iteration-variable var val data-type?)))
	     (t (iterate-make-iteration-variable var nil data-type?)
		(setq var1 (iterate-make-variable
			     (iterate-gentemp 'iterate-list-) val nil))
		(setq pseudo (list var (if fn (list fn var1) var1)))))
       (rplacd (last step) (list var1))
       (and stepvar (iterate-make-variable stepvar stepper nil))
       (setq stepper (list var1 step) et `(null ,var1))
       (if (not pseudo) `(() ,stepper ,et () () () ,et ())
	   (if (eq (car step) 'cdr) `(,et ,pseudo () ,stepper)
	       `((null (setq ,@stepper)) () () ,pseudo ,et () () ,pseudo)))))


(defun iterate-for-arithmetic (var val data-type? kwd)
  ; Args to iterate-sequencer:
  ; indexv indexv-type variable? vtype? sequencev? sequence-type
  ; stephack? default-top? crap prep-phrases
  (iterate-sequencer
     var (or data-type? 'fixnum) nil nil nil nil nil nil `(for ,var ,kwd ,val)
     (cons (list kwd val)
	   (iterate-gather-preps
	      '(from upfrom downfrom to upto downto above below by)
	      nil))))


(defun iterate-named-variable (name)
    (let ((tem (iterate-tassoc name iterate-named-variables)))
       (cond ((null tem) (iterate-gentemp))
	     (t (setq iterate-named-variables (delete tem iterate-named-variables))
		(cdr tem)))))


; Note:  path functions are allowed to use iterate-make-variable, hack
; the prologue, etc.
(defun iterate-for-being (var val data-type?)
   ; FOR var BEING something ... - var = VAR, something = VAL.
   ; If what passes syntactically for a pathname isn't, then
   ; we trap to the DEFAULT-ITERATE-PATH path;  the expression which looked like
   ; a path is given as an argument to the IN preposition.  Thus,
   ; by default, FOR var BEING EACH expr OF expr-2
   ; ==> FOR var BEING DEFAULT-ITERATE-PATH IN expr OF expr-2.
   (let ((tem nil) (inclusive? nil) (ipps nil) (each? nil) (attachment nil))
     (if (or (iterate-tequal val 'each) (iterate-tequal val 'the))
	 (setq each? 't val (car iterate-source-code))
	 (push val iterate-source-code))
     (cond ((and (setq tem (iterate-tassoc val iterate-path-keyword-alist))
		 (or each? (not (iterate-tequal (cadr iterate-source-code)
						'and))))
	      ;; FOR var BEING {each} path {prep expr}..., but NOT
	      ;; FOR var BEING var-which-looks-like-path AND {ITS} ...
	      (iterate-pop-source))
	   (t (setq val (iterate-get-form (list 'for var 'being)))
	      (cond ((iterate-tequal (car iterate-source-code) 'and)
		       ;; FOR var BEING value AND ITS path-or-ar
		       (or (null each?)
			   (iterate-simple-error
			      "Malformed BEING EACH clause in ITERATE" var))
		       (setq ipps `((of ,val)) inclusive? t)
		       (iterate-pop-source)
		       (or (iterate-tmember (setq tem (iterate-pop-source))
					    '(its his her their each))
			   (iterate-simple-error
			      "found where ITS or EACH expected in ITERATE path"
			      tem))
		       (if (setq tem (iterate-tassoc
					(car iterate-source-code)
					iterate-path-keyword-alist))
			   (iterate-pop-source)
			   (push (setq attachment
				       `(in ,(iterate-get-form
					      `(for ,var being \.\.\. in))))
				 ipps)))
		    ((not (setq tem (iterate-tassoc
				       (car iterate-source-code)
				       iterate-path-keyword-alist)))
		       ; FOR var BEING {each} a-r ...
		       (setq ipps (list (setq attachment (list 'in val)))))
		    (t ; FOR var BEING {each} pathname ...
		       ; Here, VAL should be just PATHNAME.
		       (iterate-pop-source)))))
     (cond ((not (null tem)))
	   ((not (setq tem (iterate-tassoc 'default-iterate-path
					   iterate-path-keyword-alist)))
	      (iterate-simple-error "Undefined ITERATE iteration path"
				 (cadr attachment))))
     (setq tem (funcall (cadr tem) (car tem) var data-type?
			(nreconc ipps (iterate-gather-preps (caddr tem) t))
			inclusive? (caddr tem) (cdddr tem)))
     (and iterate-named-variables
	  (iterate-simple-error "unused USING variables" iterate-named-variables))
     ; For error continuability (if there is any):
     (setq iterate-named-variables nil)
     ;; TEM is now (bindings prologue-forms . stuff-to-pass-back)
     (do ((l (car tem) (cdr l)) (x)) ((null l))
       (if (atom (setq x (car l)))
	   (iterate-make-iteration-variable x nil nil)
	   (iterate-make-iteration-variable (car x) (cadr x) (caddr x))))
     (setq iterate-prologue (nconc (reverse (cadr tem)) iterate-prologue))
     (cddr tem)))


(defun iterate-gather-preps (preps-allowed crockp)
   (do ((token (car iterate-source-code) (car iterate-source-code)) (preps nil))
       (nil)
     (cond ((iterate-tmember token preps-allowed)
	      (push (list (iterate-pop-source)
			  (iterate-get-form `(for \... being \... ,token)))
		    preps))
	   ((iterate-tequal token 'using)
	      (iterate-pop-source)
	      (or crockp (iterate-simple-error
			    "USING used in illegal context"
			    (list 'using (car iterate-source-code))))
	      (do ((z (car iterate-source-code) (car iterate-source-code)) (tem))
		  ((atom z))
		(and (or (atom (cdr z))
			 (not (null (cddr z)))
			 (not (symbolp (car z)))
			 (and (cadr z) (not (symbolp (cadr z)))))
		     (iterate-simple-error
		        "bad variable pair in path USING phrase" z))
		(cond ((not (null (cadr z)))
		         (and (setq tem (iterate-tassoc
					   (car z) iterate-named-variables))
			      (iterate-simple-error
			         "Duplicated var substitition in USING phrase"
				 (list tem z)))
			 (push (cons (car z) (cadr z)) iterate-named-variables)))
		(iterate-pop-source)))
	   (t (return (nreverse preps))))))

(defun iterate-add-path (name data)
    (setq iterate-path-keyword-alist
	  (cons (cons name data)
		(delete (iterate-tassoc name iterate-path-keyword-alist)
			iterate-path-keyword-alist
			:test #'eq)))
    nil)


(defmacro define-iterate-path (names &rest cruft)
  "(DEFINE-ITERATE-PATH NAMES PATH-FUNCTION LIST-OF-ALLOWABLE-PREPOSITIONS
DATUM-1 DATUM-2 ...)
Defines PATH-FUNCTION to be the handler for the path(s) NAMES, which may
be either a symbol or a list of symbols.  LIST-OF-ALLOWABLE-PREPOSITIONS
contains a list of prepositions allowed in NAMES. DATUM-i are optional;
they are passed on to PATH-FUNCTION as a list."
  (setq names (if (atom names) (list names) names))
  (let ((forms (mapcar #'(lambda (name) `(iterate-add-path ',name ',cruft))
		       names)))
    `(eval-when (eval load compile) ,@forms)))


(defun iterate-sequencer (indexv indexv-type
			  variable? vtype?
			  sequencev? sequence-type?
			  stephack? default-top?
			  crap prep-phrases)
   (let ((endform nil) (sequencep nil) (test nil)
	 (step ; Gross me out!
	       (1+ (or (iterate-typed-init indexv-type) 0)))
	 (dir nil) (inclusive-iteration? nil) (start-given? nil) (limit-given? nil))
     (and variable? (iterate-make-iteration-variable variable? nil vtype?))
     (do ((l prep-phrases (cdr l)) (prep) (form) (odir)) ((null l))
       (setq prep (caar l) form (cadar l))
       (cond ((iterate-tmember prep '(of in))
		(and sequencep (iterate-simple-error
				  "Sequence duplicated in ITERATE path"
				  (list variable? (car l))))
		(setq sequencep t)
		(iterate-make-variable sequencev? form sequence-type?))
	     ((iterate-tmember prep '(from downfrom upfrom))
	        (and start-given?
		     (iterate-simple-error
		        "Iteration start redundantly specified in ITERATE sequencing"
			(append crap l)))
		(setq start-given? t)
		(cond ((iterate-tequal prep 'downfrom) (setq dir 'down))
		      ((iterate-tequal prep 'upfrom) (setq dir 'up)))
		(iterate-make-iteration-variable indexv form indexv-type))
	     ((cond ((iterate-tequal prep 'upto)
		       (setq inclusive-iteration? (setq dir 'up)))
		    ((iterate-tequal prep 'to)
		       (setq inclusive-iteration? t))
		    ((iterate-tequal prep 'downto)
		       (setq inclusive-iteration? (setq dir 'down)))
		    ((iterate-tequal prep 'above) (setq dir 'down))
		    ((iterate-tequal prep 'below) (setq dir 'up)))
		(and limit-given?
		     (iterate-simple-error
		       "Endtest redundantly specified in ITERATE sequencing path"
		       (append crap l)))
		(setq limit-given? t)
		(setq endform (iterate-maybe-bind-form form indexv-type)))
	     ((iterate-tequal prep 'by)
		(setq step (if (iterate-constantp form) form
			       (iterate-make-variable
				 (iterate-gentemp 'iterate-step-by-)
				 form 'fixnum))))
	     (t ; This is a fatal internal error...
	        (iterate-simple-error "Illegal prep in sequence path"
				   (append crap l))))
       (and odir dir (not (eq dir odir))
	    (iterate-simple-error
	       "Conflicting stepping directions in ITERATE sequencing path"
	       (append crap l)))
       (setq odir dir))
     (and sequencev? (not sequencep)
	  (iterate-simple-error "Missing OF phrase in sequence path" crap))
     ; Now fill in the defaults.
     (setq step (list indexv step))
     (cond ((member dir '(nil up))
	      (or start-given?
		  (iterate-make-iteration-variable indexv 0 indexv-type))
	      (and (or limit-given?
		       (cond (default-top?
			        (iterate-make-variable
				  (setq endform (iterate-gentemp
						  'iterate-seq-limit-))
				  nil indexv-type)
				(push `(setq ,endform ,default-top?)
				      iterate-prologue))))
		   (setq test (if inclusive-iteration? '(> . args)
				  '(>= . args))))
	      (push '+ step))
	   (t (cond ((not start-given?)
		       (or default-top?
			   (iterate-simple-error
			      "Don't know where to start stepping"
			      (append crap prep-phrases)))
		       (iterate-make-iteration-variable indexv 0 indexv-type)
		       (push `(setq ,indexv
				    (,(iterate-typed-arith '1- indexv-type)
				     ,default-top?))
			     iterate-prologue)))
	      (cond ((and default-top? (not endform))
		       (setq endform (iterate-typed-init indexv-type)
			     inclusive-iteration? t)))
	      (and (not (null endform))
		   (setq test (if inclusive-iteration? '(< . args)
				  '(<= . args))))
	      (push '- step)))
     (and (and (numberp (caddr step)) (= (caddr step) 1))	;Generic arith
	  (rplacd (cdr (rplaca step (if (eq (car step) '+) '1+ '1-)))
		  nil))
     (rplaca step (iterate-typed-arith (car step) indexv-type))
     (setq step (list indexv step))
     (setq test (iterate-typed-arith test indexv-type))
     (setq test (subst (list indexv endform) 'args test))
     (and stephack? (setq stephack? `(,variable? ,stephack?)))
     `(() ,step ,test ,stephack?
       () () ,test ,stephack?)))


(defun iterate-sequence-elements-path (path variable data-type
				       prep-phrases inclusive?
				       allowed-preps data)
    allowed-preps ; unused
    (let ((indexv (iterate-named-variable 'index))
	  (sequencev (iterate-named-variable 'sequence))
	  (fetchfun nil) (sizefun nil) (type nil) (default-var-type nil)
	  (crap `(for ,variable being the ,path)))
       (cond ((not (null inclusive?))
	        (rplacd (cddr crap) `(,(cadar prep-phrases) and its ,path))
		(iterate-simple-error "Can't step sequence inclusively" crap)))
       (setq fetchfun (car data)
	     sizefun (car (setq data (cdr data)))
	     type (car (setq data (cdr data)))
	     default-var-type (cadr data))
       (list* nil nil ; dummy bindings and prologue
	      (iterate-sequencer
	         indexv 'fixnum
		 variable (or data-type default-var-type)
		 sequencev type
		 `(,fetchfun ,sequencev ,indexv) `(,sizefun ,sequencev)
		 crap prep-phrases))))



(defmacro define-iterate-sequence-path (path-name-or-names fetchfun sizefun
				     &optional sequence-type element-type)
  "Defines a sequence iiteration path.  PATH-NAME-OR-NAMES is either an
atomic path name or a list of path names.  FETCHFUN is a function of
two arguments, the sequence and the index of the item to be fetched.
Indexing is assumed to be zero-origined.  SIZEFUN is a function of
one argument, the sequence; it should return the number of elements in
the sequence.  SEQUENCE-TYPE is the name of the data-type of the
sequence, and ELEMENT-TYPE is the name of the data-type of the elements
of the sequence."
    `(define-iterate-path ,path-name-or-names
	iterate-sequence-elements-path
	(of in from downfrom to downto below above by)
	,fetchfun ,sizefun ,sequence-type ,element-type))


;;;; Setup stuff


(mapc #'(lambda (x)
	  (mapc #'(lambda (y)
		    (setq iterate-path-keyword-alist
			  (cons `(,y iterate-sequence-elements-path
				  (of in from downfrom to downto
				      below above by)
				  ,@(cdr x))
				(delete (iterate-tassoc
					  y iterate-path-keyword-alist)
					iterate-path-keyword-alist
					:test #'eq :count 1))))
		(car x)))
      '( ((element elements) elt length sequence)
	;The following should be done by using ELEMENTS and type dcls...
	  ((vector-element 
	    vector-elements 
	    array-element    ;; Backwards compatibility -- DRM
	    array-elements)
	   aref length vector)
	  ((simple-vector-element simple-vector-elements
	    simple-general-vector-element simple-general-vector-elements)
	   svref simple-vector-length simple-vector)
	  ((bits bit bit-vector-element bit-vector-elements)
	     bit bit-vector-length bit-vector bit)
	  ((simple-bit-vector-element simple-bit-vector-elements)
	     sbit simple-bit-vector-length simple-bit-vector bit)
	  ((character characters string-element string-elements)
	   char string-length string string-char)
	  ((simple-string-element simple-string-elements)
	   schar simple-string-length simple-string string-char)
	)
      )


(pushnew 'iterate *features*)  ;; Common-Lisp says this is correct.
(pushnew :iterate *features*)  ;; But Lucid only understands this one.

(defun initial-value (x) x nil)
(defun variable-declarations (type &rest vars) type vars nil)

