(DEFPARAMETER *THM-WARNING* 

"This theorem proving program is a tool for computer science
research.  It is provided without license, copyright, or fee in a
spirit of scientific cooperation.  We have tried our very best to make
this system logically sound, but we cannot accept financial
responsibility for problems that arise with its use.  Therefore, we
make the following disclaimer:  Robert S. Boyer, J Strother Moore,
Computational Logic, Inc., and other parties provide this program on
an \"as is\" basis without warranty of any kind, either expressed or
implied, including, but not limited to, the implied warranties of
merchantability and fitness for a particular purpose.

Although we impose no copyright or license restrictions on the use and
distribution of this program, we adamantly insist that it be changed
only under the condition that all changes be prominently described and
accurately attributed.  A notice of any change should be included at
the beginning of the file basis.lisp and announced to the user by
terminal output when BOOT-STRAP or NOTE-LIB are called.

We also insist, as a matter of scientific accuracy, that any
announcement of the proof of a theorem by any modification of this
program clearly include in the announcement the fact that changes were
made to the program.

To avoid this warning, (SETQ *THM-WARNING-FLG* NIL).
")

(DEFVAR *THM-WARNING-FLG* T)

(EVAL-WHEN (LOAD EVAL COMPILE)
	   (CHK-BASE-AND-PACKAGE 10 *PACKAGE*))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                             ;
;                              Our Theorem Prover                             ;
;                                                                             ;
;                       R. S. Boyer and J Strother Moore                      ;
;                                                                             ;
;                                 Austin, Texas                               ;
;                                                                             ;
;                                   circa 1986                                ;
;                                                                             ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;                            LANGUAGES AND MACHINES

;   We began working on this code in 1972 in Edinburgh at the Metamathematics
;   Unit.  The first version was written in POP-2 for an ICL 4130.  In Palo
;   Alto in 1974 we translated it into Interlisp-10 for a Xerox MAXC and a DEC
;   PDP10/BBN-TENEX.  We continued developing it at Xerox PARC, SRI, and the
;   University of Texas on two DEC-2060s.  In 1983, we translated it into
;   Maclisp for a DEC-2060 a Honeywell Multics and into Zetalisp for a
;   Symbolics 3600.  John Nagle made it compatible with Franz Lisp for the VAX.
;   In 1987 we converted it to run under Common Lisp.

;                                   OWNERSHIP

;   The development of this system has been primarily financed by the National
;   Science Foundation and the Office of Naval Research.  The NSF Grant Policy
;   Manual NSF-77-47, Revised October 1979, states in paragraph 754.2 "Data
;   banks and software, produced with the assistance of NSF grants, having
;   utility to others in addition to the grantee, shall be made available to
;   users, at no cost to the grantee, by publication or, on request, by
;   duplication or loan for reproduction by others. ... Any out of pocket
;   expenses incurred by the grantee in providing information to third parties
;   may be charged to the third party."  The conversion to Common Lisp was
;   paid for by DARPA. 


;                                 DOCUMENTATION

;   The 1978 version of this system was carefully described in "A Computational
;   Logic," Academic Press, 1979.  But much has changed.  (1) An extensive
;   amount of "linear" arithmetic has been built-in and is described in
;   "Integrating Decision Procedrues into Heuristic Theorem Provers:  A Case
;   Study of Linear Arithmetic," Robert S. Boyer and J Strother Moore,
;   Institute for Computing Science, Technical Report ICSCS-CMP-44, University of
;   Texas at Austin, Austin, Texas 78712, Jan. 1985.  (2) We have added the
;   notion of metafunctions described in "Metafunctions:  Proving Them Correct
;   and Using Them Efficiently as New Proof Procedures" in "The Correctness
;   Problem in Computer Science," R. S. Boyer and J S. Moore (editors),
;   Academic Press, 1981.  (3) Equality reasoning in the ground case is
;   complete.  (4) The mechanism for guessing well-founded relations at
;   definition time has been vastly simplified but now requires more explicit
;   guidance from the user in the nontrivial cases.  (5) A variety of hints can
;   be given to the theorem-prover.

;   The user's manual for the 1978 version of the theorem-prover is extremely
;   out of date.  The best we can offer at this time is the file CODE.DOC, "A
;   Poor Substitute for a User's Manual for a Theorem Prover for a
;   Computational Logic."


;				LISP PRIMITIVES

;   This file, BASIS, has all of the SPECIAL declarations for our
;   theorem-prover.  We load the compiled version of the file BASIS before we
;   compile the rest of the theorem-prover.


(DEFPARAMETER THEOREM-PROVER-FILES
  (QUOTE (BASIS GENFACT EVENTS CODE-1-A CODE-B-D CODE-E-M CODE-N-R CODE-S-Z
		IO PPR))) 

;				    ERRORS

;   The following macro lets us protect against ERROR1's in a way
;   analogous to ERRSET's protection against errors.  The spec is that
;   if form causes no ERROR1s then (ERROR1-SET form) returns (LIST form)
;   if form causes a SOFT ERROR1, the ERROR1-SET returns NIL.  HARD ERROR1s
;   ignore ERROR1-SET and cause ERRORs, which may be a simple reset
;   in some lisps.

(DEFMACRO ERROR1-SET (FORM)
  `(LET ((ERROR1-SET-FLG T))
     (CATCH (QUOTE ERROR1-SET) (LIST ,FORM))))


;			SUITABILITY OF THE COMMON LISP

(DEFCONSTANT PRINTING-COMMON-LISP-CHARACTERS
             '(#\!
               #\" ; " This spurious double quotation mark is to overcome
                   ; the cleverness of the ZMACS text editor on a Symbolics
                   ; running Genera 7, which claims that otherwise this
                   ; file appears to have unbalanced parentheses.
               #\# #\$ #\% #\& #\' #\( #\) #\* #\+ #\, #\- #\. #\/
	       #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9
	       #\: #\; #\< #\= #\> #\? #\@
	       #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L
	       #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z
	       #\[ #\\ #\] #\^ #\_ #\`
	       #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l
	       #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z
	       #\{ #\| #\} #\~ ))

(DEFCONSTANT ASCII-CODES-FOR-PRINTING-COMMON-LISP-CHARACTERS
  '(33 34 35 36 37 38 39 40 41 42 43 44 45 46 47
       48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63
       64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79
       80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95
       96 97 98 99 100 101 102 103 104 105 106 107 108
       109 110 111 112 113 114 115 116 117 118 119 120
       121 122 123 124 125 126))

(DEFUN CHK-FOR-SUITABILITY-OF-THIS-COMMON-LISP ()
  #|(COND ((NOT (AND (TYPEP (1- (EXPT 2 31)) 'FIXNUM)
                   (TYPEP (- (EXPT 2 31)) 'FIXNUM)))
         (ERROR "THIS COMMON LISP DOES NOT SUPPORT FIXNUMS ~
                 IN THE RANGE 2^31-1 TO -2^31.")))|#
  (ITERATE FOR CHAR IN PRINTING-COMMON-LISP-CHARACTERS
	   AS N IN ASCII-CODES-FOR-PRINTING-COMMON-LISP-CHARACTERS
	   DO
	   (COND ((NOT (= N (CHAR-CODE CHAR)))
		  (ERROR "This is not an ascii based Common Lisp ~
                   because character ~A does not have code ~A."
			 CHAR N)))))

(CHK-FOR-SUITABILITY-OF-THIS-COMMON-LISP)


;				   SYMBOLPs

;	 VARIABLE DECLARATIONS FOR THE OPERATING SYSTEM DEPENDENT CODE

(DEFCONSTANT EVENT-SEPARATOR-STRING "


")

(DEFPARAMETER IPOSITION-ALIST NIL)

(DEFVAR LIB-FILE)

(DEFVAR LIB-VARS)

(DEFVAR LIB-ATOMS-WITH-PROPS)

(DEFVAR LIB-ATOMS-WITH-DEFS)

(DEFVAR LIB-PROPS)

;   LIB-PROPS is automatically set by ADD-SUB-FACT -- when called with INIT arg
;   = T -- to be the list, in reverse priority order, of all properties
;   declared to be part of the state of the system.

(DEFPARAMETER LINEL-VALUE 79)

(DEFPARAMETER PROVE-FILE NIL)

;   Theorem-prover output is sent to PROVE-FILE.  Warnings and error messages
;   are also sent to TTY-FILE, if it is different from PROVE-FILE.  PROVE-FILE
;   and TTY-FILE are initially set to NIL, which means that output goes to the
;   default output, which is initially the terminal.

(DEFPARAMETER *RANDOM-SEED* 0)

(DEFPARAMETER *COMPILE-FUNCTIONS-FLG* NIL)

(DEFPARAMETER A-VERY-RARE-CONS (CONS NIL NIL))

;		DEFUNSs FOR THE OPERATING SYSTEM DEPENDENT CODE

(DEFUN ALPHORDER (X Y)
  (COND ((NUMBERP X)
	 (COND ((NUMBERP Y) (<= X Y))
	       (T T)))
	((NUMBERP Y) NIL)
	(T (STRING<= (SYMBOL-NAME X)
		     (SYMBOL-NAME Y)))))

(DEFUN ASSOC-EQ (X Y)
  (ITERATE FOR Z IN Y WHEN (EQ X (CAR Z)) DO (RETURN Z)))

(DEFUN COMPILE-IF-APPROPRIATE-AND-POSSIBLE (FNS)

;   If a function foo is defined in our theory, a function *1*Foo is defined in
;   Lisp.  Sometimes during the course of a proof, *1*Foo may be executed to
;   compute the value of foo on certain values.  There is a speed benefit to
;   compiling *1*Foo.  In Maclisp, the compiler is not in the same Lisp with
;   the theorem-prover; in Zetalisp, the compiler is resident.  The *.LISP
;   files produced by MAKE-LIB may be compiled after loading the compilation of
;   BASIS into the compiler.  Hence it is possible to obtain the speed of
;   compiled functions in the Maclisp version of the theorem-prover, at the
;   expense of making a library, running a separate compilation, and using
;   NOTE-LIB to load the *.LIB file and the compilation of the .LISP file.

  (COND (*COMPILE-FUNCTIONS-FLG*
	 (ITERATE FOR FN IN FNS DO
		  (COND ((NOT (TYPEP (SYMBOL-FUNCTION FN)
				     (QUOTE COMPILED-FUNCTION)))
			 (COMPILE FN)))))))

(DEFUN EXTEND-FILE-NAME (FILE EXTENSION)
  (STRING (PACK (LIST FILE (QUOTE |.|) EXTENSION))))

(DEFUN GET-TAIL (ATM PROP)
  (ITERATE FOR TAIL ON (SYMBOL-PLIST ATM) BY (QUOTE CDDR)
	   WHEN (EQ PROP (CAR TAIL))
	   DO (RETURN TAIL)))

(DEFUN IDATE ()
  (POWER-EVAL
   (ITERATE FOR I FROM 1 TO 6
	    AS J IN (MULTIPLE-VALUE-LIST (GET-DECODED-TIME))
	    COLLECT
	    (COND ((= I 6) (- J 1900))
		  (T J)))
   100))

(DEFUN INTERSECTION-EQ (X Y)
  (ITERATE FOR A IN X WHEN (MEMBER-EQ A Y) COLLECT A))

(DEFUN IPOSITION (FILE N FLG)
  (LET (PAIR)
    (COND ((NULL (SETQ PAIR (ASSOC-EQ FILE IPOSITION-ALIST)))
	   (SETQ IPOSITION-ALIST
		 (CONS (SETQ PAIR (CONS FILE 0)) IPOSITION-ALIST))))
    (COND ((NULL N) (CDR PAIR))
	  (FLG (PROG1 (CDR PAIR) (RPLACD PAIR (+ N (CDR PAIR)))))
	  (T (PROG1 (CDR PAIR) (RPLACD PAIR N))))))

(DEFUN ITERPRI (FILE)
  (IPOSITION FILE 0 NIL)
  (TERPRI FILE))

(DEFUN ITERPRIN (N FILE)
  (ITERATE FOR I FROM 1 TO N DO (ITERPRI FILE)))

(DEFUN ITERPRISPACES (N FILE)
  (ITERPRI FILE)
  (TABULATE N FILE)
  (FORCE-OUTPUT FILE))

(DEFUN IPRINC (X FILE)
  (IPOSITION FILE
	     (OUR-FLATC X)
	     T)
  (PRINC X FILE)
  (FORCE-OUTPUT FILE))

(DEFUN ISPACES (N FILE)
  (COND ((<= N 0) NIL)
	(T (IPOSITION FILE N T)
	   (ITERATE FOR I FROM 1 TO N DO (WRITE-CHAR #\Space FILE))
	   (FORCE-OUTPUT FILE))))

(DEFUN KILL-DEFINITION (ATM)
  (REMPROP ATM (QUOTE SEXPR))
  (FMAKUNBOUND ATM)
  (SETQ LIB-ATOMS-WITH-DEFS (REMOVE ATM LIB-ATOMS-WITH-DEFS)))

(DEFUN OUR-LINEL (FILE &OPTIONAL N) FILE
  (COND ((INTEGERP N) (PROG1 LINEL-VALUE (SETQ LINEL-VALUE N)))
	(T LINEL-VALUE)))

(DEFUN MAKE-LIB (FILE)
  (LET ((*READ-BASE* 10)
	(*PRINT-BASE* 10)
	*PRINT-LEVEL*
	*PRINT-LENGTH*
	(*PRINT-CASE* :UPCASE)	
	TEMP PROP-FILE FN-FILE
	PROP-FILE-NAME
	FN-FILE-NAME
	(REVERSED-LIB-PROPS (REVERSE LIB-PROPS)))
    (CHK-INIT)
    (SETQ PROP-FILE (OPEN (EXTEND-FILE-NAME FILE 'LIB)
			  :DIRECTION :OUTPUT))
    (SETQ PROP-FILE-NAME
	  (NAMESTRING (TRUENAME PROP-FILE)))
    (PRINT (LIST (QUOTE INIT-LIB) (KWOTE LIB-PROPS) (KWOTE LIB-VARS))
	   PROP-FILE)
    (ITERATE FOR VAR IN LIB-VARS
	     DO (PRINT (LIST (QUOTE SETQ) VAR (KWOTE (SYMBOL-VALUE VAR)))
		       PROP-FILE))
    (PRINT (LIST (QUOTE SETQ)
		 (QUOTE LIB-ATOMS-WITH-PROPS)
		 (KWOTE LIB-ATOMS-WITH-PROPS))
	   PROP-FILE)
    (PRINT (LIST (QUOTE SETQ)
		 (QUOTE LIB-ATOMS-WITH-DEFS)
		 (KWOTE LIB-ATOMS-WITH-DEFS))
	   PROP-FILE)
    (ITERATE FOR ATM IN LIB-ATOMS-WITH-PROPS DO
	     (PRINT
	      (LIST (QUOTE PUT1-LST)
		    (KWOTE ATM)
		    (KWOTE
		     (ITERATE FOR PROP IN REVERSED-LIB-PROPS
			      NCONC
			      (COND ((SETQ TEMP (GET-TAIL ATM PROP))
				     (LIST PROP (CADR TEMP)))))))
	      PROP-FILE))
    (ITERATE FOR ATM IN (REVERSE LIB-ATOMS-WITH-DEFS) DO
	     (PRINT (LIST (QUOTE PUT1-LST)
			  (KWOTE ATM)
			  (KWOTE
			   (LIST (QUOTE SEXPR)
				 (LIST (QUOTE LAMBDA)
				       (CADR (SETQ TEMP
						   (GET ATM
							(QUOTE SEXPR))))
				       (CADDR TEMP)))))
		    PROP-FILE))
    (CLOSE PROP-FILE)
    (SETQ FN-FILE (OPEN (EXTEND-FILE-NAME FILE 'LISP)
			:DIRECTION :OUTPUT))
    (SETQ FN-FILE-NAME
	  (NAMESTRING (TRUENAME FN-FILE)))
    (ITERATE FOR ATM IN (REVERSE LIB-ATOMS-WITH-DEFS) DO
	     (PRINT (LIST (QUOTE DEFUN)
			  ATM
			  (CADR (SETQ TEMP (GET ATM (QUOTE SEXPR))))
			  (CADDR TEMP))
		    FN-FILE))
    (PRINT (LIST (QUOTE COMPILE-IF-APPROPRIATE-AND-POSSIBLE)
		 (QUOTE LIB-ATOMS-WITH-DEFS))
	   FN-FILE)
    (CLOSE FN-FILE)
    (LIST PROP-FILE-NAME FN-FILE-NAME)))

(DEFUN NOTE-LIB (FILE1 FILE2)
  (COND (*THM-WARNING-FLG*
	 (IPRINC *THM-WARNING* T)))
  (COND ((BOUNDP (QUOTE LIB-FILE))
	 (KILL-LIB)))
  (LOAD FILE1)
  (LOAD FILE2)
  (LIST FILE1 FILE2))

(DEFUN OK-SYMBOLP (X)

;   From 1974 till 1983 our theorem-prover was implemented in Interlisp-10, in
;   which it is the case that, to use Zetalisp terminology, there is only one
;   package, every symbolp is interned in it, and symbolp's with the same pname
;   are EQ.  In the theory in which our system proves theorems, two LITATOMS
;   are EQUAL if and only if they have the same UNPACKs.  If our theorem-prover
;   admitted, say, CHAOS:FOO and USER:FOO as LITATOMS, then it would have to
;   conclude that they are EQUAL because their UNPACKs are the same given our
;   representation of evgs.  However, our theorem-prover also is based upon the
;   assumption that two evg's are not EQUAL in the theory if they are not EQUAL
;   LISP objects.  Therefore, we have to reject some litatoms, such as
;   CHAOS:FOO.  The rejection occurs in TRANSLATE, where we check that every
;   SYMBOLP is OK-SYMBOLP.

;   The following definition of OK-SYMBOLP is not sufficient to avoid the
;   problems mentioned above if, in MACLISP, OBARRAY does not have its initial
;   value, or if someone has mangled the contents of the OBARRAY, or if someone
;   has engaged in REMOB's.  The following definition is not sufficient to
;   avoid the problems in Zetalisp if someone has set PACKAGE to something
;   other than USER, has engaged in malicious INTERNing, has smashed strings
;   of SYMBOLPs, etc., etc., etc.

  (MULTIPLE-VALUE-BIND
   (SYMBOL INTERNEDP)
   (FIND-SYMBOL (SYMBOL-NAME X) 'USER)
   (AND INTERNEDP (EQ X SYMBOL))))

;  PACK returns a symbol interned in USER whose print name
;  consists of the characters that are printed by PRINCing
;  the members of L with *PRINT-BASE* 10.

(DEFUN PACK (L)
  (LET ((*PRINT-BASE* 10)
	(*PRINT-CASE* :UPCASE))
    (INTERN (FORMAT NIL "~{~A~}" L)
	    'USER)))

(DEFUN PRINT-DATE-LINE NIL
  (PRINT-IDATE (IDATE) PROVE-FILE))

(DEFUN READ-FILE (FILE-NAME)
  (WITH-OPEN-FILE (FILE FILE-NAME :DIRECTION :INPUT)
    (ITERATE WITH TEMP
	     WHILE (PROGN (SETQ TEMP (READ FILE NIL A-VERY-RARE-CONS))
			  (NOT (EQ A-VERY-RARE-CONS TEMP)))
	     COLLECT TEMP)))

(DEFUN STORE-DEFINITION (ATM EXPR)
  (SETF (GET ATM (QUOTE SEXPR)) EXPR)
;(SETF (SYMBOL-FUNCTION ATM) EXPR) ;what we used to do.
  (EVAL (LIST* 'DEFUN ATM (CDR EXPR))) ;what we do to overcome a bug in KCL ??
  (COMPILE-IF-APPROPRIATE-AND-POSSIBLE (LIST ATM)))

(DEFUN TIME-IN-60THS ()
  (FLOOR (* 60 (/ (GET-INTERNAL-RUN-TIME)
		  INTERNAL-TIME-UNITS-PER-SECOND))))



;   INTERLISP's SORT is apparently nonstable and frequently (perhaps always)
;   reverses elements of equal weight.  Zetalisp sort is stable.  We found
;   three occasions in the rsa and wilson proofs when this difference bit us
;   and caused a different elimination to be chosen first.  The first two times
;   we fixed it by letting it do the new elim and just seeing that the
;   appropriate lemmas were available to handle the new goals.  But on the
;   third time we decided simply to REVERSE the input list to mimic INTERLISP's
;   sort, just so we could get on with reproducing the old proofs on the new
;   machine.  Franz' sort is unstable, so to guarantee the same sorting
;   on all machines we adopt an unstable sort and use it everywhere.

(DEFUN UNSTABLE-SORT (X Y) 
  (OUR-STABLE-SORT (REVERSE X) Y))

(DEFUN OUR-STABLE-SORT (L FN)
  (COND ((OR (ATOM L)
	     (ATOM (CDR L)))
	 L)
	(T (OUR-MERGE (OUR-STABLE-SORT (ODDS L) FN)
		      (OUR-STABLE-SORT (ODDS (CDR L)) FN)
		      FN))))

(DEFUN OUR-MERGE (L M FN)
  (COND ((ATOM L) M)
	((ATOM M) L)
	((OR (FUNCALL FN (CAR L) (CAR M))
	     (NOT (FUNCALL FN (CAR M) (CAR L))))
	 (CONS (CAR L) (OUR-MERGE (CDR L) M FN)))
	(T (CONS (CAR M) (OUR-MERGE L (CDR M) FN)))))

(DEFUN UNION-EQ (X Y)
  (NCONC (ITERATE FOR X1 IN X UNLESS (MEMBER-EQ X1 Y) COLLECT X1) Y))

;   End of the operating system dependent code.


;			     VARIABLE DECLARATIONS

(DEFVAR *ALIST*)

(DEFVAR *ARGLIST*)

(DEFVAR *CONTROLLER-COMPLEXITIES*)

(DEFVAR *FILE*)

(DEFVAR *FNNAME*)

(DEFVAR *INDENT*)

(DEFVAR *TYPE-ALIST*)

;   Used by REWRITE-FNCALL to store the type-alist so that the lower level
;   jumpout can reach up and get it should it need to solidify before jumping.

;   The variables *1*BTM-OBJECTS, *1*F, *1*SHELL-QUOTE-MARK, and *1*T have
;   names that start with *1* because the built-in functions, such as
;   *1*COUNT, which could be called by some user's function *1*FOO, refer
;   to these variables.

(DEFVAR *1*BTM-OBJECTS)

;   This is just a list of all the bottom object function symbols.

(DEFCONSTANT *1*F (QUOTE *1*FALSE))

;   The explicit form of the term (FALSE).

(DEFCONSTANT *1*SHELL-QUOTE-MARK (QUOTE *1*QUOTE))

;   The mark that a constructor expression in an explicit object is represented
;   in the CADR.

(DEFCONSTANT *1*T (QUOTE *1*TRUE))

;   The explicit form of the term (TRUE).

(DEFVAR ABBREVIATIONS-USED)

;   Used by EXPAND-ABBREVIATIONS and CLAUSIFY-INPUT to record which
;   abbreviations were applied.

(DEFVAR ADD-EQUATIONS-TO-DO)

;   The second answer returned by ADD-EQUATION consisting of new equations yet
;   to be added to the pot.

(DEFPARAMETER ADD-TERM-TO-POT-LST-TEMP (LIST NIL))

;   Used by ADD-TERM-TO-POT-LST.

(DEFVAR ALIST)

(DEFVAR ALISTS)

(DEFVAR ALL-FNS-FLG)

(DEFPARAMETER ALL-LEMMAS-USED NIL)

(DEFPARAMETER ALMOST-SUBSUMES-CONSTANT (CONS NIL NIL))

;   A special constant used by ALMOST-SUBSUMES.  ALMOST-SUBSUMES detects when
;   CL1 is almost subsumed by CL2, which means that either CL1 is subsumed by
;   CL2 or else it is except for one literal whose negation is in CL2.
;   ALMOST-SUBSUMES sets ALMOST-SUBSUMES-LITERAL to this literal.  When that
;   variable is set to this constant it is interpreted as the message that all
;   the literals were subsumed.

(DEFVAR ALMOST-SUBSUMES-LITERAL)

;   Used as an extra answer by ALMOST-SUBSUMES.

(DEFPARAMETER ANCESTORS NIL)

;   List of the negations of the hypotheses REWRITE is currently trying to
;   establish in order to apply lemmas.  This list is used by RELIEVE-HYPS and
;   RELIEVE-HYPS-NOT-OK to prevent infinite backwards chaining.  The list is
;   supposed to be bound whenever a new entry is added.  Like TYPE-ALIST it is
;   a free var only for convenience.

(DEFVAR ANS)

;   Used frequently in FOO, FOO1 constructions in which FOO initializes a
;   collection site and FOO1 repeatedly adds to it.

(DEFVAR ARGS)

(DEFPARAMETER ARITY-ALIST NIL)

;   This is a list associating function names with their arities.  Once a
;   function has been defined or declared its arity is stored on its property
;   list encoded as the length of the CDR of its TYPE-PRESCRIPTION.  But it is
;   necessary to know the proposed arity of a function even before it has been
;   accepted as a legal function and its property list is set up.  Thus, this
;   list is used, by DEFN0, BOOT-STRAP, and ADD-SHELL0 to declare the arities
;   of certain functions during the act of creating them.

(DEFPARAMETER BOOK-SYNTAX-FLG NIL)

;   If T this variable causes REDO-UNDONE-EVENTS to print out the events in the
;   event list in the syntax used in the "A Computational Logic," rather than
;   as LISP forms.

(DEFPARAMETER BOOT-STRAP-INSTRS
  (QUOTE ((ADD-SHELL0 FALSE NIL FALSEP NIL)
	  (ADD-SHELL0 TRUE NIL TRUEP NIL)
	  (DEFN0 NOT (P)
	    (IF P (FALSE) (TRUE))
	    NIL T)
	  (DEFN0 AND (P Q)
	    (IF P (IF Q (TRUE) (FALSE)) (FALSE))
	    NIL T)
	  (DEFN0 OR (P Q)
	    (IF P (TRUE) (IF Q (TRUE) (FALSE)))
	    NIL T)
	  (DEFN0 IMPLIES (P Q)
	    (IF P (IF Q (TRUE) (FALSE))
		(TRUE))
	    NIL T)
	  (ADD-SHELL0 ADD1 ZERO NUMBERP ((SUB1 (ONE-OF NUMBERP) ZERO)))
	  (DEFN0 LESSP (X Y)
	    (IF (OR (EQUAL Y 0) (NOT (NUMBERP Y)))
		(FALSE)
		(IF (OR (EQUAL X 0) (NOT (NUMBERP X)))
		    (TRUE)
		    (LESSP (SUB1 X) (SUB1 Y))))
	    NIL T)
	  (PUT1 LESSP 0 LEVEL-NO)
	  (DEFN0 GREATERP (X Y) (LESSP Y X) NIL NIL)
	  (DEFN0 LEQ (X Y) (NOT (LESSP Y X)) NIL NIL)
	  (DEFN0 GEQ (X Y) (NOT (LESSP X Y)) NIL NIL)
	  (DEFN0 LEX2 (L1 L2)
	    (OR (LESSP (CAR L1) (CAR L2))
		(AND (EQUAL (CAR L1) (CAR L2))
		     (LESSP (CADR L1) (CADR L2))))
	    NIL NIL)
	  (DEFN0 LEX3 (L1 L2)
	    (OR (LESSP (CAR L1) (CAR L2))
		(AND (EQUAL (CAR L1) (CAR L2))
		     (LEX2 (CDR L1) (CDR L2))))
	    NIL NIL)
	  (DEFN0 ZEROP (X)
	    (IF (EQUAL X 0)
		T
		(IF (NUMBERP X) F T))
	    NIL T)
	  (DEFN0 FIX (X)
	    (IF (NUMBERP X) X 0)
	    NIL T)
	  (DEFN0 PLUS (X Y)
	    (IF (ZEROP X)
		(FIX Y)
		(ADD1 (PLUS (SUB1 X) Y)))
	    NIL T)
	  (ADD-AXIOM1 COUNT-NUMBERP (REWRITE)
		      (IMPLIES (NUMBERP I)
			       (EQUAL (COUNT I) I)))
	  (ADD-AXIOM1 COUNT-NOT-LESSP (REWRITE)
		      (NOT (LESSP (COUNT I) I)))
	  (ADD-SHELL0 PACK NIL LITATOM ((UNPACK (NONE-OF) ZERO)))
	  (ADD-SHELL0 CONS NIL LISTP ((CAR (NONE-OF) ZERO)
				      (CDR (NONE-OF) ZERO)))
	  (DEFN0 NLISTP (X)
	    (NOT (LISTP X))
	    NIL T)
	  (ADD-SHELL0 MINUS NIL NEGATIVEP
		      ((NEGATIVE-GUTS (ONE-OF NUMBERP)
				      ZERO)))
	  (DEFN0 DIFFERENCE (I J)
	    (IF (ZEROP I)
		0
		(IF (ZEROP J)
		    I
		    (DIFFERENCE (SUB1 I) (SUB1 J))))
	    NIL T)
	  (DEFN0 TIMES (I J)
	    (IF (ZEROP I)
		0
		(PLUS J (TIMES (SUB1 I) J)))
	    NIL T)
	  (DEFN0 QUOTIENT (I J)
	    (IF (ZEROP J)
		0
		(IF (LESSP I J)
		    0
		    (ADD1 (QUOTIENT (DIFFERENCE I J) J))))
	    NIL T)
	  (DEFN0 REMAINDER (I J)
	    (IF (ZEROP J)
		(FIX I)
		(IF (LESSP I J)
		    (FIX I)
		    (REMAINDER (DIFFERENCE I J) J)))
	    NIL T)
	  (DEFN0 LEGAL-CHAR-CODES NIL
	    '(45 48 49 50 51 52 53 54 55 56 57 65 66 67 68 69 70 71
		 72 73 74 75 76 77 78
		 79 80 81 82 83 84 85 86 87 88 89 90)
	    NIL NIL)
	  (DEFN0 ILLEGAL-FIRST-CHAR-CODES NIL
	    '(45 48 49 50 51 52 53 54 55 56 57)
	    NIL NIL)
	  (DEFN0 LENGTH (LST)
	    (IF (LISTP LST)
		(ADD1 (LENGTH (CDR LST)))
		0)
	    NIL NIL)
	  (DEFN0 MEMBER (X LST)
	    (IF (NLISTP LST)
		F
		(IF (EQUAL X (CAR LST))
		    T
		    (MEMBER X (CDR LST))))
	    NIL NIL)
	  (DEFN0 SUBSETP (X Y)
	    (IF (NLISTP X)
		T
		(IF (MEMBER (CAR X) Y)
		    (SUBSETP (CDR X) Y)
		    F))
	    NIL NIL)
	  (DEFN0 LAST (L)
	    (IF (LISTP L)
		(IF (LISTP (CDR L))
		    (LAST (CDR L))
		    L)
		L)
	    NIL NIL)
	  (DEFN0 LEGAL-CHAR-CODE-SEQ (LST)
	    (AND (LISTP LST)
		 (SUBSETP LST (LEGAL-CHAR-CODES))
		 (NOT (MEMBER (CAR LST) (ILLEGAL-FIRST-CHAR-CODES)))
		 (EQUAL (CDR (LAST LST))
			0))
	    NIL NIL)
	  (DEFN0 SYMBOLP (X)
	    (AND (LITATOM X)
		 (LEGAL-CHAR-CODE-SEQ (UNPACK X)))
	    NIL NIL)
	  (DEFN0 LOOKUP (X ALIST)
	    (IF (NLISTP ALIST)
		NIL
		(IF (AND (LISTP (CAR ALIST))
			 (EQUAL X (CAAR ALIST)))
		    (CDAR ALIST)
		    (LOOKUP X (CDR ALIST))))
	    NIL NIL)
	  (DCL0 ARITY (X))
	  (DCL0 FORMP (X))
	  (DEFN0 FORM-LSTP (X)
	    (IF (NLISTP X)
		(EQUAL X NIL)
		(AND (FORMP (CAR X))
		     (FORM-LSTP (CDR X))))
	    NIL NIL)
	  (DCL0 APPLY (X LST))
	  (DCL0 MEANING (X ALIST))
	  (DEFN0 MEANING-LST (X ALIST)
	    (IF (NLISTP X)
		NIL
		(CONS (MEANING (CAR X) ALIST)
		      (MEANING-LST (CDR X) ALIST)))
	    NIL NIL)
	  (SETUP-META-NAMES))))

;   This is the list of instructions executed by BOOT-STRAP that initialize the
;   theorem prover.  A user could not execute this sequence of commands because
;   they necessarily violate our new name conventions.  For example, axioms
;   about SUB1 mention LESSP before LESSP is defined.  But LESSP can't be
;   defined first because it recurses by SUB1 and before SUB1 were fully
;   axiomatized LESSP wouldn't be accepted as a total function.  Most of the
;   names introduced here are built-into the theorem prover in some sense.  For
;   example, LESSP is on the list of known well-founded relations, TRUE is
;   referenced by name in many routines, the axioms produced by ADD-SHELL use
;   IMPLIES and the other logical connectives, as well as LESSP and PLUS, and
;   the read-time translation routine uses PACK and CONS to translate QUOTE and
;   LIST expressions.  However, most of the logical properties of these names
;   are made known to the theorem prover in the usual way.  For example, TRUE,
;   FALSE, and ADD1 are just shells; the logical connectives are just defined
;   functions.  DIFFERENCE, TIMES, QUOTIENT and REMAINDER are in this list so
;   that efficient *1*functions can be provided for them.

(DEFPARAMETER BOOT-STRAP-MACRO-FNS (QUOTE (GREATERP LEQ GEQ)))

;   Must be a list of function names defined nonrecursively in
;   BOOT-STRAP-INSTRS.  Expanded away in translate.

(DEFPARAMETER BROKEN-LEMMAS NIL)

(DEFVAR CHRONOLOGY)

;   The list of all event names, in reverse chronological order.

(DEFVAR CL2)

(DEFPARAMETER CLAUSE-ALIST NIL)

(DEFVAR COMMONSUBTERMS)

(DEFPARAMETER COMMUTED-EQUALITY-FLG NIL)

(DEFPARAMETER CULPRIT-FUNCTION NIL)

(DEFPARAMETER CURRENT-ATM 0)

;   Atom of CURRENT-LIT, set by SIMPLIFY-CLAUSE1 and used by
;   ADD-TERM-TO-POT-LST to avoid trying to use linear to add the negation of
;   CURRENT-LIT to the pot lst when we know we have already tried it.

(DEFVAR CURRENT-CL)

;   The clause whose subterms control which functions get opened up.

(DEFPARAMETER CURRENT-LIT 0)

;   During SIMPLIFY-CLAUSE1, CURRENT-LIT is the literal we are currently trying
;   to rewrite.  ADD-EQUATIONS avoids using any POLY that descends from this
;   literal and REWRITE-SOLIDIFY avoids using this literal.  Outside of
;   SIMPLIFY-CLAUSE1, this variable should not be a term.

(DEFVAR CURRENT-SIMPLIFY-CL)

(DEFVAR CURRENT-TYPE-NO)

;   Bound in ADD-SHELL0 for using during the creation of the axioms for a
;   shell.

(DEFVAR DECISIONS)

(DEFVAR DEFINED-FUNCTIONS-TOGGLED)

;   A list of pairs of the form (name . flg) where flg is T or NIL.  The
;   first flg on the list is the current state of the DEFINED-FUNCTIONS-TOGGLED
;   flag.  When the flag is T, then all defined *1*functions are disabled.

(DEFVAR DEFINITELY-FALSE)

;   If FALSE-NONFALSEP returns T then DEFINITELY-FALSE is T if the term is
;   equal to FALSE and is NIL if the term is not equal to FALSE.

(DEFVAR DEFN-FLG)

;   Used by REWRITE to keep track of whether the current term's value will
;   ultimately become either T, F, or -- most importantly -- part of the
;   expanded body of a recursive function.  If DEFN-FLG is on and the value of
;   the term violates REWRITE-FNCALLP, the recursive rewriting of the body can
;   be aborted in JUMPOUTP.

(DEFVAR DESCENDANTS)

(DEFVAR DISABLED-LEMMAS)

;   This list contains the names of those lemmas that are currently disabled.
;   No routine that uses a lemma will consider a lemma whose name is on this
;   list.

(DEFVAR DLHDFMLA)

(DEFPARAMETER DO-NOT-GENERALIZE-FLG NIL)

(DEFPARAMETER DO-NOT-USE-INDUCTION-FLG NIL)

;   If set to T then PROVE aborts with failure as soon as a clause has to be
;   pushed for induction.

(DEFPARAMETER DOTCONS (LIST NIL NIL))

(DEFVAR ELAPSED-TIME)

(DEFPARAMETER ELIM-VARIABLE-NAMES
  (QUOTE (X Z V W D C X1 Z1 V1 W1 D1 C1 X2 Z2 V2 W2 D2 C2)))

;   This is the list of variables that can be introduced by the elimination of
;   destructors -- provided they do not occur in the conjecture being
;   generalized.  It is important for us to be able to tell how a variable was
;   introduced in the theorem.  For example, such a question is asked to
;   control repeated destructor elimination which might otherwise loop.  We use
;   the "pretty" variables on this list rather than just GENSYMimg unique names
;   because we do not like to see funny variable names in our output.  

;   IT IS CRUCIAL THAT THIS LIST
;   HAVE NO INTERSECTION
;   WITH GEN-VARIABLE-NAMES -- the
;   list used to pick vars for GENERALIZE.

(DEFPARAMETER ELIM-VARIABLE-NAMES1 NIL)

(DEFVAR ENDLIST)

(DEFPARAMETER ERROR1-SET-FLG NIL)

(DEFVAR EVENT-LST)

(DEFPARAMETER EXECUTE-PROCESSES
  (QUOTE (SIMPLIFY-CLAUSE SETTLED-DOWN-CLAUSE FERTILIZE-CLAUSE
			  ELIMINATE-DESTRUCTORS-CLAUSE GENERALIZE-CLAUSE
			  ELIMINATE-IRRELEVANCE-CLAUSE STORE-SENT)))

;   This list is just used by the IO1 function to control such things as to
;   whether to indent and print the current goal before printing the process
;   specific information.

(DEFPARAMETER EXPAND-LST NIL)

(DEFPARAMETER FAILED-EVENTS NIL)

;   List of all of the events given to REDO-UNDONE-EVENTS which returned NIL in
;   a given session with the system.

(DEFVAR FAILURE-ACTION)

(DEFPARAMETER FAILURE-MSG
  (QUOTE |************** F  A  I  L  E  D **************|))

;   This is the value that is returned by PROVE should a proof fail.  It is a
;   distinctive message that is guaranteed to catch our eyes if it is ever
;   returned as the value of PROVE.  Since we often run with the I/O going to a
;   file and just the value of PROVE being printed to the terminal, we like for
;   failures to be brought to our attention.

(DEFPARAMETER FALSE (QUOTE (QUOTE *1*FALSE)))

;   This variable is just bound to the internal form of the term (FALSE) for
;   convenient coding.

(DEFVAR FALSE-COMPOUND-RECOGNIZER-ALIST)

;   See TRUE-COMPOUND-RECOGNIZER-ALIST.

(DEFVAR FALSE-TYPE-ALIST)

(DEFVAR FILE)

(DEFVAR FLATSIZE)

(DEFVAR FMLA)

(DEFVAR FNS)

(DEFVAR FNSTACK)

(DEFPARAMETER FNS-TO-BE-IGNORED-BY-REWRITE NIL)

;   Terms beginning with function names on this list are not touched by
;   rewrite or expand abbreviations.  However, their arguments will have
;   already been worked on.

(DEFPARAMETER FORCEIN 38)

(DEFVAR FORM)

(DEFPARAMETER GEN-VARIABLE-NAMES (QUOTE (Y A U B E G H P Q R S)))

;   List of variables that can be introduced by generalize.  See
;   ELIM-VARIABLE-NAMES.

(DEFVAR GEN-VARIABLE-NAMES1)

;   Those GEN-VARIABLE-NAMES not occurring in theorem being proved.

(DEFPARAMETER GENERALIZE-LEMMA-NAMES NIL)

(DEFVAR GENERALIZE-LEMMAS)

;   The list of GENERALIZE-LEMMA records representing all known GENERALIZE
;   lemmas.

(DEFPARAMETER GENERALIZING-SKOS NIL)

(DEFVAR GENRLTLIST)

(DEFPARAMETER HEURISTIC-TYPE-ALIST NIL)

;   This type alist is used under ADD-TERMS-TO-POT-LST to determine
;   heuristically which terms are numeric and should be linearized.  Soundness
;   and tail biting are not affected by its setting.

(DEFVAR HIGHER-PROPS)

;   The list of properties with higher priority.  Used while PUT00 is recursing
;   to find out where to put PROP down.

(DEFPARAMETER HINT NIL)

;   If nonNIL, then PROVE goes straight into induction.

(DEFVAR HINTS)

;   STORE-SENT looks at the HINTS argument of PROVE-LEMMA when deciding whether
;   an induction has been done.

(DEFPARAMETER HINT-VARIABLE-ALIST
  (QUOTE ((DISABLE TEMPORARILY-DISABLED-LEMMAS NIL NIL)
          (ENABLE TEMPORARILY-ENABLED-LEMMAS NIL NIL)
          (EXPAND HINTED-EXPANSIONS T NIL)
          (DO-NOT-INDUCT DO-NOT-USE-INDUCTION-FLG NIL NIL)
          (DO-NOT-GENERALIZE DO-NOT-GENERALIZE-FLG NIL NIL)
          (HANDS-OFF FNS-TO-BE-IGNORED-BY-REWRITE NIL NIL)
          (NO-BUILT-IN-ARITH NO-BUILT-IN-ARITH-FLG NIL NIL))))

;   This is a list of 4-tuples interpreted by APPLY-HINTS.  Each element of the
;   list is of the form:

;   (visible-name internal-variable-name translate-flg default-value).

;   Whenever there is a hint whose CAR is one of the visible-names, the
;   corresponding internal-variable-name is set by APPLY-HINTS to the CDR of
;   the hint.  If the translate-flg is T, the CDR of the hint is taken as a
;   list of terms and each element of it is TRANSLATEd and the internal
;   variable is set to the resulting list.  It is assumed that at the top level
;   of the system we have arranged -- e.g., by an appropriate DEFVAR -- for
;   each of the internal variables to have the value given by the
;   default-value.  The UNWIND-PROTECT in PROVE-LEMMA, in which hints are
;   processed, makes sure the internal variables are restored to their default
;   values after the PROVE-LEMMA has terminated.  Thus, if a variable is
;   DEFVARd to the default value in BASIS and is never set in our code then its
;   value is always the default value except when you are executing under a
;   PROVE-LEMMA containing a hint with the corresponding visible-name.  That
;   is, you can regard those variables as having been bound by PROVE-LEMMA to
;   the user specified values or to their global default values otherwise.

(DEFPARAMETER HINTED-EXPANSIONS NIL)

;   Used to pass information from APPLY-HINTS to SETUP regarding which terms
;   the user wants expanded.

(DEFVAR HIST-ENTRY)

(DEFVAR ID-IFF)

(DEFVAR INDENT)

(DEFVAR INDUCTION-CONCL-TERMS)

(DEFPARAMETER INDUCTION-HYP-TERMS NIL)

(DEFVAR INST-HYP)

(DEFPARAMETER IN-ADD-AXIOM-FLG NIL)

(DEFPARAMETER IN-BOOT-STRAP-FLG NIL)

(DEFPARAMETER IN-REDO-UNDONE-EVENTS-FLG NIL)

(DEFPARAMETER IN-PROVE-LEMMA-FLG NIL)

(DEFPARAMETER IO-FN (QUOTE IO1))

;   The name of the function called by IO to do the printing during a proof.
;   IO is called from several of the theorem proving routines.  By redefining
;   IO-FN we can alter the amount of IO1 we see.  We usually set IO-FN to
;   either IO1 or NO-OP.

(DEFPARAMETER IO-TIME 0)

;   Used to sum up the total amount of time spent in IO during PROVE.

(DEFVAR LAST-CLAUSE)

(DEFVAR LAST-EXIT)

;   When RELIEVE-HYPS1 fails, the value of LAST-EXIT encodes the reason we
;   failed.

(DEFVAR LAST-HYP)

;   When RELIEVE-HYPS1 fails, LAST-HYP is set to the HYP that was not relieved.

(DEFVAR LAST-PRIN5-WORD)

(DEFPARAMETER LAST-PRINEVAL-CHAR (QUOTE |.|))

;   Supposedly this is the last character printed under PRINEVAL, but actually
;   it is only accurate when the last character was a punctuation mark.
;   Otherwise, it is some arbitrary non-punctuation character.

(DEFVAR LAST-PRINT-CLAUSES)

(DEFPARAMETER LAST-PROCESS NIL)

(DEFPARAMETER LINEARIZE-ASSUMPTIONS-STACK NIL)

(DEFPARAMETER LEFTMARGINCHAR NIL)

;   This is the character that IO1 and PPR will print along the left margin of
;   the proof tree.

(DEFPARAMETER LEMMA-DISPLAY-FLG NIL)

(DEFPARAMETER LEMMA-TYPES (QUOTE (REWRITE ELIM GENERALIZE META)))

;   For each lemma type x there must be a function CHK-ACCEPTABLE-x-LEMMA and
;   ADD-x-LEMMA.

(DEFPARAMETER LEMMA-STACK NIL)

(DEFVAR LEMMAS-USED-BY-LINEAR)

;   When ADD-TERMS-TO-POT-LST returns CONTRADICTION this list contains the
;   names of the lemmas used.

(DEFVAR LINEAR-ASSUMPTIONS)

;   When ADD-TERMS-TO-POT-LST returns CONTRADICTION this list contains the
;   terms assumed true during the linearization.

(DEFPARAMETER LITATOM-FORM-COUNT-ALIST NIL)

(DEFPARAMETER LITS-THAT-MAY-BE-ASSUMED-FALSE NIL)

;   During the construction of the POT-LST in SIMPLIFY-CLAUSE0, we wish to have
;   available as hypotheses the negations of the literals in the clause we are
;   trying to prove.  This variable contains those lits during that
;   construction.  As lemmas, those lits get stored in POLYs.  Then, during
;   SIMPLIFY-CLAUSE1's use of linear, we are careful, in ADD-EQUATIONS, not to
;   use any POLY that descends from the CURRENT-LIT, by checking that
;   CURRENT-LIT is not a MEMBer of the lemmas used to produce the POLY.  During
;   all calls of REWRITE except those under the construction of the POT-LST in
;   SIMPLIFY-CLAUSE0, this variable should be NIL.

(DEFPARAMETER LITS-TO-BE-IGNORED-BY-LINEAR NIL)

;   Polys descending from the lits in this list are ignored by the linear
;   package.

(DEFVAR MAIN-EVENT-NAME)

;   All the undo information saved by ADD-FACT is saved on the property list of
;   this atom.  Thus, one of the main acts of creating an event is to bind this
;   name to the event name, so that subsequent ADD-FACTs know who is
;   responsible.

(DEFVAR MARG2)

;   The PPR variable specifying the righthand margin.

(DEFVAR MASTER-ROOT-NAME)

;   Root name for all files created automatically by the theorem prover.  The
;   function MASTER-ROOT-NAME uses this variable as its value and sets it, when
;   NIL, to the user's name.

(DEFVAR MATCH-TEMP)

;   Smashed freely in our pattern matcher.

(DEFVAR MATCH-X)

(DEFPARAMETER META-NAMES
  (QUOTE (APPLY MEANING MEANING-LST ARITY FORMP FORM-LSTP)))

;   These are the names that must not appear in definitions and axioms and that
;   the MEANING and FORMP simplifiers do not know how to handle.

(DEFVAR MINREM)

(DEFPARAMETER MUST-BE-FALSE NIL)

(DEFPARAMETER MUST-BE-TRUE NIL)

(DEFVAR NAME)

(DEFVAR NAMES)

(DEFVAR NEXT-MEMO-KEY)

(DEFVAR NEXT-MEMO-VAL)

(DEFVAR NEXTIND)

(DEFVAR NEXTNODE)

(DEFPARAMETER NILCONS (CONS NIL NIL))

(DEFPARAMETER NO-BUILT-IN-ARITH-FLG NIL)

;   If T the arithmetic package and the storage of arithmetic lemmas is
;   disabled.

(DEFVAR NONCONSTRUCTIVE-AXIOM-NAMES)

;   The names of all of the axioms added with ADD-AXIOM.  We can accept the
;   proof of correctness of a metafunction only when this list is empty.

(DEFVAR NUMBER-OF-VARIABLES)

(DEFVAR OBJECTIVE)

(DEFPARAMETER OBVIOUS-RESTRICTIONS NIL)

(DEFPARAMETER ORIG-LEMMA-STACK NIL)

(DEFPARAMETER ORIG-LINEARIZE-ASSUMPTIONS-STACK NIL)

(DEFPARAMETER ORIGEVENT NIL)

(DEFVAR ORIGTHM)

(DEFCONSTANT PARAGRAPH-INDENT 5)

;   The number of spaces put out by PRIN5 when it sees the # token.

(DEFVAR PARENT)

(DEFVAR PARENT-HIST)

(DEFVAR POS)

(DEFPARAMETER PPR-MACRO-LST
  (QUOTE ((NOT . CONVERT-NOT) (CONS . CONVERT-CONS)
	  (CAR . CONVERT-CAR-CDR) (CDR . CONVERT-CAR-CDR)
	  (QUOTE . CONVERT-QUOTE))))

;   Alist used by PPR to convert terms to their abbreviations for output.

(DEFVAR PPR-MACRO-MEMO)

(DEFVAR PPRFILE)

(DEFPARAMETER PPRFIRSTCOL 35)

(DEFPARAMETER PPRMAXLNS 10000)

(DEFPARAMETER PRINEVAL-FNS
  (QUOTE (= AND EQUAL OR NOT EQ EQLENGTH !CLAUSE !CLAUSE-SET !PPR
	    !PPR1 LENGTH LENGTH-TO-ATOM
	    !PPR-LIST !LIST PLURALP QUOTE QUOTE PQUOTE CAR CDR
	    FN-SYMB FFN-SYMB ARGN FARGN ARGS FARGS QUOTEP FQUOTEP)))

;   This is the list of LISP functions that PRINEVAL -- actually PEVAL -- can
;   evaluate.  To add a function to the list you must make sure that the
;   function mentions no specvars other than those already declared, and that
;   the function is essentially a LAMBDA -- rather than an NLAMBDA -- in that
;   all of its args get evald.

(DEFVAR PROCESS)

(DEFVAR PROCESS-CLAUSES)

(DEFVAR PROCESS-HIST)

(DEFVAR PROP)

(DEFVAR PROPLIST)

(DEFVAR PROVE-TERMINATION-LEMMAS-USED)

(DEFVAR PROVE-TIME)

(DEFPARAMETER R-ALIST NIL)

(DEFVAR RECOGNIZER-ALIST)

;   An alist that associates with each shell recognizer, e.g., NUMBERP, the
;   type set recognized by that predicate, e.g., the bit mask representing the
;   set whose only element is the type class for NUMBERs.  Obviously, the list
;   is used to help determine if recognizer expressions can be simplified to
;   TRUE (when the type set of the argument is the type set recognized) or
;   FALSE (when the type set of the argument does not intersect with the one
;   recognized).

(DEFVAR RECORD-DECLARATIONS)

(DEFVAR RECORD-TEMP)

;   Smashed repeatedly during our record package manipulations.

(DEFVAR RELIEVE-HYPS-NOT-OK-ANS)

(DEFVAR REMAINDER)

(DEFVAR SCRIBE-FLG)

(DEFVAR SETQ-LST)

(DEFVAR SHELL-ALIST)

;   An alist associating each shell name, e.g., CONS, with its type number.
;   The type set of the shell -- that is, the set containing only that type of
;   object -- is just the bit mask with one 1 in it, at the bit whose position
;   is the class's type no.  Thus, the type number for FALSE is 0, TRUE 1, etc.
;   The alist is used when determining the type of an expression beginning with
;   the shell name.

(DEFVAR SHELL-POCKETS)

;   A list of pockets consisting of a shell name and the list of its
;   destructors, with the shell name in the car.

(DEFVAR SIMPLIFY-CLAUSE-MAXIMALLY-CLAUSES)

(DEFVAR SIMPLIFY-CLAUSE-MAXIMALLY-HIST)

(DEFVAR SIMPLIFY-CLAUSE-POT-LST)

;   The ADD-EQUATIONS-WITH-LEMMAS of the top-level clause in SIMPLIFY-CLAUSE.

(DEFVAR SINGLETON-TYPE-SETS)

;   A list of type sets of shells with no components.  If a shell has no
;   components then an expression beginning with the shell name represents a
;   unique constant, e.g., (TRUE) or (FALSE) or other shells the user might
;   introduce such as (ERROR).  If an expression is known to have as its type
;   set one of the singleton type sets, the expression is known to be equal to
;   the corresponding object.

(DEFVAR SPACELEFT)

(DEFPARAMETER STACK NIL)

(DEFVAR STARTLIST)

(DEFCONSTANT STRING-WEIRD (QUOTE |*1*|))

(DEFCONSTANT STRING-WEIRD2 (QUOTE |*2*|))

(DEFCONSTANT STRING-WEIRD3 (QUOTE |*3*|))

(DEFVAR T2)

(DEFVAR TEMP-TEMP)

;   Used freely by anyone, but liable to be set by a call of any function.

(DEFVAR TEMP1)

(DEFPARAMETER TEMPORARILY-DISABLED-LEMMAS NIL)

(DEFPARAMETER TEMPORARILY-ENABLED-LEMMAS NIL)

(DEFPARAMETER TERMS-TO-BE-IGNORED-BY-REWRITE NIL)

(DEFVAR TEST-LST)

(DEFVAR THM)

(DEFVAR TOTAL-MISC-TIME)

;   These three variables are used to accumulate the times consumed by
;   the individual events executed by REDO-UNDONE-EVENTS.  The vars
;   are initialized there and eventually printed there and are incremented
;   by STOP-STATS.

(DEFVAR TOTAL-PROVE-TIME)

(DEFVAR TOTAL-IO-TIME)

(DEFPARAMETER TRANSLATE-TO-LISP-TIME 0)

;   Incremented by TRANSLATE-TO-LISP to record the amount of time spent in
;   optimizing user defined functions.

(DEFCONSTANT TREE-INDENT 2)

;   The number of spaces IO1 indents when printing out the tree structure of
;   the proof.

(DEFCONSTANT TREE-LINES 2)

;   The number of lines IO1 skips between each node of the tree structure in a
;   proof.

(DEFPARAMETER TRUE (QUOTE (QUOTE *1*TRUE)))

;   A variable bound to internal form of the term (TRUE) to make coding more
;   convenient.

(DEFPARAMETER TRUE-CLAUSE (LIST TRUE))

;   The clause whose only literal is the (TRUE) literal.  This just makes
;   coding more convenient.

(DEFVAR TRUE-COMPOUND-RECOGNIZER-ALIST)

;   A list of triples of the form (fn type-set . name) encoding the fact
;   that the event named name establishes that when (fn X) is true,
;   x has type set type-set.  Used by SMART-ASSUME-TRUE-FALSE to forward
;   chain from (fn X) to typeset info about X.  See ACCEPTABLE-COMPOUND-
;   RECOGNIZER-LEMMAP for a discussion.

(DEFPARAMETER TRUE-TYPE-ALIST NIL)

(DEFPARAMETER TTY-FILE NIL)

;   The name of the file to which error and warning messages will be printed.

(DEFPARAMETER TYPE-ALIST NIL)

;   An alist used by TYPE-SET -- and hence almost every routine in the theorem
;   prover -- to associate with a term its type bits.  The list is always
;   protected by rebinding it when a new entry must be added.  However it has
;   become a GLOBAL free var out of convenience.  TYPE-SET cannot be trusted
;   unless TYPE-ALIST is accurately set.

(DEFCONSTANT TYPE-SET-BOOLEAN 3)

;   The bit pattern representing the set containing only the type classes TRUE
;   and FALSE.

(DEFCONSTANT TYPE-SET-CONS 16)

;   Type set of CONS exprs.

(DEFCONSTANT TYPE-SET-FALSE 1)

;   The bit pattern representing the set containing only the type class FALSE.

(DEFCONSTANT TYPE-SET-LITATOMS 8)

;   Type set of PACK exprs.

(DEFCONSTANT TYPE-SET-NEGATIVES 32)

;   Type set of MINUS exprs.

(DEFCONSTANT TYPE-SET-NUMBERS 4)

;   The bit pattern representing the set containing only the type class of 0
;   and ADD1 -- i.e., that set recognized by NUMBERP.

(DEFVAR TYPE-SET-TERM1)

(DEFCONSTANT TYPE-SET-TRUE 2)

;   The bit pattern representing the set containing only the type class TRUE.

(DEFCONSTANT TYPE-SET-UNKNOWN -1)

;   The bit pattern representing the set of all type classes.

(DEFPARAMETER UN-PRODUCTIVE-PROCESSES
  (QUOTE (SETTLED-DOWN-CLAUSE STORE-SENT POP SUBSUMED-ABOVE
			      SUBSUMED-BY-PARENT SUBSUMED-BELOW
			      FINISHED)))

;   Used by IO1 to determine if the descendants list of the current process
;   should be printed out.  Some processes, such as simplification, lead to n
;   new clauses, where n=0 means the parent was proved.  For unproductive
;   processes, such as SETUP, the list of descendants is meaningless since the
;   process does not change the current goal -- as far as the IO is concerned.

(DEFPARAMETER UNDONE-BATCH-COMMANDS NIL)

(DEFVAR UNDONE-EVENTS)

(DEFPARAMETER UNDONE-EVENTS-STACK NIL)

(DEFVAR UNIFY-SUBST)

(DEFVAR UNIVERSE)

(DEFPARAMETER USE-NO-LEMMAS-FLG NIL)

;   When non-NIL this flag prevents REWRITE from using rewrite lemmas, axioms,
;   and recursive definitions.  It is still free to use built in information
;   (e.g., about EQUAL) and the type set information through
;   TYPE-PRESCRIPTIONs, RECOGNIZER-ALIST, etc.  The option is used when PROVE
;   is first given a theorem and we want to eliminate the propositional
;   calculus stuff in it -- expanding the IMPLIES and NOTs etc -- without
;   wasting time trying to rewrite the interior recursive part of the theorem
;   until we have dug out all the hypotheses and put the thing into clausal
;   form.

(DEFVAR VAL)

(DEFVAR VAR-ALIST)

(DEFPARAMETER WELL-ORDERING-RELATIONS (QUOTE (LESSP LEX2 LEX3)))

;   This is the list of all known well-founded relations -- the name is
;   misleading.  A function gets to be on this list by an act of god -- i.e.,
;   the user -- since the theorem prover cannot prove that a relation is
;   well-founded.

(DEFPARAMETER ZERO (QUOTE (QUOTE 0)))

;   Internal representation of (ZERO).

(DEFMACRO *1*IF (X Y Z) `(COND ((EQ ,X *1*F) ,Z) (T ,Y)))

(DEFMACRO ADD-SUB-FACT-BODY (&REST X) (GENERATE-ADD-SUB-FACT1 X))

(DEFMACRO ACCESS (&REST X) (ACCESS-MACRO (CAR X) (CADR X) (CADDR X)))

(DEFMACRO ARGN (&REST TAIL) (ARGN-MACRO TAIL))

(DEFMACRO BINDINGS (&REST TAIL) (BINDINGS-MACRO TAIL))

(DEFMACRO CHANGE (&REST X)
  (CHANGE-MACRO (CAR X) (CADR X) (CADDR X) (CADDDR X)))

(DEFMACRO DISABLEDP (NAME)
  `(AND (NOT (MEMBER-EQ (SETQ TEMP-TEMP ,NAME) TEMPORARILY-ENABLED-LEMMAS))
        (OR (MEMBER-EQ TEMP-TEMP TEMPORARILY-DISABLED-LEMMAS)
            (AND (SETQ TEMP-TEMP (ASSOC-EQ TEMP-TEMP DISABLED-LEMMAS))
                 (CDDR TEMP-TEMP)))))

(DEFMACRO FARGN (&REST TAIL) (FARGN-MACRO TAIL))

(DEFMACRO FARGS (X) `(CDR ,X))

(DEFMACRO FCONS-TERM (&REST TAIL) (CONS (QUOTE CONS) TAIL))

(DEFMACRO FCONS-TERM* (&REST TAIL) (CONS (QUOTE LIST) TAIL))

(DEFMACRO FFN-SYMB (X) `(CAR ,X))

(DEFMACRO FN-SYMB (&REST TAIL) (FN-SYMB-MACRO TAIL))

(DEFMACRO FQUOTEP (X) `(EQ (CAR ,X) (QUOTE QUOTE)))

(DEFMACRO LOGBIT (N) `(ASH 1 ,N))

(DEFMACRO LOGDIFF (X Y) `(BOOLE BOOLE-ANDC2 ,X ,Y))

(DEFMACRO MAKE (&REST X) (MAKE-MACRO (CAR X) (CDR X)))

(DEFMACRO MATCH (&REST X) (MATCH-MACRO (CAR X) (CADR X)))

(DEFMACRO MATCH! (&REST X) (MATCH!-MACRO (CAR X) (CADR X)))

(DEFMACRO NVARIABLEP (X) `(CONSP ,X))

(DEFMACRO PQUOTE (X) `(QUOTE ,X))

(DEFMACRO PRIND (X FILE)
  `(LET ((TEMP ,X))
     (PRINC TEMP ,FILE)
     (SETQ POS (+ POS
		  (OUR-FLATC TEMP)))))


(DEFMACRO QUOTEP (X) `(AND (CONSP (SETQ TEMP-TEMP ,X))
			   (EQ (CAR TEMP-TEMP) (QUOTE QUOTE))))

(DEFMACRO SWAP (X Y) `(SETQ ,X (PROG1 ,Y (SETQ ,Y ,X))))

(DEFMACRO WRITE-CHAR1 (X FILE)
  `(PROGN (WRITE-CHAR ,X ,FILE)
	  (SETQ POS (1+ POS))))

(DEFMACRO TYPE-PRESCRIPTION (X)
  `(CDAR (GET ,X (QUOTE TYPE-PRESCRIPTION-LST))))

(DEFMACRO VALUEP (X) `(QUOTEP ,X))

(DEFMACRO VARIABLEP (X) `(ATOM ,X))

(DEFUN *1*CAR (X1)
  (COND ((ATOM X1) 0)
	((EQ (CAR X1) (QUOTE *1*QUOTE)) 0)
	(T (CAR X1))))

(DEFUN *1*CDR (X1)
  (COND ((ATOM X1) 0)
	((EQ (CAR X1) (QUOTE *1*QUOTE)) 0)
	(T (CDR X1))))

(DEFUN ACCESS-MACRO (RECORD-NAME FIELD RECORD)
  (COND ((CADDR (ASSOC-EQ RECORD-NAME RECORD-DECLARATIONS))
	 (LIST (QUOTE CAR)
	       (CELL (GET-FIELD-NUMBER RECORD-NAME FIELD)
		     RECORD)))
	(T (LIST (QUOTE COND)
		 (LIST (LIST (QUOTE AND)
			     (LIST (QUOTE CONSP)
				   (COND ((CONSP RECORD)
					  (LIST (QUOTE SETQ)
						(QUOTE RECORD-TEMP)
						RECORD))
					 (T RECORD)))
			     (LIST (QUOTE EQ)
				   (LIST (QUOTE CAR)
					 (COND ((CONSP RECORD)
						(QUOTE RECORD-TEMP))
					       (T RECORD)))
				   (KWOTE RECORD-NAME)))
		       (LIST (QUOTE CAR)
			     (CELL (GET-FIELD-NUMBER RECORD-NAME FIELD)
				   (COND ((CONSP RECORD)
					  (QUOTE RECORD-TEMP))
					 (T RECORD)))))
		 (LIST T (LIST (QUOTE ACCESS-ERROR)
			       (KWOTE RECORD-NAME)))))))

(DEFUN ADD-TO-SET (X Y)
  (COND ((MEMBER-EQUAL X Y) Y)
	(T (CONS X Y))))

(DEFUN ARGN-MACRO (TAIL)
  (COND ((INTEGERP (CADR TAIL))
	 (SUB-PAIR (QUOTE (TERM CELL N))
		   (LIST (CAR TAIL)
			 (CELL (CADR TAIL) (QUOTE TEMP-TEMP))
			 (CADR TAIL))
		   (QUOTE (COND ((NOT (EQ (CAR (SETQ TEMP-TEMP TERM))
					  (QUOTE QUOTE)))
				 (CAR CELL))
				(T (ARGN0 TEMP-TEMP N))))))
	(T (CONS (QUOTE ARGN0)
		 TAIL))))

(DEFUN BINDINGS-MACRO (TAIL)
  (IF (ATOM TAIL)
      NIL
      `(CONS (CONS ,(CAR TAIL) ,(CADR TAIL))
	     ,(BINDINGS-MACRO (CDDR TAIL)))))

(DEFUN CELL (N FIELD)
  (COND ((= N 0) FIELD)
	(T (LIST (QUOTE CDR)
		 (CELL (1- N) FIELD)))))

(DEFUN CREATE-LEMMA-STACK (N)
  (SETQ ORIG-LEMMA-STACK (SETQ LEMMA-STACK (CREATE-STACK1 N)))
  (RPLACA LEMMA-STACK (QUOTE TOP))
  NIL)

(DEFUN CREATE-LINEARIZE-ASSUMPTIONS-STACK (N)
  (SETQ ORIG-LINEARIZE-ASSUMPTIONS-STACK
	(SETQ LINEARIZE-ASSUMPTIONS-STACK (CREATE-STACK1 N)))
  (RPLACA LINEARIZE-ASSUMPTIONS-STACK (QUOTE TOP))
  NIL)

(DEFUN CREATE-STACK1 (N)
  (LET (STK)
    (SETQ STK (ITERATE FOR I FROM 1 TO (* 2 N) COLLECT NIL))
    (ITERATE FOR TAIL ON STK BY (QUOTE CDDR) UNTIL (NULL (CDDR TAIL))
	     DO (RPLACA (CDDDR TAIL)
			TAIL))
    STK))

(DEFUN DEFEVENT-APPLY (X NAME FN MIN-ARGS MAX-ARGS)
  (COND ((OR (AND (ATOM X) X)
	     (AND (NOT (ATOM X)) (CDR (OUR-LAST X))))
	 (ERROR1 (PQUOTE (PROGN |Arg| |lists| |must| |end| |in| NIL |.|))
		 ()
		 (QUOTE HARD)))
	((OR (< (LENGTH X) MIN-ARGS)
	     (> (LENGTH X) MAX-ARGS))
	 (COND ((= MIN-ARGS MAX-ARGS)
		(ERROR1 (PQUOTE (PROGN (!PPR NAME NIL) |takes|
                                       (!PPR N NIL) |arguments|
				       |.|))
			`((NAME . ,NAME)
			  (N . ,MIN-ARGS))
			(QUOTE SOFT)))
	       (T (ERROR1 (PQUOTE (PROGN (!PPR NAME NIL) |takes| |from|
					 (!PPR I NIL) |to| (!PPR J NIL)
					 |arguments| |.|))
			  `((NAME . ,NAME)
			    (I . ,MIN-ARGS)
			    (J . ,MAX-ARGS))
			  (QUOTE SOFT)))))
	(T (CONS FN
		 (APPEND (ITERATE FOR ARG IN X COLLECT (LIST (QUOTE QUOTE) ARG))
			 (ITERATE FOR I FROM 1
				  TO (- MAX-ARGS (LENGTH X))
				  COLLECT NIL))))))

(DEFUN EQLENGTH (X N) (EQUAL N (LENGTH X)))

(DEFUN FARGN-MACRO (TAIL)
  (COND ((INTEGERP (CADR TAIL))
	 (LIST (QUOTE CAR)
	       (CELL (CADR TAIL)
		     (CAR TAIL))))
	(T (LIST (QUOTE NTH)
		 (CADR TAIL)
		 (CAR TAIL)))))

(DEFUN FN-SYMB-MACRO (TAIL)
  (SUBST (CAR TAIL)
	 (QUOTE TERM)
	 (QUOTE (COND ((NOT (EQ (QUOTE QUOTE) (CAR (SETQ TEMP-TEMP TERM))))
		       (CAR TEMP-TEMP))
		      (T (FN-SYMB0 (CADR TEMP-TEMP)))))))

(DEFUN GET-FIELD-NUMBER (RECORD-NAME LITATOM)
  (SETQ TEMP-TEMP (ASSOC-EQ RECORD-NAME RECORD-DECLARATIONS))
  (COND ((ITERATE FOR I FROM (COND ((CADDR TEMP-TEMP) 0)
				   (T 1))
		  AS FIELD IN (CADR TEMP-TEMP) WHEN (EQ FIELD LITATOM)
		  DO (RETURN I)))
	(T (TERPRI T)
	   (PRINC
	    (QUOTE |***** Undeclared record name or illegal field name *****|)
	    T)
	   (TERPRI T)
	   (SPACES 6 T)
	   (PRIN1 (LIST RECORD-NAME LITATOM) T)
	   (ITERPRI T)
	   1)))

(DEFUN KWOTE (X) `(QUOTE ,X))

(DEFUN MAKE-MACRO (RECORD ARGLIST)
  (COND ((NOT (SETQ TEMP-TEMP (ASSOC-EQ RECORD RECORD-DECLARATIONS)))
	 (ITERPRI T)
	 (IPRINC
	  (QUOTE |***** Undeclared record name in MAKE expression *****|)
	  T)
	 (ITERPRI T)
	 (SPACES 6 T) (PRIN1 RECORD T) (TERPRI T)
	 NIL)
	((= (LENGTH ARGLIST) (LENGTH (CADR TEMP-TEMP)))
	 (COND ((CADDR TEMP-TEMP)
		(CONS (QUOTE LIST) ARGLIST))
	       (T (CONS (QUOTE LIST)
			(CONS (KWOTE RECORD) ARGLIST)))))
	(T (IPRINC (QUOTE |***** Wrong number of args *****|) T)
	   (TERPRI T)
	   (SPACES 6 T)
	   (PRIN1 (CONS (QUOTE MAKE) (CONS RECORD ARGLIST)) T)
	   (ITERPRI T)
	   NIL)))

(DEFUN MATCH-MACRO (TERM PAT)
  (COND ((CONSP TERM)
	 (LIST (QUOTE PROGN)
	       (LIST (QUOTE SETQ) (QUOTE MATCH-TEMP) TERM)
	       (MATCH1-MACRO (QUOTE MATCH-TEMP) PAT)))
	(T (MATCH1-MACRO TERM PAT))))

(DEFUN MATCH!-MACRO (TERM PAT)
  (LIST (QUOTE OR)
	(MATCH-MACRO TERM PAT)
	(QUOTE (ERROR "MATCH! failed!"))))

(DEFUN MATCH1-MACRO (TERM PAT)
  (LET (TEST-LST SETQ-LST)
    (MATCH2-MACRO TERM PAT)
    (LIST (QUOTE COND)
	  (CONS
	   (COND ((NULL TEST-LST) T)
		 ((NULL (CDR TEST-LST)) (CAR TEST-LST))
		 (T (CONS (QUOTE AND) TEST-LST)))
	   (NCONC1 SETQ-LST T)))))

(DEFUN MATCH2-MACRO (TERM PAT)
  (COND ((ATOM PAT)
	 (COND ((EQ PAT (QUOTE &)) NIL)
	       ((OR (EQ PAT T) (EQ PAT NIL))
		(PRINC (QUOTE |***** Attempt to smash T or NIL ignored *****|)
		       T)
		(TERPRI T)
		(SPACES 6 T)
		(PRIN1 (CONS (QUOTE MATCH) MATCH-X) T)
		(ITERPRI T))
	       ((SYMBOLP PAT)
		(SETQ SETQ-LST (NCONC1 SETQ-LST (LIST (QUOTE SETQ) PAT TERM))))
	       (T (SETQ TEST-LST
			(NCONC1 TEST-LST (LIST (QUOTE EQUAL) PAT TERM))))))
	((EQ (QUOTE CONS) (CAR PAT))
	 (SETQ TEST-LST (NCONC1 TEST-LST (LIST (QUOTE CONSP) TERM)))
	 (MATCH2-MACRO (LIST (QUOTE CAR) TERM) (CADR PAT))
	 (MATCH2-MACRO (LIST (QUOTE CDR) TERM) (CADDR PAT)))
	((EQ (QUOTE QUOTE) (CAR PAT))
	 (COND ((SYMBOLP (CADR PAT))
		(SETQ TEST-LST
		      (NCONC1 TEST-LST
			      (LIST (QUOTE EQ)
				    (LIST (QUOTE QUOTE) (CADR PAT))
				    TERM))))
	       (T (SETQ TEST-LST (NCONC1 TEST-LST
					 (LIST (QUOTE EQUAL)
					       (LIST (QUOTE QUOTE)
						     (CADR PAT))
					       TERM))))))
	(T (COND ((NOT (EQ (CAR PAT)
			   (QUOTE LIST)))
		  (SETQ PAT (CONS (QUOTE LIST)
				  (CONS (LIST (QUOTE QUOTE)
					      (CAR PAT))
					(CDR PAT))))))
	   (ITERATE FOR SUBPAT IN (CDR PAT) DO (SETQ TEST-LST
						     (NCONC1 TEST-LST
							     (LIST (QUOTE CONSP)
								   TERM)))
		    (MATCH2-MACRO (LIST (QUOTE CAR)
					TERM)
				  SUBPAT)
		    (SETQ TERM (LIST (QUOTE CDR)
				     TERM)))
	   (SETQ TEST-LST (NCONC1 TEST-LST (LIST (QUOTE EQ)
						 TERM NIL))))))

(DEFUN MEMBER-EQ (X Y)
  (ITERATE FOR Z ON Y WHEN (EQ X (CAR Z)) DO (RETURN Z)))

(DEFUN MEMBER-EQUAL (X Y)
  (ITERATE FOR Z ON Y WHEN (EQUAL X (CAR Z)) DO (RETURN Z)))

(DEFUN NCONC1 (X Y) (NCONC X (LIST Y)))

(DEFUN RECORD-DECLARATION (RECORD-NAME FIELD-LST CHEAP)
  (LET (LST)
    (COND ((NOT (BOUNDP (QUOTE RECORD-DECLARATIONS)))
	   (SETQ RECORD-DECLARATIONS NIL)))
    (COND ((NOT (OR (CONSP FIELD-LST) (EQ FIELD-LST NIL)))
	   (ERROR "Illegal field list:  ~A" FIELD-LST))
	  ((NOT (OR (EQ CHEAP T) (EQ CHEAP NIL)))
	   (ERROR "Illegal cheap flag: ~A" CHEAP)))
    (SETQ LST (LIST RECORD-NAME (COPY-TREE FIELD-LST) CHEAP))
    (COND ((MEMBER-EQUAL LST RECORD-DECLARATIONS) NIL)
	  ((ASSOC-EQ (CAR LST) RECORD-DECLARATIONS)
	   (SETQ RECORD-DECLARATIONS
		 (CONS LST (REMOVE (ASSOC-EQ (CAR LST) RECORD-DECLARATIONS)
				   RECORD-DECLARATIONS :TEST #'EQUAL)))
	   (FORMAT T "~A redefined.~%" (CAR LST)))
	  (T (SETQ RECORD-DECLARATIONS
		   (CONS LST RECORD-DECLARATIONS))))
    RECORD-NAME))

(DEFUN RECORD-DECLARATION-LST (X)
  (ITERATE FOR TUPLE IN X DO (APPLY (FUNCTION RECORD-DECLARATION) TUPLE)))

(DEFUN SPACES (N FILE)
  (COND ((<= N 0) NIL)
	(T (ITERATE FOR I FROM 1 TO N DO (WRITE-CHAR #\Space FILE)))))

(DEFUN SPELL-NUMBER (N)
  (CASE N
	(0 (QUOTE |zero|))
	(1 (QUOTE |one|))
	(2 (QUOTE |two|))
	(3 (QUOTE |three|))
	(4 (QUOTE |four|))
	(5 (QUOTE |five|))
	(6 (QUOTE |six|))
	(7 (QUOTE |seven|))
	(8 (QUOTE |eight|))
	(9 (QUOTE |nine|))
	(10 (QUOTE |ten|))
	(OTHERWISE
	 N)))

(DEFUN CHANGE-MACRO (RECORD-NAME FIELD RECORD VALUE)
  (COND ((CADDR (ASSOC-EQ RECORD-NAME RECORD-DECLARATIONS))
	 (LIST (QUOTE RPLACA)
	       (CELL (GET-FIELD-NUMBER RECORD-NAME FIELD) RECORD)
	       VALUE))
	(T (LIST (QUOTE COND)
		 (LIST (LIST (QUOTE AND)
			     (LIST (QUOTE CONSP)
				   (COND ((CONSP RECORD)
					  (LIST (QUOTE SETQ)
						(QUOTE RECORD-TEMP)
						RECORD))
					 (T RECORD)))
			     (LIST (QUOTE EQ)
				   (LIST (QUOTE CAR)
					 (COND ((CONSP RECORD)
						(QUOTE RECORD-TEMP))
					       (T RECORD)))
				   (KWOTE RECORD-NAME)))
		       (LIST (QUOTE RPLACA)
			     (CELL (GET-FIELD-NUMBER RECORD-NAME FIELD)
				   (COND ((CONSP RECORD)
					  (QUOTE RECORD-TEMP))
					 (T RECORD)))
			     VALUE))
		 (LIST T (LIST (QUOTE ACCESS-ERROR)
			       (KWOTE RECORD-NAME)))))))

(DEFUN OUR-LAST (L)
  (COND ((ATOM L) L)
	(T (ITERATE UNTIL (ATOM (CDR L)) DO (SETQ L (CDR L))
		    FINALLY (RETURN L)))))

(DEFUN SUB-PAIR (L1 L2 X)
  (COND ((ITERATE FOR Z IN L2 AS Y IN L1 WHEN (EQUAL Y X)
		  THEREIS (PROGN (SETQ TEMP-TEMP Z) T))
	 TEMP-TEMP)
	((ATOM X) X)
	(T (CONS (SUB-PAIR L1 L2 (CAR X)) (SUB-PAIR L1 L2 (CDR X))))))

(RECORD-DECLARATION-LST
 (QUOTE
   ((CANDIDATE (SCORE CONTROLLERS CHANGED-VARS UNCHANGEABLE-VARS
		      TESTS-AND-ALISTS-LST JUSTIFICATION INDUCTION-TERM
		      OTHER-TERMS)
	       NIL)

;   This record is used to represent one of the plausible inductions that must
;   be considered.  The SCORE represents some fairly artificial estimation of
;   how many terms in the conjecture want this induction.  CONTROLLERS is the
;   list of the controllers -- including unchanging controllers -- for all the
;   inductions merged to form this one.  The CHANGED-VARS is a list of all
;   those variables that will be instantiated (non-identically) in some
;   induction hypotheses.  Thus, CHANGED-VARS include both variables that
;   actually contribute to why some measure goes down and variables like
;   accumulators that are just along for the ride.  UNCHANGEABLE-VARS is a list
;   of all those vars which may not be changed by any substitution if this
;   induction is to be sound for the reasons specified.  TESTS-AND-ALISTS-LST
;   is a list of TESTS-AND-ALISTS which indicates the case analysis for the
;   induction and how the various induction hypotheses are obtained via
;   substitutions.  JUSTIFICATION is the JUSTIFICATION that justifies this
;   induction, and INDUCTION-TERM is the term that suggested this induction and
;   from which you obtain the actuals to substitute into the template.
;   OTHER-TERMS are terms whose induction candidates have been merged into this
;   one for heuristic reasons.

    (GENERALIZE-LEMMA (NAME TERM) NIL)

;   This record records a GENERALIZE lemma with name NAME.  The TERM is just a
;   well-formed formula, assumed to be a theorem of course, translated but
;   possibly involving IMPLIES and the binary AND.  These records are collected
;   on the TYPE-NAME-AND-LEMMA-LST and used when a term, x, is generalized by
;   scanning the list for all formulas involving x -- modulo a unification of
;   course -- and adding to the hypothesis of the theorem, before it is
;   generalized, the appropriately instantiated formulas found.

    (JUSTIFICATION (SUBSET MEASURE-TERM RELATION LEMMAS) NIL)

;   Consider the INDUCTION-MACHINE for some function.  This record gives one
;   justification for it.  In particular, MEASURE-TERM, which is a term on a
;   subset of the formals of the function, decreases according to the
;   well-founded relation RELATION in each hypothesis of the INDUCTION-MACHINE.
;   SUBSET is the measured subset of formals.  The fact that the measure
;   decreases can be proved using the lemmas in the list LEMMAS.

    (LINEAR-LEMMA (NAME HYPS CONCL MAX-TERM) NIL)

;   Internal form of a LINEAR lemma.  NAME is the event name, HYPS is the list
;   of hypotheses, and POLY is the LINEARIZEd conclusion.

    (LINEAR-POT (VAR POSITIVES NEGATIVES) T)

    (MEASURE-RULE (CONDITION-LIST THE-LESSER STRENGTH-SIGN THE-GREATER
				  INDUCTION-LEMMA-NAME MEASURE)
		  NIL)

;   A record used to store the representation of an INDUCTION lemma.  The
;   MEASURE field contains the pair (C . R) where C is a measure function name
;   and R is a well-founded relation name.  The name of the lemma represented
;   by the rule is stored in the INDUCTION-LEMMA-NAME field.  The hypotheses of
;   the lemma are in the CONDITION-LIST field.  The form of the conclusion of
;   the lemma depends on the contents of the STRENGTH-SIGN field.  If that
;   field contains a - (minus sign) then the conclusion is:

;        (R (C . THE-LESSER)  (C .  THE-GREATER)),

;   where THE-LESSER and THE-GREATER are the contents of the fields of those
;   names.  Otherwise, the STRENGTH-SIGN field contains a 0 and the conclusion
;   is the disjunction of the one given above and the term:

;        (EQUAL (C . THE-LESSER) (C . THE-GREATER)).

    (POLY (CONSTANT ALIST ASSUMPTIONS LITERALS LEMMAS) T)

;   CONSTANT is a INTEGERP.  ALIST is an alist of pairs of the form (ti . ki) where
;   ti is a term and ki is a INTEGERP.  ASSUMPTIONS is a list of terms.  LITERALS
;   is a list of terms.  LEMMAS is a list of lemma names, (LIST 'MARK)'s -- as
;   constructed by ADD-TERMS-TO-POT-LST, or a term.  A POLY represents an
;   implication hyps -> concl, where hyps is the conjunction of ASSUMPTIONS and
;   concl is CONSTANT + k1*t1 + ... + kn*tn <= 0, over the integers.  Every
;   POLY in SIMPLIFY-CLAUSE-POT-LST is being assumed.  See ADD-TERMS-TO-POT-LST
;   for more details about LITERALS and LEMMAS.

    (REWRITE-RULE (NAME HYPS CONCL LOOP-STOPPER) NIL)

;   These records are used to represent rewrite rules.  The NAME field contains
;   the name of the lemma or axiom.  The HYPS is a list of terms whose
;   conjunction is the hypothesis of the lemma.  The CONCL is the term which is
;   the conclusion of the lemma.  The LOOP-STOPPER is an alist dotting vars to
;   vars and used to prevent infinite rewrite loops due to permutative rewrite
;   rules.  The rewrite cannot be performed if the instantiation of the CDR of
;   some pair is lexicographically bigger than or equal to -- see
;   LOOP-STOPPER-GTE -- the instantiation of the CAR.

    (TESTS-AND-ALISTS (TESTS ALISTS) NIL)

;   A list of these records forms the TESTS-AND-ALISTS-LST component of a
;   CANDIDATE record.  The TESTS field contains a list of forms and the ALISTS
;   field a list of alists.  If the conjunction of the tests is true then some
;   measure goes down on the n-tuples produced by each alist.  The alist
;   contains explicitly pairs of the form (v . v) if v is an unchanging
;   controller.  The soundness of MERGE-CANDS rests on this fact.

    (TESTS-AND-CASE (TESTS CASE) NIL)

;   This is like a TESTS-AND-CASES except that the CASE component contains
;   exactly one case.  A list of TESTS-AND-CASEs is a flattened machine.

    (TESTS-AND-CASES (TESTS CASES) NIL)

;   A list of these compose a machine (see INDUCTION-MACHINE) The TESTS field
;   contains a list of terms whose conjunction must be true before the machine
;   can "execute" the cases.  The CASES field contains a list of arglists of
;   the recursive calls governed by the the tests.  By the nature of the
;   machine, all TESTS are previously defined concepts and only the CASES
;   involve the new function.

    (TYPE-PRESCRIPTION-NAME-AND-PAIR (NAME PAIR) NIL)

;   The TYPE-PRESCRIPTION-LST property is a list of these records.  The NAME is
;   the name of the rewrite lemma -- or definition -- that gave rise to the
;   type prescription pair (ts . args) PAIR for the function under which this
;   type prescription is hung.

    (TYPE-RESTRICTION (TERM TYPE-SET DEFAULT)
		      NIL))))

;   This record is used to represent processed type restrictions as found on
;   the destructors of shells.  The TERM component is a normalized term
;   composed of recognizers applied to the variable X, possibly negated, and
;   possibly disjoined or conjoined with binary OR or AND.  The TYPE-SET
;   component is the corresponding bit mask.  The DEFAULT is the specified
;   default object satisfying the type set.  At prove time no one looks at
;   TERM.  It is examined during add shells and is used in the rewrite rules
;   added.

;                            STACK-INITIALIZATION

;  The LEMMA-STACK is circular, and we don't want it accidentally printed.
(PROGN (CREATE-LEMMA-STACK 10) (QUOTE LEMMA-STACK))

;  The LEMMA-STACK is circular, and we don't want it accidentally printed.
(PROGN (CREATE-LINEARIZE-ASSUMPTIONS-STACK 10) 
       (QUOTE LINEARIZE-ASSUMPTIONS-STACK))



