/*  PLAISTED.PL  */
/*  Shelved on the 3rd of October 1988  */


%        C Prolog theorem prover based on the
%         simplified problem reduction format
%
%        (c) 1986, 1987 by David A. Plaisted
%
%                Release 2
%              April 6, 1987
%
% The top level call is try(File) where File has clauses converted to
% Horn clauses as follows:
%
%   The clauses p or q, p or (not q), (not p) or q, (not p) or (not q)
%   would be expressed as
%
%   p :- not(q).
%   p :- q.
%   q :- p.
%   false :- p, q.
%
% To do only forward chaining, no backward chaining, assert(f_only).
% This may be incomplete.  To do only back chaining, assert(b_only).
%
% To avoid size computations of subgoals and solutions, assert(nosize).
% This can make the prover significantly faster.
%
% To bound the maximum size of a literal in a proof, assert(maxsize).
% This often helps very much.
% 
% There are two constants that influence how much is charged for a
% lemma that has already been proved.  They are solution_size_multiplier
% and proof_size_multiplier.  Default values are 0.1 and 0.4.
% The former weights the size of the solution and the latter
% weights the length of the proof used to derive it.  To
% set them type solution_size_multiplier(M) and proof_size_multiplier(N).
% Good values to try are (0.125, 0.0) and (0, 1.0).
%
% To not record proofs, assert(noproof).  This can save a lot of memory.
%
% For more features and instructions see the file of examples for this
% prover.  The prover has facilities for term rewriting, for example.
% Also, there is a convenient interface to Prolog source code from
% the theorem prover.
%
% Suggested allocations are -g 1000 -h 1000 -t 700 in C Prolog 1.5
%
     :- print('For a trace of the proof type proof_trace.'), nl.
     :- print('To turn this off, type proof_untrace.'), nl.

     try(File) :-
    try(File, 100000).  % default limit 100000
     try(File, N) :- 
    print(File), nl,
    abolish_clauses,
    see(File), preprocess(File),
    close(File),
    X is cputime,
    abolish(count,1),
    abolish(count2,1),
    abolish(count3,1),
    assert(count(0)),
    assert(count2(0)),
    assert(count3(0)),
    print_flags,
    ((nosave ; nosave(_)) -> assert(nosaves) ; true),!,
    (ach2(5, N) ;
     print('proof not found')), nl,
    Y is cputime, Z is Y - X,
    print(Z), print(' cpu seconds used'), nl,
    count(I), print(I), print(' inferences done'), nl,
    abolish(nosave,1),
    !.

% SIZE gives size of proof, S gives bound on split nesting.

     ach2(SIZE, MAX) :- 
    (nosplits -> S = 0 ; S is (SIZE // 6)),
%   abolish(done, 2),   % retract old done assertions
    print('.'), nl,
    (f_only -> BSIZE = 1 ; BSIZE = SIZE),   % only forward chaining
    achR((false :- []),(false :- B),0,_,BSIZE,[],Proof,S),
    B == [],
    print('proof found'), nl,
    print(Proof), nl,
    print('size of proof '), print(SIZE), nl, !.
     ach2(SIZE, MAX) :-
    S is (SIZE // 6),
    FSIZE is (2 * SIZE) // 3,
    (SIZE =< 9 -> SIZE1 is SIZE + 2 ; SIZE1 is (SIZE * 4) // 3),
    SIZE1 =< MAX,
    (nosave ; b_only ; forward_chain(FSIZE, S)), !,
    back_simplify,
    ach2(SIZE1, MAX).

     print_flags :-
    member(X,[nosave,nosave(_),f_only,b_only,nosize,maxsize,
          nosplits,solution_size_mult(_),
          proof_size_mult(_),noproof]),
    clause(X,true),
    print(X), print(' is asserted'), nl, fail.

     print_flags.

     abolish_clauses :-
    abolish(ach, 8),
    abolish(replace_rule,2),
    abolish(rewrite_rule,2),
    abolish(rewrites,0),
    abolish(replacements,0),
    abolish(irreducible,1),
    abolish(done,2),
    abolish(solution,5),
    abolish(used,3),
    abolish(nosaves,0),
    abolish(f_chain,2).

     increment :-
    count(X), Y is X+1,
    retract(count(_)), !,
    assert(count(Y)).

     increment2 :-
    count2(X), Y is X+1,
    retract(count2(_)), !,
    assert(count2(Y)).

     increment3 :-
    count3(X), Y is X+1,
    retract(count3(_)), !,
    asserta(count3(Y)).     % asserta prevents infinite loop
                    % via backtracking
     :- op( 1200, xfx, [ : ]).
     :- op( 1200, xfx, [ :-- ]).
     :- op( 1200, xfx, [ :-. ]).

     :- abolish(solution_size_mult,1).
     :- abolish(proof_size_mult,1). 
     :- assert(proof_size_mult(0.4)).
     :- assert(solution_size_mult(0.1)).

% filter splits to avoid duplicated or contradictory elements

     filter(L) :-
    member(not(X), L),
    memq(X,L),
    pprint(filter(L)), !, fail.
     filter(L) :- append(L1, [X|L2], L),
    memq(X,L2), pprint(filter(L)),
    !, fail.
     filter(L).

% preprocess clauses

     preprocess(File) :-
    read(X), X \== 'end_of_file',
    print(X), nl,
    assert_solutions(X),
    convert_clause(_,X,Y),
    passert(Y),
    preprocess(File).

     preprocess(File) :- nl.

% assert solutions for clauses read in

     assert_solutions(rewrite(U,V)) :- !.
     assert_solutions(replace(U,V)) :- !.
     assert_solutions(X) :-
    (X = (L :- B) ; X = (L :-- B) ; X = (L :-. B)),
    (vars_in(B) ; L \== false), !.  % variable in body or L not false
     assert_solutions(X) :-
    (X = (L :- B) ; X = (L :-- B) ; X = (L :-. B)),!,
    make_list(B,BL),
    length(BL,Len),
    (Len =< 1 ->
        (member(prolog(_),BL) -> true ;
        linearize_term(L,LL,V1,V2),
        passert(solution(0,(LL :- BL),V1,V2,X))) ;
        true).
     assert_solutions(X) :-     % do for assertions
    linearize_term(X,XN,V1,V2),
    passert(solution(0,(XN :- []),V1,V2,X)).    % assert solution

     vars_in(X) :- var(X), !.
     vars_in((X,Y)) :- vars_in(X) ; vars_in(Y).

     make_list((X,Y),Z) :- !,       % convert comma notation to list
    make_list(X,Z1),
    make_list(Y,Z2),
    append(Z1,Z2,Z).
     make_list(X,[X]).

% convert clauses from external to internal format

%    ach(X, AG, M, N, SIZE, G, Proof, Splits)  X is current goal,
%    AG is achieved goal, which is an instance of X with possibly
%    some negative literals added to the body, M is bound on
%    size of proof so far, N is bound on proof size after solving
%    goal X, SIZE is upper bound on proof size, G is list of higher
%    goals, Proof is the proof, Splits is number of splits allowed.

     convert_clause(D,rewrite(LHS,RHS),rewrite_rule(LHS,RHS)) :- !,
    (\+ rewrites -> assert(rewrites) ; true).

     convert_clause(D,replace(LHS,RHS),replace_rule(LHS,RHS)) :- !,
    (\+ replacements -> assert(replacements) ; true).

     convert_clause(D,(X1 :-- X2),      % don't charge for :--
            (ach((Y1 :- B),(Y1 :- AB),M,N,SIZE,G,(Y1 :- Proof), S) :-
            Unif,
            Y2,
            increment)) :-
    !, convert_body(X2,Y2,L,M,N,SIZE,G,Proof,B,AB,S),
    linearize_term(X1,Y1,Z1,Z2), make_unifier(Z1,Z2,Unif).

     convert_clause(D,(X1 :-. X2),      % charge one for :-.
            (ach((Y1 :- B),(Y1 :- AB),M,N,SIZE,G,(Y1 :- Proof), S) :-
            M+1 =< SIZE, Unif,
            M1 is M+1, Y2,
            increment)) :-
    !, convert_body(X2,Y2,L,M1,N,SIZE,G,Proof,B,AB,S),
    linearize_term(X1,Y1,Z1,Z2), make_unifier(Z1,Z2,Unif).

     convert_clause(D,(X1 :- X2),
            (ach((Y1 :- B),(Y1 :- AB),M,N,SIZE,G,(Y1 :- Proof), S) :-
            M+L0 =< SIZE, Unif,
            M1 is M+L0, Y2,
            increment)) :-
    !, convert_body(X2,Y2,L,M1,N,SIZE,G,Proof,B,AB,S),
    (nonvar(D) -> L0 = D ; L0 = L),     % give increment if specified
    linearize_term(X1,Y1,Z1,Z2), make_unifier(Z1,Z2,Unif).

     convert_clause(D, X1,          % assertion
            (ach((Y1 :- B),(Y1 :- B),M,M,SIZE,G,Y1,S) :-
            nosaves,        % units already solutions,
            Unif,           % so not needed usually
            increment)) :- !,
    linearize_term(X1,Y1,Z1,Z2), make_unifier(Z1,Z2,Unif).

     convert_body(X, (nonvar(X),
            achR((X :- B),(X :- AB),M,N,SIZE,G,Proof,S)),
          1,M,N,SIZE,G,Proof,B,AB,S) :- var(X), !.
                % subgoal is a variable.  Require
                % instantiations to be non-variables.

     convert_body((X1,X2),(Y1,Y2),L,M,N,SIZE,G,(Proof1,Proof2),B,AB,S) :-
    !, convert_body(X1,Y1,L1,M,N1,SIZE,G,Proof1,B,AB1,S),
    convert_body(X2,Y2,L2,N1,N,SIZE,G,Proof2,AB1,AB,S),
    L is L1+L2.

     convert_body(prolog(X), X,         % call Prolog
          0,M,M,SIZE,G,prolog(X),B,B,S) :- !.

     convert_body(X, achR((X :- B),(X :- AB),M,N,SIZE,G,Proof,S),
          1,M,N,SIZE,G,Proof,B,AB,S).   % subgoal

% forward chaining part

     forward_chain(SIZE, S) :- 
    pprint('BEGIN FORWARD CHAINING'),
    assert(f_chaining), forward_infer(SIZE, S), !,
    abolish(f_chaining,0),
    pprint('END FORWARD CHAINING').

     forward_infer(SIZE, S) :-
    achR((X :- []), AG, 0, Y, SIZE, [], P, S),
    AG == (false :- []),!.

     forward_infer(_,_).

     passert(X) :- (passert1 -> (print(assert(X)),nl) ; true), assertz(X).
     passerta(X) :- (passert1 -> (print(assert(X)),nl) ; true), asserta(X).

     passert :- assert(passert1).
     unpassert :- retract(passert1).
     proof_trace :- assert(proof_trace1).
     proof_untrace :- retract(proof_trace1).
     solution_size_multiplier(M) :-
    abolish(solution_size_mult,1),
    assert(solution_size_mult(M)).
     proof_size_multiplier(M) :-
    abolish(proof_size_mult,1),
    assert(proof_size_mult(M)).
     pprint(X) :- proof_trace1,!,print(X),nl.
     pprint(X).

% only limited depth of forward chaining permitted

     achR(X, AG, M, N, SIZE, G, P, S) :-
    f_chaining, increment2, length(G,Len), Len >= 2, !, fail.

% deal with :- in the subgoal

     achR((L :- B), (L :- B3), M, N, SIZE, G, assume(L2, P), S) :-
    nonvar(L), L = (L1 :- L2), !,
    S1 is S + 1,
    ((append(B,[L2],B1),    % add extra assumption to the end
      achR((L1 :- B1), (L1 :- B2), M, N, SIZE, G, P, S1),
      append(B3,[L2],B2)) ; % remove assumption from the end
     (split_member(L2,B),
      achR((L1 :- B), (L1 :- B3), M, N, SIZE, G, P, S))).

% goal seen already, fail

     achR(X, AG, M, N, SIZE, G, P, S) :- memq(X, G),!,fail.

% do assumption axioms here if not caching

     achR((L :- [X|Y]), (L :- [X|Y]), N, N, SIZE, G, L, S) :-
    (nosave -> true ; nonvar(L), nosave(L)),
    split_member(L, [X|Y]),
    pprint(assumption((L :- [X|Y]))).

     achR((not(L) :- B), (not(L) :- [not(L)|B]), N, N,
        SIZE, G, not(L), S) :-
    \+ nosplits,
    (nosave(not(L)) -> true ; nosave),
    length(B,Len), (N > S -> Len < S ; true),
    pprint(assumption((not(L) :- [not(L)|B]))).

     achR((not(L) :- [X|Y]), AG, N, N, SIZE, G, P, S) :- 
    memq(L, [X|Y]),
    pprint(fail((not(L) :- [X|Y]))), !, fail.
        % fail because don't want not(L) literals in L split case.

% do fast, don't record solutions if nosave is asserted

     achR(X, AG, M, N, SIZE, G, P, S) :-
    X = (L :- B), L \== false, nosave, !,
    pprint(attempt(X)),
    ach(X, AG, M, N, SIZE, [X|G], P, S),
    pprint(solved(AG)).

% do fast, don't record solutions if nosave(X), that is, no
% caching or splitting for X

     achR(X, AG, M, N, SIZE, G, P, S) :-
    X = (L :- B), nonvar(L), nosave(L), !,
    pprint(attempt(X)),
    ach(X, AG, M, N, SIZE, [X|G], P, S),
    pprint(solved(AG)).

% record new goal on list of goals

     achR((X :- B1), (X :- AB2), M, N, SIZE, G, P, S) :- D is SIZE-M,
    (maxsize ->
     \+ big_subgoal((X :- B1), SIZE) ; true), % delete large subgoals
    replace_rewrite(X,X2,X2N),  % do rewriting and replacements
    copy((X2 :- B1), (Y :- B2)),
    remove_false(B1,B1F),
    (B2 \== [] ->
     (filter(B2),
      ((X2 :- B1) \== (Y :- B2) -> numbervars((Y :- B2), 0, _) ; true),
                    % ground term
      \+ unit_delete(B2),       % delete subgoal if test succeeds
      remove_false(B2,B2F),
      delete_units(B1F,B2F,B1S,B2S)) ;  % delete units known
     (Y = X2N,          % take care of B2 == [] case
      B2F = B2, B2S = B2,
      B1S = B2)),
    (memq((X :- B1S),G) -> (pprint(recursion_fail((X :- B1S))),fail)
         ; true),   % test if simplified X is on stack
    increment3,
    count3(I),!,
    pprint(attempt(D,id(I),(X2 :- B1S))),
    (already_solved((Y :- B2),P) ->
     (AB2 = B1, N = M,
      pprint(already_solved(id(I),(X :- B1)))) ;
     (achR2((X2 :- B1S), (X2 :- AB), (Y :- B2S), M, N, SIZE, G, P, D, S,I),
      merge(B1,AB,AB2,N,S))).

% assumption axiom rules

     achR2((not(L) :- B), (not(L) :- [not(L)|B]), (L1 :- B1), N, N,
        SIZE, G, not(L), _, S, I) :-
    \+ nosplits,
    length(B,Len), (N > S -> Len < S ; true),
    \+ unit_delete([L1]),       % don't do if L is proven
    pprint(assumption(id(I),(not(L) :- [not(L)|B]))).
        % extra one needed for other split case later

     achR2((L :- [X|Y]), (L :- [X|Y]), _, N, N, SIZE, G, L, _, S, I) :-
    split_member(L, [X|Y]),
    pprint(assumption(id(I),(L :- [X|Y]))).

% Use old solutions first, even if subgoal has never been seen before

    achR2((X :- B1), (X :- AB), (Y :- B2), M, N, SIZE, G, P, D, S,I) :-
%   ((solution(F,(XN :- []),V1,V2,P), XN = X, B4 = []) ;
%                   % try short solutions first
%   (solution(F,(XN :- [M1]),V1,V2,P), XN = X, B4 = [M1]) ;
%   (solution(F,(XN :- [M1,M2|M3]),V1,V2,P), XN = X, B4 = [M1,M2|M3])),
    solution(F,(X :- B4),V1,V2,P),
    (maxsize -> \+ big_subgoal((X :- B4), SIZE) ; true),
    solution_size(F,(X :- B4),N1),  % compute size of instantiated goal
    N is M + N1,
    N =< SIZE,
    length(B4,Len),         % don't use long solutions
    (N > S -> Len =< S ; true),
    unify_lists(V1,V2),
    remove_false(B4,B4F),
    split_subset(B4F,B1),
    merge(B1,B4,AB,N,S),
    pprint(use_old_solution(F,id(I),(X :- B4))),
    increment.

% Goal already seen, do not look for new solutions

     achR2((X :- B1), (X :- AB), (Y :- B2), M, N, SIZE, G, P, D, S,I) :-
    \+ f_only,
    done(E,(Y :- B2)), E>=D,
    pprint(already_seen(D,id(I),(X :- B1))),
    !, fail.

% call ach and then possibly try to delete negative literals by
% case analysis

     achR2((L :- B),(L :- B4),X,M,N,SIZE,G,P,D,S,I) :-
    ach((L :- B), (L :- B2), M, N, SIZE, [(L :- B)|G], Pold, S),
%   E1 is N1 - M,
%   record_solution(E1,(L :- B2),Pold,start(I)),
    ((\+ splittable((L :- B),B2,X,G)) -> (B4=B2, Pold = P) ;
     pprint(attempt_splits(D,id(I),(L :- B2))),
     length(B,Len),
     do_splits((L :- B2), (L :- B4), Len, N, _, SIZE, G, Pold, P, S, I)),
    E is N - M,
    length(G, Len2),
    record_solution(E,(L :- B4),SIZE,Len2,P,I).

% criterion for deciding whether to do case analysis

     splittable((L :- B),[(not _)|_],(L2 :- B2), G) :-
    (L == L2; L == false),  % ground head literal or later
                % becomes false
    \+ (L = not(_)),    % non negated head
    ((L = and(U,V) ; L = and(U) ;
      G = [or(U)|W] ; G = [or(U,V)|W]) -> fail ; true), !.
        % don't split and goals or arguments of or goals

%    splittable((L :- B), (L2 :- B2), G) :-
%   L = or(U,V),        % allow non-ground or goals to be split
%   \+ (G = [or(U,V)|W]),!. % if not inside other or goals

     achR2((L :- B), AG, Y, M, N, SIZE, G, P, D, S, I) :-
    \+ f_only,
    (proof_trace1 -> (print(done(D,id(I),(L :- B))), nl) ; true),!,
    record_done(D,(L :- B),I),!,fail.
            % don't make achR succeed when it shouldn't

% do_splits tries to delete negative literals by splitting
% this function is only called for splittable subgoals.  The 
% test for splittability is a syntactic one.  This seems best
% unless semantic information is available that certain
% subgoals are provable.

     do_splits((L :- [not(LL)|B]), AG, Len, M, N, SIZE, G, Pold, P, S, I) :-
    length(B, Len2),
    Len2 >= Len,        % can remove more literals by splitting
    M1 is M,        % charge zero for splits
    M1 =< SIZE,
    L \== not(LL),      % test that assumption axiom not just used
                % see if can delete negative literal by a split
    achR((L :- [LL|B]), (L :- B1), M1, N1, SIZE, G, P1, S),
    split_subset(B1,[LL|B]),    % check that no neg. literals added
                % now maybe delete more negative literals
    E is N1 - M,
%   record_solution(E,(L :- B), cases((not(LL) : Pold), (LL : P1)),
%       split(I)),
                % record ``resolvent'' of split cases
    do_splits((L :- B), AG, Len, N1, N, SIZE, G, cases((not(LL) : Pold),
        (LL : P1)), P, S, split(I)).

% try also the case where no negative literals are deleted

     do_splits(X, X, Len, M, M, SIZE, G, P, P, S, I).

% we would like somehow to delay recording of solutions in do splits
% until splitting is done for easy retracting of less general ones.

% record solutions.  Fail if solution exists already since old solution
% could probably have been used instead (unless recently generated).

     record_solution(D,X,SIZE,Len,P) :- record_solution(D,X,SIZE,Len,P,_).

     record_solution(D,(L :- B),SIZE,Len,P,I) :-
    memq(L,B),!.    % don't record if L is in B

     record_solution(D,(L0 :- B0),SIZE,Len2,P,I) :- !, 
    (replacements ->
     replace(L0, L0R, _) ; L0 = L0R),
    (rewrites -> rewrite_solution((L0R :- B0), (L :- B)); % do rewriting
        (L0R :- B0) = (L :- B)),
    filter(B),             % omit tautologies and repeats
    copy((L :- B),(L1 :- B1)),
    ((L :- B) \== (L1 :- B1) -> numbervars((L1:- B1),0,_) ; true),
                    % ground term
    !,
    unit_simplify(B,B1,BS,B1S), % delete literals if negation proved
    \+ unit_delete(B1S),
    (maxsize -> \+ big_subgoal((L :- BS), SIZE) ; true),
%   solution_size((L :- BS), N1),
    record_numbervard(D,(L :- BS),(L1 :- B1S),Len2,P,I),!.

     solution_size(D, (L :- BS), N1) :- % compute size of instantiated goal
    (nosize -> Len = 0 ; length(BS,Len)), 
    (top_connective(L) -> N0 = 0, NS = 0 ;
     flatsize(L, N0),
     flatsize(BS, NS)),
    proof_size_mult(M1),
    solution_size_mult(M2),
    N1 is floor(((N0 + NS) * M2) + ((D + Len) * M1)),!.

     record_numbervard(D,X0,(X :- B1),Len,P,I) :-
    solution(E,(X :- B2),V1,V2,_),V1=V2,
    E =< D, subset(B2,B1),!,
    pprint(repeated_solution(D,id(I),X0)),
    ((X0 = (false :- _) ; Len =< 1 ;
     already_solved((false :- []))) -> true). % fail if not false :- _
    % found more general solution already   % and not done

     record_numbervard(D,(X :- B),(Y :- BY),Len,P,I) :-
%   retract_old(D,(X :- B),Z,V1,V2),% retract old solutions
    (proof_trace1 ->
       (nl,print(newly_solved(D,id(I),(X :- B))),nl,nl) ; true),
    (nonvar(X) ->
     process_equation((X :- B),Save) ; Save = true),
%   ((var(I) ; (I \== no , \+ (I = start(_)) , \+ (I = split(_))))
%        -> make_clause(D,(X :- B),(Y :- BY)) ; true),
%           % assert solutions as clauses
    ((X :- B) == (false :- []) -> Proof = P ;
        Proof = lemma((X :- B))),   % record proof of false :- []
    linearize_term(X,Z,V1,V2),
    passertf(Save,solution(D,(Z :- B),V1,V2,Proof),(Y :- BY)),
    ((Y \== false, negate(Y,YN),
      (already_solved((YN :- BY),_) ; memq(YN,BY))
                      ) ->   % negation of Y proven with splits
         (pprint(negation_solved((YN :- BY))),
%         solution_size((false :- BY),D1),
          (already_solved(D,(false :- BY),_) ;
           (% retract_old(D,(false :- B),_,_,_),
            (proof_trace1 -> 
         (nl,print(newly_solved(D,id(I),(false :- B))),nl,nl) ; true),
            (B == [] -> Proof2 = cases((YN : (YN :- BY)), (Y : P)) ;
                Proof2 = lemma((false :- B))),
            passertf(true,
         solution(D,(false :- B),[],[],Proof2),(false :- BY))
%       , make_clause(D,(false :- B), (false :- BY))
           )));
      true),!.


     passertf(true,solution(D,(false :- [not(L)|B]),[],[],P),
           (false :- [not(LG)|BG])) :- !,
    (noproof -> P1 = [] ; P1 = P),
    passerta(solution(D,(false :- [not(L)|B]),[],[],P1)),
%   solution_size((L :- B),D1),
    ((solution(E,(LG :- B2),W1,W2,_),W1=W2,
      E =< D, subset(B2,BG)) ;
     (linearize_term(L,LN,V1,V2),
      passert(solution(D,(LN :- B),V1,V2,transpose(P1))))),!.

     passertf(true,solution(D,X,V1,V2,P),_) :- !, 
    (noproof -> P1 = [] ; P1 = P),
    passerta(solution(D,X,V1,V2,P1)).

     passertf(reverse,solution(D,((X = Y) :- []),V1,V2,P),_) :- !,
    (noproof -> P1 = [] ; P1 = P),  
    passerta(solution(D,((Y = X) :- []),V1,V2,P1)).

     passertf(false,S,_) :- !.

     passertf(_,_,_).   % just in case

     process_equation(((X = Y) :- []),false) :-
    copy((X = Y),(X2 = Y2)),        % don't save if have
    numbervars((X2 = Y2),0,_),      % rewrite rule already
    (rewrite_rule(X2,Y2) ; rewrite_rule(Y2,X2)),!.

     process_equation(((X = Y) :- []),Save) :- !,
    see(user), tell(user),
    (\+ proof_trace1 -> (print('New equation '),
        print((X = Y)), nl) ; true),
    print('To orient this equation left to right, type l.'),nl,
    print('to orient this equation right to left, type r.'),nl,
    print('to do neither type n., to delete type d.'),nl,
    read(Response),
    (Response = n -> Save = true ;
     (Response = d -> Save = false ;
      (Response = l -> (convert_clause(D,rewrite(X,Y),C),passerta(C),
        abolish(irreducible,1), Save = true)        ;
       (Response = r -> (convert_clause(D,rewrite(Y,X),C),passerta(C),
        abolish(irreducible,1), Save = reverse)     ;
        print('invalid response, try again'),nl,
        process_equation(((X = Y) :- []),Save))))),
    Save \== false.     % fail if delete equation

     process_equation(_,true).

     big_subgoal((L :- B), SIZE) :-
    \+ top_connective(L),
    ((flatsize2(L,FS), FS > SIZE) ;
     (member(X,B), flatsize(X,FX),
      FX > SIZE)), !.

     record_done(D,(X :- B1),I) :- copy((X :- B1),(Y :- B2)),
    ((X :- B1) \== (Y :- B2) -> numbervars((Y :- B2),0,_) ; true),
                    % ground term
    done(E,(Y :- B2)), E >= D,
    !.  % solved more general goal already
     record_done(D,(X :- B1),I) :-
    passert(done(D,(X :- B1))),!.

     back_simplify :-       % delete unsimplified solutions
    maybe_new_rewrites,
    clause(solution(_,S,V1,V2,_),true,Ref), \+ erased(Ref),
    V1 = V2,
    rewrite_solution(S,S2),
    S \== S2, erase(Ref),
    pprint(retracting(S)), fail.

     back_simplify :-       % delete unsimplified rewrites
    maybe_new_rewrites,
    clause(rewrite_rule(L,R),true,Ref), \+ erased(Ref),
    rewrite_atom((L = R), (L2 = R2)),
    (L = R) \== (L2 = R2), erase(Ref),
    pprint(retracting(rewrite_rule(L, R))), fail.

     back_simplify :-       % delete redundant solutions
    clause(solution(D,(L :- B),V1,V2,_),true,Ref), \+ erased(Ref),
    V1=V2,
    copy((L :- B), LC),
    ((L :- B) \== LC -> numbervars((L :- B),0,_) ; true),
            % test for ground term
    clause(solution(E,(L :- B2),W1,W2,_),true,Ref2), \+ erased(Ref2),
    W1=W2,
    (E < D ; (E = D, Ref \== Ref2)),
    append(_,B2,B),     % fewer hypotheses
    erase(Ref),
    pprint(retracting(solution(D,(L :- B)))), fail.

     back_simplify.

     maybe_new_rewrites :-
    rewrites,
    solution(_,((_=_):-[]),_,_,_),!.

     top_connective(X) :-
    nonvar(X), member(X,[or(_),or(_,_),and(_),and(_,_)]), !.

     remove_false(X,X) :- nosplits,!.

     remove_false([not(X)|Y],Z) :- !,remove_false(Y,Z).
     remove_false([X|Y],[X|Z]) :- !,remove_false(Y,Z).
     remove_false([],[]).

     retract_all(X) :- retract(X), fail.
     retract_all(X).

     unit_simplify([L|B],[L1|B1],BS,B1S) :- % delete literals from body.
    already_solved((L1 :- B1),_),!,     % Assume second
    unit_simplify(B,B1,BS,B1S).     % arg is numbervars of first.
                        % return third and fourth args.
     unit_simplify([L|B],[L1|B1],[L|BS],[L1|B1S]) :-
    unit_simplify(B,B1,BS,B1S).

     unit_simplify([],[],[],[]).

     delete_units([L|B],[L1|B1],BS,B1S) :-  % delete from body derived
    \+ nosplits,
    solution(_,(L1 :- []),V1,V2,_), V1=V2, !,
                        % literals.  Assume second
    delete_units(B,B1,BS,B1S).      % arg is numbervars of first.
                        % return third and fourth args.
     delete_units([L|B],[L1|B1],[L|BS],[L1|B1S]) :-
    delete_units(B,B1,BS,B1S).

     delete_units([],[],[],[]).

     unit_delete(BG) :- 
    \+ nosplits,
    \+ nosave,
    member(L,BG),
    negate(L,LN),
    solution(_,(LN :- []),V1,V2,_), V1=V2,
    pprint(unit_delete(BG)),!.

     already_solved((X :- B1), P) :-        % assume X, B1 ground
    solution(D,(X :- B2),V1,V2,P), V1=V2,
    subset(B2,B1),!.
     already_solved((X :- B),transpose(P)) :-
    X \== false, negate(X,Z),
    already_solved((false :- [Z|B]), P).

     already_solved(D,(X :- B1),P) :-   % keep depth info for this one
    solution(E,(X :- B2),V1,V2,P),V1=V2,
    E =< D, subset(B2,B1),
    ((X :- B1) \== (false :- []) ->
     pprint(using_solution(E,(X :- B2))) ; true),
    !.
     already_solved(D,(X :- B),transpose(P)) :-
    X \== false, negate(X,Z),
    already_solved(D,(false :- [Z|B]), P).

% auxiliary functions

     memq(X,[Y|Z]) :- X == Y,!.
     memq(X,[Y|Z]) :- memq(X,Z).

     member(X,Z) :-
    append(_,[X|_],Z).

     split_member(X,Y) :- memq(X,Y),!.

     split_member(X,Y) :- member1(X,Y).

     member1(X,[Y|Z]) :-
    (unify(X,Y) ; member1(X,Z)).

     subset(X,Z) :- var(X), !.      % allows variables at end of lists
     subset([],Z).
     subset([X|Y],Z) :-
    member(X,Z), subset(Y,Z).

     split_subset(X,Z) :- var(X), !.    % allows variables at end of lists
     split_subset([X|Y],Z) :-       % does true unification
    member1(X,Z),
    split_subset(Y,Z).
     split_subset([],Z).

% merge(B1, B2, B3, N, S) adds negative literals of B2 to B1 to produce B3,
% fails if length of B3 is larger than S.  N is effort so far.
% (Allow S unrestricted if N is small, for theorems that can be proven
% mostly by simplification)

     merge(B,[],B,N,S).
     merge(B1,[L|B2],B4,N,S) :-     % preserve order in B2
    merge(B1,B2,B3,N,S),
    merge_lit(B3,L,B4,N,S).
     merge_lit(B3,not(L),B4,N,S) :-
    memq(L,B3),!,fail.      % would create L, not(L) in same body
     merge_lit(B3,not(L),B4,N,S) :- !,  % detect negative literal
    merge_falselit(B3,not(L),B4,N,S).
     merge_lit(B3,L,B3,N,S).        % ignore positive literals
     merge_falselit(B3,L,B3,N,S) :-
    memq(L,B3),!.
     merge_falselit(B3,L,B3,N,S) :-
    member1(L,B3).
     merge_falselit(B3,L,[L|B3],N,S) :-
    (N =< S ;           % allow long splits for short time
     (length(B3,Len), Len < S)),!.  % for theorems that are mostly
                    %  simplification

     atom(not(X),X) :- !.   % remove ``not'' if it is there
     atom(X,X).

     negate(not(X),X) :- !.   % remove ``not'' if there, add if not
     negate(X,not(X)).

     append([], L, L).
     append([X|L1],Y,[X|L2]) :- append(L1, Y, L2).

     flatsize(X,N) :- nosize, !, N = 0.

     flatsize(X,N) :- flatsize2(X,N), !.

     flatsize2(X,N) :-
    var(X),!,N=1.

     flatsize2(X,N) :-
    X =.. [F|L],
    list_size(L,M), N is M + 1.

     list_size([],0).
     list_size([X|Y],N) :-
    flatsize2(X,M1),
    list_size(Y,M2),
    N is M1 + M2.

% routines to do top level replacement and general rewriting

    replace_rewrite(X,Y,YN) :-
    (replacements -> replace(X,X1,XN1),
        ((rewrites,\+ f_chaining) -> rewrite(X1,Y,XN1,XN2) ;
            XN2 = XN1 , Y = X1) ;
     ((rewrites, \+ f_chaining) -> rewrite(X,Y,XN2) ;
        (Y = X , copy(Y,XN2)))),
    (Y == XN2 -> YN = Y ;
        (copy(Y,YN),
         numbervars(YN,0,_))),!.


    replace(X,Z,ZN) :-  % X is input, Z output, ZN possible ground instance
    copy(X,XN),
    (X == XN -> % ground term
     replace(X,Z,XN,ZN) ;
     (numbervars(XN,0,_),
      replace(X,Z,XN,ZN))).

    replace(X,Z,XN,ZN) :-
    replace1(X,Y,XN,YN), !,
    replace(Y,Z,YN,ZN).

    replace(X,X,XN,XN).

    replace1(L,M,LN,MN) :-  % do one step replacing at top level
    (L == LN ->     % like rewrite1 below
        (replace_rule(L,M), copy(M,MN)) ;
        (copy(LN,LNC),
         (LN == LNC ->  % ground instance
          (clause(replace_rule(LN,MN),true,Ref),
           clause(replace_rule(L,M),true,Ref)) ;
          (copy(L,LC),
           numbervars(LC,0,_),
           clause(replace_rule(LC,MN),true,Ref),
           clause(replace_rule(L,M),true,Ref))))).

% In rewrite(L,M,LN,MN), L is input term, M is rewritten term,
% LN is possibly ground instance of L, MN is possibly ground
% instance of M.

    rewrite1(L,M,LN,MN) :-  % do one step rewriting at top level
    (L == LN ->     % ground term
        (rewrite_rule(L,M), copy(M,MN)) ;
        (copy(LN,LNC),
         (LN == LNC ->  % ground instance
          (clause(rewrite_rule(LN,MN),true,Ref),
           clause(rewrite_rule(L,M),true,Ref)) ;
          (copy(L,LC),
           numbervars(LC,0,_),
           clause(rewrite_rule(LC,MN),true,Ref),
           clause(rewrite_rule(L,M),true,Ref))))).

    rewrite_filter(X,X) :- var(X),!.        % can't rewrite a variable

%   rewrite_filter(and(X,Y),and(X,Y)) :- !. % don't rewrite a conjunction,
                    % wait and rewrite subgoals separately

    rewrite_filter(X,X) :-
    \+ top_connective(X),
    copy(X,Y),
    irreducible(Y), !.  % if irreducible, stop.

    rewrite(X,Y,YN) :-
    rewrite_filter(X,Y), !.

    rewrite(X,Y,YN) :-          % add third argument, numbervars'd
    copy(X,XN),         % term
    (X == XN -> % ground term
     rewrite0(X,Y,XN,YN) ;
     (numbervars(XN,0,_),
      rewrite0(X,Y,XN,YN))).

    rewrite(X,Y,XN,YN) :-
    rewrite_filter(X,Y),!,XN = YN.

    rewrite(X,Y,XN,YN) :-
    rewrite0(X,Y,XN,YN).

    rewrite0(X,Z,XN,ZN) :-      % do outermost rewriting
    rewrite1(X,Y,XN,YN),!,
    pprint(rewrite(X,Y)),
    rewrite(Y,Z,YN,ZN).

    rewrite0(X,Z,XN,ZN) :-      % reduce subterms, assert
    rewrite_args(X,Y,XN,YN),!,  % irreducible if so
    (X \== Y -> rewrite2(Y,Z,YN,ZN) ;
        (Y = Z,
        (top_connective(Y) -> true ;
           (copy(Y,W), numbervars(W,0,_),
            passert(irreducible(W)))))).

    rewrite2(X,Z,XN,ZN) :-      % do one rewrite at top level
    rewrite1(X,Y,XN,YN),!,      % then innermost rewriting
    pprint(rewrite(X,Y)),
    rewrite(Y,Z,YN,ZN).

    rewrite2(X,X,XN,XN) :- 
    (top_connective(X) -> true ;
     (copy(X,W),
      numbervars(W,0,_),        % assert irreducible term
      passert(irreducible(W)))).

% routines to rewrite left hand sides of equations

    rewrite1_lhs((X = Y),(Z = Y),ZN) :-  % do one step top level rewriting
    rewrite1(X,Z,ZN),        % of left hand side of equation
    Y \== Z.

    rewrite1_lhs((X = Y),(Z = Y),XN,ZN) :- % do one step top level rewriting
    rewrite1(X,Z,XN,ZN),
    Y \== Z.

    rewrite_lhs_filter(X) :- var(X).    % can't rewrite a variable

    rewrite_lhs_filter(X) :- copy(X,X1), % if irreducible, stop
    irreducible(X1).

    rewrite_lhs((X = Y),(X = Y),XN) :-
    rewrite_lhs_filter(X),!.

    rewrite_lhs((X = Y),E2,ZN) :-
    copy(X,XN),
    (X == XN -> rewrite0_lhs((X = Y),E2,XN,ZN) ;
     (numbervars(XN,0,_),
      rewrite0_lhs((X = Y),E2,XN,ZN))).

    rewrite_lhs((X = Y),(X = Y),XN,XN) :-
    rewrite_lhs_filter(X),!.

    rewrite_lhs(E1,E2,XN,ZN) :-
    rewrite0_lhs(E1,E2,XN,ZN).

    rewrite0_lhs((X = Y),(Z = Y),XN,ZN) :-% do outermost rewriting
    rewrite1_lhs((X = Y),(X1 = Y),XN,X1N), !,
    pprint(rewrite(X,X1)),
    rewrite_lhs((X1 = Y),(Z = Y),X1N,ZN).   % do numbervars again

    rewrite0_lhs((X = Y),(Z = Y),XN,ZN) :- % reduce subterms
    rewrite_args(X,X1,XN,X1N),!,
    (X \== X1 -> rewrite2_lhs((X1 = Y),(Z = Y),X1N,ZN) ;
        X1 = Z).

    rewrite2_lhs((X = Y),(Z = Y),XN,ZN) :-   % do one rewrite at top level
    rewrite1_lhs((X = Y),(X1 = Y),XN,X1N),!, % then outermost rewriting
    pprint(rewrite(X,X1)),
    rewrite_lhs((X1 = Y),(Z = Y),X1N,ZN).

    rewrite2_lhs((X = Y),(X = Y),XN,XN) :- !.   % done rewriting, may not be
                        % irreducible

    rewrite_list(X,Y,YN) :-
    copy(X,XN),
    (X == XN -> rewrite_list(X,Y,XN,YN) ;
     (numbervars(XN,0,_),
      rewrite_list(X,Y,XN,YN))).

    rewrite_list([],[],[],[]).      % rewrite a list of terms
    rewrite_list([X1|Y1],[X2|Y2],[XN1|YN1],[XN2|YN2]) :-
    rewrite(X1,X2,XN1,XN2),
    rewrite_list(Y1,Y2,YN1,YN2),!.

    rewrite_args(X,Y,YN) :-     % add third argument
    copy(X,XN),
    (X == XN -> rewrite_args(X,Y,XN,YN) ;
        (numbervars(XN,0,_),
        rewrite_args(X,Y,XN,YN))).

    rewrite_args(X,Y,XN,YN) :-      % rewrite arguments of a term
    X =.. [F|XL],
    XN =.. [F|XNL],
    rewrite_list(XL,YL,XNL,YNL),
    Y =.. [F|YL],
    YN =.. [F|YNL].

    rewrite_atom((X = Y),(Z = W)) :- !,
    rewrite(Y,W,_),!,
    rewrite_lhs((X = W),(Z = W),_). % treat equations differently

%   rewrite_atom(and(X,Y),and(X,Y)) :- !.

    rewrite_atom(X,Y) :- rewrite_args(X,Y,YN).

    rewrite_literal(X,X) :- var(X), !.

    rewrite_literal(not(X),not(Y)) :- !, rewrite_atom(X,Y).

    rewrite_literal(X,Y) :- rewrite_atom(X,Y).

    rewrite_solution((L :- B), (L1 :- B1)) :-
    rewrite_literal(L,L1),
    rewrite_list(B,B1,_),!.

% linearize_term creates a linearized version of a term i.e.
% linearize_term(p(X,X),p(X,V),[X],[V])
% call at top level by linearize_term(Term,X,Y,Z).

% To make possible fast unification
% using Prolog's built in unification without occurs check.

    linearize_term(Term, Newterm, [], []) :-
    copy(Term, Newterm), Term == Newterm, !.    % ground term

    linearize_term(Term, Newterm, Vars1, Vars2) :-
    linearize(Term, Newterm, [], [], Oldvars, Newvars),
    pairs(Oldvars, Newvars, Vars1, Vars2, Vars3), !.

% Makes use of predicate linearize, with arguments as follows:
% linearize(Input_term, Linearized_version, Previous_oldvars,
%       Previous_linearized_vars, New_oldvars, New_linearized_vars)
% so linearize(p(X,X),p(U,V),[],[],[X,X],[U,V])
% call at top level by linearize(Input_term, Y, [], [], U, V).

     linearize(R,S,O1,O2,[R|O1],[S|O2]) :- var(R), memq(R,O1), !.
    % generate a new variable S after the first time a
    % variable is seen

     linearize(R,R,O1,O2,[R|O1],[R|O2]) :- var(R), !.
    % don't generate a new variable S

     linearize(R,S,Old1,Old2,New1,New2) :-
    R =.. [F|Rlist],
    linearize2(Rlist,Slist,Old1,Old2,New1,New2),
    S =.. [F|Slist].

     linearize2([],[],Old1,Old2,Old1,Old2) :- !.

     linearize2([R|Rlist],[S|Slist],Old1,Old2,New1,New2) :-
    linearize(R,S,Old1,Old2,Mid1,Mid2),!,
    linearize2(Rlist,Slist,Mid1,Mid2,New1,New2).

% pairs takes two lists and finds matching variables.  For
% example, pairs([X,X,Y],[X1,X2,X3],[X1],[X2],[X]).  This
% makes use of the lists returned by linearize.

     pairs([X|L1],[Y|L2],[Y|Z1],[W|Z2],V) :- memq(X,L1), !,
    firstmatch(X,L1,L2,W),
    pairs(L1,L2,Z1,Z2,Z3),
    remove_duplicates([X|Z3],V).

     pairs([X|L1],[Y|L2],Z1,Z2,Z3) :- pairs(L1,L2,Z1,Z2,Z3).

     pairs([],[],[],[],[]).

     firstmatch(X,[Y|L1],[Z|L2],Z) :- X==Y,!.
     firstmatch(X,[Y|L1],[Z|L2],W) :-
    firstmatch(X,L1,L2,W).

     remove_duplicates([X|Y],[U|Y]) :- memq(X,Y), !.
     remove_duplicates(X,X).

     unify_lists([],[]) :- !.       % unify lists of terms
     unify_lists([X|Y],[U|V]) :-
    unify(X,U),
    unify_lists(Y,V).

     make_unifier([],[],true).      % make a unifier for two lists
     make_unifier([X],[Y],unify(X,Y)) :- !.
     make_unifier([X|U],[Y|V],(unify(X,Y), Rest)) :-
    make_unifier(U,V,Rest).

% functions from O'Keefe

     occurs_check(Term, Var) :-
    var(Term), !, Term \== Var.
     occurs_check(Term, Var) :-
    functor(Term, _, Arity),
    occurs_check(Arity, Term, Var).

     occurs_check(0, _, _) :- !.
     occurs_check(N, Term, Var) :-
    arg(N, Term, Arg),
    occurs_check(Arg, Var),
    M is N-1, !,
    occurs_check(M, Term, Var).

     unify(X,Y) :- X == Y, !.               % special case
%    unify(X,Y) :- copy(X,XC), X == XC, !, X = Y.   % ground term
     unify(X,Y) :- unify1(X,Y), !.

     unify1(X, Y) :- var(X), var(Y), !, X = Y.
     unify1(X, Y) :- var(X), !, occurs_check(Y, X), X = Y.
     unify1(X, Y) :- var(Y), !, occurs_check(X, Y), Y = X.
     unify1(X, Y) :- atomic(X), !, X = Y.
     unify1(X, Y) :-
    functor(X, F, N), functor(Y, F, N),
    unify1(N, X, Y).

     unify1(0, X, Y) :- !.
     unify1(N, X, Y) :- arg(N, X, Xn), arg(N, Y, Yn),
    unify1(Xn, Yn),
    M is N-1, !,
    unify1(M, X, Y).

     var_member_chk(Var, [Head|_]) :- Head == Var, !.
     var_member_chk(Var, [_|Tail]) :- var_member_chk(Var, Tail).

     variables_of(Term, Vars) :- variables_of(Term, [], Vars).

     variables_of(Term, Sofar, Sofar) :- var(Term),
    var_member_chk(Term, Sofar), !.
     variables_of(Term, Sofar, [Term|Sofar]) :- var(Term), !.
     variables_of(Term, Sofar, Vars) :-
    functor(Term, _, N),
    variables_of(N, Term, Sofar, Vars).

     variables_of(0, _, Vars, Vars) :- !.
     variables_of(N, Term, Sofar, Vars) :-
    arg(N, Term, Arg),
    variables_of(Arg, Sofar, Mid),
    M is N-1, !,
    variables_of(M, Term, Mid, Vars).

% subsumes tests if Specific is an instance of General, if so
% General is bound to Specific

    subsumes(General, Specific) :-
    variables_of(Specific, Vars),
    subsumes(General, Specific, Vars).

    subsumes(General, Specific, Vars) :-
    var(General),
    var_member_chk(General, Vars),
    !, General == Specific.
    subsumes(General, Specific, Vars) :-
    var(General), !, General = Specific.    % binds
    subsumes(General, Specific, Vars) :-
    nonvar(Specific),
    functor(General, Functionsymbol, Arity),
    functor(Specific, Functionsymbol, Arity),
    subsumes(Arity, General, Specific, Vars).

    subsumes(0, _, _, _) :- !.
    subsumes(N, General, Specific, Vars) :-
    arg(N, General, GenArg),
    arg(N, Specific, SpeArg),
    subsumes(GenArg, SpeArg, Vars),
    M is N-1, !,
    subsumes(M, General, Specific, Vars).

% test for subsumption without binding variables.

    subsumes_chk(General, Specific) :-
    fails_to_subsume(General, Specific), !, fail. %double negation
    subsumes_chk(_,_).

    fails_to_subsume(General, Specific) :-
    subsumes(General, Specific), !, fail. % undo bindings
    fails_to_subsume(_, _).
    
% replace variables by constants of form $VAR(N) for successive
% integers N, do from left to right, yields a ground instance
% of given term
% Call at top level by numbervars(Term, 0, _).

    numbervars('$VAR'(L), L, M) :- !,
    M is L+1.
    numbervars(Term, K, M) :-
    functor(Term, _, N),
    numbervars(0, N, Term, K, M), !.

    numbervars(N, N, Term, M, M) :- !.
    numbervars(I, N, Term, K, M) :-
    J is I+1,
    arg(J, Term, Arg),
    numbervars(Arg, K, L), !,
    numbervars(J, N, Term, L, M).

% make a copy of a term

    copy(Old, New) :-
    asserta(copy(Old)),
    retract(copy(Mid)), !,
    New = Mid.
