/*
 *         Copyright (C) Argonne National Laboratory
 *
 *   Argonne does not guarantee this software in any manner and is
 *   not responsible for any damages that may result from its use.
 *   Furthermore, Argonne does not provide any formal support for this
 *   software.  This is an experimental program.  This software
 *   or any part of it may be freely copied and redistributed,
 *   provided that this paragraph is included in each source file.
 *
 */

/*
 *  process.c -- Routines to handle the processing of generated clauses.
 *
 */

#include "header.h"

#ifndef ROO

/*************
 *
 *    post_process(c, lst) -- finish processing a clause
 *
 *    The clause has already been integrated, indexed, appended to
 *    Sos.  This routine looks for unit conflict, does back subsumption,
 *    and possibly generates more clauses (factoring, back demod, hot
 *    lists, etc.).  Any newly generated and kept clauses will be
 *    appended to lst and will wait their turn to be post_processed.
 *
 *************/

static void post_process(c, lst)
struct clause *c;
struct list *lst;
{
    struct clause *d, *e;
    struct clause_ptr *cp1, *cp2;

    if (Flags[BACK_DEMOD].val) {
        /* c was made into a new demodulator */
	if (c->first_lit != NULL &&
            TP_BIT(c->first_lit->atom->bits, NEW_DEMOD_BIT)) {
	    CLEAR_BIT(c->first_lit->atom->bits, NEW_DEMOD_BIT);
	    d = cl_find(c->id + 1);  /* demod id is 1 more than clause id */
	    if (Flags[PRINT_BACK_DEMOD].val)
		printf(">>>> Starting back demodulation with %d.\n", d->id);
	    CLOCK_START(BACK_DEMOD_TIME)
	    back_demod(d, c, lst);
	    CLOCK_STOP(BACK_DEMOD_TIME)
	    }
	}

    if (Flags[BACK_SUB].val) {
	CLOCK_START(BACK_SUB_TIME)
	cp1 = back_subsume(c);
	CLOCK_STOP(BACK_SUB_TIME)
	while (cp1 != NULL) {
	    e = cp1->c;
	    if (e->container != Passive) {
		Stats[CL_BACK_SUB]++;
		if (Flags[PRINT_BACK_SUB].val)
		    printf("%d back subsumes %d.\n", c->id, e->id);
		un_index_lits_all(e);
		if (e->container == Usable)
		    un_index_lits_clash(e);
		else if (e->container == Sos)
		    Stats[SOS_SIZE]--;
		rem_from_list(e);
		hide_clause(e);
		}
	    cp2 = cp1;
	    cp1 = cp1->next;
	    free_clause_ptr(cp2);
	    }
	}

    if (Flags[FACTOR].val) {
	CLOCK_START(FACTOR_TIME)
	all_factors(c, lst);
	CLOCK_STOP(FACTOR_TIME)
	}

}  /* post_process */


/*************
 *
 *    post_proc_all(lst_pos, lst)
 *
 *************/

void post_proc_all(lst_pos, lst)
struct clause *lst_pos;
struct list *lst;
{
    struct clause *c;

    CLOCK_START(POST_PROC_TIME)
    if (lst_pos == NULL)
	c = lst->first_cl;
    else
	c = lst_pos->next_cl;

    while (c != NULL) {
        post_process(c, lst); /* this may alter c->next_cl */
	c = c->next_cl;
        /* following moved from end of infer_and_process 19 Jan 90 */
	if (Flags[REALLY_DELETE_CLAUSES].val)
	    /* clauses hidden by back demod, back subsumption */
	    /* also empty clauses are hidden */
	    del_hidden_clauses();
	}

    CLOCK_STOP(POST_PROC_TIME)
}  /* post_proc_all */

#endif  /* not ROO */

/*************
 *
 *    infer_and_process(giv_cl)
 *
 *    The inference rules append kept clauses to Sos.  After the
 *    inference rule is finished, the newly kept clauses are
 *    `post_process'ed (unit_conflict, back subsump, etc.).
 *
 *************/

void infer_and_process(giv_cl)
struct clause *giv_cl;
{
    struct clause *c, *sos_pos;
    struct int_ptr *ip;

    if (Flags[CONTROL_MEMORY].val)
	control_memory();
    
    if (Flags[BINARY_RES].val) {
	sos_pos = Sos->last_cl;  /* Save position of last clauses in Sos. */
	bin_res(giv_cl);  /* Inf rule appends newly kept clauses to Sos. */
#ifndef ROO
	/* Now post_process new clauses in Sos. */
	/* (Post_process may append even more clauses to Sos. Do them all.) */
        /* (ROO does not do this.) */
	post_proc_all(sos_pos, Sos);
#endif
	}

    /* For subsequent inference rules, check that the given clause  */
    /*  is still in usable (not back demodulated or back subsumed). */

    if (giv_cl->container == Usable && Flags[HYPER_RES].val) {
	sos_pos = Sos->last_cl;
	hyper_res(giv_cl);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

    if (giv_cl->container == Usable && Flags[NEG_HYPER_RES].val) {
	sos_pos = Sos->last_cl;
	neg_hyper_res(giv_cl);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

    if (giv_cl->container == Usable && Flags[UR_RES].val) {
	sos_pos = Sos->last_cl;
	ur_res(giv_cl);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

    if (giv_cl->container == Usable && Flags[PARA_INTO].val) {
	sos_pos = Sos->last_cl;
	para_into(giv_cl);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

    if (giv_cl->container == Usable && Flags[PARA_FROM].val) {
	sos_pos = Sos->last_cl;
	para_from(giv_cl);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

    if (giv_cl->container == Usable && Flags[LINKED_UR_RES].val) {
	sos_pos = Sos->last_cl;
	linked_ur_res(giv_cl);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

    if (giv_cl->container == Usable && Flags[LINKED_HYPER_RES].val) {
	sos_pos = Sos->last_cl;
	linked_hyper_res(giv_cl);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

    if (giv_cl->container == Usable && Flags[DEMOD_INF].val) {
	sos_pos = Sos->last_cl;
	c = cl_copy(giv_cl);
	ip = get_int_ptr();
	ip->i = giv_cl->id;
	c->parents = ip;
	Stats[CL_GENERATED]++;
	pre_process(c, 0, Sos);
#ifndef ROO
	post_proc_all(sos_pos, Sos);
#endif
	}

}  /* infer_and_process */

/*************
 *
 *    int proc_gen(c, input, demod_flag_ptr)
 *
 *    input -> c is an input clause, and some tests should not be performed.
 *
 *    This routine takes a generated clause and:
 *        -.  (optional) print the clause
 *        -.  (optional) n_resolution check
 *        -.  (if order_eq || lex_order_vars) renumber vars
 *        -.  demodulates it (if any demoulators are present)
 *        -.  handle evaluable literals
 *        -.  (optional) flip equalities
 *        -.  merges identical literals
 *        -.  (optional) sorts literals
 *        -.  (optional & !input) max literals test
 *        -.  if tautology, delete c
 *        -.  (optional & !input) max weight test
 *        -.  (optional & !imput) delete_identical_nested_skolems
 *        -.  (optional) forward subsumption
 *        -.  (optional) unit_deletion
 *        -.  renumber vars if not already done
 *        -.  (optional & !input) max_distinct_vars check
 *
 *    Return 0 if clause should be deleted.
 *
 *************/

int proc_gen(c, input, demod_flag_ptr)
struct clause *c;
int input;
int *demod_flag_ptr;
{
    struct clause *e;
    int wt, i, already_renumbered;


    if (Flags[VERY_VERBOSE].val) {
	printf("\n  ");
	CLOCK_START(PRINT_CL_TIME)
	print_clause(stdout, c);
	CLOCK_STOP(PRINT_CL_TIME)
	}

    if (Flags[N_RESOLUTION].val && !n_resolution_check(c)) {
	Stats[CL_WT_DELETE]++;
	return(0);
	}

    if (Flags[ORDER_EQ].val || Flags[LEX_ORDER_VARS].val) {
	/* renumber variables now, else renumber after forward subsumption */
	if (renumber_vars(c) == 0) {
	    Stats[CL_VAR_DELETES]++;
	    print_clause(stdout, c);
	    return(0);
	    }
	already_renumbered = 1;
	}
    else
	already_renumbered = 0;

    if (Demodulators->first_cl != NULL || Internal_flags[DOLLAR_PRESENT]) {
	CLOCK_START(DEMOD_TIME)
	demod_cl(c);
	CLOCK_STOP(DEMOD_TIME)
	if (Flags[VERY_VERBOSE].val) {
	    printf("  after demodulation: ");
	    CLOCK_START(PRINT_CL_TIME)
            print_clause(stdout, c);
            CLOCK_STOP(PRINT_CL_TIME)
	    }
	/* (variables might no longer start x,y,z,... .) */
	}

    /* False lits of c may be deleted even if test fails. */
    if (Internal_flags[DOLLAR_PRESENT] && lit_t_f_reduce(c)) {
	Stats[CL_TAUTOLOGY]++;
	return(0);
	}

    if (Flags[ORDER_EQ].val) {
	if (Flags[LEX_RPO].val)
	    *demod_flag_ptr = order_equalities_lrpo(c);
	else
	    *demod_flag_ptr = order_equalities(c);
	/* demod_flag is set if c is an eq unit that should be a demodulator */
	/* 0->no, 1->yes, 2->lex_dep */
	/* (variables might no longer start x,y,z,... .) */
	}
    else
	*demod_flag_ptr = 0;

    /* Unit deletion could be placed here, but it seems to work better */
    /* when delayed until after forward subsumption. */

    cl_merge(c);        /* merge identical literals */

    if (Flags[SORT_LITERALS].val)
	sort_lits(c);

    if (!input && Parms[MAX_LITERALS].val != 0) {
	if (num_literals(c) > Parms[MAX_LITERALS].val) {
	    Stats[CL_WT_DELETE]++;
	    return(0);
	    }
	}

    if (tautology(c)) {
	Stats[CL_TAUTOLOGY]++;
	return(0);
	}

    if (!input && Parms[MAX_WEIGHT].val != 0) {
	CLOCK_START(WEIGH_CL_TIME)
	wt = weight_cl(c, Weight_purge_gen_index);
	CLOCK_STOP(WEIGH_CL_TIME)
	if (wt > Parms[MAX_WEIGHT].val) {
            if (Flags[VERY_VERBOSE].val)
		printf("  deleted because weight=%d.\n", wt);
	    Stats[CL_WT_DELETE]++;
	    return(0);
	    }
	}

    if (!input && Flags[DELETE_IDENTICAL_NESTED_SKOLEM].val) {
	if (ident_nested_skolems(c)) {
	    Stats[CL_WT_DELETE]++;
	    return(0);
	    }
	}

    if (Flags[FOR_SUB].val) {
	CLOCK_START(FOR_SUB_TIME)
	e = forward_subsume(c);
	CLOCK_STOP(FOR_SUB_TIME)
	if (e != NULL) {
	    if (Flags[VERY_VERBOSE].val)
		printf("  subsumed by %d.\n", e->id);
	    Stats[CL_FOR_SUB]++;
	    if (e->container == Sos)
		Stats[FOR_SUB_SOS]++;
	    if (e->id < 100)
		Subsume_count[e->id]++;
	    else
		Subsume_count[0]++;
	    return(0);
	    }
	}

    if (Flags[UNIT_DELETION].val && num_literals(c) > 1) {
	CLOCK_START(UNIT_DEL_TIME)
        i = unit_del(c);
	CLOCK_STOP(UNIT_DEL_TIME)
	/* If any unit deletion occurred, and it's now an equality unit,  */
	/* and dynamic_demod is set, call order_equalities (again) to     */
	/* see if it should be a demodulator. */
	if (i && Flags[DYNAMIC_DEMOD].val && num_literals(c) == 1 &&
	                       c->first_lit->atom->varnum == POS_EQ) {
	    if (Flags[LEX_RPO].val)
		*demod_flag_ptr = order_equalities_lrpo(c);
	    else
		*demod_flag_ptr = order_equalities(c);
	    }
	
	}

    if (already_renumbered == 0 && renumber_vars(c) == 0) {
	Stats[CL_VAR_DELETES]++;
	print_clause(stdout, c);
	return(0);
	}

    if (!input && Parms[MAX_DISTINCT_VARS].val != -1) {
	/* this is here becuase vars must be renumbered */
	if (distinct_vars(c) > Parms[MAX_DISTINCT_VARS].val) {
	    Stats[CL_WT_DELETE]++;
	    return(0);
	    }
	}

    return(1);

}  /* proc_gen */

/*************
 *
 *    pre_process(c, input, lst)
 *
 *************/

void pre_process(c, input, lst)
struct clause *c;
int input;
struct list *lst;
{
    int i, demod_flag;
    struct clause *e;

    CLOCK_START(PRE_PROC_TIME)
    i = proc_gen(c, input, &demod_flag);
    if (!i) {
	cl_del_non(c);
	CLOCK_STOP(PRE_PROC_TIME)
	return;
	}

#ifdef ROO

    /* The following kludge is so tha Task b knows where to put kept clause. */

    if (lst == Usable)
	c->id = -1;
    else if (lst == Sos)
	c->id = -2;
    else
	c->id = 0;

    store_in_k(c);

#else  /* not ROO */

    CLOCK_START(KEEP_CL_TIME)
    cl_integrate(c);
    index_lits_all(c);
    if (lst == Usable)
	index_lits_clash(c);
    append_cl(lst, c);
    if (lst == Sos)
	Stats[SOS_SIZE]++;
    c->weight = weight_cl(c, Weight_pick_given_index);
    Stats[CL_KEPT]++;
    CLOCK_STOP(KEEP_CL_TIME)

    if (input || Flags[PRINT_KEPT].val) {
	printf("** KEPT: ");
	CLOCK_START(PRINT_CL_TIME)
	print_clause(stdout, c);
	CLOCK_STOP(PRINT_CL_TIME)
	}

    if (num_literals(c) == 1 && c->first_lit->atom->varnum == POS_EQ) {

	if ((input || Flags[PRINT_NEW_DEMOD].val) &&
	       Flags[DYNAMIC_DEMOD_ALL].val && demod_flag == 0) {
	    printf("++++ cannot make into demodulator: ");
	    print_clause(stdout, c);
	    }

	if (Flags[NEW_FUNCTIONS].val)
	    i = new_function(c, lst);
	else
	    i = 0;

	/* don't try to make into demod if new function introduced */

	if (demod_flag != 0 && Flags[DYNAMIC_DEMOD].val && i == 0)
	    /* Make sure that there are no calls to cl_integrate */
	    /* between KEEP and here, because id of dynamic demodulator */
            /* must be 1 more than the id of the KEPT copy.             */
	    new_demod(c, demod_flag);
	}

    e = check_for_proof(c);

    if (Parms[MAX_PROOFS].val != 0 &&
	Stats[EMPTY_CLAUSES] >= Parms[MAX_PROOFS].val) {
	cleanup();
	exit(0);
	}

#endif

    CLOCK_STOP(PRE_PROC_TIME)

}  /* pre_process */

