/*
 *         Copyright (C) Argonne National Laboratory
 *
 *   Argonne does not guarantee this software in any manner and is
 *   not responsible for any damages that may result from its use.
 *   Furthermore, Argonne does not provide any formal support for this
 *   software.  This is an experimental program.  This software
 *   or any part of it may be freely copied and redistributed,
 *   provided that this paragraph is included in each source file.
 *
 */

/*
 *  misc.c -- Miscellaneous routines.
 *
 */

#include "header.h"

/*************
 *
 *    init() -- initialize global variables
 *
 *************/

void init()
{
    Is_pos_lits = get_is_tree();  /* index for forward subsumption */
    Is_neg_lits = get_is_tree();  /* index for forward subsumption */
    Demod_imd = get_imd_tree();   /* index for demodulation */

    Eq_sym_num = str_to_sn("$eq_infix", 2);
    Cons_sym_num = str_to_sn("$cons", 2);
    Nil_sym_num = str_to_sn("$nil", 0);
    Ignore_sym_num = str_to_sn("$IGNORE", 1);
    Conditional_demodulator_sym_num = str_to_sn("$CONDITIONAL", 3);
    Chr_sym_num = str_to_sn("$CHR", 1);

    clock_init();
    init_options();

}  /* init */

/*************
 *
 *    read_all_input()
 *
 *************/

void read_all_input()
{
    struct list *l;
    struct term *t, *t1;
    struct clause *c, *c2;
    int rc, error, list_errors, i;
    struct formula_ptr *fp;

    CLOCK_START(INPUT_TIME)
    Usable = get_list();
    Sos = get_list();
    Demodulators = get_list();
    Passive = get_list();

    t = read_term(stdin, &rc);
    while (t != NULL || rc == 0) {
	error = 0;
	if (t == NULL)
	    error = 1;
	else if (t->type != COMPLEX)
	    error = 1;
	else if (str_ident("set", sn_to_str(t->sym_num))) {
	    if (change_flag(stdout, t, 1)) {
		print_term(stdout, t); printf(".\n");
		}
	    }
	else if (str_ident("clear", sn_to_str(t->sym_num))) {
	    if (change_flag(stdout, t, 0)) {
		print_term(stdout, t); printf(".\n");
		}
	    }
	else if (str_ident("assign", sn_to_str(t->sym_num))) {
	    if (change_parm(stdout, t)) {
		print_term(stdout, t); printf(".\n");
		}
	    }
	else if (str_ident("list", sn_to_str(t->sym_num))) {
	    t1 = t->farg->argval;
	    if (t1->type == COMPLEX || t->farg->narg != NULL) {
		printf("ERROR, bad argument to list: ");
		print_term(stdout, t); printf(".\n");
		Stats[INPUT_ERRORS]++;
		}
	    else if (str_ident("usable", sn_to_str(t1->sym_num)) ||
		     str_ident("axioms", sn_to_str(t1->sym_num))) {
		if (str_ident("axioms", sn_to_str(t1->sym_num)))
		    fprintf(stderr, "NOTICE: Please change 'axioms' to 'usable'.\n");
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		l = read_cl_list(stdin, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		else if (Flags[PROCESS_INPUT].val == 0) {
		    c = l->first_cl;
		    while (c != NULL) {
			Stats[INPUT_ERRORS] += process_linked_tags(c);
			cl_integrate(c);
			c = c->next_cl;
			}
		    }
		print_cl_list(stdout, l);
		append_lists(Usable,l);
		}
	    else if (str_ident("sos", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		l = read_cl_list(stdin, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		else if (Flags[PROCESS_INPUT].val == 0) {
		    c = l->first_cl;
		    while (c != NULL) {
			cl_integrate(c);
			c = c->next_cl;
			}
		    }
		print_cl_list(stdout, l);
		append_lists(Sos,l);
		}
	    else if (str_ident("demodulators", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		l = read_cl_list(stdin, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		c = l->first_cl;
		while (c != NULL) {
                    if (check_input_demod(c))
		        cl_integrate(c);
		    else {
			Stats[INPUT_ERRORS]++;
			printf("ERROR, bad demodulator: ");
			print_clause(stdout, c);
			}
		    c = c->next_cl;
		    }
		print_cl_list(stdout, l);
		append_lists(Demodulators,l);
		}
	    else if (str_ident("passive", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		l = read_cl_list(stdin, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		c = l->first_cl;
		/* always integrate, becuase never pre_processed */
		while (c != NULL) {
		    cl_integrate(c);
		    c = c->next_cl;
		    }

		print_cl_list(stdout, l);
		append_lists(Passive,l);
		}
	    else {
		if (str_ident("axioms", sn_to_str(t1->sym_num)))
		    fprintf(stderr, "Name of axioms list is now 'usable'.\n");
		printf("ERROR, unknown list: ");
		print_term(stdout, t); printf(".\n");
		l = read_cl_list(stdin, &list_errors);
		print_cl_list(stdout, l);
		Stats[INPUT_ERRORS]++;
		}
	    }
	else if (str_ident("formula_list", sn_to_str(t->sym_num))) {
	    t1 = t->farg->argval;
	    if (t1->type == COMPLEX || t->farg->narg != NULL) {
		printf("ERROR, bad argument to list: ");
		print_term(stdout, t); printf(".\n");
		Stats[INPUT_ERRORS]++;
		}
	    else if (str_ident("usable", sn_to_str(t1->sym_num)) ||
		     str_ident("axioms", sn_to_str(t1->sym_num))) {
		if (str_ident("axioms", sn_to_str(t1->sym_num)))
		    fprintf(stderr, "NOTICE: Please change 'axioms' to 'usable'.\n");
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		fp = read_formula_list(stdin, &list_errors);
		print_formula_list(stdout,fp);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		else {
		    CLOCK_START(CLAUSIFY_TIME)
		    l = clausify_formula_list(fp);
		    CLOCK_STOP(CLAUSIFY_TIME)
		    if (Flags[PROCESS_INPUT].val == 0) {
			c = l->first_cl;
			while (c != NULL) {
			    cl_integrate(c);
			    c = c->next_cl;
			    }
			}
		    printf("\n-------> usable clausifies to:\n\nlist(usable).\n");
		    print_cl_list(stdout, l);
		    append_lists(Usable,l);
		    }
		}
	    else if (str_ident("sos", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		fp = read_formula_list(stdin, &list_errors);
		print_formula_list(stdout,fp);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		else {
		    CLOCK_START(CLAUSIFY_TIME)
		    l = clausify_formula_list(fp);
		    CLOCK_STOP(CLAUSIFY_TIME)
		    if (Flags[PROCESS_INPUT].val == 0) {
			c = l->first_cl;
			while (c != NULL) {
			    cl_integrate(c);
			    c = c->next_cl;
			    }
			}
		    printf("\n-------> sos clausifies to:\n\nlist(sos).\n");
		    print_cl_list(stdout, l);
		    append_lists(Sos,l);
		    }
		}
	    else if (str_ident("passive", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		fp = read_formula_list(stdin, &list_errors);
		print_formula_list(stdout,fp);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		else {
		    CLOCK_START(CLAUSIFY_TIME)
		    l = clausify_formula_list(fp);
		    CLOCK_STOP(CLAUSIFY_TIME)
		    c = l->first_cl;
		    /* always integrate, becuase never pre_processed */
		    while (c != NULL) {
			cl_integrate(c);
			c = c->next_cl;
			}
		    
		    printf("\n-------> passive clausifies to:\n\nlist(passive).\n");
		    print_cl_list(stdout, l);
		    append_lists(Passive,l);
		    }
		}
	    else {
		if (str_ident("axioms", sn_to_str(t1->sym_num)))
		    fprintf(stderr, "Name of axioms list is now 'usable'.\n");
		printf("ERROR, unknown formula_list: ");
		print_term(stdout, t); printf(".\n");
		l = read_cl_list(stdin, &list_errors);
		print_cl_list(stdout, l);
		Stats[INPUT_ERRORS]++;
		}
	    }
	else if (str_ident("weight_list", sn_to_str(t->sym_num))) {
	    t1 = t->farg->argval;
	    if (t1->type != NAME || t->farg->narg != NULL) {
		printf("ERROR, bad argument to Weight_list: ");
		print_term(stdout, t); printf(".\n");
		Stats[INPUT_ERRORS]++;
		}
	    else if (str_ident("purge_gen", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		if (Weight_purge_gen != NULL) {
		    printf("----> ERROR, already have purge weight list.\n");
		    Stats[INPUT_ERRORS] ++;
		    }
		Weight_purge_gen = read_list(stdin, &list_errors, 0);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		Weight_purge_gen_index = get_is_tree();
		set_wt_list(Weight_purge_gen, Weight_purge_gen_index, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		print_list(stdout, Weight_purge_gen);
		}
	    else if (str_ident("pick_given", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		if (Weight_pick_given != NULL) {
		    printf("----> ERROR, already have pick weight list.\n");
		    Stats[INPUT_ERRORS] ++;
		    }
		Weight_pick_given = read_list(stdin, &list_errors, 0);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		Weight_pick_given_index = get_is_tree();
		set_wt_list(Weight_pick_given, Weight_pick_given_index, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		print_list(stdout, Weight_pick_given);
		}
	    else if (str_ident("pick_and_purge", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		if (Weight_pick_given != NULL || Weight_purge_gen != NULL) {
		    printf("----> ERROR, already have pick weight list or purge weight list.\n");
		    Stats[INPUT_ERRORS] ++;
		    }
		Weight_pick_given = Weight_purge_gen = read_list(stdin, &list_errors, 0);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		Weight_pick_given_index = Weight_purge_gen_index = get_is_tree();
		set_wt_list(Weight_pick_given, Weight_pick_given_index, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		print_list(stdout, Weight_pick_given);
		}
	    else if (str_ident("terms", sn_to_str(t1->sym_num))) {
		printf("\n");
		print_term(stdout, t);
		printf(".\n");
		if (Weight_terms != NULL) {
		    printf("----> ERROR, already have term weight list.\n");
		    Stats[INPUT_ERRORS] ++;
		    }
		Weight_terms = read_list(stdin, &list_errors, 0);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		Weight_terms_index = get_is_tree();
		set_wt_list(Weight_terms, Weight_terms_index, &list_errors);
		if (list_errors != 0)
		    Stats[INPUT_ERRORS] += list_errors;
		print_list(stdout, Weight_terms);
		}
	    else {
		printf("ERROR, unknown Weight_list: ");
		print_term(stdout, t); printf(".\n");
		Weight_pick_given = read_list(stdin, &list_errors, 0);
		print_list(stdout, Weight_pick_given);
		Stats[INPUT_ERRORS]++;
		}
	    }
	else if (str_ident("lex", sn_to_str(t->sym_num))) {
	    if (t->farg == NULL || t->farg->narg != NULL || proper_list(t->farg->argval) == 0) {
		printf("ERROR, argument of lex term is not a list: ");
		print_term_nl(stdout, t);
		Stats[INPUT_ERRORS]++;
		}
	    else {
		printf("\n");
	        print_term(stdout, t);
		printf(".\n");
	        set_lex_vals(t);
		}
	    }
	else if (str_ident("lrpo_lr_status", sn_to_str(t->sym_num)) ||
	         str_ident("lrpo_rl_status", sn_to_str(t->sym_num))) {
	    if (t->farg == NULL || t->farg->narg != NULL || proper_list(t->farg->argval) == 0) {
		printf("ERROR, argument of lrpo_status term is not a list: ");
		print_term_nl(stdout, t);
		Stats[INPUT_ERRORS]++;
		}
	    else {
		printf("\n");
	        print_term(stdout, t);
		printf(".\n");
	        if (str_ident("lrpo_lr_status", sn_to_str(t->sym_num)))
		    set_lrpo_status(t, LRPO_LR_STATUS);
		else
		    set_lrpo_status(t, LRPO_RL_STATUS);
		}
	    }
	else if (str_ident("skolem", sn_to_str(t->sym_num))) {
	    if (t->farg == NULL || t->farg->narg != NULL || proper_list(t->farg->argval) == 0) {
		printf("ERROR, argument of skolem term is not a list: ");
		print_term_nl(stdout, t);
		Stats[INPUT_ERRORS]++;
		}
	    else {
		printf("\n");
	        print_term(stdout, t);
		printf(".\n");
	        set_skolem(t);
		}
	    }
	else if (str_ident("special_unary", sn_to_str(t->sym_num))) {
	    if (t->farg == NULL || t->farg->narg != NULL || proper_list(t->farg->argval) == 0) {
		printf("ERROR, argument of special_unary term is not a list: ");
		print_term_nl(stdout, t);
		Stats[INPUT_ERRORS]++;
		}
	    else {
		printf("\n");
	        print_term(stdout, t);
		printf(".\n");
	        set_special_unary(t);
		Internal_flags[SPECIAL_UNARY_PRESENT] = 1;
		}
	    }
	    
	else
	    error = 1;

	if (error) {
	    Stats[INPUT_ERRORS]++;
	    if (t != NULL) {
		printf("ERROR, command not found: ");
		print_term(stdout, t); printf(".\n");
		}
	    }
	if (t != NULL)
	    zap_term(t);
	t = read_term(stdin, &rc);
	}

    CLOCK_STOP(INPUT_TIME)

    if (Stats[INPUT_ERRORS] == 0) {

	dependent_options();
	check_options();

	mark_evaluable_symbols();

	/* index demodulators */

	if (Demodulators->first_cl != NULL && Flags[DEMOD_LINEAR].val == 0) {
	    Demod_imd = get_imd_tree();
	    c = Demodulators->first_cl;
	    while (c != NULL) {
		imd_insert(c, Demod_imd);
		c = c->next_cl;
		}
	    }
	
	/* index <passive list, don't pre_process, even if flag is set */
	
	for (c = Passive->first_cl; c != NULL; c = c->next_cl)
	    index_lits_all(c);
	
	if (Flags[PROCESS_INPUT].val) {
	    CLOCK_START(PROCESS_INPUT_TIME)
	    printf("\n------------> process usable:\n");
	    l = Usable;
	    Usable = get_list();
	    c = l->first_cl;
	    while (c != NULL) {
		c2 = c;
		c = c->next_cl;
		cl_clear_vars(c2);  /* destroy input variable names */
		pre_process(c2, 1, Usable);
		}
	    free_list(l);
	    c2 = NULL;
#ifndef ROO
	    post_proc_all((struct clause *) NULL, Usable);
#endif
	    printf("\n------------> process sos:\n");
	    l = Sos;
	    Sos = get_list();
	    c = l->first_cl;
	    while (c != NULL) {
		c2 = c;
		c = c->next_cl;
		cl_clear_vars(c2);  /* destroy input variable names */
		pre_process(c2, 1, Sos);
		}
	    free_list(l);
	    c2 = NULL;
#ifndef ROO	    
	    post_proc_all((struct clause *) NULL, Sos);
#endif	    
	    printf("\n------------> done processing input.\n");
	    CLOCK_STOP(PROCESS_INPUT_TIME)
	    }
	else {  /* index usable and sos (not passive) */
	    for (c = Usable->first_cl; c != NULL; c = c->next_cl) {
		index_lits_clash(c);
		index_lits_all(c);
		}
	    for (c = Sos->first_cl; c != NULL; c = c->next_cl)
		index_lits_all(c);
	    }
	c = Sos->first_cl;
	for (c = Sos->first_cl, i = 0; c != NULL; c = c->next_cl, i++)
	    if (Flags[INPUT_SOS_FIRST].val)
		c->weight = -MAX_INT;
	    else
		c->weight = weight_cl(c, Weight_pick_given_index);
	Stats[SOS_SIZE] = i;
	}

    fflush(stdout);

}  /* read_all_input */

/*************
 *
 *    set_lex_vals(t) - t is a lex term with a list as its one and only argument.
 *    Set lexical values of the members to 1, 2, 3, ... .
 *
 *************/

void set_lex_vals(t)
struct term *t;
{
    struct rel *r;
    int i;
    struct sym_ent *p;

    for (r = t->farg, i = 1; r->argval->sym_num != Nil_sym_num; r = r->argval->farg->narg, i++) {
	p = sn_to_node(r->argval->farg->argval->sym_num);
	if (p == NULL) {
	    output_stats(stdout, 4);
	    fprintf(stderr, "ABEND, error processing lex_term.\007\n");
	    fprintf(stdout, "ABEND, error processing lex_term: ");
	    print_term_nl(stdout, t);
	    exit(1);
	    }
	else
	    p->lex_val = i;
	}
}  /* set_lex_vals */

/*************
 *
 *    set_lrpo_status(t, val) - t is a lex term with a list as its one and only argument.
 *    Set lrpo_status values of the members to 1.
 *
 *************/

void set_lrpo_status(t, val)
struct term *t;
int val;
{
    struct rel *r;
    struct sym_ent *p;

    for (r = t->farg; r->argval->sym_num != Nil_sym_num; r = r->argval->farg->narg) {
	p = sn_to_node(r->argval->farg->argval->sym_num);
	if (p == NULL) {
	    output_stats(stdout, 4);
	    fprintf(stderr, "ABEND, error processing lrpo_status.\007\n");
	    fprintf(stdout, "ABEND, error processing lrpo_status: ");
	    print_term_nl(stdout, t);
	    exit(1);
	    }
	else
	    p->lex_rpo_status = val;
	}
}  /* set_lrpo_status */

/*************
 *
 *    set_special_unary(t) - t is a lex term with a list as its one and only argument.
 *    Set special_unary values of the members to 1.
 *
 *************/

void set_special_unary(t)
struct term *t;
{
    struct rel *r;
    struct sym_ent *p;

    for (r = t->farg; r->argval->sym_num != Nil_sym_num; r = r->argval->farg->narg) {
	p = sn_to_node(r->argval->farg->argval->sym_num);
	if (p == NULL) {
	    output_stats(stdout, 4);
	    fprintf(stderr, "ABEND, error processing set_special_unary.\007\n");
	    fprintf(stdout, "ABEND, error processing set_special_unary: ");
	    print_term_nl(stdout, t);
	    exit(1);
	    }
	else
	    p->special_unary = 1;
	}
}  /* set_special_unary */

/*************
 *
 *    set_skolem(t) - t is a lex term with a list as its one and only argument.
 *
 *    Set the major function symbol (including constants) of each member of the
 *    list to be a skolem symbol.  (This is called only when skolem symbols
 *    are not created by skolemization by OTTER.)
 *
 *************/

void set_skolem(t)
struct term *t;
{
    struct rel *r;
    struct sym_ent *p;

    for (r = t->farg; r->argval->sym_num != Nil_sym_num; r = r->argval->farg->narg) {
	p = sn_to_node(r->argval->farg->argval->sym_num);
	if (p == NULL) {
	    output_stats(stdout, 4);
	    fprintf(stderr, "ABEND, error processing set_skolem.\007\n");
	    fprintf(stdout, "ABEND, error processing set_skolem: ");
	    print_term_nl(stdout, t);
	    exit(1);
	    }
	else
	    p->skolem = 1;
	}
}  /* set_skolem */

/*************
 *
 *    free_all_mem()
 *
 *************/

void free_all_mem()
{
    struct clause *c;

    c = find_last_cl(Usable);
    while (c != NULL) {
        rem_from_list(c);
	un_index_lits_clash(c);
	un_index_lits_all(c);
	cl_del_int(c);
	c = find_last_cl(Usable);
	}
    free_list(Usable);
    Usable = NULL;

    c = find_last_cl(Sos);
    while (c != NULL) {
	Stats[SOS_SIZE]--;
        rem_from_list(c);
	un_index_lits_all(c);
	cl_del_int(c);
	c = find_last_cl(Sos);
	}
    free_list(Sos);
    Sos = NULL;

    c = find_last_cl(Passive);
    while (c != NULL) {
        rem_from_list(c);
	un_index_lits_all(c);
	cl_del_int(c);
	c = find_last_cl(Passive);
	}
    free_list(Passive);
    Passive = NULL;
    
    c = find_last_cl(Demodulators);
    while (c != NULL) {
        rem_from_list(c);
	if (Flags[DEMOD_LINEAR].val == 0)  /* if imd indexing */
	    imd_delete(c, Demod_imd);
	cl_del_int(c);
	c = find_last_cl(Demodulators);
	}
    free_list(Demodulators);
    Demodulators = NULL;
    
    if (Demod_imd != NULL) {
	free_imd_tree(Demod_imd);
	Demod_imd = NULL;
	}

    /* Weight_purge_gen and Weight_pick_given might point to the same list */

    if (Weight_purge_gen != NULL) {
	weight_index_delete(Weight_purge_gen_index);
	zap_list(Weight_purge_gen);
	if (Weight_purge_gen == Weight_pick_given) {
	    Weight_pick_given = NULL;
	    Weight_pick_given_index = NULL;
	    }
	Weight_purge_gen = NULL;
	Weight_purge_gen_index = NULL;
	}

    if (Weight_pick_given != NULL) {
	weight_index_delete(Weight_pick_given_index);
	zap_list(Weight_pick_given);
	Weight_pick_given = NULL;
	Weight_pick_given_index = NULL;
	}

    if (Weight_terms != NULL) {
	weight_index_delete(Weight_terms_index);
	zap_list(Weight_terms);
	Weight_terms = NULL;
	Weight_terms_index = NULL;
	}
    free_is_tree(Is_pos_lits);
    free_is_tree(Is_neg_lits);
    Is_pos_lits = Is_neg_lits = NULL;

    del_hidden_clauses();
    free_sym_tab();
}  /* free_all_mem */

/*************
 *
 *    output_stats(fp, level) -- print memory, clause, and time stats
 *
 *************/

void output_stats(fp, level)
FILE *fp;
int level;
{
    int i, j;

    if (level == 0)
	;  /* do nothing */
    else {
	if (level == 4)
	    print_options(fp);

	if (level == 1) {
	    print_mem_brief(fp);
	    print_stats_brief(fp);
	    print_times_brief(fp);
	    }
	else {
	    print_mem(fp);
	    print_stats(fp);
	    print_times(fp);
	    if (level >= 3) {
		fprintf(fp, "\nForward subsumption counts, subsumer:number_subsumed.\n");
		for (i = 0; i < 10; i++) {
		    for (j = 1; j < 10; j++)
			fprintf(fp, "%2d:%-4d ", 10*i+j, Subsume_count[10*i+j]);
		    if (i < 9)  /* don't do 100 */
			fprintf(fp, "%2d:%-4d\n", 10*i+10, Subsume_count[10*i+10]);
		    else
			fprintf(fp, "\n");
		    }
		fprintf(fp, "All others: %d.\n", Subsume_count[0]);
		}
	    }
	}
}  /* output_stats */

/*************
 *
 *    print_stats(fp)
 *
 *************/

void print_stats(fp)
FILE *fp;
{
    fprintf(fp, "\n-------------- statistics -------------\n");
    fprintf(fp, "clauses input            %7ld\n", Stats[CL_INPUT]);
    fprintf(fp, "clauses given            %7ld\n", Stats[CL_GIVEN]);
    fprintf(fp, "clauses generated        %7ld\n", Stats[CL_GENERATED]);
  if (Flags[FACTOR].val)
    fprintf(fp, "  (factors generated)    %7ld\n", Stats[FACTORS]);
    fprintf(fp, "demod & eval rewrites    %7ld\n", Stats[REWRITES]);
  if (Parms[MAX_WEIGHT].val != 0 || Parms[MAX_LITERALS].val != 0 || Flags[N_RESOLUTION].val)
    fprintf(fp, "clauses wt,lit,sk delete %7ld\n", Stats[CL_WT_DELETE]);
    fprintf(fp, "tautologies deleted      %7ld\n", Stats[CL_TAUTOLOGY]);
    fprintf(fp, "clauses forward subsumed %7ld\n", Stats[CL_FOR_SUB]);
  if (Flags[ANCESTOR_SUBSUME].val)    
    fprintf(fp, "cl not subsumed due to ancestor_subsume %7ld\n", Stats[CL_NOT_ANC_SUBSUMED]);
    fprintf(fp, "  (subsumed by sos)      %7ld\n", Stats[FOR_SUB_SOS]);
  if (Flags[UNIT_DELETION].val || Flags[LINKED_UNIT_DEL].val)
    fprintf(fp, "unit deletions           %7ld\n", Stats[UNIT_DELETES]);
    fprintf(fp, "clauses kept             %7ld\n", Stats[CL_KEPT]);

#ifdef ROO
    fprintf(fp, "clauses almost kept      %7ld\n", Stats[ROO_ALMOST_KEPT]);
#endif

  if (Flags[DYNAMIC_DEMOD].val)
    fprintf(fp, "new demodulators         %7ld\n", Stats[NEW_DEMODS]);
    fprintf(fp, "empty clauses            %7ld\n", Stats[EMPTY_CLAUSES]);
  if (Flags[BACK_DEMOD].val)
    fprintf(fp, "clauses back demodulated %7ld\n", Stats[CL_BACK_DEMOD]);
    fprintf(fp, "clauses back subsumed    %7ld\n", Stats[CL_BACK_SUB]);

    fprintf(fp, "sos size                 %7ld\n", Stats[SOS_SIZE]);
    fprintf(fp, "Kbytes malloced          %7ld\n", Stats[K_MALLOCED]);

  if (Flags[LINKED_UR_RES].val) {
    fprintf(fp, "linked UR depth hits     %7ld\n", Stats[LINKED_UR_DEPTH_HITS]);
    fprintf(fp, "linked UR deduct hits    %7ld\n", Stats[LINKED_UR_DED_HITS]);
    }

    /* The following are output only if not 0. */
    /* They aren't errors, but they are anomalies. */

    if (Stats[CL_VAR_DELETES] != 0)
	fprintf(fp, "cl deletes, too many vars  %7ld\n", Stats[CL_VAR_DELETES]);
    if (Stats[FPA_OVERLOADS] != 0)
	fprintf(fp, "fpa argument overloads     %7ld\n", Stats[FPA_OVERLOADS]);
    if (Stats[FPA_UNDERLOADS] != 0)
	fprintf(fp, "fpa argument underloads    %7ld\n", Stats[FPA_UNDERLOADS]);

	
}  /* print_stats */

/*************
 *
 *    print_stats_brief(fp)
 *
 *************/

void print_stats_brief(fp)
FILE *fp;
{
    fprintf(fp, "\n-------------- statistics -------------\n");
    fprintf(fp, "clauses generated        %7ld\n", Stats[CL_GENERATED]);
    fprintf(fp, "clauses kept             %7ld\n", Stats[CL_KEPT]);
#ifdef ROO
    fprintf(fp, "clauses almost kept      %7ld\n", Stats[ROO_ALMOST_KEPT]);
#endif
    fprintf(fp, "clauses forward subsumed %7ld\n", Stats[CL_FOR_SUB]);
    fprintf(fp, "clauses back subsumed    %7ld\n", Stats[CL_BACK_SUB]);
}  /* print_stats_brief */

/*************
 *
 *    p_stats()
 *
 *************/

void p_stats()
{
    print_stats(stdout);
}  /* p_stats */

/*************
 *
 *    print_times(fp)
 *
 *************/

void print_times(fp)
FILE *fp;
{
    long t, min, hr;

    fprintf(fp, "\n----------- times (seconds) -----------\n");
    t = run_time();
    fprintf(fp, "run time         %8.2f  ", t / 1000.);
    t = t / 1000; hr = t / 3600; t = t % 3600; min = t / 60; t = t % 60;
    fprintf(fp, "                 (run time  %ld hr, %ld min, %ld sec)\n", hr, min, t); 
    fprintf(fp, "system time      %8.2f\n", system_time() / 1000.);

#ifdef ROO

    fprintf(fp, "task A time      %8.2f\n", clock_val(TASK_A_TIME) /1000.);
    fprintf(fp, "task B time      %8.2f\n", clock_val(TASK_B_TIME) /1000.);
    fprintf(fp, "task C time      %8.2f\n", clock_val(TASK_C_TIME) /1000.);
    fprintf(fp, "task D time      %8.2f\n", clock_val(TASK_D_TIME) /1000.);
    fprintf(fp, "master work time %8.2f\n", clock_val(MASTER_WORK_TIME) /1000.);
    fprintf(fp, "process time nodes%7.2f\n", clock_val(PROCESS_TIME_NODE_LIST) /1000.);

#endif

    fprintf(fp, "input time       %8.2f\n", clock_val(INPUT_TIME) / 1000.);
    fprintf(fp, "  clausify time  %8.2f\n", clock_val(CLAUSIFY_TIME) / 1000.);
  if (Flags[PROCESS_INPUT].val)
    fprintf(fp, "  process input  %8.2f\n", clock_val(PROCESS_INPUT_TIME) / 1000.);
  if (Flags[BINARY_RES].val)
    fprintf(fp, "binary_res time  %8.2f\n", clock_val(BINARY_TIME) / 1000.);
  if (Flags[HYPER_RES].val)
    fprintf(fp, "hyper_res time   %8.2f\n", clock_val(HYPER_TIME) / 1000.);
  if (Flags[NEG_HYPER_RES].val)
    fprintf(fp, "neg_hyper_res time%7.2f\n", clock_val(NEG_HYPER_TIME) / 1000.);
  if (Flags[UR_RES].val)
    fprintf(fp, "UR_res time      %8.2f\n", clock_val(UR_TIME) / 1000.);
  if (Flags[PARA_INTO].val)
    fprintf(fp, "para_into time   %8.2f\n", clock_val(PARA_INTO_TIME) / 1000.);
  if (Flags[PARA_FROM].val)
    fprintf(fp, "para_from time   %8.2f\n", clock_val(PARA_FROM_TIME) / 1000.);
  if (Flags[LINKED_UR_RES].val)
    fprintf(fp, "linked_ur time   %8.2f\n", clock_val(LINKED_UR_TIME) / 1000.);
    fprintf(fp, "pre_process time %8.2f\n", clock_val(PRE_PROC_TIME) / 1000.);
    fprintf(fp, "  demod time     %8.2f\n", clock_val(DEMOD_TIME) / 1000.);
    fprintf(fp, "  weigh cl time  %8.2f\n", clock_val(WEIGH_CL_TIME) / 1000.);
    fprintf(fp, "  for_sub time   %8.2f\n", clock_val(FOR_SUB_TIME) / 1000.);
  if (Flags[UNIT_DELETION].val)
    fprintf(fp, "  unit_del time  %8.2f\n", clock_val(UNIT_DEL_TIME) / 1000.);
    fprintf(fp, "  renumber time  %8.2f\n", clock_val(RENUMBER_TIME) / 1000.);
    fprintf(fp, "  keep cl time   %8.2f\n", clock_val(KEEP_CL_TIME) / 1000.);
    fprintf(fp, "  print_cl time  %8.2f\n", clock_val(PRINT_CL_TIME) / 1000.);
    fprintf(fp, "  conflict time  %8.2f\n", clock_val(CONFLICT_TIME) / 1000.);
    fprintf(fp, "post_process time%8.2f\n", clock_val(POST_PROC_TIME) / 1000.);
  if (Flags[BACK_DEMOD].val)
    fprintf(fp, "  back demod time%8.2f\n", clock_val(BACK_DEMOD_TIME) / 1000.);
    fprintf(fp, "  back_sub time  %8.2f\n", clock_val(BACK_SUB_TIME) / 1000.);
  if (Flags[FACTOR].val)
    fprintf(fp, "  factor time    %8.2f\n", clock_val(FACTOR_TIME) / 1000.);
    fprintf(fp, "lex_rpo time     %8.2f\n", clock_val(LRPO_TIME) / 1000.);
}  /* print_times */

/*************
 *
 *    print_times_brief(fp)
 *
 *************/

void print_times_brief(fp)
FILE *fp;
{

    fprintf(fp, "\n----------- times (seconds) -----------\n");
    fprintf(fp, "run time         %8.2f\n", run_time() / 1000.);
  if (Flags[BINARY_RES].val)
    fprintf(fp, "binary_res time  %8.2f\n", clock_val(BINARY_TIME) / 1000.);
  if (Flags[HYPER_RES].val)
    fprintf(fp, "hyper_res time   %8.2f\n", clock_val(HYPER_TIME) / 1000.);
  if (Flags[NEG_HYPER_RES].val)
    fprintf(fp, "neg_hyper   time %8.2f\n", clock_val(NEG_HYPER_TIME) / 1000.);
  if (Flags[UR_RES].val)
    fprintf(fp, "UR_res time      %8.2f\n", clock_val(UR_TIME) / 1000.);
  if (Flags[PARA_INTO].val)
    fprintf(fp, "para_into time   %8.2f\n", clock_val(PARA_INTO_TIME) / 1000.);
  if (Flags[PARA_FROM].val)
    fprintf(fp, "para_from time   %8.2f\n", clock_val(PARA_FROM_TIME) / 1000.);
  if (Flags[LINKED_UR_RES].val)
    fprintf(fp, "linked_ur time   %8.2f\n", clock_val(LINKED_UR_TIME) / 1000.);
    fprintf(fp, "for_sub time     %8.2f\n", clock_val(FOR_SUB_TIME) / 1000.);
    fprintf(fp, "back_sub time    %8.2f\n", clock_val(BACK_SUB_TIME) / 1000.);
    fprintf(fp, "conflict time    %8.2f\n", clock_val(CONFLICT_TIME) / 1000.);
  if (Demodulators->first_cl != NULL || Internal_flags[DOLLAR_PRESENT])
    fprintf(fp, "demod time       %8.2f\n", clock_val(DEMOD_TIME) / 1000.);
}  /* print_times_brief */

/*************
 *
 *    p_times()
 *
 *************/

void p_times()
{
    print_times(stdout);
}  /* p_times */

/*************
 *
 *    append_lists(l1, l2) -- append l2 to l1 and free the header node l2
 *
 *************/

void append_lists(l1, l2)
struct list *l1;
struct list *l2;
{
    struct clause *c;

    if (l1->last_cl != NULL)  /* if l1 not empty */
        l1->last_cl->next_cl = l2->first_cl;
    else
	l1->first_cl = l2->first_cl;

    if (l2->first_cl != NULL) {  /* if l2 not empty */
	l2->first_cl->prev_cl = l1->last_cl;
	l1->last_cl = l2->last_cl;
	}

    c = l2->first_cl;
    while (c != NULL) {
	c->container = l1;
	c = c->next_cl;
	}

    free_list(l2);
}  /* append_lists */

/*************
 *
 *    struct term *copy_term(term) -- Return a copy of the term.
 *
 *    The bits field is not copied.
 *
 *************/

struct term *copy_term(t)
struct term *t;
{
    struct rel *r, *r2, *r3;
    struct term *t2;

    t2 = get_term();
    t2->type = t->type;
    t2->sym_num = t->sym_num;
    t2->varnum = t->varnum;
    if (t->type != COMPLEX)
	return(t2);
    else {
	r3 = NULL;
	r = t->farg;
	while (r != NULL) {
	    r2 = get_rel();
	    if (r3 == NULL)
		t2->farg = r2;
	    else 
		r3->narg = r2;
	    r2->argval = copy_term(r->argval);
	    r3 = r2;
	    r = r->narg;
	    }
	return(t2);
	}
}  /* copy_term */

/*************
 *
 *    int biggest_var(term)  --  return largest variable number (-1 if none)
 *
 *************/

int biggest_var(t)
struct term *t;
{
    struct rel *r;
    int i, j;

    if (t->type == VARIABLE)
	return(t->varnum);
    else if (t->type == NAME) 
	return(-1);
    else {
	r = t->farg;
	i = -1;
	while (r != NULL) {
	    j = biggest_var(r->argval);
	    if (j > i)
		i = j;
	    r = r->narg;
	    }
	return(i);
	}
}  /* biggest_var */

/*************
 *
 *    zap_list(term_ptr) -- Free a list of nonintegrated terms.
 *
 *************/

void zap_list(p)
struct term_ptr *p;
{
    struct term_ptr *q;

    while (p != NULL) {
	zap_term(p->term);
	q = p;
	p = p->next;
	free_term_ptr(q);
	}
}  /* zap_list */

/*************
 *
 *     int occurs_in(t1, t2) -- Does t1 occur in t2?
 *
 *     term_ident is used to check identity.
 *
 *************/

int occurs_in(t1, t2)
struct term *t1;
struct term *t2;
{
    struct rel *r;

    if (term_ident(t1, t2))
	return(1);
    else if (t2->type != COMPLEX)
	return(0);
    else {
	r = t2->farg;
	while (r != NULL && occurs_in(t1, r->argval) == 0) 
	    r = r->narg;
	return(r != NULL);
	}
}  /* occurs_in */

/*************
 *
 *    int sn_occur(sn, t)
 *
 *    Is sn the sym_num of t or any subterms of t?
 *
 *************/

int sn_occur(sn, t)
int sn;
struct term *t;
{
    struct rel *r;
    int occurs;

    if (t->type != COMPLEX)
	return(t->sym_num == sn);
    else if (t->sym_num == sn)
	return(1);
    else {
	occurs = 0;
	r = t->farg;
	while (r != NULL && occurs == 0) {
	    occurs = sn_occur(sn, r->argval);
	    r = r->narg;
	    }
	return(occurs);
	}
}  /* sn_occur */

/*************
 *
 *    is is_atom(t) -- Is t an atom?
 *
 *    A term is an atom iff it is not a variable and varnum != 0.
 *    (The varnum field of an atom gives its type---equality, answer, evaluable, etc.)
 *
 *************/

int is_atom(t)
struct term *t;
{
    return(t->type != VARIABLE && t->varnum != 0);
}  /* is_atom */

/*************
 *
 *    int id_nested_skolems(t)
 *
 *    Does t or any of its subterms have the identical_nested_skolems property?
 *
 *************/

static int id_nested_skolems(t)
struct term *t;
{
    struct rel *r;
    int occurs;

    if (t->type != COMPLEX)
	return(0);
    else {
	occurs = 0;
	if (is_skolem(t->sym_num)) {
	    r = t->farg;
	    while (r != NULL && occurs == 0) {
		occurs = sn_occur(t->sym_num, r->argval);
		r = r->narg;
		}
	    }
	if (occurs)
	    return(1);
	else {
	    occurs = 0;
	    r = t->farg;
	    while (r != NULL && occurs == 0) {
		occurs = id_nested_skolems(r->argval);
		r = r->narg;
		}
	    return(occurs);
	    }
	}
}  /* id_nested_skolems */

/*************
 *
 *    int ident_nested_skolems(c)
 *
 *    Do any of the terms in clause c have the
 *    identical_nested_skolems property?
 *
 *************/

int ident_nested_skolems(c)
struct clause *c;
{
    struct literal *l;
    int occurs;

    l = c->first_lit;
    occurs = 0;
    while (l != NULL && occurs == 0) {
	occurs = id_nested_skolems(l->atom);
	l = l->next_lit;
	}
    return(occurs);
}  /* ident_nested_skolems */

/*************
 *
 *    int ground(t) -- is a term ground?
 *
 *************/

int ground(t)
struct term *t;
{
    struct rel *r;
    int ok;

    if (t->type == NAME)
	return(1);
    else if (t->type == VARIABLE)
	return(0);
    else { /* COMPLEX */
	ok = 1;
	for (r = t->farg; r != NULL && ok; r = r->narg)
	    ok = ground(r->argval);
	return(ok);
	}
}  /* ground */

/*************
 *
 *    void cleanup()
 *
 *************/

void cleanup()
{
    int i;

    printf("\n------------ END OF SEARCH ------------\n");

    if (Flags[PRINT_LISTS_AT_END].val) {
	printf("\nlist(usable).\n"); print_cl_list(stdout, Usable);
	printf("\nlist(sos).\n"); print_cl_list(stdout, Sos);
	if (Demodulators != NULL) {
	    printf("\nlist(demodulators).\n");
	    print_cl_list(stdout, Demodulators);
	    }
	printf("\n");
	}

#ifndef ROO
    if (Flags[FREE_ALL_MEM].val)
	free_all_mem();
#endif  /* not ROO */

    if (Parms[STATS_LEVEL].val >= 2)
	print_options(stdout);

    output_stats(stdout, Parms[STATS_LEVEL].val);

#ifdef ROO

    print_mem_from_struct(stdout, &(Glob->memory_stats));

    /* print communication lists only if length <= 200 */
    
    printf("\nk_list InDeX:\n");
    for (i = 0; i < K_INDEX_SIZE; i++)
	if (Glob->k_index[i] != NULL) {
	    printf("first of weight %d: ", i);
	    print_clause(stdout, Glob->k_index[i]);
	    }
    
    printf("\nk_list:\n");
    if (Glob->k_length > 200)
	printf("    size=%d.\n", Glob->k_length);
    else
	print_cl_list(stdout, Glob->k_list);

    printf("\nm_list:\n");
    if (Glob->m_length > 200)
	printf("    size=%d.\n", Glob->m_length);
    else
        print_cl_ptr_list(stdout, &Glob->m_list);

    printf("\nn_list:\n");
    if (Glob->n_length > 200)
	printf("    size=%d.\n", Glob->n_length);
    else
	print_cl_ptr_list(stdout, &Glob->n_list);
    
    printf("\nK-list high-water-mark = %d\n", Glob->k_hwm);
    printf("M-list high-water-mark = %d\n", Glob->m_hwm);
    printf("N-list high-water-mark = %d\n", Glob->n_hwm);

#endif  /* ROO */

    printf("The job finished        %s", get_time());

}  /* cleanup */

/*************
 *
 *    int check_stop()  --  Should the search be terminated?
 *
 *    return:
 *        0 if we should not stop;
 *        1 if we should stop because of max_given option;
 *        2 if we should stop because of max_seconds option;
 *        3 if we should stop because of max_gen option;
 *        4 if we should stop because of max_kept option.
 *
 *************/

int check_stop()
{
    long given, seconds, gen, kept;
    int max_given, max_seconds, max_gen, max_kept;

#ifdef ROO
    given = Glob->giv_cl_count;
#else
    given = Stats[CL_GIVEN];
#endif
    seconds = run_time() / 1000;
    gen = Stats[CL_GENERATED];
    kept = Stats[CL_KEPT];

    max_given = Parms[MAX_GIVEN].val;
    max_seconds = Parms[MAX_SECONDS].val;
    max_gen = Parms[MAX_GEN].val;
    max_kept = Parms[MAX_KEPT].val;

    if (max_given != 0 && given >= max_given)
	return(1);
    else if(max_seconds != 0 && seconds >= max_seconds)
	return(2);
    else if (max_gen != 0 && gen >= max_gen)
	return(3);
    else if (max_kept != 0 && kept >= max_kept)
	return(4);
    else
	return(0);
}  /* check_stop */

/*************
 *
 *    report() -- possibly report statistics and times
 *
 *************/

void report()
{
    static int next_report;
    float runtime;

    if (next_report == 0)
	next_report = Parms[REPORT].val;

    runtime = run_time() / 1000.;
    if (runtime >= next_report) {
	printf("\n----- report at %9.2f seconds ----- %s", runtime, get_time());
	output_stats(stdout, Parms[STATS_LEVEL].val);
	fprintf(stderr, "A report (%.2f seconds) has been sent to the output file.\007\n", runtime);
	while (runtime >= next_report)
	    next_report += Parms[REPORT].val;
	}
}  /* report */

/*************
 *
 *    void reduce_weight_limit()
 *
 *************/

void reduce_weight_limit()
{
    int i, new_limit, g;

    /* example:  3975 means at CL_GIVEN==39, reduce limit to 75 */ 

    i = Parms[REDUCE_WEIGHT_LIMIT].val;
    g = i / 100;
    new_limit = i % 100;

    if (Stats[CL_GIVEN] == g) {
	Parms[MAX_WEIGHT].val = new_limit;
	printf("\nreducing weight limit to %d.\n", new_limit);
	}
}  /* reduce_weight_limit */

/*************
 *
 *    void control_memory()
 *
 *************/

void control_memory()
{
    static int next_control_point = 0;
    int sos_distribution[500];
    int i, j, wt, n, control;
    struct clause *c;

    if (Parms[MAX_MEM].val != 0 && total_mem()*3 > Parms[MAX_MEM].val) {
	if (!next_control_point)
	    control = 1;
	else if (next_control_point == Stats[CL_GIVEN])
	    control = 1;
	else
	    control = 0;
	}
    else
	control = 0;

    if (control) {
	next_control_point = Stats[CL_GIVEN] + 10;
	for (i = 0; i < 500; i++)
	    sos_distribution[i] = 0;
	for (c = Sos->first_cl; c; c = c->next_cl) {
	    if (c->weight < 0)
		wt = 0;
	    else if (c->weight >= 500)
		wt = 499;
	    else
		wt = c->weight;
	    sos_distribution[wt]++;
	    }

	i = 0; n = 0;
	while (i < 500 && n*20 <= Stats[SOS_SIZE]) {
	    n += sos_distribution[i];
	    i++;
	    }
	i--;
	
	/* reset weight limit to i */

	if (i < Parms[MAX_WEIGHT].val || Parms[MAX_WEIGHT].val == 0) {
	    Parms[MAX_WEIGHT].val = i;
	    fprintf(stderr, "\007\n\n");
	    printf("\nResetting weight limit to %d.\n\n", i);
	    fprintf(stderr, "\nResetting weight limit to %d.\n\n", i);
	    printf("sos_size=%d\n", Stats[SOS_SIZE]);
	    printf("weight: number of sos clauses with that weight\n");
	    for (j = 0; j < 100; j++)
		printf("%d:  %d\n", j, sos_distribution[j]);
	    }
	}
	
}  /* control_memory */

/*************
 *
 *    proof_message()
 *
 *************/

static void proof_message()
{

#ifdef TP_SUN

    /* system call cuserid (to get username) seems to be on SUN only */

    long i;
    char s[L_cuserid];

    i = run_time();  /* i is milliseconds */
    
    if (i > 10000) {
	/* If more than 10 seconds, print excitedly. (Now that's real AI!) */
	cuserid(s);   /* get username */
	fprintf(stderr, "--- We have a proof, %s, WE HAVE A PROOF!! --", s);
	}
    else
	fprintf(stderr, "---------------- PROOF ----------------");

    fprintf(stderr, "\007\n\n");

#else

    fprintf(stderr, "---------------- PROOF ----------------");
    fprintf(stderr, "\007\n\n");

#endif

}  /* proof_message */

/*************
 *
 *    print_proof(fp, c)
 *
 *************/

void print_proof(fp, c)
FILE *fp;
struct clause *c;
{
    struct clause_ptr *cp1, *cp2, *cp3;

    cp1 = NULL;
    get_ancestors(c, &cp1);
    while (cp1 != NULL) {
	cp3 = cp1->next;
	if (cp3 != NULL && cp3->c->parents != NULL &&
	                   cp3->c->parents->i == NEW_DEMOD_RULE &&
			   cp3->c->parents->next->i == cp1->c->id) {
	    /* skip over dynamic demodulator copy */
	    fprintf(fp, "%d,", cp3->c->id);
	    print_clause(fp, cp1->c);
	    cp2 = cp1;
	    cp1 = cp3->next;
	    free_clause_ptr(cp2);
	    free_clause_ptr(cp3);
	    }
	else {
	    print_clause(fp, cp1->c);
	    cp2 = cp1;
	    cp1 = cp1->next;
	    free_clause_ptr(cp2);
	    }
	}
}  /* print_proof */

/*************
 *
 *    struct clause *check_for_proof(c)
 *
 *************/

struct clause *check_for_proof(c)
struct clause *c;
{
    struct clause *e;
    struct clause_ptr *cp1, *cp2;
    int number_of_lits;

    e = NULL;
    number_of_lits = num_literals(c);
    if (number_of_lits == 0) {
#ifdef ROO
	ALOG_LOG(Pid, 9, c->id, "");
	Glob->proofs_found++;
#endif
        printf("\n-----> EMPTY CLAUSE at %6.2f sec ----> ",
                         run_time() / 1000.);

	print_clause(stdout, c);
	printf("\n");
#ifndef ROO
	if (c->container == Sos)
	    Stats[SOS_SIZE]--;
	Stats[CL_KEPT]--;  /* don't count empty clauses */
	rem_from_list(c);  /* pre_process has already KEPT it */
#endif	
	hide_clause(c);
	Stats[EMPTY_CLAUSES]++;
	e = c;
	if (Flags[PRINT_PROOFS].val) {
	    proof_message();
	    printf("Level of proof is %d, ", proof_level(e)-1);
	    printf("length is %d.\n", proof_length(e)-1);
	    printf("\n---------------- PROOF ----------------\n\n");
	    print_proof(stdout, e);
	    printf("\n------------ end of proof -------------\n\n");
	    fflush(stdout);
	    }
	}
    else if (number_of_lits == 1) {
	CLOCK_START(CONFLICT_TIME)
	cp1 = unit_conflict(c);
	CLOCK_STOP(CONFLICT_TIME)
	while (cp1 != NULL) {  /* empty clause from unit conflict */
            e = cp1->c;
#ifdef ROO
	    ALOG_LOG(Pid, 9, e->id, "");
	    Glob->proofs_found++;
#endif
	    cp2 = cp1->next;
	    free_clause_ptr(cp1);
	    cp1 = cp2;

	    cl_integrate(e);
	    printf("\n----> UNIT CONFLICT at %6.2f sec ----> ",
			 run_time() / 1000.);
	    print_clause(stdout, e);
	    printf("\n");
	    hide_clause(e);
	    if (Flags[PRINT_PROOFS].val) {
		proof_message();
		printf("Level of proof is %d, ", proof_level(e)-1);
		printf("length is %d.\n", proof_length(e)-1);
		printf("\n---------------- PROOF ----------------\n\n");
		print_proof(stdout, e);
		printf("\n------------ end of proof -------------\n\n");
		fflush(stdout);
		}
	    }
	}

    return(e);  /* NULL if no proof was found */


}  /* check_for_proof */

