;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

;;; ---------------------------------------------------------------------
;;; --                                                                 --
;;; --                     THE MVL STATIC VIEWER                       --
;;; --    (and most of the infrastructure for the dynamic viewer)      --
;;; --                                                                 --
;;; ---------------------------------------------------------------------

(in-package "MVL")

;;; This file constains the code that implements the static viewer. A lot
;;; of this code is also used by the dynamic viewer, specially the book
;;; keeping and drawing functions.

;;; The file is organized in the following sections:

;;; 0. Data Structures.
;;; 1. Viewer Set up.
;;; 2. The static display functions.
;;; 3. Book keeping utilities.
;;; 4. Gnode drawing functions.
;;; 5. Scrolling, Repainting and Resizing.
;;; 6. The Menus of the System.
;;; 7. Viewing utilities.
;;; 8. The Overviewer and its functions.
;;; 9. The Magnifier and its function.
;;; 10. Mouse in and out methods for gnodes.
;;; 11. Overviewer Gnode Selection.

;;; [0] DATA STRUCTURES
;;;     ---------------

;;; Global Variables
;;; ----------------

;;; The variable that will hold the viewer (the window where
;;; the proof effort will be displayed).

(defvar *viewer* nil)

;;; The variable holding the overviewer (the small window that shows
;;; a view of the whole tree).

(defvar *overviewer* nil)

;;; The variable holding the magnifier (the window that comes up to
;;; show the complete proposition and truth value of a task when
;;; the user requests it).

(defvar *magnifier* nil)

;;; The menu displayed when the user presses the right button on a 
;;; visible gnode.

(defvar *gnode-menu* nil)

;;; The menu displayed when the user presses the right button on the mat
;;; of the viewer window (on the white surface).

(defvar *viewer-menu* nil)

;;; The menu displayed in the magnifier when the user presses the
;;; right button.

(defvar *magnifier-menu* nil)

;;; The menu displayed in the overviewer during dynamic displaying to
;;; redisplay the whole tree.

(defvar *overviewer-menu* nil)

;;; The menu displayed when the user clicks on a gnode in the overviewer
;;; window.

(defvar *overviewer-gnode-menu* nil)

;;; Each visible task is represented in the viewer by a gnode (for
;;; graphical node). This variable holds the list of all the gnodes 
;;; that are displayed in the viewer (not just the ones showing in
;;; the viewer window).
;;; The list is ordered by the x position of the node in the extent.
;;; This allows us to compute the nodes contained in a region rather
;;; quickly.

(defvar *gnodes* nil)

;;; The propositions that are currently shown in the viewer. Note
;;; that only tasks that pass the display test are members.
;;; A hash table is used for this. We want to find gnodes very
;;; fast. This is important in the dynamic viewer, where we are
;;; given a task to place in the tree and we want to find its
;;; parent very fast.

(defvar *gprops* nil)

;;; A list of the proof efforts that have been shared by two or more
;;; tasks.

(defvar *viewer-reusable-proofs* nil)

;;; This is the region of the extent that the viewer is currently
;;; exposing. It changes with scrolling and resizing.

(defvar *viewer-region* nil)

;;; A list of all the nodes that are visible at the moment. This is
;;; used to determine whether a node should draw to its parents or
;;; not. The idea is to avoid drawing when the object is not in the
;;; visible area of the extent.

(defvar *visible-gnodes* nil)

;;; This variable is a list of gnodes which have been selected by the
;;; user. It is empty when no nodes are selected.

(defvar *selected-gnodes* nil)

;;; These two variables hold the size of the active part of the extent
;;; that contains gnodes. Notice that the extent of the viewer is
;;; never changed from its maximum value (32000 by 32000).

(defvar *viewer-extent-width* 0)
(defvar *viewer-extent-height* 0)

;;; The region that is currently hilighted in the overviewer.

(defvar *overviewer-region* (cw::make-region))

;;; How many real pixels stand for one overviewer pixel. (The overviewer
;;; is a reduced view of the extent).

(defvar *overviewer-x-scale* nil)
(defvar *overviewer-y-scale* nil)

;;; This array records the furthest position to the left which is
;;; occupied at each level. Since the extent's maximum size is 32000
;;; pixels, and the vertical separation of gnodes is 100, we divide
;;; the whole extent into 320 levels of 100 pixels each.

(defvar *viewer-xedge*
    (make-array 320 :element-type 'integer
		:initial-element 200))

;;; This variable is set to t when a gnode is drawn in an x position
;;; to the left of the current edge at the appropriate level (it is
;;; being drawn on top of already existing gnodes).

(defvar *collisions-occured* nil)

;;; The proposition of the task that roots the proof tree displayed.

(defvar *viewer-root-prop* nil)

;;; This variable is used to compute the x coordinates of the left
;;; edge of a tree (the left edge is made of the left most leaf of
;;; each level). We use it to see how far a tree can be shifted to
;;; the left without hitting the next tree.

(defvar *left-edge* nil)

;;; This variable is used to compute the x coordinates of the right
;;; edge of a tree (the right edge is made of the right most leaf of
;;; each level). We use it to see how far a tree can be shifted to
;;; the right without hitting the next tree.

(defvar *right-edge* nil)

;;; If this variable is set to t, dynamic viewing is on.

(defvar *dynamic-viewer-active* nil)

;;; This variable is used to determine whether the user wants to invoke
;;; the dynamic viewer while proving or not. The functions below set this
;;; variable.

(defvar *dynamic-viewing* nil)
(defvar *dynamic-bc-trace*)

(defun dynamic-view () 
  (setf *dynamic-viewing* t)
  (unless *dynamic-bc-trace* (dynamic-bc-trace))
  'DONE)

(defun dynamic-unview () 
  (setf *dynamic-viewing* nil))

;;; This variable can be set to the task currently selected in the
;;; viewer.

(defvar *c* nil)

;;; When this variable is set to t, all events to the viewer and overviewer
;;; are trapped and stoped from causing any function invocation. The functions
;;; below set and unset this variable.

(defvar *viewer-locked* nil)

(defun lock-viewer () (setf *viewer-locked* t))
(defun unlock-viewer () (setf *viewer-locked* nil))

;;; This variable stores the gnode that "roots" the current display of the
;;; viewer. This is the gnode which is closest to the top of the window. Its
;;; purpose is to remember what gnode the user wants to see and to keep the
;;; viewer there through reorganizations and repaintings.

(defvar *viewer-current-root* nil)

;;; These variables are used to remember user requests that take place while
;;; we cannot attend to them because the necessary data structures are under
;;; revision. This is most common during adding a new subtree in dynamic
;;; viewing.

(defvar *static-display-signaled* nil)
(defvar *overviewer-scroll-signaled* nil)

;;; This variable is set to t when we enter dynamic-plant-and-reshape and
;;; to nil when we leave the function. This is necessary because when MVL
;;; breaks during dynamic viewing, the system thinks that it is in the
;;; middle of planting and thus will not attend to user requests (scrolling,
;;; etc) till the end. Thus during the break the viewer is frozen. With this
;;; variable know that if we are not *within-planting* then it is safe to
;;; allow the user to scroll.

(defvar *within-planting* nil)

;;; This array will contain the gnodes at each level sorted by their
;;; x position. It is very important to find out which parts of the
;;; tree lie to the right and below a certain node.

(defvar *viewer-spots-taken*
    (make-array 320 :initial-element nil))

;;; Variables used for determining what tasks should be visible in the
;;; viewer. Functions required to set them follow.

(defvar *static-bc-trace* nil)
(defvar *static-anytime-trace* nil)
(defvar *dynamic-bc-trace* nil)
(defvar *dynamic-anytime-trace* nil)

(defun static-bc-trace (&optional (pat '(?*)))
  (add-trace '*static-bc-trace* pat))

(defun static-bc-untrace (&optional (pat '(?*)))
  (remove-trace '*static-bc-trace* pat))

(defun dynamic-bc-trace (&optional (pat '(?*)))
  (add-trace '*dynamic-bc-trace* pat))

(defun dynamic-bc-untrace (&optional (pat '(?*)))
  (remove-trace '*dynamic-bc-trace* pat))

(defun static-anytime-trace (&optional (pat '(?*)))
  (add-trace '*static-anytime-trace* pat)
  'done)

(defun static-anytime-untrace (&optional (pat '(?*)))
  (remove-trace '*static-anytime-trace* pat))

(defun dynamic-anytime-trace (&optional (pat '(?*)))
  (add-trace '*dynamic-anytime-trace* pat)
  'done)

(defun dynamic-anytime-untrace (&optional (pat '(?*)))
  (remove-trace '*dynamic-anytime-trace* pat))

(defun viewer-static-traced-task (task &aux (prop (consider-prop task)))
  (or (and *static-bc-trace* (traced-prop prop *static-bc-trace*))
      (and *static-anytime-trace* (inference-p task)
	   (traced-prop prop *static-anytime-trace*))))

(defun viewer-dynamic-traced-task (task &aux (prop (consider-prop task)))
  (or (and *dynamic-bc-trace* (traced-prop prop *dynamic-bc-trace*))
      (and *dynamic-anytime-trace* (inference-p task)
	   (traced-prop prop *dynamic-anytime-trace*))))

;;; Global Parameters
;;; -----------------

;;; The function in this variable is called to determine whether a task
;;; is to be shown in the viewer. The default is the function
;;; viewer-traced-task defined above.

(defparameter *display-test* #'mvl::viewer-dynamic-traced-task)

;;; The width and height of the viewer window.

(defparameter *viewer-size* '(800 700))

;;; The width and height of the overviewer window.

(defparameter *overviewer-size* '(500 300))

;;; When this variable is set to t, the viewer's scroll bars
;;; become active and can be used. The overviewer provides
;;; all the functionality required to move in the viewer
;;; conveniently.

(defparameter *viewer-scroll-bars* nil)

;;; The size of the graphic object that is to represent a node in the
;;; tree of the proof being viewed.

(defparameter *gnode-size* '(10 10))

;;; This holds the maximum number of characters which is displayed
;;; in the viewer itself when we write the proposition and the truth
;;; value of a task.

(defparameter *viewer-max-chars* 40)

;;; The horizontal minimum separation of gnodes. The center of two
;;; gnodes on the same level will be at least 250 pixels away.

(defparameter *gnode-separation* 250)

;;; This variable is used to control how big a spread between
;;; sibling gnodes we can tolerate. The value of this variable is
;;; multiplied by *gnode-separation* and if two siblings are further
;;; than the result, we try to get the closer together. This is
;;; used to decide when shifting a sibling to the right is required.

(defparameter *gnode-tolerable-spread* 1)

;;; 2. Structures
;;;    ----------

;;; The gnode represents a graphic object. We try to cache a lot so that
;;; redrawing is fast.

(defstruct (gnode 
	    (:conc-name gn-)
	    (:print-function print-gnode))
  task                              ; task this gnode represents.
  prop                              ; uniquely identifies gnode.
  type                              ; inference, clause or modal.
  prover                            ; the prover invoked for this node.
  dad                               ; the parent gnode not necessarily
                                    ;  the task's parent.
  kids                              ; the children gnodes.
  area                              ; the area the gnode occupies on the
                                    ;  extent.
  actv-reg                          ; the gnode's active region.
  position                          ; the center of the graphic
                                    ;  object in extent coordinates.
  ov-position                       ; position of this gnode in the overviewer.
  prop-text                         ; the proposition of this task converted
                                    ;  to a string shorter than
                                    ;  *viewer-max-chars*.
  prop-pos                          ; the position at which the prop-text
                                    ;  is written.
  truth-text                        ; the truth-value of this task converted
                                    ;  to a string shorter than
                                    ;  *viewer-max-chars*.
  truth-pos                         ; the position at which the truth-text
                                    ;  is written.
  to-dad                            ; position from which to connect to dad.
  to-kids                           ; position from which to connect to kids.
  draw-args                         ; a list of args to which we
                                    ;  apply the drawing procedure
                                    ;  appropriate to each node type.
  hilite-args                       ; a list of args describing the hilite
                                    ;  rectangle we draw when the mouse
				    ;  enters the active region of the gnode.
  drawn-as-active		    ; whether the gnode is being shown as
				    ;  active or dead in the viewer.
  selected                          ; t if the gnode is selected.
  )

(defun print-gnode (gnode stream print-depth)
  (declare (ignore print-depth))
  (format stream "GNODE for ~a" (gn-task gnode)))

;;; The gprop is used to find a gnode fast given its task or
;;; proposition.

(defstruct (gprop (:conc-name gp-))
  task                              ; The task corresponding to the prop.
  gnode)                            ; The corresponding gnode in the
                                    ; viewer.

;;; This structure represents the magnifier window. In it we cache the
;;; positions at which to draw the proposition, the truth value and the
;;; enclosing rectangles.

(defstruct (magnifier (:conc-name mag-))
  window
  task
  prop-area
  prop-pos
  truth-area
  truth-pos
  prop-rect
  truth-rect
  )

;;; FUNCTIONS:
;;; =========

;;; [1] Viewer Set up
;;;     -------------

;;; init-viewer. This function creates the viewer window and
;;;              initializes the data structures we need in the
;;; static viewer display. An important task is to add our
;;; defined methods to the viewer window. We also initialize
;;; the menus. Notice that the viewer window is *not* activated
;;; at this point.

(defun init-viewer (analysis type)
  (let* ((cw::*make-window-stream-actions* nil)
	 (width (car *viewer-size*))
	 (height (cadr *viewer-size*))
	 (bottom (- 32000 height))
	 (left 0)
	 (font (cw::open-font :courier :plain 10
			      :weight :bold))
	 (viewer (cw::make-window-stream
		  :bottom 48
		  :left 34
		  :border-width 4
		  :width (car *viewer-size*)
		  :height (cadr *viewer-size*)
		  :font font
		  :title (format nil "MVL ~a Viewer: ~a"
				 type
				 (analysis-prop analysis)))))
    (when *viewer-scroll-bars*
      (cw::enable-window-stream-extent-scrolling viewer 
						 :vertical t
						 :horizontal t))
    (setf (cw::window-stream-extent-width viewer) 32000)
    (setf (cw::window-stream-extent-height viewer) 32000)
    (setf (cw::window-stream-y-offset viewer) bottom)
    (setf (cw::window-stream-truncate-lines-p viewer) t)
    (cw::modify-window-stream-method viewer 
	   :repaint :after 'viewer-repaint)
    (cw::modify-window-stream-method viewer
	   :right-button-down :after 'general-menu-action)
    (cw::modify-window-stream-method viewer
	   :left-button-down :after 'general-left-button-actions)
    (cw::modify-window-stream-method viewer
	   :reshape-notify :after 'viewer-reshape)
    (setf *viewer-region* (cw::make-region :bottom bottom
				       :left left
				       :width width
				       :height height))
    (init-gnode-menu)
    (init-viewer-menu)
    (init-overviewer-menu)
    (init-overviewer-gnode-menu)
    (setf *viewer-reusable-proofs* (init-reusable-proofs analysis))
    (setf *viewer* viewer)
    (setf cw::*draw-text-background-p* t)
    (setf *gnodes* nil)
    (setf *visible-gnodes* nil)
    (setf *gprops* (make-hash-table :test #'equal))
    (setf *viewer-locked* nil)
    (setf *collisions-occured* nil)
    (setf *selected-gnodes* nil)
    (setf *viewer-xedge*  (make-array 320 :element-type 'integer
				      :initial-element 100))))

;;; init-reusable-proofs. Returns a list of all the proof 
;;;                       invocations that have been shared 
;;; by two or more tasks.

(defun init-reusable-proofs (analysis)
  (let ((all-proofs (analysis-proofs analysis))
	result)
    (dolist (one-proof all-proofs result)
      (when (proof-slaves one-proof)
	(push one-proof result)))))

;;; [2] The Static Display Functions
;;;     ----------------------------

;;; static-display. This is the function seen by the user (the one we
;;;                 export). When called with an analysis of a finished
;;; proof it displays the proof tree. We start by flushing old viewers
;;; and related windows. Then we call plant and reshape to arrange the
;;; nodes into a tree. If no nodes were visible (neither bc-trace nor
;;; anytime-trace) are on, no gnodes are created and the variable
;;; *gnodes* is null. In such a case we report that no nodes were
;;; visible and end. Otherwise, we activate the viewer, resize the
;;; extent of the window to the smallest rectangular region that 
;;; contains all the gnodes, place the viewer so that the root of the
;;; tree is shown, initialize the overviewer and display the whole
;;; tree in it and we are done.

(defun static-display (analysis)
  (flush-viewer-and-related-windows)
  (unless *static-bc-trace* (static-bc-trace))
  (let ((task (analysis-main-task analysis)))
    (init-viewer analysis "Static")
      (lock-viewer)
      (setf *display-test* #'viewer-static-traced-task)
      (plant-and-reshape task)
      (unlock-viewer)
      (if *gnodes*
	  (progn (enter-gnodes-in-spots-taken
		  (prop-find-gnode *viewer-root-prop*) 0)
		 (cw::activate *viewer*)
		 (resize-downstairs-world)
		 (root-on-proposition *viewer-root-prop*)
		 (init-overviewer)
		 (static-overviewer-display 
		  (prop-find-gnode *viewer-root-prop*))
		 (viewer-repaint *viewer*))
	(format t "~%STATIC-DISPLAY: no tasks are visible!")))
  'DONE)

;;; plant-and-reshape. This function directs the tree construction
;;;                    process. There are three things to do: find
;;; out where all the nodes will go (this is the hard part, done by
;;; plant-and-reshape), sort all the gnodes by their x coordinate so
;;; that the scrolling functions work, and anchor the nodes
;;; completely (once we know where a gnode will be placed we can
;;; compile all the other positions, regions, etc that the gnode
;;; needs in order to be redrawn).

(defun plant-and-reshape (task)
  (grow-the-tree task nil 0)
  (sort-gnodes-by-x)
  (when *gnodes*
    (anchor-completely (find-gnode task))))

;;; grow-the-tree. This is the function that arranges the nodes into
;;;                a tree. It tries to build the most compact tree
;;; possible, being constrained by always placing a parent at the
;;; position which is the average of the positions of its first and
;;; last child. The algorithm traverses the tree from top to bottom
;;; but only draws when working backwards to the top (traversal is
;;; top-down and drawing is bottom-up).

;;; Note that "position" refers to the x coordinate only. The y position
;;; is no problem. Also this algorithm always respects the order of the 
;;; children of a task.

;;; The idea is to always draw the leafs at the lefmost available 
;;; position of their level. Then we work backwards, positioning
;;; the parent at the average of the positions of its first and
;;; last children. As we place the parent, we may notice that we are
;;; placing it at a position which is more to the left than the next
;;; available position at its level. At this point, we signal a
;;; collision and its magnitude (how far we are from the next
;;; available position at this level). Eventually we get back to
;;; the calling node which discovers that a collision has occured.
;;; From *collisions-occured* we can determine which was the worst
;;; crash (and by how much we missed) and we fix this by shifting the
;;; crashed subtree to the right by this amount.

;;; But, as we shift a subtree to the right, we separate it from its
;;; siblings that were already planted next to it, when it might be
;;; the case that the subtree did not crash to its siblings but to
;;; a tree further away. So, what we need to do is see if we can keep
;;; the already planted siblings to the shifting subtree.

;;; A description in pseudocode follows:

;;; grow-the-tree task parent-gnode y-level
;;;  if (task is visible) then
;;;     let gnode be the gnode created for task
;;;         children be the inference-children of task
;;;       if (the gnode is a leaf) then
;;;          plant the leaf at the next available position of level
;;;                y-level + 1
;;;       else
;;;          for each inference-child of task
;;;              grow-the-tree child gnode (1+ y-level)
;;;              correct-mistakes gnode y-level
;;;          end for
;;;          plant the gnode at the position which is the average of
;;;           the positions of its first and last children
;;;       end if
;;;  else
;;;   for each inference-child of task
;;;       grow-the-tree child parent-gnode y-level
;;;       correct-mistakes parent-gnode (1- y-level)
;;;   end for
;;;  end if
;;; end grow-the-tree

;;; where correct mistakes is:

;;;  when (collisions have occured)
;;;    shift the tree rooted at child to the left
;;;          by the necessary amount
;;;  when (the gnode just positioned is "too far away"
;;;        from it closest planted sibling)
;;;    shift the subtrees rooted at all its already
;;;          planted siblings to the right as much as
;;;          possible

(defun grow-the-tree (task parent-gnode y-level)
  (if (funcall *display-test* task)
    (let ((gnode (or (find-gnode task)
		     (create-gnode task parent-gnode)))
	  (children (inference-children task)))
      (when parent-gnode (adopt gnode parent-gnode))
      (if (null children) 
	  (plant-a-leaf gnode y-level)
	(dolist (child children (plant-a-what gnode y-level))
	  (grow-the-tree child gnode (1+ y-level))
	  (correct-mistakes gnode y-level))))
    (dolist (child (inference-children task))
      (grow-the-tree child parent-gnode y-level)
      (correct-mistakes parent-gnode (1- y-level)))))

(defun correct-mistakes (gnode y-level) 
  (when gnode
    (when *collisions-occured*
      (shift-tree-by-dx *collisions-occured* (car (gn-kids gnode))
			(1+ y-level)))
    (when (and (> (length (gn-kids gnode)) 1)
	       (> (- (gnode-x-pos (car (gn-kids gnode)))
		     (gnode-x-pos (cadr (gn-kids gnode))))
		  (* *gnode-tolerable-spread* *gnode-separation*)))
      (shift-left-siblings (cdr (gn-kids gnode))
			   (1+ y-level) (list (car (gn-kids gnode)))))))

(defun plant-a-what (gnode y-level)
  (if (gn-kids gnode)
    (plant-a-branch gnode y-level)
    (plant-a-leaf gnode y-level)))

;;; shift-left-siblings. When we shift a tree rooted at gnode to the right
;;;                      due to collisions, the siblings of the shifted
;;; gnode might be left too far away to the left and this may not be 
;;; necessary because the collisions might have been caused by a tree not
;;; rooted at any of them. Thus, this function is called to try to bring
;;; the siblings back together by shifting the already planted ones (the
;;; ones to the left of the current gnode) to the right as much as possible.
;;; We work recursively starting with the sibling closest to the current
;;; gnode and ending with the first sibling planted.

;;; To see how much we can shift each one we compute two lists: the x
;;; coordinates of the rightmost node at each level of the tree rooted
;;; at the first sibling, and the x coordinates of the left most node
;;; at each level of the tree we get by joining all the subtrees rooted
;;; at the already shifted siblings. Once we have the two lists we find
;;; the distance between the edges of the tree at each level and the
;;; smallest of these differences turns out to be the amount by which we
;;; can safely shift the sibling. If this amount is 0, we don't do anything,
;;; otherwise we shift every node in the subtree by the amount.

;;; Note that at the start, already-shifted contains just the current gnode, 
;;; and siblings contains all its siblings already planted. As we go along,
;;; we shift the first sibling, remove it from the siblings list and add it 
;;; to the already-shifted list.

(defun shift-left-siblings (siblings y-level already-shifted)
  (cond ((null siblings) nil)
	(t 
	 (setf *left-edge* nil)
	 (setf *right-edge* nil)
	 (let* ((first-sib (car siblings))
		(left (compile-left-edge-for-kids already-shifted y-level))
		(right (compile-right-edge first-sib y-level))
		(dx (find-dx left right)))
	   (unless (zerop dx)
	     (shift-subtree-by-dx first-sib y-level dx))
	   (shift-left-siblings (cdr siblings)
				y-level 
				(append already-shifted (list first-sib)))
	   ))))

;;; find-dx. Given lists representing the right edge of the tree we want to
;;;          shift (sibling), and the left edge of the tree we have already
;;; shifted (shifter), we find the distance between the edges of the trees
;;; at each level and pick the smallest of these. We return this value which
;;; is the amount by which we can safely shift the sibling to the right.

(defun find-dx (left right)
  (let ((safe-dx 32000)
	lpos rpos)
    (setf left (reverse left))
    (setf right (reverse right))
    (loop
      (when (or (null left) (null right))
	(return-from find-dx safe-dx))
      (setf rpos (cadr (pop right)))
      (setf lpos (cadr (pop left)))
      (setf safe-dx (min safe-dx
			 (- lpos (+ rpos *gnode-separation*)))))))

;;; compile-left-edge. This functions builds a list which holds the
;;;                    x coordinates of the lefmost node at each level
;;; of the tree rooted at gnode. We use the global variable *left-edge*.
;;; Our strategy is to just record the x coordinate of the *first* node
;;; we find at each level.

(defun compile-left-edge (gnode level)
  (unless (assoc level *left-edge*)
    (push (list level (gnode-x-pos gnode)) *left-edge*))
  (compile-left-edge-for-kids (gn-kids gnode) (1+ level)))

;;; compile-left-edge-for-kids. This function calls compile-left-edge for
;;;                             each gnode in the list children. Notice
;;; that we reverse the kids in order that the first kid represents the
;;; leftmost tree (gn-kids holds the children in reverse planting order).
;;; This function was defined (as opposed to being an inline section of
;;; compile-left-edge) because it is needed by shift-left-siblings to
;;; compile the left edge of the subtrees rooted at the already planted
;;; siblings as if it were just one tree.

(defun compile-left-edge-for-kids (children level)
  (dolist (kid (reverse children) *left-edge*)
    (compile-left-edge kid level)))

;;; compile-right-edge. This functions builds a list which holds the
;;;                     x coordinates of the rightmost node at each level
;;; of the tree rooted at gnode. We use the global variable *right-edge*.
;;; Our strategy is to just record the x coordinate of the *first* node
;;; we find at each level. Notice that now, since we do not reverse
;;; gn-kids, the first node we find is the rightmost one.

(defun compile-right-edge (gnode level)
  (unless (assoc level *right-edge*)
    (push (list level (gnode-x-pos gnode)) *right-edge*))
  (compile-right-edge-for-kids (gn-kids gnode) (1+ level)))

;;; Note that the kids should be sorted so that the first one is the
;;; right most one.

(defun compile-right-edge-for-kids (children level)
  (dolist (kid children *right-edge*)
    (compile-right-edge kid level)))
 
;;; shift-tree-by-dx. When we plant a tree and find that collisions have
;;;                   occured, this function is called to shift the tree
;;; to the right to correct the situation. The parameter all-collisions
;;; holds a list of the levels at which collisions occured and the severity
;;; of the collisions (by how much we missed). We pick the worst collision
;;; and shift the whole tree by its amount.

(defun shift-tree-by-dx (all-collisions gnode y-level)
  (let* ((worst-crash (find-worst-crash all-collisions))
	 (dx (+ *gnode-separation* (car worst-crash))))
    (shift-subtree-by-dx gnode y-level dx)
    (setf *collisions-occured* nil)))

;;; shift-subtree-by-dx. This function shifts the subtree rooted at root
;;;                      dx pixels to the right.

(defun shift-subtree-by-dx (root y-level dx)
  (let* ((pos (gn-position root))
	 (x (cw::position-x pos))
	 (y (cw::position-y pos))
	 (new-x (+ x dx)))
    (anchor-gnode root new-x y)
    (when (> new-x (xedge y-level))
      (setf (aref *viewer-xedge* y-level) new-x))
    (dolist (child (gn-kids root))
      (shift-subtree-by-dx child (1+ y-level) dx))))

;;; find-worst-crash. Given a list of the levels and the amounts of
;;;                   the collisions we return the worst one (the one
;;; with the biggest miss).

(defun find-worst-crash (all-collisions)
  (car (sort all-collisions #'> :key #'car)))

;;; plant-a-branch. This function is used to plant a gnode that is not a
;;;                 leaf. This is done *after* its children have been
;;; planted and so, the position of the parent is the average of the
;;; positions of its first and last child. Once we have the x coordinate
;;; we call generic plant to do book keeping.

(defun plant-a-branch (gnode y-level)
  (let ((y (level-to-y y-level))
	(x (avg-kids (gn-kids gnode))))
    (anchor-gnode gnode x y)
    (generic-plant x y-level)))

;;; plant-a-leaf. This function is called to plant a gnode which is a leaf.
;;;               We plant it at the next available position which is
;;; given by looking at the xedge array. Once we have the x coordinate
;;; we call generic plant to do book keeping.

(defun plant-a-leaf (gnode y-level)
  (let ((y (level-to-y y-level))
	(x (+ (xedge y-level) *gnode-separation*)))
    (anchor-gnode gnode x y)
    (generic-plant x y-level)))

;;; generic-plant. After a gnode has been planted at position x and level
;;;                y-level, this function is called to reset the next
;;; available position at y-level and to set the variable *collisions-
;;; occured* if necessary. Note that a collision has occured when the
;;; x coordinate of the new gnode is *less* than the next available
;;; position at the desired level.

(defun generic-plant (x y-level)
  (when (> (+ *gnode-separation* (xedge y-level)) x)
    (push (list (- (xedge y-level) x) y-level) *collisions-occured*))
  (when (> x (xedge y-level))
    (setf (aref *viewer-xedge* y-level) x)))

;;; resize-downstairs-world. After the tree has been drawn we 
;;;                          resize the extent of the viewer
;;; to the smallest rectangular area enclosing the whole tree. This
;;; makes the scroll bars more useful.

(defun resize-downstairs-world ()
  (setf (cw::window-stream-extent-bottom *viewer*)
    (- (level-to-y (find-bottom)) 100))
  (setf (cw::window-stream-extent-height *viewer*)
    (- 32000 (cw::window-stream-extent-bottom *viewer*)))
  (setf (cw::window-stream-extent-width *viewer*)
    (+ (find-max-xedge) 200))
  (setf *viewer-extent-height* (cw::window-stream-extent-height *viewer*))
  (setf *viewer-extent-width* (cw::window-stream-extent-width *viewer*)))

;;; avg-kids. Given a list of children, returns the position where their
;;;           parent should be placed. This position is the average
;;; of the positions of the first and last sibling.

(defun avg-kids (siblings)
  (/ (+ (gnode-x-pos (car siblings))
	(gnode-x-pos (car (last siblings))))
     2))

;;; find-bottom. This function finds the level at which the tree ends. For
;;;              this it traverses the xedge array to find the first level
;;; which still has the initial value (200).

(defun find-bottom ()
  (dotimes (i 320 319)
    (when (< (xedge i) 200)
      (return-from find-bottom (1- i)))))

;;; find-max-xedge. Finds the rightmost position at which a node has been
;;;                 planted in the whole tree.

(defun find-max-xedge ()
  (let ((result 0))
    (dotimes (i 320 result)
      (when (> (xedge i) result)
	(setf result (xedge i))))))

;;; xedge. Given a depth, it returns the next available position at that
;;;        depth in extent coordinates.

(defun xedge (level)
  (aref *viewer-xedge* level))

;;; level-to-y. Given a depth in the tree, this function returns the y
;;;             coordinate at which nodes of this level are drawn in the
;;; extent.

(defun level-to-y (level)
  (- 31900 (* level 100)))

;;; y-to-level. The inverse of level-to-y.

(defun y-to-level (y)
  (/ (- 31900 y) 100))

;;; [3] Book keeping Utilities
;;;     ----------------------

;;; create-gnode. This function creates the gnode data structure. Note
;;;               that this is done before we know where the gnode will
;;; be placed in the viewer, and so, all the fields depending on this
;;; information are not specified. After creation, we enroll the gnode
;;; in the *gnodes* list and in the *gprops* list. The new gnode is
;;; returned. This function is called for the visible tasks only once.

(defun create-gnode (task parent-gnode)
  (let ((gnode (make-gnode
		:task       task
		:prop      (consider-prop task)
		:type      (typecase task
			     (inference 'inference)
			     (modal 'modal)
			     (modal-clause 'modal-clause))
		:prover    (typecase task
			     (inference (inference-proof task))
			     (t nil))
		:dad       parent-gnode
		:prop-text (fit-the-text (consider-prop task))
		:truth-text (fit-the-text (consider-truth-value task))
		:position (cw::make-position)
		:area (cw::make-region))))
    (enroll-gnode gnode)
    (enroll-gprop gnode)
    gnode))

;;; anchor-gnode. This function places the gnode at coordinates x and y.
;;;               Since it just sets the position of the gnode, it is
;;; very cheap! This function is used during the arranging process when
;;; a node is moved a number of times before it is in the final position.

(defun anchor-gnode (gnode x y)
  (let* ((pos (gn-position gnode))
	 (text (gn-prop-text gnode))
	 (to-left (max 50 (1+ (truncate (* 3 (length text))))))
	 (area (gn-area gnode)))
    (setf (cw::position-x pos) x)
    (setf (cw::position-y pos) y)
    (setf (cw::region-width area) (* 2 to-left))
    (setf (cw::region-height area) 100)
    (setf (cw::region-left area) (- x to-left))
    (setf (cw::region-bottom area) (- y 10))))

;;; anchor-completely. Once the arranging process is done, we have to
;;;                    fill in each gnode with the fields related to
;;; its position. This is very expensive since there are lots of regions,
;;; positions and active regions to set, and thus it is done outside the
;;; arranging functions. This function traverses the finished tree in DFS
;;; manner doing a complete anchor of each gnode. The functions that follow
;;; this function do just that and are self explanatory.

(defun anchor-completely (root)
  (let ((children (gn-kids root)))
    (complete-anchor-gnode root)
    (dolist (child children)
      (anchor-completely child))))

(defun complete-anchor-gnode (gnode)
  (let* ((pos (gn-position gnode))
	 (x (cw::position-x pos))
	 (y (cw::position-y pos))
	 (text (gn-prop-text gnode))
	 (to-left (max 50 (1+ (truncate (* 3 (length text))))))
	 (width (* 2 to-left))
	 (height 100)
	 (left (- x to-left))
	 (bottom (- y 10))
	 (region (cw::make-region :bottom bottom
				  :left left
				  :height height
				  :width width)))
    (anchor-gnode-active-region gnode bottom left height width)
    (anchor-gnode-areas-and-positions gnode region x y)
    (anchor-gnode-text-and-hilite gnode)
    (when *dynamic-viewer-active* 
      (check-extent-limits x y)
      (enroll-proof-in-reusable-proofs gnode))
    gnode))

(defun anchor-gnode-areas-and-positions (gnode region x y)
  (setf (gn-area gnode) region)
  (setf (gn-position gnode)
    (cw::make-position :x x :y y))
  (setf (gn-to-dad gnode)
    (cw::make-position :x x :y (+ y 40)))
  (setf (gn-to-kids gnode) 
    (cw::make-position :x x :y (- y (+ (/ (cadr *gnode-size*) 2) 5))))
  (setf (gn-draw-args gnode) (create-draw-args (gn-task gnode) x y)))

(defun anchor-gnode-active-region (gnode bottom left height width)
  (let ((actv (gn-actv-reg gnode)))
    (unless (gn-actv-reg gnode)
      (setf actv
	(cw::make-active-region :left (truncate left) 
				:bottom bottom
				:width width :height height
				:parent *viewer*
				:activate-p t))
      (cw::modify-active-region-method actv :mouse-cursor-in
				       :after 'viewer-mouse-in)
      (cw::modify-active-region-method actv :mouse-cursor-out
				       :after 'viewer-mouse-out)
      (cw::modify-active-region-method actv :button
				       :after 'direct-to-window)
      (setf (gn-actv-reg gnode) actv))
    (setf (cw::active-region-left actv) (truncate left))
    (setf (cw::active-region-bottom actv) bottom)
    (setf (cw::active-region-width actv) width)
    (setf (cw::active-region-height actv) height)))

;;; NOTE: each character drawn is 6 pixels wide.
;;;       This depends on the font size.

(defun anchor-gnode-text-and-hilite (gnode)
  (let* ((to-top (/ (cadr *gnode-size*) 2))
	 (prop (gn-prop-text gnode))
	 (truth (gn-truth-text gnode))
	 (pos (gn-position gnode))
	 (orig-x (cw::position-x pos))
	 (orig-y (cw::position-y pos))
	 x y)
    ;; ABOUT TO ANCHOR THE TRUTH VALUE
    (setf y (+ orig-y to-top 5))
    (setf x (- orig-x (* 3 (length truth))))  ;; why 3? See the note.
    (setf (gn-truth-pos gnode) 
          (cw::make-position :x x :y y))
    ;; ABOUT TO ANCHOR THE PROPOSITION
    (setf y (+ y 13))
    (setf x (- (cw::position-x pos) (* 3 (length prop))))
    (setf (gn-prop-pos gnode) (cw::make-position :x x :y y))
    ;; ABOUT TO ANCHOR THE HILITE AREA
    (setf (gn-hilite-args gnode)
      (list *viewer* 
	    (cw::make-position :x (- orig-x 130) :y (+ orig-y to-top 2))
	    260
	    25))))

;;; fit-the-text. Given a proposition, it reduces it to a string which
;;;               is truncated when it reaches *viewer-max-chars*
;;; characters. First, filter is called to transform the proposition
;;; such that there are no carriage returns, tabs or multiple spaces.
;;; If the string returned is longer than *viewer-max-chars* we truncate
;;; it and add "..." to show the user the proposition has been cut.
;;; Otherwise we just returned the filtered text.

(defun fit-the-text (prop)
  (let ((text (filter (format nil "~a" prop))))
    (if (> (length text) *viewer-max-chars*)
	(concatenate 'string (subseq text 0 (- *viewer-max-chars* 3))
		"...")
      text)))

;;; filter. Given a string, it gets rid of all carriage returns, tabs
;;;         and multiple spaces. The resulting string separates its
;;; tokens by just one space.

(defun filter (string)
  (let ((garbage-bag '(#\space #\tab #\newline))
	(new-string "")
	(start 0)
	(len (length string))
	(i 0))
    (loop
      (if (member (elt string i) garbage-bag)
	  (progn (setf new-string 
		   (concatenate 'string  new-string 
		      (concatenate 'string (sub-string string start i)
				   " ")))
		 (setf i (find-start string i len garbage-bag))
		 (setf start i))
	(incf i))
      (when (= i (length string))
	(return-from filter
	  (concatenate 
	      'string new-string (sub-string string start i)))))))

;;; find-start. Returns the position of the first character in string
;;;             that is not in garbage-bag.

(defun find-start (string i len garbage-bag)
  (loop
    (incf i)
    (when (or (= i len)
	      (not (member (elt string i) garbage-bag)))
      (return-from find-start i))))

;;; sub-string. Returns the substring of string from start to end.

(defun sub-string (string start end)
  (if (= start end)
      ""
    (subseq string start end)))

;;; create-draw-args. Each task type has its own representation (square,
;;;                   circle and triangle. To make redrawing fast, we
;;; compile in advance the list of arguments that the appropriate drawing
;;; function will require (draw-rectangle, draw-circle and draw-triangle
;;; expect different things). When redrawing, we just apply the appropriate
;;; function to these arguments.

(defun create-draw-args (task x y)
  (typecase task
    (inference (create-inference-icon x y))
    (modal (create-modal-icon x y))
    (modal-clause (create-modal-clause-icon x y))))

;;; INFERENCE is represented by a RECTANGLE.

(defun create-inference-icon (x y)
  (list *viewer*
	(cw::make-position :x (- x (/ (car *gnode-size*) 2))
		       :y (- y (/ (cadr *gnode-size*) 2)))
	(car *gnode-size*)
	(cadr *gnode-size*)))

;;; MODAL-CLAUSE is represented by a FILLED CIRCLE.

(defun create-modal-clause-icon (x y)
  (list *viewer*
	(cw::make-position :x x :y y)
	(/ (car *gnode-size*) 2)))

;;; MODAL is represented by a FILLED TRIANGLE

(defun create-modal-icon (x y)
  (list *viewer*
	(cw::make-position :x (- x (/ (car *gnode-size*) 2))
			   :y (+ y (/ (cadr *gnode-size*) 2)))
	(cw::make-position :x (+ x (/ (car *gnode-size*) 2))
			   :y (+ y (/ (cadr *gnode-size*) 2)))
	(cw::make-position :x x
			   :y (- y (/ (cadr *gnode-size*) 2)))))

;;; adopt. This function is called when a child is created. It sets the
;;;        appropriate fields to show that it is a child of its parent.

(defun adopt (gnode parent-gnode)
  (unless (gn-dad gnode)
    (setf (gn-dad gnode) parent-gnode))
  (unless (member gnode (gn-kids parent-gnode))
    (setf (gn-kids parent-gnode) (push gnode (gn-kids parent-gnode)))))

;;; enroll-gnode. Each time a gnode is created it is placed in the
;;;               list *gnodes*. This list is used for scrolling and
;;; redrawing at which point it needs the *gnodes* sorted by x coordinate.
;;; This is done once we are done with the placing process.

(defun enroll-gnode (gnode)
  (when (null *gnodes*)
    (setf *viewer-root-prop* (gn-prop gnode))
    (unless *viewer-current-root* (setf *viewer-current-root* gnode)))
  (unless (member gnode *gnodes*)
    (push gnode *gnodes*)))

;;; sort-gnodes-by-x. Sorts the gnodes in *gnodes* by their x position
;;;                   in ascending order.

(defun sort-gnodes-by-x ()
  (setf *gnodes* (sort *gnodes* #'< :key #'gnode-x-pos)))

;;; gnode-x-pos. Given a gnode it returns its x coordinate.

(defun gnode-x-pos (gnode)
  (cw::position-x (gn-position gnode)))

;;; gnode-y-pos. Given a gnode it returns its y coordinate.

(defun gnode-y-pos (gnode)
  (cw::position-y (gn-position gnode)))

;;; enroll-gprop. Given a gnode, it creates its gprop and enters it into
;;;               the hash table *gprops*.

(defun enroll-gprop (gnode)
  (setf (gethash (gn-prop gnode) *gprops*)
    (cons (make-gprop :task (gn-task gnode) :gnode gnode)
	  (gethash (gn-prop gnode) *gprops*))))

;;; find-gnode. Given a task, we return its gnode. For this we first
;;;             look into *gprops* to give us the all the gnodes that
;;; share the proposition of the task, and then select the right one.

(defun find-gnode (parent-task)
  (let ((subjects (gethash (consider-prop parent-task) *gprops*)))
    (dolist (one-subj subjects nil)
      (when (equal parent-task (gp-task one-subj))
	(return-from find-gnode (gp-gnode one-subj))))))

;;; prop-find-gnode. Like find-gnode but it takes a proposition as its
;;;                  argument. The problem here is that there may be
;;; more than one gnode associated with a proposition (different types
;;; of tasks). If no type is specified we default to inference.
;;; If no inference gnode is found, we return any one that we can find.
;;; If the user specifies a type and it is not found, nil is
;;; returned.

(defun prop-find-gnode (prop &optional (type nil))
  (let ((subjects (gethash prop *gprops*)))
    (if subjects
	(if type
	    (find-of-type subjects type)
	  (let ((result (find-of-type subjects 'inference)))
	    (if result
		result
	      (gp-gnode (car subjects)))))
      nil)))

;;; find-of-type. Given a list of gprops associated to a proposition,
;;;               this function returns the gnode of type type if
;;; found in the list and nil otherwise.

(defun find-of-type (gprops type)
  (dolist (one-subj gprops nil)
    (when (equal type (gn-type (gp-gnode one-subj)))
      (return-from find-of-type (gp-gnode one-subj)))))

;;; find-visible-gnode. Given a position, we look in the list of visible
;;;                     gnodes to find one that includes this position.
;;; If none of the visible gnodes contains the position nil is returned,
;;; otherwise the gnode is returned. This is used to catch mousedown events
;;; in the viewer. If the mouse is on a gnode we pop up the gnode menu.
;;; Otherwise we pop up the viewer menu.

(defun find-visible-gnode (position)
  (dolist (gnode *visible-gnodes* nil)
    (when (cw::region-contains-position-p (gn-area gnode) position)
      (return-from find-visible-gnode gnode))))

;;; enter-gnodes-in-spots-taken. Enters the gnode into the *viewer-spots-
;;;                              taken-array*. This is done only during
;;; static viewing just after we are done planting the whole tree. This
;;; data structure is necessary in the static display to implement the
;;; mousable gnodes in the overviewer.

(defun enter-gnodes-in-spots-taken (root level)
  (cond ((null root) nil)
	(t (setf (aref *viewer-spots-taken* level)
	     (cons root (aref *viewer-spots-taken* level)))
	   (dolist (child (gn-kids root))
	     (enter-gnodes-in-spots-taken child (1+ level))))))
  
;;; [4] Drawing Functions
;;;     -----------------

;;; draw-gnode. This function draws a gnode on the viewer. Notice that
;;;             the gnode itself is only drawn when it is visible in the
;;; viewer. If it is visible and it is not already a member of *visible-gnodes*
;;; it is made one. The we check the type of the gnode and call the
;;; appropriate drawing function for each type. Then we call the functions
;;; connect-to-children and connect-to-parent even if the gnode is not
;;; visible. We do this because its parent and/or children might be visible
;;; in the viewer and the connections should be shown.

(defun draw-gnode (gnode)
  (when (cw::regions-intersect-p *viewer-region* (gn-area gnode))
    (unless (member gnode *visible-gnodes*)
      (push gnode *visible-gnodes*))
    (write-text gnode)
    (case (gn-type gnode)
      (inference (draw-inference-gnode gnode))
      (modal (apply 'cw::draw-filled-triangle
		    (gn-draw-args gnode)))
      (modal-clause (apply 'cw::draw-filled-circle
			   (gn-draw-args gnode)))))
  (connect-to-children gnode)    
  (connect-to-dad gnode))

;;; draw-inference-gnode. Draws an inference gnode. If the inference
;;;                       is active it draws it as a rectangle. If not,
;;; it draws it as a rectangle with and x inside.

(defun draw-inference-gnode (gnode)
  (let ((draw-args (gn-draw-args gnode))
	(task (gn-task gnode)))
    (setf (gn-drawn-as-active gnode) (inference-active-p task))
    (if (inference-active-p task)
	(apply 'cw::draw-rectangle draw-args)
      (let* ((pos (cadr (gn-draw-args gnode)))
	     (x (cw::position-x pos))
	     (y (cw::position-y pos))
	     (xx (+ x (car *gnode-size*)))
	     (yy (+ y (cadr *gnode-size*))))
	(apply 'cw::draw-rectangle draw-args)
	(cw::draw-line *viewer* pos 
		       (cw::make-position :x xx :y yy))
	(cw::draw-line *viewer*
		       (cw::make-position :x x :y yy)
		       (cw::make-position :x xx :y y))))))

(defun inference-active-p (task &aux (p (inference-proof task)))
  (and (proof-p p) (not (completed-proof p))))

;;; connect-to-dad. Connects the gnode to its parent gnode by drawing
;;;                 a line to it. This only happens if it has a parent.

(defun connect-to-dad (gnode)
  (let ((dad (gn-dad gnode)))
    (when dad
      (let ((dad-pos (gn-to-kids dad)))
	(when dad-pos
	  (cw::draw-line *viewer* dad-pos (gn-to-dad gnode)))))))

;;; connect-to-children. Connects the gnode to its children by drawing
;;;                      lines to them.
;;; The children are attatched to the gnode by a small vertical line.


(defun connect-to-children (gnode)
  (let ((children (gn-kids gnode))
	(to-kids (gn-to-kids gnode)))
    (when children
      (cw::draw-line *viewer*
		     to-kids
		     (cw::make-position :x (cw::position-x to-kids)
					:y (+ (cw::position-y to-kids) 5))))
    (dolist (child children)
      (cw::draw-line *viewer*
		     to-kids
		     (gn-to-dad child)))))

;;; write-text. Writes the text associated to a gnode. Everything is
;;;             compiled in advance except the display of the number
;;; of the prover invoked if any. This is only done when the invocation
;;; for this node is shared by other gnodes. If the gnode is currently
;;; selected, we call draw-selected-node-mark to draw the arrow that
;;; signals the connection.

(defun write-text (gnode)
  (let* ((gn-prover (gn-prover gnode))
	 prover proof-pos
	 (pos (gn-position gnode))
	 x y)
    ;; ABOUT TO WRITE THE TRUTH VALUE
    (unless (gn-truth-pos gnode)
      (format t "WRITE-TEXT: calling with NULL truth pos.")
      (setf *c* gnode))
    (cw::set-window-stream-cursor-position *viewer* (gn-truth-pos gnode))
    (format *viewer* (gn-truth-text gnode))
    (finish-output *viewer*)
    ;; ABOUT TO WRITE THE PROPOSITION
    (cw::set-window-stream-cursor-position *viewer* (gn-prop-pos gnode))
    (format *viewer* (gn-prop-text gnode))
    (finish-output *viewer*)
    ;; ABOUT TO WRITE THE PROOF INVOCATION
    (unless (proof-p gn-prover)
      (setf gn-prover 
	(find gn-prover *viewer-reusable-proofs* 
	      :key #'proof-prop :test #'samep)))
    (setf proof-pos (position gn-prover *viewer-reusable-proofs*))
    (when proof-pos
      (setf prover (format nil "~a" proof-pos))
      (setf y (- (cw::position-y pos) 4))
      (setf x (+ (cw::position-x pos) 10))
      (setf (cw::window-stream-x-position *viewer*) x)
      (setf (cw::window-stream-y-position *viewer*) y)
      (format *viewer* prover)
      (finish-output *viewer*))
    ;; ABOUT TO WRITE THAT THE NODE IS SELECTED
    (when (gn-selected gnode)
      (draw-selected-node-mark gnode))))

;;; draw-selected-node-mark. This function is called when gnode is a
;;;                          selected node. It draws an arrow next to
;;; the gnode's icon.

(defun draw-selected-node-mark (gnode)
  (let ((p1 (cw::make-position))
	(p2 (cw::make-position))
	(pos (gn-position gnode)))
    ;; p1 IS THE HEAD OF THE ARROW
    (setf (cw::position-x p1) (- (cw::position-x pos) 10))
    (setf (cw::position-y p1) (cw::position-y pos))
    ;; p2 IS THE TAIL AND WE DRAW IT
    (setf (cw::position-x p2) (- (cw::position-x p1) 10))
    (setf (cw::position-y p2) (cw::position-y pos))
    (cw::draw-line *viewer* p1 p2 :brush-width 2)
    ;; p2 IS THE TWO POINTS THAT MAKE THE HEAD.
    (setf (cw::position-x p2) (- (cw::position-x p1) 3))
    (setf (cw::position-y p2) (+ (cw::position-y pos) 3))
    (cw::draw-line *viewer* p1 p2 :brush-width 2)
    (setf (cw::position-y p1) (+ (cw::position-y p1) 1))
    (setf (cw::position-y p2) (- (cw::position-y p1) 3))
    (cw::draw-line *viewer* p1 p2 :brush-width 2)))

;;; clear-selected-node-mark. This function clears the arrow that shows
;;;                           the user that a gnode is selected. It is
;;; called when the node is deselected.

(defun clear-selected-node-mark (gnode)
  (let* ((x (gnode-x-pos gnode))
	 (y (gnode-y-pos gnode))
	 (arrow-reg (cw::make-region :bottom (- y 4)
				     :left   (- x 21)
				     :width  12
				     :height 8)))
    (cw::clear-area *viewer* arrow-reg)))

;;; [5] Scrolling, Resizing, Repainting
;;;     -------------------------------

;;; viewer-repaint. Called each time the viewer's repaint method is
;;;                 invoked. If no region is provided, we assume that
;;; the whole window is to be repainted. A region is provided when
;;; the repainting is due to scrolling that exposes a new area of the
;;; extent. We reset the *viewer-region* variable, clear the area to
;;; be repainted, redraw the area, and hilite the new area shown in the
;;; viewer in the overviewer.

(defun viewer-repaint (ws &optional region)
  (unless *viewer-locked*
    (unless region
      (setf region 
	(cw::make-region
	 :bottom (cw::window-stream-y-offset ws)
	 :left   (cw::window-stream-x-offset ws)
	 :width  (cw::window-stream-width ws)
	 :height (cw::window-stream-height ws))))
    (reset-viewer-region ws)
    (viewer-clear ws region)
    (viewer-redraw region)
    (if *dynamic-viewer-active*
	(dynamic-hilite-visible-region)
      (hilite-visible-region))))

;;; reset-viewer-region. Updates *viewer-region* to represent the
;;;                      area of the extent that is currently
;;; visible in the viewer.

(defun reset-viewer-region (viewer)
  (setf (cw::region-bottom *viewer-region*)
    (cw::window-stream-y-offset viewer))
  (setf (cw::region-left *viewer-region*)
    (cw::window-stream-x-offset viewer)))

;;; viewer-clear. Clears the region region in the viewer. Since the region
;;;               exposed by the viewer might have changed, we call
;;; reset-visible-gnodes to find out which of the gnodes which were
;;; visible in the old exposed region are still visible after the change.

(defun viewer-clear (ws region)
  (when region
      (cw::clear-area ws region))
  (reset-visible-gnodes))

;;; reset-visible-gnodes. Removes from *visible-gnodes* all the
;;;                       gnodes that were visible before the change
;;; in region but which are not visible anymore.

(defun reset-visible-gnodes ()
  (setf *visible-gnodes*
    (delete-if-not #'(lambda (gnode)
		       (cw::regions-intersect-p 
			*viewer-region* (gn-area gnode)))
		   *visible-gnodes*)))

;;; viewer-redraw. Redraws the region of the viewer passed in region.
;;;                We find all gnodes that belong to the
;;; new region and draw them. After that, we find all the gnodes
;;; the are located in the currently visible region, and also one screen
;;; to the left and to the right. Then we connect all these gnodes to their
;;; parents and children. This is necessary because although a gnode might
;;; not show in the current viewer region, its connections to parent and
;;; children might cross the current viewer region. Notice that the
;;; width of the area we check is 3 screens. This means that if a parent
;;; and a child are more than 3 screens away, the connection might be
;;; momentarily lost while scrolling.

(defun viewer-redraw (region)
  (let* ((x (cw::region-left region))
	 (gnodes (collect-exposed-gnodes 
		  region 
		  (- x 100)
		  (+ x (cw::region-width region) 100))))
    (dolist (oneNode gnodes)
      (draw-gnode oneNode))
    (setf (cw::region-height region) (cw::region-height *viewer-region*))
    (setf (cw::region-bottom region) (cw::region-bottom *viewer-region*))
    (setf (cw::region-width region) (* (cw::region-width *viewer-region*) 3))
    (setf (cw::region-left region) (- (cw::region-left *viewer-region*) 
				      (cw::region-width *viewer-region*)))
    (setf x (cw::region-left region))
    (setf gnodes (collect-exposed-gnodes 
		  region 
		  (- x 100)
		  (+ x (cw::region-width region) 100)))
    (dolist (oneNode gnodes)
      (connect-to-dad oneNode)
      (connect-to-children oneNode))))
	  
;;; collect-exposed-gnodes. Finds all the gnodes that are located within
;;;                         region. The region we are interested in
;;; starts at xstart and ends at xend.

(defun collect-exposed-gnodes (region xstart xend)
  (let ((xgnodes (collect-relevant-x *gnodes* xstart xend))
	result)
    (if xgnodes
	(dolist (oneXgnode xgnodes result)
	  (when (cw::regions-intersect-p region (gn-area oneXgnode))
	    (push oneXgnode result)))
      nil)))

;;; collect-relevant-x. Returns all the gnodes that are located within
;;;                     x coordinates xstart and xend, regardless of
;;; their vertical possition.

(defun collect-relevant-x (gnodes xstart xend)
  (let (result)
    (dolist (gnode gnodes result)
      (if (> (gnode-x-pos gnode) xend)
	  (return-from collect-relevant-x result)
	(when (> (gnode-x-pos gnode) xstart)
	    (push gnode result))))))

;;; viewer-reshape. This function is called when the viewer is resized
;;;                 by the user and the reshape method is invoked.
;;; Resizing works merely by changing the width and height fields of the
;;; window-stream structure. The problem with this is that the image
;;; we are currently viewing unexpectedly jumps to another location as we
;;; resize the window. This function will compute the amount by which
;;; the image shifts and reposition it at its original position.

(defun viewer-reshape (ws region)
  (declare (ignore region))
  (unless *viewer-locked*
    (let* ((old-bottom (cw::region-bottom *viewer-region*))
	   (old-width (cw::region-width *viewer-region*))
	   (old-height (cw::region-height *viewer-region*))
	   (old-left (cw::region-left *viewer-region*))
	   (r1-bottom (+ old-bottom old-height))
	   (r1-left (cw::region-left *viewer-region*))
	   (r1-height (- (cw::window-stream-height ws) old-height))
	   (r1-width (cw::window-stream-width ws))
	   (r2-bottom old-bottom)
	   (r2-left (+ r1-left old-width))
	   (r2-height old-height)
	   (r2-width (- (cw::window-stream-width ws) old-width))
	   (r1 (cw::make-region :bottom r1-bottom :left r1-left
				:width r1-width :height r1-height))
	   (r2 (cw::make-region :bottom r2-bottom :left r2-left
				:width r2-width :height r2-height)))
      (resize-viewer-region ws)
      (reset-visible-gnodes)
      (viewer-clear ws r1)
      (viewer-clear ws r2)
      (viewer-redraw r1)
      (viewer-redraw r2)
      (cw::scroll *viewer* 
		  (cw::make-region 
		   :left old-left
		   :bottom (- r1-bottom 
			      (cw::region-height *viewer-region*)))))))
  
;;; resize-viewer-region. When the window is resized, this function is
;;;                       called to reflect the changes in *viewer-region*.

(defun resize-viewer-region (ws)
  (setf (cw::region-width *viewer-region*)
    (cw::window-stream-width ws))
  (setf (cw::region-height *viewer-region*)
    (cw::window-stream-height ws)))
      
;;; [6] Menus
;;;     -----

;;; init-gnode-menu. Creates the gnode menu which pops up when the uses
;;;                  triggers a mouse down event on a gnode.

(defun init-gnode-menu ()
  (let ((item1 (cw::make-menu-item "Inspect [L]" 'inspect))
	(item2 (cw::make-menu-item "Root" 'root))
	(item3 (cw::make-menu-item "Center" 'center))
	(item4 (cw::make-menu-item 
		"Display in Listener [S-L]" 'listener-display))
	(item5 (cw::make-menu-item "Display in Separate Window [C-L]"
				   'window-display))
	(item6 (cw::make-menu-item "Setf *C* to Object"
				   'set-to-c)))
    (setf *gnode-menu* 
	  (cw::make-menu :items (list item1 item2 item3 item4 item5
				      item6)))))

;;; general-menu-action. When a mouse down event is cought in the viewer
;;;                      we need to determine whether it is a gnode event
;;; or a general event (not on a gnode). For this we find the position of
;;; the mouse and find whether this position is contained by the area of
;;; one of the gnodes currently visible. If so, we pop up the gnode menu,
;;; asnd otherwise we pop up the viewer menu.

(defun general-menu-action (ws mouse-state &optional event)
  (declare (ignore event))
  (unless *viewer-locked*
    (let* ((position (cw::mouse-state-position mouse-state))
	   (gnode (find-visible-gnode position)))
      (if gnode
	  (gnode-menu-action gnode ws)
	(viewer-menu-action ws position)))))

;;; gnode-menu-action. Pops up the gnode menu and handles the returned
;;;                    value by calling appropriate functions.

(defun gnode-menu-action (gnode ws)
  (let ((menu-action (cw::pop-up-menu *gnode-menu*
				      :position (gn-position gnode)
				      :window ws)))
    (case menu-action
      (inspect (progn (setf *package* (find-package "MVL"))
		      (wt:winspect (gn-task gnode))))
      (root    (root-on-proposition nil 
				    :task-type nil 
				    :gnode gnode))
      (center  (center-on-proposition nil
				      :task-type nil
				      :gnode gnode))
      (listener-display (expand-text gnode))
      (window-display (magnifier-display gnode))
      (set-to-c (setf *c* (gn-task gnode))))))
  
;;; expand-text. Prints in the listener the proposition and truth value of
;;;              the current gnode.

(defun expand-text (gnode)
  (format t "~2%~a" (gn-prop gnode))
  (format t "~%~a" (consider-truth-value (gn-task gnode))))

;;; init-viewer-menu. Creates the general viewer menu which is poped
;;;                   up when the user clicks on an area of the viewer
;;; not occupied by a gnode.

(defun init-viewer-menu ()
  (let ((item1 (cw::make-menu-item "Display Root [C-L]" 'Root))
	(item2 (cw::make-menu-item "Repaint [L]" 'repaint))
	(item3 (cw::make-menu-item "Flush" 'flush))
	(item4 (cw::make-menu-item "Show Overviewer [Sh-L]" 'overviewer)))
    (setf *viewer-menu* (cw::make-menu 
			 :items (list item1 item4 item2 item3)))))

;;; viewer-menu-action. Pops up the viewer menu and handles the returned
;;;                     value by calling appropriate functions.

(defun viewer-menu-action (ws position)
  (let ((menu-action (cw::pop-up-menu *viewer-menu*
				      :position position
				      :window ws)))
    (case menu-action
      (root (root-on-proposition *viewer-root-prop*))
      (repaint (cw::repaint ws))
      (flush (flush-viewer-and-related-windows))
      (overviewer (overviewer-to-surface cw::*all-window-streams*)))))

;;; flush-viewer-and-related-windows.

(defun flush-viewer-and-related-windows ()
  (setf cw::*draw-text-background-p* nil)
  (when *viewer* (cw::flush *viewer*))
  (when *overviewer* (cw::flush *overviewer*))
  (when *magnifier* 
    (cw::flush (mag-window *magnifier*))
    (setf *magnifier* nil)))
	
;;; overviewer-to-surface. Brings the overviewer to the front of the screen.

(defun overviewer-to-surface (windows)
  (dolist (w windows)
    (when (equal "MVL Overviewer" (cw::window-stream-title w))
      (cw::expose w)
      (return-from overviewer-to-surface t))))

;;; general-overviewer-menu. This function determines whether the user
;;;                          has clicked on a gnode or not. If so, the
;;; overviewer gnode menu is poped. Otherwise the overviewer menu is
;;; shown.

(defun general-overviewer-menu (ws mouse-state &optional event)
  (let ((gnode (find-object-pointed-at mouse-state)))
    (if gnode
	(overviewer-gnode-menu gnode)
      (overviewer-menu ws mouse-state event))))

;;; toggle-gnode-selection. This function is called when the user presses
;;;                         the middle mouse button in the overviewer. If
;;; the click happens on a gnode the gnode changes selection status 
;;; (gets selected if not selected and vice versa). If the click is not on
;;; a gnode nothing happens.

(defun toggle-gnode-selection (ws mouse-state &optional event)
  (declare (ignore ws event))
  (let ((gnode (find-object-pointed-at mouse-state)))
    (when gnode
      (if (gn-selected gnode)
	  (deselect gnode)
	(select gnode)))))

;;; find-object-pointed-at. This function returns the object the user is
;;;                         pointing at in the overviewer window. If the
;;; user points at empty space nil is returned. To find the object we turn
;;; ov coordinates into extent coordinates, find the y level at which the
;;; key took place and examine all nodes in this level to find whether the
;;; point clicked on belongs to them.

(defun find-object-pointed-at (mouse-state)  
  (let* ((position (cw::mouse-state-position mouse-state))
	 (ov-x (cw::position-x position))
	 (ov-y (cw::position-y position))
	 (x (unscale-x ov-x))
	 (y (round-up-to-hundreds (unscale-y ov-y)))
	 (y-level (floor (y-to-level y)))
	 (clicked-pos (cw::make-position :x x :y y))
	 (suspect-gnodes (aref *viewer-spots-taken* y-level))
	 (too-far-to-the-right (+ x *gnode-separation*)))
    (dolist (gnode suspect-gnodes nil)
      (when (cw::region-contains-position-p (gn-area gnode) clicked-pos)
	(return-from find-object-pointed-at gnode))
      (when (> (gnode-x-pos gnode) too-far-to-the-right)
	(return-from find-object-pointed-at nil)))))

(defun round-up-to-hundreds (num)
  (+ 100 (* 100 (truncate num 100))))

;;; init-overviewer-menu. Initializes the overviewer menu.

(defun init-overviewer-menu ()
  (let ((item1 (cw::make-menu-item "Reorganize" 'reorganize))
	(item2 (cw::make-menu-item "Repaint" 'repaint))
	(item3 (cw::make-menu-item "Break" 'break)))
    (setf *overviewer-menu* (cw::make-menu 
			     :items (list item1 item2 item3)))))

;;; overviewer-menu. Pops up the overviewer menu and takes the
;;;                  appropriate action requested by the user.

(defun overviewer-menu (ws mouse-state &optional event)
  (declare (ignore event))
  (unless *viewer-locked*
    (let* ((position (cw::mouse-state-position mouse-state)))
      (let ((menu-action (cw::pop-up-menu *overviewer-menu*
					  :position position
					  :window ws)))
	(case menu-action
	  (reorganize (signal-static-display))
	  (repaint (overviewer-repaint))
	  (break (signal-break)))))))

;;; init-overviewer-gnode-menu. Initializes the overviewer gnode menu.

(defun init-overviewer-gnode-menu ()
  (let ((item1 (cw::make-menu-item "Select" 'select))
	(item2 (cw::make-menu-item "Deselect" 'deselect))
	(item3 (cw::make-menu-item "Deselect All" 'deselect-all))
	(item4 (cw::make-menu-item "Setf *C* to Object" 'set-to-c)))
    (setf *overviewer-gnode-menu* 
      (cw::make-menu :items (list item1 item2 item3 item4)))))

;;; overviewer-gnode-menu. Pops up the overviewer-gnode menu and
;;;                        processes the user's choice.

(defun overviewer-gnode-menu (gnode)
  (unless *viewer-locked*
    (let ((menu-action (cw::pop-up-menu *overviewer-gnode-menu*
					:position (gn-ov-position gnode)
					:window *overviewer*)))
      (case menu-action
	(select (select gnode))
	(deselect (deselect gnode))
	(deselect-all (deselect-all))
	(set-to-c (setf *c* (gn-task gnode)))))))

;;; overviewer-repaint. Redraws the tree in the overviewer.

(defun overviewer-repaint ()
  (cw::clear *overviewer*)
  (static-overviewer-display (prop-find-gnode *viewer-root-prop*)))

;;; general-left-button-actions. Instead of using the menus, the user
;;;                              has the choice of using the left mouse
;;; button in conjunction whith other keys. When the user does one
;;; of these actions, we need to determine whether the action desired
;;; belongs to the gnode actions or to the general actions. For that we
;;; determine wheter the click was on a gnode. If so we call the function
;;; gnode-left-button-actions to handle it and otherwise we call the
;;; function to handle viewer left button actions.

(defun general-left-button-actions (ws mouse-state &optional event)
  (declare (ignore event))
  (let* ((position (cw::mouse-state-position mouse-state))
	 (buttons (cw::mouse-state-button-state mouse-state))
	 (gnode (find-visible-gnode position)))
    (if gnode
	(gnode-left-button-actions gnode buttons)
      (viewer-left-button-actions ws buttons))))

;;; viewer-left-button-actions. Called to handle a left-mouse-button/key
;;;                             combination.

(defun viewer-left-button-actions (ws buttons)
  (cond ((logtest cw::*shift-key-down* buttons)
	 (overviewer-to-surface cw::*all-window-streams*))
	((logtest cw::*control-key-down* buttons)
	 (root-on-proposition *viewer-root-prop*))
	(t (cw::repaint ws))))

;;; viewer-left-button-actions. Called to handle a left-mouse-button/key
;;;                             combination.

(defun gnode-left-button-actions (gnode buttons)
  (cond ((logtest cw::*shift-key-down* buttons)
	 (expand-text gnode))
	((logtest cw::*control-key-down* buttons)
	 (magnifier-display gnode))
	(t (progn (setf *package* (find-package "MVL")) 
		  (wt:winspect (gn-task gnode))))))

;;; [7] Utilities
;;;     ---------

;;; root-on-proposition. This utility finds the proposition given by the 
;;;                      user and displays its gnode at the top/center
;;; point of the viewer. The result is a view of the subtree
;;; this proposition roots.

(defun root-on-proposition (prop &key (task-type 'inference)
				       (gnode nil))
  (unless gnode
    (setf gnode (prop-find-gnode prop task-type)))
  (when gnode
    (let* ((gnode-x (gnode-x-pos gnode))
	   (gnode-y (gnode-y-pos gnode))
	   (width (cw::window-stream-width *viewer*))
	   (height (cw::window-stream-height *viewer*))
	   (x-offset (- gnode-x (/ width 2)))
	   (y-offset (- gnode-y (- height 60)))
	   (region (cw::make-region :bottom y-offset :left x-offset)))
      (cw::scroll *viewer* region)
      'DONE)))

;;; center-on-proposition. Finds the gnode corresponding to prop
;;;                        and scrolls so that it is placed at the
;;; center of the screen. If no type is given the gnode corresponding
;;; to the inference gnode is found.

(defun center-on-proposition (prop &key (task-type 'inference)
					(gnode nil))
  (unless gnode
    (setf gnode (prop-find-gnode prop task-type)))
  (when gnode
    (let* ((gnode-x (gnode-x-pos gnode))
	   (gnode-y (gnode-y-pos gnode))
	   (width (cw::window-stream-width *viewer*))
	   (height (cw::window-stream-height *viewer*))
	   (old-x-offset (cw::window-stream-x-offset *viewer*))
	   (old-y-offset (cw::window-stream-y-offset *viewer*))
	   (mid-x (+ (/ width 2) old-x-offset))
	   (mid-y (+ (/ height 2) old-y-offset))
	   (x-offset (+ old-x-offset (- gnode-x mid-x)))
	   (y-offset (+ old-y-offset (- gnode-y mid-y)))
	   (region (cw::make-region :bottom y-offset :left x-offset)))
      (cw::scroll *viewer* region)
      'DONE)))

;;; [8] The Overview of the Tree
;;;     ------------------------

;;; Note on Drawing Lines:
;;; ----

;;; Lines should always be drawn in the same direction!!! Our convention
;;; is to *always* draw connections from the parent to its children.

;;; If this is not done, round off errors make a line from A to B be
;;; different from a line from B to A.
;;; Note on the Hilite Rectangles:
;;; ----

;;; A rectangle is used to show the user what area of the world is visible
;;; currently in the viewer. This rectangle is just a frame during
;;; dynamic viewing and it becomes a filled rectangle when dynamic
;;; viewing is done.

;;; A number of functions in this section deal with the hilite rectangle
;;; drawing them in very similar ways. They were written to deal with
;;; strange special cases that do arise.

;;; Note on Overviewer Gnode Drawing Functions:
;;; ----

;;; Gnodes are drawn in the overviewer and very often they change possitions,
;;; which means that they have to be erased and redrawn in order to move them.

;;; If the gnode that has to be redrawn is in the visible region of the
;;; viewer, in the overviewer it will lie within the rectangle that shows this
;;; region. 

;;; If this rectangle is just a frame, erasing can be accomplished
;;; by redrawing the gnode using the boole-clr mode (clear), and drawing
;;; can be accoimplished by redrawing the gnode using the boole-1 mode
;;; (set).

;;; If the rectangle is a filled rectangle (this happens when we are not
;;; viewing dynamically), the gnodes inside of it are drawn using the xor
;;; mode, and thus we can erase by drawing the gnode with xor. We can then
;;; redraw using xor again.

;;; The code that draws gnodes in the overviewer is always checking for
;;; these cases.

;;; End of Notes.
;;; ------------

;;; init-overviewer. This function creates and initializes the
;;;                  overviewer (a reduced view of the whole proof tree).
;;; In this function we set the scale variables *overviewer-x/y-scale*
;;; which show how many real pixels one pixel in the overviewer
;;; represents.

(defun init-overviewer ()
  (let* ((cw::*make-window-stream-actions* nil)
	 (overviewer (cw::make-window-stream
		      :bottom 
		      (- (cw::window-stream-extent-height cw::*root-window*)
			 (cadr *overviewer-size*)
			 30)
		      :left
		      (- (cw::window-stream-extent-width cw::*root-window*)
			 (car *overviewer-size*)
			 30)
		      :border-width 1
		      :width (car *overviewer-size*)
		      :height (cadr *overviewer-size*)
		      :title (format nil "MVL Overviewer"))))
    (cw::activate overviewer)
    (cw::control-mouse-cursor-move-events 
     overviewer :pointer-motion nil
     :button-motion t :last-only-p t :hint t)
    (cw::modify-window-stream-method overviewer 
	   :left-button-down :after 'overviewer-scroll)
    (cw::modify-window-stream-method overviewer 
	   :right-button-down :after 'general-overviewer-menu)
    (cw::modify-window-stream-method overviewer
	   :middle-button-down :after 'toggle-gnode-selection)
    (cw::modify-window-stream-method overviewer
	   :mouse-cursor-move :after 'overviewer-drag)
    (setf *overviewer* overviewer)
    (setf *overviewer-x-scale* (/ *viewer-extent-width*
				  (car *overviewer-size*)))
    (setf *overviewer-y-scale* (/ *viewer-extent-height*
				  (cadr *overviewer-size*)))))

;;; static-overviewer-display. Shows in the overviewer the whole tree
;;;                            displayed in the viewer. Then it initializes
;;; the hilited region which shows what part of the tree is being shown in
;;; the viewer.

(defun static-overviewer-display (root)
  (when root
    (shrink-forest root)
    (init-overviewer-hilite)))

;;; update-overviewer-region. Each time a hilite method is called it is
;;;                           because the visible area of the extent
;;; changes. This function mirrors this change in the *overviewer-region*
;;; variable.

(defun update-overviewer-region ()
  (setf (cw::region-left *overviewer-region*)
    (scale-to-ov-x (cw::region-left *viewer-region*)))
  (setf (cw::region-bottom *overviewer-region*)
    (scale-to-ov-y (cw::region-bottom *viewer-region*)))
  (setf (cw::region-height *overviewer-region*)
    (/ (cw::region-height *viewer-region*) *overviewer-y-scale*))
  (setf (cw::region-width *overviewer-region*)
    (/ (cw::region-width *viewer-region*) *overviewer-x-scale*)))

;;; init-overviewer-hilite. Draws a black rectangle in the overviewer which
;;;                         shows the area of the tree that the viewer is
;;; currently exposing.

(defun init-overviewer-hilite ()
  (update-overviewer-region)
  (funcall (if *dynamic-viewer-active*
	     #'cw::draw-rectangle
	     #'cw::draw-filled-rectangle)
	   *overviewer*
	   (cw::make-position :x (cw::region-left *overviewer-region*)
			      :y (cw::region-bottom *overviewer-region*))
	   (cw::region-width *overviewer-region*)
	   (cw::region-height *overviewer-region*)
	   :operation boole-xor))

;;; shrink-forest. This function draws the whole tree in the overviewer.
;;;                It does a DFS traversal.

(defun shrink-forest (root)
  (let ((children (gn-kids root)))
    (ov-draw-gnode root)
    (dolist (child children)
      (shrink-forest child))))

;;; ov-draw-gnode. Draws one gnode in the overviewer. We then connect it
;;;                to its children. A gnode is represented by a point
;;; with brush-width 4.

;;; Note that if the gnode is selected we call ov-draw-selected-gnode
;;; to draw it.

(defun ov-draw-gnode (gnode)
  (let* ((pos (gn-position gnode))
	 (x (cw::position-x pos))
	 (y (cw::position-y pos))
	 (scaled-x (scale-to-ov-x x))
	 (scaled-y (scale-to-ov-y y))
	 (scaled-pos (cw::make-position :x scaled-x :y scaled-y)))
    (setf (gn-ov-position gnode) scaled-pos)
    (if (gn-selected gnode)
	(ov-draw-selected-gnode gnode
				(if *dynamic-viewer-active* 
				    boole-1 boole-xor))
      (cw::draw-point *overviewer* scaled-pos :brush-width 4))
     (ov-connect-to-children gnode scaled-pos)))
  
;;; ov-connect-to-children. Connects a gnode in the overviewer to its
;;;                         children. The lines do not connect to the
;;; middle of the node but the the edges. This solves problems that arise
;;; whe we use xor and we write a line to the middle of the node.

;;; Note that the lines that connect the gnode to its children do not
;;; touch the gnode at all. That is why we need to increase and decrease
;;; the y coordinate at which the lines begin. The constants that you
;;; see in this function were found empirically.

;;; Note that all the functions that draw gnodes and connections in the
;;; overviewer have to do this in exactly the same way. Otherwise lines
;;; do not get cleared well and a mess results.

(defun ov-connect-to-children (gnode parent-pos)
  (let ((children (gn-kids gnode))
	(pos (cw::make-position))
	(draw-pos (cw::make-position))
	(par-draw-pos (cw::make-position)))
    (setf (cw::position-x par-draw-pos) (cw::position-x parent-pos))
    (setf (cw::position-y par-draw-pos) 
      (- (cw::position-y parent-pos)
	 (if (gn-selected gnode) 4 3)))
    (dolist (child children)
      (setf (cw::position-x pos)
	(scale-to-ov-x (gnode-x-pos child)))
      (setf (cw::position-y pos)
	(scale-to-ov-y (gnode-y-pos child)))
      (setf (cw::position-x draw-pos) (cw::position-x pos))
      (setf (cw::position-y draw-pos)
	(+ (cw::position-y pos) (if (gn-selected child) 3 2)))
      (cw::draw-line *overviewer* par-draw-pos draw-pos))))
		     
;;; hilite-visible-region. Moves the black rectangle in the overviewer
;;;                        to show the area that is currently exposed
;;; in the viewer. For this we first redraw the old rectangle in XOR
;;; mode to erase it, and then we draw the new rectangle (also in XOR
;;; mode to avoid covering the drawn gnodes).

(defun hilite-visible-region () 
  (when (member *overviewer* cw::*all-window-streams*)
    (cw::draw-filled-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)
    (update-overviewer-region)
    (cw::draw-filled-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)))

;;; Just like function above but hilites with an empty rectangle.

(defun dynamic-hilite-visible-region () 
  (when (member *overviewer* cw::*all-window-streams*)
    (cw::draw-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)
    (update-overviewer-region)
    (cw::draw-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)))

(defun dynamic-clear-hilite ()
  (when (member *overviewer* cw::*all-window-streams*)
    (cw::draw-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)))

(defun dynamic-show-hilite ()
  (when (member *overviewer* cw::*all-window-streams*)
    (update-overviewer-region)
    (cw::draw-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)))

;;; revert-to-reverse-hilite. This function is called when the dynamic
;;;                           viewer is done, to change the hilite from
;;; a thin frame to a filled rectangle.

(defun revert-to-reverse-hilite ()
  (when (member *overviewer* cw::*all-window-streams*)
    (cw::draw-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)
    (cw::draw-filled-rectangle 
     *overviewer*
     (cw::make-position :x (cw::region-left *overviewer-region*)
			:y (cw::region-bottom *overviewer-region*))
     (cw::region-width *overviewer-region*)
     (cw::region-height *overviewer-region*)
     :operation boole-xor)))

;;; careless-hilite-visible-region. Just like the function above but 
;;;                                 does not clear the old hilite area.

(defun careless-hilite-visible-region ()
  (update-overviewer-region)
  (cw::draw-filled-rectangle 
   *overviewer*
   (cw::make-position :x (cw::region-left *overviewer-region*)
		      :y (cw::region-bottom *overviewer-region*))
   (cw::region-width *overviewer-region*)
   (cw::region-height *overviewer-region*)
   :operation boole-xor))

;;; overviewer-scroll. This function is called when the user clicks on the
;;;                    overviewer. It will scroll the viewer so that the
;;; clicked point is at the center of the display. For this we translate
;;; the clicked point to viewer coordinates and just tell the viewer to
;;; scroll to it.

(defun overviewer-scroll (ws mouse-state &optional event)
  (declare (ignore event ws))
  (if (and *dynamic-viewer-active*
	   (not (equal *static-display-signaled* 'stoped))
	   *within-planting*)
      (signal-overviewer-scroll mouse-state)
    (do-overviewer-scroll mouse-state)))

(defun do-overviewer-scroll (mouse-state)
  (unless *viewer-locked*
    (let* ((position (cw::mouse-state-position mouse-state))
	   (ov-x (cw::position-x position))
	   (ov-y (cw::position-y position))
	   (x (unscale-x ov-x))
	   (y (unscale-y ov-y)))
      (setf x (- x (/ (cw::region-width *viewer-region*) 2)))
      (setf y (- y (* (cw::region-height *viewer-region*) 0.75)))
      (cw::scroll *viewer* (cw::make-region :left x :bottom y))
      (let ((new-current-root (find-current-root *visible-gnodes*)))
	(if new-current-root
	    (setf *viewer-current-root* new-current-root)
	  (setf *viewer-current-root* 
	    (prop-find-gnode *viewer-root-prop*)))))))

;;; find-current-root. This function is called after scrolling to return the
;;;                    gnode that "roots" the new area displayed. This is
;;; the first gnode found closest to the top of the window. This is used in
;;; the dynamic display to remember what node the user wants to see.

(defun find-current-root (visible-gnodes)
  (let (result
	(top-y 0))
    (dolist (gnode visible-gnodes result)
      (when (> (gnode-y-pos gnode) top-y)
	(setf top-y (gnode-y-pos gnode))
	(setf result gnode)))))

;;; overviewer-drag. This is called when the user moves the mouse in the
;;;                  overviewer while holding a button down. Each time
;;; an event is generated, this function calls overviewer-scroll to
;;; move the viewer to the translated new position of the mouse in 
;;; the overviewer.

(defun overviewer-drag (ws mouse-state event)
  (unless *viewer-locked*
    (cw::get-mouse-state ws mouse-state)
    (overviewer-scroll ws mouse-state event)))

;;; scale-to-ov-x. Translates a viewer x coordinate into the corresponding
;;;                overviewer value.

(defun scale-to-ov-x (x)
    (truncate (/ x *overviewer-x-scale*)))

;;; unscale-x. Translates an overviewer x coordinate into its corresponding
;;;            viewer coordinate.

(defun unscale-x (ov-x)
  (* ov-x *overviewer-x-scale*))

;;; scale-to-ov-x. Translates a viewer y coordinate into the corresponding
;;;                overviewer value.

(defun scale-to-ov-y (y)
    (truncate (/ (- y (- 32000 *viewer-extent-height*))
		 *overviewer-y-scale*)))

;;; unscale-y. Translates an overviewer y coordinate into its corresponding
;;;            viewer coordinate.

(defun unscale-y (ov-y)
  (+ (- 32000 *viewer-extent-height*)
     (* ov-y *overviewer-y-scale*)))

;;; [9] The Magnifier
;;;     -------------

;;; magnifier-display. The magnifier is a separate window in which the 
;;;                    proposition and truth value of a gnode are displayed.
;;; This function displays in the magnifier the proposition and truth
;;; value of the gnode passed.
	   
(defun magnifier-display (gnode)
  (unless *magnifier* (init-magnifier))
  (let ((mag (mag-window *magnifier*))
	(trect (mag-truth-rect *magnifier*))
	(prect (mag-prop-rect *magnifier*))
	(prop (truncate-to-fit (format nil "~a" (gn-prop gnode)) 9))
	(truth (truncate-to-fit
		(format nil "~a" (consider-truth-value (gn-task gnode)))
		21)))
    (setf (mag-task *magnifier*) (gn-task gnode))
    (cw::expose mag)
    (cw::clear-area mag (mag-prop-area *magnifier*))
    (cw::clear-area mag (mag-truth-area *magnifier*))
    (cw::set-window-stream-cursor-position mag (mag-prop-pos *magnifier*))
    (format mag prop)
    (cw::set-window-stream-cursor-position mag (mag-truth-pos *magnifier*))
    (format mag truth)
    (cw::draw-rectangle mag (car prect)
			(cadr prect) (caddr prect)
			:brush-width 2)
    (cw::draw-rectangle mag (car trect)
			(cadr trect) (caddr trect)
			:brush-width 2)))

;;; truncate-to-fit. Truncates a string so that it will fit in the number
;;;                  of lines in lines. This is used because the size of
;;; the prop and truth rectangles of the magnifier are fixed.

(defun truncate-to-fit (text lines)
  (let ((returns 0))
    (dotimes (i (length text) text)
      (when (char= (aref text i) #\Newline)
	(incf returns)
	(when (= returns lines)
	  (return-from truncate-to-fit 
	    (concatenate 'string (subseq text 0 i) "...")))))))

;;; init-magnifier. Initializes the magnifier window. Note that as opposed
;;;                 to the other windows in the viewer system, the variable
;;; *magnifier* holds a structure with the window as a field
;;; (not the window itself).

(defun init-magnifier ()
  (let* ((cw::*make-window-stream-actions* nil)
	 (font (cw::open-font :courier :plain 10
			     :weight :bold))
	 (pb 325) (pl 5) (ph 140) (pw 480)
	 (tb 5) (tl 5) (th 290) (tw 480)
	 (mag (cw::make-window-stream
	       :bottom 48
	       :left 34
	       :border-width 4
	       :width 500
	       :height 500
	       :font font
	       :title "MVL Magnifier"))
	 (p-area (cw::make-region :bottom pb  :left pl
				  :height ph :width pw))
	 (p-pos (cw::make-position :x (+ pl 5) :y (- (+ pb ph) 10)))
	 (t-area (cw::make-region :bottom tb :left tl
				  :height th :width tw))
	 (t-pos (cw::make-position :x (+ tl 5) :y (- (+ tb th) 10))))
    (setf (cw::window-stream-position mag)
      (cw::make-position :x 617 :y 17))
    (cw::modify-window-stream-method mag
	   :right-button-down :after 'magnifier-menu-action)
    (init-magnifier-menu)
    (cw::activate mag)
    (setf *magnifier* (make-magnifier
		       :window mag
		       :prop-area p-area
		       :prop-pos  p-pos
		       :truth-area t-area
		       :truth-pos  t-pos
		       :prop-rect (list (cw::make-position 
					 :x (- pl 2) :y (+ pb 2))
					(+ pw 4) (+ ph 4))
		       :truth-rect (list (cw::make-position 
					  :x (- tl 2) :y (+ tb 2))
					 (+ tw 4) (+ th 4))))
    (cw::draw-rectangle mag
			(car (mag-prop-rect *magnifier*))
			(cadr (mag-prop-rect *magnifier*))
			(caddr (mag-prop-rect *magnifier*))
			:brush-width 2)
    (cw::draw-rectangle mag
			(car (mag-truth-rect *magnifier*))
			(cadr (mag-truth-rect *magnifier*))
			(caddr (mag-truth-rect *magnifier*))
			:brush-width 2)
    (cw::set-window-stream-cursor-position mag
         (cw::make-position :x (+ pl 10) :y (+ pb ph 15)))
    (format mag "Proposition")
    (cw::set-window-stream-cursor-position mag
	 (cw::make-position :x (+ tl 10) :y (+ tb th 15)))
    (format mag "Truth Value")))
				
;;; init-magnifier-menu. Initializes and creates the magnifier menu.

(defun init-magnifier-menu ()
  (let ((item1 (cw::make-menu-item "Flush" 'flush)))
    (setf *magnifier-menu*
	  (cw::make-menu :items (list item1)))))

;;; magnifier-menu-action. When the user clicks on the magnifier window,
;;;                        the magnifier menu is poped up and the
;;; approppriate action taken.

(defun magnifier-menu-action (ws mouse-state &optional event)
  (declare (ignore mouse-state event))
  (let ((menu-action (cw::pop-up-menu *magnifier-menu*
				      :window ws)))
    (case menu-action
      (flush (flush-magnifier)))))

;;; flush-magnifier. Gets rid of the magnifier and clears the *magnifier*
;;;                  variable.

(defun flush-magnifier ()
  (cw::flush (mag-window *magnifier*))
  (setf *magnifier* nil))
  
;;; [10] Viewer Mouse in and out methods
;;;      -------------------------------

;;; viewer-mouse-in. This function is called when the mouse enters a
;;;                  mouse sensitive region. We have defined one such
;;; region for each gnode so that mouse-enter events can be caught and
;;; the gnode hilited. In this function we find the gnode that contains
;;; the entered region and frame it (draw a rectangle around its text).

(defun viewer-mouse-in (region mouse-state transition)
  (declare (ignore mouse-state transition))
  (let ((the-gnode (find-aroused-gnode region)))
    (when the-gnode
      (frame-gnode the-gnode))))

;;; find-aroused-gnode. Iterates through the visible nodes to find
;;;                     the gnode containing the entered region. This gnode
;;; is returned.

(defun find-aroused-gnode (region)
  (dolist (gn *visible-gnodes* nil)
    (when (equal (gn-actv-reg gn) region)
      (return-from find-aroused-gnode gn))))

;;; frame-gnode. Draws a rectangle around the propostion and truth value of
;;;              the gnode gn.

(defun frame-gnode (gn)
  (apply 'cw::draw-rectangle (gn-hilite-args gn)))

;;; viewer-mouse-out. This is called when an active region is exited from.
;;;                   The function finds the gnode that contains the active
;;; region and unframes it.

(defun viewer-mouse-out (region mouse-state transition)
  (declare (ignore mouse-state transition))
  (let ((the-gnode (find-aroused-gnode region)))
    (when the-gnode
      (unframe-gnode the-gnode))))

;;; unframe-gnode. Clears the frame of the gnode as created by frame-gnode.

(defun unframe-gnode (gn)
  (apply 'cw::draw-rectangle 
	 (append (gn-hilite-args gn)
		 (list ':operation boole-xor))))

;;; direct-to-window. When the user is pressing more than one button down,
;;;                   this function is called. If the left button is pressed
;;; we cause a left-button-down event that we can catch. Otherwise we check
;;; if the right button is pressed and do likewise.

(defun direct-to-window (region mouse-state transition)
  (declare (ignore region))
  (let ((buttons (cw::mouse-state-button-state mouse-state)))
    (cond ((logtest cw::*left-button-down* buttons)
	   (cw::left-button-down *viewer* mouse-state transition))
	  ((logtest cw::*right-button-down* buttons)
	   (cw::right-button-down *viewer* mouse-state transition)))))

;;; [11] Overviewer Gnode Selection
;;;      --------------------------

;;; The user can select gnodes in the overviewer. This section contains the
;;; functions that implement selection and deselection.

;;; In the overviewer, selected gnodes are drawn bigger than normal 
;;; gnodes and with a white point in the middle. These two operations 
;;; do the job.

;;; (cw::draw-point *overviewer* position :brush-width 6)
;;; (cw::draw-point *overviewer* position :brush-width 2 :operation boole-xor)

;;; In the viewer, a selected gnode is recognized by an arrow that points
;;; to it on its left. 

;;; The drawing functions in the viewer and overviewer have been modified
;;; to deal with this.

;;; select. This function selects a gnode if it is not already selected.
;;;         In the overviewer we need to erase the gnode and draw it
;;; again as a selected gnode. A problem is that the gnode might be in the
;;; currently visible area and we need to decide how to erase and draw the
;;; gnode according to two cases:

;;; Case 1 happens when we are in the middle of dynamic viewing. Here, the
;;; visible area is an empty rectangle and we can thus use boole-clr
;;; to clear the gnode and boole-1 to draw the gnode as selected. 

;;; Case 2 happens when we are done with dynamic viewing and the visible
;;; area is a filled rectangle. In this case, using boole-xor works well
;;; for erasing and redrawing.

;;; So, this function starts by finding out what operators will be used
;;; for drawing and redrawing. Then the gnode is erased by calling 
;;; draw-connections, we tell the gnode it has been selected and we
;;; push it into the list of selected gnodes. Then we call draw-connections
;;; again to redraw the gnode, and we finish by marking the gnode as
;;; selected in the viewer if it is currently visible.

(defun select (gnode)
  (unless (gn-selected gnode)   
    (let ((op1 (if *dynamic-viewer-active* boole-clr boole-xor))
	  (op2 (if *dynamic-viewer-active* boole-1 boole-xor)))
      (draw-connections gnode (gn-dad gnode) (gn-kids gnode) op1 t)
      (setf (gn-selected gnode) t)
      (push gnode *selected-gnodes*)
      (draw-connections gnode (gn-dad gnode) (gn-kids gnode) op2 nil)
      (when (member gnode *visible-gnodes*)
	(draw-selected-node-mark gnode)))))

;;; deselect. This function deselects a gnode if it is already selected.
;;;           It is symmetric to select.

(defun deselect (gnode)
  (when (gn-selected gnode)   
    (let ((op1 (if *dynamic-viewer-active* boole-clr boole-xor))
	  (op2 (if *dynamic-viewer-active* boole-1 boole-xor)))
      (draw-connections gnode (gn-dad gnode) (gn-kids gnode) op1 t)
      (setf (gn-selected gnode) nil)
      (setf *selected-gnodes* (remove gnode *selected-gnodes*))
      (draw-connections gnode (gn-dad gnode) (gn-kids gnode) op2 nil)
      (when (member gnode *visible-gnodes*)
	(clear-selected-node-mark gnode)))))

;;; deselect-all. This function call deselect on all the gnodes that are
;;;               selected.

(defun deselect-all ()
  (when (and (streamp *viewer*) (open-stream-p *viewer*))
    (dolist (gnode *selected-gnodes*)
      (deselect gnode))))

;;; ov-draw-selected-gnode. Draws a selected gnode in the overviewer.
;;;                         Note that it creates the white dot by
;;; xoring if we are not dynamic viewing and by clearing otherwise.

(defun ov-draw-selected-gnode (gnode op)
  (let ((position (gn-ov-position gnode)))
    (cw::draw-point *overviewer* position :brush-width 6 :operation op)
    (when (= op boole-1)
      (setf op boole-clr))
    (cw::draw-point *overviewer* position :brush-width 2 :operation op)))

;;; ov-clear-selected-gnode. Clears the selected gnode in the overviewer.

(defun ov-clear-selected-gnode (gnode op)
  (let ((position (gn-ov-position gnode)))
    (cw::draw-point *overviewer* position :brush-width 2 :operation op)
    (cw::draw-point *overviewer* position :brush-width 6 :operation op)))

;;; select-tasks. This function calls select on every gnode that
;;;               corresponds to a task in tasks.

(defun select-tasks (tasks) (de/select-tasks tasks #'select))

(defun de/select-tasks (tasks fn &aux gnode)
  (when (and (streamp *viewer*) (open-stream-p *viewer*))
    (dolist (task tasks)
      (when (setf gnode (find-gnode task)) (funcall fn gnode)))))

;;; deselect-tasks. This function calls deselect-task on every gnode
;;;                 that corresponds to a task in tasks.

(defun deselect-tasks (tasks) (de/select-tasks tasks #'deselect))

;;; selected-tasks. Returns the tasks selected by the user.

(defun selected-tasks ()
  (mapcar #'gn-task *selected-gnodes*))

