;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; global variable defined:

;; *completed-database*	t or nil depending on whether approach to unground 
;;			negative subgoals is as suggested by Chan or Ginsberg
;;			respectively

;; Much of this code involves computing (or recomputing) the truth value
;; to be associated to any particular task.  There are three ways in
;; which this is done:

;; 1.  The simplest is to merely compute the truth value as things
;; stand.

;; 2.  We may also want to compute the *hypothetical* truth value of a
;; task, assuming that some other tasks work out as well as possible.

;; 3.  Finally, we may want to compute the truth value in a destructive
;; way, updating all of the associated data structures to reflect the
;; change.

(defvar *active-bilattice*)
(defvar binding-dag)
(defvar bdg-to-truth-val)
(defvar bdg-true)
(defvar bdg-false)
(defvar bdg-unknown)
(defvar bdg-bottom)

;; We begin with case (1), because that's the simplest.  There are three
;; cases, depending on the type of the task.  But first, we check to
;; see if we've already got the answer cached away.

;; At the end, we may have to plug in the bindings everywhere.  This
;; appears to only be a problem for nonmodal tasks, and there only if the
;; plugging function on the active bilattice is nontrivial.

(defparameter *use-cache* t)

(defun compute-truth-value (task)
  (or (and *use-cache* (cached-computation task))
      (cache-computation
       task
       (ctv-plug (typecase task
		   (modal (compute-modal-truth-value task))
		   (modal-clause (compute-modal-clause-truth-value task))
		   (t (compute-proof-truth-value task)))
		 task))))

;; Do we already have an answer for this computation stored away?  First
;; look with eql, then with the full equality check on the binding
;; bilattice.

;; The entries on the cache list are of the form ((v1 ... vn) . v) where
;; the vi are the truth values of the children and v is the value for the
;; parent.

(defun cached-computation
    (task &aux temp (kid-vals (mapcar #'consider-truth-value
				      (consider-children task)))
	       (cache (consider-cache task)))
  (when (setq temp (or (find-in-cache cache kid-vals #'eql)
		       (find-in-cache cache kid-vals
				      (bilattice-eq bdg-to-truth-val))))
    (cdr temp)))

;; Look for the children in the cache, where the kids are a match if
;; they match the truth values in (car cache) for a particular cache
;; entry.

(defun find-in-cache (cache kid-vals test)
  (find-if #'(lambda (c) (every test kid-vals (car c)))
	   cache))

;; To cache a computation, just construct the list of kids' values, and
;; cons onto the computed result.  But we don't cache slave truth values,
;; since there aren't any "children".

(defun cache-computation (task result)
  (when (and *use-cache* (not (and (inference-p task) (slaved-task task))))
    (push (cons (mapcar #'consider-truth-value (consider-children task))
		result)
	  (consider-cache task)))
  result)

;; Here is where we handle plugging into a computed answer.  The
;; conditions are as described above: not a modal task, plugging function
;; not trivial, more than one child (so that there is some "cross effect"
;; in the bindings being considered) and truth value not unknown (in
;; which case the plug would do nothing).

(defun ctv-plug (ans task)
  (if (ctv-needs-plug ans task) (ctvp-1 ans) ans))

(defun ctv-needs-plug (ans task)
  (and (not (modal-p task)) (bilattice-plug *active-bilattice*)
       (cdr (consider-children task))
       (not (mvl-unk ans bdg-to-truth-val))))

(defun ctvp-1 (f &aux flag)
  (mapc #'(lambda (x &aux (val (mvl-plug (dag-entry-val x) 
					 (dag-entry-pt x))))
	      (unless (mvl-eq val (dag-entry-val x))
		(setq flag t)
		(setf (dag-entry-val x) val))
	      x)
	  (dag-fn-list f))
  (if flag (simplify f) f))

;; For a modal task, pass these truth values off to the combining
;; function associated to the modal operator.

(defun compute-modal-truth-value (task)
  (apply (modal-op-fn (modal-op task)) (modal-exp-args task)))

;; get the arguments for a modal expression.  task is the task whose
;; truth value is being computed.  It's pretty easy -- we just walk down
;; the list of children.  The only catch is that we have to be careful to
;; insert any parametric arguments by pulling them off of the proposition
;; being considered, and to invert the truth value if the argument type
;; means it's negated.

(defun modal-exp-args (task &aux (kids (modal-children task)))
  (mapcar #'(lambda (type term)
	      (cond ((eql type 0) term)
		    (t (setq term (consider-truth-value (pop kids)))
		       (if (case type
			     ((nil) t)
			     (1 (not (modal-parity task)))
			     (-1 (modal-parity task)))
			   (mvl-not term bdg-to-truth-val)
			 term))))
	  (modal-op-args (modal-op task)) (cdr (consider-prop task))))

;; For a MODAL-CLAUSE task, just combine the positive and negative
;; subtasks.  In some instances, this function may be called before
;; these subtasks are established, so we'll need to return unknown.

(defun compute-modal-clause-truth-value (task)
  (mvl-plus (safe-value (modal-clause-pos task))
	    (mvl-not (safe-value (modal-clause-neg task)) bdg-to-truth-val)
	    bdg-to-truth-val))

(defun safe-value (task)
  (if task (consider-truth-value task) bdg-unknown))

;; For a nonmodal task, the truth value can be computed using the
;; lookup-value and the truth values of the subtasks.  The justification
;; information for the given task is stored on the <just> property of
;; the task in the form of an ATMS-lattice value associated to each
;; binding for which a justification is known.  There is one additional
;; subtlety -- for a modal task, the parent tasks may actually use
;; either the modal proposition itself, or its negation.  For this
;; reason, modal tasks have a <parity> property that indicates whether
;; the negated (NIL) or nonnegated (T) version of the modal sentence is
;; used.  We also absorb the new binding into the existing answer using
;; bdg-absorb.

;; If this task is inheriting its value from another, we can simply get
;; the value of the other task and then do a plug.

(defun compute-proof-truth-value (task)
  (if (slaved-task task)
      (compute-slaved-truth-value task)
    (let ((ans (inference-base-value task)))
      (dolist (item (inference-just task)
		(bdg-absorb ans (proof-vars (inference-proof task))))
	(dolist (conj (cdr item))
	  (setq ans (mvl-plus ans (conj-to-val (car item) conj)
			      bdg-to-truth-val)))))))

;; A task is slaved if it is not the first entry on the list of tasks used
;; by the associated invocation of the prover.

(defun slaved-task (task)
  (not (eql task (proof-task (inference-proof task)))))

(defun compute-slaved-truth-value (task &aux (proof (inference-proof task)))
  (mvl-plug (consider-truth-value (proof-task proof))
	    (cdr (assoc task (proof-slaves proof)))
	    bdg-to-truth-val))

;; Here is where we absorb any superfluous variables in the answers returned
;; by the first-order prover.  Of course, if the absorption function is
;; trivial, we can just drop the variables.  If it isn't ...

(defun bdg-absorb (fn vars)
  (if (bilattice-absorb *active-bilattice*)
      (ba-1 fn vars)
    (dag-change-dag fn #'(lambda (x) (meaningful-bdgs x vars)))))

;; ... then we have to group the answers by the apparently relevant
;; bindings and, for each, absorb it into the overall answer.  To
;; understand mvl-absorb, see the description of absorption in load.lisp.

(defun ba-1 (fn vars &aux alist relevant entry)
  (dolist (item (dag-fn-list fn) 
	    (dag-accumulate-fn binding-dag *active-bilattice* alist))
    (setq relevant (meaningful-bdgs (dag-entry-pt item) vars)
	  entry (assoc relevant alist :test #'equal-binding))
    (if entry
	(setf (cdr entry)
	  (mvl-absorb (cdr entry) (dag-entry-pt item) (dag-entry-val item)))
      (push (cons relevant (dag-entry-val item)) alist))))

;; given a conjunct, construct a truth value.  Combine the truth values
;; after including information from the bindings associated with a
;; particular justification.

(defun conj-to-val (bdg conj)
  (include-bdg-in-truth-value bdg (combine-for-proof conj) #'bilattice-or))

(defun combine-for-proof (tasks)
  (if tasks
      (mvl-t-ground (reduce (bilattice-and bdg-to-truth-val)
			    (mapcar #'adjust-truth-value tasks))
		    bdg-to-truth-val)
    bdg-true))

;; modify a truth value to include a particular "root" binding.  If the
;; root binding is null, just return the given truth value.  Otherwise,
;; append the given binding to the bindings of the truth value.

(defun include-bdg-in-truth-value
    (bdg value &optional (modify #'bilattice-plus))
  (if bdg
      (dag-change-dag value #'(lambda (x) (append-binding-lists x bdg))
		      modify)
    value))

;; when computing the truth value contributed by a given task, if it is
;; a nonmodal task, just negate the given truth value (unless it's a
;; place holder for a truth value).  If it is a modal task, then check to
;; see whether it is used positively or negatively in the parent, and
;; negate the truth value if necessary.

(defun adjust-truth-value (task)
  (if (proof-normal-value task)
      (consider-truth-value task)
    (mvl-not (consider-truth-value task) bdg-to-truth-val)))

(defun proof-normal-value (task)
  (typecase task
    (modal (modal-parity task))
    (inference (null (consider-prop task)))
    (modal-clause t)))

;; Here we *recompute* the truth value assigned to a task, making
;; whatever destructive modifications are appropriate as we go, including
;; recomputing the truth values of the parents.  The way it works is as
;; follows:

;; We get the new truth value and see if it's different from the
;; original one.  Assuming that it is, we tell the user if tracing is on.
;; If the task has a parent, we recompute *its* truth value; similarly
;; for any slaves of the task.  If the task has no parent, we set the
;; current value of the overall analysis to the value computed for this
;; task, ignroing exceptions if appropriate.

;; We also have to deal with the fact that some pending answers might
;; now be relevant.  Suppose that we just changed the truth value of a
;; task t0 at depth d, and that the previous task that changed truth
;; value was t1.  Now if c is any descendant of t0 that is *not* a
;; descendant of t1, a pending answer for c might be relevant if the
;; highest task that it affected is a sibling of t1 or higher.  If it's
;; below t1, then it still can't have any effect.  The same thing is true
;; if the truth value of t0 *didn't* change, provided that the truth
;; value of t1 did.

;; To implement this, recompute-truth-value accepts an optional argument
;; that is the last task changing truth value, and uses the following
;; function in relevance.lisp:

;; recheck-descendants (task subtask) reactivates pending answers that
;; are descendants of task but not of subtask and that affect something at
;; depth equal to 1+ that of task or greater.

;; Finally, if this task has any slaves, we have to recompute their truth
;; values as well.

;; One other caveat: if the user has some sort of dynamic display going
;; on, so that he is watching the prover, update-dynamic-display should
;; handle this.

(defun recompute-truth-value (task &optional last-task)
  (update-dynamic-display task)
  (unless (mvl-eq (consider-truth-value task)
		  (setf (consider-truth-value task) (compute-truth-value task))
		  bdg-to-truth-val)
    (show-truth-value task)
    (if (consider-parent task)
	(recompute-truth-value (consider-parent task) task)
      (setf (analysis-current-value analysis)
	(absorb-exceptions (consider-truth-value task))))
    (when (and (inference-p task) (not (slaved-task task)))
      (mapc #'(lambda (x) (recompute-truth-value (car x)))
	    (task-slaves task))))
  (recheck-descendants task last-task))

(defparameter *completed-database* nil)

(defun absorb-exceptions (value)
  (if *completed-database* value (ignore-exceptions value)))

;; If *completed-database* is NIL, we don't care about "exceptions"
;; to current answers.  The easiest way to deal with this is just to
;; disjoin all the values from any particular binding to the root.
;; ie-1 does this recursively, and then we simplify the whole thing.

(defun ignore-exceptions (fn &aux (bilattice (dag-fn-bilattice fn)) list)
  (setq fn (copy-entire-fn fn)
	list (dag-fn-list fn))
  (mapc #'(lambda (x) (setf (dag-entry-val x) (ie-1 x list bilattice))) list)
  (simplify fn))

;; here we do the disjunction.  The dag-fn structure already includes
;; information about points where x is a successor.

(defun ie-1 (x orig bilattice &aux (d (dag-entry-pt x)) (v (dag-entry-val x)))
  (dolist (item orig v)
    (when (member d (dag-entry-succ item))
      (setq v (mvl-or v (dag-entry-val item) bilattice)))))

(defun task-slaves (task)
  (proof-slaves (inference-proof task)))

#-Allegro
(defun update-dynamic-display (x) x)

;; This next group of functions computes the *hypothetical* truth value
;; of a task; what we really want is a list of (v1 v2 . assumptions)
;; where v1 <> v2 given the assumptions about the other tasks.
;; hypothetical-differences is responsible for computing this.

;; The first thing we need to do is adjust the new value in case the
;; parent task is a quantification task.  Then we compute a list of
;; (v .  assumptions) for each possible set of assumptions, combine the
;; results in the obvious way and return the changes.

(defun hypothetical-differences
    (tasks news &aux (parent (consider-parent (car tasks))))
  (declare (ignore ccheck))
  (when (quantifier-p parent)
    (let ((parity (modal-parity parent))
	  (qvars (second (consider-prop parent))))
      (when (atom qvars) (setq qvars (list qvars)))
      (setq news
	(mapcar #'(lambda (n x)
		    (recompute-for-quantification n (consider-truth-value x)
						  x parity qvars))
		news tasks))))
  (compute-differences parent tasks news))

;; Is this task considering a quantified sentence?  Yes, if it's modal
;; and involves the right quantifier depending on the parity.

(defun quantifier-p (task)
  (and (modal-p task)
       (eq (car (consider-prop task))
	   (if (modal-parity task) 'forall 'exists))))

;; Here is where we figure out what to use for the new values in the
;; quantified case.  The story is that the given task is under a
;; quantified parent, and we are considering the values NEW instead of
;; OTHER.  The problem is that NEW may fail to impact the quantified
;; value because it doesn't supply values for all bindings of the
;; quantified variable.

;; We first find all the bindings where NEW takes a better value than
;; OTHER; these are the "relevant" bindings.  Adjust-value then makes
;; suitable changes to NEW using this information.

(defun recompute-for-quantification (new other task parity qvars)
  (if (proof-p (inference-proof task))
      (adjust-value new (diff-bdgs new other parity) parity qvars)
    new))

(defun diff-bdgs (x y parity)
  (delete-if #'(lambda (b) (funcall (if parity #'k-le #'k-ge)
				    (get-val b x) (get-val b y)))
	     (combine-2 (dag-dot (dag-fn-dag x)) (dag-eq (dag-fn-dag x))
			(all-dag-pts x) (all-dag-pts y))))

(defun combine-2 (dot eq list new-list &aux result)
  (dolist (new-pt new-list (delete-duplicates result :test eq))
    (dolist (old-pt list)
      (pushconc (funcall dot new-pt old-pt) result))))

;; Here we compute the new value.  Here's the story -- the value of VAL
;; is better than the original at all the points in BDGS.  So at any
;; point b that will be mapped to the same set of bdgs as BDGS will be
;; mapped to, we assume that we've gotten a value of val as well by
;; disjoining (or conjoining for an existential task) the value val takes
;; at BDGS with the current value at b.

;; To make it a little easier, we begin by constructing qlist, a list of
;; (bdgs . new-bdgs) for every element of the original dag-fn where the
;; quantification will affect the bindings (i.e., bdgs and new-bdgs
;; differ).  Now for each bdg in bdgs, if the quantification matters, we
;; work through the elements of qlist; if the new-bdgs match but the
;; original bindings didn't, we add a new entry at the old bindings.

(defun adjust-value (val bdgs parity qvars &aux dag-list qlist)
  (setf val (copy-entire-fn val)
	dag-list (dag-fn-list val)
	qlist (delete nil 
		      (mapcar #'(lambda (b &aux (pt (dag-entry-pt b)))
				  (multiple-value-bind (qbdg qflag)
				      (make-qbdg pt qvars)
				    (when qflag (cons pt qbdg))))
			      dag-list)))
  (dolist (bdg bdgs (simplify val))
    (multiple-value-bind (qbdg qflag) (make-qbdg bdg qvars)
      (when qflag
	(dolist (item qlist)
	  (when (and (equal-binding (cdr item) qbdg)
		     (not (equal-binding (car item) bdg)))
	    (add-dag (car item)
		     (funcall (if parity #'mvl-or #'mvl-and)
			      (dag-entry-val (find-entry (car item) dag-list))
			      (get-val qbdg val))
		     val)))))))

;; Given bindings and a list of variables, return two values -- the
;; bindings with the variables removed, and a flag to indicate that the
;; two are different.
		   
(defun make-qbdg (bdg qvars &aux flag)
  (values (remove-if #'(lambda (b) (when (member (car b) qvars) (setq flag t)))
		     bdg)
	  flag))		     

;; Here is where we actually do the work.  Part of this involves finding
;; every other task that might be an assumption, and then doing the
;; calculation making all possible assumptions about the values taken by
;; these tasks.  We assume that any subset of them might succeed; by
;; "succeed" we mean take the value TRUE if the original task is of type
;; inference or the value BOTTOM if it's of type modal.

;; ORIG below is a list of the values originally taken by the subtasks;
;; HOLES is a list of the tasks about which we'll make assumptions and
;; HOLE-VALS are the original values of these tasks.  SUBSETS is a list
;; of the subsets of HOLES.  OLD-IS-DIFF indicates whether any of the old
;; values is a change from those in ORIG (typically not, but quantification
;; adjustments may change that), and NEW-IS-DIFF is similar for the new
;; values.  Here's how it goes:

;; First, we set the truth values to the old values, provided that
;; they're different from the current ones.  Then we compute all the
;; hypothetical truth values.  The new values and old values are always
;; different, so we set up the new values and then compute the
;; hypothetical values again.  Then we reset the original values if
;; needed and construct the actual list of differences to be returned.

;; This final list only includes assumptions for which new and old are
;; different, and if there are assumption sets a1 and a2 with a2 a subset
;; of a1 and the new value of a1 being the same as that for a2, we can
;; drop a1.

(defparameter *show-hypothetical-computations* nil)

;; apparently they are different but old is *also* different and new
;; is then the same as old was?

(defun compute-differences
    (task subtasks news 
     &aux (current-val (consider-truth-value task)) new-val
	  (orig (mapcar #'consider-truth-value subtasks))
	  (holes (find-holes task subtasks))
	  (hole-vals (mapcar #'consider-truth-value holes)) result)
  (when *show-hypothetical-computations*
    (deselect-all) (select-tasks subtasks))
  (dolist (set (subsets holes))
    (setq new-val (prog2 (set-truth-values subtasks news)
		      (new-hole-value task holes hole-vals set t)
		    (set-truth-values subtasks orig)))
    (when (and (not (mvl-eq current-val new-val bdg-to-truth-val))
	       (setq new-val
		 (new-val-difference (new-hole-value task holes hole-vals
						     set nil)
				     new-val)))
      (push (cons new-val set) result)))
  (delete-subsumed-entries result
			   #'(lambda (x y)
			       (and (subsetp (cdr y) (cdr x))
				    (mvl-eq (car x) (car y) bdg-to-truth-val))
			       )))

(defun new-val-difference (old-val new-val)
  (unless (mvl-eq old-val new-val bdg-to-truth-val)
    new-val))

;; return a list of all subsets of the given set.

(defun subsets (set)
  (if set
      (let ((temp (subsets (cdr set))))
	(nconc temp (mapcar #'(lambda (x) (cons (car set) x)) temp)))
    (list nil)))

;; Here we actually compute the new value, assuming that the elements of
;; subset all take the value new.  We change these values, compute the
;; answer, and then reset the values.  Of course, if subset is empty, we
;; don't have to do all that.  And if subset is empty and there is no
;; change in truth value, we can just use the current value instead of
;; recomputing it.

(defun new-hole-value (task holes hole-vals subset change)
  (cond (subset (prog2 (mapc #'(lambda (x) (setf (consider-truth-value x)
					     (assumed-truth-value x task)))
			     subset)
		    (compute-truth-value task)
		  (set-truth-values holes hole-vals)))
	(change (compute-truth-value task))
	(t (consider-truth-value task))))

;; what value do we want to assume for the given task?  If it's modal, use
;; bottom.  If it's normal, we use true.  If it's a modal child of negative
;; parity (when proof-normal-value fails), we use false.

(defun assumed-truth-value (task parent)
  (cond ((modal-p parent) bdg-bottom)
	((proof-normal-value task) bdg-true)
	(t bdg-false)))

;; set the truth values of a bunch of tasks to the given values.

(defun set-truth-values (tasks values)
  (mapc #'(lambda (task val) (setf (consider-truth-value task) val))
	tasks values))

;; Return a list of all the "holes," which are the children of the given
;; task that are *not* on the given list of subtasks, don't have the
;; assumed value (since the assumption would then do nothing) and:
;;  1.  If the task is modal, the hole simply needs to be active.
;;  2.  If the task is inferential, the hole needs to be part of a
;;      justification that uses a task in SUBTASKS.

(defun find-holes (task subtasks)
  (delete-if #'(lambda (x) 
		 (mvl-eq (consider-truth-value x) (assumed-truth-value x task)
			 bdg-to-truth-val))
	     (typecase task
	       (modal (holes subtasks (consider-children task)))
	       (inference (proof-holes task subtasks)))))

;; by a "hole" we mean a task that is still active.  BUT, the current
;; task shouldn't count; that's the one we're assuming succeeds!  If
;; require-modal is T, we also remove any task that isn't modal.

(defun holes (sublist list &optional require-modal)
  (remove-if #'(lambda (task) 
		 (or (and require-modal (not (modal-p task)))
		     (member task sublist)
		     (not (active-task task))))
	     list))

;; A task is active if either:
;;  1.  It is of type inference, with an inference-proof that is itself
;;      active because it is incomplete, or
;;  2.  It has an active child.

;; It's more subtle, though.  The second arg is a "parity" that flips at
;; consistency checks (we want to know if the original task can become
;; *more* true).  So there are the following cases:
;;  1.  If not of type inference, see if some child is active.
;;  2.  If of type inference, flip parity if nonmonotonic.  Then if parity
;;      T (the good case):
;;     2a.  If proof is incomplete, return T (it's active).
;;     2b.  If any child is active, return T.
;;  3.  If parity is NIL, then case (2a) goes away, and we just return T
;;      if some child is active.

(defun active-task (task &optional (parity t))
  (when (inference-p task)
    (when (inference-nonmon task) (setq parity (not parity)))
    (let ((p (inference-proof task)))
      (when (and parity (not (and (proof-p p) (completed-proof p))))
	(return-from active-task t))
      (setq task (proof-task p))))
  (some #'(lambda (task) (active-task task parity)) (consider-children task)))

;; Holes in a justification.  For each element on the justification
;; list, we ignore the binding and just look at the cnf expression.  Then
;; for each conjunction in that, if the given subtask is a member, we
;; call HOLES to find the holes.  The subtasks being removed are required
;; to be modal (the others are consistency checks).

(defun proof-holes (task subtasks)
  (delete-duplicates
     (mapcan #'(lambda (x) (romt-1 (cdr x) subtasks)) (inference-just task))))

(defun romt-1 (cnf subtasks)
  (mapcan #'(lambda (x) (when (some #'(lambda (y) (member y subtasks)) x)
			  (holes subtasks x t))) 
	  cnf))
