;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;;; This function reads forms from the file whose name is in inf and
;;; prints errors on the terminal.  When done, the function returns the
;;; total number of errors and exits.

;; The format of the test file is as follows:

;; <exp-1>
;; <value-returned-1>
;; 
;; <exp-2>
;; <value-returned-2>
;; 
;; 	.
;; 	.
;; 	.
;; 
;; <exp-n>
;; <value-returned-n>

;; The expressions are evaluated and the values returned are compared to
;; the <value-returned-i>, where ** is used as a wildcard in
;; <value-returned-i>.

;; As an example, see the file mvl.test, which is used by the function
;; (test-mvl) to test the entire mvl system.

;; here it is.  Some preliminary junk, then read an expression; if EOF,
;; return the number of errors.  Otherwise, get the next expression; if
;; it's EOF, warn the user about the unexpected EOF and return.

;; If you have an expression and an expected value, evaluate the
;; expression and check to see if the answer is as expected.

(defun mvl-test-file (inf &key description &aux (file (mvl-file inf "test")))
  (when description (format t "~%Testing ~a." description))
  (when file
    (with-open-file (in file)
      (do (test expect actual (n 0))
	  (nil)
	(cond ((eq 'stop (setq test (read in nil 'stop)))
	       (return n))
	      ((eq 'stop (setq expect (read in nil 'stop)))
	       (warn "~%Early end of file for ~a.~%" inf)
	       (return n))
	      ((not (test-match expect (setq actual (eval test))))
	       (incf n)
	       (format t "~%~a - ~s~%~a not ~a"
		       inf test actual expect)))))))

;; demo is a little harder.  We have to check for "pause" (which waits
;; for a cr to continue) and "quiet" (which evals the next form
;; silently).  The check for *stop-demo* inside a quiet evaluation is
;; to allow the user to exit easily from nested demos.

(defparameter *stop-demo* (list nil))

(defun mvl-demo-file (inf &key description 
		      &aux (file (mvl-file inf "demo")) ans)
  (when description (format t "~2&Demonstrating ~a." description))
  (when file
    (with-open-file (in file)
      (do (form)
	  ((eq 'stop (setq form (read in nil 'stop))))
	(cond ((and (atom form) (string-equal (string form) "pause"))
	       (do () (nil)
		 (format t "~2%Enter a form or <CR> to continue: ")
		 (setq form (read-line))
		 (when (string-equal form "") (return))
		 (when (string-equal form "stop")
		   (return-from mvl-demo-file *stop-demo*))
		 (format t "~2&Evaluating: ~a" form)
		 (format t "~%Returns:    ~a" (eval (read-from-string form)))))
	      ((and (atom form) (string-equal (string form) "quiet"))
	       (when (eql (eval (read in nil)) *stop-demo*)
		 (return-from mvl-demo-file *stop-demo*)))
	      (t (format t "~2&Evaluating: ~a" form)
		 (format t "~%Returns:    ~a" (setq ans (eval form)))
		 (when (eql ans *stop-demo*) 
		   (return-from mvl-demo-file *stop-demo*))))))))

;; Check to see if two expressions match.  e1 is what you expected; e2
;; is what you computed.

;; If e1 is ** (the wildcard), just succeed.  If the two expressions
;; are equal or equal-answers, you can also succeed.  Finally, if they 
;; are both lists, invoke test-match recursively.

(defun test-match (e1 e2)
  (or (eq e1 '**) (equalp e1 e2) (equal-answer e1 e2)
      (and (listp e1) (listp e2) 
	   (test-match (car e1) (car e2))
	   (test-match (cdr e1) (cdr e2)))))

;; test-mvl invokes mvl-test-file on the file "mvl" and returns only the
;; first value (the number of errors).

(defun test-mvl ()
  (with-mvl-invocation ()
    (time (prog1 (mvl-test-file "mvl") (format t "~2%")))))

;; demo-mvl invokes mvl-demo-file on "mvl"

(defun demo-mvl ()
  (with-mvl-invocation ()
    (mvl-demo-file "mvl")))
