;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; simple temporal reasoning.  The set of time points is just the integers;
;; the dot of t1 and t2 is simply the maximum of the two.

(defun time-dag-dot (t1 t2)
  (list (max t1 t2)))

(defvar *active-bilattice*)

(defun time-dag ()
  (first (bilattice-components *active-bilattice*)))

(defparameter *time-dag*
	      (make-dag :root 0 :eq #'= :leq #'>= :dot #'time-dag-dot
			:long "Temporal order" :short "Time"))

;; The underlying bilattice used includes justification information --
;; it's the bilattice of functions from contexts into t/f/u/bot.

(defun time-base-bilattice ()
  (second (bilattice-components *active-bilattice*)))

(defparameter *time-base-bilattice*
	      (lattice-to-dag-to-bilattice *first-order-atms-lattice*
					   *first-order-bilattice*))

;; A truth function that is true at the given time, and unknown
;; immediately thereafter.

(defun true-at (time &aux (b (time-base-bilattice))
			  (ans (make-tv (time-dag) b time (bilattice-true b))))
  (add-dag (1+ time) (bilattice-unknown b) ans)
  ans)

;; delay a dag function.  Just advance each time in the list.

(defun delay (time v)
  (dag-change-dag v #'(lambda (x) (+ x time))))

(defparameter *delay* (make-modal-op :name 'delay :fn #'delay :args '(0 1)))

;; propagation is easy here -- just remove any points where the function
;; is unknown.

(defun propagate (fn)
  (dag-remove-unknown fn))

;; remove unknown points from a dag-list (but not from the root!)

(defun dag-remove-unknown (f &aux (bilattice (dag-fn-bilattice f)))
  (dag-drop f #'(lambda (entry) (mvl-unk (dag-entry-val entry) bilattice))))

(defparameter *propagate*
    (make-modal-op :name 'propagate :fn #'propagate :args '(x)))

;; get a value at a particular time, and then return the constant
;; function at that value.

(defun extract-value (time fn)
  (make-root (time-dag) (time-base-bilattice) (get-val time fn)))

(defparameter *extract-value*
    (make-modal-op :name 'at :fn #'extract-value :args '(0 1)))

;; assemble all the pieces; the modal operators are called DELAY,
;; PROPAGATE and AT.

(defparameter *time-bilattice* 
    (dag-bilattice *time-dag* *time-base-bilattice*))

(bilattice-has-modal-op *time-bilattice* *delay*)
(bilattice-has-modal-op *time-bilattice* *propagate*)
(bilattice-has-modal-op *time-bilattice* *extract-value*)

(describe-bilattice *time-bilattice* "Temporal reasoning" "Temporal" #\t)

(bilattice *time-bilattice*)

;; A quick version has the first-order bilattice as the base bilattice
;; instead of including the justification information.

(defparameter *quick-time-bilattice* 
    (dag-bilattice *time-dag* *first-order-bilattice*))

(bilattice-has-modal-op *quick-time-bilattice* *delay*)
(bilattice-has-modal-op *quick-time-bilattice* *propagate*)
(bilattice-has-modal-op *quick-time-bilattice* *extract-value*)

(describe-bilattice *quick-time-bilattice*
		    "Cheap temporal reasoning" "Temporal (quick)" #\q)

(bilattice *quick-time-bilattice*)
