;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; The basic aim in this file is to establish the junk needed to execute
;; cutoff tests -- mostly the functions associated with each tag.  To do
;; that, we first have to define all of the cutoff functions, and then to
;; associate them to the tags of the same name.  Dot-with-not and
;; splitting-point-below are also defined here.

;; functions defined:
;; 
;; bilattice comparison functions:
;; 
;;   k-le k-lt k-ge k-gt k-not-le k-not-lt k-not-ge k-not-gt
;;   t-le t-lt t-ge t-gt t-not-le t-not-lt t-not-ge t-not-gt
;; 
;; dot-with-not (x)		compute x.-x
;; splitting-point-below (x)	the splitting point below x (in a
;; 				canonically grounded bilattice)

(defvar unknown)
(defvar *active-bilattice*)
(defvar *always-succeeds*)
(defvar *never-succeeds*)

;; auxiliary functions k-le lt ge gt not-le not-lt not-ge not-gt etc

;; as an example, x is k-le y if x.y=x; any bilattice may also supply a
;; special (and presumably faster!) version of k-le and t-le which are
;; then used to compute all of the other comparitors.

;; If the bilattice is supplied, use that; otherwise, use the standard
;; one.

(defun k-le (x y &optional (b *active-bilattice*))
  (or (eql x y) (funcall (bilattice-kle b) x y)))

(defun k-ge (x y &optional (b *active-bilattice*))
  (k-le y x b))

(defun t-le (x y &optional (b *active-bilattice*))
  (or (eql x y) (funcall (bilattice-tle b) x y)))

(defun t-ge (x y &optional (b *active-bilattice*))
  (t-le y x b))

(defun k-lt (x y &optional (b *active-bilattice*))
  (and (not (mvl-eq x y b)) (k-le x y b)))

(defun k-gt (x y &optional (b *active-bilattice*))
  (and (not (mvl-eq x y b)) (k-le y x b)))

(defun t-lt (x y &optional (b *active-bilattice*))
  (and (not (mvl-eq x y b)) (t-le x y b)))

(defun t-gt (x y &optional (b *active-bilattice*))
  (and (not (mvl-eq x y b)) (t-le y x b)))

(defun k-not-le (x y &optional (b *active-bilattice*)) (not (k-le x y b)))
(defun k-not-ge (x y &optional (b *active-bilattice*)) (not (k-le y x b)))
(defun t-not-le (x y &optional (b *active-bilattice*)) (not (t-le x y b)))
(defun t-not-ge (x y &optional (b *active-bilattice*)) (not (t-le y x b)))
(defun k-not-lt (x y &optional (b *active-bilattice*)) (not (k-lt x y b)))
(defun k-not-gt (x y &optional (b *active-bilattice*)) (not (k-gt x y b)))
(defun t-not-lt (x y &optional (b *active-bilattice*)) (not (t-lt x y b)))
(defun t-not-gt (x y &optional (b *active-bilattice*)) (not (t-gt x y b)))

(defun dot-with-not (x &optional (b *active-bilattice*))
  (mvl-dot x (mvl-not x b) b))

;; on the tag's tag-not property is that tag that is the t-flip of the
;; given tag

(defun tag-not (tag) (get tag 'tag-not))

(defmacro set-tag-not (symbol tag)
  `(setf (get ',symbol 'tag-not) ',tag))

(set-tag-not t-le t-ge)
(set-tag-not t-ge t-le)
(set-tag-not t-lt t-gt)
(set-tag-not t-gt t-lt)
(set-tag-not t-not-le t-not-ge)
(set-tag-not t-not-ge t-not-le)
(set-tag-not t-not-lt t-not-gt)
(set-tag-not t-not-gt t-not-lt)
(set-tag-not k-le k-le)
(set-tag-not k-ge k-ge)
(set-tag-not k-lt k-lt)
(set-tag-not k-gt k-gt)
(set-tag-not k-not-le k-not-le)
(set-tag-not k-not-ge k-not-ge)
(set-tag-not k-not-lt k-not-lt)
(set-tag-not k-not-gt k-not-gt)

;; cutoffs are expressed in dnf, where the base-level elements are
;; tests.  A <test> is a field consisting of a truth value and a tag.
;; The story is that the truth value x has to satisfy the given property
;; with respect to the given value.

(defstruct (test (:constructor make-test (value tag)))
  value tag)

;; here, we take cutoffs c and produce a new cutoffs d such that x
;; satisfies d if and only if the negation of x satisfies c.  Of course,
;; we can do it one conjunct at a time ...

(defun cutoffs-not (cutoffs &optional (b *active-bilattice*))
  (mapcar #'(lambda (x) (cutoffs-not-1 x b)) cutoffs))

;; and within a conjunct, we can do it one disjunct at a time, so all we
;; really have to do is "invert" a test (value . tag), which we can do
;; simply by negating the value and finding the tag that is the
;; "tag-not" of the given one.

(defun cutoffs-not-1 (c b)
  (mapcar #'(lambda (x) 
	      (make-test (mvl-not (test-value x) b) (tag-not (test-tag x))))
	  c))

;; to test a value and a cutoffs, make sure that the test passes every
;; conjunct in the cutoffs

(defun test (value cutoffs &optional (bilattice *active-bilattice*))
  (every #'(lambda (x) (test-1 value x bilattice)) cutoffs))

;; to test a value on a disjunction, it has to pass some clause

(defun test-1 (value disj-cutoffs bilattice)
  (some #'(lambda (x) (test-2 value x bilattice)) disj-cutoffs))

(defun test-2 (value test bilattice)
  (funcall (symbol-function (test-tag test))
	   value (test-value test)
	   bilattice))

;; The splitting point below x is gt(x).-gt(x) + gf(x).-gf(x)

(defun splitting-point-below (x &optional (b *active-bilattice*))
  (mvl-plus (dot-with-not (mvl-t-ground x b) b)
	    (dot-with-not (mvl-f-ground x b) b) b))
