;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; Here is all the stuff needed for the control calculations -- but not
;; the control calculations themselves.  The following functions are
;; defined here:

;; recheck-descendants (task subtask) reactivates pending answers that
;; are descendants of task but not of subtask and that affect something at
;; depth equal to that of subtask or greater.  subtask might be NIL.

;; return-answer (main-task)	what can you return now?

;; relevant-answer (ans proof) Takes an answer and proof and returns a
;; list of <relevance>s describing conditions under which the answer is
;; relevant.

(defvar proof)
(defvar analysis)

;; For each task that is a descendant of task (but not of subtask), if
;; there is a pending answer that managed to have some affect up to depth
;; one below the task (or higher!), reset the cdr of that pending answer
;; to indicate that it's time to think about it again.  If any relevance
;; in the cached relevance list affects something at this level, dump them
;; all and recompute them.

(defun recheck-descendants (task subtask 
			    &aux (depth (1+ (consider-depth task))))
  (recheck-task task depth)
  (dolist (sub (consider-children task))
    (unless (eq sub subtask) (recheck-descendants-1 sub depth))))

(defun recheck-descendants-1 (task limit)
  (recheck-task task limit)
  (mapc #'(lambda (x) (recheck-descendants-1 x limit))
	(consider-children task)))

;; Here we recheck a single task.  If it's of type inference and it's
;; been associated with a proof effort, we look at each returned answer.
;; If it's pending for the given task below the depth limit, we mark it
;; as actively pending and reactivate the task.  Similar check to decide
;; whether or not to dump the cached relevant-answer computation.  (In
;; the check itself, it's possible that (cdr x) is NIL because there is
;; no way for an answer to affect anything at all.  In that case, we
;; assume it has an effect at this level of the analysis only.)

(defun recheck-task (task limit &aux temp proof)
  (when (and (inference-p task) (proof-p (setq proof (inference-proof task))))
    (dolist (ans (proof-pending-answers proof))
      (when (and (setq temp (cdr ans)) (<= temp limit))
	(setf (cdr ans) nil)))
    (setf (proof-relevance proof)
      (delete-if #'(lambda (x)
		     (if (cdr x)
			 (some #'(lambda (r &aux (a (relevance-affects r)))
				   (or (null a) (<= (consider-depth a) limit)))
			       (cdr x))
		       (<= (1+ (consider-depth task)) limit)))
		 (proof-relevance proof)))))

;; Returning answers.  Just prune out everything that doesn't satisfy
;; the termination test.

(defun return-answer (&aux (terminate (analysis-terminate analysis)))
  (dag-prune (analysis-current-value analysis)
	     #'(lambda (x) (test x terminate))))

;; Here is the actual relevance computation.  It's easy except for the
;; fact that two impacted tasks may be siblings.  So what we do use the
;; depth associated with each task to work from the bottom up.  We
;; actually work with a small structure, each entry of which is a task, a
;; new value, and a list of the assumptions made.

(defstruct (rel (:constructor make-rel (current new assumptions)))
  current new assumptions)

;; relevant-answer takes an answer and a proof and returns a list of
;; relevances computed from it.  The work is done by relevant-tasks; all
;; we do here is set it up.  Here, we either return a cached answer or
;; call ra-1 to do the work.

(defun relevant-answer (ans proof &aux (r (proof-relevance proof)) temp)
  (if (setq temp (assoc ans r :test #'equal-answer))
      (cdr temp)
    (prog1 (setq temp (ra-1 ans proof))
      (push (cons ans temp) (proof-relevance proof)))))

;; Here is actual computation.  Task is the associated task and orig the
;; current truth value.  The new value includes the contribution from
;; ans; then we call include-new-rel to set up all the possible
;; assumptions (including any slaves) and relevant-tasks to do the work.

(defun ra-1 (ans proof
	     &aux (task (proof-task proof))
		  (orig (consider-truth-value task))
		  (new (mvl-plus orig
				 (make-tv binding-dag *active-bilattice*
					  (answer-binding ans)
					  (answer-value ans))
				 bdg-to-truth-val)))
  (unless (mvl-eq orig new bdg-to-truth-val)
    (relevant-tasks (include-new-rel task new nil))))

;; Set up a list of rels, each a current task, new value and assumption
;; list.  At this point, it's easy.  We just construct one possible
;; assumption for each task that uses the given proof.

(defun include-new-rel (current new ass)
  (cons (make-rel current new ass)
	(when (inference-p current)
	  (mapcar #'(lambda (x) 
		      (make-rel (car x) (mvl-plug new (cdr x) bdg-to-truth-val)
				ass))
		  (task-slaves current)))))

;; Here is the basic loop to find the relevant tasks.  The input is a
;; list of <rel>s, and the output is a list of <relevance>s that can then
;; be processed by the control mechanism.

;; We begin by selecting the rels to work on; then we compute the
;; hypothetical truth value(s) of the parent and call relevant-tasks
;; recursively.  At the end, we have to take the union of all the results
;; computed.  relevant-tasks can finish when the selected task has no
;; parent.

;; In a bit more detail, what we do is the following:
;;  1.  We set selected-rels to a list of the rels being considered
;;      together with all of their considered siblings.  These are
;;      displayed if *show-relevance-computation* is T.
;;  2.  We remove the selected-rels from the list of rels, since these
;;      have now been investigated.
;;  3.  Rel-finish then finishes up the work.

(defparameter *show-relevance-computation* nil)

(defun relevant-tasks (rels &optional answer &aux selected)
  (cond (rels
	 (setq selected (select-rels rels))
	 (when *show-relevance-computation*
	   (format t "~%** Relevance computation **")
	   (dolist (r selected)
	     (format t "~%Task:   ~a~%New:  ~a~%" 
		     (rel-current r) (rel-new r))))
	 (rel-finish (collapse-rels selected) (set-difference rels selected)
		     answer))
	(t answer)))

;; Select rels to work on.  Pick the deepest task available, and all of
;; its siblings.

(defun select-rels (rels &aux (deep (car rels)) temp
			      (depth (consider-depth (rel-current deep))))
  (dolist (r (cdr rels))
    (when (> (setq temp (consider-depth (rel-current r))) depth)
      (setq deep r depth temp)))
  (remove (consider-parent (rel-current deep)) rels
	  :key #'(lambda (r) (consider-parent (rel-current r)))
	  :test-not #'eql))

;; It may be that a task is affected by each of two routes; when this
;; happens, we add the two new truth values using mvl-plus and keep only
;; one rel for the task in question.

(defun collapse-rels (rels &aux temp match sum)
  (cond ((cdr rels)
	 (setq temp (remove-duplicates rels :key #'rel-current))
	 (dolist (item rels temp)
	   (unless (find item temp)
	     (setq match (member (rel-current item) temp :key #'rel-current)
		   sum (mvl-plus (rel-new item) (rel-new (car match))
				 bdg-to-truth-val))
	     (cond ((mvl-eq sum (rel-new item) bdg-to-truth-val)
		    (setf (car match) item))
		   ((not (mvl-eq sum (rel-new (car match)) bdg-to-truth-val))
		    (setf (car match) 
		      (make-rel (rel-current item) sum
				(union (rel-assumptions item) 
				       (rel-assumptions (car match))))))))))
	(t rels)))

;; Here we complete the calculation.  parent-values is a list of
;; (new . assumptions) for the parent task.  There are three 
;; possibilities:
;; 
;; 1.  parent-values is NIL, either because the parent is not affected
;;     or because there is no parent but the answer doesn't matter.  In
;;     this case, we have to take all the rels and construct <relevance>s
;;     from them, adding these to the current answer and then calling
;;     relevant-tasks again on what's left.
;; 2.  the task has no parent.  Now we know we're done, and return a new
;;     answer to the whole problem.  We construct a <relevance> reflecting
;;     this and return normally.
;; 3.  the task has a parent.  Now we just recur, adjusting the sets of
;;     assumptions suitably.

(defun rel-finish (rels others answer
		   &aux (parent-values (next-step rels))
			(old-ass (reduce #'union
					 (mapcar #'rel-assumptions rels))))
  (if parent-values
      (let ((parent (consider-parent (rel-current (car rels)))))
	(if parent
	    (rel-finish-continue parent-values others answer parent old-ass)
	  (rel-finish-success (rel-new (car rels)) answer old-ass)))
    (rel-finish-failure rels others answer old-ass)))

;; Case 3 above.  For each parent-value, we update the answer by calling
;; relevant-tasks recursively.  We make a new rel that combines any new
;; assumptions with the old ones.

(defun rel-finish-continue (parent-values others answer parent old-ass)
  (dolist (val parent-values answer)
    (setq answer
      (relevant-tasks (nconc (include-new-rel parent (car val)
					      (union (cdr val) old-ass))
			     others)
		      answer))))

;; Case 2 above.  For each parent value and each task, we make a new
;; relevance that points to that task, has an <affects> field of NIL
;; because a new final answer was constructed, and then returns that new
;; answer and all of the assumptions used.

(defun rel-finish-success (value answer old-ass)
  (cons (make-relevance nil value old-ass) answer))

;; Case 1 above.  For each rel, and for each task, we make a new
;; relevance that indicates how far we got, the new answer at that level,
;; and the assumptions used.  Then we call relevant-tasks again to finish
;; up.

(defun rel-finish-failure (rels others answer old-ass)
  (dolist (rel rels)
    (push (make-relevance (rel-current rel) (rel-new rel) old-ass) answer))
  (relevant-tasks others answer))

;; Take the next step in the relevance computation.  If there is no
;; parent, just record whether the new answer matters.  If there is a
;; parent, look at all the hypothetical new truth values for the parent
;; and for any cases where the new and old values are different, that
;; will be a continuation of the given path.

(defun next-step (rels)
  (if (consider-parent (rel-current (car rels)))
      (compute-hypotheticals rels)
    (change-is-relevant (rel-new (car rels)))))

;; Here we have some rels and need to compute the hypothetical truth
;; values for the parent.  We invoke hypothetical-differences to return a
;; list of (new-value . assumptions-made).

(defun compute-hypotheticals (rels)
  (hypothetical-differences (mapcar #'rel-current rels)
			    (mapcar #'rel-new rels)))

;; Here is the main task, and a new answer.  Is it really different?
;;  1.  If the old answer has something that passes the success test,
;;  	then there are two cases:
;;    1a.  If *succeed-with-bound* is true (the default), the new answer
;;         must be k-not-ge the old at some such point.
;;    1b.  If *succeed-with-bound* is false, the new answer must *fail*
;;         the success test at some such point.
;;  2.  Otherwise, the results must be different after you prune everything
;;  	that doesn't pass the cutoff test.

(defun change-is-relevant
    (new &aux old-list (old (analysis-current-value analysis))
	      (cuts (analysis-cutoffs analysis))
	      (succ (analysis-succeeds analysis)))
  (setq new (absorb-exceptions new))
  (if (setq old-list (remove-if-not #'(lambda (x)
					(test (dag-entry-val x) succ))
				    (dag-fn-list old)))
      (some (if (analysis-succeed-with-bound analysis)
		#'(lambda (item)
		    (k-not-ge (get-val (dag-entry-pt item) new)
			      (dag-entry-val item)))
	      #'(lambda (item)
		  (not (test (get-val (dag-entry-pt item) new) succ))))
	    old-list)
    (not (mvl-eq (answer-to-return old cuts)
		 (answer-to-return new cuts)
		 bdg-to-truth-val))))

(defun answer-to-return (val cutoffs)
  (dag-prune val #'(lambda (x) (test x cutoffs))))

