;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; Plan representation and basic operations thereon.

;; a PLAN is a partial order on a set of actions, and is represented as
;; a structure.  The first slot is the action, one of the following:
;;  1a. a variable
;;  1b. a list like (move a b)
;;  1c. initial (the dummy initial action)
;;  1d. terminal (the dummy final action)

;; The remaining slots in the plan are as follows:
;;  2. A list of other actions (plans) that must precede this one
;;  3. If this plan is an immediate (and presumably only) successor
;;     of the one pointing to it, IMM is T.  If actions can be
;;     interspersed between the two, IMM is NIL.
;;  4. A mark that is used in merging and other constructions.
;;  5. A list of the variables in the plan.  (T until computed.)
;;  6. A hash value that is used to cache results of the merge and
;;     instance computations.
;; The last two slots are computed when needed and then stored forever.

;; All PLANs begin with a terminal action and eventually ground out in
;; the initial action, which has no predecessors.  The generic plan has
;; the initial action as a non-immediate predecessor to the dummy action
;; that terminates the plan; the initial state corresponds to the plan
;; that has the initial action as an *immediate* predecessor to the
;; terminal action.

(defstruct (plan (:print-function print-plan) 
	    (:constructor make-plan (action preds &optional imm mark)))
  action preds imm mark 
  (vars t) (hash nil))

;; Get the variables in a plan.  If already computed (i.e., stored value
;; is a list), use that.  Otherwise, vars-in does it pretty simply.

(defun get-plan-vars (plan &aux (v (plan-vars plan)))
  (if (listp v) v
    (setf (plan-vars plan) (compute-vars plan))))

(defun compute-vars (plan)
  (reduce #'union (mapcar #'get-plan-vars (plan-preds plan))
	  :initial-value (vars-in (plan-action plan))))

;; Get the hashed value of a plan.  If it's already computed, use that.
;; Otherwise, compute it and store it in the plan-hash slot.

(defun get-plan-hash (plan)
  (or (plan-hash plan) (setf (plan-hash plan) (compute-plan-hash plan))))

;; Hash a plan.  Replace all the vars in the action with NIL, and cons
;; the action onto the front of a list of the hashed preds.  Then push T
;; onto the front if it's immediate and call sxhash to do the work.

(defun compute-plan-hash
    (plan &aux (hpreds (mapcar #'get-plan-hash (plan-preds plan)))
	       (action (plan-action plan)))
  (push (plug action (napcar #'list (vars-in action))) hpreds)
  (when (plan-imm plan) (push t hpreds))
  (sxhash hpreds))

;; here is the mark manipulation stuff.  The way it works is that when
;; we process a node, we push a new symbol onto the front of the node's
;; mark and push the node itself onto mark-list.  Then at the end, we pop
;; the marks of everything in mark-list.
 
(defvar mark-list)
(defvar finished-mark)

;; remark restores all the marks on the mark-list

(defun remark () 
  (mapc #'(lambda (x) (pop (plan-mark x))) mark-list))

;; save-mark takes a plan, pushes it onto the mark-list and marks it as
;; finished.

(defun save-mark (plan)
  (push plan mark-list)
  (push finished-mark (plan-mark plan)))

;; has a plan already been processed?

(defun marked (plan)
  (eql (car (plan-mark plan)) finished-mark))

;; slightly more complex versions of the above, where the finshed mark is
;; actually a cons.  In marking, return the answer.

(defun mark-with-cons (plan ans)
  (push plan mark-list)
  (push (cons finished-mark ans) (plan-mark plan))
  ans)

;; if it's marked with a cons, presumably the cdr of the mark is the
;; answer to be returned.

(defun marked-with-cons (plan &aux (pmark (car (plan-mark plan))))
  (and (listp pmark) (eql (car pmark) finished-mark)
       (cdr pmark)))

;; with-marks takes a function call (fn args).  It sets the finish mark
;; to be the function, then invokes the function and restores the marks.
;; The function itself should check to see if any particular action is
;; marked and mark the action when it's done with it.

(defmacro with-marks ((fn &rest args))
  `(let (mark-list (finished-mark ',fn))
     (prog1 (,fn ,@args) (remark))))

;; marked-defun is rather more involved.  The arguments are a function
;; being defined, an argument list, the object being traversed, an answer
;; to return when the traversal stops, and the body of the function to be
;; evaluated.  The arguments may be conses; if so, the cdr is the initial
;; value to be passed to the function that does the work, with the car
;; being the name of the variable.  Mark the plan at the end, but return
;; the last form in the function body as the result.

(defmacro marked-defun (fn args plan answer &rest body
			&aux (temp (concsym fn "-1")))
  `(progn (defun ,fn (,@(remove-if #'consp args))
	    (with-marks (,temp ,@(mapcar #'(lambda (x)
					     (if (consp x) (cdr x) x)) 
					 args))))
	  (defun ,temp (,@(mapcar #'(lambda (x) (if (consp x) (car x) x))
					 args))
	    (cond ((marked ,plan) ,answer)
		  (t (prog1 (progn ,@(subst temp fn body))
		       (save-mark ,plan)))))))

;; marked walk is more involved still.  It assumes that the structure
;; being traversed is PLAN and that the accumulated answer is ANS.  The
;; function might have other arguments, of course, and those are the
;; second arg of marked-walk.  The initial value for ANS is provided and
;; the rest of the macro is the code to be evaluated and returned once
;; PLAN's predecessors have been dealt with recursively.

(defmacro marked-walk (fn other-args initial-value &rest body)
  `(marked-defun ,fn (plan ,@other-args (ans . ,initial-value)) plan ans 
		 (dolist (item (plan-preds plan))
		   (setq ans (,fn item ,@other-args ans)))
		 ,@body))

;; remove the marks from a plan.

(marked-walk unmark () nil (setf (plan-mark plan) nil))

;; How many actions are in the given plan?  Start at -2 (the initial and
;; terminal actions don't count), and then count.

(marked-walk count-actions () -2 (1+ ans))

;; the root of the plan dag has an ongoing link from the initial to
;; terminal action.  The initial situation has an immediate link.

(defparameter plan-root (make-plan 'terminal (list (make-plan 'initial nil))))
(defparameter plan-init 
    (make-plan 'terminal (list (make-plan 'initial nil t))))

(defun plan-from-action (a)
  (make-plan 'terminal (list (make-plan a (list (make-plan 'initial nil))
					t))))

;; Stuff for printing plans.

(defparameter *print-bounds* nil)	; print dummy initial/terminal actions?
(defvar *print-simply* nil)		; give up and just print structure?
(defparameter *print-marks* nil)	; print marks?

(defun print-plan (plan stream print-depth)
  (declare (ignore print-depth))
  (if *print-simply*
      (format stream "~:[~;*~]t:~a~@[ m:~a~]~@[ p:~a~]" 
	      (plan-imm plan) (plan-action plan) 
	      (and *print-marks* (plan-mark plan))
	      (plan-preds plan))
    (format stream "~a" (printable-plan plan))))

;; A printable plan is an instance of the PLAN-PRINT structure.  This
;; has a type (serial, parallel or an action's mark) and a list of
;; entries (each of which is a printable plan).  There are also three
;; special actions: sequence (an arbitrary sequence of actions can be
;; inserted here), initial and terminal.

(defstruct (plan-print (:print-function print-plan-2)
	    (:constructor make-plan-print (type entries)))
  type entries)

(defparameter serial-mark (make-plan-print nil 'sequence))

;; After the conversion: if serial, separate entries by ".." or by
;; nothing as appropriate.  Parallel, separate them by "&".  Actions are
;; just printed out.  If it's an action, we enclose it in <> instead of
;; the usual lisp (), which means that we have to check to see if it's a
;; list like (move a b) or a single variable like ?3.  We also reverse
;; the order of the entries if it's serial.  Finally, if *print-bounds*
;; is nil, we don't print the terminal and initial actions.

(defun print-plan-2 (plan stream print-depth 
		      &aux (type (plan-print-type plan))
			   (ent (plan-print-entries plan)))
  (declare (ignore print-depth))
  (when (and (null *print-bounds*) (eql type 'serial))
    (setq ent 
      (remove-if #'(lambda (e)
		     (member (plan-print-entries e) '(initial terminal)))
		 ent)))
  (format stream
	  (cond ((eql type 'serial) "~@<[~;~I~{~a~^~:_ ~}~;]~:>")
		((eql type 'parallel) "~@<[~;~I~{~a~^~:_ & ~}~;]~:>")
		((eql ent 'sequence) "~@<..~:>")
		((listp ent) "~<<~;~I~{~a~^~:_ ~}~;>~>~@[~a~]")
		(t "~<<~;~a~;>~:>~@[~a~]"))
	  (if (eql type 'serial) (reverse ent) ent) (and *print-marks* type)))

;; Make a plan printable.  Look at the current latest point:
;;  1.  If it has no predecessors, it's easy.
;;  2.  If there are predecessors, printable-preds does the work.  Then we
;;      tack on the leading action unless suppress-first is non-NIL.

(defun printable-plan (plan &optional suppress-first
		       &aux (preds (plan-preds plan)))
  (cond ((and preds suppress-first) (printable-preds plan preds))
	(preds (add-serial (printed-action plan) (printable-preds plan preds)
			   (plan-imm (car preds))))
	(suppress-first (make-plan-print 'parallel nil))
	(t (printed-action plan))))

(defun printed-action (plan)
  (make-plan-print (plan-mark plan) (plan-action plan)))

;; add p1 to p2, looking for an answer of the given type.  Either or
;; both might be of that type already.

(defun add-general (p1 p2 imm? type &aux ans)
  (setq ans (make-plan-print type (if (eql (plan-print-type p2) type)
				      (plan-print-entries p2) (list p2))))
  (unless imm? (push serial-mark (plan-print-entries ans)))
  (if (eql (plan-print-type p1) type)
      (setf (plan-print-entries ans) 
	(append (plan-print-entries p1) (plan-print-entries ans)))
    (push p1 (plan-print-entries ans)))
  ans)

(defun add-serial (p1 p2 imm?)
  (add-general p1 p2 imm? 'serial))

(defun add-parallel (p1 p2)
  (if (null (plan-print-entries p2)) p1
    (add-general p1 p2 t 'parallel)))

;; This is the hard case.
;;   1.  If there is only one predecessor, it's a serial plan and we just
;;       do that.
;;   2.  If there is a collection of plans that everything has to go
;;       through (a "fan point"), we convert everything up to the fan point
;;       to a plan, then everything from the fan point on, and serialize
;;       the two results.
;;   3.  If none of these cases applies, it's handled by print-with-split.

(defun printable-preds (plan preds &aux fan)
  (cond ((null (cdr preds))
	 (printable-plan (car preds)))
	((setq fan (fan-in plan))
	 (add-serial (convert-with-fan plan fan)
		     (fan-to-printable-plan fan)
		     (plan-imm (car fan))))
	(t (print-with-split preds))))

;; In finding fan-in points, we are looking for a collection of actions
;; that "split" the basic plan, in the sense that the collection is
;; mutually unordered but every other point is either before or after
;; every element of the set.  We also don't want to use either the plan
;; itself or predecessors of it.

(defun fan-in (plan &aux (preds (plan-all-preds plan)))
  (some #'(lambda (p) (and (not (eql p plan))
			   (not (member p (plan-preds plan)))
			   (fan-in-1 p preds)))
	preds))

;; Here we work through the plan, trying to build up a fan from a given
;; seed.  For each predecessor, if it's before the seed, it has to be
;; before all of them.  Similarly if it's after.  If neither, it has
;; to be unordered with respect to all the other seeds, and has to have
;; the right relationship with all of the other precs/succs.

;; One other trick -- if the seed is s, there is no point adding to s
;; anything ahead of s in the pred list; we've already tried that and it
;; didn't work.

(defun fan-in-1 (seed preds &aux flag precs succs)
  (setq seed (list seed))
  (dolist (p preds seed)
    (unless (cond ((eql p (car seed)) (setq flag t))
		  ((plan-precedes p (car seed))
		   (when (every #'(lambda (x) (plan-precedes p x)) (cdr seed))
		     (push p precs)))
		  ((plan-precedes (car seed) p)
		   (when (every #'(lambda (x) (plan-precedes x p)) (cdr seed))
		     (push p succs)))
		  (flag
		   (when (and (notany #'(lambda (x) (or (plan-precedes x p)
							(plan-precedes p x)))
				      (cdr seed))
			      (every #'(lambda (x) (plan-precedes x p)) precs)
			      (every #'(lambda (x) (plan-precedes p x)) succs))
		     (push p (cdr seed)))))
      (return))))

;; early precedes late if they are the same, or if early precedes a
;; predecessor of late.

(marked-defun plan-precedes (early late) late nil
	      (if (eql early late) t
		(some #'(lambda (x) (plan-precedes early x))
		      (plan-preds late))))

;; to print out a plan up to a fan, we first prune any predecessors that
;; are in the fan, and then print it out normally!  Then we restore the
;; predecessors as they were originally.

(defun convert-with-fan (plan fan &aux (temps (remove-fan plan fan nil)))
  (prog1 (printable-plan plan t)
    (mapc #'(lambda (x) (setf (plan-preds (car x)) (cdr x))) temps)))

;; To remove elements of a fan from a plan, we check to see if any of
;; the top-level predecessors are in the fan.  If so, we save a list of
;; these predecessors (together with the plan, so we know what to put
;; back) and then remove those in the fan.  Then we call remove-fan again
;; on what's left.

(defun remove-fan (plan fan ans)
  (when (some #'(lambda (pred) (member pred fan)) (plan-preds plan))
    (push (cons plan (plan-preds plan)) ans)
    (setf (plan-preds plan) 
      (remove-if #'(lambda (x) (member x fan)) (plan-preds plan))))
  (dolist (item (plan-preds plan) ans)
    (setq ans (remove-fan item fan ans))))

;; print out a fan.  This is easy.

(defun fan-to-printable-plan (fan)
  (printable-plan (make-plan nil fan) t))

;; Here is the last case.  There are two possibilities: we can split the
;; current actions into two independent subsets (so that the entire plan
;; is parallel), or we can't.  (In which case we basically give up,
;; splitting off the first action and calling the result parallel even
;; though there are known to be intersections.)

;; We establish temp as a list of all predecessors to each top-level one.
;; Then if we can partition this into two subsets that are disjoint, we
;; split out those top-level predecessors that are contained in the first
;; of the two partitioning sets.  Otherwise, we just split out the first
;; top-level predecessor.

;; Then it's easy.  We parallelize a plan made from the split and one
;; made from what's left.

(defun print-with-split 
    (preds &aux (temp (mapcar #'(lambda (p) (plan-all-preds p)) preds))
		(part (partition temp))
		(split (if part
			   (remove nil
				   (mapcar #'(lambda (x y)
					       (when (member (car y) part) x))
					   preds temp))
			 (list (car preds)))))
  (add-parallel (printable-plan (make-plan nil split) t)
		(printable-plan (make-plan nil (set-difference preds split))
				t)))

;; Here we have a plan and want to return all predecessors of it.  A
;; simple recursive walk.

(marked-walk plan-all-preds () nil (cons plan ans))

;; Now we have a list S of sets and want to split it into P and Q such
;; that the union of all the sets in P is disjoint from the union of all
;; the sets in Q.  If we can do so, we return U, the union of the elements
;; of P.

;; The way we do this is as follows: we put the first set in P and set
;; U=P.  Now any set that intersects any set in P must also go in P; we
;; remove such sets from S as we add them to U.  If at any point we fail
;; to find such a set, we know we're done and can return U.  If at any
;; point we run out of sets to look at, we know we can't partition the
;; original sets and return failure.

(defun partition (sets &aux (p (pop sets)))
  (do (flag)
      ((null sets) (return nil))
    (setq flag nil
	  sets (remove-if #'(lambda (set) (when (intersection set p)
					    (setq flag t p (union set p))))
			  sets))
    (unless flag (return p))))

;; Two plans are the same if they are eql, or if their actions and imm
;; flags are the same and they have equal sets of predecessors.

(defun plan-same (p1 p2)
  (or (eql p1 p2)
      (and (eql (plan-imm p1) (plan-imm p2))
	   (action-same (plan-action p1) (plan-action p2))
	   (set-equal (plan-preds p1) (plan-preds p2) #'plan-same))))

;; Actions are the same if they are both equal or both variables. Note
;; that (move a ?) and (move a ?1) are *not* the same action, since
;; (holds (at a ?)) holds after one but not the other.

(defun action-same (a1 a2)
  (if (varp a1) (varp a2) (equal a1 a2)))

;; plugging.  If the bdgs are nil, no copy is ever made.

(defun plan-plug (plan bdgs)
  (if bdgs (with-marks (pp-1 plan bdgs))
    plan))

;; It's a little bit subtle because once we plug into an action, that
;; action may be the predecessor of other actions that we haven't gotten
;; to yet.  So the marks pushed onto the existing mark are of the form
;;   (finished-mark . result)
;; where result is the result of plugging the bindings into this action.

(defun pp-1 (plan bdgs)
  (cond ((marked-with-cons plan))
	((some #'(lambda (v) (assoc v bdgs)) (get-plan-vars plan))
	 (mark-with-cons plan (make-plan (plug (plan-action plan) bdgs)
					 (mapcar #'(lambda (x) (pp-1 x bdgs))
						 (plan-preds plan))
					 (plan-imm plan)
					 (plan-mark plan))))
	(t plan)))

;; destructive version.  Just walk through, doing the plug on the action.

(defun nplan-plug (plan bdgs)
  (if bdgs (npp plan bdgs) plan))

(marked-defun npp (plan bdgs) plan plan
	      (when (some #'(lambda (v) (assoc v bdgs)) (get-plan-vars plan))
		(setf (plan-action plan) (plug (plan-action plan) bdgs)
		      (plan-vars plan) (vars-in (plug (plan-vars plan) bdgs)))
		(mapc #'(lambda (x) (npp x bdgs)) (plan-preds plan)))
	      plan)

;; Code to merge two plans.  In some sense, this is easy.  We figure out
;; which plan is smaller, mark it, and then do everything incrementally.
;; We also check to see if the bigger plan is an instance of the smaller;
;; that makes the merge computation *real* easy!

(defun merge-plans (p1 p2 &aux c1 c2)
  (cond ((eq p1 plan-root) (list p2))
	((eq p2 plan-root) (list p1))
	((< (setq c1 (count-actions p1)) (setq c2 (count-actions p2)))
	 (mp-0 p1 p2 nil))
	(t (mp-0 p2 p1 (eql c1 c2)))))

(defun mp-0 (p1 p2 try-both)
  (or (merge-from-instance p2 p1)
      (and try-both (merge-from-instance p1 p2))
      (copy-list (merge-internal p1 p2 t))))

(defun merge-from-instance (p1 p2)
  (when (merge-internal p2 p1 nil) (list p1)))

;; Here is the guts of merge-plan.  We mark the actions in p1 from 0
;; upward, call mp-2 to do the dirty work, and then unmark everything in
;; sight.  The result is a list of (bdgs . plan), where plan has the
;; ordering information right but hasn't had any plugs done to it.  So we
;; do the plug and return.

;; There is some cleanup at the end.  If we've located a point x in the
;; merge, and also a predecessor p of x, it's possible that there is
;; another path from x to p that has arisen from another source or some
;; such.  (This can only happen if p is not immediate, of course.)

;; So we look at all the plans and subplans.  For each, we look at each
;; predecessor p.  If p is marked and not immediate, and if p precedes
;; some predecessor of plan other than p itself, we remove p from the
;; list of plan's predecessors.  true-mark is a simple utility that
;; ignores the junk that we might have put on the marks during the merge.

(defun merge-internal (p1 p2 merge? &aux result)
  (mark p1 -1)
  (setq result (mp-2 p1 (list (cons nil (make-separate p2))) merge?))
  (unmark p1)
  (if merge? (cleanup-merge result) (cleanup-inst result)))

(defun cleanup-merge (b-ps)
  (napcar #'(lambda (b-p)
	      (delete-redundant-arcs (cdr b-p))
	      (unmark (cdr b-p))
	      (nplan-plug (cdr b-p) (car b-p)))
	  b-ps))

(defun cleanup-inst (b-ps)
  (napcar #'(lambda (b-p)
	      (delete-if #'(lambda (x) (eql (car x) (cdr x))) (car b-p)))
	  b-ps))

(marked-defun delete-redundant-arcs (plan) plan plan
	      (mapc #'delete-redundant-arcs (plan-preds plan))
	      (setf (plan-preds plan)
		  (remove-if #'(lambda (p)
				 (and (not (plan-imm p))
				      (true-mark (plan-mark p))
				      (some #'(lambda (x)
						(and (not (eql x p))
						     (plan-precedes p x)))
					    (plan-preds plan))))
			     (plan-preds plan)))
	      plan)

;; mark a plan.  Just walk through it, marking anything that doesn't yet
;; have a mark.  The mark is (n . t), where n is an integer used to
;; continue to mark the action in the result (to help keep track of the
;; constraints) and t means that we haven't yet merged this node into the
;; final answer.

(defun mark (plan count)
  (setf (plan-mark plan) (cons (incf count) t))
  (dolist (item (plan-preds plan) count)
    (unless (plan-mark item) (setq count (mark item count)))))

;; The first thing we have to do is make separate copies of the two
;; plans if their action lists overlap, since this will raise absolute
;; havoc with the marking mechanism.  So we first check for the overlap
;; (indicated by a mark in p2 being a cons -- this being the mark set up
;; for p1); if so, we recopy the plan.

(defun make-separate (plan)
  (if (overlap plan) (recopy plan) plan))

(marked-defun overlap (plan) plan nil
	      (or (consp (plan-mark plan)) (some #'overlap (plan-preds plan))))

;; The action is here.  But before we get started, note that we can't
;; merge p1 into p2 an action at a time, because we don't really know
;; where to put the immediate actions in p1.  So what we actually do is
;; to merge p1 into p2 a *segment* at a time, where a segment consists of
;; a non-immediate action and then a maximal sequence of immediate
;; predecessors.

;; As we do the merge, we maintain a list of (bdg . plan) pairs or B-P
;; pairs.  merge-internal has set this list to the given bindings and p2.
;; We find the terminating segment in p1; if it isn't done (the cdr of
;; the terminal action in the segment is still T), we invoke mp-2
;; recursively on what precedes it.  Then we do the merge on the segment
;; itself, and set the cdr of the terminal action to NIL.

;; The final argument is T if we're doing a merge and NIL if we're doing
;; an instance check.

(defun mp-2 (p1 b-ps merge? &aux (seg (terminal-segment p1)))
  (dolist (item (plan-preds (car (last seg))))
    (when (cdr (plan-mark item))
      (setq b-ps (mp-2 item b-ps merge?))))
  (setq b-ps (mapcan #'(lambda (b-p) (seg-merge seg b-p merge?)) b-ps))
  (setf (cdr (plan-mark p1)) nil)
  b-ps)

;; Find the terminal segment of a plan.  Just walk along, pushing
;; immediate predecessors until you run out.

(defun terminal-segment (plan &aux (ans (list plan)))
  (do () (nil)
    (if (setq plan (immediate-predecessor plan))
	(push plan ans)
      (return (nreverse ans)))))

(defun immediate-predecessor (plan &aux (preds (plan-preds plan)))
  (and preds (plan-imm (car preds)) (car preds)))

;; Here we merge the segment.  We begin by merging the terminal action
;; only; action-merge returns a list of MERGED structures.  The fields in
;; this structure are as follows:
;;  bp		is the usual bdg-plan combination
;;  perfect	indicates whether the match is perfect
;;  place	where in the new plan is the action located?
;;  root	where in the new plan is the end of the segment located?
;; This list has the property that *late* elements correspond to points
;; where the segment was found early in p2.  This is important because we
;; want to use only the first perfect match we find -- provided that we
;; can also locate the remainder of the segment.

;; Now we work through the rest of the segment.  Then at the end, we
;; drop anything from the overall list that precedes a still-perfect
;; match.

(defstruct (merged (:constructor make-merged (bp perfect place root)))
  bp perfect place root)

(defun seg-merge (seg b-p merge? &aux merges ans)
  (setq merges (action-merge (car seg) (car b-p) (cdr b-p) merge?))
  (dolist (imm (cdr seg))
    (setq merges (mapcan #'(lambda (m) (imm-merge imm m merge?)) merges)))
  (setq ans merges)
  (mapl #'(lambda (ms) (when (merged-perfect (car ms)) (setq ans ms)))
	merges)
  (napcar #'merged-bp ans))

;; Here we try to merge the action into a plan, returning a list of
;; MERGEDs.  There are two ways to do this.  We can be lazy and simply
;; stick the action into the plan in parallel, or we can try to actually
;; locate it in the plan somehow.  We put the first case at the beginning
;; so that we drop it if any perfect merges were found.

(defun action-merge
    (action bdgs plan merge?
     &aux (ans (merge-action-find action plan plan bdgs merge?)) temp)
  (if (and merge? (setq temp (append-action action plan)))
      (cons (make-merged (cons bdgs temp) t (car (plan-preds temp))
			 (car (plan-preds temp)))
	    ans) 
    ans))

;; just appending the action is a bit easier.  We stick it on, and then
;; find the actions in plan that are marked with the same marker as the
;; predecessors to the given action.  The actions are now predecessors to
;; the newly added action, not predecessors to the original action.

;; It's a bit subtle because some of the action's predecessors may have
;; been mapped to immediate actions in the new plan.  So we have to
;; change the predecessors to point to the non-immediate actions that
;; follow them.  The reason we can do this is that we know the
;; predecessor isn't immediate in ACTION (we're at the terminal action in
;; the segment), so ACTION's immediate predecessors haven't been matched
;; yet.  We can't give the immediate action in the new plan a second
;; successor, so we have to back up.  Of course, if we *can't* back up,
;; then the attempt to append the action to the new plan fails.  At the
;; end, we have to cater to the possibility that ACTION is immediate (for
;; example), pushing the other actions in the new plan past it if so.
;; push-past-immediate does this destructively.

(defun append-action (action plan &aux (a (plan-action action)))
  (unless (member a '(initial terminal))
    (let* ((preds (find-marks (remove-if #'plan-imm (plan-preds action)) plan))
	   (ipreds (remove-if-not #'plan-imm preds))
	   (temp (when ipreds (later-than-immediate plan ipreds))) ans)
      (when (or (null ipreds) (= (length ipreds) (length temp)))
	(setq preds (nconc (delete-duplicates temp)
			   (set-difference preds ipreds)))
	(setq ans 
	  (make-plan 'terminal
		     (cons (make-plan a preds nil (plan-mark action))
			   (set-difference (plan-preds plan) preds))))
	(when (push-past-immediate ans) ans)))))

(marked-walk later-than-immediate (points) nil
	     (unless (or (eql (plan-action plan) 'terminal)
			 (plan-imm plan))
	       (do ((imm plan)) (nil)
		 (if (setq imm (immediate-predecessor imm))
		     (when (member imm points) (push plan ans))
		   (return))))
	     ans)

;; Now we actually try to locate the action in the original plan.  This
;; is made harder by the fact that if we do the merge (changing the
;; action in orig), we want the newly constructed action to get the marks
;; that used to be in orig -- but *not* the currents mark in orig (quite
;; often finished-mark, which won't do us any good at all).  When we do
;; the insertion, we pop the mark until it starts with a number.

;; So we begin.  If the new place already precedes the given action, we
;; don't have to do anything, including trying to merge with predecessors
;; of the place.  Any merge here or with a predecessor would cause a
;; loop.  Otherwise, for each predecessor to this plan, we invoke
;; merge-action-find recursively to attempt to match the action to the
;; predecessors, accumulating the results.  We also merge the action
;; here, if it's legal to do so.

(marked-defun merge-action-find (action orig place bdgs merge?) place nil
	      (unless (merge-precedes action orig place)
		(nconc (delete nil (napcar #'(lambda (b)
					       (absorb-action action orig place
							      b bdgs merge?))
					   (legal-merge action place bdgs
							merge?)))
		       (mapcan #'(lambda (x)
				   (merge-action-find action orig x bdgs
						      merge?))
			       (plan-preds place)))))

;; PLACE (in the PLAN getting the action merged in) effectively precedes
;; ACTION (in the other plan) if PLACE precedes anything marked by
;; the mark of a predecessor to ACTION (all such predecessors having
;; been located by now).

(defun merge-precedes (action plan place)
  (mark-precedes (mapcar #'plan-mark (not-imm-preds action))
		 plan place nil))

(defun not-imm-preds (plan)
  (do (imm) (nil)
    (if (setq imm (immediate-predecessor plan))
	(setq plan imm)
      (return (plan-preds plan)))))

(defun mark-precedes (marks plan place flag)
  (setq flag (or flag (member (true-mark (plan-mark plan)) marks)))
  (if (eql plan place) flag
    (some #'(lambda (x) (mark-precedes marks x place flag))
	  (plan-preds plan))))

;; Here we check to see if it's legal to merge ACTION at PLACE.  This
;; means that the action and place have to unify (we use the instance
;; check if appropriate instead), and also that no temporal constraints
;; are introduced if we're doing an instance check.  The unification is
;; cheaper, so we do that first and return the result if everything else
;; checks out as well.

;; In the instance computation, if the action has extra predecessors,
;; merging it in would break the instance requirements.  (The action is
;; from the set with fewer actions, so we're trying to show that orig is
;; an instance of action.)  We don't worry about ACTION's immediate
;; predecessors, since they are part of the current segment and will be
;; dealt with separately later.

(defun legal-merge (action place bdgs merge?)
  (and (setq bdgs (unify-actions merge? nil action place bdgs))
       (or merge? (every #'(lambda (pred)
			     (or (plan-imm pred)
				 (precedes (plan-mark pred) place)))
			 (plan-preds action)))
       bdgs))

;; Here we absorb a single action into a plan at the given place; the
;; hard part is getting the constraints right.  It all happens only if we
;; are doing a merge as opposed to an instance check, of course.  (For an
;; instance check, it's already known to be ok.)

;; We first have to be careful not to include all the finished-mark
;; stuff in the marks; we just want the original marks.  We also have to
;; pass the mark from the given action to the new place.  Most of this is
;; handled by recopy.

;; What we want to do is to add to the predecessors of PLACE all of the
;; non-immediate predecessors of ACTION; all these predecessors have been
;; located in PLAN by now.  (The immediate predecessors are handled
;; later.)  We use find-marks to locate them.  Having found these
;; predecessors, we add them to the predecessors of the action.  Then we
;; simplify the result using combine-predecessors to keep only the latest
;; of the predecessors.

;; We also have to handle any immediacy stuff.  Note that ACTION is not
;; immediate, although PLACE might be.  This means that all that can go
;; wrong is that the new action has multiple predecessors, of which one
;; is immediate.  (This can only happen in merging, since that is the
;; only time we might add new non-immediate predecessors to action.)
;; This problem is handled by push-past-immediate.

;; Finally, recopy is used to make a copy of PLAN where all of this
;; modification is done destructively.  It also locates the PLACE in the
;; new plan.

;; At the end, we construct a MERGED to return.

(defvar new-places)

(defun absorb-action (action plan place bdgs old-bdgs merge?
		      &aux new-place preds fail?)
  (setq plan (recopy plan (list place))
	new-place (car new-places))
  (setf (plan-mark new-place) (plan-mark action))
  (when (and merge? (setq preds (find-marks (plan-preds action) plan)))
    (setf (plan-preds new-place)
      (combine-predecessors preds (plan-preds new-place)))
    (setq fail? (not (push-past-immediate new-place))))
  (unless fail? 
    (make-merged (cons bdgs plan) 
		 (perfect-match bdgs old-bdgs plan new-place)
		 new-place new-place)))

;; push nonimmediate predecessors of plan past any immediate predecessors,
;; returning NIL if something went wrong and T otherwise.

;; First set temp to the immediate predecessor.  If there's a problem, set
;; new-loc to temp and then advance new-loc until it has no immediate
;; predecessor.  If new-loc is now initial, you're stuck.  Otherwise, push
;; all the remaining predecessors onto the predecessor list for new-loc,
;; set the unique predecessor of plan to temp, and return T.

(defun push-past-immediate (plan &aux (preds (plan-preds plan)) temp)
  (when (and (cdr preds) (setq temp (member-if #'plan-imm preds)))
    (let ((new-loc (car temp)))
      (do (imm) (nil)
	(if (setq imm (immediate-predecessor new-loc))
	    (setq new-loc imm)
	  (return)))
      (when (eql (plan-action new-loc) 'initial)
	(return-from push-past-immediate nil))
      (pushconc (delete (car temp) preds :count 1) (plan-preds new-loc))
      (setf (plan-preds plan) (list (car temp)))))
  t)

;; A match is perfect if the bindings match and if the place is connected
;; only by immediate links to the terminal action.

(defun perfect-match (bdgs old-bdgs plan place)
  (and (equal-binding bdgs old-bdgs)
       (or (not (plan-imm place))
	   (immediate-path plan place))))

;; Is there a path from the terminal node of plan to place?

(defun immediate-path (plan place)
  (do ()
      ((eql plan place) t)
    (unless (setq plan (immediate-predecessor plan)) (return))))

;; Here we have an action a and a MERGED; the action is an immediate
;; predecessor of the given place p in the merged and we want to merge it
;; in.  There are four cases:

;; 1. [found] p has an immediate predecessor.  In this case, it has to
;; merge with action and the match can continue to be perfect.  The other
;; cases apply for merge? = T only, since we can't add arcs or merge the
;; immediate a with something non immediate in the instance case.

;; 2. [direct] a merges with a predecessor to p.  Now we have to make
;; this predecessor immediate, pushing the others past it if possible.
;; The match can continue to be perfect.

;; 3. [other] a merges with a point x that is neither after p (which
;; would cause a loop) or before it (which won't work because there are
;; points caught in the middle).  We're going to have to move x before p,
;; so x can't be immediate.

;; 4. [append] we can just stick it on as a new predecessor to p,
;; sorting out the result with push-past-immediate.

;; In the following computation, we do append last because it's
;; destructive.  But we return append first because if any other case
;; is perfect, we want to drop append.

(defun imm-merge (action merged merge? 
		  &aux (imm (immediate-predecessor (merged-place merged)))
		       (b-p (merged-bp merged)))
  (cond (imm (imm-merge-found action merged merge? imm b-p))
	(merge?
	 (delete nil
		 (let ((direct (imm-merge-direct action merged b-p))
		       (other (imm-merge-other action (cdr b-p) (cdr b-p)
					       (car b-p) merged nil))
		       (append (imm-merge-append action merged (cdr b-p))))
		   (nconc append other direct))))))

;; 1. [found] p has an immediate predecessor.  In this case, it has to
;; merge with action and the match can continue to be perfect.

(defun imm-merge-found (action merged merge? imm b-p)
  (napcar #'(lambda (b) (make-new-merged merged b (cdr b-p) (car b-p) imm))
	  (unify-actions merge? t action
			 (car (plan-preds (merged-place merged)))
			 (car b-p))))

(defun make-new-merged (merged bdgs plan old-bdgs place)
  (make-merged (cons bdgs plan)
	       (and (merged-perfect merged) (equal-binding bdgs old-bdgs))
	       place (merged-root merged)))

;; 2. [direct] a merges with a predecessor to p.  Now we have to make
;; this predecessor immediate, pushing the others past it if possible.
;; The match can continue to be perfect.

(defun imm-merge-direct (action merged b-p)
  (mapcan #'(lambda (pred) (imd-1 action merged b-p pred))
	  (plan-preds (merged-place merged))))

(defun imd-1 (action merged b-p pred)
  (napcar #'(lambda (b) (imd-2 merged b-p pred b (car b-p)))
	  (unify-actions t t action pred (car b-p))))

;; Here we do the work for a specific predecessor and binding list.  We
;; find the pred in the new plan and make it immediate.  Then we try to
;; push the nonimmediate predecessors out of the way and make a new
;; MERGED if we manage it.

(defun imd-2 (merged b-p pred bdgs old-bdgs
	      &aux (new (recopy (cdr b-p) (list pred (merged-place merged)))))
  (setf (plan-imm (car new-places)) t)
  (when (push-past-immediate (second new-places))
    (make-new-merged merged bdgs new old-bdgs (car new-places))))

;; 3. [other] a merges with a point x that is neither before p (which
;; would cause a loop) or after it (which won't work because there are
;; points caught in the middle).  We're going to have to move x after p,
;; so x can't be immediate.

;; This part of the code finds the possible matches.

(marked-defun imm-merge-other (action orig plan bdgs merged ans) plan ans
	      (let ((place (merged-place merged)))
		(cond ((plan-precedes plan place) ans)
		      (t (dolist (item (plan-preds plan))
			   (setq ans (imm-merge-other action orig item bdgs
						      merged ans)))
			 (if (or (plan-imm plan) (plan-precedes place plan))
			     ans
			   (nconc (napcar #'(lambda (b)
					      (imo-1 orig plan merged b))
					  (unify-actions t t action plan bdgs))
				  ans))))))

;; Here we do the merge.  x is the point unifying with the action, p is
;; the action's successor and r is the root of the segment being
;; analyzed.  We begin by taking all the actions that have x as a
;; predecessor and changing that predecessor to r.  Then we make x the
;; predecessor to p in the new plan, and the predecessors to x the
;; combined predecessors to x and to p.  We also make x immediate.  At
;; the end, we have to make sure that any immediate predecessors of x
;; are handled correctly.

(defun imo-1 (orig x merged bdgs &aux (p (merged-place merged))
				      (r (merged-root merged)) new)
  (setq new (move-x-to-r orig x r p)
	p (third new-places) x (second new-places))
  (setf (plan-preds x) (combine-predecessors (plan-preds x) (plan-preds p))
	(plan-preds p) (list x)
	(plan-imm x) t)
  (when (push-past-immediate x)
    (make-merged (cons bdgs new) nil x (car new-places))))

;; Make a copy of the plan, changing every predecessor that's x to r.
;; In mxr, we can stop when we get to x or to r, since they are presumed
;; to be temporally unordered.  Otherwise, we simply substitute one for
;; the other in the predecessor list.  (In fact, since we do the subst
;; before dealing with the predecessors, we know that x will have been
;; substituted out and need only stop when we get to r.)

;; One other detail.  If r is a direct predecessor of plan and x is any
;; other predecessor of plan, x will eventually be replaced with r, which
;; is redundant.  So we drop r from the predecessor list of plan in this
;; case.

(defun move-x-to-r (plan x r p &aux (ans (recopy plan (list r x p))))
  (with-marks (mxr ans (second new-places) (car new-places)))
  ans)

(defun mxr (plan old new)
  (unless (or (eql plan new) (marked plan))
    (save-mark plan)
    (when (and (member new (plan-preds plan)) (plan-precedes old plan))
      (popf (plan-preds plan) new :count 1))
    (mapc #'(lambda (p) (mxr p old new))
	  (setf (plan-preds plan) (nsubst new old (plan-preds plan))))))

;; 4. [append] we can just stick it on as a new predecessor to p,
;; sorting out the result with push-past-immediate.

;; This is destructive; we've already made a copy of the plan at the
;; successor to action even if we did so only using append-action.

;; Consider the predecessors of action, which have been located in plan
;; by now.  Any that already precedes place can be dropped as a
;; predecessor, since that information is redundant.  This even works if
;; it's a direct predecessor of place, since push-past-immediate will
;; move it back after action in any case.

(defun imm-merge-append (action merged plan &aux (a (plan-action action)))
  (unless (eql a 'initial)
    (let* ((place (merged-place merged))
	   (preds (remove-if #'(lambda (x) (plan-precedes x place))
			     (find-marks (plan-preds action) plan))))
      (push (make-plan a preds t (plan-mark action)) (plan-preds place))
      (when (push-past-immediate place)
	(list merged)))))

;; What follows is a variety of utilities used in the merge construction.

;; Here we have a set of marks and a plan; we want to find all the
;; subplans of plan that are marked with marks in the set.

(defun find-marks (search-for plan)
  (when search-for (fm plan (mapcar #'plan-mark search-for))))

(marked-walk fm (marks) nil
	     (when (member (plan-mark plan) marks) (push plan ans))
	     ans)

;; Figure out if the given mark already precedes the action, in the
;; sense that either it is the action's mark, or the mark of a
;; predecessor to the action.  (In merging, actions may be different and
;; we use the marks to tell what's what.)

(defun precedes (mark action)
  (when mark (prec-1 mark action)))

(marked-defun prec-1 (mark plan) plan nil
	      (or (eql mark (true-mark (plan-mark plan))) 
		  (some #'(lambda (x) (prec-1 mark x)) (plan-preds plan))))

;; unifier/instance check

(defun unify-actions (merge? imm? action place bdgs)
  (when (or (null (plan-mark place)) (cdr (plan-mark place))
	    (and imm? (equal (plan-mark place) '(-1))))
    (setq action (plan-action action) place (plan-action place))
    (if (or (member action '(initial terminal)) 
	    (member place '(initial terminal)))
	(when (eql action place) (list bdgs))
      (funcall (if merge? #'unifyp+ #'action-inst) action place
	       (list (copy-alist bdgs))))))

;; a version of the unifier that accepts an incoming binding list.  Code
;; taken from match.lisp, give or take.  The last line of unify+ has been
;; modified.

(defvar unify-dummies)
(defvar unify*flg)

(defun unifyp+ (p q bdgs &aux unify-dummies (blists (unify+ p q bdgs)))
  (cond ((car blists)
	 (napcar #'copy-alist (cdr blists))
	 (napcar #'subst-bdgs blists)
	 (delete 'fail blists))
	(t blists)))

(defun unify+ (p q bdgs &aux unify*flg)
  (if (or (atom p) (atom q)) (setq p (list p) q (list q)))
  (unify-1 p q bdgs nil))

;; when is a1 an instance of a2?  It's basically inst-1, but there is
;; one problem.  The two actions (move x ?) and (move x ?1) are not the
;; same because x is at ? after one and at ?1 after the other.  So they
;; can't be instances of each other either (since action-inst is supposed
;; to be antisymmetric).  That means that if the binding list constructed
;; binds a new variable to another *variable*, it's no good.

(defun action-inst (a1 a2 bdgs
		    &aux (ans (inst-1 (list a1) (list a2) bdgs nil)))
  (if (varp a1) ans
    (delete-if #'(lambda (b)
		   (some #'(lambda (x)
			     (and (not (eql (car x) (cdr x)))
				  (not (member (car x) (car bdgs) :key #'car))
				  (varp (cdr x))))
			 b))
	       ans)))

;; Recopying a plan is a lot like plug; we have to be sure that we use a
;; copy of a predecessor that's already been copied.  The old marks can
;; be found by popping the given marks until we get to a number.  The
;; global variable new-places is set to a list that gives us the new
;; plans corresponding to the elements of PLACES.

(defun recopy (plan &optional places)
  (setq new-places places)
  (with-marks (recopy-1 plan)))

(defun recopy-1 (plan)
  (or (marked-with-cons plan)
      (let ((ans (make-plan (plan-action plan)
			    (mapcar #'recopy-1 (plan-preds plan))
			    (plan-imm plan) 
			    (true-mark (plan-mark plan)))))
	(nsubst ans plan new-places)
	(mark-with-cons plan ans))))

(defun true-mark (mark)
  (when mark
    (if (numberp (car mark)) mark (true-mark (cdr mark)))))

;; Combine the predecessors from two points, tossing any that precede
;; others.

(defun combine-predecessors (list1 list2)
  (nconc (remove-predecessors list1 list2)
	 (remove-predecessors list2 list1)
	 (intersection list1 list2)))

;; remove from LIST any plans that are already known to be preceded by
;; an element of OTHERS.

(defun remove-predecessors (list others)
  (remove-if #'(lambda (p) (some #'(lambda (o) (plan-precedes p o)) others))
	     list))

;; Code to tell if plan1 is an instance of plan2.

(defun plan-instance (p1 p2)
  (cond ((or (eql p1 p2) (eq p2 plan-root)) (list nil))
	((eq p2 plan-init) (eq p1 plan-init))
	((not (or (eq p1 plan-root) (eq p1 plan-init)))
	 (unless (> (count-actions p2) (count-actions p1))
	   (copy-list (merge-internal p2 p1 nil))))))

;; Here we coalesce two plans that hopefully are identical except that
;; two specific actions occur in reverse order in the two plans.

;; If this is to happen, the plans must have the same number of actions;
;; now coalesce-p checks to see if they can be combined.  If so, ptr is
;; returned as a cons of the two swappable actions (the one that is first
;; in p1 is the car), and coalesce-doit actually constructs the new plan.

(defun coalesce (p1 p2 &aux ptr)
  (when (and (eql (count-actions p1) (count-actions p2))
	     (setq ptr (car (coalesce-p p1 p2 (cons nil nil)))))
    (coalesce-doit p1 ptr)))

;; If p1 and p2 are to be merged, there are two cases.  If they have the
;; same action, then we can move along to the predecessors.  If the
;; initial action is different, then coalesce-p-mismatch checks to make
;; sure the mismatch can be handled.

(defun coalesce-p (p1 p2 mismatch)
  (or (and (action-same (plan-action p1) (plan-action p2))
	   (coalesce-p-continue (plan-preds p1) (plan-preds p2) mismatch))
      (coalesce-p-mismatch p1 p2 mismatch)))

;; This is the easy case; we just have to check to make sure that the
;; predicates are equal, where equal means we can coalesce them given the
;; currently-discovered mismatch.

(defun coalesce-p-continue (l1 l2 mismatch &aux temp)
  (when (set-equal l1 l2
		   #'(lambda (x y) (when (setq temp (coalesce-p x y mismatch))
				     (setq mismatch temp))))
    mismatch))

;; Here we actually have a mismatch.  If there is already a mismatch, it
;; had better be the same as the current one.  If not, then each action
;; has to have a single predecessor that is the same as the other action.
;; In addition, the plans that follow the predecessors had better be the
;; same.  This is the case where we actually construct the mismatch.

(defun coalesce-p-mismatch (p1 p2 mismatch)
  (if (car mismatch)
      (when (and (eql p1 (car mismatch)) (eql p2 (cdr mismatch))) mismatch)
    (let ((pr1 (plan-preds p1)) (pr2 (plan-preds p2)))
      (when (and pr1 (null (cdr pr1)) pr2 (null (cdr pr2))
		 (not (plan-imm p1)) (not (plan-imm p2))
		 (action-same (plan-action (car pr1)) (plan-action p2))
		 (action-same (plan-action (car pr2)) (plan-action p1))
		 (set-equal (plan-preds (car pr1)) (plan-preds (car pr2))
			    #'plan-same))
	(cons p1 p2)))))

;; Here we construct the merged plan.  We make a copy of the existing
;; plan, using new-places to record where merge-pt is in the new plan.
;; Now we hang onto the predecessor of merge-copy, and replace them with
;; the predecessors of that predecessor.  Then push-ptr makes sure that
;; everything that points to the first predecessor points to the second
;; as well.

(defun coalesce-doit (p1 merge-pt &aux temp)
  (setq p1 (recopy p1 (list merge-pt)))
  (setq merge-pt (car new-places)
	temp (car (plan-preds merge-pt)))
  (setf (plan-preds merge-pt) (plan-preds temp))
  (push-ptr p1 (car new-places) temp)
  p1)

(marked-defun push-ptr (plan old new) plan nil
	      (if (member old (plan-preds plan))
		  (push new (plan-preds plan))
		(mapc #'(lambda (p) (push-ptr p old new)) (plan-preds plan))))

;; Similar but subtraction.  The actions have to be the same, and the
;; instance computation (which we presumably know succeeds) has to return
;; (NIL).  subtract-p does the rest of the check; subtract-doit does the
;; work.

(defun subtract (p1 p2 &aux ptr)
  (when (and (= (count-actions p1) (count-actions p2))
	     (null (car (plan-instance p2 p1)))
	     (setq ptr (subtract-p p1 p2)))
    (subtract-doit p2 (car ptr) (cdr ptr))))

;; Here is the check to see if p1 can be subtracted from p2.  There are
;; two possibilities:

;; 1.  p2 has a pred not in p1.  This first time this happens, this is
;; the point at which the unordered kids (in p2) show up.  The (unique)
;; extra pred in p2 is the p2 version of the point that is early in p1,
;; and we call it early-2.  The second (and presumably last!) time, it is
;; because some predecessor of early-1 in p1 (perhaps even the initial
;; action) is now reported as being a predecessor of the point that
;; happens to be late-2.

;; 2.  p1 has a pred not in p2.  This pred must be early-1, since it's
;; the ordered version early-2.  early-1 and early-2 must be the same
;; plan.  (We handle the late points first, to make sure that early-2 is
;; established before we get here.)  The other part of the check can also
;; be done now; p1, with the extra pred (late-1) removed, must be the
;; same plan as late-2.  But it's not quite so simple; late-2 will have
;; predecessors that don't show up in p1 because they are predecessors of
;; late-1.  So we remove from late-2 anything known to also precedes
;; early-2.  Now late-2 and early-1 (with late-1 removed) should be the
;; same plan.

;; At the end, we return the dotted pair (late-2 . early-2).

(defun subtract-p (p1 p2 &aux early-2 late-2 late-2-list)
  (declare (special early-2 late-2 late-2-list))
  (when (sp-1 p1 p2) (cons late-2 early-2)))

(defun sp-1 (p1 p2 &aux (pr1 (plan-preds p1)) (pr2 (plan-preds p2)) found)
  (declare (special early-2 late-2 late-2-list))
  (dolist (item pr2)
    (cond ((setq found (find-action item pr1))
	   (unless (eql (plan-imm item) (plan-imm found)) (return-from sp-1)))
	  (early-2
	   (cond ((or (eql early-2 item) (eql late-2 item)))
		 ((or late-2 (not (member p2 late-2-list)))
		  (return-from sp-1))
		 ((setq late-2 p2))))
	  ((plan-imm item) (return-from sp-1))
	  (t (setq early-2 item late-2-list (remove item pr2)))))
  (dolist (item pr1 t)
    (unless
	(if (setq found (find-action item pr2))
	    (and (eql (plan-imm item) (plan-imm found)) (sp-1 item found))
	  (and (plan-same early-2 item)
	       (plan-same (make-plan (plan-action p1) (remove item pr1))
			  (make-plan (plan-action late-2)
				     (remove-if #'(lambda (p) 
						    (plan-precedes p early-2))
						(plan-preds late-2))))))
      (return))))

(defun find-action (plan list)
  (find (plan-action plan) list :key #'plan-action :test #'action-same))

;; Here we actually do the subtraction.  If early and late are the
;; special points, then everything that points to early and late should
;; point to early one (this is what sub-ptr does), and early should point
;; to late and the things it used to point to.

(defun subtract-doit (plan late early &aux temp)
  (setq plan (recopy plan (list early late))
	early (car new-places) late (second new-places)
	temp (combine-predecessors (list late) (plan-preds early)))
  (sub-ptr plan early late)
  (setf (plan-preds early) temp)
  plan)

(marked-defun sub-ptr (plan early late) plan nil
	      (if (member early (plan-preds plan))
		  (popf (plan-preds plan) late :count 1)
		(mapc #'(lambda (p) (sub-ptr p early late))
		      (plan-preds plan))))

;; Plan dag!  Assemble the pieces ...

(defparameter *plan-dag*
	      (make-dag :root plan-root :eq #'plan-same :leq #'plan-instance
			:dot #'merge-plans :add #'coalesce :sub #'subtract
			:long "Plans" :short "Plans"
			:vars #'(lambda (x) (copy-list (get-plan-vars x)))
			:plug #'plan-plug))

;; Most of the time is spent doing plan instance computations, which are
;; done over and over again for the same plans.  So we do something like
;; what Norvig suggests in his common lisp book to cache the results.

;; It's complicated by the fact that we can't hash plans directly;
;; sxhash returns a value independent of the values at the slots.  So we
;; use get-plan-hash to hash plans ourselves.  In the memoized version,
;; we get a hash key by hashing the args using args-to-hash, then see if
;; we've got an entry that actually matches this one as given by
;; plan-match.  If so, we use that answer (perhaps doing a plug if
;; necessary); if not, we push the args and answer onto the hash entry
;; for the given key.

(defun memo (fn name &aux (table (make-hash-table :test #'equal)))
  (setf (get name 'memo) table)
  #'(lambda (&rest args &aux (key (args-to-hash args)) temp bdgs)
      (if (setq temp 
	    (find-if #'(lambda (x) (setq bdgs (plan-match (car x) args)))
		     (gethash key table)))
	  (adjust-for-return (cdr temp) (car bdgs))
	(prog1 (setq temp (apply fn args))
	  (push (cons args temp) (gethash key table))))))

;; Hash an arglist by hashing the plans yourself and leaving the other
;; args alone.

(defun args-to-hash (args)
  (mapcar #'(lambda (x) (if (plan-p x) (get-plan-hash x) x)) args))

;; Two arglists match if they are samep -- but we already know the lists
;; are ok (because the hash table checks), so we only have to check the
;; plans, which we do in pm-1.  The binding list is built globally and
;; stored in blist.

(defun plan-match (entry args &aux blist)
  (declare (special blist))
  (when (every #'(lambda (x y) (if (plan-p x) (pm-1 x y) (eql x y)))
	       entry args)
    (samep-answer blist)))

;; Two plans match if they have the same immediate flags, the actions
;; are samep, they have the same number of preds, and the preds all
;; match.

(defun pm-1 (p1 p2)
  (and (eql (plan-imm p1) (plan-imm p2))
       (samep1 (plan-action p1) (plan-action p2))
       (= (length (setq p1 (plan-preds p1)))
	  (length (setq p2 (plan-preds p2))))
       (every #'pm-1 p1 p2)))

;; Here we do any plug involved in returning the answer, which is either
;; a list of plans or a list of bdg lists.

(defun adjust-for-return (ans bdgs)
  (when ans 
    (mapcar (if (plan-p (car ans))
		#'(lambda (x) (plan-plug x bdgs))
	      #'(lambda (x) (plug x bdgs)))
	    ans)))

;; From the Norvig book.

(defun memoize (fn-name)
  (setf (symbol-function fn-name)
    (memo (symbol-function fn-name) fn-name)))

(defun clear-memoize (fn-name &aux (table (get fn-name 'memo)))
  (when table (clrhash table)))

(memoize 'merge-internal)
