;;; -*- Mode: LISP; Syntax: Common-lisp; Package: MVL; Base: 10 -*-

(in-package "MVL")

;; This file constructs the bilattice used by the planner.  This is
;; constructed as the set of functions from the dag of plans into the
;; first-order bilattice.

(defvar *active-bilattice*)

(defun plan-base-bilattice ()
  (second (bilattice-components *active-bilattice*)))

;; Dag-fns that are true or true by default at a given plan.  The
;; default case takes a 2nd arg (a level), and uses true at that level.

(defun plan-true-at (plan &aux (base (plan-base-bilattice)))
  (make-tv *plan-dag* base plan (bilattice-true base)))

(defun plan-default-at 
    (plan &optional (level 1) &aux (base (plan-base-bilattice)))
  (make-tv *plan-dag* base plan (cons level (bilattice-true base))))

;; delay a dag-fn by a certain amount of time.  plan-advance does the work.

(defun plan-delay (time v)
  (dag-change-dag v #'(lambda (x) (plan-advance x time))))

;; advance a plan by a certain amount of time.  Just push that many
;; dummy actions onto the end of the plan and set the now-p flag
;; appropriately.

(defun plan-advance (plan amount)
  (if (plusp amount)
      (make-plan 'terminal (pa-1 (plan-preds plan) amount) t)
    plan))

(defun pa-1 (plans amount)
  (dotimes (i amount plans)
    (setq plans (list (make-plan (new-?var) plans t)))))

(defparameter *plan-delay* 
    (make-modal-op :name 'delay :fn #'plan-delay :args '(0 1)))

;; The guts of the planner are in some sense right here -- in the
;; propagation function, which is the formal analog of what's been called
;; the modal truth criterion elsewhere.  This is enough of a mess that we
;; do it using dag-accumulate-fn.  We also ensure that when we process a
;; point, all predecessors to that point have already been processed.  By
;; "process" a point, we mean adding a new element to the list of points
;; to be accumulated based on the point being considered.

;; There are the following cases:

;; 1. The point is ongoing, and the value is not unknown.  This should
;; never happen; the view we take is that the ongoing value overwrites
;; everything after it, so that no successor of the given point need be
;; considered.  (Handled 3rd below.)

;; 2.  The point is ongoing and the value is unknown.  If there is an
;; immediate version of the point on the list (with value other than
;; unknown), we do nothing since the immediate version will overwrite
;; this one.  Otherwise, we pass the ongoing point to the list of values
;; to be accumulated.  (Handled 4th below.)

;; 3.  The point is immediate and the value is not unknown.  We push the
;; corresponding ongoing point onto the accumulation list.  We also push
;; the point onto the propagated list (see reconstruct-with-propagation).
;; (Handled 2nd below.)

;; 4.  The point is immediate and the value is unknown.  We push the
;; point onto a list of failing actions to deal with later.  (Handled 1st
;; below.)

(defun plan-propagate (fn &aux (bilattice (dag-fn-bilattice fn)))
  (if (mvl-unk (dag-fn-root-val fn) bilattice)
      (plan-prop-1 fn bilattice)
    (make-root (dag-fn-dag fn) bilattice (dag-fn-root-val fn))))

;; Here we do the work.  remaining is a list of points still to be
;; considered; completed is a list of points that are done.  We begin by
;; finding an element of remaining with the property that every other
;; unprocessed element does not precede it.  We remove the element from
;; remaining and add it to completed.  Then we handle the four cases as
;; above.

;; At the end, finish-propagation cleans up.

(defun plan-prop-1
    (fn b &aux (dag-list (dag-fn-list fn)) list propagated failed)
  (do ((remaining (copy-list (all-dag-pts fn)))
       x pt immediate unk completed temp)
      ((null remaining) (finish-propagation propagated failed list b))
    (setq remaining
      (delete-if #'(lambda (y)
		     (when (every #'(lambda (z)
				      (or (member z completed)
					  (not (member y (dag-entry-imm z)))))
				  dag-list)
		       (setq x y)))
		 remaining :count 1)
      pt (find-entry x dag-list)
      immediate (immediate-predecessor x)
      unk (mvl-unk (dag-entry-val pt) b))
    (push pt completed)
    (cond ((and immediate unk) (push pt failed))
	  (immediate (push (cons (make-ongoing x) (dag-entry-val pt)) list)
		     (push (car list) propagated))
	  ((not unk)
	   (push (cons x (dag-entry-val pt)) list)
	   (setq remaining (nset-difference remaining (dag-entry-succ pt))))
	  ((or (not (setq temp
		      (find-if #'(lambda (p) (immediate-version-p p x))
			       (dag-entry-imm pt))))
	       (mvl-unk (dag-entry-val (find-entry temp dag-list)) b))
	   (push (cons x (dag-entry-val pt)) list)))))

(defun make-ongoing (plan &optional (imm (immediate-predecessor plan)))
  (if imm
      (make-plan 'terminal
		 (list (make-plan (plan-action imm) (plan-preds imm))))
    plan))

(defun make-immediate (plan &aux (act (car (plan-preds plan))))
  (make-plan 'terminal
	     (list (make-plan (plan-action act) (plan-preds act) t))))

(defun immediate-version-p (imm ong &aux (i (immediate-predecessor imm)))
  (and i (plan-same (make-ongoing imm i) ong)))

;; At the end, we call this function to clean up.  propagated is a list
;; of points where we propagated a value, failed a list of points where
;; the (immediate) value was unknown, and list is the (plan .  value)
;; points we're working with.

;; To understand the principal complexity, suppose that we invoke
;; plan-prop-1 on a dag-fn that's true at x and false at y (both
;; immediate).  Since making x and y ongoing produces two plans that
;; merge (the originals didn't), we need to specify values at the merges.
;; If x is last, the function should be true; if y last, false.  The
;; remainder of the code deals with failed actions.

(defun finish-propagation (propagated failed list b 
			   &aux (ans (merge-tails propagated list b)))
  (if failed (handle-failures failed ans) ans))

;; Here we handle the final actions in the plans.  final is the
;; propagated action that is going to be last, and other is some other
;; action.  We don't have to do anything if the values agree, of course.
;; Otherwise, we merge the plans.  If we just get final or other back
;; again, we don't have to do anything since the given value is correct.
;; Failing that, we split final out of the merged plan and add a new
;; point to the propagated function.

(defun merge-tails (propagated list b)
  (when (cdr propagated)
    (dolist (final propagated)
      (dolist (other propagated)
	(unless (or (eql final other) (mvl-eq (cdr final) (cdr other) b))
	  (mapc #'(lambda (p)
		    (unless (or (plan-same p (car final))
				(plan-same p (car other)))
		      (push (cons p (cdr final)) list)))
		(napcar #'make-ongoing
			(merge-plans (make-immediate (car final))
				     (car other))))))))
  (dag-accumulate-fn *plan-dag* b list))

;; Here we handle failed actions.  Each element of failed is a
;; dag-entry, so we can process them sequentially.  For each one, we
;; figure that the final action is the failing one, so we pop that off.
;; Then we get the value that the function apparently has there, return
;; the final action to the plan, make it ongoing, and then add it to the
;; overall answer.  At the end, we simplify everything.

(defun handle-failures (failed fn)
  (do (x)
      ((null failed) (simplify fn))
    (setq failed
      (delete-if #'(lambda (y &aux (pt (dag-entry-pt y)))
		     (unless
			 (some #'(lambda (z) (member pt (dag-entry-succ z)))
			       failed)
		       (setq x (dag-entry-pt y))))
		 failed :count 1))
    (add-dag (make-ongoing x) (get-val (drop-final x) fn) fn)))

;; drop the final action from a plan

(defun drop-final (plan)
  (make-plan 'terminal (plan-preds (car (plan-preds plan)))))

(defparameter *plan-propagate*
    (make-modal-op :name 'propagate :fn #'plan-propagate :args '(x)))

;; here is something that ignores "exceptions" to plans.

(defun plan-ignore-exceptions (fn &aux list)
  (setq fn (copy-entire-fn fn)
	list (dag-fn-list fn))
  (mapc #'(lambda (x) (setf (dag-entry-val x) (pie-1 x list))) list)
  (simplify fn))

;; here we do the disjunction. It's easy because the dag-fn structure
;; already includes information about points where x is the successor.  We
;; stop when there are no new actions in the next part of the plan, since
;; that's good enough -- imposing a constraint isn't a change of measure
;; zero because there are only a finite number of actions to consider.

(defun pie-1 (x orig &aux (d (dag-entry-pt x)) (v (dag-entry-val x))
			  (actions (cdr (nbutlast (all-actions d)))))
  (dolist (item orig v)
    (when (and (member d (dag-entry-succ item))
	       (not (set-equal (cdr (nbutlast (all-actions 
					       (dag-entry-pt item))))
			       actions
			       #'action-same)))
      (setq v (mvl-or v (dag-entry-val item) (plan-base-bilattice))))))

(marked-walk all-actions () nil (cons (plan-action plan) ans))

(defparameter *ignore-exceptions*
    (make-modal-op :name 'x 
		   :fn #'(lambda (x) (mvl-t-ground (plan-ignore-exceptions x)))
		   :args '(t)))

(defun make-plan-bilattice 
    (&aux (b (dag-bilattice *plan-dag* *first-order-bilattice*)))
  (describe-bilattice b "Planning" "Planning" #\z)
  (bilattice-has-modal-op b *plan-delay*)
  (bilattice-has-modal-op b *plan-propagate*)
  (bilattice-has-modal-op b *extract-value*)
  (bilattice-has-modal-op b *ignore-exceptions*)
  b)

(defparameter *plan-bilattice* (make-plan-bilattice))

(bilattice *plan-bilattice*)

(defun make-hplan-bilattice 
    (&aux (b (dag-bilattice *plan-dag* *hierarchy-bilattice*)))
  (describe-bilattice b "Hierarchical planning" "Hierarchical planning" #\x)
  (bilattice-has-modal-op b *plan-delay*)
  (bilattice-has-modal-op b *plan-propagate*)
  (bilattice-has-modal-op b *extract-value*)
  (bilattice-has-modal-op b *ignore-exceptions*)
  b)

(defparameter *hplan-bilattice* (make-hplan-bilattice))

(bilattice *hplan-bilattice*)
