;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: USER; Base: 10 -*-

;;
;; MVL package, file-system, and system definitions.
;;
;; In order to load the system on a new machine, edit the default
;; pathname specified in the defsystem form, which can be found by
;; searching for the string `***'.  If ANSI, set the logical
;; pathname "MVL" to the directory where MVL is kept.

;; The first thing we do is decide whether we're using the ANSI
;; standard or Steele's 1984 version of Common Lisp.  The way this
;; is done is to see if in-package is a function (1984) or a macro
;; (ANSI).

(eval-when (compile load)
  (when (macro-function 'in-package) (pushnew :ANSI *features*)))

;; MVL package interface.

#+ANSI (defpackage "MVL" (:use "COMMON-LISP"))
#+ANSI (in-package "MVL")

#-ANSI (in-package "MVL" :use '(lisp))

;; Note: the names of modal operators are also exported.  The function
;; load-logic-1 performs the exporting when a bilattice defining modal
;; operators is loaded.

(export '(|=>|				;operators supposedly understood by MVL
	  |<=|
	  |<=>|
	  if
	  iff
	  ?
	  ?*

	  activate			;operators in the manual appendix
	  activate-theory-mechanism
	  *active-bilattice*
	  *active-theory*
	  add-dag
	  *always-succeeds*
	  answer
	  answer-binding
	  answer-value
	  anytime-trace
	  anytime-untrace
	  append-bilattice
	  append-binding-lists
	  at
	  *atms-bilattice*
	  bagof
	  bc
	  bc-trace
	  bc-untrace
	  bc-unwatch
	  bc-watch
	  bcs
	  bdg-bottom
	  bdg-false
	  bdg-to-truth-val
	  bdg-true
	  bdg-unknown
	  bilattice
	  *bilattices*
	  binding-dag
	  binding-le
	  bottom
	  *break-node*
	  choose-proof
	  circum
	  *circum-trace*
	  circums
	  *circumscription-bilattice*
	  cnf
	  *cnf-atms-bilattice*
	  combine-1
	  combine-fns
	  compile-mvl
	  *completed-database*
	  cont-analysis
	  cont-prover
	  contents
	  create-mvl-invocation
	  cutoffs-not
	  dag-accumulate-fn
	  dag-bilattice
	  dag-change-dag
	  dag-change-tv
	  dag-drop
	  dag-fn-list
	  dag-list-of-answers
	  dag-prune
	  *default-bilattice*
	  delay
	  delete-bdg
	  demo-mvl
	  denotes
	  *depth-limit*
	  deselect-tasks
	  dnf
	  done
	  dot-with-not
	  dot-with-star
	  dynamic-anytime-trace
	  dynamic-anytime-untrace
	  dynamic-bc-trace
	  dynamic-bc-untrace
	  dynamic-unview
	  dynamic-view
	  empty
	  empty-theory
	  equal-answer
	  equal-binding
	  *equivalence-translation*
	  erase
	  erase-from-theory
	  existing-mvl-invocation
	  false
	  fc
	  fc-trace
	  fc-untrace
	  *first-order-atms-bilattice*
	  *first-order-bilattice*
	  fn-and
	  fn-comp
	  fn-dot
	  fn-dws
	  fn-eq
	  fn-not
	  fn-or
	  fn-plus
	  get-bdg
	  get-mvl
	  get-val
	  *global*
	  groundp
	  hierarchy-bilattice
	  *hierarchy-bilattice*
	  *if-translation*
	  includes
	  index
	  init-analysis
	  init-prover
	  instp
	  invoke
	  k-ge
	  k-gt
	  k-le
	  k-lt
	  k-not-ge
	  k-not-gt
	  k-not-le
	  k-not-lt
	  known
	  lattice-to-bilattice
	  lattice-to-dag
	  lattice-to-dag-to-bilattice
	  lisp-defined
	  lisp-predicate
	  load-logic
	  load-mvl
	  lookup
	  lookup-call
	  lookups
	  lookups-call
	  make-answer
	  make-plan
	  make-root
	  make-tv
	  matchp
	  monotonic-cutoff
	  *monotonic-test*
	  mvl-and
	  mvl-demo-file
	  mvl-dot
	  mvl-eq
	  mvl-file
	  *mvl-invocation*
	  mvl-load
	  mvl-not
	  mvl-or
	  mvl-plug
	  mvl-plus
	  mvl-print
	  mvl-save
	  mvl-t-ground
	  mvl-test-file
	  mvl-unload
	  mvl-vars
	  negate
	  *never-succeeds*
	  new-*var
	  new-?var
	  new-val
	  normal-form
	  *overviewer-size*
	  pause
	  *plan-bilattice*
	  plan-from-action
	  plan-init
	  plan-true-at
	  plan-vars
	  plug
	  prfacts
	  *probability-bilattice*
	  product-bilattice
	  propagate
	  property
	  props
	  quiet
	  rem-mvl
	  samep
	  *save-nodes*
	  select-tasks
	  selected-tasks
	  set-equal
	  simplify
	  splice-bilattice
	  splitting-point-below
	  standardize-operators
	  stash
	  stash-value
	  state
	  state-in-theory
	  static-anytime-trace
	  static-anytime-untrace
	  static-bc-trace
	  static-bc-untrace
	  static-display
	  std-cutoffs
	  *success-test*
	  t-ge
	  t-gt
	  t-le
	  t-lt
	  t-not-ge
	  t-not-gt
	  t-not-le
	  t-not-lt
	  test
	  test-mvl
	  theory-active-p
	  theory-contents
	  theory-dag
	  *time-bilattice*
	  true
	  true-at
	  true-in-theory
	  truth-value
	  unifyp
	  unincludes
	  unindex
	  unknown
	  unstash
	  unstash-facts-with-atom
	  unstash-value
	  unstate
	  unstate-from-theory
	  value
	  varp
	  varp*
	  varp?
	  vars-in
	  vartype
	  *viewer-scroll-bars*
	  *viewer-size*
	  with-mvl-invocation))

(defmacro read-time-case (first-case &rest other-cases)
  "Do the first case, where cases are specified with #+ or possibly #-
  conditionals.  The final, default case has no conditional."
  (declare (ignore other-cases))
  first-case)

#-ANSI
(defmacro with-compilation-unit (options &body body)
  (declare (ignore options))
  `(progn .,body))

;; MVL system definition.

;; Ensure a defsystem macro, using a simple parser to translate the
;; defsystem into a list of files in the correct dependency order.

(defmacro system-source-files (name)
  `(get (canonicalize-system ,name) :system-source-files))

(defmacro system-default-pathname (name)
  `(get (canonicalize-system ,name) :system-default-pathname))
   
(defmacro defsystem (name options &body body)
  (let ((default-pathname (getf options :default-pathname))
	(modules nil)
	(definition nil))
    (dolist (clause (copy-list body))	;copy-list for safety ...
      (ecase (first clause)
	((:module :module-group)
	 (push (cons (second clause) (third clause)) modules))
	((:serial :parallel :definitions)
	 (setq definition (rest clause)))))
    (when (null definition)
      (setq definition (mapcar #'car (reverse modules))))
    (labels ((parse-system (list modules)
	       (etypecase (first list)
		 ;; Base case.
		 (null nil)
		 ;; A string is a file name.
		 (string (cons (first list)
			       (parse-system (rest list) modules)))
		 ;; A symbol is a module name.
		 (symbol (parse-system 
			  (append (cdr (assoc (first list) modules))
				  (rest list))
			  (remove (first list) modules :key #'car)))
		 ;; A cons is another definition.
		 (cons (parse-system (append (rest (first list)) (rest list))
				     modules)))))
      `(setf (system-source-files ',name)
	 (mapcar #'(lambda (file)
		     #+ANSI (make-pathname :host ,default-pathname :name file)
		     #-ANSI (merge-pathnames file ,default-pathname))
		 ',(parse-system definition modules))
	 (system-default-pathname ',name)
	 ,default-pathname))))

(defun canonicalize-system (name)
  (values (intern (string-upcase (string name)) 'keyword)))

;; Define the MVL system.
;;
;; *** Edit the value of :default-pathname to reflect the location of
;; the source code on your machine or include a form like
;;  (setf (logical-pathname-translations "MVL")
;;        '(("*.*" "/u/ginsberg/mvl/")))
;; when initializing common lisp.

(defsystem mvl
    (:default-pathname #+ANSI "MVL" #-ANSI "/u/ginsberg/mvl/"
     :default-package 'mvl
     :pretty-name "Multivalued Logics")
  (:module basic ("symbols" "bindings" "match" "tags" "index"))
  (:module invocations ("invocations"))
  (:module query ("query"))
  (:module load ("load"))
  (:module manipulate ("product" "function" "surgery" "hierarchy" "dag"
				 "lattice"))
  (:module bilattices ("fol" "atms" "atms-cnf" "def" "fo-atms" "circ"
			     "prob" "temporal" "plan"))
  (:module dags ("plan-dag" "theory-dag" "binding-dag"))
  (:module logic ("logic"))
  (:module functions1 ("cnf" "rep" "lookup" "first-order"))
  (:module functions2 ("bc"))
  (:module functions3 ("first-order-user-interface"
		       "first-order-mvl-interface" "bc-user"
		       "bc-interface" "values" "relevance" "control" "fc" 
		       "attach" "test"))
  (:module machine-dependent
	   #.(prog1 #+Allegro '("fi" "viewer" "dynamic-viewer")
		    nil))
  (:serial basic invocations query load manipulate bilattices dags
	   logic functions1 functions2 functions3 machine-dependent))

;; System manipulation functions.

;; Load MVL.  Also updates the *features* list and invokes (empty) to
;; clean out the deductive database, if there is one.

(defun load-mvl ()
  (when (fboundp 'empty) (funcall 'empty))
  (mapc #'load (system-source-files :mvl))
  (pushnew :mvl *features*)
  (values))

;; Compile MVL.  Should only be used when source code is changed.

(defun compile-mvl (&key recompile)
  (when (fboundp 'empty) (funcall 'empty))
  (read-time-case
   #+Allegro
   (progn
     (if recompile 
	 (mapc #'(lambda (f) (progn (compile-file f) (load f)))
	       (system-source-files :mvl))
       (apply #'tpl::cload-command (system-source-files :mvl))))
   (progn
     recompile				;prevent compiler warning
     (mapc #'(lambda (f) (with-compilation-unit () (load (compile-file f))))
	   (system-source-files :mvl))))
  (pushnew :mvl *features*)
  (values))

;; MVL file-system interface.

;; Return a pathname, merging with the system default pathname.

(defun mvl-file (file &optional (ext "mvl") &key no-error)
  (unless (pathname-type file)
    (setq file (make-pathname :type ext :defaults file)))
  (let ((pathname #+ANSI (make-pathname :type ext :defaults file
					:host (system-default-pathname :mvl))
		  #-ANSI (merge-pathnames file
					  (system-default-pathname :mvl))))
    (or (probe-file file)
	(probe-file pathname)
	(and no-error pathname)
	(cerror "Prompt for replacement pathname"
		"MVL File not found for ~:@(~A~)." pathname)
	(progn
	  (format t "~%New pathname: ")
	  (mvl-file (read-line))))))
